/**EWQuakeAlert generates email messages when receiving HYP2000ARC
 * messages from Earthworm.
 * Copyright (C) 2009 Ruben S. Luis
 * This program is free software: you ca redistribute it an/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be usefull,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with EWQuakeAlert. If not, see <http://www.gnu.org/licenses/>.
 **/
package ewquakealert;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Vector;
import javax.imageio.ImageIO;

/**
 *
 * @author RL197718
 */
public class PrintPhases {
    Vector<Phase> phases;

    PrintPhases(Vector<Phase> phases, String filename) {
        this.phases = phases;


        // Create an image to save
        RenderedImage rendImage = myCreateImage();
        // Write generated image to a file
        try {
            // Save as PNG
            File file = new File(filename);
            ImageIO.write(rendImage, "png", file);

            EWQuakeAlertView.println("Produced image with acquired picks: " +
                    filename);
            // Save as JPEG
            //file = new File("newimage.jpg");
            //ImageIO.write(rendImage, "jpg", file);
        }
        catch (IOException e) {
            EWQuakeAlertView.println("Failed to produce image with acquired picks.");
        }
    }




    
    private RenderedImage myCreateImage() {

        int nPhases = phases.size();
        int width = Configuration.graphWidth;

        double minStartTime = Double.MAX_VALUE;
        double maxEndTime = 0;


        for (int i=0; i<nPhases; i++) {
            if (phases.elementAt(i).dataRetrieved) {
                if (phases.elementAt(i).time[0] < minStartTime)
                    minStartTime = phases.elementAt(i).time[0];
                if (phases.elementAt(i).time[phases.elementAt(i).time.length-1] > maxEndTime)
                    maxEndTime = phases.elementAt(i).time[phases.elementAt(i).time.length-1];
            }
        }
        double deltaX = (double)(width) / (maxEndTime - minStartTime);

        //Prepare data for drawing
        int graphheight = Configuration.graphHeight;
        int height = graphheight * (nPhases) + 20;

        // Create a buffered image in which to draw
        BufferedImage bufferedImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);

        // Create a graphics contents on the buffered image
        Graphics2D g2d = bufferedImage.createGraphics();

        // Draw graphics
        g2d.setColor(Color.white);
        g2d.fillRect(0, 0, width, height);

        for (int curElement = 0; curElement < nPhases; curElement++) {
            if (phases.elementAt(curElement).dataRetrieved) {

                int[] xPoints = new int[phases.elementAt(curElement).time.length];
                int[] yPoints = new int[phases.elementAt(curElement).data.length];

                for (int i=0; i<xPoints.length; i++)
                    xPoints[i] = (int)((phases.elementAt(curElement).time[i] -
                            minStartTime) * deltaX);



                int graphyDelta = Configuration.graphHeight * curElement;
                double[] yPointsTemp = new double[phases.elementAt(curElement).data.length];
                double meanData = 0;
                for (int i=0; i<yPoints.length; i++)
                    meanData += (double)(phases.elementAt(curElement).data[i]);
                meanData = meanData / (double)(yPoints.length);

                double maxData = Double.NEGATIVE_INFINITY;
                for (int i=0; i<yPoints.length; i++) {
                    yPointsTemp[i] = (double)(phases.elementAt(curElement).data[i]) - meanData;
                    if ((yPointsTemp[i] > 0)&(yPointsTemp[i] > maxData)) maxData = yPointsTemp[i];
                    if ((yPointsTemp[i] < 0)&(-yPointsTemp[i] > maxData)) maxData = -yPointsTemp[i];
                }

                for (int i=0; i<yPoints.length; i++)
                    yPoints[i] = (int)(graphyDelta + graphheight/2 - (yPointsTemp[i] / maxData) * graphheight/2);


                g2d.setColor(Color.blue);
                g2d.drawPolyline(xPoints, yPoints, xPoints.length);
            }

            
            //Draw phase label
            String labelStr = phases.elementAt(curElement).Sta + "." +
                    phases.elementAt(curElement).Chan + "." +
                    phases.elementAt(curElement).Net + "." +
                    phases.elementAt(curElement).Loc;
            g2d.setColor(Color.BLACK);
            g2d.drawString(labelStr, 10, 13 + curElement * graphheight);

            //Draw separating line
            g2d.setColor(Color.black);
            g2d.drawLine(0, (curElement + 1) * graphheight, width, (curElement + 1) * graphheight);

            //Draw phase indicating line
            
            double xPos = (phases.elementAt(curElement).StartTime/1000 -
                    minStartTime);
            int xPhase = (int)(xPos * deltaX);
            g2d.setColor(Color.RED);
            g2d.drawLine(xPhase, (curElement + 1) * graphheight,
                    xPhase, (curElement) * graphheight);

            //Draw phase type
            g2d.drawString(phases.elementAt(curElement).P, xPhase+1, 13 + curElement * graphheight);

            //Draw Coda Termination line
            xPos += phases.elementAt(curElement).CDuration/1000;
            xPhase += (int)(xPos * deltaX);
            g2d.setColor(Color.GREEN);
            g2d.drawLine(xPhase, (curElement + 1) * graphheight,
                    xPhase, (curElement) * graphheight);

            
        }

        //Plot time scale
        int yPos = graphheight * (nPhases);
        int NMARKS = 4;
        for (int i=0; i<NMARKS; i++) {
            int xPos = (int)(42 + (double)(i) * (double)(width-85)/(NMARKS-1));
            long timePos = (long)((xPos / deltaX + minStartTime ) * 1000);
            SimpleDateFormat df = new SimpleDateFormat("HH:mm:ss.SSS");
            //System.out.println(xPos + ": " + df.format(new Date(timePos)));

            //Draw small vertical line
            g2d.setColor(Color.black);
            g2d.drawLine(xPos, yPos-2, xPos, yPos+2);
            //DrawText
            g2d.drawString(df.format(new Date(timePos)), xPos - 40, yPos + 15);
        }

        // Graphics context no longer needed so dispose it
        g2d.dispose();

        return bufferedImage;
    }
}
