package pickewanalysis.waveform;

import edu.iris.dmc.seedcodec.Codec;
import edu.sc.seis.seisFile.mseed.Blockette1000;
import edu.sc.seis.seisFile.mseed.DataHeader;
import edu.sc.seis.seisFile.mseed.DataRecord;
import edu.sc.seis.seisFile.mseed.SeedRecord;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

public class WaveFormGroup extends ArrayList<WaveForm> {
    //Constants

    private static SimpleDateFormat miniseedDateFormat = new SimpleDateFormat("yyyy,DDD,HH:mm:ss.SSS");
    public double maxVal = 0, minVal = 0;

    public WaveForm addEmptyWaveForm(String stat, String chan, String net, String loc,
            long starttime, double samprate, int nsamp) {
        WaveForm output = new WaveForm(stat, chan, net, loc,
                starttime, samprate, new double[nsamp]);
        add(output);
        return output;
    }

    public long getStartTime() {
        if (this.isEmpty()) {
            return -1;
        }
        long earliestTime = Long.MAX_VALUE;
        for (WaveForm wave : this) {
            if (wave.getStartTime() < earliestTime) {
                earliestTime = wave.getStartTime();
            }
        }
        return earliestTime;
    }

    public long getEndTime() {
        if (this.isEmpty()) {
            return -1;
        }
        long latestTime = 0;
        for (WaveForm wave : this) {
            if (wave.getEndTime() > latestTime) {
                latestTime = wave.getEndTime();
            }
        }
        return latestTime;
    }

    public WaveFormGroup splitStation(String stat, String chan, String net, String loc) {
        WaveFormGroup output = new WaveFormGroup();
        for (WaveForm wave : this) {
            if (wave.isSCNL(stat, chan, net, loc)) {
                output.add(wave);
            }
        }
        return output;
    }

    public boolean containsStation(String stat, String chan, String net, String loc) {
        for (WaveForm wave : this) {
            if (wave.isSCNL(stat, chan, net, loc)) {
                return true;
            }
        }
        return false;
    }

    public List<WaveFormGroup> splitStations() {
        List<WaveFormGroup> groups = new ArrayList<WaveFormGroup>();
        boolean foundGroup = false;
        for (WaveForm wave : this) {
            for (WaveFormGroup curGroup : groups) {
                if (curGroup.containsStation(wave.stat, wave.chan, wave.net, wave.loc)) {
                    curGroup.add(wave);
                    foundGroup = true;
                    break;
                }
            }
            if (!foundGroup) {
                WaveFormGroup newGroup = new WaveFormGroup();
                newGroup.add(wave);
                groups.add(newGroup);
            }
        }
        return groups;
    }

    public static WaveFormGroup parseTankFile(String filename) throws Exception {
        //Open file
        RandomAccessFile Finput = new RandomAccessFile(filename, "r");
        //Reserve memory
        byte[] data = new byte[(int) Finput.length()];
        //Load data
        long readBytes = Finput.read(data);
        if (readBytes != data.length) {
            throw new Exception("Unable to read file");
        }
        //Close file
        Finput.close();
        return parseTraceBuf2(data);
    }

    public static WaveFormGroup parseTraceBuf2(byte[] data) throws Exception {
        //Byte buffer
        ByteBuffer buffer = ByteBuffer.wrap(data);
        //Create waveform group
        WaveFormGroup waveGroup = new WaveFormGroup();
        //Variables to be used for parsing
        int pinno, curPos;
        byte[] aux = new byte[9];
        char sampleType, sampleSize;
        //Parse messages
        while (buffer.hasRemaining()) {
            //Create new message
            WaveForm wave = new WaveForm();
            //header
            //Try to find the datatype
            curPos = buffer.position();
            sampleType = (char) buffer.get(curPos + 57);
            sampleSize = (char) buffer.get(curPos + 58);
            if (sampleType == 'i') {
                buffer.order(ByteOrder.LITTLE_ENDIAN);
            }
            //Read data from header
            pinno = buffer.getInt();
            wave.nsamp = buffer.getInt();
            wave.starttime = (long) (buffer.getDouble() * 1000.0);
            wave.endtime = (long) (buffer.getDouble() * 1000.0);
            wave.samprate = buffer.getDouble();
            buffer.get(aux, 0, 7);
            wave.stat = new String(aux, 0, 7).trim();
            buffer.get(aux, 0, 9);
            wave.net = new String(aux, 0, 9).trim();
            buffer.get(aux, 0, 4);
            wave.chan = new String(aux, 0, 4).trim();
            buffer.get(aux, 0, 3);
            wave.loc = new String(aux, 0, 3).trim();
            //Extra data which is not important for this
            buffer.position(buffer.position() + 9);
            //Read samples
            wave.samples = new double[wave.nsamp];
            if (sampleSize == '4') {
                for (int i = 0; i < wave.nsamp; i++) {
                    wave.samples[i] = (double)buffer.getInt();
                }
            } else {
                for (int i = 0; i < wave.nsamp; i++) {
                    wave.samples[i] = (double) buffer.getShort();
                }
            }
            //Calculate waveform average
            wave.average = 0;
            for (int i = 0; i < wave.nsamp; i++) {
                wave.average += wave.samples[i];
            }
            wave.average /= (double) wave.nsamp;
            //Store waveform
            waveGroup.add(wave);
        }
        return waveGroup;
    }

    public static WaveFormGroup parseMSEEDFile(String filename) throws Exception {
        //Open file
        RandomAccessFile Finput = new RandomAccessFile(filename, "r");
        //Reserve memory
        byte[] data = new byte[(int) Finput.length()];
        //Load data
        long readBytes = Finput.read(data);
        if (readBytes != data.length) {
            throw new Exception("Unable to read file");
        }
        //Close file
        Finput.close();
        //Create waveform group
        WaveFormGroup waveGroup = new WaveFormGroup();
        //Decompress data
        DataInputStream Binput = new DataInputStream(new ByteArrayInputStream(data));
        while (Binput.available() > 0) {
            //Create new message
            WaveForm wave = new WaveForm();
            //Read seed data
            SeedRecord sr = SeedRecord.read(Binput);
            DataRecord dr = (DataRecord) sr;
            DataHeader dh = dr.getHeader();
            byte[] basedata = dr.getData();
            //Get blockette1000 to determine byte order and compression scheme
            Blockette1000 b1000 = (Blockette1000) (sr.getBlockettes(1000))[0];
            int[] intSamples = new Codec().decompress(
                    b1000.getEncodingFormat(),
                    basedata,
                    dh.getNumSamples(),
                    b1000.isLittleEndian()).getAsInt();
            wave.nsamp = intSamples.length;
            wave.samples = new double[wave.nsamp];
            for (int i=0;i<wave.nsamp;i++) {
                wave.samples[i] = (double)intSamples[i];
            }
            //Calculate waveform average
            wave.average = 0;
            for (int i = 0; i < wave.nsamp; i++) {
                wave.average += wave.samples[i];
            }
            wave.average /= (double) wave.nsamp;

            wave.starttime =
                    miniseedDateFormat.parse(
                    dh.getStartTime().substring(0, 21)).getTime();

            wave.endtime =
                    miniseedDateFormat.parse(
                    dh.getEndTime().substring(0, 21)).getTime();
            wave.stat = dh.getStationIdentifier().trim();
            wave.chan = dh.getChannelIdentifier().trim();
            wave.net = dh.getNetworkCode().trim();
            wave.loc = dh.getLocationIdentifier();
            wave.samprate = (double) dh.getSampleRate();
            //Store waveform
            waveGroup.add(wave);
        }
        return waveGroup;
    }

    @Override
    public String toString() {
        String output = String.format(Locale.UK, "%5s.%3s.%2s.%2s %23s %23s %5s\n",
                "STAT", "CHN", "NT", "LC",
                "START TIME", "END TIME", "NSAMP");
        for (WaveForm wave : this) {
            output += wave.toString() + "\n";
        }
        return output;
    }

    public double[] getAllSamples(long starttime, long endtime) {
        double samprate = this.get(0).samprate;
        int nsamp = (int) ((double) (endtime - starttime) * samprate / 1000.0);
        double[] samples = new double[nsamp];
        for (WaveForm wave : this) {
            if (wave.getStartTime() < endtime && wave.getEndTime() > starttime) {
                //Process wave
                double[] wavesamp = wave.getSamples();
                int recordOffset = (int) ((double) (wave.getStartTime() - starttime) * samprate / 1000.0);
                int readOffset = 0;
                if (recordOffset < 0) {
                    readOffset = -recordOffset;
                    recordOffset = 0;
                }
                int readLength = wave.getSampleCount() - readOffset;
                if (endtime < wave.getEndTime()) {
                    //readLength -= (int) ((double) (wave.getEndTime() - endtime) * samprate / 1000.0 + 2);
                    readLength = (int) ((double) (endtime - wave.getStartTime()) * samprate / 1000.0) - readOffset;
                }
                System.arraycopy(wavesamp, readOffset, samples, recordOffset, readLength);
            }
        }
        return samples;
    }

    public double[] getAllSamplesZeroAverage(long starttime, long endtime) {
        return getAllSamplesZeroAverage(starttime, endtime, false);
    }

    public double[] getAllSamplesZeroAverage(long starttime, long endtime, boolean zeroaverage) {
        double[] samples = getAllSamples(starttime, endtime);
        double average = 0;
        if (!zeroaverage) {
            for (int i = 0; i < samples.length; i++) {
                average += samples[i];
            }
            average /= samples.length;
        }
        double[] zsamples = new double[samples.length];
        maxVal = Double.NEGATIVE_INFINITY;
        minVal = Double.POSITIVE_INFINITY;
        for (int i = 0; i < samples.length; i++) {
            zsamples[i] = samples[i] - average;
            if (zsamples[i] > maxVal) {
                maxVal = zsamples[i];
            }
            if (zsamples[i] < minVal) {
                minVal = zsamples[i];
            }
        }
        return zsamples;
    }
}
