package pickewanalysis.ewpicker;

import java.math.BigDecimal;
import java.util.Locale;
import pickewanalysis.ewmessages.CodaSCNL;
import pickewanalysis.ewmessages.EWMessageGroup;
import pickewanalysis.ewmessages.PickSCNL;
import pickewanalysis.waveform.WaveForm;
import pickewanalysis.waveform.WaveFormGroup;

public class EWPicker {

    private EWPickerParameters Gparm;
    private PickEWStation Sta;
    //Handle output waveforms for analysis
    public WaveFormGroup rdatGroup = new WaveFormGroup();
    public WaveFormGroup charGroup = new WaveFormGroup();
    public WaveFormGroup ltaGroup = new WaveFormGroup();
    public WaveFormGroup staGroup = new WaveFormGroup();
    public WaveFormGroup staltaGroup = new WaveFormGroup();
    private WaveForm curRdat;
    private WaveForm curChar;
    private WaveForm curLta;
    private WaveForm curSta;
    private WaveForm curStaLta;
    // Output data handling 
    public EWMessageGroup messages = new EWMessageGroup();
    //Handle individual samples
    int sample_index;
    int SeqNo = 0;

    //Construct
    public EWPicker(PickEWStation station, EWPickerParameters parameters) {
        this.Gparm = parameters;
        this.Sta = station;
        //Reset the station
        Sta.initVar();
        //Restart output waveforms
        rdatGroup = new WaveFormGroup();
        charGroup = new WaveFormGroup();
        ltaGroup = new WaveFormGroup();
        staGroup = new WaveFormGroup();
        staltaGroup = new WaveFormGroup();
    }

    public EWMessageGroup analyze(WaveFormGroup wfg) {
        for (WaveForm tracebuf : wfg) {
            //Prepare a cycle
            double GapSizeD;
            int GapSize;
            //Check if waveform is from the correct station
            if (!tracebuf.isSCNL(Sta.stat, Sta.chan, Sta.net, Sta.loc)) {
                continue;
            }
            if (Sta.first == 1) {
                Sta.endtime = (double) tracebuf.getEndTime() / 1000.0;
                Sta.first = 0;
                continue; //Strange, discards the first wave
            }
            //Prepare output waveforms
            curRdat = rdatGroup.addEmptyWaveForm(Sta.stat, Sta.chan, Sta.net, Sta.loc,
                    tracebuf.getStartTime(), tracebuf.getSamplingRate(),
                    tracebuf.getSampleCount());
            curChar = charGroup.addEmptyWaveForm(Sta.stat, Sta.chan, Sta.net, Sta.loc,
                    tracebuf.getStartTime(), tracebuf.getSamplingRate(),
                    tracebuf.getSampleCount());
            curSta = staGroup.addEmptyWaveForm(Sta.stat, Sta.chan, Sta.net, Sta.loc,
                    tracebuf.getStartTime(), tracebuf.getSamplingRate(),
                    tracebuf.getSampleCount());
            curLta = ltaGroup.addEmptyWaveForm(Sta.stat, Sta.chan, Sta.net, Sta.loc,
                    tracebuf.getStartTime(), tracebuf.getSamplingRate(),
                    tracebuf.getSampleCount());
            curStaLta = staltaGroup.addEmptyWaveForm(Sta.stat, Sta.chan, Sta.net, Sta.loc,
                    tracebuf.getStartTime(), tracebuf.getSamplingRate(),
                    tracebuf.getSampleCount());
            //Compute gaps
            GapSizeD = tracebuf.getSamplingRate() * ((double) tracebuf.getStartTime() / 1000.0 - Sta.endtime);
            if (GapSizeD < 0.0) { //Invalid time going backwards
                GapSize = 0;
            } else {
                GapSize = (int) (GapSizeD + 0.5);
            }
            //Interpolate missing samples
            if ((GapSize > 1) && (GapSize <= Gparm.MaxGap)) {
                //TODO: Interpolation
                System.err.println("Interpolation required but not yet supported.");
            }
            //Announce large sample gaps
            if (GapSize > Gparm.MaxGap) {
                //TODO: Announce graphically
                System.err.println("Large gap detected.");
            }
            //For big gaps, enter restart mode
            //Compute STAs and LTAs without picking
            if (Restart(tracebuf.getSampleCount(), GapSize) == 1) {
                for (int i = 0; i < tracebuf.getSampleCount(); i++) {
                    Sample(tracebuf.getIntSample(i));
                }
            } else {
                PickRA(tracebuf);
            }
            //Save time and amplitude of the end of the current message
            Sta.enddata = tracebuf.getIntSample(tracebuf.getSampleCount() - 1);
            Sta.endtime = (double) tracebuf.getEndTime() / 1000.0;
        }
        return messages;
    }

    private int Restart(int nsamp, int GapSize) {


        if (GapSize > Gparm.MaxGap) {
            Sta.initVar();
            Sta.ns_restart = nsamp;
            return 1;
        } else {
            if (Sta.ns_restart >= Gparm.RestartLength) {
                return 0;
            } else {
                Sta.ns_restart += nsamp;
                return 1;
            }
        }
    }

    private void Sample(int LongSample) {
        double rdif;
        double edat;
        final double small_double = 1.0e-10;

        //Store present value of filtered data
        Sta.rold = Sta.rdat;

        //Compute new value of filtered data
        Sta.rdat = (Sta.rdat * Sta.RawDataFilt)
                + (double) (LongSample - Sta.old_sample) + small_double;

        //Compute 1st difference of filtered data
        rdif = Sta.rdat - Sta.rold;

        //Store integer data value
        Sta.old_sample = LongSample;

        //Compute characteristic function
        edat = (Sta.rdat * Sta.rdat) + (Sta.CharFuncFilt * rdif * rdif);

        //Compute esta
        Sta.esta += Sta.StaFilt * (edat - Sta.esta);

        //Compute elta
        Sta.elta += Sta.LtaFilt * (edat - Sta.elta);

        //Compute eref
        Sta.eref = Sta.elta * Sta.EventThresh;

        //Compute eabs
        Sta.eabs = (Sta.RmavFilt * Sta.eabs)
                + ((1.0 - Sta.RmavFilt) * Math.abs(Sta.rdat));

        //Update output waveforms
        curRdat.addSample(Sta.rdat);
        curChar.addSample(edat);
        curSta.addSample(Sta.esta);
        curLta.addSample(Sta.elta);
        curStaLta.addSample(Sta.esta / Sta.elta);
    }

    private void PickRA(WaveForm WaveBuf) {
        int event_found;
        int event_active;
        sample_index = -1;
        Pick Pick = Sta.pick;
        Coda Coda = Sta.coda;

        if (Pick.status > 0 || Coda.status > 0) {
            event_active = EventActive(WaveBuf);
            if (event_active == 1) {
                return;
            }
            if (event_active <= 0) {
                messages.add(new PickEWDebugMsg((long) (Sta.pick.time * 1000), event_active));
            }
            //Go to search mode
        }
        //Search mode
        while (true) {
            event_found = ScanForEvent(WaveBuf);
            if (event_found == 0) {
                return;
            }
            event_active = EventActive(WaveBuf);
            if (event_active == 1) {
                return;
            }
            if (event_active <= 0) {
                messages.add(new PickEWDebugMsg((long) (Sta.pick.time * 1000), event_active));
            }
        }
    }

    private int ScanForEvent(WaveForm WaveBuf) {
        Pick Pick = Sta.pick;
        Coda Coda = Sta.coda;
        //Set pick and coda calculations to inactive mode
        Pick.status = Coda.status = 0;
        while (++sample_index < WaveBuf.getSampleCount()) {
            int old_sample;
            int new_sample;
            double old_eref;
            new_sample = WaveBuf.getIntSample(sample_index);
            old_sample = Sta.old_sample;
            old_eref = Sta.eref;
            //Update sample
            Sample(new_sample);
            //Station is assumed dead
            if (Sta.eabs > Sta.DeadSta) {
                continue;
            }
            //Has the sta abruplty increased?
            if (Sta.esta > Sta.eref) {
                int wi; //Window index
                //Initialize pick variables
                Pick.time = (double) WaveBuf.getStartTime() / 1000
                        + (double) sample_index / WaveBuf.getSamplingRate();
                Coda.len_win = 0;
                Coda.len_sec = 0;
                for (wi = 0; wi < 6; wi++) {
                    Coda.aav[wi] = 0;
                }
                Sta.crtinc = Sta.eref / Sta.Erefs;
                Sta.ecrit = old_eref;
                Sta.evlen = 0;
                Sta.isml = 0;
                Sta.k = 0;
                Sta.m = 1;
                Sta.mint = 0;
                Sta.ndrt = 0;
                Sta.next = 0;
                Sta.nzero = 0;
                Sta.rlast = Sta.rdat;
                Sta.rsrdat = 0;
                Sta.sarray[0] = new_sample;
                Sta.tmax = Math.abs(Sta.rdat);
                Sta.xfrz = 1.6 * Sta.eabs;
                //Threshold for big ZC
                Sta.xdot = new_sample - old_sample;
                Sta.rbig = ((Sta.xdot < 0) ? -Sta.xdot : Sta.xdot) / 3;
                Sta.rbig = (Sta.eabs > Sta.rbig) ? Sta.eabs : Sta.rbig;
                //Compute cocrit and the sign of coda len
                if (Sta.eabs > (Sta.AltCoda * Sta.CodaTerm)) { //Big
                    Sta.cocrit = Sta.PreEvent * Sta.eabs;
                    Coda.len_out = -1;
                } else {
                    Sta.cocrit = Sta.CodaTerm;
                    Coda.len_out = 1;
                }
                Pick.status = Coda.status = 1;
                return 1;
            }
        }
        return 0;
    }

    private int EventActive(WaveForm WaveBuf) {
        Pick Pick = Sta.pick;
        Coda Coda = Sta.coda;
        //An event is active. See if it shoud be declared
        while (++sample_index < WaveBuf.getSampleCount()) {
            int new_sample;
            new_sample = WaveBuf.getIntSample(sample_index);
            Sample(new_sample);
            //BEGIN CODA CALCULATION
            if (Coda.status == 1) {
                int lwindow = (int) (WaveBuf.getSamplingRate() + 0.5);
                if (Coda.len_win != 72) {
                    lwindow *= 2;
                }
                Sta.rsrdat += Math.abs(Sta.rdat);
                //Save windows specified in the pwin array
                if (++Sta.ndrt >= lwindow) {
                    double ave_abs_val;
                    int pwin[] = new int[]{1, 2, 3, 4, 5, 6, 7, 8, 10, 12, 16, 20, 24,
                        28, 32, 36, 40, 44, 48, 56, 64, 70, 73};
                    if (Coda.len_win++ == pwin[Sta.k]) {
                        for (int i = 5; i > 0; i--) {
                            Coda.aav[i] = Coda.aav[i - 1];
                        }
                        Sta.k++;
                    }

                    //Compute coda length in seconds
                    Coda.len_sec = (2 * Coda.len_win) - 1;
                    if (Coda.len_sec == 145) {
                        Coda.len_sec = 144;
                    }
                    //Flush pick from buffer if coda is long enough
                    if ((Pick.status == 2) && (Coda.len_sec >= Sta.MinCodaLen)) {
                        ReportPick();
                        Pick.status = 0;
                        if (Gparm.NoCoda == 1) {
                            Coda.status = 0;//Artificially terminate coda
                        }
                    }
                    //Compute and sabe average absolute value
                    ave_abs_val = Sta.rsrdat / (double) lwindow;
                    Coda.aav[0] = (int) (ave_abs_val + 0.5);
                    //Initialize counter and coda amp sum
                    Sta.ndrt = 0;
                    Sta.rsrdat = 0.0;
                    //See if coda calculation is over
                    if ((Coda.len_win == 73) || (ave_abs_val < Sta.cocrit)) {
                        if (Coda.len_sec < Sta.MinCodaLen) {
                            return -1;
                        }
                        Coda.len_out *= Coda.len_sec;
                        Coda.status = 2;
                    }
                }
            }
            //The pick has been reported
            //The coda is complete but hasn't been reported
            if (Coda.status == 2) {
                if (Pick.status == 0) {
                    ReportCoda();
                    Coda.status = 0;
                }
                if (Pick.status == 2) {
                    ReportPick();
                    ReportCoda();
                    Pick.status = Coda.status = 0;
                }
            }
            //A pick is active
            if (Pick.status == 1) {
                int i;
                int k;
                int noise;
                int itrm;
                double xon;
                double xpc;
                double xp0, xp1, xp2;
                //Save first 10 points after pick
                if (++Sta.evlen < 10) {
                    Sta.sarray[Sta.evlen] = new_sample;
                }
                //Store current data if it is extreme
                if (Sta.next < 3) {
                    double adata;
                    adata = Math.abs(Sta.rdat);
                    if (adata > Sta.tmax) {
                        Sta.tmax = adata;
                    }
                }
                //Test for large zero crossing
                if (Math.abs(Sta.rdat) >= Sta.rbig) {
                    if (Sign(Sta.rdat, Sta.rlast) != Sta.rdat) {
                        Sta.nzero++;
                        Sta.rlast = Sta.rdat;
                    }
                }
                //Increment zc interval counter
                if (++Sta.mint > Sta.MaxMint) {
                    return -2;
                }
                //Test for small zc
                if (Sign(Sta.rdat, Sta.rold) == Sta.rdat) {
                    continue;
                }
                //Small zc found, reset interval counter
                Sta.mint = 0;
                //Update ecrit
                Sta.ecrit += Sta.crtinc;
                Sta.isml++;
                if (Sta.esta > Sta.ecrit) {
                    Sta.isml = 0;
                }
                //Store extrema of preceeding half cycle
                if (Sta.next < 3) {
                    Pick.xpk[Sta.next++] = Sta.tmax;
                    if (Sta.next == 1) {
                        double vt3;
                        vt3 = Sta.tmax / 3.0;
                        Sta.rbig = (vt3 > Sta.rbig) ? vt3 : Sta.rbig;
                    }
                    Sta.tmax = 0.0;
                }
                //Compute itrm
                itrm = Sta.Itr1 + (Sta.m / Sta.Itr1);
                if (Sta.m > 150) {
                    itrm = 50;
                }
                //See if pick is over
                if ((++Sta.m != Sta.MinSmallZC) && (Sta.isml < itrm)) {
                    continue; //Not over
                }
                //See if pick is a noise pick
                //System.out.println(String.format(Locale.UK,
                //        "xpk: %.01f %.01f %.01f  m: %d  nzero: %d",
                //        Pick.xpk[0], Pick.xpk[1], Pick.xpk[2], Sta.m, Sta.nzero));
                noise = 1;
                for (i = 0; i < 3; i++) {
                    if (Pick.xpk[i] >= (double) Sta.MinPeakSize) {
                        if ((Sta.m == Sta.MinSmallZC)
                                && (Sta.nzero >= Sta.MinBigZC)) {
                            noise = 0;
                        }
                        break;
                    }
                }
                if (noise == 1) {
                    return -3;
                }
                //A valid pick was found, determine first motion
                Pick.FirstMotion = ' ';
                for (k = 0; true; k++) {
                    if (Sta.xdot <= 0) {
                        if ((Sta.sarray[k + 1] > Sta.sarray[k]) || (k == 8)) {
                            if (k == 0) {
                                break;
                            }
                            Pick.FirstMotion = 'D';
                            break;
                        }
                    } else {
                        if ((Sta.sarray[k + 1] < Sta.sarray[k]) || (k == 8)) {
                            if (k == 0) {
                                break;
                            }
                            Pick.FirstMotion = 'U';
                            break;
                        }
                    }
                }
                //Pick weight calculation
                xpc = (Pick.xpk[0] > Math.abs((double) Sta.sarray[0]))
                        ? Pick.xpk[0] : Pick.xpk[1];
                xon = Math.abs((double) Sta.xdot / Sta.xfrz);
                xp0 = Pick.xpk[0] / Sta.xfrz;
                xp1 = Pick.xpk[1] / Sta.xfrz;
                xp2 = Pick.xpk[2] / Sta.xfrz;
                Pick.weight = 3;
                if ((xp0 > 2.0) && (xon > 0.5) && (xpc > 25.0)) {
                    Pick.weight = 2;
                }
                if ((xp0 > 3.0) && ((xp1 > 3.0) || (xp2 > 3.0)) && (xon > 0.5)
                        && (xpc > 100.0)) {
                    Pick.weight = 1;
                }
                if ((xp0 > 4.0) && ((xp1 > 6.0) || (xp2 > 6.0)) & (xon > 0.5)
                        && (xpc > 200.0)) {
                    Pick.weight = 0;
                }
                Pick.status = 2; //Calculated but not reported
                //Report pick and coda
                if (Coda.status == 2) {
                    ReportPick();
                    ReportCoda();
                    Pick.status = Coda.status = 0;
                }
            }
            //End pick calculation
            if ((Pick.status == 0) && (Coda.status == 0)) {
                return 0;
            }
        }
        return 1; //Event is still active
    }

    private void ReportPick() {
        // Round pick arrival time to the 10-th of a second 
        BigDecimal bd = new BigDecimal(Sta.pick.time).setScale(2, BigDecimal.ROUND_HALF_DOWN);
        // Procuce pick_scnl message 
        PickSCNL newPick = new PickSCNL(
                1,
                255,
                SeqNo += Gparm.NoCoda,
                Sta.getSCNLStr(),
                Sta.pick.FirstMotion,
                Sta.pick.weight,
                bd.doubleValue(),
                new int[]{(int) (Sta.pick.xpk[0] + 0.5), (int) (Sta.pick.xpk[1] + 0.5), (int) (Sta.pick.xpk[2] + 0.5)});
        messages.add(newPick);
        //System.out.println(newPick.toString());
    }

    private void ReportCoda() {
        if (Gparm.NoCoda == 1) {
            return;
        }
        CodaSCNL newCoda = new CodaSCNL(
                1,
                255,
                SeqNo++, //Same seqno has last pick
                Sta.getSCNLStr(),
                new int[]{Sta.coda.aav[0], Sta.coda.aav[1], Sta.coda.aav[2], Sta.coda.aav[3], Sta.coda.aav[4], Sta.coda.aav[5]},
                Sta.coda.len_out,
                (double) curRdat.getStartTime() / 1000 + (double) sample_index / curRdat.getSampleCount());
        messages.add(newCoda);
        //System.out.println(newCoda.toString());
    }

    private double Sign(double x, double y) {
        if (x == 0.) {
            return 0.;
        }

        if (x < 0.) {
            if (y < 0.) {
                return x;
            } else {
                return -x;
            }
        } else {
            if (y < 0.) {
                return -x;
            } else {
                return x;
            }
        }
    }
    /*
    //Construct
    public EWPicker(PickEWStation station, EWPickerParameters pickParams) {
    this.Sta = station;
    Sta.initVar();
    this.pickParams = pickParams;
    }
    
    //Clean all and start again
    public void clearAll() {
    Sta.initVar();
    //lastSampleTime = Double.NEGATIVE_INFINITY;
    messages = new EWMessageGroup();
    //firstTrace = true;
    //Restart output waveforms
    rdatGroup = new WaveFormGroup();
    charGroup = new WaveFormGroup();
    ltaGroup = new WaveFormGroup();
    staGroup = new WaveFormGroup();
    staltaGroup = new WaveFormGroup();
    }
    
    //Analyze a list of TraceBufs
    public EWMessageGroup analyze(WaveFormGroup wfg) {
    //Analyze each waveform on the group that matches the set station
    for (int counter = 0; counter < wfg.size(); counter++) {
    //Set current waveforms
    tracebuf = wfg.get(counter);
    //Match requested station
    if (!tracebuf.isSCNL(Sta.stat, Sta.chan, Sta.net, Sta.loc)) {
    //Not the correct station, move to next waveform
    continue;
    }
    //System.out.println(curWave.toString());
    
    //Prepare output waveforms            
    curRdat = rdatGroup.addEmptyWaveForm(Sta.stat, Sta.chan, Sta.net, Sta.loc,
    tracebuf.getStartTime(), tracebuf.getSamplingRate(),
    tracebuf.getSampleCount());
    curChar = charGroup.addEmptyWaveForm(Sta.stat, Sta.chan, Sta.net, Sta.loc,
    tracebuf.getStartTime(), tracebuf.getSamplingRate(),
    tracebuf.getSampleCount());
    curSta = staGroup.addEmptyWaveForm(Sta.stat, Sta.chan, Sta.net, Sta.loc,
    tracebuf.getStartTime(), tracebuf.getSamplingRate(),
    tracebuf.getSampleCount());
    curLta = ltaGroup.addEmptyWaveForm(Sta.stat, Sta.chan, Sta.net, Sta.loc,
    tracebuf.getStartTime(), tracebuf.getSamplingRate(),
    tracebuf.getSampleCount());
    curStaLta = staltaGroup.addEmptyWaveForm(Sta.stat, Sta.chan, Sta.net, Sta.loc,
    tracebuf.getStartTime(), tracebuf.getSamplingRate(),
    tracebuf.getSampleCount());
    
    //Start pick computations
    if (Sta.first==1) {
    //lastSampleTime = (double) tracebuf.getEndTime() / 1000;
    Sta.endtime = (double) tracebuf.getEndTime() / 1000;
    //firstTrace = false;
    Sta.first = 0;
    }
    
    //double GapSizeD = tracebuf.getSamplingRate()
    //        * ((double) tracebuf.getStartTime() / 1000 - lastSampleTime);
    double GapSizeD = tracebuf.getSamplingRate()
     * ((double) tracebuf.getStartTime() / 1000 - Sta.endtime);
    int GapSize = 0;
    if (GapSizeD < 0) { // Overlap - Discard 
    GapSize = 0;
    } else {
    GapSize = (int) (GapSizeD + 0.5);
    }
    // Small gap, interpolate 
    if ((GapSize > 1) && (GapSize <= pickParams.MaxGap)) {
    //TODO: Interpolation
    System.out.println("Interpolation required. Not yet supported. " + GapSize);
    }
    
    // Large gap, anounce restart 
    if (GapSize > pickParams.MaxGap) {
    System.out.println("Large gap detected at: "
    + new SimpleDateFormat("yyyy.MM.dd HH:mm:ss.SSS").format(tracebuf.getStartTime())
    + ". Restarting picker.");
    }
    
    //Save last endtime of this message
    //lastSampleTime = (double) tracebuf.getEndTime() / 1000;
    
    //In restart mode, compute samples without calculating picks
    if (Restart(Sta, tracebuf.getSampleCount(), GapSize)) {
    //System.out.println("Restart cycle: " + tracebuf.nsamples);
    for (int i = 0; i < tracebuf.getSampleCount(); i++) {
    Sample((int)tracebuf.getSamples()[i]);
    }
    } else {
    PickRA();
    }
    
    Sta.enddata = (int)tracebuf.getSamples()[tracebuf.getSampleCount()-1];
    Sta.endtime = (double) tracebuf.getEndTime() / 1000;
    }
    return messages;
    }
    
    //The method to retrieve picks
    private void PickRA() {
    //Process sample
    int event_found;
    int event_active;
    //Reset sample counter 
    sample_index = -1;
    
    //Check if a pick is active
    if ((Sta.pick.status > 0) || (Sta.coda.status > 0)) {
    //It is active
    event_active = EventActive(Sta);
    if (event_active == 1) {
    return; //Event active at end of tracebuf stream
    }
    //Add debug message
    if (event_active <= 0) {
    messages.add(new PickEWDebugMsg((long) (Sta.pick.time * 1000), event_active));
    }
    }
    //Search Mode
    while (true) {
    event_found = ScanForEvent();
    if (event_found == 1) {
    //System.out.println("Event found at " + Sta.pick.time);
    } else {
    //System.out.println("Event not found.");
    return;
    }
    //It is active
    event_active = EventActive(Sta);
    if (event_active == 1) {
    //System.out.println("Event active for next trace.");
    return; //Event active at end of tracebuf stream
    }
    }
    }
    
    //Equivalent to function EventActive
    private int EventActive(PickEWStation Sta) {
    while ((++sample_index) < tracebuf.getSampleCount()) {
    //Start
    int new_sample = (int)tracebuf.getSamples()[sample_index];
    
    //Update sample calculations
    Sample(new_sample);
    
    // BEGIN CODA CALCULATION
    //The CODA is active. Measure its length and amplitudes
    if (Sta.coda.status == 1) {
    int lwindow = (int) (tracebuf.getSamplingRate() + 0.5); //Coda window in samples
    if (Sta.coda.len_win != 72) {
    lwindow *= 2;
    }
    Sta.rsrdat += Math.abs(Sta.rdat);
    
    //Save windows specified in the pwin array
    if (++Sta.ndrt >= lwindow) {
    double ave_abs_val;
    int[] pwin = new int[]{1, 2, 3, 4, 5, 6, 7, 8, 10, 12, 16, 20, 24,
    28, 32, 36, 40, 44, 48, 56, 64, 70, 73};
    
    if (Sta.coda.len_win++ == pwin[Sta.k]) {
    for (int i = 5; i > 0; i--) {
    Sta.coda.aav[i] = Sta.coda.aav[i - 1];
    }
    Sta.k++;
    }
    
    //Compute coda length in seconds
    Sta.coda.len_sec = (2 * Sta.coda.len_win) - 1;
    if (Sta.coda.len_sec == 145) {
    Sta.coda.len_sec = 144;
    }
    
    //Flush pick from buffer if coda is long enough
    if ((Sta.pick.status == 2) && (Sta.coda.len_sec >= Sta.MinCodaLen)) {
    //Main.Debug("Coda->len_sec: " + Coda.len_sec);
    ReportPick();
    //Main.Debug("Pick Detected!");
    Sta.pick.status = 0;
    
    if (pickParams.NoCoda == 1) {
    //Artificially terminate coda calculation
    Sta.coda.status = 0;
    }
    }
    
    //Compute and save average absolute value
    ave_abs_val = Sta.rsrdat / lwindow;
    Sta.coda.aav[0] = (int) (ave_abs_val + 0.5);
    
    //Initialize counter and coda amp sum
    Sta.ndrt = 0;
    Sta.rsrdat = 0;
    
    //See if the coda calculation is over.
    if ((Sta.coda.len_win == 73) || (ave_abs_val < Sta.cocrit)) {
    if (Sta.coda.len_sec < Sta.MinCodaLen) {
    return -1;
    }
    Sta.coda.len_out *= Sta.coda.len_sec;
    Sta.coda.status = 2; //Coda complete but not reported
    }
    }
    }
    
    //The pick has been reported. The coda is complete
    //but has not been reported
    if (Sta.coda.status == 2) {
    if (Sta.pick.status == 0) {
    ReportCoda();
    //Main.Debug("Coda detected!");
    Sta.coda.status = 0;
    }
    if (Sta.pick.status == 2) {
    //Main.Debug("Pick detected!");
    ReportPick();
    ReportCoda();
    //Main.Debug("Coda detected!");
    Sta.pick.status = Sta.coda.status = 0;
    }
    }
    
    //END CODA CALCULATION
    //BEGIN PICK CALCULATION
    //A Pick is active
    if (Sta.pick.status == 1) {
    int i;              //Peak index
    int k;              //Index into sarray
    int noise;          //1 if ievent is noise
    int itrm;           //Number of small counts allowed before pick is declared over
    double xon;         //Used in pick weight calculation
    double xpc;
    double xp0, xp1, xp2;
    
    //Save first 10 points after pick for first motion determination
    if (++Sta.evlen < 10) {
    Sta.sarray[Sta.evlen] = new_sample;
    }
    
    //Store current data if it is a new extreme value
    if (Sta.next < 3) {
    double adata = Math.abs(Sta.rdat);
    if (adata > Sta.tmax) {
    Sta.tmax = adata;
    }
    }
    
    //Test for large zero crossing
    if (Math.abs(Sta.rdat) >= Sta.rbig) {
    if (Sign(Sta.rdat, Sta.rlast) != Sta.rdat) {
    Sta.nzero++;
    Sta.rlast = Sta.rdat;
    //Main.Debug("nzero: " + Sta.nzero);
    }
    }
    
    //Increment ZC crossing interval counter or terminate pick if no ZC have occured
    if (++Sta.mint > (Integer) Sta.MaxMint) {
    //System.out.println("Too many samples without a zero crossing.");
    return -2;
    }
    
    //Test for small ZC
    if (Sign(Sta.rdat, Sta.rold) == Sta.rdat) {
    continue;
    }
    
    //Small ZC found, reset zero crossing interval counter
    Sta.mint = 0;
    
    //Update ecrit and determine whether at this crossing esta
    //is still above ecrit.
    Sta.ecrit += Sta.crtinc;
    Sta.isml++;
    if (Sta.esta > Sta.ecrit) {
    Sta.isml = 0;
    }
    
    //Store extrema of preceeding half cycle
    if (Sta.next < 3) {
    Sta.pick.xpk[Sta.next++] = Sta.tmax;
    
    
    if (Sta.next == 1) {
    double vt3;
    vt3 = Sta.tmax / 3;
    Sta.rbig = (vt3 > Sta.rbig) ? vt3 : Sta.rbig;
    }
    
    Sta.tmax = 0;
    }
    
    //Compute itrm
    itrm = Sta.Itr1 + (Sta.m / Sta.Itr1);
    
    if (Sta.m > 150) {
    itrm = 50;
    }
    
    //See if the pick is over
    if ((++Sta.m != Sta.MinSmallZC) && (Sta.isml < itrm)) {
    continue; //Its not over
    }
    
    //See if the pick was a noise pick. If so, terminate it.
    
    //System.out.println("xpk: "
    //+ (int) (Sta.pick.xpk[0] + 0.5) + " "
    //+ (int) (Sta.pick.xpk[1] + 0.5) + " "
    //+ (int) (Sta.pick.xpk[2] + 0.5)
    //+ "  m: " + Sta.m
    //+ "  nzero: " + Sta.nzero);
    
    noise = 1;
    
    for (i = 0; i < 3; i++) {
    if (Sta.pick.xpk[i] >= Sta.MinPeakSize) {
    if ((Sta.m == Sta.MinSmallZC)
    && (Sta.nzero >= Sta.MinBigZC)) {
    noise = 0;
    }
    break;
    }
    }
    
    if (noise == 1) {
    //System.out.println("Noise Pick");
    return -3;
    }
    
    //A valid pick was found, determine the first motion
    Sta.pick.FirstMotion = ' '; //Unknown
    
    for (k = 0; true; k++) {
    if (Sta.xdot <= 0) {
    if ((Sta.sarray[k + 1] > Sta.sarray[k]) || (k == 8)) {
    if (k == 0) {
    break;
    }
    Sta.pick.FirstMotion = 'D'; //First motion is down
    break;
    }
    } else {
    if ((Sta.sarray[k + 1] < Sta.sarray[k]) || (k == 8)) {
    if (k == 0) {
    break;
    }
    Sta.pick.FirstMotion = 'U'; //First motion is down
    break;
    }
    }
    }
    
    //Pick weight calculation
    xpc = (Sta.pick.xpk[0] > Math.abs((double) Sta.sarray[0])) ? Sta.pick.xpk[0] : Sta.pick.xpk[1];
    xon = Math.abs((double) Sta.xdot / Sta.xfrz);
    xp0 = Sta.pick.xpk[0] / Sta.xfrz;
    xp1 = Sta.pick.xpk[1] / Sta.xfrz;
    xp2 = Sta.pick.xpk[2] / Sta.xfrz;
    
    Sta.pick.weight = 3;
    
    if ((xp0 > 2) && (xon > 0.5) && (xpc > 25)) {
    Sta.pick.weight = 2;
    }
    
    if ((xp0 > 3) && ((xp1 > 3) || (xp2 > 3)) && (xon > 0.5) && (xpc > 100)) {
    Sta.pick.weight = 1;
    }
    
    if ((xp0 > 4) && ((xp1 > 6) || (xp2 > 6)) && (xon > 0.5) && (xpc > 200)) {
    Sta.pick.weight = 0;
    }
    
    Sta.pick.status = 2; //Pick calculated but not reported
    
    //Report pick and coda
    if (Sta.coda.status == 2) {
    //Main.Debug("Pick detected!");
    ReportPick();
    ReportCoda();
    //Main.Debug("Coda detected!");
    Sta.pick.status = Sta.coda.status = 0;
    }
    }
    
    // END PICK CALCULATION
    
    //If the pick is over and coda measurement is done, scan for a new event
    if ((Sta.pick.status == 0) && (Sta.coda.status == 0)) {
    return 0;
    }
    }
    return 1; // Event is still active
    }
    
    //Equivalent to function ScanForEvent
    private int ScanForEvent() {
    //System.out.println("Scanning.");
    
    //Reset pick and coda status
    Sta.pick.status = Sta.coda.status = 0;
    
    //Loop through all available samples
    while ((++sample_index) < tracebuf.getSampleCount()) {
    //Start
    int new_sample = (int)tracebuf.getSamples()[sample_index];
    int old_sample = Sta.old_sample;
    double old_eref = Sta.eref;
    
    //Update sample stuff
    Sample(new_sample);
    
    //Station is assumed dead when eabs>Deadsta
    if (Sta.eabs > Sta.DeadSta) {
    continue;
    }
    
    //Has the sta abruptly increased?
    if (Sta.esta > Sta.eref) {
    int wi; //Window index
    
    //Initialize pick variables
    Sta.pick.time = (double) tracebuf.getStartTime() / 1000 + (double) sample_index / tracebuf.getSamplingRate();
    //System.out.println("Pick time: " + new SimpleDateFormat("yyyyMMddHHmmss.SSS").format((long)(Sta.pick.time*1000)));
    //System.out.println("Pick sample: " + sample_index);
    Sta.coda.len_win = 0; //Coda length in windows
    Sta.coda.len_sec = 0; //Coda length in seconds
    
    
    for (wi = 0; wi < 6; wi++) {
    Sta.coda.aav[wi] = 0;
    }
    
    Sta.coda.aav[0] = 0;
    Sta.coda.aav[1] = 0;
    Sta.coda.aav[2] = 0;
    Sta.coda.aav[3] = 0;
    Sta.coda.aav[4] = 0;
    Sta.coda.aav[5] = 0;
    
    Sta.crtinc = Sta.eref / (Double) Sta.Erefs;
    Sta.ecrit = old_eref;
    Sta.evlen = 0;
    Sta.isml = 0;
    Sta.k = 0;
    Sta.m = 1;
    Sta.mint = 0;
    Sta.ndrt = 0;
    Sta.next = 0;
    Sta.nzero = 0;
    Sta.rlast = Sta.rdat;
    Sta.rsrdat = 0;
    Sta.sarray[0] = new_sample;
    Sta.tmax = Math.abs(Sta.rdat);
    Sta.xfrz = 1.6 * Sta.eabs;
    
    //Compute threshold for big zerocrossings
    Sta.xdot = new_sample - old_sample;
    Sta.rbig = ((Sta.xdot < 0) ? -Sta.xdot : Sta.xdot) / 3;
    Sta.rbig = (Sta.eabs > Sta.rbig) ? Sta.eabs : Sta.rbig;
    
    //Compute cocrit and the sign of coda.len_out
    if (Sta.eabs > ((Double) Sta.AltCoda * (Double) Sta.CodaTerm)) {
    Sta.cocrit = (Double) Sta.PreEvent * Sta.eabs;
    Sta.coda.len_out = -1;
    } else {
    Sta.cocrit = (Double) Sta.CodaTerm;
    Sta.coda.len_out = 1;
    }
    
    Sta.pick.status = Sta.coda.status = 1; //Set pick and coda
    return 1;
    }
    }
    return 0; // Message ended with no event
    }
    
    // Equivalent to function Sample
    private double rdif = 0;
    private double edat = 0;
    
    private void Sample(int LongSample) {
    Sta.rold = Sta.rdat;
    
    Sta.rdat = (Sta.rdat * Sta.RawDataFilt) + (double) (LongSample - Sta.old_sample) + 1.0e-10;
    //currdat.addSample((int) Sta.rdat);
    
    //Compute 1st difference of filtered data
    rdif = Sta.rdat - Sta.rold;
    
    //Store integer data value
    Sta.old_sample = LongSample;
    
    //Compute characteristic funcion
    edat = (Sta.rdat * Sta.rdat) + (Sta.CharFuncFilt * rdif * rdif);
    
    //Compute esta, the short-term average of edat
    Sta.esta += Sta.StaFilt * (edat - Sta.esta);
    //curesta.addSample((int) Sta.esta);
    
    //Compute elta, the long-term average of edat
    Sta.elta += Sta.LtaFilt * (edat - Sta.elta);
    //curelta.addSample((int) Sta.elta);
    
    //Compute eref, the reference level to event checking
    Sta.eref = Sta.elta * Sta.EventThresh;
    
    //System.out.println("Ratio: " + (Sta.esta/Sta.elta));
    
    //Compute eabs, the running mean absolute value of rdat
    Sta.eabs = (Sta.RmavFilt * Sta.eabs)
    + ((1.0 - Sta.RmavFilt) * Math.abs(Sta.rdat));
    
    //Update output waveforms
    curRdat.addSample(Sta.rdat);
    curChar.addSample(edat);
    curSta.addSample(Sta.esta);
    curLta.addSample(Sta.elta);
    curStaLta.addSample(Sta.esta / Sta.elta);
    }
    
    // sign function
    private double Sign(double x, double y) {
    if (x == 0) {
    return 0;
    }
    
    if (x < 0) {
    if (y < 0) {
    return x;
    } else {
    return -x;
    }
    } else {
    if (y < 0) {
    return -x;
    } else {
    return x;
    }
    }
    }
    
    // Restart function (see restart.c)
    private boolean Restart(PickEWStation Sta, int nsamp, int GapSize) {
    if (GapSize > pickParams.MaxGap) {
    Sta.initVar();
    Sta.ns_restart = nsamp;
    return true;
    } else {
    if (Sta.ns_restart >= pickParams.RestartLength) {
    return false;
    } else {
    Sta.ns_restart += nsamp;
    return true;
    }
    }
    }
    
    // Reporting picks
    private static int SeqNo = 0; //Generic sequence number for picker
    
    private void ReportPick() {
    // Round pick arrival time to the 10-th of a second 
    BigDecimal bd = new BigDecimal(Sta.pick.time).setScale(2, BigDecimal.ROUND_HALF_DOWN);
    // Procuce pick_scnl message 
    PickSCNL newPick = new PickSCNL(
    1,
    255,
    SeqNo,
    Sta.getSCNLStr(),
    Sta.pick.FirstMotion,
    Sta.pick.weight,
    bd.doubleValue(),
    new int[]{(int) (Sta.pick.xpk[0] + 0.5), (int) (Sta.pick.xpk[1] + 0.5), (int) (Sta.pick.xpk[2] + 0.5)},
    (double) tracebuf.getStartTime() / 1000 + (double) sample_index / tracebuf.getSampleCount());
    messages.add(newPick);
    //System.out.println(newPick.toString());
    }
    
    private void ReportCoda() {
    CodaSCNL newCoda = new CodaSCNL(
    1,
    255,
    SeqNo++, //Same seqno has last pick
    Sta.getSCNLStr(),
    new int[]{Sta.coda.aav[0], Sta.coda.aav[1], Sta.coda.aav[2], Sta.coda.aav[3], Sta.coda.aav[4], Sta.coda.aav[5]},
    Sta.coda.len_out,
    (double) tracebuf.getStartTime() / 1000 + (double) sample_index / tracebuf.getSampleCount());
    messages.add(newCoda);
    //System.out.println(newCoda.toString());
    }
    // Output data handling 
    public EWMessageGroup messages = new EWMessageGroup();
    
    // Parameters to compute picks 
    //private double lastSampleTime = Double.NEGATIVE_INFINITY; //Acquisition time of the last sample
    private int sample_index; //Sample index
    //private boolean firstTrace = true;
    
    // Objects for handling trace and station data 
    private WaveForm tracebuf; //Currently used tracebuf
    private WaveForm curRdat;
    private WaveForm curChar;
    private WaveForm curLta;
    private WaveForm curSta;
    private WaveForm curStaLta;
    private PickEWStation Sta; //Same as STATION from pick_ew code
    
    // pick_ew parameters 
    private EWPickerParameters pickParams;
    //Handle output waveforms for analysis
    public WaveFormGroup rdatGroup = new WaveFormGroup();
    public WaveFormGroup charGroup = new WaveFormGroup();
    public WaveFormGroup ltaGroup = new WaveFormGroup();
    public WaveFormGroup staGroup = new WaveFormGroup();
    public WaveFormGroup staltaGroup = new WaveFormGroup();
     */
}