package pickewanalysis;

import java.awt.Color;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;
import javax.swing.JFileChooser;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import pickewanalysis.ewmessages.EWMessageGroup;
import pickewanalysis.ewpicker.EWPicker;
import pickewanalysis.ewpicker.EWPickerParameters;
import pickewanalysis.ewpicker.PickEWStation;
import pickewanalysis.ewpicker.PickEWStationGroup;
import pickewanalysis.waveform.WaveFormGroup;
import pickewanalysis.wavesources.WaveSource;
import pickewanalysis.wavesources.WaveSourceStation;

public class PickEWAnalysis extends javax.swing.JFrame {

    /** Creates new form PickEWAnalysis */
    public PickEWAnalysis() {
        initComponents();
    }

    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jScrollPane1 = new javax.swing.JScrollPane();
        StationsList = new javax.swing.JTable();
        jScrollPane2 = new javax.swing.JScrollPane();
        StationParametersTable = new javax.swing.JTable();
        jScrollPane3 = new javax.swing.JScrollPane();
        PreferencesTable = new javax.swing.JTable();
        jLabel1 = new javax.swing.JLabel();
        starttimeField = new javax.swing.JTextField();
        jLabel2 = new javax.swing.JLabel();
        endtimeField = new javax.swing.JTextField();
        setTimeButton = new javax.swing.JButton();
        leftButton = new javax.swing.JButton();
        rightButtom = new javax.swing.JButton();
        zoomOutButton = new javax.swing.JButton();
        zoomInButton = new javax.swing.JButton();
        centerButton = new javax.swing.JButton();
        tracePanel = new pickewanalysis.TracePanel();
        menuBar = new javax.swing.JMenuBar();
        fileMenu = new javax.swing.JMenu();
        openMenuItem = new javax.swing.JMenuItem();
        saveAsMenuItem = new javax.swing.JMenuItem();
        exitMenuItem = new javax.swing.JMenuItem();
        editMenu = new javax.swing.JMenu();
        settingsMenuItem = new javax.swing.JMenuItem();
        filterMenuItem = new javax.swing.JMenuItem();
        helpMenu = new javax.swing.JMenu();
        aboutMenuItem = new javax.swing.JMenuItem();

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
        setTitle("PickEWAnalysis");

        StationsList.setAutoCreateRowSorter(true);
        StationsList.setFont(new java.awt.Font("Tahoma", 0, 10));
        StationsList.setModel(new StationsListTableModel("", null));
        StationsList.setAutoResizeMode(javax.swing.JTable.AUTO_RESIZE_OFF);
        StationsList.setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
        StationsList.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
            public void valueChanged(ListSelectionEvent e)
            {
                stationSelected();
            }
        });
        jScrollPane1.setViewportView(StationsList);

        StationParametersTable.setFont(new java.awt.Font("Tahoma", 0, 10));
        StationParametersTable.setModel(new StationParametersTableModel(null, null));
        jScrollPane2.setViewportView(StationParametersTable);

        PreferencesTable.setFont(new java.awt.Font("Tahoma", 0, 10));
        PreferencesTable.setModel(new PreferencesTableModel(config, this));
        jScrollPane3.setViewportView(PreferencesTable);

        jLabel1.setFont(new java.awt.Font("Tahoma", 0, 10));
        jLabel1.setText("Start Time");

        starttimeField.setFont(new java.awt.Font("Tahoma", 0, 10));
        starttimeField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                starttimeFieldKeyReleased(evt);
            }
        });

        jLabel2.setFont(new java.awt.Font("Tahoma", 0, 10));
        jLabel2.setText("End Time");

        endtimeField.setFont(new java.awt.Font("Tahoma", 0, 10));
        endtimeField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                endtimeFieldKeyReleased(evt);
            }
        });

        setTimeButton.setFont(new java.awt.Font("Tahoma", 0, 10));
        setTimeButton.setText("Set From Source");
        setTimeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                setTimeButtonActionPerformed(evt);
            }
        });

        leftButton.setFont(new java.awt.Font("Tahoma", 0, 8));
        leftButton.setText("<");
        leftButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                leftButtonActionPerformed(evt);
            }
        });

        rightButtom.setFont(new java.awt.Font("Tahoma", 0, 8));
        rightButtom.setText(">");
        rightButtom.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                rightButtomActionPerformed(evt);
            }
        });

        zoomOutButton.setFont(new java.awt.Font("Tahoma", 0, 8));
        zoomOutButton.setText("<>");
        zoomOutButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                zoomOutButtonActionPerformed(evt);
            }
        });

        zoomInButton.setFont(new java.awt.Font("Tahoma", 0, 8));
        zoomInButton.setText("><");
        zoomInButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                zoomInButtonActionPerformed(evt);
            }
        });

        centerButton.setFont(new java.awt.Font("Tahoma", 0, 8));
        centerButton.setText("O");
        centerButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                centerButtonActionPerformed(evt);
            }
        });

        tracePanel.setBackground(new java.awt.Color(255, 255, 255));

        javax.swing.GroupLayout tracePanelLayout = new javax.swing.GroupLayout(tracePanel);
        tracePanel.setLayout(tracePanelLayout);
        tracePanelLayout.setHorizontalGroup(
            tracePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 508, Short.MAX_VALUE)
        );
        tracePanelLayout.setVerticalGroup(
            tracePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 495, Short.MAX_VALUE)
        );

        fileMenu.setMnemonic('f');
        fileMenu.setText("File");

        openMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_O, java.awt.event.InputEvent.CTRL_MASK));
        openMenuItem.setText("Open Stations File ...");
        openMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                openMenuItemActionPerformed(evt);
            }
        });
        fileMenu.add(openMenuItem);

        saveAsMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_S, java.awt.event.InputEvent.CTRL_MASK));
        saveAsMenuItem.setMnemonic('a');
        saveAsMenuItem.setText("Save Stations File As ...");
        saveAsMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                saveAsMenuItemActionPerformed(evt);
            }
        });
        fileMenu.add(saveAsMenuItem);

        exitMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_Q, java.awt.event.InputEvent.CTRL_MASK));
        exitMenuItem.setMnemonic('x');
        exitMenuItem.setText("Exit");
        exitMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                exitMenuItemActionPerformed(evt);
            }
        });
        fileMenu.add(exitMenuItem);

        menuBar.add(fileMenu);

        editMenu.setMnemonic('e');
        editMenu.setText("Edit");

        settingsMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_E, java.awt.event.InputEvent.CTRL_MASK));
        settingsMenuItem.setMnemonic('t');
        settingsMenuItem.setText("Settings");
        settingsMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                settingsMenuItemActionPerformed(evt);
            }
        });
        editMenu.add(settingsMenuItem);

        filterMenuItem.setText("Filter Parameter Tool");
        filterMenuItem.setEnabled(false);
        editMenu.add(filterMenuItem);

        menuBar.add(editMenu);

        helpMenu.setMnemonic('h');
        helpMenu.setText("Help");

        aboutMenuItem.setMnemonic('a');
        aboutMenuItem.setText("About");
        aboutMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                aboutMenuItemActionPerformed(evt);
            }
        });
        helpMenu.add(aboutMenuItem);

        menuBar.add(helpMenu);

        setJMenuBar(menuBar);

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jScrollPane1, javax.swing.GroupLayout.PREFERRED_SIZE, 144, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addComponent(setTimeButton, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(jScrollPane2, javax.swing.GroupLayout.DEFAULT_SIZE, 129, Short.MAX_VALUE)
                    .addComponent(jScrollPane3, javax.swing.GroupLayout.DEFAULT_SIZE, 129, Short.MAX_VALUE)
                    .addComponent(jLabel1)
                    .addComponent(starttimeField)
                    .addComponent(jLabel2)
                    .addComponent(endtimeField))
                .addGap(18, 18, 18)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createSequentialGroup()
                        .addComponent(leftButton, javax.swing.GroupLayout.PREFERRED_SIZE, 50, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(rightButtom, javax.swing.GroupLayout.PREFERRED_SIZE, 50, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(zoomOutButton, javax.swing.GroupLayout.PREFERRED_SIZE, 50, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(zoomInButton, javax.swing.GroupLayout.PREFERRED_SIZE, 50, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(centerButton, javax.swing.GroupLayout.PREFERRED_SIZE, 50, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addComponent(tracePanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createSequentialGroup()
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                            .addComponent(leftButton, javax.swing.GroupLayout.PREFERRED_SIZE, 20, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(rightButtom, javax.swing.GroupLayout.PREFERRED_SIZE, 20, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(zoomOutButton, javax.swing.GroupLayout.PREFERRED_SIZE, 20, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(zoomInButton, javax.swing.GroupLayout.PREFERRED_SIZE, 20, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(centerButton, javax.swing.GroupLayout.PREFERRED_SIZE, 20, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(tracePanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                    .addComponent(jScrollPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 521, Short.MAX_VALUE)
                    .addGroup(layout.createSequentialGroup()
                        .addComponent(jScrollPane2, javax.swing.GroupLayout.PREFERRED_SIZE, 309, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                        .addComponent(jScrollPane3, javax.swing.GroupLayout.PREFERRED_SIZE, 68, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                        .addComponent(jLabel1)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(starttimeField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                        .addComponent(jLabel2)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(endtimeField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(14, 14, 14)
                        .addComponent(setTimeButton)))
                .addContainerGap())
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void exitMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_exitMenuItemActionPerformed
        System.exit(0);
    }//GEN-LAST:event_exitMenuItemActionPerformed

    private void settingsMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_settingsMenuItemActionPerformed
        config.setLocationRelativeTo(this);
        config.setVisible(true);
        update();
    }//GEN-LAST:event_settingsMenuItemActionPerformed

    private void starttimeFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_starttimeFieldKeyReleased
        try {
            config.starttime = df.parse(starttimeField.getText().trim());
            viewStart = config.starttime.getTime();
            starttimeField.setForeground(Color.BLACK);
            if (checkTimeConsistency()) {
                updateWaveData();
            }
        } catch (Exception e) {
            starttimeField.setForeground(Color.RED);
        }

    }//GEN-LAST:event_starttimeFieldKeyReleased

    private void endtimeFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_endtimeFieldKeyReleased
        try {
            config.endtime = df.parse(endtimeField.getText().trim());
            viewEnd = config.endtime.getTime();
            endtimeField.setForeground(Color.BLACK);
            if (checkTimeConsistency()) {
                updateWaveData();
            }
        } catch (Exception e) {
            endtimeField.setForeground(Color.RED);
        }
    }//GEN-LAST:event_endtimeFieldKeyReleased

    private void setTimeButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_setTimeButtonActionPerformed
        //First try the default times
        config.endtime = new Date((long) (System.currentTimeMillis() / 1000) * 1000);
        config.starttime = new Date(config.endtime.getTime() - 60000);
        //Get the currently selected station
        if (curStation == null) {
            //Not available... leave with the default times
            update();
            return;
        }
        //Cycle though sources to try and find the first which contains this station
        for (WaveSource source : config.sources) {
            WaveSourceStation station = source.findStation(curStation.stat, curStation.chan,
                    curStation.net, curStation.loc);
            
            if (station == null) {
                continue;
            }
            long st = station.getStartTime();
            long et = station.getEndTime();
            long delta = et - st;
            if (delta < 0) {
                //Something is wrong with this tation... discard it
                continue;
            }
            if (delta > MAXTIME) {
                //Reset starttime to fit maximum time
                st = et - MAXTIME;
            }
            //Set new configuration times
            config.starttime = new Date(st);
            config.endtime = new Date(et);
            //Set new view times
            this.viewStart = st;
            this.viewEnd = et;
            //Update everything
            update();
        }
    }//GEN-LAST:event_setTimeButtonActionPerformed

    private void leftButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_leftButtonActionPerformed
        //Check if already at the left of the available window
        if (viewStart <= config.starttime.getTime()) {
            return;
        }
        long shiftSpan = (long) ((double) (viewEnd - viewStart) * 0.2 + 0.5);
        long newStart = viewStart - shiftSpan;
        if (newStart < config.starttime.getTime()) {
            newStart = config.starttime.getTime();
        }
        long delta = viewStart - newStart;
        viewStart = newStart;
        viewEnd -= delta;
        updateTracePanel();
    }//GEN-LAST:event_leftButtonActionPerformed

    private void rightButtomActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_rightButtomActionPerformed
        //Check if already at the right of the available window
        if (viewEnd >= config.endtime.getTime()) {
            return;
        }
        long shiftSpan = (long) ((double) (viewEnd - viewStart) * 0.2 + 0.5);
        long newEnd = viewEnd + shiftSpan;
        if (newEnd > config.endtime.getTime()) {
            newEnd = config.endtime.getTime();
        }
        long delta = viewEnd - newEnd;
        viewEnd = newEnd;
        viewStart -= delta;
        updateTracePanel();
    }//GEN-LAST:event_rightButtomActionPerformed

    private void zoomOutButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_zoomOutButtonActionPerformed
        zoom(1.2);
    }//GEN-LAST:event_zoomOutButtonActionPerformed

    private void zoomInButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_zoomInButtonActionPerformed
        zoom(1.0 / 1.2);
    }//GEN-LAST:event_zoomInButtonActionPerformed

    private void centerButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_centerButtonActionPerformed
        viewStart = config.starttime.getTime();
        viewEnd = config.endtime.getTime();
        updateTracePanel();
    }//GEN-LAST:event_centerButtonActionPerformed

    private void saveAsMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_saveAsMenuItemActionPerformed
        //Save stations file
        PickEWStationGroup stations = ((StationsListTableModel) StationsList.getModel()).getStations();
        if (stations == null || stations.isEmpty()) {
            System.err.println("No stations available for saving.");
        }
        JFileChooser filechooser = new JFileChooser();
        filechooser.setCurrentDirectory(new File("./"));
        int aux = filechooser.showSaveDialog(null);
        if (aux == JFileChooser.APPROVE_OPTION) {
            File selectedfile = filechooser.getSelectedFile();
            if (selectedfile != null) {

                try {
                    BufferedWriter Foutput = new BufferedWriter(new FileWriter(selectedfile));
                    Foutput.write("#####################################################");
                    Foutput.newLine();
                    Foutput.write("#  Pick_EW Stations File created automaticaly.      #");
                    Foutput.newLine();
                    Foutput.write("#  Date: " + new Date().toString() + "             #");
                    Foutput.newLine();
                    Foutput.write("#####################################################");
                    Foutput.newLine();
                    for (PickEWStation station : stations) {
                        System.out.println(station.toStationLine());
                        Foutput.write(station.toStationLine());
                        Foutput.newLine();
                    }
                    Foutput.flush();
                    Foutput.close();
                } catch (Exception e) {
                    //Unable to save file
                    System.err.println("Unable to save stations file.");
                }
            }
        }

    }//GEN-LAST:event_saveAsMenuItemActionPerformed

    private void aboutMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_aboutMenuItemActionPerformed
        new AboutDialog(this, true).setVisible(true);
    }//GEN-LAST:event_aboutMenuItemActionPerformed

    private void openMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_openMenuItemActionPerformed
        JFileChooser filechooser = new JFileChooser();
        filechooser.setCurrentDirectory(new File("./"));
        int aux = filechooser.showOpenDialog(null);
        if (aux == JFileChooser.APPROVE_OPTION) {
            File selectedfile = filechooser.getSelectedFile();
            if (selectedfile != null) {
                config.StationsFile = selectedfile.getAbsolutePath();
                try {
                    config.save(config.configFile);
                } catch (Exception e) {
                    e.printStackTrace();
                    //Do nothing
                }
            }
            update();
        }
    }//GEN-LAST:event_openMenuItemActionPerformed

    public static void main(final String args[]) {
        //Set UTC Time
        System.setProperty("user.timezone", "Etc/Universal");
        TimeZone.setDefault(TimeZone.getTimeZone("UTC"));

        //Start GUI
        java.awt.EventQueue.invokeLater(new Runnable() {

            public void run() {
                //Create main window
                PickEWAnalysis mainWindow = new PickEWAnalysis();
                //Load configuration
                configFileName = DEFAULT_CONFIG_FILENAME;
                if (args.length > 0) {
                    configFileName = args[0];
                }
                try {
                    mainWindow.config = Configuration.load(configFileName);
                } catch (Exception e) {
                    System.err.println("Unable to open configuration file");
                    e.printStackTrace();
                    System.err.println("Starting default configuration.");
                    mainWindow.config = new Configuration();
                    try {
                        mainWindow.config.save(configFileName);
                    } catch (Exception ex) {
                        System.err.println("Unable to save new configuration file.");
                        System.err.println("Settings will not be saved");
                    }
                }

                //Update everything
                mainWindow.update();
                //Center on screen
                mainWindow.setLocationRelativeTo(null);
                //Start
                mainWindow.setVisible(true);
            }
        });
    }

    //Method to update the contents of the main window
    //It uses the smaller update methods to rebuild the entire main window
    private void update() {
        int rowIndex = StationsList.getSelectedRow();
        updateStationsList();
        updateParametersTable();
        updatePreferencesTable();
        updateTimeFields();
        if (rowIndex != -1) {
            StationsList.setRowSelectionInterval(rowIndex, rowIndex);
        }
        StationsList.getColumnModel().getColumn(0).setPreferredWidth(110);
        StationsList.getColumnModel().getColumn(1).setPreferredWidth(10);
        updateWaveData();
    }

    private boolean updateStationsList() {
        StationsList.setModel(new StationsListTableModel(config.StationsFile,
                config));
        return true;
    }

    private boolean updateParametersTable() {
        StationParametersTable.setModel(new StationParametersTableModel(curStation, this));
        return true;
    }

    private boolean updatePreferencesTable() {
        PreferencesTable.setModel(new PreferencesTableModel(config, this));
        return true;
    }

    private boolean updateTimeFields() {
        starttimeField.setText(df.format(config.starttime));
        endtimeField.setText(df.format(config.endtime));
        checkTimeConsistency();
        return true;
    }

    private boolean updateWaveData() {
        //Basic chechs
        if (curStation == null) {
            return false;
        }
        //Clear current group
        curGroup = null;
        for (WaveSource source : config.sources) {
            if (!source.isSCNL(curStation.stat, curStation.chan,
                    curStation.net, curStation.loc,
                    config.starttime.getTime(), config.endtime.getTime())) {
                continue;
            }
            //Found a source, read data
            try {
                curGroup = source.read(curStation.stat, curStation.chan,
                        curStation.net, curStation.loc,
                        config.starttime.getTime(), config.endtime.getTime());
                if (curGroup == null) {
                    continue;
                }
            } catch (Exception e) {
                //Do nothing... move to next
                new ErrorDialog("Error reading data: " + e.getMessage(), "Error", this, true).setVisible(true);
                continue;
            }
            if (curGroup == null) {
                new ErrorDialog("Unable to load data.", "Error", this, true).setVisible(true);
            }
        }
        return updateTracePanel();
    }

    public boolean updateTracePanel() {
        if (curGroup == null) {
            //No available data
            tracePanel.update(viewStart, viewEnd, curStation, null, null);
            return false;
        };
        //Data is available. Pick it
        //Create pick parameters object
        EWPickerParameters pickerParams = new EWPickerParameters(
                config.RestartLength, config.MaxGap, config.NoCoda);
        //Create pick object
        EWPicker picker = new EWPicker(curStation, pickerParams);

        //Analyze
        EWMessageGroup mg = picker.analyze(curGroup);
        //For debuging
        /*
        System.out.println("Pick Messages");
        for (EWMessage m:mg) {
            if (m.getType()>0) {
                System.out.println(m.toString());
            }
        }
         * 
         */

        //Build waveform array
        WaveFormGroup[] groups = new WaveFormGroup[]{
            curGroup,
            picker.rdatGroup,
            picker.charGroup,
            picker.staGroup,
            picker.ltaGroup,
            picker.staltaGroup};

        //Send update to panel
        tracePanel.update(viewStart, viewEnd, curStation, groups, mg);
        return false;
    }

    private boolean checkTimeConsistency() {
        long delta = config.endtime.getTime() - config.starttime.getTime();
        boolean output = true;
        if (delta <= 0) {
            //System.out.println("Negative time span.");
            output = false;
        }
        if (delta > MAXTIME) {
            //System.out.println("Time span exceeds " + (int)(MAXTIME/1000) + " seconds.");
            output = false;
        }
        if (output) {
            //System.out.println("Time span accepted. " + (delta/1000));
            starttimeField.setForeground(Color.BLACK);
            endtimeField.setForeground(Color.BLACK);
        } else {
            starttimeField.setForeground(Color.RED);
            endtimeField.setForeground(Color.RED);
        }
        return output;
    }

    public void zoom(double k) {
        long curSpan = viewEnd - viewStart;
        double newSpan = (double) curSpan * k;
        double center = (viewEnd + viewStart) / 2;
        double newViewStart = center - newSpan / 2;
        double newViewEnd = center + newSpan / 2;
        if (newViewStart <= config.starttime.getTime()) {
            viewStart = config.starttime.getTime();
        } else {
            viewStart = (long) (newViewStart + 0.5);
        }
        if (newViewEnd >= config.endtime.getTime()) {
            viewEnd = config.endtime.getTime();
        } else {
            viewEnd = (long) (newViewEnd + 0.5);
        }
        updateTracePanel();
    }

    public void zoom(long startT, long endT) {
        if (startT < config.starttime.getTime()) {
            viewStart = config.starttime.getTime();
        } else {
            viewStart = startT;
        }
        if (endT > config.endtime.getTime()) {
            viewEnd = config.endtime.getTime();
        } else {
            viewEnd = endT;
        }
        updateTracePanel();
    }

    private void stationSelected() {
        //Set the current station
        int selectedRow = StationsList.getSelectedRow();
        if (selectedRow==-1) {
            //There is no selected row
            return;
        }
        int rowIndex = StationsList.convertRowIndexToModel(selectedRow);
        curStation = (PickEWStation) ((StationsListTableModel) StationsList.getModel()).getValueAt(rowIndex, 0);
        //System.out.println("Selected Station: " + curStation.toString());

        //Set viewing times
        if (viewStart == 0 || viewEnd == 0) {
            viewStart = config.starttime.getTime();
            viewEnd = config.endtime.getTime();
        }

        //Update parameters table
        updateParametersTable();

        //Update data and trace panel
        updateWaveData();
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JTable PreferencesTable;
    private javax.swing.JTable StationParametersTable;
    private javax.swing.JTable StationsList;
    private javax.swing.JMenuItem aboutMenuItem;
    private javax.swing.JButton centerButton;
    private javax.swing.JMenu editMenu;
    private javax.swing.JTextField endtimeField;
    private javax.swing.JMenuItem exitMenuItem;
    private javax.swing.JMenu fileMenu;
    private javax.swing.JMenuItem filterMenuItem;
    private javax.swing.JMenu helpMenu;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JScrollPane jScrollPane2;
    private javax.swing.JScrollPane jScrollPane3;
    private javax.swing.JButton leftButton;
    private javax.swing.JMenuBar menuBar;
    private javax.swing.JMenuItem openMenuItem;
    private javax.swing.JButton rightButtom;
    private javax.swing.JMenuItem saveAsMenuItem;
    private javax.swing.JButton setTimeButton;
    private javax.swing.JMenuItem settingsMenuItem;
    private javax.swing.JTextField starttimeField;
    private pickewanalysis.TracePanel tracePanel;
    private javax.swing.JButton zoomInButton;
    private javax.swing.JButton zoomOutButton;
    // End of variables declaration//GEN-END:variables
    //Custom Static Variables
    static String configFileName;
    //Custom Variables
    Configuration config; //The configuration
    PickEWStation curStation; //The current station
    //WaveSource curSource;
    WaveFormGroup curGroup;
    long viewStart, viewEnd;
    //Custom Constants
    public static final String DEFAULT_CONFIG_FILENAME = "PickEWAnalysisConfig.xml";
    public static final SimpleDateFormat df = new SimpleDateFormat("yyyyMMddHHmmss");
    public static final long MAXTIME = 600000; //Maximum time in miliseconds
    public static final long MARGIN = 15000; //Margin for picker
}
