package edu.iris.Fissures.seed.director;

import edu.iris.Fissures.seed.builder.*;
import edu.iris.Fissures.seed.exception.*;
import edu.iris.Fissures.seed.container.*;

/**
 * Abstract class representing all Director classes for exporting data 
 * from an Object Container to any number of output formats, as dictated
 * by the Builder Class registered with it.
 *
 * @author Robert Casey
 * @version 4/30/2002
 */

public abstract class ExportDirector {
	
	/**
	 * Instantiate without assigning a Builder or Container to it.
	 */
	public ExportDirector () {
	}
	
	/**
	 * Assign a Builder to the Director.
	 */
	public ExportDirector (ExportBuilder builder) {
		assignBuilder(builder);
	}
	
	/**
	 * Assign a Container and Builder to the Director.
	 */
	public ExportDirector (ObjectContainer container, ExportBuilder builder) {
		assignContainer(container);
		assignBuilder(builder);
	}
	
	/**
	 * Assign a Container, Builder, and Template to the Director.
	 */
	public ExportDirector (ObjectContainer container, ExportBuilder builder, ExportTemplate expTemplate) {
		assignContainer(container);
		assignBuilder(builder);
		assignTemplate(expTemplate);
	}
	
	// public methods
	
	/**
	 * Assign a builder object to this director.
	 */
	public void assignBuilder (ExportBuilder builder) {
		this.builder = builder;
	}
	
	/**
	 * Assign a container object to this director to read objects from.
	 */
	public void assignContainer (ObjectContainer container) {
		this.container = container;
	}
	
	/**
	 * Assign optional export template to the director.
	 */
	public void assignTemplate (ExportTemplate expTemplate) {
		this.expTemplate = expTemplate;
	}
	
	/**
	 * Signal to begin sending build commands to the Builder for a
	 * single output volume.
	 * Returns the number of logical records generated.
	 */
	public int construct () throws Exception {
		if (container == null) throw new BuilderException ("Container not assigned");
		if (builder == null) throw new BuilderException ("Builder not assigned");
		if (expTemplate == null) throw new BuilderException ("Template not assigned");
		//
		int recCount = 0;   // count the number of logical records generated by the builder
		startVolume(); // do any director initializations necessary to start a new volume
		builder.startVolume(); // signal the builder to begin a new volume
		//
		int refNum = -1;
		while (true) {
			// get next builder object requirement, in the form of an object type number.
			// the series of object types is scripted in the builder, which also contains loops.
			//System.err.println("DEBUG: calling builder.getNext()");
			int objType = builder.getNext();
			//System.err.println("DEBUG: builder.getNext() gets type: " + objType);
			// if objType is -1, then the builder requires no more objects, the sequence is complete
			if (objType == -1) break;  // no more object requirements
			if (objType == -2) {
				expTemplate.reset();  // end of pass signal...reset template indices to 1
				continue;  // next in while loop
			}
			// track parent-child context and possibly modify the object type number to include
			// parent reference.
			long newObjType = getContext(objType);
			//System.err.println("DEBUG: calling expTemplate.getNext(" + objType + ")");
			refNum = expTemplate.getNext(newObjType);  // get the next refNum in the template sequence, 0 means group break
			Object nextObject = null;
			try {
                                // this is not called if a group (loop) break -- resulting in nextObject being null
				if (refNum > 0) nextObject = getObject(refNum);   // get the object corresponding to that refNum
			} catch(BuilderException e) {
				// this exception is thrown when we are unable to get the
				// object...let's point out the object type referred to
				System.err.println("Exception: " + e);
				//
				// temporary: dump the contents of the template to stderr
				expTemplate.debugDump();
				// throw the exception
				throw new BuilderException("Called for object refNum " + refNum + " of type " + objType + " converted to new object type " + newObjType);
			}
			//
			// give the builder the next object to build with...null for nextObject is permissible,
			// signalling to the builder that there are no other objects of the requested type available.
			// builder returns the number of *new* logical records it has constructed for this call.
			//
			// DEBUG:
			//if (nextObject != null) {
				//System.err.println("DEBUG: call builder.build(" + ((Blockette)nextObject).getLookupId() + ")");
			//}
			// END DEBUG
			//
                        // a null nextObject signifies a group break, and the builder.build call will ensure script loop exit
			recCount += builder.build(nextObject);  // the value returned can sometimes be zero
			//
		}
		builder.finish();   // finally, the builder is exhausted of requirements...tell the builder to complete its tasks
		return recCount++;  // return the total number of logical records reported by the builder
	}
	
	// abstract methods
	
	/**
	 * Using the provided BuilderFilter, pick out matching blockette objects 
	 * from the ObjectContainer to populate the ExportTemplate.
	 * This process must be implemented by the concrete ExportDirector so
	 * as to query for only relevant object types.
	 */
	public abstract void fillTemplate(BuilderFilter templateFilter) throws Exception;
	
	/**
	 * Signal concrete director class that we are starting a new volume.
	 * Any concrete-specific initialization is taken care of here.
	 */
	protected abstract void startVolume();
	
	/**
	 * Get the object corresponding to refNum.  The returned object should be a
	 * copy of the one in the container, such that modifying its contents will not
	 * alter the contents of the object in the container.
	 * Concrete class will know how to access type-specific object-base.
	 */
	protected abstract Object getObject(int refNum) throws Exception;
	
	/**
	 * Get context ID.
	 * Get a long integer that places a certain object type in a family of objects
	 * that points back to its parent ID.  The context of an object is it's
	 * specific parent and the order it appears inside the parent in relation to
	 * other objects of the same type also inside the parent.  Concrete class
	 * implementations may vary.
	 */
	protected abstract long getContext(int objType) throws Exception;
	
	// instance variables
	
	protected ExportBuilder builder = null;      // handle to assigned Builder
	protected ObjectContainer container = null;  // handle to assigned Container
	protected ExportTemplate expTemplate = null;    // handle to assigned Template
	
}
