/**EWQuakeAlert generates email messages when receiving HYP2000ARC
 * messages from Earthworm.
 * Copyright (C) 2009 Ruben S. Luis
 * This program is free software: you ca redistribute it an/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be usefull,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with EWQuakeAlert. If not, see <http://www.gnu.org/licenses/>.
 **/
package ewquakealert;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.Vector;

/**
 *
 * @author RL197718
 */
public class EWMessage {
    int instalationID, moduleID, messageType;

    //Variables for TYPE_HYPO2000ARC messages
    String[] phasesStr;
    String hypOrigin, eventID;
    double hypLatitude, hypLongitude, hypDepth, hypMag;
    Vector<Phase> phases;

    //Generic
    String filename="";

    EWMessage(String rawmsg) {
        //Check if rawmsg is valid
        this.instalationID = Integer.valueOf(rawmsg.substring(0,3).trim());
        this.moduleID = Integer.valueOf(rawmsg.substring(3,6).trim());
        this.messageType = Integer.valueOf(rawmsg.substring(6,9).trim());

        switch (messageType) {
        case 3:
            //TYPE_HEARTBEAT
            break;
        case 14:
            //TYPE_HYP2000ARC
            process_HYP2000ARC(rawmsg);
            break;
        }
    } //end construct


    private void process_HYP2000ARC(String rawmsg) {
        //Fill in message data
        String newline = System.getProperty("line.separator");
        String[] lines = rawmsg.split("[" + newline + "]+");
        //for (int i=0; i<lines.length; i++) System.out.println(lines[i]);

        //hypocenter
        String hyp = lines[0].substring(9,lines[0].length());
        //System.out.println("<<" + hyp + ">>");

        String hypDate = hyp.substring(0,8);
        String hypTime = hyp.substring(8,16);

        hypOrigin = hypDate.substring(0,4) + "." +
            hypDate.substring(4,6) + "." +
            hypDate.substring(6,8) + " " +
            hypTime.substring(0,2) + ":" +
            hypTime.substring(2,4) + ":" +
            hypTime.substring(4,6) + "." +
            hypTime.substring(6,8);

        String hypLat_deg = hyp.substring(16,18);
        String hypLat_Pos = hyp.substring(18,19);
        String hypLat_min = hyp.substring(19,23);

        hypLatitude = Double.valueOf(hypLat_deg.trim()) + Double.valueOf(hypLat_min.trim())/100/60;
        if (hypLat_Pos.equals("S")) hypLatitude = -hypLatitude;

        String hypLon_deg = hyp.substring(23,26);
        String hypLon_Pos = hyp.substring(26,27);
        String hypLon_min = hyp.substring(27,31);

        hypLongitude = -(Double.valueOf(hypLon_deg.trim()) + Double.valueOf(hypLon_min.trim())/100/60);
        if (hypLon_Pos.equals("E")) hypLongitude = -hypLongitude;

        hypDepth = Double.valueOf(hyp.substring(31,36).trim())/100;
        hypMag = Double.valueOf(hyp.substring(147,150).trim())/100;

        eventID = hyp.substring(136,146).trim();

        //phases
        phasesStr = new String[(lines.length-4)/2];
        for (int i=0; i<phasesStr.length; i++) phasesStr[i] = lines[(i+1)*2];
        phases = new Vector<Phase>();
        for (int i=0; i<phasesStr.length; i++) addStation(phasesStr[i]);




    } //end process_HYP2000ARC method


    public boolean sendEWMsgEmail() {
        boolean output = false;

        NumberFormat nf = NumberFormat.getNumberInstance(new Locale("en", "US"));
        DecimalFormat df = (DecimalFormat)nf;
        df.applyPattern("#0.000");


        String subject = "Earthquake Notification ID: " + eventID +
                    " - Magnitude: " + hypMag ;
        
        if ((!Configuration.sendPicks)&&(!Configuration.useWaveServer)) {
            //Use only text email
            String newline = System.getProperty("line.separator");

            

            String msgBody =
                "Origin time: " + hypOrigin + newline +
                "Latitude: " + df.format(hypLatitude) + "º" + newline +
                "Longitude: " + df.format(hypLongitude) + "º" + newline +
                "Depth: " + hypDepth + " km" + newline +
                "Magnitude: " + hypMag + newline + newline + newline + newline +
                "Picks:" + newline;

            for (int i=0; i<phasesStr.length; i++) msgBody = msgBody.concat(phasesStr[i] + newline);

            sendEmail x = new sendEmail(Configuration.destEmail,subject,msgBody);
            x.start();
        }
        else {
            //Use html email
            String msgBody = "<html><body>" +
                    "<p><font size=\"2\" face=\"courier\">";
            //Include google static map
            if (Configuration.sendPicks)
                msgBody = msgBody +
                        "<img src=\"" +
                        "http://www.google.com/staticmap?" +
                        "center=" + hypLatitude + "," + hypLongitude+ "&" +
                        "markers=" + hypLatitude + "," + hypLongitude+ ",red&" +
                        "zoom=7&" +
                        "size=450x250&" +
                        "maptype=satellite&" +
                        "key=ABQIAAAAqyzzbiK9siI-msWQ3Dfp9BT5BQoDbTDyj7mFdda6I1pbLwBJfxTN_7I3G8UyLk2bVL6yTDex1dIbGQ" +
                        "\"></img><br><br>";
            //Include location text
            msgBody = msgBody +
                    "<b>Location</b><br>" +
                    "Origin time: " + hypOrigin + "<br>" +
                    "Latitude: " + df.format(hypLatitude) + "<br>" +
                    "Longitude: " + df.format(hypLongitude) + "<br>" +
                    "Depth: " + hypDepth + " km<br>" +
                    "Magnitude: " + hypMag + "<br>";
            if ((Configuration.useWaveServer)&&(!filename.equals("")))
                msgBody = msgBody +
                        "<br><b>Waveforms</b><br><img src=\"" + filename + "\"></img><br>";
            //Include picks text
            msgBody = msgBody + "<br><b>Picks</b><br>";
            for (int i=0; i<phasesStr.length; i++)
                msgBody = msgBody + phasesStr[i] + "<br>";
            msgBody = msgBody + "</font></p>" +
                    "</body></html>";

            //Finally send email
            sendEmailAttach y = new sendEmailAttach(subject, msgBody, filename);
            y.start();


        }
        //Update main window
        EWQuakeAlertView.updateLabels(this);
        return output;
    } //end sendEWMsgEmail



    public void sendEWSMS() {
        NumberFormat nf = NumberFormat.getNumberInstance(new Locale("en", "US"));
        DecimalFormat df = (DecimalFormat)nf;
        df.applyPattern("#0.00");

        int index = 0;
        index = hypOrigin.indexOf(" ");
        String originutf = hypOrigin.substring(0,index) + "%20" +
                hypOrigin.substring(index+1);


        String SMSText = "EWQuakeAlert:%20" + eventID + "%0A" +
                "Date:%20" + originutf + "%0A" +
                "Magnitude:%20" + hypMag + "%0A" +
                "Lat:%20" + df.format(hypLatitude) + "%0A" +
                "Lon:%20" + df.format(hypLongitude);
        //System.out.println(SMSText);

        for (int i=0; i<Configuration.SMSdestinations.length; i++)
            if (Configuration.SMSdestinations[i].length()>0)
                new sendSMS(Configuration.SMSdestinations[i],SMSText);
    } //end sendSMS










    public boolean openPhases() {
        WaveServer WS = new WaveServer();

        boolean test = WS.getData(phases);

        if (test) {
            filename = eventID + ".png";

            new PrintPhases(phases, filename);
            Configuration.last_png = filename;
        }
        return test;
    } //end openPhases




    public void addStation(String stationLine) {
        //Read and interpred a station line to add to the vector of stations
        //String aux = stationLine.substring(12, 15).trim();
        //if (!aux.equals("P"))
        //    return;

        phases.addElement(new Phase());
        //System.out.println(stationLine);
        //System.out.println(">" + stationLine.substring(111,113));

        phases.lastElement().Sta = stationLine.substring(0, 5).trim();
        phases.lastElement().Net = stationLine.substring(5, 7).trim();
        phases.lastElement().Chan = stationLine.substring(9, 12).trim();
        phases.lastElement().Loc = stationLine.substring(111, 113).trim();
        phases.lastElement().P = stationLine.substring(14, 15).trim();

        if (phases.lastElement().P.equals(""))
            phases.lastElement().P = stationLine.substring(46,48).trim();

        phases.lastElement().FstMotion = stationLine.substring(15, 16).trim();
        phases.lastElement().Weight = stationLine.substring(16, 17).trim();

        int year = Integer.valueOf(stationLine.substring(17, 21));
        int month = Integer.valueOf(stationLine.substring(21,23));
        int day = Integer.valueOf(stationLine.substring(23,25));
        int hour = Integer.valueOf(stationLine.substring(25,27));
        int min = Integer.valueOf(stationLine.substring(27,29));

        double sec = Double.valueOf(stationLine.substring(29,34).trim())/100;
        if (phases.lastElement().P.equals("S"))
            sec = Double.valueOf(stationLine.substring(41,46).trim())/100;
        //int intsec = (int)sec;
        Calendar cal = Calendar.getInstance();
        cal.set(year, month - 1, day, hour, min, 0);
        //cal.set(2009, 9, 14, 16, 35 ,0);
        long refTime = cal.getTimeInMillis();
        //Remove residual second
        refTime /= 1000;
        refTime *= 1000;

        phases.lastElement().StartTime =
                (double)refTime+(sec * 1000);

        phases.lastElement().CDuration = Double.valueOf(stationLine.substring(87,91).trim());
        //System.out.println("Message: " + stationLine);
        //System.out.println("Coda: " + phases.lastElement().CDuration);
        
        phases.lastElement().EndTime = phases.lastElement().StartTime +
                phases.lastElement().CDuration * 1000;
    }
} //end EWMessage class
