/*************************************************************************
 *   v02wws_ack.C                                                        *
 *                                                                       *
 *   Program to read COSMOS V0 files, convert to tracebuf2 and           *
 *   export to a Winston Waveserver that uses ack protocol.              *
 *                                                                       *
 *   The tracebuf packets are aligned on 1 second boundaries so          *
 *   that data in the Winston Waveserver can be overwritten.             *
 *                                                                       *
 *   This is basically export_ack with a V0 reader frontend.             *
 *                                                                       *
 *   The use of ACKnowledgement optimizes the speed with which           *
 *   data can be stuffed into the Winston.                               *
 *                                                                       *
 *   This program is based on nq2wws_ack, by Jim Luetgert.               *
 *                                                                       *
 *   Pete Lombard 2011-10-18                                             *
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <math.h>
#include <time.h>
#include <errno.h>
#include <signal.h>

extern "C" {
#include <earthworm.h>
#include <kom.h>
#include <trace_buf.h>
#include <transport.h>
#include <imp_exp_gen.h>
#include <mem_circ_queue.h>
#include <priority_level.h>   /* for EW_PRIORITY definitions only */
#include <socket_ew.h>
#include <time_ew.h>
}

#include "v02wws_ack.h"
#include "v0_multiplexor.h"

int ShutMeDown;		/* shut down flag */

char   progname[15];
char   V0FilesInDir[FILE_NAM_LEN];     /* directory from whence cometh the          */
                                       /* v0 input files */
char   V0FilesOutDir[FILE_NAM_LEN];    /* directory to which we put processed files */
char   V0FilesErrorDir[FILE_NAM_LEN];  /* directory to which we put problem files   */

/* CONSTANTS */
#define TIME_SLOP 3 /* seconds */

/* Threads in this source file
*****************************/
thr_ret SocketSender  ( void * );    /* thread to write to socket            */
thr_ret SocketRecv    ( void * );    /* thread to read from socket           */
thr_ret MessageStacker( void * );    /* used to pass messages between main   */
                                     /*    thread and SocketSender           */
/* Other Thread things
*********************/
#define THREAD_STACK 8192
static unsigned TidSocketSend;       /* SocketSender thread id                */
static unsigned TidSocketRecv;       /* SocketRecv thread id                  */
static unsigned TidStacker;          /* Thread moving messages from transport */
                                     /*   to queue                            */
/* Thread flags
**************/
#define THREAD_OFF    0              /* thread has not been started         */
#define THREAD_TEST   1              /* thread alive, main is testing reset */
#define THREAD_ALIVE  2              /* thread alive and well               */
#define THREAD_ERR   -1              /* thread encountered error quit       */
/* The xxStatus variables are for filing complaints */
volatile int MessageStackerStatus = THREAD_OFF;
volatile int SocketSenderStatus   = THREAD_OFF;
volatile int SocketRecvStatus     = THREAD_OFF;
/* The xxAlive variables are for proving vitality */
volatile int MessageStackerAlive  = THREAD_OFF;
volatile int SocketSenderAlive    = THREAD_OFF;
volatile int SocketRecvAlive      = THREAD_OFF;

volatile int LiveConnection=0;       /* 0= not connected   1=connected to client */

/* Socket status values
***********************/
#define SOCK_CLOSED      0           /* No sockets set up yet                     */
#define SOCK_NEW         1           /* New socket net yet connected              */
#define SOCK_NOT_CONN    2           /* No connection after timeout               */
#define SOCK_CONNECTED   3           /* connected to client                       */
volatile int SocketStatus = SOCK_CLOSED;

/* Global socket things
**********************/
int    PassiveSocket = 0;            /* Socket descriptor; passive socket     */
int    ActiveSocket;                 /* Socket descriptor; active socket      */

/* Message stacking queue
************************/
QUEUE OutQueue;                      /* from mem_circ_queue.h,c; sets up linked   */
                                     /*    list via malloc and free               */

/* Send Queue stuff: used for asynchronous tracking of acknowledgments
*********************************************************************/
typedef struct _SEND_QUEUE {
    char         *msg;                 /* pointer to raw message (not bin-escaped!)*/
    long          msglen;              /* length of raw message                    */
    MSG_LOGO      logo;                /* logo used in WriteSocket                 */
    unsigned char sequence;            /* sequence number used in WriteSocket      */
    volatile int  status;              /* current status of this message           */
} SEND_QUEUE;

SEND_QUEUE *SQ    = NULL;            /* allocate to proper length                */
char       *SQmsg = NULL;            /* allocate to SendQueueLen*MaxMsgSize;     */
                                     /*   have SQ.msg point in here.             */
int           SendQueueLen = 100;    /* default length of send queue (1-254 OK)  */
unsigned char SendSequence = 0;      /* next sequence number for WriteSocket;    */
                                     /*  also used as index into the SendQueue.  */
                                     /*  (value restricted: 0 to SendQueueLen)   */

/* Definitions of SendQueue status: */
#define SQS_NEW         1            /* new pkt dequeued from MessageStacker     */
#define SQS_SENT        2            /* pkt sent to socket; waiting for ACK      */
#define SQS_ACK_RCVD    3            /* ACK received, slot can be (re)used       */

/* Message Buffers to be allocated
*********************************/
static char *Rawmsg        = NULL;   /* "raw" retrieved msg for main thread      */
static char *SSbinmsg      = NULL;   /* SocketSender's binary-escaped msg buffer */
static char *SRmsg         = NULL;   /* SocketRecv's incoming message buffer     */
static char  inBuffer[INBUFFERSIZE]; /* working input buffer for SocketRecv      */

/* Heart beat stuff
*******************/
time_t LastRcvAlive  = 0;  /* Last time we heard from our client */
time_t LastSendAlive = 0;  /* Last time we sent a heartbeat to our client */
time_t now;                /* current time, used for timing heartbeats */
time_t MyLastBeat;         /* time of last local (into Earthworm) hearbeat */
int    ImportAlive   = 1;  /* switch for noting if our import is sending heart beats ok */

extern int  errno;

static    SHM_INFO  Region;       /* shared memory region to use for i/o    */

char *Argv0;            /* pointer to executable name */
pid_t MyPid;        /* Our own pid, sent with heartbeat for restart purposes */

/* Things to read or derive from configuration file
**************************************************/
static char    RingName[MAX_RING_STR]; /* name of transport ring for i/o    */
static char    MyModName[MAX_MOD_STR]; /* speak as this module name/id      */
static int     LogSwitch;              /* 0 if no logfile should be written */
static int     HeartBeatInt;           /* seconds between heartbeats        */
static char    ServerIPAdr[16];        /* server's IP address               */
static int     ServerPort;             /* Server's well-known port number   */
static long    MaxMsgSize;             /* max size for input/output msgs    */
static int     RingSize;               /* max messages in output circular buffer */
static int     SendAliveInt;           /* Send alive messages this often    */
static char    SendAliveText[MAX_ALIVE_STR]; /* Text of alive message. Max size is traditional */
static int     RcvAliveInt;            /* Expect client heartbeats this often */
static char    RcvAliveText[MAX_ALIVE_STR];  /* Text of client's alive messages */
static char    DefLocationCode[TRACE2_LOC_LEN];  /* configured location code, *
						  * since v0 doesn't have any */
static int     Verbose=0;              /* changed to 1 by Verbose command       */
int Debug;
int err_exit;			       /* an v02ring_die() error number */
int nchans;
DATABUF chanbuf[MAXCHANS];

static int     SocketTimeoutLength=0;  /* Length of timeouts on SOCKET_ew calls */
/* Socket timeouts are not handled (well) in export, so there is not a
 * whole lot of point to setting SocketTimeoutLength to any useful value.
 * Currently it is set to atleast the RcvAliveInt. */

static int     SOCKET_ewDebug = 0;     /* Set to 1 for SOCKET_ew debug statements*/

static int     HBflag = 0;             /* Set to 1 for heartbeat statements*/

/* Things to look up in the earthworm.h tables with getutil.c functions
**********************************************************************/
static long          RingKey;       /* key of transport ring for i/o      */
static unsigned char InstId;        /* local installation id              */
static unsigned char MyModId;       /* Module Id for this program         */
static unsigned char TypeHeartBeat;
static unsigned char TypeError;
static unsigned char TypeTrace2;

/* Error messages used by export
********************************/
#define  ERR_MISSMSG       0   /* message missed in transport ring        */
#define  ERR_TOOBIG        1   /* retreived msg too large for buffer      */
#define  ERR_NOTRACK       2   /* msg retreived; tracking limit exceeded  */
#define  ERR_SOCKETSEND    3   /* trouble writing to socket               */
#define  ERR_SOCKETACCEPT  4   /* error accepting socket connection       */
#define  ERR_QUEUE         5   /* error queueing message for sending      */
#define  ERR_NOCONN        6   /* no connection after timeout             */
#define  ERR_CONN          7   /* Connection accepted; cancels ERR_NOCONN */
#define  ERR_SOCKETREAD    8   /* trouble reading from socket             */
static char  errText[256];     /* string for log/error messages           */

/* Functions in this source file
*******************************/
void  v02wws_config  ( char * );
void  v02wws_lookup  ( void );
void  v02wws_status  ( unsigned char, short, char * );
void  v02wws_shutdown( int );
void  v02wws_free    ( void );
int   SendPacket     ( SEND_QUEUE *sq );
int   WriteToSocket  ( int, char *, long, MSG_LOGO *, unsigned char );
int   binEscape      ( char*, long , char* , long* , long );

int v0_to_tbuf(v0_cosmos *v0, int *tbufcount);
int queue_putmsg( long recsize, char* trace_buffer);

/**********************************************************************
 *                                main()                              *
 *   Program to read COSMOS v0 data files from a directory, convert   *
 *   them to tracebufs aligned on integer seconds and export them     *
 *   over a socket with acknowledgement.
 *                                                                    *
 *   This is designed for use with Winston WaveServer, but could be   *
 *   used with any import_ack.                                        *
 **********************************************************************/

int main( int argc, char **argv )
{
    /* Socket variables: */
    int    on = 1;
    int    clientLen;
    char   client_ip[16];    /* IP address of client from inet_ntoa   */
    struct sockaddr_in  skt;
    struct sockaddr_in  client;

    /* Other variables: */
    char      whoami[50];
    time_t    timeNow;       /* current time                          */
    int       res;
    long      recsize;       /* size of retrieved message             */
    MSG_LOGO  reclogo;       /* logo of retrieved message             */
    int       AcceptTimeout; /* time in milliseconds for accept_ew()  */
    int       sockErr;
    char     *ptr;           /* temp pointer for setting up SendQueue */
    int       i;

    /* The timers and time limits for assertaining thread life */
    time_t messageStackerTimer;
    time_t socketSenderTimer;
    time_t socketRecvTimer;
    time_t messageStackerTimeout;
    time_t socketSenderTimeout;
    time_t socketRecvTimeout;

    sprintf(whoami, (char*)"%s: %s: ", progname, (char*)"main");

    /* Catch broken socket signals
    ******************************/
#ifdef SIGPIPE
    (void)sigignore(SIGPIPE);
#endif

    /* Check command line arguments
     ******************************/
    Argv0 = argv[0];
    sprintf(progname, (char*)"%s", Argv0);
    if( argc != 2 )
    {
	fprintf( stderr, (char*)"Usage: %s <configfile>\n", Argv0 );
	return( 0 );
    }

    /* Initialize name of log-file & open it
     ****************************************/
    logit_init( argv[1], 0, 512, 1 );

    /* init some locals */
    err_exit = -1;

    /* Read the configuration file(s) & lookup info from earthworm.h
     ****************************************************************/
    v02wws_config( argv[1] );
    v02wws_lookup();
    logit( (char*)"et" , (char*)"%s Read command file <%s>\n", whoami, argv[1] );

    /* Reinitilize the logging level
     ********************************/
    logit_init( argv[1], 0, 512, LogSwitch );

    /* Set SOCKET_ew debugging on/off
     *********************************/
    setSocket_ewDebug(SOCKET_ewDebug);

    /* Introduce ourselves
     **********************/
    logit((char*)"",(char*)"2010-10-18 version: "
	  "I do re-sends, I expect ACKs, I check thread life.\n");

    /* Heartbeat parameters sanity checks
     ************************************/
    /* Do a sanity check on the SocketTimeoutLength.
       If it is -1 (no timeout; block on the call) then
       leave it alone. */
   
    if( SocketTimeoutLength != -1 &&
	1000*(RcvAliveInt)  >= SocketTimeoutLength ) {
	/* WARNING: This code insures that export's socket thread will never
	 * survive a socket timeout.  By setting them to greater than the
	 * RcvAliveInt, the main thread will clobber the other threads and
	 * perform a restart, any time that the socket actually hits a timeout.
	 * This was done, because there is currently no code in export.c to handle
	 * socket timeouts, (and no real reason to.)
	 * DavidK 04/16/2001
	 **************************************************************************/
	logit((char*)"",(char*)"Socket timeout (%d ms) is less than incoming heartrate "
	      "(%d sec)", SocketTimeoutLength, RcvAliveInt );
	SocketTimeoutLength = 1000*(RcvAliveInt);
	logit((char*)"", (char*)"Setting socket timeout to %d ms\n", SocketTimeoutLength);
    }

    /* Get our own Pid for restart purposes
     ***************************************/
    MyPid = getpid();
    if(MyPid == -1) {
	logit((char*)"e", (char*)"%s Cannot get pid; exiting!\n", whoami);
	return(0);
    }

    /* Allocate space for input/output messages for all threads
     ***********************************************************/
    /* Buffers for SocketSender thread: */
    if( ( SSbinmsg = (char *) malloc(MaxMsgSize*2) ) ==  NULL ) {
	logit( (char*)"e", (char*)"%s error allocating SSbinmsg; exiting!\n", whoami );
	v02wws_free();
	return( -1 );
    }

    /* Buffers for Send Queue */
    if( ( SQ = (SEND_QUEUE *)calloc((size_t)SendQueueLen,
                                    sizeof(SEND_QUEUE)) ) ==  NULL ) {
	logit( (char*)"e", (char*)"%s error allocating SendQueue; exiting!\n", whoami );
	v02wws_free();
	return( -1 );
    }
    if( ( SQmsg = (char *)calloc((size_t)SendQueueLen*MaxMsgSize,
                                 sizeof(char)) ) ==  NULL ) {
	logit( (char*)"e", (char*)"%s error allocating SQmsg; exiting!\n", whoami );
	v02wws_free();
	return( -1 );
    }
    /* Initialize Send Queue */
    for( i=0, ptr=SQmsg; i<SendQueueLen; i++, ptr+=MaxMsgSize ) {
	SQ[i].msg    = ptr;          /* point to memory in SQmsg   */
	SQ[i].status = SQS_ACK_RCVD; /* this slot can be (re)used! */
    }

    /* Buffer for SocketRecv thread: */
    if( ( SRmsg = (char *) malloc(MaxMsgSize+1) ) ==  NULL ) {
	logit( (char*)"e", (char*)"%s error allocating SRmsg; exiting!\n", whoami );
	v02wws_free();
	return( -1 );
    }

    /* initialize the channel buffers */
    for (i=0; i < MAXCHANS; i++) {
	chanbuf[i].buf = NULL;
	chanbuf[i].bufLen = 0;
	chanbuf[i].bufptr = 0;
	strcpy(chanbuf[i].sncl, " ");
    }

    /* Attach to Input/Output shared memory ring; beat heart first time
     ******************************************************************/
    tport_attach( &Region, RingKey );
    v02wws_status( TypeHeartBeat, 0, (char*)"" );
    time(&MyLastBeat);

    /* Create a Mutex to control access to message stacking queue
     ************************************************************/
    CreateMutex_ew();

    /* Initialize the message stacking queue
     ***************************************/
    initqueue( &OutQueue, (unsigned long)RingSize,(unsigned long)MaxMsgSize );

    /************************/
 retry:
    /************************/
    /* Come here after communications with client have been broken,     */
    /* the socket has been closed, and the threads have been terminated. */

    /* Initialize the socket system
     *******************************/
    SocketSysInit();

    if( ( PassiveSocket = socket_ew( PF_INET, SOCK_STREAM, 0) ) == -1 ) {
	logit( (char*)"et", (char*)"%s Error opening socket; exiting!\n", whoami );
	tport_detach( &Region );
	return( -1 );
    }
    SocketStatus = SOCK_NEW;

    /* Fill in server's socket address structure
********************************************/
    memset( (char *) &skt, '\0', sizeof(skt) );
    skt.sin_family = AF_INET;
    skt.sin_port   = htons( (short)ServerPort );
#if defined(_LINUX) || defined(_MACOSX)
    if ((int)(skt.sin_addr.s_addr = inet_addr(ServerIPAdr)) == -1) {
#else
    if ((int)(skt.sin_addr.S_un.S_addr = inet_addr(ServerIPAdr)) == -1) {
#endif
	logit( (char*)"e", (char*)"%s inet_addr failed for ServerIPAdr <%s>;"
	       " exiting!\n", whoami, ServerIPAdr );
	return( -1 );
    }

    /* Allows the server to be stopped and restarted
     ************************************************/
    on=1;
    if( setsockopt( PassiveSocket, SOL_SOCKET, SO_REUSEADDR,
		    (char *)&on, sizeof(char *) ) != 0 ) {
	logit( (char*)"et", (char*)"%s Error on setsockopt; exiting!\n", whoami );
	perror((char*)"Export setsockopt");
	v02wws_shutdown( -1 );
    }

    /* Bind socket to a name
     ************************/
    if ( bind_ew( PassiveSocket, (struct sockaddr *) &skt, sizeof(skt)) ) {
	logit((char*)"et", (char*)"%s error binding socket; exiting.\n", whoami );
	perror((char*)"Export bind error");
	v02wws_shutdown( -1 );
    }

    /* Prepare for connect requests
     *******************************/
    if( listen_ew( PassiveSocket, 0 ) ) {
	logit((char*)"et", (char*)"%s socket listen error; exiting!\n", whoami );
	v02wws_shutdown( 1 );
    }

    /* Accept a connection (blocking)
     *********************************/
    clientLen = sizeof( client );
    logit( (char*)"et", (char*)"%s Waiting for new connection.\n", whoami );
    AcceptTimeout = 1000 * HeartBeatInt;

    while( (ActiveSocket = accept_ew( PassiveSocket,
				      (struct sockaddr*) &client,
				      &clientLen, AcceptTimeout) ) == INVALID_SOCKET ) {
	sockErr = socketGetError_ew();
	if( sockErr == WOULDBLOCK_EW ) {
	    if( SocketStatus == SOCK_NEW ) {
		SocketStatus = SOCK_NOT_CONN;
		sprintf( errText, (char*)"No connection after %d seconds\n", HeartBeatInt );
		v02wws_status( TypeError, ERR_NOCONN, errText );
	    }
	} else {
	    logit((char*)"et", (char*)"%s error on accept: %d %s\n\t; exiting!\n", whoami,
		  sockErr, strerror(sockErr) );
	    v02wws_shutdown(-1);
	}
	/* Are we supposed to shut down? */
	if( tport_getflag( &Region ) == TERMINATE ) goto shutdown;
	if( tport_getflag( &Region ) == MyPid     ) goto shutdown;

	/* Beat our heart so statmgr knows we're alive */
	v02wws_status( TypeHeartBeat, 0, (char*)"" );
	time(&MyLastBeat);
    }

    strcpy( client_ip, inet_ntoa(client.sin_addr) );
    LiveConnection = 1; /*970623:ldd*/
    if( SocketStatus == SOCK_NOT_CONN ) {
	/* we cried before, so now we have to let them know we're OK */
	sprintf( errText, (char*)"Connection accepted from IP address %s\n", client_ip);
	v02wws_status( TypeError, ERR_CONN, errText );
    } else {
	logit((char*)"et", (char*)"%s Connection accepted from IP address %s\n", whoami, client_ip );
    }
    SocketStatus = SOCK_CONNECTED;

    /* Start the message stacking thread if it isn't already running.
     ****************************************************************/
    if( MessageStackerStatus != THREAD_ALIVE) {
	if( StartThread( MessageStacker, (unsigned)THREAD_STACK, &TidStacker ) == -1 ) {
	    logit( (char*)"e",
		   (char*)"%s Error starting  MessageStacker thread; exiting!\n", whoami );
	    v02wws_shutdown( -1 );
	}
	MessageStackerStatus  = THREAD_ALIVE;
	MessageStackerAlive   = THREAD_ALIVE; /* assume the best */
	/* There are no inherent time-delays in the stacker */
	messageStackerTimeout = TIME_SLOP;
    }

    /* Start the socket writing thread
     *********************************/
    if( StartThread( SocketSender, (unsigned)THREAD_STACK, &TidSocketSend ) == -1 ) {
	logit( (char*)"e", (char*)"%s Error starting SocketSender thread; exiting!\n", whoami );
	v02wws_shutdown( -1 );
    }
    /* It will set its status, but set it here to prevent timing problems */
    SocketSenderStatus = THREAD_ALIVE;
    SocketSenderAlive  = THREAD_ALIVE;
    /* Timeout is at worst sending an old message, a heartbeat, and a message.
       Each is at worst one socketTimeout */
    socketSenderTimeout = 3*(SocketTimeoutLength/1000) + TIME_SLOP;

    /* Start the socket reading thread
     *********************************/
    if( StartThread( SocketRecv, (unsigned)THREAD_STACK, &TidSocketRecv ) == -1 ) {
	logit( (char*)"e", (char*)"%s Error starting SocketRecv thread; exiting!\n", whoami );
	v02wws_shutdown( -1 );
    }
    /* It will set its status, but set it here to prevent timing problems */
    SocketRecvStatus  = THREAD_ALIVE;
    SocketRecvAlive   = THREAD_ALIVE;
    /* At most one slurp from the socket */
    socketRecvTimeout = (SocketTimeoutLength/1000) + TIME_SLOP;

    /* set import heartbeat timer & flag
     ************************************/
    time(&LastRcvAlive);  /* from distant client */
    ImportAlive = 1;  /* assume we'll be hearing from our import */

    /* Start main export monitoring loop for current connection
     **********************************************************/
    while( tport_getflag( &Region ) != TERMINATE && 
	   tport_getflag( &Region ) != MyPid         ) {
	/* See if our import has beat it's heart
         ****************************************/
	time(&timeNow);
	if(difftime(timeNow,LastRcvAlive) > (double)RcvAliveInt && RcvAliveInt != 0 ) {
	    logit((char*)"et", (char*)"%s lost import heartbeat\n", whoami);
	    ImportAlive = 0; /* conclusion: no heartbeats from our import partner */
	}

	/* see how our socket thread is are feeling
         ******************************************/
	/* Each thread complains by setting its ...Status to -1, and exiting. */
	if( SocketSenderStatus != THREAD_ALIVE  ||
	    SocketRecvStatus   != THREAD_ALIVE  ||
	    ImportAlive        != 1 ) {
	    logit((char*)"et", (char*)"%s restarting. This procedure may hang. "
		  "Make sure restartMe is set in my .desc file\n", whoami  );
               
	    /* The code below occasionally hangs before the "waiting for new connection"
	     * statement. The solution is to set the "restartMe" option in the .desc
	     * file, which will cause me to be restarted by statmgr/startstop. */
	    /* One cause of hanging is that KillThread() does nothing about mutexes held
	     * by the threads being killed. If a thread holds a mutex when it is killed,
	     * the mutex is held forever. It would be better to use thread cancellation
	     * once earthworm is free from Solaris threads. PNL, 1/10/00 */
       
	    closesocket_ew( ActiveSocket,  SOCKET_CLOSE_IMMEDIATELY_EW );
	    closesocket_ew( PassiveSocket, SOCKET_CLOSE_IMMEDIATELY_EW );
	    SocketStatus = SOCK_CLOSED;
	    LiveConnection = 0;
	    /* Stop the socket threads, but leave the MessageStacker running. */
	    (void)KillThread(TidSocketSend);
	    SocketSenderStatus = THREAD_OFF;
	    SocketSenderAlive  = THREAD_OFF;
	    (void)KillThread(TidSocketRecv);
	    SocketRecvStatus   = THREAD_OFF;
	    SocketRecvAlive    = THREAD_OFF;

	    /* go & re-try from scratch */
	    goto retry;
	}

	/* See if threads are really running
         ************************************/
	/* The scheme is to knock down the thread's 'alive' flags
	   and observe that the threads are healthy enough to
	   raise them up again within a reasonable amount of time */
	if( MessageStackerAlive == THREAD_ALIVE ) { /* Hurray, it's running */
	    MessageStackerAlive = THREAD_TEST; /* see if it gets reset */
	    time( &messageStackerTimer );      /* within the timelimit */
	} else {
	    if( difftime( messageStackerTimer, time(NULL)) > messageStackerTimeout ) {
	    /* Very weird and very bad. Exit */
		logit((char*)"et",(char*)"%s MessageStacker thread hung. Exiting\n", whoami);
		v02wws_shutdown(-1);
	    }
	}

	if( SocketSenderAlive == THREAD_ALIVE ) { /* Hurray, it's running */
	    SocketSenderAlive = THREAD_TEST; /* see if it gets reset */
	    time( &socketSenderTimer );      /* within the timelimit */
	} else {
	    if( difftime( socketSenderTimer, time(NULL)) > socketSenderTimeout ) {
		/* Very weird and very bad. Exit */
		logit((char*)"et",(char*)"%s SocketSender thread hung. Exiting\n", whoami);
		v02wws_shutdown(-1);
	    }
	}

	if( SocketRecvAlive == THREAD_ALIVE ) { /* Hurray, it's running */
	    SocketRecvAlive = THREAD_TEST; /* see if it gets reset */
	    time( &socketRecvTimer );      /* within the timelimit */
	} else {
	    if( difftime( socketRecvTimer, time(NULL)) > socketRecvTimeout ) {
		/* Very weird and very bad. Exit */
		logit((char*)"et",(char*)"%s SocketRecv thread hung. Exiting\n", whoami);
		v02wws_shutdown(-1);
	    }
	}

	/* Beat the heart into the transport ring
         ****************************************/
	time(&now);
	if( difftime(now,MyLastBeat) > (double)HeartBeatInt ) {
	    v02wws_status( TypeHeartBeat, 0, (char*)"" );
	    time(&MyLastBeat);
	}

	/* take a brief nap; added 970624:ldd
         ************************************/
	sleep_ew(100);
    } /*end while of monitoring loop */

 shutdown:
    /* Shut it down
     ***************/
    logit((char*)"t", (char*)"%s termination requested; exiting!\n", whoami );
    v02wws_shutdown(0);
}
/*************************** end of main **************************/


/**********************************************************************
 *                          SocketRecv()                              *
 * Read socket for client heartbeats and set global variable showing  *
 * time of last heartbeat. Main thread will take it from there.       *
 * Also read socket for acknowledgments.                              *
 *                                                                    *
 * If things go badly, set the global SocketStatus non-zero to alert  *
 * the main thread.                                                   *
 **********************************************************************/

thr_ret SocketRecv( void *dummy )
{
    char        whoami[50];
    static int  state;
    static char chr;
    static int  nr;                 /* total #chars in inBuffer     */
    static long inchar;             /* current position in inBuffer */
    static long nchar;              /* current position in SRmsg    */
    static char startCharacter=STX; /* ASCII STX characer  */
    static char endCharacter  =ETX; /* ASCII ETX character */

    sprintf(whoami, (char*)"%s: %s: ", progname, (char*)"SocketRecv");
	
    SocketRecvStatus = THREAD_ALIVE;

    /* Initialize these variables once per active socket
     ***************************************************/
    /* Working loop: receive and process messages
     * We are either (0) initializing,
     *               (1) looking for a message start,
     *               (2) assembling a message
     * The variable `state' determines our mood.
     *********************************************/
    state=SEARCHING_FOR_MESSAGE_START;  /* we're initializing */

    /* Start of New code Section DK 11/20 Multi-byte Read
     * Set inchar to be nr-1, so that when inchar is incremented, they will be the
     * same and a new read from socket will take place.  Set chr to 0 to indicate
     * a null character, since we haven't read any yet.
     */
    inchar=-1;
    nr=0;
    chr=0;
    /* End of New code Section */

    while(1) {  /* loop over socket reads */
	/* Tell the main thread we're OK. Do this each pass thru the
	 * loop, as the main thread will keep resetting it to THREAD_OFF,
	 * just to see us raise it as a sign of continued life.
	 ****************************************************************/
	SocketRecvAlive = THREAD_ALIVE;

	/* Read from buffer
         ******************/
	/* Get next char operation */
	if (++inchar == nr) {
	    /* Read from socket operation */
	    nr=recv_ew(ActiveSocket,inBuffer,INBUFFERSIZE,0,SocketTimeoutLength);
	    if (nr<=0)  { /* Connection Closed */
		sprintf (errText, (char*)"%s recv_ew()", whoami);
		SocketPerror(errText);
		logit((char*)"et", (char*)"%s Bad socket read: %d\n", whoami, nr);
		SocketRecvStatus = THREAD_ERR; /* complain to main thread     */
		KillSelfThread();              /* main thread will restart us */
	    }
	    inchar=0;
	    /* End of Read from socket operation */
	}
	chr=inBuffer[inchar];
	/* End of Get next char operation */

	/* Initialization:
	 * Throw all away until we see a start character
	 ***********************************************/
	if (state==SEARCHING_FOR_MESSAGE_START)	{
	    if (chr == startCharacter) state=ASSEMBLING_MESSAGE; /* OK to start assembing message */
	    nchar=0; /* next character position to load */
	    continue;
	}

	/* Seeking message start:
	 * The next char had better be a start character
	 ***********************************************/
	if (state==EXPECTING_MESSAGE_START) {
	    nchar=0;
	    if (chr != startCharacter) { /* then we're eating garbage */
		logit((char*)"et", (char*)"%s unexpected character from client\n", whoami);
		state=SEARCHING_FOR_MESSAGE_START; /* initialization mode */
		continue;
	    } else {   /* we got the message start, and we're in assembly mode */
		nchar=0; /* start with firsts char position */
		state=ASSEMBLING_MESSAGE; /* go into assembly mode */
		continue;
	    }
	}

	/* In the throes of assembling a message
         ***************************************/
	if (state==ASSEMBLING_MESSAGE) {
	    /* We have a complete message
             ****************************/
	    if (chr==endCharacter) {  /* end of the message is at hand */
		SRmsg[nchar]=0; /* terminate as a string */

		/* Client's heartbeat */
		if(strcmp(&SRmsg[9],RcvAliveText)==0) {
		    if(Verbose || HBflag) logit((char*)"et", (char*)"%s Received heartbeat\n", whoami);
		}

		/* Acknowledgment */
		else if(strncmp(&SRmsg[9],(char*)"ACK:",4)==0) {
		    int  ack;
		    char ackstr[4];
		    strncpy( ackstr, &SRmsg[13], 3 );  ackstr[3] = 0;
		    ack = atoi( ackstr );
		    if( ack == HEARTSEQ ) {
			if(Verbose) logit((char*)"et", (char*)"%s Received ACK:%3d (heartbeat)\n", whoami, ack);
		    } else {
			SEND_QUEUE *sq = &(SQ[ack%SendQueueLen]);
			sq->status     = SQS_ACK_RCVD;
			if(Verbose) {
			    logit((char*)"et", (char*)"%s Received ACK:%3d\n", whoami, ack);
			}
		    }
		}

		/* Got a non-heartbeat, non-acknowledgment message!  */
		/* We don't know to whom we're talking; stop thread! */
		else {
		    logit((char*)"et", (char*)"%s Weird message from client:\n%s\n", whoami, SRmsg );
		    SocketRecvStatus = THREAD_ERR; /* complain to main thread     */
		    KillSelfThread();              /* main thread will restart us */
		}

		/* Both heartbeats and acks prove that import is alive. */
		/* Reset the recv timer and get ready for the next msg  */
		time(&LastRcvAlive);
		state=EXPECTING_MESSAGE_START;
		continue;
	    }

	    /* Keep building the message
             ***************************/
	    else {
		if( nchar < MaxMsgSize ) { /* then there's still room in the buffer */
		    SRmsg[nchar++]=chr;
		} else {  /* freakout: message won't fit */
		    logit((char*)"et",
			  (char*)"%s receive buffer overflow after %ld bytes\n", whoami, MaxMsgSize);
		    state=SEARCHING_FOR_MESSAGE_START; /* initialize again */
		    nchar=0;
		    continue;
		}
	    }
	} /* end of state==ASSEMBLING_MESSAGE processing */
    }  /* end while of loop over characters in inBuffer */

#ifndef _WINNT
    return (thr_ret)0;
#endif
}  /* end of SocketRecv thread */


/************************** SocketSender ******************************
 * Pull a messsage from the queue, and push it out the socket with a  *
 * known sequence number.                                             *
 *                                                                    *
 * If things go badly, set the global SocketStatus non-zero to alert  *
 * the main thread.                                                   *
 **********************************************************************/

thr_ret SocketSender( void *dummy )
{
    SEND_QUEUE *sq;   /* working pointer into SendQueue   */
    time_t      now, lasttime=0;  /* heartbeat things */
    MSG_LOGO    hrtlogo;
    char        whoami[50];
    int         ret;
    int         ifirst, ilast;    /* indices for looking in SendQueue */
    int         oldstatus, i;
    int         nresent = 0;

    sprintf(whoami, (char*)"%s: %s: ", progname, (char*)"SocketSender");
    
    /* Tell the main thread we're ok; initialize stuff
     **************************************************/
    SocketSenderStatus = THREAD_ALIVE;
    SocketSenderAlive  = THREAD_ALIVE;
    hrtlogo.instid = InstId;
    hrtlogo.mod    = MyModId;
    hrtlogo.type   = TypeHeartBeat;

    /* Check status of all msgs in SendQueue, resend if necessary
     ************************************************************/
    ifirst = SendSequence;
    ilast  = SendSequence + SendQueueLen;
    logit((char*)"et", (char*)"%s Reviewing SendQueue for non-ACKed msgs...\n", whoami );
    for( i=ifirst; i<ilast; i++ ) {
	sq = &(SQ[i%SendQueueLen]);
	if( sq->status == SQS_ACK_RCVD ) continue;

	/* Must be SQS_NEW or SQS_SENT: send (re-send) packet!
         *****************************************************/
	oldstatus = sq->status;
	if( SendPacket( sq ) != 0 ) {
	    SocketSenderStatus = THREAD_ERR;  /* complain to main thread */
	    KillSelfThread();             /* main thread will restart us */
	}
	sq->status = SQS_SENT; /* updates to SQS_ACK_RCVD in SocketRecv */
	if( Verbose ) {
	    /*      if( oldstatus == SQS_NEW ) {
		    exportfilter_logmsg( sq->msg, sq->msglen, sq->logo.type,
		    "sent2socket:    " );
		    } else {
		    exportfilter_logmsg( sq->msg, sq->msglen, sq->logo.type,
		    "re-sent2socket: " );
		    }   */
	}
	nresent++;
    } /* end check status for loop */
    if (nresent) logit((char*)"et", (char*)"%s Re-sent %d non-ACKed msgs from SendQueue.\n",
                      whoami, nresent );
    else        logit((char*)"et", (char*)"%s SendQueue was all ACKed; no msgs re-sent.\n",
                      whoami, nresent );

    /*----------------------- main SocketSender loop ------------------------*/

    while( 1 ) {
	sq = &(SQ[SendSequence]);  /* ptr to next available SendQueue slot */

	/* Tell the main thread we're still alive
         ****************************************/
	SocketSenderAlive = THREAD_ALIVE;

	/* Send alive message to client program if it's time to
         ******************************************************/
	time( &now );
	if( (now-lasttime)>=SendAliveInt  &&  SendAliveInt != 0 ) {
	    if( WriteToSocket( ActiveSocket, SendAliveText,
			       (long)strlen(SendAliveText),
			       &hrtlogo, HEARTSEQ ) != 0 ) {
		sprintf( errText, (char*)"error sending alive msg to socket" );
		v02wws_status( TypeError, ERR_SOCKETSEND, errText );
		SocketSenderStatus = THREAD_ERR; /* complain to main thread */
		KillSelfThread();            /* main thread will restart us */
	    }
	    if(Verbose || HBflag) logit((char*)"et", (char*)"%s Sent alive msg\n", whoami );
	    lasttime = now;
	}

	/* See if next SendQueue slot is ready to (re)fill.
	 * If not, sleep a tiny bit and check again, until it's ACKed.
	 * SQS_SENT is the only status that we have to do this for.
	 * SQS_ACK_RCVD status slots can be reused immediately.
	 * SQS_NEW status is impossible here (if SocketSender thread were
	 * killed by main thread between dequeue and SendPacket, status
	 * could be SQS_NEW, but on thread restart, status would've been
	 * changed to SQS_SENT in "check status" loop before the main loop).
	 *******************************************************************/
	if( sq->status == SQS_SENT ) {
	    if(Verbose) logit( (char*)"et", (char*)"Waiting for ACK in SendQueue[%d]\n",
			       SendSequence );
	    while( sq->status != SQS_ACK_RCVD ) {
		sleep_ew( 10 );
	    }
	    if(Verbose) logit( (char*)"et", (char*)"Rcvd ACK; re-using SendQueue[%d]\n",
			       SendSequence );
	}

	/* Put next message from MessageStacker queue into SendQueue
         ***********************************************************/
	RequestMutex();
	ret = dequeue( &OutQueue, sq->msg, &(sq->msglen), &(sq->logo) );
	ReleaseMutex_ew();
	if( ret < 0 ) {  /* -1 means empty queue */
	    sleep_ew(100);
	    continue;
	}
	sq->status   = SQS_NEW;
	sq->sequence = SendSequence;
	SendSequence = (SendSequence+1)%SendQueueLen;

	/* Send the just-dequeued message
         ********************************/
	if( SendPacket( sq ) != 0 ) {
	    SocketSenderStatus = THREAD_ERR; /* complain to main thread */
	    KillSelfThread();            /* main thread will restart us */
	}
	sq->status = SQS_SENT; /* updates to SQS_ACK_RCVD in SocketRecv */
    } /* End of SocketSender while(1) loop */
#ifndef _WINNT
    return (thr_ret)0;
#endif
}

/******************************************************************
 * SendPacket()  Do the binary escape and write to socket         *
 ******************************************************************/
int SendPacket( SEND_QUEUE *sq )
{
    char    whoami[50];
    long binlen;  /* length of binary-escaped message */

    sprintf(whoami, (char*)"%s: %s: ", progname, (char*)"SendPacket");

    /* Make the message safe for binary text
     ***************************************/
    if( binEscape( sq->msg, sq->msglen,
		   SSbinmsg, &binlen, MaxMsgSize*2 ) < 0 ) {
	logit((char*)"et",(char*)"%s binEscape overflow; bad news!\n", whoami);
	return( -2 );
    }

    /* Write the escaped message to the socket.
     ******************************************/
    if( WriteToSocket( ActiveSocket, SSbinmsg, binlen,
		       &(sq->logo), sq->sequence ) != 0 ) {
	sprintf( errText, (char*)"error sending msg to socket." );
	v02wws_status( TypeError, ERR_SOCKETSEND, errText );
	return( -1 );
    }
    return( 0 );
}


/***************************binEscape******************************
 *    make a binary message safe for shipping: insert an escape   *
 *    character before any bit pattern which looks like a sacred  *
 *    character. That will allow the receiving code to recognize  *
 *    it as of the data, and not as a sacred character.           *
 *    Sacred characters are: STX (02), ETX (03), and, of course,  *
 *                           ESC (27) itsself                     *
 *                                                                *
 *    Returns 0 if ok, -1 if sanitized message might not fit      *
 ******************************************************************/

int binEscape( char* inmsg, long inSize,
               char* outmsg, long* outSize, long outMaxSize )
{
    static char startCharacter=STX; /* from imp_exp_gen.h */
    static char endCharacter=ETX;
    static char escape=ESC;
    static char tmp;
    int    i;

    /* Test output buffer size only once */
    if (outMaxSize < 2 * inSize) return(-1);

    *outSize = 0; /* index to next byte in outgoing message */
    for( i=0;i<inSize; i++ ) {  /*loop over bytes in input message */
	tmp=inmsg[i];
	if( tmp==startCharacter || tmp==endCharacter || tmp==escape ) {
	    outmsg[(*outSize)++]=escape;
	    outmsg[(*outSize)++]=tmp;
	} else {
	    outmsg[(*outSize)++]=tmp;
	}
    }
    return(0);
}


/*************************** WriteToSocket ************************
 *    send a message logo and message to the socket               *
 *    returns  0 if there are no errors                           *
 *            -1 if any errors are detected                       *
 ******************************************************************/

int WriteToSocket( int ActiveSocket, char *msg, long msglength,
                   MSG_LOGO *logo, unsigned char seq )
{
    char etext[128];        /* error text */
    char asciilogo[17];     /* ascii version of seq#, outgoing logo */
    char startmsg = STX;    /* flag for beginning of message        */
    char endmsg   = ETX;    /* flag for end of message              */
    int  rc;
    char    whoami[50];

    sprintf(whoami, (char*)"%s: %s: ", progname, (char*)"WriteToSocket");
    
    /* Send "start of transmission" flag, sequence#, ascii logo
    **********************************************************/
    sprintf( asciilogo, (char*)"%cSQ:%3d%3d%3d%3d", startmsg, (int) seq,
	     (int) logo->instid, (int) logo->mod, (int) logo->type  );
    rc = send_ew( ActiveSocket, asciilogo, strlen(asciilogo), 0, SocketTimeoutLength );
    if( rc != (int)strlen(asciilogo) ) {
	sprintf( etext, (char*)"%s socket send of message logo", whoami );
	SocketPerror( etext );
	return( -1 );
    }

    /* Send message; break it into chunks if it's big!
     *************************************************/
    rc = send_ew( ActiveSocket, msg, msglength, 0, SocketTimeoutLength);
    if( rc == -1 ) {
	sprintf( etext, (char*)"%s message send error", whoami );
	SocketPerror( etext );
	return( -1 );
    }

    /* Send "end of transmission" flag
     *********************************/
    rc = send_ew( ActiveSocket, &endmsg, 1, 0, SocketTimeoutLength);
    if( rc != 1 ) {
	sprintf( etext, (char*)"%s socket send EOT flag", whoami );
	SocketPerror( etext );
	return( -1 );
    }

    return( 0 );
}


/********************** Message Stacking Thread *******************
 *          Move messages from v0 files to memory queue           *
 ******************************************************************/
thr_ret MessageStacker( void *dummy )
{
    long       recsize;	/* size of retrieved message             */
    MSG_LOGO   reclogo; /* logo of retrieved message             */
    int        i, res, ret;
    int        NumOfTimesQueueLapped= 0; /* number of messages lost due to queue lap */
    int        NumOfElementsIv0ueue; /* number of messages in the queue  */
    int        NumOfTBufs; /* number of tracebufs burst from this file  */
    int        NumOfseeds; /* number of mSEED records burst from this file  */

    char    whoami[50];
    FILE   *fp;
    char    fname[FILE_NAM_LEN];         /* name of ascii version of evt file */
    char    outFname[FILE_NAM_LEN];      /* Full name of output file */
    char    errbuf[100];                 /* Buffer for error messages */
    int     FileOK, rc;
    v0_cosmos *v0;
    v0_multiplexor *vmx;
    time_t stime, etime;	
    clock_t start;
    double  secs, tics;	   
    SEND_QUEUE *sq;   /* working pointer into SendQueue   */

    sprintf(whoami, (char*)"%s: %s: ", progname, (char*)"MessageStacker");
    
    /* Change working directory to "V0FilesInDir" - where the files should be
     ************************************************************************/
    if ( chdir_ew( V0FilesInDir ) == -1 ) {
	logit( (char*)"e", (char*)"Error. Can't change working directory to %s\n Exiting.", V0FilesInDir );
	goto error;
    }
    if(Debug) logit( (char*)"e", (char*)"%sChanged working directory to %s\n ", whoami, V0FilesInDir );
	
    /* Start main export service loop for current connection
     ********************************************************/
    while( 1 ) {
	/* Tell the main thread we're ok
         ********************************/
	MessageStackerAlive = THREAD_ALIVE;

	/* Start of working loop over files.
         ************************************/
	while ((GetFileName (fname)) != 1) {    
	    time(&stime);
	    start = clock();
	    /* Open the file for reading only.  Since the file name
	       is in the directory we know the file exists, but it
	       may be in use by another process.  If so, fopen_excl()
	       will return NULL.  In this case, wait a second for the
	       other process to finish and try opening the file again.
	    ******************************************************/
	    if ( strcmp(fname,(char*)"core")==0 ) {
		logit((char*)"et", (char*)"%s Deleting core file <%s>; \n", whoami, fname );
		if ( remove( fname ) != 0) {
		    logit((char*)"et", (char*)"%s Cannot delete core file <%s>; exiting!", whoami, fname );
		    break;
		}
		continue;
	    }

	    try {
		vmx = new v0_multiplexor(fname);
	    
		logit((char*)"et", (char*)"*** Begin Processing file %s *** \n", fname);
		if(Debug) {
		    logit((char*)"e", (char*)"\n\n ************************************************************ \n");
		    logit((char*)"et", (char*)"\n *** Begin Processing file %s *** \n", fname);
		    logit((char*)"e", (char*)"\n ************************************************************ \n");
		}

		/* Prime the data buffers
                 *************************/
		nchans = 0;
		for (i=0; i < MAXCHANS; i++) {
		    chanbuf[i].bufptr = 0;
		    strcpy(chanbuf[i].sncl, (char*)" ");
		}

		/* Read the mseed file
                 ************************/
		NumOfTBufs = 0;
		FileOK = TRUE;
		
		try {
		    while ( (v0 = vmx->getNext()) != NULL ) {
			
			if ( v0->isDataless()) {
			    logit((char*)"et", (char*)"%s is dataless V0; skipping\n", fname);
			    break;
			}
			if (v0->getData() == NULL ) {
			    logit((char*)"et", (char*)"Could not process %s COSMOS V0 information\n", fname);
			    FileOK = FALSE;
			    continue;
			}
			if (v0->getStageIndex() != 0) {
			    logit((char*)"et", (char*)"Could not process %s COSMOS, not a stage V0 file\n", fname);
			    continue;
			}

			if (Debug) {
			    logit((char*)"e", (char*)"\nS_C_N_L: %s_%s_%s_%s  nsamp: %d samprate: %d\n", 
				  v0->getStationCode(), v0->getChannelCode(), 
				  v0->getNetworkCode(), DefLocationCode, 
				  v0->getNumSamples(), v0->getSampleRate());
			}
	                
			v0_to_tbuf(v0, &NumOfTBufs);
		    }
		}
		catch ( v0_cosmos::format_exception fme ) {
		    logit((char*)"et", (char*)"Format Exception Caught with file %s\n", fname); 
		    logit((char*)"e", (char*)"Skipping this file.\n"); 
		    FileOK = FALSE;		    
		}
	    } 
	    catch (v0_multiplexor::file_io_exception f) {
		logit((char*)"et", (char*)"Problem attempting to read from file %s\n", fname);
		logit((char*)"e", (char*)"Skipping this file.\n"); 
		FileOK = FALSE;
	    }
	    if (vmx != NULL) {
		delete vmx;
		vmx = NULL;
	    }
	    
	    /* Dispose of file
             *********************/
	    if (FileOK) {
		/* all ok; move the file to the output directory */

		sprintf (outFname, (char*)"%s/%s", V0FilesOutDir, fname);
		if ( rename( fname, outFname ) != 0 ) {
		    logit( (char*)"et", (char*)"Error moving %s: %s\n", fname, strerror(errno));
		    goto error;
		}
		if (Debug) logit( (char*)"e", (char*)"Moved %s to %s\n", fname, V0FilesOutDir);
	    }
	    else {      /* something blew up on this file. Preserve the evidence */
		logit((char*)"e",(char*)"Error processing file %s\n",fname);

		/* move the file to the error directory */
		sprintf (outFname, (char*)"%s/%s", V0FilesErrorDir, fname);

		if (rename (fname, outFname) != 0 ) {
		    logit( (char*)"et", (char*)"Fatal: Error moving %s: %s\n", fname, strerror(errno));
		    goto error;
		} 
		if(Debug) logit( (char*)"et", (char*)"Moved %s to %s\n", fname, V0FilesErrorDir);
		continue;
	    }
	        
	    time(&etime);
	    tics = (double)(clock() - start);
	    secs = tics / (double)CLOCKS_PER_SEC;
	    logit((char*)"et",(char*)"Done with file %s %.1f seconds.  %d TraceBufs\n",
		  fname, difftime(etime,stime), NumOfTBufs);
	    sleep_ew(100);
	}
	        
	sleep_ew(2000);

    } /* end of while */

    /* we're quitting
     *****************/
 error:
    MessageStackerStatus = THREAD_ERR; /* file a complaint to the main thread */
    KillSelfThread(); /* main thread will restart us */
    MessageStackerAlive = THREAD_ERR; /* Assume the kill worked. */

#ifndef _WINNT
    return (thr_ret)0;
#endif
}


/********************************************************************
 *  v0_to_tbuf converts COSMOS v0 to TraceBuf2 and                  *
 *  writes to queue with tracebufs starting on integer seconds.     *
 ********************************************************************/

static TracePacket trace_buffer;
int v0_to_tbuf(v0_cosmos *v0, int *tbufcount) 
{
    char            whoami[50];
    TRACE2_HEADER  *trace2_hdr;		/* ew trace header */
    int             i, j, k, offset, out_message_size;
    int*            inpTracePtr;
    char*           myTracePtr;
    char            sncl[20];
    int             NumOfElementsIv0ueue; /* number of messages in the queue  */
    const int       MSECS_PER_SEC = 1000;
    struct tm       startTM;
    long *data;    
    sprintf(whoami, (char*)"%s: %s: ", progname, (char*)"mseed_to_tbuf");

    /*** Pre-fill some of the tracebuf header ***/
    trace2_hdr = (TRACE2_HEADER *) &trace_buffer.trh;
    memset((void*) trace2_hdr, 0, sizeof(TRACE2_HEADER));
    trace2_hdr->pinno = 0;		/* Unknown item */
    trace2_hdr->samprate = v0->getSampleRate();
    strcpy(trace2_hdr->sta, v0->getStationCode());
    strcpy(trace2_hdr->net, v0->getNetworkCode());
    strcpy(trace2_hdr->chan, v0->getChannelCode());
    strcpy(trace2_hdr->loc, DefLocationCode);
    if (0 == strncmp(trace2_hdr->loc, (char*)"  ", 2) || 0 == strlen(trace2_hdr->loc))
	strcpy(trace2_hdr->loc,(char*)"--");
    sprintf(sncl, (char*)"%s_%s_%s_%s", trace2_hdr->sta,trace2_hdr->net,trace2_hdr->chan,trace2_hdr->loc);
	
#if defined( _SPARC )
    strcpy(trace2_hdr->datatype, (char*)"s4");
#elif defined( _INTEL )
    strcpy(trace2_hdr->datatype, (char*)"i4");
#endif
    trace2_hdr->version[0]=TRACE2_VERSION0;
    trace2_hdr->version[1]=TRACE2_VERSION1;
	
    /*** Move the data into the floating buffer. ***/
    k = -1;
    for(i=0; i < nchans; i++) {
	if (strcmp(chanbuf[i].sncl, sncl) == 0) k = i;
    }
    data = v0->getData();
    
    if (k == -1) {
	/*** New channel. Start floating buffer at next integer second. ***/
	k = nchans++;
	strcpy(chanbuf[k].sncl, sncl);
	if (chanbuf[k].bufLen < v0->getNumSamples()) {
	    chanbuf[k].bufLen = v0->getNumSamples();
	    chanbuf[k].buf = (long*)realloc(chanbuf[k].buf, sizeof(long) * v0->getNumSamples());
	    if (chanbuf[k].buf == NULL) {
		logit((char*)"et", (char*)"Error allocating channel buffer memory\n");
		exit(1);
	    }
	}

	offset = (v0->getStartMsecs() == 0) ? 0 :
	    (int)(((double)MSECS_PER_SEC - v0->getStartMsecs())*trace2_hdr->samprate)/ (double)MSECS_PER_SEC;
	for (i = offset, j = 0; i < v0->getNumSamples(); i++, j++) 
	    chanbuf[k].buf[j] = data[i];

	chanbuf[k].bufptr = j;

	// Prepare tm structure for gmtime_ew, which uses mktime()
	startTM.tm_sec = v0->getStartSecs();
	startTM.tm_min = v0->getStartMin();
	startTM.tm_hour = v0->getStartHour();
	startTM.tm_mday = v0->getStartMDay();
	startTM.tm_mon = v0->getStartMonth() - 1;
	startTM.tm_year = v0->getStartYear() - 1900;
	startTM.tm_wday = 0;  // ignored by mktime()
	startTM.tm_yday = 0;  // ignored by mktime()
	startTM.tm_isdst = 0;

	if (Debug) {
	    logit((char*)"et", (char*)"%sk: %d v0 start: %04d/%02d/%02d %02d:%02d:%02d.%03d\n",
		  whoami, k,
		  v0->getStartYear(), v0->getStartMonth(), 
		  v0->getStartMDay(), v0->getStartHour(), 
		  v0->getStartMin(), v0->getStartSecs(), 
		  v0->getStartMsecs());
	}
	
	chanbuf[k].starttime = timegm_ew(&startTM) + ((offset > 0) ? 1 : 0);
	if (Debug) {
	    char dbuf[24];
	    logit((char*)"et",(char*)"%sk: %d EW start: %s\n", whoami, k, datestr23(chanbuf[k].starttime, 
										  dbuf, 23));
	}
	
	chanbuf[k].endtime   = timegm_ew(&startTM) + (double)(v0->getNumSamples() - 1)/v0->getSampleRate();
	chanbuf[k].endtime   = chanbuf[k].endtime - 1;
	if(Debug) logit((char*)"et", (char*)"%sk: %d offset: %d numsamps: %d \n", whoami, k, offset, v0->getNumSamples()); 
    } else {
	for(i = 0, j = chanbuf[k].bufptr; i < v0->getNumSamples(); i++, j++) 
	    chanbuf[k].buf[j] = data[i];
	chanbuf[k].bufptr = j;
    }
	
    if (Debug) {
	trace2_hdr->starttime = (double) timegm_ew(&startTM) + ((double)(v0->getStartMsecs())/MSECS_PER_SEC);
	trace2_hdr->endtime = trace2_hdr->starttime +
	    (v0->getNumSamples() - 1) / v0->getSampleRate();
	logit((char*)"et", (char*)"%sRcvd from mSEED   %s_%s_%s_%s %d Time: %f  %f \n",  whoami,
	      trace2_hdr->sta, trace2_hdr->chan, trace2_hdr->net, trace2_hdr->loc, 
	      v0->getNumSamples(),
	      trace2_hdr->starttime, trace2_hdr->endtime);
    }
	
    // transfer data one second at a time, skipping the last fraction of second
    while (chanbuf[k].bufptr >= trace2_hdr->samprate) {
		
	trace2_hdr->nsamp     = trace2_hdr->samprate;
	trace2_hdr->starttime = chanbuf[k].starttime;
	chanbuf[k].endtime    = chanbuf[k].starttime + (double)(trace2_hdr->nsamp - 1)/trace2_hdr->samprate;
	trace2_hdr->endtime   = chanbuf[k].endtime;

	myTracePtr = &trace_buffer.msg[0] + sizeof(TRACE2_HEADER);        
	inpTracePtr = (int*) &chanbuf[k].buf[0];        
	
	/* Move the trace */
	memcpy( (void*)myTracePtr, (void*)inpTracePtr, trace2_hdr->nsamp*sizeof(long) ); 
    	
	out_message_size = sizeof(TRACE2_HEADER)+sizeof(int)*trace2_hdr->nsamp;
		
	if (out_message_size >= MaxMsgSize) {
	    logit((char*)"et", (char*)"%s Error - out_message_size (%d) >= MaxMsgSize (%d)\n", whoami, 
		  out_message_size, MaxMsgSize);
	    out_message_size = MaxMsgSize;
	}
		
	if(Debug) {
	    logit((char*)"et", (char*)"%sSENT to EARTHWORM %s_%s_%s_%s %d Time: %f  %f %f %d %d %d\n", whoami, 
		  trace2_hdr->sta, trace2_hdr->chan, 
		  trace2_hdr->net, trace2_hdr->loc, trace2_hdr->nsamp, 
		  trace2_hdr->starttime, trace2_hdr->endtime, trace2_hdr->endtime-trace2_hdr->starttime, 
		  out_message_size, chanbuf[k].bufptr, j-i);
	}
	/* transport it off to the queue */
	if ( queue_putmsg((long) out_message_size, (char*)&trace_buffer) != 0) {
	    logit((char*)"et", (char*)"%s Fatal Error sending trace via queue_putmsg()\n", whoami);
	    ShutMeDown = TRUE;
	    err_exit = V02WWS_DEATH_EW_PUTMSG;
	} else {
	    *tbufcount = *tbufcount + 1;
	    if ( Verbose==TRUE ) {
		fprintf(stderr, (char*)"SENT to EARTHWORM %s_%s_%s_%s %d Time: \n", 
			trace2_hdr->sta, trace2_hdr->chan, 
			trace2_hdr->net, trace2_hdr->loc, trace2_hdr->nsamp);
	    }
	}
		
	/* Reset the floating buffer */
	for (i = 0, j = trace2_hdr->nsamp; j < chanbuf[k].bufptr; i++, j++) 
	    chanbuf[k].buf[i] = chanbuf[k].buf[j];
	chanbuf[k].bufptr -= trace2_hdr->nsamp;
	chanbuf[k].starttime += 1.0;
    }
             
    return 0;
} 

/******************************************************************
 *          Move tracebufs from v0 files to memory queue        *
 ******************************************************************/
int queue_putmsg( long recsize, char* trace_buffer)
{
    char      whoami[50];
    MSG_LOGO  reclogo;                  /* logo of retrieved message             */
    int		  res, ret, sleep, min;
    int       NumOfTimesQueueLapped= 0; /* number of messages lost due to queue lap */
    int       NumOfElementsInQueue;     /* number of messages in the queue  */

    sprintf(whoami, (char*)"%s: %s: ", progname, (char*)"queue_putmsg");
    sleep = min = 0;
    RequestMutex();
    NumOfElementsInQueue = getNumOfElementsInQueue(&OutQueue);
    ReleaseMutex_ew();
    if(Debug) logit((char*)"et", (char*)"%sQueue has %d elements.  \n", whoami, NumOfElementsInQueue); 
    while(NumOfElementsInQueue > RingSize-10) {
	if(++sleep > 120) {
	    logit((char*)"et", (char*)"%sQueue may be stalled.  \n", whoami);
	    min++;
	    sleep = 0;
	}
	sleep_ew(1000);
	MessageStackerAlive = THREAD_ALIVE;
	RequestMutex();
	NumOfElementsInQueue = getNumOfElementsInQueue(&OutQueue);
	if(Debug) logit((char*)"et", (char*)"Queue has %d elements.  %d\n", NumOfElementsInQueue, sleep); 
	ReleaseMutex_ew();
    }
    if(sleep > 0) logit((char*)"et", (char*)"Queue was filled. Slept %d minutes %d sec.  \n", min/2, sleep); 

    /* Process retrieved msg (res==GET_OK,GET_MISS,GET_NOTRACK)
     ***********************************************************/
    trace_buffer[recsize] = '\0';
	
    reclogo.instid = InstId;
    reclogo.mod    = MyModId;
    reclogo.type   = TypeTrace2;
	
    /* put it into the 'to be shipped' queue */
    /* the thread SocketSender is in the biz of de-queueng and sending */
    RequestMutex();
    ret=enqueue( &OutQueue, trace_buffer, recsize, reclogo );
    ReleaseMutex_ew();
	
    if ( ret!= 0 ) {
	if (ret==-2) {  /* Serious: quit */
	    /* Currently, ev0ueue() in mem_circ_queue.c never returns this error. */
	    sprintf(errText,(char*)"internal queue error. Terminating.");
	    v02wws_status( TypeError, ERR_QUEUE, errText );
	    logit((char*)"et", (char*)"%s%s  \n", whoami, errText);
	    goto error;
	}
	if (ret==-1) {
	    sprintf(errText,(char*)"queue cannot allocate memory. Lost message.");
	    v02wws_status( TypeError, ERR_QUEUE, errText );
	    logit((char*)"et", (char*)"%s%s  \n", whoami, errText);
	}
	if (ret==-3  &&  LiveConnection) { /* Log only while client's connected */
	    /* Queue is lapped too often to be logged to screen.
	     * Log circular queue laps to logfile.
	     * Maybe queue laps should not be logged at all.   */
		
	    NumOfTimesQueueLapped++;
	    if (!(NumOfTimesQueueLapped % 5)) {
		logit((char*)"t", (char*)"%s Circular queue lapped 5 times. Messages lost.\n", whoami);
		if (!(NumOfTimesQueueLapped % 100)) {
		    logit( (char*)"et", (char*)"%s Circular queue lapped 100 times. Messages lost.\n", whoami);
		}
	    }
	}
    }
    return 0;

    /* we're quitting
     *****************/
 error:
    MessageStackerStatus = THREAD_ERR; /* file a complaint to the main thread */
    KillSelfThread(); /* main thread will restart us */
    MessageStackerAlive = THREAD_ERR; /* Assume the kill worked. */
}


/*****************************************************************************
 *  v02wws_config() processes command file(s) using kom.c functions;         *
 *                    exits if any errors are encountered.               *
 *****************************************************************************/
void v02wws_config( char *configfile )
{
    int      ncommand;     /* # of required commands you expect to process   */
    char     init[20];     /* init flags, one byte for each required command */
    int      nmiss;        /* number of required commands that were missed   */
    char    *com;
    char     processor[20];
    int      nfiles;
    int      success;
    int      i;
    char*    str;

    /* Set to zero one init flag for each required command
     *****************************************************/
    ncommand = 15;
    for( i=0; i<ncommand; i++ )  init[i] = 0;
    HBflag = 1;

    /* Set the default location code, which may be overridder by configuration */
    strcpy(DefLocationCode, (char*)"--");
    
    /* Open the main configuration file
     **********************************/
    nfiles = k_open( configfile );
    if ( nfiles == 0 ) {
	logit( (char*)"e",
	       (char*)"%s: Error opening command file <%s>; exiting!\n",
	       Argv0, configfile );
	exit( -1 );
    }

    /* Process all command files
     ***************************/
    while(nfiles > 0)   /* While there are command files open */
    {
	while(k_rd())    /* Read next line from active file  */
	{
	    com = k_str();   /* Get the first token from line */

	    /* Ignore blank lines & comments
             *******************************/
	    if( !com )           continue;
	    if( com[0] == '#' )  continue;

	    /* Open a nested configuration file
             **********************************/
	    if( com[0] == '@' ) {
		success = nfiles+1;
		nfiles  = k_open(&com[1]);
		if ( nfiles != success ) {
		    logit( (char*)"e",
			   (char*)"%s: Error opening command file <%s>; exiting!\n",
			   Argv0, &com[1] );
		    exit( -1 );
		}
		continue;
	    }
	    strcpy( processor, (char*)"v02wws_config" );

	    /* Process anything else as a command
             ************************************/
	    /*0*/    if( k_its((char*)"LogFile") ) {
		LogSwitch = k_int();
		init[0] = 1;
	    }
	    /*1*/    else if( k_its((char*)"MyModuleId") ) {
		str = k_str();
		if(str) strcpy( MyModName, str );
		init[1] = 1;
	    }
	    /*2*/    else if( k_its((char*)"RingName") ) {
		str = k_str();
		if(str) strcpy( RingName, str );
		init[2] = 1;
	    }
	    /*3*/    else if( k_its((char*)"HeartBeatInt") ) {
		HeartBeatInt = k_int();
		init[3] = 1;
	    }

	    /* IP Address to bind to socket
             ******************************/
	    /*4*/    else if( k_its((char*)"ServerIPAdr") ) {
		str = k_str();
		if(str) {
		    if( strlen(str) >= sizeof( ServerIPAdr ) ) {
			logit( (char*)"e",
			       (char*)"%s: ServerIPAdr <%s> too long; exiting!\n",
			       Argv0, str );
			exit( -1 );
		    }
		    strcpy( ServerIPAdr, str );
		}
		init[4] = 1;
	    }

	    /* Well-known server port number
             *******************************/
	    /*5*/    else if( k_its((char*)"ServerPort") ) {
		ServerPort = k_int();
		init[5] = 1;
	    }

	    /* Maximum size (bytes) for incoming/outgoing messages
             *****************************************************/
	    /*6*/    else if( k_its((char*)"MaxMsgSize") ) {
		MaxMsgSize = k_long();
		init[6] = 1;
	    }

	    /* Maximum number of messages in outgoing circular buffer
             ********************************************************/
	    /*7*/    else if( k_its((char*)"RingSize") ) {
		RingSize = k_long();
		if(RingSize < 50) RingSize = 50;
		init[7] = 1;
	    }

	    /* Interval (seconds) between alive msgs to client
             *************************************************/
	    /*8*/    else if( k_its((char*)"SendAliveInt") ) {
		SendAliveInt = k_int();
		init[8] = 1;
	    }

	    /* Text of alive message to client
             **********************************/
	    /*9*/   else if( k_its((char*)"SendAliveText") ) {
		str=k_str();
		if(str && strlen(str)<(size_t)MAX_ALIVE_STR) {
		    strcpy(SendAliveText,str);
		    init[9]=1;
		}
	    }

	    /* Interval (seconds) between alive msgs from client
             ***************************************************/
	    /*10*/   else if( k_its((char*)"RcvAliveInt") ) {
		RcvAliveInt = k_int();
		init[10] = 1;
	    }

	    /* Text of alive message from client
             ***********************************/
	    /*11*/   else if( k_its((char*)"RcvAliveText") ) {
		str=k_str();
		if(str && strlen(str)<(size_t)MAX_ALIVE_STR) {
		    strcpy(RcvAliveText,str);
		    init[11]=1;
		}
	    }
     
	    /* Input directory containing v0 files
             ****************************************/
	    /*12*/	else if( k_its((char*)"V0FilesInDir") ) {
		str = k_str();
		if(str) strcpy( V0FilesInDir, str );
		init[12] = 1;
	    }
     
	    /* Output directory for processed v0 files
             ********************************************/
	    /*13*/	else if( k_its((char*)"V0FilesOutDir") ) {
		str = k_str();
		if(str) strcpy( V0FilesOutDir, str );
		init[13] = 1;
	    }
     
	    /* Output directory for problem v0 files
             ******************************************/
	    /*14*/	else if( k_its((char*)"V0FilesErrorDir") ) {
		str = k_str();
		if(str) strcpy( V0FilesErrorDir, str );
		init[14] = 1;
	    }

	    /* Optional: location code
             ******************************************/
	    else if( k_its((char*)"LocationCode") ) {
		str = k_str();
		if(str) {
		    if (strlen(str) < TRACE2_LOC_LEN-1) {
			strcpy(DefLocationCode, str );
		    } else {	
			logit( (char*)"e",
			       (char*)"%s: LocationCode <%s> too long (max 2 char); exiting!\n",
			       Argv0, str );
			exit( -1 );
		    }
		}
	    }

	    /* Optional: Debug
             ******************/
	    else if ( k_its( (char*)"Debug" ) ) {
		Debug = 1;
		/* turn on the LogFile too! */
		LogSwitch = 1;
	    }  

	    /* Optional: Reset length of Send Queue
             **************************************/
	    else if( k_its((char*)"SendQueueLength") ) {
		int tmp;
		tmp = k_int();
		if( tmp < 1  ||  tmp > 254 ) {
		    logit((char*)"et", (char*)"%s: Invalid SendQueueLength %d (valid values: 1-254);"
			  " exiting!\n", Argv0, tmp );
		    exit( -1 );
		}
		SendQueueLen = (unsigned char)tmp;
	    }

	    /* Optional: Timeout in milliseconds for IP Socket routines
             **********************************************************/
	    else if(k_its((char*)"SocketTimeout") ) {
		if(RcvAliveInt == 0)
		{
		    SocketTimeoutLength = k_int();
		}
	    }

	    /* Optional: Turn on socket debug logging -> BIG LOG FILES!!
             ***********************************************************/
	    else if(k_its((char*)"SocketDebug") ) {
		SOCKET_ewDebug = k_int();
	    }

	    /* Optional cmd: Turn on heartbeat logging 
             ********************************************************/
	    else if( k_its((char*)"HBDebug") ) {
		HBflag = k_int();
	    }

	    /* Optional: Turn on debug logging -> BIG LOG FILES!!
             *****************************************************/
	    else if( k_its((char*)"Verbose") ) {
		Verbose = 1;
	    }



	    /* Unknown command
             *****************/
	    else {
		logit( (char*)"e", (char*)"%s: <%s> Unknown command in <%s>.\n",
		       Argv0, com, configfile );
		continue;
	    }

	    /* See if there were any errors processing the command
             *****************************************************/
	    if( k_err() ) {
		logit( (char*)"e",
		       (char*)"%s: Bad <%s> command for %s() in <%s>; exiting!\n",
		       Argv0, com, processor, configfile );
		exit( -1 );
	    }
	}  /* while reading from a file */
	nfiles = k_close();
    } /* while files are open */

    /* After all files are closed, check init flags for missed commands
     ******************************************************************/
    nmiss = 0;
    for( i=0; i<ncommand; i++ )  if( !init[i] ) nmiss++;
    if( nmiss ) {
	logit( (char*)"e", (char*)"%s: ERROR, no ", Argv0 );
	if ( !init[0] )   logit( (char*)"e", (char*)"<LogFile> "      );
	if ( !init[1] )   logit( (char*)"e", (char*)"<MyModuleId> "   );
	if ( !init[2] )   logit( (char*)"e", (char*)"<RingName> "     );
	if ( !init[3] )   logit( (char*)"e", (char*)"<HeartBeatInt> " );
	if ( !init[4] )   logit( (char*)"e", (char*)"<ServerIPAdr> "  );
	if ( !init[5] )   logit( (char*)"e", (char*)"<ServerPort> "   );
	if ( !init[6] )   logit( (char*)"e", (char*)"<MaxMsgSize> "   );
	if ( !init[7] )   logit( (char*)"e", (char*)"<RingSize> "     );
	if ( !init[8] )   logit( (char*)"e", (char*)"<SendAliveInt> " );
	if ( !init[9] )   logit( (char*)"e", (char*)"<SendAliveText>" );
	if ( !init[10] )  logit( (char*)"e", (char*)"<RcvAliveInt> "  );
	if ( !init[11] )  logit( (char*)"e", (char*)"<RcvAliveText>"  );
	if ( !init[12] ) logit( (char*)"e", (char*)"<V0FilesInDir>"   );
	if ( !init[13] ) logit( (char*)"e", (char*)"<V0FilesOutDir>"  );
	if ( !init[14] ) logit( (char*)"e", (char*)"<V0FilesErrorDir>");
	logit( (char*)"e", (char*)"command(s) in <%s>; exiting!\n", configfile );
	exit( -1 );
    }

    return;
}

/****************************************************************************
 *  v02wws_lookup( )   Look up important info from earthworm.h tables       *
 ****************************************************************************/
void v02wws_lookup( void )
{
    /* Look up keys to shared memory regions
     ***************************************/
    if( ( RingKey = GetKey(RingName) ) == -1 ) {
	logit( (char*)"e",
	       (char*)"%s:  Invalid ring name <%s>; exiting!\n",
	       Argv0, RingName);
	exit( -1 );
    }

    /* Look up installations of interest
     ***********************************/
    if( GetLocalInst( &InstId ) != 0 ) {
	logit( (char*)"e",
	       (char*)"%s: error getting local installation id; exiting!\n",
	       Argv0 );
	exit( -1 );
    }

    /* Look up modules of interest
     *****************************/
    if( GetModId( MyModName, &MyModId ) != 0 ) {
	logit( (char*)"e",
	       (char*)"%s: Invalid module name <%s>; exiting!\n",
	       Argv0, MyModName );
	exit( -1 );
    }

    /* Look up message types of interest
     ***********************************/
    if( GetType( (char*)"TYPE_HEARTBEAT", &TypeHeartBeat ) != 0 ) {
	logit( (char*)"e",
	       (char*)"%s: Invalid message type <TYPE_HEARTBEAT>; exiting!\n", Argv0 );
	exit( -1 );
    }
    if( GetType( (char*)"TYPE_ERROR", &TypeError ) != 0 ) {
	logit( (char*)"e",
	       (char*)"%s: Invalid message type <TYPE_ERROR>; exiting!\n", Argv0 );
	exit( -1 );
    }
    if ( GetType( (char*)"TYPE_TRACEBUF2", &TypeTrace2 ) != 0 ) {
	logit( (char*)"e",
	       (char*)"%s: Message type <TYPE_TRACEBUF2> not found in earthworm_global.d; exiting!\n", progname);
        exit(-1);
    } 
    return;
}

/***************************************************************************
 * v02wws_status() builds a heartbeat or error message & puts it into      *
 *                 shared memory.  Writes errors to log file & screen.     *
 ***************************************************************************/
void v02wws_status( unsigned char type, short ierr, char *note )
{
    MSG_LOGO    logo;
    char        msg[256];
    long        size;
    time_t      t;

    /* Build the message
     *******************/
    logo.instid = InstId;
    logo.mod    = MyModId;
    logo.type   = type;

    time( &t );

    if( type == TypeHeartBeat )
	sprintf( msg, (char*)"%ld %ld\n\0", (long) t, (long) MyPid);
    else if( type == TypeError )
    {
	sprintf( msg, (char*)"%ld %hd %s\n\0", (long) t, ierr, note);
	logit( (char*)"et", (char*)"%s(%s): %s\n", Argv0, MyModName, note );
    }
    size = strlen( msg );   /* don't include the null byte in the message */

    /* Write the message to shared memory
     ************************************/
    if( tport_putmsg( &Region, &logo, size, msg ) != PUT_OK )
    {
	if( type == TypeHeartBeat ) {
	    logit((char*)"et",(char*)"%s(%s):  Error sending heartbeat.\n",
		  Argv0, MyModName );
	}
	else if( type == TypeError ) {
	    logit((char*)"et", (char*)"%s(%s):  Error sending error:%d.\n",
		  Argv0, MyModName, ierr );
	}
    }

    return;
}

/******************************************************************
 * v02wws_shutdown()  take care of all final matters and exit     *
 ******************************************************************/
void v02wws_shutdown( int status )
{
    /* Stop threads
     *********************************/
    if( SocketSenderStatus   != THREAD_OFF ) (void)KillThread(TidSocketSend);
    if( SocketRecvStatus     != THREAD_OFF ) (void)KillThread(TidSocketRecv);
    if( MessageStackerStatus != THREAD_OFF ) (void)KillThread(TidStacker);

    /* Close sockets if they exist
     *****************************/
    if( SocketStatus == SOCK_CONNECTED )
	closesocket_ew( ActiveSocket,  SOCKET_CLOSE_IMMEDIATELY_EW );
    if( SocketStatus != SOCK_CLOSED )
	closesocket_ew( PassiveSocket, SOCKET_CLOSE_IMMEDIATELY_EW );

    /* Clean up filter, memory, etc
     ******************************/
    v02wws_free();
    tport_detach( &Region );
    exit( status );
}

/***************************************************************************
 * v02wws_free()  free all previously allocated memory                     *
 ***************************************************************************/
void v02wws_free( void )
{
    free (SSbinmsg);          /* SocketSender's binary-escaped msg buffer */
    free (SRmsg);             /* SocketRecv's incoming message buffer     */
    free (SQmsg);             /* SendQueue message storage                */
    free (SQ);                /* SendQueue parameter storage              */
    return;
}
