package org.trinet.waveserver.rt;
import java.util.*;
import java.io.*;
import java.net.*;
//import org.trinet.jasi.*;
import org.trinet.util.*;
import org.trinet.jasi.seed.SeedReader;


/* NOTE - A patch to fix verbose logging of data recovery failure:
        if (retVal == TN_FAILURE) retVal = TN_NODATA;   // TEMPORARY PATCH UNTIL DWS CODES FIXED
   is inserted in all doGetXXX() routines for CASE TN_TCP_ERROR_RESP messages values.
*/

/** Implementations of a WaveClient API adapted with modifications from the
* original "TRINET C++ WaveClient API" by Patrick Small. Methods allow the
* recovery of data from a list of server connections created using the
* addServer() methods. The getXXX() methods can then be used to retrieve data
* from the server list. Note that the getXXX() methods usually return the data
* into a user initialized input parameter object and the methods return a status
* code indicating success or failure. Some methods throw exceptions if an error
* is encountered during processing. Refer the the TriNet WaveClient/WaveServer
* documentation for further descriptions of the TCP message protocols and the
* roles served by the classes in this package.
*<p>
*<b> Note - some methods may alias input/output parameter object references,
*    when aliased, the data members reflect any external modification to the data.
*    If preservation of the original object data is required, as a precaution clone
*    the data of the external reference before modifying or setting the data values.
*</b>
*
* @see TrinetReturnCodes
* @see TrinetTCPMessageTypes
* @see TCPConn
* @see TCPMessage */
public class WaveClient extends org.trinet.util.WaveClient
    implements TrinetReturnCodes, TrinetTCPMessageTypes

{

/** Constant identifying rapid wave server default pool size in milliseconds */
    public static final long RWS_MAX_MILLISECS_IN_POOL = 8640000l; // 1 day

/** Constant identifying rapid wave server (memory based) default port number*/
    public static final int RWS_PORT_ID = 6500;

/** Constant identifying disk wave server (file based) default port number*/
    public static final int DWS_PORT_ID = 6501;

/** Constant identifying primary server role */
    public static final int TN_PRIMARY = 1;

/** Maximum message sequence number value. */
    public static final int DEFAULT_MAX_SEQNUM = 2048;

/** Default time to wait for a server response. */
    public static final int DEFAULT_MAX_TIMEOUT_MILLISECS = 30000;

/** Time to wait for a server response. */
    private int maxTimeout = DEFAULT_MAX_TIMEOUT_MILLISECS;

/** Default number of retries for a connection attempt. */
    public static final int DEFAULT_MAX_RETRIES = 3;

/** Number of reconnection attempts. */
    private int maxRetries = DEFAULT_MAX_RETRIES;

/** HashMap buckets. */
    static final int INITIAL_HASH_CAPACITY = 8;

/** Map storing server connections. */
    private Map servers = new Hashtable(INITIAL_HASH_CAPACITY);

/** True == truncate waveforms at first time gap. */
    private boolean truncateAtTimeGap = false;

/** True == verify waveforms. */
    private boolean verifyWaveforms = true;

/** TCPMessage number generator */
    private Counter sequenceNumberCounter;

/** Default constructor, default waveform modes: ignore time gaps and verifies waveforms.
*   Initializes request message sequence number to zero.
*/
    public WaveClient() {
        sequenceNumberCounter = new Counter(0, DEFAULT_MAX_SEQNUM);
    }

/** constructor sets client properties to the values parsed from the specified input property file.<p>
*<b>
* Valid properties:<br>
*</b>
* <pre>
* keyString                valueString<br>
* server            name:port name2:port name3:port ... ("\" at EOL continues theinput on next line)
* verifyWaveforms   true or false                       (default=true)
* truncateAtTimeGap true or false                       (default=false)
* maxTimeout        int >= 0                            (default=30000 ms, 0 => infinite, socket reads)
* maxRetries        int >= 0                            (default=3 reconnects attempted on socket error)
* </pre>
*   @exception WaveClientException input property file not found; invalid input property specification
*
*
*/
    public WaveClient(String propertyFileName) throws WaveClientException {
        String keyStr = null;
        String valueStr = null;
        int maxSequenceNumber = DEFAULT_MAX_SEQNUM;

        try {

            PropertyResourceBundle propRsrcBdl = new PropertyResourceBundle(new FileInputStream(propertyFileName));
            Enumeration keys = propRsrcBdl.getKeys();

            while (keys.hasMoreElements()) {

                keyStr = (String) keys.nextElement();
                valueStr = propRsrcBdl.getString(keyStr);

                if (keyStr.equalsIgnoreCase("servers")) {
                    StringTokenizer strToke = new StringTokenizer(valueStr, " :\t\n\r\f");
                    if ( strToke.countTokens()%2 != 0 )
                        throw new WaveClientException("invalid server property string; must be serverName:portNumber pairs");
                    while (strToke.hasMoreTokens()) {
                        String serverName = strToke.nextToken();
                        int serverPort = Integer.parseInt(strToke.nextToken());
                        this.addServer(serverName, serverPort);
                    }
                }

                else if (keyStr.equalsIgnoreCase("verifyWaveforms")) {
                    if (valueStr.equalsIgnoreCase("false")) verifyWaveforms = false;
                }

                else if (keyStr.equalsIgnoreCase("truncateAtTimeGap")) {
                    if (valueStr.equalsIgnoreCase("true")) truncateAtTimeGap = true;
                }

                else if (keyStr.equalsIgnoreCase("maxTimeout")) {
                    this.maxTimeout = Integer.parseInt(valueStr);
                }

                else if (keyStr.equalsIgnoreCase("maxRetries")) {
                    this.maxRetries = Integer.parseInt(valueStr);
                }
                else if (keyStr.equalsIgnoreCase("maxSequence")) {
                    maxSequenceNumber = Integer.parseInt(valueStr);
                }
                else {
                    throw new WaveClientException("Property resource file:" + propertyFileName + " has unknown key: " + keyStr);
                }
            }
            sequenceNumberCounter = new Counter(0, maxSequenceNumber);
        }
        catch (Exception ex)  {
            ex.printStackTrace();
            throw new WaveClientException("WaveClient constructor failed; propertyFile: " + propertyFileName  +
            " key: " + keyStr + " value: " + valueStr);
        }
    }


/** Default constructor, sets mode to ignore time gaps, returning all time segments.
*   Initializes request message sequence number to zero.
*   @param maxRetries          number of server reconnections to attempt if connection is broken or input timesout.
*   @param maxTimeoutMillisecs timeout milliseconds to wait for input to complete before attempting a reconnection.
*   @param verifyWaveforms     true == (default) check all packet headers for data consistency; false == accept packets as is.
*   @param truncateAtTimeGap   false == (default) return all time-series segments; false == return only time-series before 1st gap.
*   @exception java.lang.IllegalArgumentException input parameter < 0
*/
    public WaveClient(int maxRetries, int maxTimeoutMilliSecs, boolean verifyWaveforms, boolean truncateAtTimeGap) {
        if (maxRetries < 0 || maxTimeoutMilliSecs < 0)
            throw new IllegalArgumentException("WaveClient constructor invoked with input parameter < 0");

//  removed -  @param maxSequenceNumber   maximum TCPMessage sequence number for the generated request messages.
//        maxSequence < 0 throw new Illegal ...
//        sequenceNumberCounter = new Counter(0, maxSequenceNumber);

        this.maxRetries = maxRetries;
        this.maxTimeout = maxTimeoutMilliSecs;
        this.verifyWaveforms = verifyWaveforms;
        this.truncateAtTimeGap = truncateAtTimeGap;
        sequenceNumberCounter = new Counter(0, DEFAULT_MAX_SEQNUM);
    }

/** Closes socket connections for any server in the server list. */
    protected void finalize() {
        this.close();
    }

/** Returns the maximum TCPMessage number allowed for the generated request messages. */
    public int getMaxSequenceNumber() {
        return sequenceNumberCounter.rolloverValue;
    }

/** Returns the maximum number of re-connection, re-requests attempted on any server connection before failing. */
    public int getMaxRetries() {
        return maxRetries;
    }

/** Sets the maximum number of re-connection, re-requests attempted on any server connection error before failing.
*   @exception java.lang.IllegalArgumentException input parameter < 0
*/
    public int setMaxRetries(int retries) {
        if (maxRetries < 0 )
            throw new IllegalArgumentException("WaveClient.setMaxRetries(int) input parameter < 0");
        return(maxRetries = retries);
    }

/** Returns the maximum timeout milliseconds for input from a server responses to complete. */
    public int getMaxTimeoutMilliSecs() {
        return maxTimeout;
    }

/** Sets the maximum timeout milliseconds to allowed for input from a server responses to complete
*   before an java.io.InterruptedIOException is thrown by client socket.
*   @exception java.lang.IllegalArgumentException input parameter < 0
*/
    // Made public 7/17/2000 - DDG
    public int setMaxTimeoutMilliSecs(int millisecs) {
        if (millisecs < 0)
            throw new IllegalArgumentException("WaveClient.setMaxTimeoutMilliSecs(int) input parameter < 0");
        maxTimeout = millisecs;
//        Collection connList = getServerPollList();
        Collection connList = servers.values();
        Iterator iter = connList.iterator();
        while(iter.hasNext()){
            ((TCPConn) iter.next()).setDefaultTimeout(maxTimeout);
        }
				return(maxTimeout);
    }

/** Sets mode to truncate any requested time-series data at first time tear.
* Data requested by the getData(...) , getPacketData(...) , getJasiWaveformDataXXX(...) and getTimes(...) methods
* @see #getData(String, String, String, Date, int, Collection)
* @see #getData(Channel, TimeWindow, Waveform)
* @see #getJasiWaveformDataRaw(org.trinet.jasi.Waveform)
* @see #getJasiWaveformDataSeries(org.trinet.jasi.Waveform)
* @see #getPacketData(Channel, TimeWindow)
* @see #getTimes(Channel, Collection)
*/
    public void setTruncatetAtTimeGap(boolean value) {
        truncateAtTimeGap = value;
    }

/** Returns true if mode is set to truncate any requested time-series data at first time tear.
* Data requested by the getData(...) and getPacketData(...) and getTimes(...) methods.
* @see #getData(String, String, String, Date, int, Collection)
* @see #getData(Channel, TimeWindow, Waveform)
* @see #getJasiWaveformDataRaw(org.trinet.jasi.Waveform)
* @see #getJasiWaveformDataSeries(org.trinet.jasi.Waveform)
* @see #getPacketData(Channel, TimeWindow)
* @see #getTimes(Channel, Collection)
*/
    public boolean isTruncateAtTimeGap() {
        return truncateAtTimeGap;
    }

/** Returns true if mode is set to ignore all time gaps any requested time-series data, no data truncation.
* Data requested by the getData(...) and getPacketData(...) and getTimes(...) methods.
* @see #getData(String, String, String, Date, int, Collection)
* @see #getData(Channel, TimeWindow, Waveform)
* @see #getJasiWaveformDataRaw(org.trinet.jasi.Waveform)
* @see #getJasiWaveformDataSeries(org.trinet.jasi.Waveform)
* @see #getPacketData(Channel, TimeWindow)
* @see #getTimes(Channel, Collection)
*/
    public boolean isIgnoreTimeGaps() {
        return ! truncateAtTimeGap;
    }

/** Gets the verification state setting for all retrieved time series data.
* True implies data packets are checked for channel and time consistency.
* Default value == true;
*/
    public boolean isVerifyWaveforms() {
        return verifyWaveforms;
    }

/** Sets the default verification mode for any time series data retrieved from a server.
* True value implies all data packets are checked for channel and time consistency,
* which requires decoding data headers and adds overhead to the processing.
* If waveforms are reliably transferred from servers set state to false for faster response.
* Default value == true;
*/
    public boolean setWaveformVerify(boolean value) {
        return(verifyWaveforms = value);
    }

/** Returns the number of servers currently known by this WaveClient
*/
    public int numberOfServers() {
        return servers.size();

    }
/** Returns an array of Strings each string describes a server known by this WaveClient.
*/
    public String [] listServers() {
        String [] clients = new String [servers.size()];
        Set keySet = servers.keySet();
        Iterator iter = keySet.iterator();
        int index = 0;
        while (iter.hasNext()) {
            clients[index] = ((Client) iter.next()).toString();
            index++;
        }
        return clients;
    }

/** Adds the specified host and port number to the polled servers collection.
* Returns true if successful, false if error occurs, or server already is in list.
*/
    public boolean addServer(String host, int port) {
        return addServer(new Client(host, port)); // InetAddress ?
    }

/** Adds the host and port number specified in the input Client object to the servers collection.
* Servers in the collection are polled sequentially to locate the requested  data.
* If a server does not have the data, the next server in the iteration, if any is polled to satisfy a data request.
* Returns true if successful, false if error occurs, or server already is in list.
*/
    public boolean addServer(Client client) {

        boolean retVal = true;

        TCPConn conn = null;
        try {
            if (client == null)
                throw new NullPointerException("WaveClient.addServer() input Client parameter is null");
            if (servers.containsKey(client)) {
                System.err.println("Error WaveClient.addServer() : " + client.toString() + " already in list of servers");
                return false;
            }
            conn = new TCPConnClient(client);
            conn.setDefaultTimeout(maxTimeout);
        }
        catch (UnknownHostException ex) {
            System.err.println("Error WaveClient.addServer() Unknown host: " + client.host + " port: " + client.port);
            retVal = false;
        }
        catch (IOException ex) {
            System.err.println("Error WaveClient.addServer() IO error connecting to " + client.host + " port: " + client.port);
            System.err.println(ex.toString());
            retVal = false;
        }
        catch (NullPointerException ex) {
            System.err.println("WaveClient.addServer() input Client parameter is null");
            retVal = false;
        }
        catch (Exception ex) {
            System.err.println("Error WaveClient.addServer() caught exception connecting to " +client.host+ " port: " +client.port);
            System.err.println(ex.toString());
            ex.printStackTrace();
            retVal = false;
        }

        if (retVal == true) {
            if (servers.put(client, conn) != null) {
                System.err.println("Error WaveClient.addServer() : " + conn.toString() + " already in list of servers");
                retVal =  false;
            }
        }
        return retVal;
    }

/** Removes the specified host and port number from the polled server collection.
*   Returns true if successful, false if an error occurs or input parameter is null.
*/
    public boolean removeServer(String host, int port) {
        return removeServer(new Client(host, port)); // InetAddress ?
    }

/** Removes the host and port number specified in the specified Client object from the polled
* server collection and closes its socket connection.
* Returns true if successful, false if an error occurs or input parameter is null.
*/
    public boolean removeServer(Client client) {

        if (client == null) {
            System.err.println("WaveClient removeServer() input Client parameter is null");
            return false;
        }

        TCPConn conn = (TCPConn) servers.get(client);

        if (conn == null) {
            System.err.println("Error WaveClient.removeServer() server: " + client.toString() + " not found");
            return false;
        }
        else {
            conn.close();
            servers.remove(client);
        }
        return true;
    }

/** Closes all connections for the entire server list. */
    public void close() {
        // get all available servers
        Collection connList = servers.values();
        Iterator iter = connList.iterator();
        while (iter.hasNext()) {
            ((TCPConn) iter.next()).close();
        }
    }

/** Requests time series data for the specified data channel, start time and duration.
* If data is found the data is converted into FloatTimeSeries objects returned as elements added to the input collection.
* The input collection is cleared and new data appended.
* Returns TN_SUCCESS if successful, TN_NODATA if no data are found, else an TRINET error return code.
* <b>By default data packet headers are checked for data consistency, for faster response disable verification.</b>
* @see #setWaveformVerify(boolean)
* @see #isVerifyWaveforms()
* @see #getData(Channel, TimeWindow, Waveform)
* @exception java.lang.NullPointerException input collection parameter is null
* @exception java.lang.IllegalArgumentException Valid Channel/TimeWindow cannot be constructed from the input parameters.
*/
    public int getData(String net, String sta, String chn, Date beginTimestamp, int durationSeconds, Collection timeSeriesList) {
        if (timeSeriesList == null)
            throw new NullPointerException("WaveClient.getData() input FloatTimeSeries collection parameter is null");
        // Clear any existing FloatTimeSeries structures
        timeSeriesList.clear();

        Channel chan = null;
        try {
            chan = new Channel(net, sta, chn);
        }
        catch (IllegalArgumentException ex) {
            System.err.println("WaveClient.getData(): " + ex.getMessage());
            ex.fillInStackTrace();
            throw ex;
        }

        TimeWindow timeWindow = null;
        try {
            timeWindow = new TimeWindow( beginTimestamp, new Date( beginTimestamp.getTime() + (durationSeconds * 1000) ) );
        }
        catch (IllegalArgumentException ex) {
            System.err.println("WaveClient.getData(): " + ex.getMessage());
            ex.fillInStackTrace();
            throw ex;
        }

        Waveform waveform = new Waveform();

        // Retrieve the waveform data
        int retVal = getData(chan, timeWindow, waveform);
        if (retVal < TN_SUCCESS) {
            if (retVal == TN_NODATA) return retVal;
        }

        if (! waveform.trim(timeWindow)) {
            System.err.println("Error WaveClient.getData() Unable to trim waveform to time window");
            return TN_FAILURE;
        }
        chan.sampleRate =  waveform.getSampleRate();
        if (chan.sampleRate <= 0.0) {
            System.err.println("Error WaveClient.getData() waveform sample rate is undefined.");
            return TN_FAILURE;
        }

        List dataCountsList = new ArrayList();

        retVal = TN_SUCCESS;

        try {
            while (true) {
                Waveform waveformSegment = waveform.next();
                if (waveformSegment == null) {
                    if (timeSeriesList.size() > 0) retVal = TN_SUCCESS;
                    else {
                        retVal = TN_NODATA;
                        System.err.println("WaveClient.getData() end of data found for channel: " + chan.toString());
                    }
                    break;
                }
                if (! waveformSegment.getCounts(dataCountsList)) {
                    System.err.println("Error WaveClient.getData() Unable to get counts from waveform." );
                    retVal = TN_FAILURE;
                    break;
                }
                if (dataCountsList.size() != 1) {
                    System.err.println("Error WaveClient.getData() count list has " + dataCountsList.size() + " elements");
                    retVal = TN_FAILURE;
                    break;
                }
                // Note - Only one element, don't need iterator for list now.
                DataCounts dataCounts = (DataCounts) dataCountsList.get(0);
                if (dataCounts.dataList == null) {
                    System.err.println("Error WaveClient.getData() null data counts list.");
                    retVal = TN_FAILURE;
                    break;
                }
                // Changed block below to avoid extra object conversion re changes to Waveform - aww

                /*
                // Convert DataCount List of Number object counts to primitive array of counts
                    int numberOfSamples = dataCounts.dataList.size();
                    float [] samples = new float [numberOfSamples];
                    for (int index = 0; index < numberOfSamples; index++) {
                        samples[index] = ((Float) dataCounts.dataList.get(index)).floatValue();
                    }
                    FloatTimeSeries timeSeries =
                          new FloatTimeSeries( dataCounts.startTimestamp,
                                         new Date( dataCounts.startTimestamp.getTime() +
                                             (long) ( 1000. * (double)(numberOfSamples - 1) / waveform.getSampleRate() ) ),
                                             (float []) dataCounts.dataList.get(index);
                                             samples);
                */

                float [] samples =  (float []) dataCounts.dataList.get(0);
                int numberOfSamples = samples.length;
                FloatTimeSeries timeSeries =
                          new FloatTimeSeries( dataCounts.startTimestamp,
                                               new Date( dataCounts.startTimestamp.getTime() +
                                               (long) ( 1000. * (double)(numberOfSamples - 1) / waveform.getSampleRate() ) ),
                                               samples);
                timeSeriesList.add(timeSeries);
            }
        }
        catch (WaveformDataException ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
            retVal = TN_FAILURE;
        }
        catch (Exception ex) {
            System.err.println("Error WaveClient.getData() caught exception");
            ex.printStackTrace();
            retVal = TN_FAILURE;
        }
        return retVal;
    }

/** Requests a list of TimeRange objects describing the available timeseries for the specified channel string descriptors.
* Clears the input TimeRange list and appends the data returned in the server response, if any, to this list.
* Returns TN_SUCCESS if successful, TN_NODATA if no data is found, else a TRINET error return code.
* @exception java.lang.NullPointerException input collection parameter is null
* @exception java.lang.IllegalArgumentException Valid Channel cannot be constructed from the input parameters.
*/
    public int getTimes(String net, String sta, String chn, Collection timeRangeList) {
        if (timeRangeList == null)
            throw new NullPointerException("WaveClient.getTimes() input TimeRange collection parameter is null");
        // Clear any existing TimeRange structures
        timeRangeList.clear();

        Channel chan = null;
        try {
            chan = new Channel(net, sta, chn);
        }
        catch (IllegalArgumentException ex) {
            System.err.println("WaveClient.getTimes(): " + ex.getMessage());
            ex.fillInStackTrace();
            throw ex;
        }

        List timeWindowList = new ArrayList();
        int retVal = this.getTimes(chan, timeWindowList);
        if (retVal < TN_SUCCESS) return retVal;

        Iterator iter = timeWindowList.iterator();
        while (iter.hasNext()) {
            TimeWindow timeWindow = (TimeWindow) iter.next();
            timeRangeList.add(timeWindow.getTimeRange());
        }
        return retVal;
    }

/** Requests the sample rate for the seismic channel specified by the input string parameters.
* Returns the sample rate data returned in the server response.
* Returns Double.NaN if no data was available.
* @exception java.lang.IllegalArgumentException Valid Channel cannot be constructed from the input parameters.
* @exception WaveClientException error occurred recovering sample rate data
*/
    public double getSampleRate (String net, String sta, String chn ) throws WaveClientException {
        Channel chan = null;
        try {
            chan = new Channel(net, sta, chn);
        }
        catch (IllegalArgumentException ex) {
            System.err.println("WaveClient.getSampleRate(): " + ex.getMessage());
            ex.fillInStackTrace();
            throw ex;
        }

        int status = getSampleRate(chan);
        if (status < TN_SUCCESS) {
            if (status == TN_NODATA) return Double.NaN;
            else throw new WaveClientException("WaveClient.getSampleRate() returned error code: " + getStatusMessage(status));
        }
        return chan.sampleRate;
    }

/** Closes the connection object's socket and attempts to re-establish new socket connection with the same host and port.
* Returns true upon success, false otherwise.
*/
    private static boolean reconnect(TCPConn conn) {
        System.err.println("Attempting WaveClient.reconnect() to " + conn.socket.toString());
        boolean retVal = true;
        try {
            //done by reset: conn.close();
            conn.reset();
         }
         catch (IOException ex) {
             System.err.println("Error WaveClient.reconnect() io error reconnecting to " + conn.socket.toString());
             System.err.println(ex.getMessage());
             ex.printStackTrace();
             retVal = false;
         }
         catch (Exception ex) {
             System.err.println("Error WaveClient.reconnect() caught exception reconnecting to " + conn.socket.toString());
             System.err.println(ex.getMessage());
             ex.printStackTrace();
             retVal = false;
         }
         System.err.println("Reconnect successful");
         return retVal;
    }

/** Attempts to sends the input TCPMessage to the server host described by the TCPConn socket.
* Returns true if successful, false otherwise.
*/
    private static boolean sendRequest(TCPConnClient conn, TCPMessage requestMessage) {
        return conn.send(requestMessage);
    }

/** Returns a TCPMessage from the server host described by the TCPConn socket.
* If no response if completed by the specified timeout interval the request is aborted.
* Returns null if unsuccessful.
*/
    private static TCPMessage receiveResponse(TCPConnClient conn, int timeoutMilliSecs) {
        TCPMessage responseMessage = null;
        try {
            responseMessage = conn.receive(timeoutMilliSecs);
        }
        catch (TCPConnException ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (Exception ex) {
            System.err.println("Error WaveClient.receiveResponse() caught exception ");
            ex.printStackTrace();
        }
        return responseMessage;
    }

/** Requests data from the server host described by the input TCPConn via a requesting TCPMessage.
* If no response if completed by the set number of connection retries the request is aborted.
* Returns the server generated response TCPMessage, or null if unsuccessful.
*/
    private TCPMessage requestReply(TCPConnClient conn, TCPMessage requestMessage) {
        TCPMessage responseMessage = null;
        for (int retry = 0; retry <= maxRetries; retry++) {
            if (sendRequest(conn, requestMessage)) {
                responseMessage = receiveResponse(conn, maxTimeout);
                if (responseMessage == null) {
                    System.err.println( "Error WaveClient.requestReply() invalid response message" );
                }
                else break;  // success
            }
            else System.err.println("Error WaveClient.requestReply() Unable to send request message to server");

            // reset connection
            if (maxRetries > 0) {
                if (reconnect(conn)) continue; // retry again with a new request/response iteration
                System.err.println("Error WaveClient.requestReply() reconnection attempt: " + String.valueOf(retry) + " failed");
                break; // failure can't reconnect;
            }
        }
        return responseMessage;
    }

/** Returns the TCMessage constructed from the specified input values.
* @exception WaveClientException message construction failed.
*/
    private static TCPMessage prepareMessage(int messageType, int requestSequenceNumber, TrinetSerial [] fields)
                 throws WaveClientException {

        TCPMessage requestMessage = null;

        try {
            if (fields == null)
                throw new NullPointerException("WaveClient.prepareMessage(...) null fields [] input parameter.");
            int fieldsLength = fields.length;
            // Construct the get data request message, prepare the request parameters
            requestMessage = new TCPMessage(messageType, fieldsLength);
            requestMessage.appendField(requestSequenceNumber);
            for (int index = 0; index < fieldsLength; index++) {
                requestMessage.appendField(fields[index].toByteArray());
            }
        }
        catch (Exception ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
            throw new WaveClientException("Error WaveClient.prepareMessage() caught exception creating request message fields");
        }
        return requestMessage;
    }

/** Returns the reference to the input responseMessage TCPMessage reference.
* @exception WaveClientException input parameter response message is null.
* @exceptionWaveClientIdException response sequence number does not match the request sequence number.
* @exception java.io.IOException response message sequence number cannot be parsed.
*/
    private static TCPMessage verifyResponseMessage(TCPMessage responseMessage, int requestSequenceNumber)
                   throws IOException, WaveClientException {
        if (responseMessage == null) throw new WaveClientException("WaveClient.verifyResponseMessage() null message");
        // Verify the message sequence number
// must set the message iterator
        int responseSequenceNumber = responseMessage.nextFieldValueAsInt();
        if (responseSequenceNumber != requestSequenceNumber)
            throw new WaveClientIdException("Error WaveClient.verifyResponseMessage() response id does not match request id: "
                                             + responseSequenceNumber + " != " + requestSequenceNumber);
        return responseMessage;
    }

/** Generates a request message, requests a server reply, then checks the server reponse for validity.
* Returns the response message.
* @exception WaveClientException error processing message.
* @exception java.io.IOException connection error or error parsing serialized data stream.
*/
    private TCPMessage processMessage(TCPConnClient conn, int requestMessageType, TrinetSerial [] fields)
             throws IOException, WaveClientException {
        // Create a new message identifier
        int requestSequenceNumber = sequenceNumberCounter.plus();

        // Create a request message type
        TCPMessage requestMessage = prepareMessage(requestMessageType, requestSequenceNumber, fields);

        // Submit the request, check for response
        TCPMessage responseMessage = requestReply(conn, requestMessage);

        // Check validity of message, if no exception thrown, return response message data
        return verifyResponseMessage(responseMessage, requestSequenceNumber);
    }

/** Processes a "getData" request message type.
* Returns TN_SUCCESS if the server response provided data, else returns an error code.
*/
    private int doGetData(TCPConnClient conn, Channel chan, TimeWindow requestedTimeWindow, Collection dataSegmentList) {
        // Prepare the request parameters
        TrinetSerial [] fields = new TrinetSerial[2];
        fields[0] = chan;
        fields[1] = requestedTimeWindow;
        int retVal = TN_FAILURE;
        try {
            // Check validity of message and get its content: either error code or data
            TCPMessage responseMessage = processMessage(conn, TN_TCP_GETDATA_REQ, fields);
            switch (responseMessage.getMessageType()) {
                case TN_TCP_ERROR_RESP:
                    retVal = responseMessage.nextFieldValueAsInt();
                    if (retVal == TN_FAILURE) retVal = TN_NODATA;  // TEMPORARY PATCH UNTIL DWS CODES FIXED
                    break;
                case TN_TCP_GETDATA_RESP:
                    chan.sampleRate = responseMessage.nextFieldValueAsDouble ();
                    int numberOfSegments = responseMessage.nextFieldValueAsInt();
                    for (; numberOfSegments > 0; numberOfSegments--) {
                        dataSegmentList.add(new DataSegment(responseMessage.nextFieldValueAsBytes()));
                    }
                    retVal = TN_SUCCESS;
                    break;
                default:
                    System.err.println("Error WaveClient.doGetData() Unrecognized response message");
            }
        }
        catch (WaveClientException ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (IOException ex) {
            System.err.println("Error WaveClient.doGetData() Failure parsing response message");
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (Exception ex) {
            System.err.println("Error WaveClient.doGetData() caught exception.");
            ex.printStackTrace();
        }
        return retVal;
    }

/** Processes a "getSampleRate" request message type.
* Returns TN_SUCCESS if the server response provided data, else returns an error code.
*/
    private int doGetSampleRate(TCPConnClient conn, Channel chan) {
        // Prepare the request parameters
        TrinetSerial [] fields = new TrinetSerial[1];
        fields[0] = chan;
        int retVal = TN_FAILURE;
        try {
            // Check validity of message and get its content: either error code or data
            TCPMessage responseMessage = processMessage(conn, TN_TCP_GETRATE_REQ, fields);
            switch (responseMessage.getMessageType()) {
                case TN_TCP_ERROR_RESP:
                    retVal = responseMessage.nextFieldValueAsInt();
                    if (retVal == TN_FAILURE) retVal = TN_NODATA;  // TEMPORARY PATCH UNTIL DWS CODES FIXED
                    break;
                case TN_TCP_GETRATE_RESP:
                    chan.sampleRate = responseMessage.nextFieldValueAsDouble ();
                    retVal = TN_SUCCESS;
                    break;
                default:
                    System.err.println("Error WaveClient.doGetSampleRate() Unrecognized response message");
            }
        }
        catch (WaveClientException ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (IOException ex) {
            System.err.println("Error WaveClient.doGetSampleRate() Failure parsing response message");
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (Exception ex) {
            System.err.println("Error WaveClient.doGetSampleRate() caught exception.");
            ex.printStackTrace();
        }
        return retVal;
    }

/** Processes a "getChannels" request message type.
* Returns TN_SUCCESS if the server response provided data, else returns an error code.
*/
    private int doGetChannels(TCPConnClient conn, Collection channelList) {
        // Prepare the request parameters
        TrinetSerial [] fields = new TrinetSerial[0];
        int retVal = TN_FAILURE;
        try {
            // Check validity of message and get its content: either error code or data
            TCPMessage responseMessage = processMessage(conn, TN_TCP_GETCHAN_REQ, fields);
            switch (responseMessage.getMessageType()) {
                case TN_TCP_ERROR_RESP:
                    retVal = responseMessage.nextFieldValueAsInt();
                    if (retVal == TN_FAILURE) retVal = TN_NODATA;  // TEMPORARY PATCH UNTIL DWS CODES FIXED
                    break;
                case TN_TCP_GETCHAN_RESP:
                    int numberOfChannels = responseMessage.nextFieldValueAsInt();
                    for (; numberOfChannels > 0; numberOfChannels--) {
                        channelList.add(new Channel(responseMessage.nextFieldValueAsBytes()));
                    }
                    retVal = TN_SUCCESS;
                    break;
                default:
                  System.err.println("Error WaveClient.doGetChannels() Unrecognized response message");
            }
        }
        catch (WaveClientException ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (IOException ex) {
            System.err.println("Error WaveClient.doGetChannels() Failure parsing response message");
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (Exception ex) {
            System.err.println("Error WaveClient.doGetChannels() caught exception.");
            ex.printStackTrace();
        }
        return retVal;
    }

/** Processes a "getTimes" request message type.
* Returns TN_SUCCESS if the server response provided data, else returns an error code.
*/
    private int doGetTimes(TCPConnClient conn, Channel chan, Collection timeWindowList) {
        // Prepare the request parameters
        TrinetSerial [] fields = new TrinetSerial[1];
        fields[0] = chan;
        int retVal = TN_FAILURE;
        try {
            // Check validity of message and get its content: either error code or data
            TCPMessage responseMessage = processMessage(conn, TN_TCP_GETTIMES_REQ, fields);
            switch (responseMessage.getMessageType()) {
                case TN_TCP_ERROR_RESP:
                    retVal = responseMessage.nextFieldValueAsInt();
                    if (retVal == TN_FAILURE) retVal = TN_NODATA;  // TEMPORARY PATCH UNTIL DWS CODES FIXED
                    break;
                case TN_TCP_GETTIMES_RESP:
                    int numberOfWindows = responseMessage.nextFieldValueAsInt();
                    try {
                        for (; numberOfWindows > 0; numberOfWindows--) {
                            timeWindowList.add(new TimeWindow(responseMessage.nextFieldValueAsBytes()));
                        }
                        retVal = TN_SUCCESS;
                    }
                    catch (IllegalArgumentException ex) {
                        System.err.println("WaveClient.doGetTimes(): " + ex.toString());
                        retVal = TN_FAILURE;
                    }
                    break;
                default:
                    System.err.println("Error WaveClient.doGetTimes() Unrecognized response message");
            }
        }
        catch (WaveClientException ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (IOException ex) {
            System.err.println("Error WaveClient.doGetTimes() Failure parsing response message");
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (Exception ex) {
            System.err.println("Error WaveClient.doGetTimes() caught exception.");
            ex.printStackTrace();
        }
        return retVal;
    }

/** Checks the input TimeWindow collection for temporal integrity.
* @return false found an invalid TimeWindow
*/
    static boolean verifyTimeWindows(Collection timeWindowList) {
        boolean retVal = true;
        Iterator iter = timeWindowList.iterator();
        int offset = 0;
        while(iter.hasNext()) {
          TimeWindow timeWindow = (TimeWindow) iter.next();
          if (! timeWindow.isValid() ) {
            retVal = false;
            System.err.println("Warning WaveClient.verifyTimeWindows() Invalid time window detected at offset: " + offset);
            System.err.println(timeWindow.toString());
            break;
          }
          offset++;
        }
        return retVal;
    }

/** Error threshold seconds allowed between contiguous time series samples. */
    static double getEpsilon(double sampleRate) {
        return (1 + Waveform.TIME_EPSILON)/sampleRate;
    }

/** Merges contiguous TimeWindow data found in the input collection thereby reducing the total number of elements
* used to generate any data requests using this collection.
* Returns the revised collection.
*/
    static Collection mergeTimeWindows(Collection timeWindowList, double sampleRate) {
        if (timeWindowList.isEmpty()) return timeWindowList;
        TreeSet timeWindowSet = new TreeSet(timeWindowList);
        Iterator iter = timeWindowSet.iterator();
        Outer:
        while (iter.hasNext()) {                                                     // iterate over window set
            TimeWindow timeWindow = (TimeWindow) iter.next();                        // get first time window
            Inner:
            while (iter.hasNext()) {                                                 // look for another window, else done.
                TimeWindow timeWindowNext = (TimeWindow) iter.next();                // get next time window for comparison
                if (timeWindowNext.after(timeWindow)) {                              // next window temporally follows previous
                    if (timeWindow.timeGapSeconds(timeWindowNext) < getEpsilon(sampleRate)) { // small gap between
                        ((SimpleTimeRange) timeWindow.getTimeRange()).endTimestamp =
                                                    timeWindowNext.getEndTimestamp();// merge next time window into current
                        iter.remove();                                               // delete next window
                        continue Inner;                                              // loop for next window in set
                    }
                    else continue Outer;                                             // large gap keep both windows, get next pair
                }
                else {                                                               // Time windows overlap.
                    Date endTimeNext = timeWindowNext.getEndTimestamp();
                    if (endTimeNext.after(timeWindow.getEndTimestamp())){            // Partial overlap between windows
                        ((SimpleTimeRange) timeWindow.getTimeRange()).endTimestamp = endTimeNext; // merge next into current window
                    }
                    iter.remove();                                                   // delete next, complete or partial overlap
                    continue Inner;                                                  // loop for next window in set
                }
            }
            break Outer;                                                             // no more windows in set
        }
        // Clear out original collection and reload with new set contents
        timeWindowList.clear();
        iter = timeWindowSet.iterator();
        while (iter.hasNext()) {
            timeWindowList.add(iter.next());
        }
        return timeWindowList; // revised collection
    }

/** Returns the time gap seconds found between the dataSegmentNext start time and the dataSegmentLast end time.*/
    static double timeGapSeconds(DataSegment dataSegmentLast, DataSegment dataSegmentNext, double sampleRate) {
        return dataSegmentNext.getStartTimeSecs() - dataSegmentLast.getStartTimeSecs()
                         - (double) (dataSegmentLast.numberOfSamples - 1)/sampleRate;
    }

/** Returns the time gap seconds found between the input TimeWindow and the DataSegment start times.*/
    static double timeGapSeconds(TimeWindow timeWindow, DataSegment dataSegment) {
        return (double) ( dataSegment.startTimestamp.getTime() - timeWindow.getStartTimestamp().getTime() ) / 1000. ;
    }

/** Returns the time of the last sample in the input DataSegment.*/
    static Date endTime(DataSegment dataSegment, double sampleRate) {
        return new Date( dataSegment.startTimestamp.getTime() +
                         Math.round( 1000. * (double)(dataSegment.numberOfSamples - 1) / sampleRate ) );
    }

/** Checks the time spane of the input data collection, presumably returned by a server response message,
* against that of the input TimeWindow, presumably the requested time range.
* Returns true if the data times span a range equal to or greater than that of the TimeWindow.
*/
    static boolean haveRequestedData(TimeWindow timeWindow, Collection dataSegmentList, double sampleRate) {
        if (dataSegmentList.isEmpty()) return false;

        // Check for time tear between request window start and first data segment start
        Iterator iter = dataSegmentList.iterator();
        DataSegment dataSegment = (DataSegment) iter.next();
        if (dataSegment.startTimestamp.after(timeWindow.getStartTimestamp())) {
            if (timeGapSeconds(timeWindow, dataSegment) > getEpsilon(sampleRate)) {
                return false; // need more data at begining of window
            }
        }

        // Check for time tear between elements in the data segment list
        while (iter.hasNext()) {
            DataSegment dataSegmentNext = (DataSegment) iter.next();
            if( timeGapSeconds(dataSegment, dataSegmentNext, sampleRate) > getEpsilon(sampleRate) ) return false; // have time tear
            dataSegment = dataSegmentNext;
        }

        // Check for time tear between requested window end time and last data segment end time
        Date dataSegmentEndTimestamp = endTime(dataSegment, sampleRate);
        return (timeWindow.timeGapSeconds(dataSegmentEndTimestamp) > getEpsilon(sampleRate) ) ? true : false; // need data at end
    }

/** Returns a collection of minSEED packets, if any,  for the specified input channel descriptors, starting time, and duration.
*  Data returned by the server response is truncated at the first time gap if isTruncateAtTimeGap() == true.
*  @exception java.lang.IllegalArgumentException Valid Channel/TimeWindow cannot be constructed from the input parameters.
*  @exception WaveClientException error occurred during packet processing.
*/
    public List getPacketData(String net, String sta, String chn, Date beginTimestamp, int durationSeconds)
                                                                                        throws WaveClientException {
        Channel chan = null;
        try {
            chan = new Channel(net, sta, chn);
        }
        catch (IllegalArgumentException ex) {
            System.err.println("WaveClient.getPacketData(): " + ex.getMessage());
            ex.fillInStackTrace();
            throw ex;
        }

        TimeWindow timeWindow = null;
        try {
            timeWindow = new TimeWindow( beginTimestamp, new Date( beginTimestamp.getTime() + (durationSeconds * 1000) ) );
        }
        catch (IllegalArgumentException ex) {
            System.err.println("WaveClient.getPacketData(): " + ex.getMessage());
            ex.fillInStackTrace();
            throw ex;
        }

        return getPacketData(chan, timeWindow);
    }

/** Returns a collection of miniSEED packets, if any, for the specified input Channel and TimeWindow data.
*  Data returned by the server response is truncated at the first time gap if isTruncateAtTimeGap() == true.
*  Returns null if no data was found..
*  <b>By default data packet headers are checked for data consistency, for faster response disable verification.</b>
*  @see #setWaveformVerify(boolean)
*  @see #isVerifyWaveforms()
*  @see #getData(Channel, TimeWindow, Waveform)
*  @exception java.lang.NullPointerException null input parameter
*  @exception WaveClientException error occurred during processing of packet data.
*/
    public List getPacketData(Channel chan, TimeWindow timeWindow) throws WaveClientException {
        if (chan == null)
            throw new NullPointerException("WaveClient.getPacketData(Channel, TimeWindow) null Channel input parameter");
        if (timeWindow == null)
            throw new NullPointerException("WaveClient.getPacketData(Channel, TimeWindow) null TimeWindow input parameter");
        Waveform wave = new Waveform();
        int status = getData(chan, timeWindow, wave);
        if (status < TN_SUCCESS) {
            if(status == TN_NODATA) return null;
            else throw new WaveClientException("getPacketData() getData() error code: " + getStatusMessage(status));
        }
        return wave.getMiniSEEDPackets();
    }

/** Retrieves from the server hosts the time-series data for the channel and time specified in the input Waveform instance.
*  A WFSegment collection constructed from the retrieved MiniSEED packet data is assigned to the input wavefrom object.
*  Time-series data returned is truncated at the first time gap if isTruncateAtTimeGap() == true.
*  Returns TN_SUCCESS if data was found, else TN_NODATA or other TRINET error code.
*  <b>By default data packet headers are checked for data consistency, for faster response disable verification.</b>
*  @see #getJasiWaveformDataSeries(org.trinet.jasi.Waveform jasiWaveform)
*  @see #setWaveformVerify(boolean)
*  @see #isVerifyWaveforms()
*  @see #getData(Channel, TimeWindow, Waveform)
*  @see org.trinet.jasi.seed.SeedReader#getWFSegments(Collection)
*  @exception java.lang.NullPointerException input Waveform null or has null data member
*  @exception java.lang.IllegalArgumentException input Waveform channel or time data violate constraints
*/
    public int getJasiWaveformDataRaw(org.trinet.jasi.Waveform jasiWaveform) {
        if (jasiWaveform == null)
            throw new NullPointerException("WaveClient.getJasiWaveformDataRaw(jasi.Waveform) null input parameter");

	// Changed jasiWaveform.chan.channel to  jasiWaveform.chan.seedchan - DDG 9/7/2000
	// Channel chan = new Channel(jasiWaveform.chan.net, jasiWaveform.chan.sta, jasiWaveform.chan.seedchan);
        org.trinet.jasi.ChannelName channelId = jasiWaveform.getChannelObj().getChannelName();
        Channel chan = new Channel(channelId.getNet(), channelId.getSta(), channelId.getSeedchan());

        TimeWindow timeWindow =
            new TimeWindow( EpochTime.epochToDate(jasiWaveform.getEpochStart()),
                            EpochTime.epochToDate(jasiWaveform.getEpochEnd())
                          );

        Waveform wave = new Waveform();

        int status = getData(chan, timeWindow, wave);
        if (status < TN_SUCCESS) {
            if (status == TN_NODATA) return status;
            System.err.println("WaveClient.getJasiWaveformDataRaw() getData() error code: " + getStatusMessage(status));
            return status;
        }

        jasiWaveform.format.setValue(Waveform.MINISEED_TYPE);
        jasiWaveform.encoding.setValue(Waveform.STEIM1);

        jasiWaveform.setSampleRate(wave.getSampleRate());
        jasiWaveform.setSegmentList(new ArrayList(SeedReader.getWFSegments(wave.getMiniSEEDPackets())));

        return status;
    }

/** Retrieves from the server hosts the time-series data for the channel and time specified in the input Waveform instance.
*  A WFSegment collection constructed from a FloatTimeSeries collection derived from the original raw data packets (MiniSEED).
*  The original packet header format information is not preserved.
*  Time-series data returned is truncated at the first time gap if isTruncateAtTimeGap() == true.
*  Returns TN_SUCCESS if data was found, else TN_NODATA or other TRINET error code.
*  <b>By default data packet headers are checked for data consistency, for faster response disable verification.</b>
*  @see #getJasiWaveformDataRaw(org.trinet.jasi.Waveform jasiWaveform)
*  @see #setWaveformVerify(boolean)
*  @see #isVerifyWaveforms()
*  @see #getData(Channel, TimeWindow, Waveform)
*  @see org.trinet.jasi.SeedReader#getWFSegments(Collection)
*  @exception java.lang.NullPointerException input Waveform null or has null data member
*  @exception java.lang.IllegalArgumentException input Waveform channel or time data violate constraints
*/
    public int getJasiWaveformDataSeries(org.trinet.jasi.Waveform jasiWaveform) {
        if (jasiWaveform == null)
            throw new NullPointerException("WaveClient.JasiWaveformDataSeries(jasi.Waveform) null input parameter");

        ArrayList timeSeriesList = new ArrayList();

	// Changed jasiWaveform.chan.channel to  jasiWaveform.chan.seedchan - DDG 9/7/2000
	// int status = getData(jasiWaveform.chan.net, jasiWaveform.chan.sta, jasiWaveform.chan.seedchan,
        org.trinet.jasi.ChannelName channelId = jasiWaveform.getChannelObj().getChannelName();
        double startTime = jasiWaveform.getEpochStart();
        double endTime = jasiWaveform.getEpochEnd();
        int status = getData(channelId.getNet(), channelId.getSta(), channelId.getSeedchan(),
                             EpochTime.epochToDate(startTime), (int) Math.round(endTime - startTime),
                             timeSeriesList);
        if (status < TN_SUCCESS) {
            if (status == TN_NODATA) return status;
            System.err.println("WaveClient.getJasiWaveformDataSeries() getData() error code: " + getStatusMessage(status));
            return status;
        }
        int numberOfElements = timeSeriesList.size();
        jasiWaveform.setSegmentList(new ArrayList(numberOfElements));
        if (numberOfElements <= 0) return status;

        Iterator iter = timeSeriesList.iterator();
        final int BYTES_PER_SAMPLE = 4;
        org.trinet.jasi.WFSegment wfSeg = null;
        while (iter.hasNext()) {
            FloatTimeSeries ts = (FloatTimeSeries) iter.next();
            wfSeg = new org.trinet.jasi.WFSegment();
            wfSeg.bytesPerSample = BYTES_PER_SAMPLE;
            wfSeg.setChannelObj(jasiWaveform.getChannelObj());

            wfSeg.setStart(EpochTime.dateToEpoch(ts.startTimestamp));
            wfSeg.setEnd(EpochTime.dateToEpoch(ts.endTimestamp));
            //deprecated  wfSeg.lenSecs = wfSeg.tend - wfSeg.tstart;
            //deprecated  wfSeg.length = ts.samples.length * BYTES_PER_SAMPLE;

            wfSeg.samplesExpected = ts.samples.length;
            wfSeg.setSampleInterval(ts.getSampleRate()); // wfSeg.lenSecs/(wfSeg.getSampleCount() - 1));
            wfSeg.setTimeSeries(ts.samples);
            jasiWaveform.getSegmentList().add(wfSeg);
        }

        if (wfSeg != null) jasiWaveform.setSampleRate(Math.rint(1.0/wfSeg.getSampleInterval()));
        jasiWaveform.format.setValue(Waveform.MINISEED_TYPE);
        jasiWaveform.encoding.setValue(Waveform.STEIM1);

        return status;
    }

/** Requests timeseries data for the specified input Channel and TimeWindow time range.
*  Inputs the data returned in the server response, if any, to the input Waveform object.
*  The passed Waveform object should be constructed with the default Waveform constructor.
*  Trims the returned data at the first time gap if isTruncateAtTimeGap() == true.
*  Returns TN_SUCCESS if successful, else TN_NODATA, or an error code.
*  <b>By default data packet headers are checked for data consistency, for faster response disable verification.</b>
*  @see #setWaveformVerify(boolean)
*  @see #isVerifyWaveforms()
*  @exception java.lang.NullPointerException null input parameter
*/
    public int getData(Channel chan, TimeWindow timeWindow, Waveform waveform) {
        if (chan == null)
            throw new NullPointerException("WaveClient.getData(Channel, TimeWindow, Waveform) null Channel parameter");
        if (timeWindow == null)
            throw new NullPointerException("WaveClient.getData(Channel, TimeWindow, Waveform) null TimeWindow parameter");
        if (waveform == null)
            throw new NullPointerException("WaveClient.getData(Channel, TimeWindow, Waveform) null Waveform parameter");

        int retVal = TN_SUCCESS;

        Collection dataSegmentList = new TreeSet();

        // Request all available data from each server
//        Collection connList = getServerPollList(timeWindow.getStartTimestamp().getTime());
        Collection connList = servers.values();
        Iterator iter = connList.iterator();

        while (iter.hasNext()) {
            TCPConnClient conn = (TCPConnClient) iter.next();
            System.err.println("WaveClient getData() Polling server: " + conn.socket.toString());
            retVal = doGetData(conn, chan, timeWindow, dataSegmentList);

            if (retVal < TN_SUCCESS && retVal != TN_NODATA ) {
                System.err.println("WaveClient.getData() doGetData returned error code:" + getStatusMessage(retVal));
//              return retVal; // ignore error, continue iterating for now.
            }
            else System.err.println("WaveClient doGetData() returned status code:" + getStatusMessage(retVal));

          // The following ordering of the packets assumes:
          //   1) Multiple servers may have data packets for the specified channel.
          //   2) The SEED packets for a particular time window of data are identical across all wave servers.
          //   3) Each wave server could be missing one or more packets that other servers may possess.
          // Check if the request has been satisfied
            if (haveRequestedData(timeWindow, dataSegmentList, chan.sampleRate)) break; // done with request
            else System.err.println("WaveClient don't have all requested data: " + conn.socket.toString());
        }

        if (dataSegmentList.isEmpty()) {
            return TN_NODATA;
        }

//        dataSegmentList = new ArrayList(dataSegmentList);  // pass as defined, a Set, or convert to a List?

        try {
            waveform.setData(dataSegmentList, verifyWaveforms); // set verifyWaveforms false to speed up transfer
        }
        catch (WaveformDataException ex) {
            System.err.println("Error WaveClient.getData() Unable to construct modify waveform object");
            System.err.println(ex.getMessage());
            ex.printStackTrace();
            return TN_FAILURE;
        }
        catch (Exception ex) {
            System.err.println("Error WaveClient.getData() caught exception");
            ex.printStackTrace();
            return TN_FAILURE;
        }

        // Trim to the first data include time tear if necessary
        if (isTruncateAtTimeGap() && waveform.hasTimeGaps() ) {
            waveform.trimAtFirstGap();
        }
        return TN_SUCCESS;
    }

/** Requests the sample rate for the specified input seismic Channel.
* Sets the Channel sample rate to the data returned in the server response, if any.
* Returns TN_SUCCESS if successful, else TN_NODATA, or an error code.
* @exception java.lang.NullPointerException null input parameter
*/
    public int getSampleRate(Channel chan) {
        if (chan == null)
            throw new NullPointerException("WaveClient.getSampleRate(Channel) null input Channel reference");

        int retVal = TN_SUCCESS;

        chan.sampleRate = 0.0;

        // Request all available data from each server
//        Collection connList = getServerPollList();
        Collection connList = servers.values();
        Iterator iter = connList.iterator();

        while (iter.hasNext()) {
            retVal = doGetSampleRate((TCPConnClient) iter.next(), chan);
            if (retVal == TN_SUCCESS) return TN_SUCCESS;
        }
        return retVal; // TN_NODATA;
    }

/** Requests a list of Channel objects describing known seismic channels for which timeseries data is available.
* Clears the input Channel list and appends the data returned in the server response, if any, to this list.
* Returns TN_SUCCESS if successful, else TN_NODATA, or an error code.
* @exception java.lang.NullPointerException null input parameter
*/
    public int getChannels(Collection channelList) {
        if (channelList == null)
            throw new NullPointerException("WaveClient.getChannels(Collection) null input collection reference");

        int retVal = TN_SUCCESS;

        // Clear out any existing channels in the list
        channelList.clear();

        TreeSet channelSet = new TreeSet();

        // Request all available data from each server
//        Collection connList = getServerPollList();
        Collection connList = servers.values();
        Iterator iter = connList.iterator();

        while (iter.hasNext()) {
            retVal = doGetChannels((TCPConnClient) iter.next(), channelSet);
            if (retVal < TN_SUCCESS && retVal != TN_NODATA ) {
                System.err.println("WaveClient.getChannels() doGetChannels returned error code:" + getStatusMessage(retVal));
//              return retVal; // continue for now.
            }
        }
        if (channelSet.isEmpty()) return TN_NODATA;

        iter = channelSet.iterator();
        while (iter.hasNext()) {
           channelList.add(iter.next());
        }
        return TN_SUCCESS;
    }

/** Requests a list of TimeWindow objects describing the available timeseries for the input seismic Channel.
* Clears the input TimeWindow list and appends the data returned in the server response, if any, to this list.
* Returns TN_SUCCESS if successful, else TN_NODATA, or an error code.
* @exception java.lang.NullPointerException null input parameter
*/
    public int getTimes(Channel chan, Collection timeWindowList) {
        if (chan == null)
            throw new NullPointerException("WaveClient.getTimes(Channel, Collection) null input Channel reference");
        if (timeWindowList == null)
            throw new NullPointerException("WaveClient.getTimes(Collection) null input collection reference");

        // Clear out the existing time windows
        timeWindowList.clear();
        // Request all available data from each server
//        Collection connList = getServerPollList();
        Collection connList = servers.values();
        Iterator iter = connList.iterator();
        int retVal = TN_SUCCESS;
        while (iter.hasNext()) {
            TCPConnClient conn = (TCPConnClient) iter.next();
            retVal = doGetTimes(conn, chan, timeWindowList);
            if (retVal < TN_SUCCESS && retVal != TN_NODATA ) {
                System.err.println("WaveClient.getTimes() doGetTimes returned error code:" + getStatusMessage(retVal));
//              return retVal; // continue for now.
            }
        }
        if (timeWindowList.isEmpty()) return TN_NODATA;

        // Verify that the list of time windows is valid
        if (! verifyTimeWindows(timeWindowList)) {
            System.err.println( "Error WaveClient.getTimes() time window list failed verification");
            return TN_FAILURE;
        }
        // Retrieve the sample rate for this channel
        if (getSampleRate(chan) < TN_SUCCESS) {
           System.err.println("Error WaveClient.getTimes() cannot retrieve sample rate");
           return TN_FAILURE;
        }
        // Merge adjacent time windows using sample rate
        mergeTimeWindows(timeWindowList, chan.sampleRate);

        // Trim to the first time window if necessary
        if (isTruncateAtTimeGap() && (timeWindowList.size() > 1)) {
           iter = timeWindowList.iterator();
           TimeWindow timeWindow = (TimeWindow) iter.next();
           timeWindowList.clear();
           timeWindowList.add(timeWindow);
        }
        return TN_SUCCESS;
    }

/** Print to System.err the Trinet status code descriptive message. */
    public static String getStatusMessage(int status) {
        StringBuffer sb = new StringBuffer(132);
        sb.append("TN status message: ");
        switch (status) {
            case TN_SUCCESS    :
                sb.append("True/Success");
                break;
            case TN_FAILURE    :
                sb.append("False/Failure");
                break;
            case TN_EOF        :
                sb.append("End of File");
                break;
            case TN_SIGNAL     :
                sb.append("Signal");
                break;
            case TN_NODATA     :
                sb.append("No Data");
                break;
            case TN_NOTVALID   :
                sb.append("Not Valid");
                break;
            case TN_TIMEOUT    :
                sb.append("Timeout");
                break;
            case TN_BEGIN      :
                sb.append("Begin");
                break;
            case TN_END        :
                sb.append("End");
                break;
            case TN_PARENT     :
                sb.append("Parent");
                break;
            case TN_CHILD      :
                sb.append("Child");
                break;
            case TN_FAIL_WRITE :
                sb.append("Write Failure");
                break;
            case TN_FAIL_READ  :
                sb.append("Read Failure");
                break;
            case TN_BAD_SAMPRATE :
                sb.append("Bad sample rate");
                break;
            default:
                sb.append("Unknown return code: " + String.valueOf(status));
        }
        return sb.toString();
    }

/* Returns a List of the server connections in preferred polling order
* to satify a data request.
* @see #getRole()
*/
    List getServerPollList() {
        return getServerPollList(System.currentTimeMillis());
    }

/* Returns a List of the server connections in preferred polling order
* to satify a data request.
* @see #getRole()
*/
    List getServerPollList(long startTimeMilliSeconds) {
        // get all available servers
        Collection connList = servers.values();
        List serverPollList = new ArrayList(connList.size());

        Iterator iter = connList.iterator();
        while (iter.hasNext()) {                                   // puts both the dws and rws of primary at beginning of list
            TCPConnClient connClient = (TCPConnClient) iter.next();
            if ( getRole(connClient) == TN_PRIMARY) {
                if ( (System.currentTimeMillis() - startTimeMilliSeconds) < RWS_MAX_MILLISECS_IN_POOL) { // put rws first
                    if (connClient.getPort() == RWS_PORT_ID) serverPollList.add(0, connClient);
                    else serverPollList.add(connClient);
                }
                else {                                                                                     // put dws first
                    if (connClient.getPort() == DWS_PORT_ID) serverPollList.add(0, connClient);
                    else serverPollList.add(connClient);
                }
                iter.remove();
            }
        }
        serverPollList.addAll(connList);
        return serverPollList;
    }

/**
* Returns the role of the server specified by the input TCPConnClient object.
* The role is used by the WaveClient to determine the order of preferred
* polling of the server hosts to satify a data request.
*/
    public int getRole(TCPConnClient conn) {
        if (doGetRole(conn) == TN_PRIMARY) return 1;
        else return 1;
    }

/* Processes a "getRole" request message type.
* Returns the server preferred role, else returns an error code.
*/
    private int doGetRole(TCPConnClient conn) {
       return 0;
    }

/* comment out for now until implementation by server on rt systems
        // Prepare the request parameters
        TrinetSerial [] fields = new TrinetSerial[0];
        int retVal = TN_FAILURE;
        try {
            // Check validity of message and get its content: either error code or data
            TCPMessage responseMessage = processMessage(conn, TN_TCP_GETROLE_REQ, fields);
            switch (responseMessage.getMessageType()) {
                case TN_TCP_ERROR_RESP:
                case TN_TCP_GETROLE_RESP:
                    retVal = responseMessage.nextFieldValueAsInt();
                    if (retVal == TN_FAILURE) retVal = TN_NODATA;  // TEMPORARY PATCH UNTIL DWS CODES FIXED
                    break;
                default:
                    System.err.println("Error WaveClient.doGetRole() Unrecognized response message");
            }
        }
        catch (WaveClientException ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (IOException ex) {
            System.err.println("Error WaveClient.doGetRole() Failure parsing response message");
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (Exception ex) {
            System.err.println("Error WaveClient.doGetRole() caught exception.");
            ex.printStackTrace();
        }
        return retVal;
    }
*/
}
