package org.trinet.util;

import java.util.Hashtable;
import java.util.Arrays;

/** Gaussian distribution function as described by a "z-table".
Given a Gauss (Normal) distribution, this function describes the fraction of
the area under the
curve that is above or below the given standard deviation from the mean.
For example: at 1 std dev from the mean  0.1587 is above and 0.8413 is below.
A negative value of std-dev just reverses the values of above and below.
<tt>
           |<- mean
          /^\
        /  |  \ |<- one std dev (X's represent area "above")
      /    |    \
    /      |    X \
_ /        |    X X |__
</tt>
Created to support the Chauvenet's Criterion. */

public class Ztable {

// std-dev score
static final double sDev[] = {
0.00, 0.01, 0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.10, 0.11,
0.12, 0.13, 0.14, 0.15, 0.16, 0.17, 0.18, 0.19, 0.20, 0.21, 0.22, 0.23,
0.24, 0.25, 0.26, 0.27, 0.28, 0.29, 0.30, 0.31, 0.32, 0.33, 0.34, 0.35,
0.36, 0.37, 0.38, 0.39, 0.40, 0.41, 0.42, 0.43, 0.44, 0.45, 0.46, 0.47,
0.48, 0.49, 0.50, 0.51, 0.52, 0.53, 0.54, 0.55, 0.56, 0.57, 0.58, 0.59,
0.60, 0.61, 0.62, 0.63, 0.64, 0.65, 0.66, 0.67, 0.68, 0.69, 0.70, 0.71,
0.72, 0.73, 0.74, 0.75, 0.76, 0.77, 0.78, 0.79, 0.80, 0.81, 0.82, 0.83,
0.84, 0.85, 0.86, 0.87, 0.88, 0.89, 0.90, 0.91, 0.92, 0.93, 0.94, 0.95,
0.96, 0.97, 0.98, 0.99, 1.00, 1.01, 1.02, 1.03, 1.04, 1.05, 1.06, 1.07,
1.08, 1.09, 1.10, 1.11, 1.12, 1.13, 1.14, 1.15, 1.16, 1.17, 1.18, 1.19,
1.20, 1.21, 1.22, 1.23, 1.24, 1.25, 1.26, 1.27, 1.28, 1.29, 1.30, 1.31,
1.32, 1.33, 1.34, 1.35, 1.36, 1.37, 1.38, 1.39, 1.40, 1.41, 1.42, 1.43,
1.44, 1.45, 1.46, 1.47, 1.48, 1.49, 1.50, 1.51, 1.52, 1.53, 1.54, 1.55,
1.56, 1.57, 1.58, 1.59, 1.60, 1.61, 1.62, 1.63, 1.64, 1.65, 1.66, 1.67,
1.68, 1.69, 1.70, 1.71, 1.72, 1.73, 1.74, 1.75, 1.76, 1.77, 1.78, 1.79,
1.80, 1.81, 1.82, 1.83, 1.84, 1.85, 1.86, 1.87, 1.88, 1.89, 1.90, 1.91,
1.92, 1.93, 1.94, 1.95, 1.96, 1.97, 1.98, 1.99, 2.00, 2.01, 2.02, 2.03,
2.04, 2.05, 2.06, 2.07, 2.08, 2.09, 2.10, 2.11, 2.12, 2.13, 2.14, 2.15,
2.16, 2.17, 2.18, 2.19, 2.20, 2.21, 2.22, 2.23, 2.24, 2.25, 2.26, 2.27,
2.28, 2.29, 2.30, 2.31, 2.32, 2.33, 2.34, 2.35, 2.36, 2.37, 2.38, 2.39,
2.40, 2.41, 2.42, 2.43, 2.44, 2.45, 2.46, 2.47, 2.48, 2.49, 2.50, 2.51,
2.52, 2.53, 2.54, 2.55, 2.56, 2.57, 2.58, 2.59, 2.60, 2.61, 2.62, 2.63,
2.64, 2.65, 2.66, 2.67, 2.68, 2.69, 2.70, 2.71, 2.72, 2.73, 2.74, 2.75,
2.76, 2.77, 2.78, 2.79, 2.80, 2.81, 2.82, 2.83, 2.84, 2.85, 2.86, 2.87,
2.88, 2.89, 2.90, 2.91, 2.92, 2.93, 2.94, 2.95, 2.96, 2.97, 2.98, 2.99,
3.00, 3.01, 3.02, 3.03, 3.04, 3.05, 3.06, 3.07, 3.08, 3.09, 3.10, 3.11,
3.12, 3.13, 3.14, 3.15, 3.16, 3.17, 3.18, 3.19, 3.20, 3.21, 3.22, 3.23,
3.24, 3.25, 3.26, 3.27, 3.28, 3.29, 3.30, 3.31, 3.32, 3.33, 3.34, 3.35,
3.36, 3.37, 3.38, 3.39, 3.40, 3.41, 3.42, 3.43, 3.44, 3.45, 3.46, 3.47,
3.48, 3.49, 3.50};

static final double maxSdev = 3.50;

// area above the sDev
static final double below[]= {
0.5000, 0.5040, 0.5080, 0.5120, 0.5160, 0.5199, 0.5239, 0.5279, 0.5319, 0.5359, 0.5398, 0.5438,
0.5478, 0.5517, 0.5557, 0.5596, 0.5636, 0.5675, 0.5714, 0.5753, 0.5793, 0.5832, 0.5871, 0.5910,
0.5948, 0.5987, 0.6026, 0.6064, 0.6103, 0.6141, 0.6179, 0.6217, 0.6255, 0.6293, 0.6331, 0.6368,
0.6406, 0.6443, 0.6480, 0.6517, 0.6554, 0.6591, 0.6628, 0.6664, 0.6700, 0.6736, 0.6772, 0.6808,
0.6844, 0.6879, 0.6915, 0.6950, 0.6985, 0.7019, 0.7054, 0.7088, 0.7123, 0.7157, 0.7190, 0.7224,
0.7257, 0.7291, 0.7324, 0.7357, 0.7389, 0.7422, 0.7454, 0.7486, 0.7517, 0.7549, 0.7580, 0.7611,
0.7642, 0.7673, 0.7703, 0.7734, 0.7764, 0.7794, 0.7823, 0.7852, 0.7881, 0.7910, 0.7939, 0.7967,
0.7995, 0.8023, 0.8051, 0.8078, 0.8106, 0.8133, 0.8159, 0.8186, 0.8212, 0.8238, 0.8264, 0.8289,
0.8315, 0.8340, 0.8365, 0.8389, 0.8413, 0.8438, 0.8461, 0.8485, 0.8508, 0.8531, 0.8554, 0.8577,
0.8599, 0.8621, 0.8643, 0.8665, 0.8686, 0.8708, 0.8729, 0.8749, 0.8770, 0.8790, 0.8810, 0.8830,
0.8849, 0.8869, 0.8888, 0.8907, 0.8925, 0.8944, 0.8962, 0.8980, 0.8997, 0.9015, 0.9032, 0.9049,
0.9066, 0.9082, 0.9099, 0.9115, 0.9131, 0.9147, 0.9162, 0.9177, 0.9192, 0.9207, 0.9222, 0.9236,
0.9251, 0.9265, 0.9279, 0.9292, 0.9306, 0.9319, 0.9332, 0.9345, 0.9357, 0.9370, 0.9382, 0.9394,
0.9406, 0.9418, 0.9429, 0.9441, 0.9452, 0.9463, 0.9474, 0.9484, 0.9495, 0.9505, 0.9515, 0.9525,
0.9535, 0.9545, 0.9554, 0.9564, 0.9573, 0.9582, 0.9591, 0.9599, 0.9608, 0.9616, 0.9625, 0.9633,
0.9641, 0.9649, 0.9656, 0.9664, 0.9671, 0.9678, 0.9686, 0.9693, 0.9699, 0.9706, 0.9713, 0.9719,
0.9726, 0.9732, 0.9738, 0.9744, 0.9750, 0.9756, 0.9761, 0.9767, 0.9772, 0.9778, 0.9783, 0.9788,
0.9793, 0.9798, 0.9803, 0.9808, 0.9812, 0.9817, 0.9821, 0.9826, 0.9830, 0.9834, 0.9838, 0.9842,
0.9846, 0.9850, 0.9854, 0.9857, 0.9861, 0.9864, 0.9868, 0.9871, 0.9875, 0.9878, 0.9881, 0.9884,
0.9887, 0.9890, 0.9893, 0.9896, 0.9898, 0.9901, 0.9904, 0.9906, 0.9909, 0.9911, 0.9913, 0.9916,
0.9918, 0.9920, 0.9922, 0.9925, 0.9927, 0.9929, 0.9931, 0.9932, 0.9934, 0.9936, 0.9938, 0.9940,
0.9941, 0.9943, 0.9945, 0.9946, 0.9948, 0.9949, 0.9951, 0.9952, 0.9953, 0.9955, 0.9956, 0.9957,
0.9959, 0.9960, 0.9961, 0.9962, 0.9963, 0.9964, 0.9965, 0.9966, 0.9967, 0.9968, 0.9969, 0.9970,
0.9971, 0.9972, 0.9973, 0.9974, 0.9974, 0.9975, 0.9976, 0.9977, 0.9977, 0.9978, 0.9979, 0.9979,
0.9980, 0.9981, 0.9981, 0.9982, 0.9982, 0.9983, 0.9984, 0.9984, 0.9985, 0.9985, 0.9986, 0.9986,
0.9987, 0.9987, 0.9987, 0.9988, 0.9988, 0.9989, 0.9989, 0.9989, 0.9990, 0.9990, 0.9990, 0.9991,
0.9991, 0.9991, 0.9992, 0.9992, 0.9992, 0.9992, 0.9993, 0.9993, 0.9993, 0.9993, 0.9994, 0.9994,
0.9994, 0.9994, 0.9994, 0.9995, 0.9995, 0.9995, 0.9995, 0.9995, 0.9995, 0.9996, 0.9996, 0.9996,
0.9996, 0.9996, 0.9996, 0.9997, 0.9997, 0.9997, 0.9997, 0.9997, 0.9997, 0.9997, 0.9997, 0.9997,
0.9997, 0.9998, 0.9998};

     public Ztable() {

     }

     public static void dump() {
      for (int i = 0; i < sDev.length; i++) {
        System.out.println (sDev[i] + "   "+below[i]);
      }
     }

     /** Lookup the fraction of the set that is expected to be below this many standard
     * deviations away from the mean. */
     public static double getBelow(double stdDev) {

         if (stdDev > maxSdev) return 1.0;
         if (stdDev < 0.0) return getBelow(-stdDev);

         int idx = Arrays.binarySearch (sDev, stdDev);
         if (idx < 0) {
           int i = (-idx) - 1;
           if (i == 0) {
              return below[0];
           } else {
              return (below[i] + below[i-1])/2.0;
           }
         } else {

            return below[idx];
         }

     }
     /** Lookup the fraction of the set that is expected to be above this many standard
     * deviations away from the mean. */
     public static double getAbove(double stdDev) {
       return 1.0 - getBelow(stdDev);
     }

     public static void main (String args[]) {

         double val = 0.0;
         System.out.println(val + " " +getAbove(val));
         System.out.println(val + " " +getBelow(val));

         val = 0.105;
         System.out.println(val + " " +getAbove(val));
         System.out.println(val + " " +getBelow(val));

         val = 2.0;
         System.out.println(val + " " +getAbove(val));
         System.out.println(val + " " +getBelow(val));

         val = 5.0;
         System.out.println(val + " " +getAbove(val));
         System.out.println(val + " " +getBelow(val));

         val = -2.0;
         System.out.println(val + " " +getAbove(val));
         System.out.println(val + " " +getBelow(val));
         }
}
/*
// flip below/above for negative values
// below is just 1 - above
// zscore is stddevs

Normal Curve Areas

  z-score Area    Area
        Below   Above
      0   0.5     0.5
    0.01  0.504   0.496
    0.02  0.508   0.492
    0.03  0.512   0.488
    0.04  0.516   0.484
    0.05  0.5199  0.4801
    0.06  0.5239  0.4761
    0.07  0.5279  0.4721
    0.08  0.5319  0.4681
    0.09  0.5359  0.4641
    0.1   0.5398  0.4602
    0.11  0.5438  0.4562
    0.12  0.5478  0.4522
    0.13  0.5517  0.4483
    0.14  0.5557  0.4443
    0.15  0.5596  0.4404
    0.16  0.5636  0.4364
    0.17  0.5675  0.4325
    0.18  0.5714  0.4286
    0.19  0.5753  0.4247
    0.2   0.5793  0.4207
    0.21  0.5832  0.4168
    0.22  0.5871  0.4129
    0.23  0.591   0.409
    0.24  0.5948  0.4052
    0.25  0.5987  0.4013
    0.26  0.6026  0.3974
    0.27  0.6064  0.3936
    0.28  0.6103  0.3897
    0.29  0.6141  0.3859
    0.3   0.6179  0.3821
    0.31  0.6217  0.3783
    0.32  0.6255  0.3745
    0.33  0.6293  0.3707
    0.34  0.6331  0.3669
    0.35  0.6368  0.3632
    0.36  0.6406  0.3594
    0.37  0.6443  0.3557
    0.38  0.648   0.352
    0.39  0.6517  0.3483
    0.4   0.6554  0.3446
    0.41  0.6591  0.3409
    0.42  0.6628  0.3372
    0.43  0.6664  0.3336
    0.44  0.67    0.33
    0.45  0.6736  0.3264
    0.46  0.6772  0.3228
    0.47  0.6808  0.3192
    0.48  0.6844  0.3156
    0.49  0.6879  0.3121
    0.5   0.6915  0.3085
    0.51  0.695   0.305
    0.52  0.6985  0.3015
    0.53  0.7019  0.2981
    0.54  0.7054  0.2946
    0.55  0.7088  0.2912
    0.56  0.7123  0.2877
    0.57  0.7157  0.2843
    0.58  0.719   0.281
    0.59  0.7224  0.2776
    0.6   0.7257  0.2743
    0.61  0.7291  0.2709
    0.62  0.7324  0.2676
    0.63  0.7357  0.2643
    0.64  0.7389  0.2611
    0.65  0.7422  0.2578
    0.66  0.7454  0.2546
    0.67  0.7486  0.2514
    0.68  0.7517  0.2483
    0.69  0.7549  0.2451
    0.7   0.758   0.242
    0.71  0.7611  0.2389
    0.72  0.7642  0.2358
    0.73  0.7673  0.2327
    0.74  0.7703  0.2297
    0.75  0.7734  0.2266
    0.76  0.7764  0.2236
    0.77  0.7794  0.2206
    0.78  0.7823  0.2177
    0.79  0.7852  0.2148
    0.8   0.7881  0.2119
    0.81  0.791   0.209
    0.82  0.7939  0.2061
    0.83  0.7967  0.2033
    0.84  0.7995  0.2005
    0.85  0.8023  0.1977
    0.86  0.8051  0.1949
    0.87  0.8078  0.1922
    0.88  0.8106  0.1894
    0.89  0.8133  0.1867
    0.9   0.8159  0.1841
    0.91  0.8186  0.1814
    0.92  0.8212  0.1788
    0.93  0.8238  0.1762
    0.94  0.8264  0.1736
    0.95  0.8289  0.1711
    0.96  0.8315  0.1685
    0.97  0.834   0.166
    0.98  0.8365  0.1635
    0.99  0.8389  0.1611
      1   0.8413  0.1587
    1.01  0.8438  0.1562
    1.02  0.8461  0.1539
    1.03  0.8485  0.1515
    1.04  0.8508  0.1492
    1.05  0.8531  0.1469
    1.06  0.8554  0.1446
    1.07  0.8577  0.1423
    1.08  0.8599  0.1401
    1.09  0.8621  0.1379
    1.1   0.8643  0.1357
    1.11  0.8665  0.1335
    1.12  0.8686  0.1314
    1.13  0.8708  0.1292
    1.14  0.8729  0.1271
    1.15  0.8749  0.1251
    1.16  0.877   0.123
    1.17  0.879   0.121
    1.18  0.881   0.119
    1.19  0.883   0.117
    1.2   0.8849  0.1151
    1.21  0.8869  0.1131
    1.22  0.8888  0.1112
    1.23  0.8907  0.1093
    1.24  0.8925  0.1075
    1.25  0.8944  0.1056
    1.26  0.8962  0.1038
    1.27  0.898   0.102
    1.28  0.8997  0.1003
    1.29  0.9015  0.0985
    1.3   0.9032  0.0968
    1.31  0.9049  0.0951
    1.32  0.9066  0.0934
    1.33  0.9082  0.0918
    1.34  0.9099  0.0901
    1.35  0.9115  0.0885
    1.36  0.9131  0.0869
    1.37  0.9147  0.0853
    1.38  0.9162  0.0838
    1.39  0.9177  0.0823
    1.4   0.9192  0.0808
    1.41  0.9207  0.0793
    1.42  0.9222  0.0778
    1.43  0.9236  0.0764
    1.44  0.9251  0.0749
    1.45  0.9265  0.0735
    1.46  0.9279  0.0721
    1.47  0.9292  0.0708
    1.48  0.9306  0.0694
    1.49  0.9319  0.0681
    1.5   0.9332  0.0668
    1.51  0.9345  0.0655
    1.52  0.9357  0.0643
    1.53  0.937   0.063
    1.54  0.9382  0.0618
    1.55  0.9394  0.0606
    1.56  0.9406  0.0594
    1.57  0.9418  0.0582
    1.58  0.9429  0.0571
    1.59  0.9441  0.0559
    1.6   0.9452  0.0548
    1.61  0.9463  0.0537
    1.62  0.9474  0.0526
    1.63  0.9484  0.0516
    1.64  0.9495  0.0505
    1.65  0.9505  0.0495
    1.66  0.9515  0.0485
    1.67  0.9525  0.0475
    1.68  0.9535  0.0465
    1.69  0.9545  0.0455
    1.7   0.9554  0.0446
    1.71  0.9564  0.0436
    1.72  0.9573  0.0427
    1.73  0.9582  0.0418
    1.74  0.9591  0.0409
    1.75  0.9599  0.0401
    1.76  0.9608  0.0392
    1.77  0.9616  0.0384
    1.78  0.9625  0.0375
    1.79  0.9633  0.0367
    1.8   0.9641  0.0359
    1.81  0.9649  0.0351
    1.82  0.9656  0.0344
    1.83  0.9664  0.0336
    1.84  0.9671  0.0329
    1.85  0.9678  0.0322
    1.86  0.9686  0.0314
    1.87  0.9693  0.0307
    1.88  0.9699  0.0301
    1.89  0.9706  0.0294
    1.9   0.9713  0.0287
    1.91  0.9719  0.0281
    1.92  0.9726  0.0274
    1.93  0.9732  0.0268
    1.94  0.9738  0.0262
    1.95  0.9744  0.0256
    1.96  0.975   0.025
    1.97  0.9756  0.0244
    1.98  0.9761  0.0239
    1.99  0.9767  0.0233
      2   0.9772  0.0228
    2.01  0.9778  0.0222
    2.02  0.9783  0.0217
    2.03  0.9788  0.0212
    2.04  0.9793  0.0207
    2.05  0.9798  0.0202
    2.06  0.9803  0.0197
    2.07  0.9808  0.0192
    2.08  0.9812  0.0188
    2.09  0.9817  0.0183
    2.1   0.9821  0.0179
    2.11  0.9826  0.0174
    2.12  0.983   0.017
    2.13  0.9834  0.0166
    2.14  0.9838  0.0162
    2.15  0.9842  0.0158
    2.16  0.9846  0.0154
    2.17  0.985   0.015
    2.18  0.9854  0.0146
    2.19  0.9857  0.0143
    2.2   0.9861  0.0139
    2.21  0.9864  0.0136
    2.22  0.9868  0.0132
    2.23  0.9871  0.0129
    2.24  0.9875  0.0125
    2.25  0.9878  0.0122
    2.26  0.9881  0.0119
    2.27  0.9884  0.0116
    2.28  0.9887  0.0113
    2.29  0.989   0.011
    2.3   0.9893  0.0107
    2.31  0.9896  0.0104
    2.32  0.9898  0.0102
    2.33  0.9901  0.0099
    2.34  0.9904  0.0096
    2.35  0.9906  0.0094
    2.36  0.9909  0.0091
    2.37  0.9911  0.0089
    2.38  0.9913  0.0087
    2.39  0.9916  0.0084
    2.4   0.9918  0.0082
    2.41  0.992   0.008
    2.42  0.9922  0.0078
    2.43  0.9925  0.0075
    2.44  0.9927  0.0073
    2.45  0.9929  0.0071
    2.46  0.9931  0.0069
    2.47  0.9932  0.0068
    2.48  0.9934  0.0066
    2.49  0.9936  0.0064
    2.5   0.9938  0.0062
    2.51  0.994   0.006
    2.52  0.9941  0.0059
    2.53  0.9943  0.0057
    2.54  0.9945  0.0055
    2.55  0.9946  0.0054
    2.56  0.9948  0.0052
    2.57  0.9949  0.0051
    2.58  0.9951  0.0049
    2.59  0.9952  0.0048
    2.6   0.9953  0.0047
    2.61  0.9955  0.0045
    2.62  0.9956  0.0044
    2.63  0.9957  0.0043
    2.64  0.9959  0.0041
    2.65  0.996   0.004
    2.66  0.9961  0.0039
    2.67  0.9962  0.0038
    2.68  0.9963  0.0037
    2.69  0.9964  0.0036
    2.7   0.9965  0.0035
    2.71  0.9966  0.0034
    2.72  0.9967  0.0033
    2.73  0.9968  0.0032
    2.74  0.9969  0.0031
    2.75  0.997   0.003
    2.76  0.9971  0.0029
    2.77  0.9972  0.0028
    2.78  0.9973  0.0027
    2.79  0.9974  0.0026
    2.8   0.9974  0.0026
    2.81  0.9975  0.0025
    2.82  0.9976  0.0024
    2.83  0.9977  0.0023
    2.84  0.9977  0.0023
    2.85  0.9978  0.0022
    2.86  0.9979  0.0021
    2.87  0.9979  0.0021
    2.88  0.998   0.002
    2.89  0.9981  0.0019
    2.9   0.9981  0.0019
    2.91  0.9982  0.0018
    2.92  0.9982  0.0018
    2.93  0.9983  0.0017
    2.94  0.9984  0.0016
    2.95  0.9984  0.0016
    2.96  0.9985  0.0015
    2.97  0.9985  0.0015
    2.98  0.9986  0.0014
    2.99  0.9986  0.0014
      3   0.9987  0.0013
    3.01  0.9987  0.0013
    3.02  0.9987  0.0013
    3.03  0.9988  0.0012
    3.04  0.9988  0.0012
    3.05  0.9989  0.0011
    3.06  0.9989  0.0011
    3.07  0.9989  0.0011
    3.08  0.999   0.001
    3.09  0.999   0.001
    3.1   0.999   0.001
    3.11  0.9991  0.0009
    3.12  0.9991  0.0009
    3.13  0.9991  0.0009
    3.14  0.9992  0.0008
    3.15  0.9992  0.0008
    3.16  0.9992  0.0008
    3.17  0.9992  0.0008
    3.18  0.9993  0.0007
    3.19  0.9993  0.0007
    3.2   0.9993  0.0007
    3.21  0.9993  0.0007
    3.22  0.9994  0.0006
    3.23  0.9994  0.0006
    3.24  0.9994  0.0006
    3.25  0.9994  0.0006
    3.26  0.9994  0.0006
    3.27  0.9995  0.0005
    3.28  0.9995  0.0005
    3.29  0.9995  0.0005
    3.3   0.9995  0.0005
    3.31  0.9995  0.0005
    3.32  0.9995  0.0005
    3.33  0.9996  0.0004
    3.34  0.9996  0.0004
    3.35  0.9996  0.0004
    3.36  0.9996  0.0004
    3.37  0.9996  0.0004
    3.38  0.9996  0.0004
    3.39  0.9997  0.0003
    3.4   0.9997  0.0003
    3.41  0.9997  0.0003
    3.42  0.9997  0.0003
    3.43  0.9997  0.0003
    3.44  0.9997  0.0003
    3.45  0.9997  0.0003
    3.46  0.9997  0.0003
    3.47  0.9997  0.0003
    3.48  0.9997  0.0003
    3.49  0.9998  0.0002
    3.5   0.9998  0.0002
*/
