package org.trinet.util;
import java.util.*;
import org.trinet.jdbc.*;
import org.trinet.jdbc.datatypes.*;

/** Class implements a java.util.Date range. */
public class DateRange extends NumberRange implements DateRangeIF, DateStringFormatter, Cloneable, Comparable {

/** Default constructor, null bounds. */
    public DateRange() {}

/** Constructor creates minimum and maximum bounds from the input object values.
* @exception java.lang.IllegalArgumentException max < min
*/
    public DateRange(long min, long max) {
        super(new Long(min), new Long(max));
    }

/** Constructor uses EpochTime.epochToDate(double) to construct date bounds. */
    public DateRange(double minEpoch, double maxEpoch) {
        this(Math.round(minEpoch*1000.), Math.round(maxEpoch*1000.));
    }

/** Constructor creates minimum and maximum bounds from the input object values.
* @exception java.lang.IllegalArgumentException max < min
*/
    public DateRange(java.util.Date  min, java.util.Date max) {
        super();
        setMin(min);
        setMax(max);
    }

/** Constructor uses EpochTime.stringToDate(String date, String pattern) to construct date bounds. */
    public DateRange(String minDate, String maxDate, String pattern ) {
        this(EpochTime.stringToDate(minDate, pattern), EpochTime.stringToDate(maxDate, pattern)); 
    }

/** Returns milliseconds value of lower bound. */
    public long getMinTime() {
        return (min == null) ? 0 : min.longValue();
    }

/** Returns milliseconds value of upper bound. */
    public long getMaxTime() {
        return (max == null) ? 0 : max.longValue();
    }

/** Returns epoch seconds value of lower bound. */
    public double getMinEpochSecs() {
        return (min == null) ? 0. : min.doubleValue()/1000.;
    }

/** Returns epoch seconds value of upper bound. */
    public double getMaxEpochSecs() {
        return (max == null) ? 0. : max.doubleValue()/1000.;
    }

/** Returns a clone of the minimum bound. */
    public java.util.Date getMinDate() {
        return (min == null) ? null : new java.util.Date(min.longValue());
    }

/** Returns a clone of the maximum bound. */
    public java.util.Date getMaxDate() {
        return (max == null) ? null : new java.util.Date(max.longValue());
    }

/** Returns the size of the range as a double seconds. */
    public double seconds() {
        return doubleExtent()/1000.;
    }

/** Returns the size of the range as a double milliseconds. */
    public double doubleExtent() {
        return  (double) longExtent();
    }

/** Returns the size of the range as a long milliseconds. */
    public long longExtent() {
        return  (min == null || max == null) ? Long.MAX_VALUE : (max.longValue() - min.longValue());
    }

/** True if minimum bound of this instance is after the input date. */
    public boolean after(java.util.Date date) {
        return (date == null) ? true : super.after(new Long(date.getTime()));
    }

/** True if maximum bound of this instance is before the input date. */
    public boolean before(java.util.Date date) {
        return (date == null) ? true : super.before(new Long(date.getTime()));
    }

/** True if this object's range excludes input date. */
    public boolean excludes(java.util.Date date) {
        return (date == null) ? true : super.excludes(new Long(date.getTime()));
    }

/** True if this object's range contains input date. */
    public boolean contains(java.util.Date date) {
        return (date == null) ? false : super.contains(new Long(date.getTime()));
    }

/** True if this object's range excludes input range. */
    public boolean excludes(java.util.Date minDate, java.util.Date maxDate) {
        Long mind = (minDate == null) ? null : new Long(minDate.getTime());
        Long maxd = (maxDate == null) ? null : new Long(maxDate.getTime());
        return super.excludes(mind, maxd);
    }

/** True if this object's range contains input range. */
    public boolean contains(java.util.Date minDate, java.util.Date maxDate) {
        Long mind = (minDate == null) ? null : new Long(minDate.getTime());
        Long maxd = (maxDate == null) ? null : new Long(maxDate.getTime());
        return super.contains(mind, maxd);
    }

/** True if this object's range overlaps input range. */
    public boolean overlaps(java.util.Date minDate, java.util.Date maxDate) {
        Long mind = (minDate == null) ? null : new Long(minDate.getTime());
        Long maxd = (maxDate == null) ? null : new Long(maxDate.getTime());
        return super.overlaps(mind, maxd);
    }

/** True if this object's range lies within input range. */
    public boolean within(java.util.Date minDate, java.util.Date maxDate) {
        Long mind = (minDate == null) ? null : new Long(minDate.getTime());
        Long maxd = (maxDate == null) ? null : new Long(maxDate.getTime());
        return super.within(mind, maxd);
    }

/** Returns String concatenation of minimum and maximum bounds values 
* formatted as specified by the input pattern relative to UTC time zone.
* The min, max dates are separated by " ".
*/
    public String toDateString(String pattern) {
        StringBuffer sb = new StringBuffer(80);
        if (min != null)
            sb.append(EpochTime.epochToString(Math.round(min.doubleValue())/1000., pattern));
        else sb.append("NULL");
        sb.append(" ");
        if (max != null)
            sb.append(EpochTime.epochToString(Math.round(max.doubleValue())/1000., pattern));
        else sb.append("NULL");
        return sb.toString();
    }

/** Returns String concatenation of minimum and maximum bounds as
* GMT date values separated by " ".
*/
    public String toString() {
        return toDateString("yyyy/MM/dd:HH:mm:ss.SSS");
    }

/** Convenience wrapper System.out.println(toString()). */
    public void print() {
        System.out.println(toString());
    }

/** 
* Sets the minimum bound of the range to the input.
*/
    public void setMin(java.util.Date minDate) {
        if (minDate == null) min = null;
        else super.setMin(new Long(minDate.getTime()));
    }

/** 
* Sets the maximum bound of the range to the input.
*/
    public void setMax(java.util.Date maxDate) {
        if (maxDate == null) max = null;
        else super.setMax(new Long(maxDate.getTime()));
    }

/** 
* Sets the minimum, maximum bounds of the range to the input values.
*/
    public void setLimits(java.util.Date minDate, java.util.Date maxDate) {
        if (minDate != null && maxDate != null) {
            if (maxDate.before(minDate))
                throw new IllegalArgumentException("DateRange min-max bounds reversed.");
        }
        setMin(minDate);
        setMax(maxDate);
    }

/**
* Sets the appropiate minimum or maximum bound to extend the range to include the input value.
* Does a no-op if contains(Date) == true.
*/
    public void include(java.util.Date date) {
        super.include(new Long(date.getTime()));
    }

/*
    public static void main(String [] args) {

         System.out.println("++++++++ BEGIN TEST ++++++++++");

         long ms = System.currentTimeMillis();
         Date now = new Date(ms);
      
         Date ontime = new Date(ms + 360000);
         Date later = new Date(ms + 3600000);
         Date laterstill = new Date(ms + 7200000);
         Date yesterday = new Date(ms - 86400000);
         Date tomorrow =  new Date(ms + 86400000);

         DateRange dr = new DateRange(now, later);
         DateRange dr2 = new DateRange(yesterday, tomorrow);

         dr.dump1(yesterday);
         dr.dump1(ontime); dr.dump1(later); dr.dump1(tomorrow);

         dr.dump2(dr2);

         System.out.println("Test dr.setLimits(ontime, laterstill)");
         dr.setLimits(ontime, laterstill);
         dr.dump2(dr2);

         System.out.println("Test dr.include(now)");
         dr.include(now);
         dr.dump2(dr2);

         System.out.println("Test dr.include(dr2)");
         dr.include(dr2);
         dr.dump2(dr2);

         System.out.println("Test dr.setMax(laterstill); dr2.setMin(laterstill)");
         dr.setMax(laterstill);
         dr2.setMin(laterstill);
         dr.dump2(dr2);

         System.out.println("Test dr.setMax(later); dr2.setMin(laterstill)");
         dr.setMax(later);
         dr2.setMin(laterstill);
         dr.dump2(dr2);

    }

    public void dump(DateRange dr2, DateRange dr3) {
         System.out.println("Test equalsRange: ");
         System.out.println("     dr2.toString(): " + dr2.toString());
         System.out.println("     dr3.toString(): " + dr3.toString());
         System.out.println("     dr2.equalsRange(dr3) : " +  dr2.equalsRange(dr3));
         System.out.println("------------------\n");
    }

     public void dump1(Date val) {
         System.out.println("Dump Range bounds: " + min.toString() + ", " + max.toString() + " size: " +
                             EpochTime.elapsedTimeToText(seconds()) );
         System.out.println(" Test input val: " + val);
         System.out.println(" after(val)    : " + after(val));
         System.out.println(" before(val)   : " + before(val));
         System.out.println(" excludes(val) : " + excludes(val));
         System.out.println(" contains(val) : " + contains(val));
         System.out.println("------------------\n");
    }

    public void dump2(DateRange dr2) {
         System.out.println("Dump Range bounds: " + min.toString() + ", " + max.toString() + " size: " +
                             EpochTime.elapsedTimeToText(seconds()) );
         System.out.println(" Test Range2 Min,max: " + dr2.min.toString() + ", " + dr2.max.toString() );

         System.out.println(" within(dr2)       : " + within(dr2));
         System.out.println(" dr2.within(this)  : " + dr2.within(this)); 

         System.out.println(" overlaps(dr2)     : " + overlaps(dr2));
         System.out.println(" dr2.overlaps(this): " + dr2.overlaps(this));

         System.out.println(" dr2.after(this)   : " + dr2.after(this));
         System.out.println(" after(dr2)        : " + after(dr2));

         System.out.println(" dr2.before(this)  : " + dr2.before(this));
         System.out.println(" before(dr2)       : " + before(dr2));

         System.out.println(" dr2.excludes(this): " + dr2.excludes(this));
         System.out.println(" excludes(dr2)     : " + excludes(dr2));

         System.out.println(" dr2.contains(this): " + dr2.contains(this));
         System.out.println(" contains(dr2)     : " + contains(dr2));
         System.out.println("------------------\n");

    }
*/
}

