package org.trinet.jiggle;

// WFPanel.java

import java.applet.*;
import java.text.*;
import java.awt.*;
import java.util.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import org.trinet.jasi.TravelTime;
import org.trinet.jdbc.*;
import org.trinet.jasi.*;
import org.trinet.util.*;
import org.trinet.util.graphics.ColorList;

/**
 * This class it the graphical representation of one WView in a panel.
 * The size of the WF is scaled to the size of the panel.
 * If it is not explicitely set, the panel size is set by the
 * layout manager.<p>
 *
 * Class allows the WFPanel to have an alternateWf. This is a second version
 * of the contained waveform and ususally represents a filtered time series.
 *
 * Has limited mouse support for finding out where you are in the trace.
 * You must extend this class and add mouse handelers to make it dynamic.
 * @see: ActiveWFPanel()
 * @see: ZoomedWFPanel()
 */

public class WFPanel
    extends JPanel
    implements Scrollable  {

    /** Keep track of selected status.
     * If selected highlight with the selectedColor.
     */
  boolean amSelected = false;

    // Color attributes

    // Static default colors
  static final Color selectedColor	= new Color (165, 220, 255);	// sky blue
  static final Color unselectedColor	= Color.white;
  static final Color alternateUnselectedColor = ColorList.aliceBlue; //slight gray
  static final Color highlightColor	= Color.yellow;
  static final Color dragColor		= Color.orange;

    // Actual colors
  Color WFColor	        = Color.black;	// default colors
  Color TextColor       = Color.black;
  Color backgroundColor = unselectedColor;
  Color segEdgeColor    = Color.cyan;
  Color ScaleColor	= Color.black;

    /** Allow display of an alternate (filtered) wf. */
    boolean showAlternateWf = false;
    /** The filter used to produce the alternate waveform. */
    FilterIF filter = null;
    /** The alternate waveform. */
    private Waveform alternateWf = null;

    /** The row header for WFPanels in scollers. */
    private WFRowHeader rowHeader = new WFRowHeader(this);

/** the waveform view to be painted. */
    public WFView wfv;			// the waveform view to be painted

    /** Master view of the data set*/
    MasterView mv ;

    /** Initial scale factor */
    private double ScaleFacX = 1.0;
    /** Initial scale factor */
    private double ScaleFacY = 1.0;

    /** Show or don't the WFSegment boundaries in the view */
    public boolean showSegments = false;

    /** Controls if a time scale is plotted in the panel */
    boolean showTimeScale = true;

    /** Controls if an amp scale is plotted in the panel */
    boolean showAmpScale = true;

    /** If a time scale is plotted, controls if time labels are printed */
    boolean showTimeLabel = true;

    /** If true, show pick descriptions on pick marker flags. */
    boolean showPhaseDescriptions = true;

    /** If true, show channel and dist label. */
    boolean showChannelLabel = true;

    /** If true show a colored stripe (PhaseCue) where P an S should be. */
    boolean showPhaseCues = false;
    PhaseCue cue = new PhaseCue();

    /** show samples as red |'s if scale is expanded sufficiently */
    boolean showSamples = true;

    /** Show a red vertical line from median line to sample*/
    boolean showSampleBars = true;

// Optional viewport: to let paint() know limits of visible panel. This is used to insure
// that PickMarkers, etc. are visible where WFPanel is expanded
  JViewport vport;

// scaling values
  double pixelsPerSecond;
  double pixelsPerCount;

  double	 transY;	   // this is needed to convert pixel position to amp

    // reduce instatiation overhead...
  double pixelInterval, xPosition;

 // for dynmic displays of current cursor location
  Point cursorLoc = new Point();

  public boolean continuousSelection = false;

  int initPanelWidth  = 300;	// default size (just temporary, layout manager
  int initPanelHeight = 100;	// will adjust this)

//  static final Font fontBold = new Font("Monospaced", Font.BOLD, 12);
  static final DecimalFormat df1 = new DecimalFormat ( "###0.0" );  // distance format

/** Time and amp bounds of the actual data in the WFView */
   WFDataBox dataBox = new WFDataBox();

/** Time and amp bounds of this WFPanel
 *  (often longer then the dataBox to align traces in the
 *  MultiPanel or smaller if the trace is zoomed )*/
   WFSelectionBox panelBox = new WFSelectionBox();

/* Explanation of view areas used here.

Each "box" defines a 2-D area in time-amp space. Time is in seconds and amp is
in counts. These boxes are used to define and map time-amp areas into graphical
areas defined by 2-D boxes defined in terms of pixels in Height and Width.

We use 3 WFSelectionBox'es:
	1) dataBox - defines the extent of the waveform data in time and amp.
	   If there is no waveform this is null as tested by 'dataBox.isNull()'
	2) panelBox - defines the time-amp bounds of this WFPanel. Normally it is
	   the same as the dataBox for a zoomPanel. For a WFPanel in a
	   GroupPanel the panelBox time bounds will usually be equal to the
	   exteme bounds of the ViewList. This insures all data is visable and
	   time aligned. For dataless WFPanels the amp dimension is problematic
	   and is arbitrarily set to +100/-100 counts.
	3) viewBox - defines the bounds of the view. For example, in a zoomed
	   panel the viewbox represents the JViewport and the WFPanel expands
	   to accomodate the scaling of the viewBox to the JViewport. This is
	   only used by ActiveWFPanels.


     panelBox
    +===================================================================+
    |									|
    |	                        +-----------------+	 		|
    |	 	           	|dataBox          |	 		|
    |	 	            	|                 |	 		|
    |	 	                +-----------------+ 	 		|
    |									|
    +===================================================================+

*/

/**
 * Constructors
 */
  public WFPanel (){

        setBackground(unselectedColor);
	setForeground(WFColor);

        setSize(initPanelWidth, initPanelHeight);

        setCursor(new Cursor(Cursor.CROSSHAIR_CURSOR));

	//	addWFLoadListener();

    }

  public WFPanel (WFView wfview) {

        this();
        setWFView (wfview);

   }

    /** Return true if this panel is selected. */
    public boolean isSelected() {
      return amSelected;
    }
// -------------------------------------------------------------------
/**
 * Set or change the WFView to be displayed in this WFPanel.
 */
  public void setWFView (WFView wfview) {

	wfv = wfview;

	if (wfv == null) {
	  clear();
	  return;	// signal to GC to erase old view
	}

        setMasterView(wfv.mv);

	// add change listener to waveform object to handle async. load/unloading
	addWFLoadListener();

        setupWf ();

	WFColor = ComponentColor.get(wfv.getChannelObj());	// get proper color for this component

	// this seems logical...
	repaint();

    }

/** Set data and panel box sizes and creates alternate (filtered) waveform if a filter exists.
 *  Which will actually be displayed is toggled with setShowAlternateWf(). */
    public void setupWf() {
        createAlternateWf();    // only done if a filter is set and there is timeseries
	// set box bounds based on the data
        dataBox.set(getWf());		// a null wf is OK here
        setPanelBoxSize();
	((WFRowHeader)getRowHeader()).setDefaultText();
    }

    /** Create and setup the alternate (filtered) waveform. */
    void createAlternateWf() {
         if (filter != null && getRawWf() != null) {
           alternateWf = Waveform.copy(getRawWf());    // copy

	   // filter the time series of the new waveform
           alternateWf.filter(filter.getFilter((int) alternateWf.getSampleRate()));

           // set max, min, bias values                // scan
/* done by filtering
	   if (alternateWf.hasTimeSeries()) {
	      alternateWf.scanForAmps();
	      alternateWf.scanForBias();
	   }
*/
         }

    }
    /**
     * Add the listener that handles waveform load events. This is to support
     * asynchronous loading of timeseries in background.  */
    //    private void addWFLoadListener () {
    protected void addWFLoadListener () {

	if (getRawWf() != null) {
	    getRawWf().addChangeListener (new WFLoadListener());
	}

    }

    /** INNER CLASS: Change events are expected from Waveform objects when the
     *  timeseries is loaded. Repaint the WFPanel when timeseries shows up.
     *  This is to support asynchronous loading of timeseries in background. */
    class WFLoadListener implements ChangeListener {

	public void stateChanged (ChangeEvent changeEvent) {

         setupWf();
//         setShowAlternateWf(getShowAlternateWf()); // force selection of proper WF

	 //	    repaint();
	}
    }

    /** Set the filter to be used to create the alternate waveform. You only need
    * to do this once. */
    public void setFilter(FilterIF filter) {
         this.filter = filter;
         if (filter != null) {
           createAlternateWf();
         }
    }

    public void setShowTimeScale (boolean tf) {
       if (showTimeScale != tf) {
          showTimeScale = tf;
//          repaint();
       }
    }
    public boolean getShowTimeScale() {
      return showTimeScale;
    }

    public void setShowSamples (boolean tf) {
       showSamples = tf;
    }
    public boolean getShowSamples() {
      return showSamples;
    }

    public void setShowAmpScale (boolean tf) {
       if (showAmpScale != tf) {
          showAmpScale = tf;
       }
    }
    public boolean getShowAmpScale() {
      return showAmpScale;
    }

    public void setShowTimeLabel (boolean tf) {
       if (showTimeLabel != tf) {
          showTimeLabel = tf;
//          repaint();
       }
    }
    public boolean getShowTimeLabel() {
      return showTimeLabel;
    }
    public void setShowChannelLabel (boolean tf) {
       if (showChannelLabel != tf) {
          showChannelLabel = tf;
//          repaint();
       }
    }
    public boolean getShowChannelLabel() {
      return showChannelLabel;
    }
    public void setShowSegments (boolean tf) {
       if (showSegments != tf) {
          showSegments = tf;
//          repaint();
       }
    }
    public boolean getShowSegments() {
      return showSegments;
    }

    public void setShowPhaseCues(boolean tf) {
       if (showPhaseCues != tf) {
          showPhaseCues = tf;
//          repaint();
       }
    }

    public boolean getShowPhaseCues() {
      return showPhaseCues;
    }

    /** Show alternate trace if set true. */
    public void setShowAlternateWf( boolean tf) {
       if (showAlternateWf != tf) {   // its a change
          showAlternateWf = tf;
          // set box bounds based on the current WF
	  dataBox.set (getWf());		// a null wf is OK here
          setPanelBoxSize();
       }
    }

    public boolean getShowAlternateWf() {
      return showAlternateWf;
    }

    public Waveform getRawWf() {
      if (wfv == null) return null;
      return wfv.wf;
    }

    public Waveform getAlternateWf() {
      return alternateWf;
    }
/** Return the time of the sample nearest the pixel with he value of x. */
    public double getSampletimeNearX (int x) {

	double time = dtOfPixel(x);

	// force click to be on a sample, if there's a waveform
	if (getWf() != null) time = getWf().closestSampleInterval(time);

        return time;
    }
    /** Returns the currently selected waveform (raw or filtered). All access to
     *  timeseries should be via this method because it determines which one
     *  to use. */
    public Waveform getWf () {
       if (getShowAlternateWf()) return getAlternateWf();
       return getRawWf();
    }
/** Change this panel's selection state. Just changes the background color to
* indicate the panel is selected then repaints. */
         public void setSelected (boolean tf) {

           if (tf == isSelected()) return;   // no change, don't re-add listeners
           amSelected = tf;
           repaint();
    }

    public void setBackgroundColor (Color clr) {
      backgroundColor = clr;
    }
    public Color getBackgroundColor() {
          if (amSelected) {
              return selectedColor;
           } else {
//	      if (getShowAlternateWf()) return alternateUnselectedColor;
 // Use logic to test that WF was ACTUALLY filtered not just that you asked that
 // it be. This is because the filter may not work on this type of data.
 // For example, WAFilter only works on 20, 80 & 100 sps data.
	      if (getWf() != null && getWf().isFiltered()) return alternateUnselectedColor;
              return unselectedColor;
           }
    }
  /**
  * Set the size of the panel box.
  */
  public void setPanelBoxSize () {

      Waveform wf = getWf();

      if (wf != null)  {
         panelBox.set(wfv.viewSpan, wf.getMaxAmp(), wf.getMinAmp());
      } else {
      // box must have non-zero dimension else /0 error happen elsewhere
         panelBox.set(wfv.viewSpan, 1, 0);
      }
  }

  /** Return a JPanel that acts as a row header in a scroll pane.*/
  public JComponent getRowHeader() {
    return rowHeader;
     //return new WFRowHeader(this);
  }

    /** Set the WFPanel's master view. */
     public void setMasterView (MasterView mv) {
       this.mv = mv;
     }

/**
 * Set the start/end time of the panel. This may be different from the time span
 * of the waveform data. For example, traces with different starting times
 * can be time aligned in the MultiPanel by setting all their timebounds the
 * same. Setting the time bounds can also be used to clip the data.
 */
  public void setTimeBounds (double start, double end) {
	panelBox.setTimeSpan(start, end);
  }

  public void setTimeBounds (TimeSpan ts) {
	panelBox.setTimeSpan(ts);
  }

/**
 * Clear this WFPanel so that it blank when displayed on a partially full
 * page that contains many WFPanels
 */
    public void clear () {
	wfv = null;
        alternateWf = null;
	dataBox.setNull();
	panelBox.setNull();
    }

// -------------------------------------------------------------------

/**
 * Set the WFPanel's scaling factors to fit the WFSelectionBox
 */
void scalePanel () {
    // Use the timeSpan of the panel box (for alignment) and the amps
    // of the data box (for scaling)
    WFSelectionBox box = new WFSelectionBox(panelBox);;

    if (wfv.hasTimeSeries())  box.setAmpSpan(dataBox);

    setScaleFactors (box);

}

/**
 * Set this WFPanel's scaling factors so that the given WFSelectionBox
 * (in time-amp space) fits completely inside this WFPanel. And the Waveform
 * is drawn at the correct scale.
 */
    //void scalePanelToBox (WFSelectionBox box)
void setScaleFactors (WFSelectionBox box) {

    // pixels/second  (X-axis)
    if (box.getTimeSize() == 0.0) {
       pixelsPerSecond = (double) getWidth();
    } else {
       pixelsPerSecond = (double) (getWidth())  / box.getTimeSize();
    }
    // pixels/count   (Y-axis)
    if (box.getAmpSize() == 0.0) {
       pixelsPerCount = (double) getHeight();
    } else {
       pixelsPerCount  = ((double) getHeight()) / box.getAmpSize();
    }
    // Y-offset to zero in the data(pixel space is always positive, data
    // space can be negative.
    transY = box.getMaxAmp();

    // debug
    //dumpScaleValues();

//    System.out.println ("setScaleFactors = "+pixelsPerSecond+"/"+pixelsPerCount+"  "+transY);
}

/**
 * Dump values (for debugging)
 */

    public void dumpScaleValues()
    {
	System.out.println ("----------- "+wfv.getChannelObj().toString()+" -----------");

	System.out.println ("transY      " + transY);
//	System.out.println ("biasOffset  " + biasOffset);
	System.out.println ("dataBox     " + dataBox.toString() );
//	System.out.println ("viewBox     " + viewBox.toString() );
	System.out.println ("panelBox    " + panelBox.toString() );

	System.out.println ("panelWidth  " + getWidth() );
	System.out.println ("panelHeight " + getHeight() );

	System.out.println ("pix/Sec     " + pixelsPerSecond);
	System.out.println ("pix/Count   " + pixelsPerCount);
    }

    // -------------------------------------------------------------------
/**
 * Paints one WFView in the WFPanel. Must rescale every time because frame
 * may have been resized.
 */
    public void paintComponent (Graphics g)  {

     super.paintComponent(g);	// give super and delegate chance to paint

/* IMPORTANT!
 * You must recalculate ALL scaling here because this may be called as a result
 * of the frame being resized by the user with the frame border widgits.
 */
     scalePanel();

     paintBackground(g);

     paintContents(g);

    }
/**
 * Paints the contents of the panel. Called after paintBackground();
 */
    public void paintContents (Graphics g)  {

     if ( wfv == null ) return;	// bail if no WFView (no data loaded)

     paintData(g);
    }
/**
 * Shared basic painting of background.
 * This sets the panel to the color indicating it's selected. Its also
 * a workaround to stop a weird behavior where old components get
 * painted into the WFPanel on repaints. Just paint over with the background
 * color.
 */
    public void paintBackground (Graphics g) {

//       g.setColor(getBackgroundColor());
//       g.fillRect(0, 0, getSize().width, getSize().height);
               g.setColor(getBackgroundColor());
               g.fillRect(0, 0, getWidth(), getHeight());
//               g.setColor(getForeground());

       if (getShowPhaseCues()) paintPhaseCues(g);   // paint phase location cues

    }

/**
* Shared basic painting of foreground "decorations" and data.
*/
    public void paintData (Graphics g) {

// Plot all the segments in the view

    paintTimeSeries(g);

// draw a scale (should be a seperate method)

    paintScale (g);

// Put the picks on the view

    paintPickFlags (g);

// Put the amps and codas on the view

    paintAmpFlags (g);

    paintCodaFlags(g);

// Label the trace
    if (getShowChannelLabel()) {

	g.setColor(TextColor);
	// font ?
//	g.setFont(fontBold);

	String lab = wfv.getChannelObj().toDelimitedSeedString(' ') + " " +
	    df1.format(wfv.getDistance()) +"km" ;

	g.drawString (lab, 10, 20);
    }

    } // end of paint()

/**
 * Paint the time-series, i.e. the waveform in the WFView (all its segments)
 */
    public void paintTimeSeries (Graphics g) {

     Waveform wf = getWf();

	if (wf == null || !wf.hasTimeSeries()) return;

	// last sample drawn in previous segment, to connect segments on plot
	Sample previousSample;

     // synchronize to make sure another thread doesn't change the wf underneith us
      synchronized (wf) {
	 WFSegment wfseg[] = WFSegment.getArray(wf.getSegmentList());

	 // optimization step, reduces method calls in loop ??
	 int count = wfseg.length;

	 for (int i = 0; i < count; i++) {

	    // handle continuity between contiguous segments (time tears)
	    if ( (i > 0) &&
             WFSegment.areContiguous(wfseg[i-1], wfseg[i])) {
	   	   previousSample = wfseg[i-1].getLastSample();
	      } else {
		   previousSample = null;
	    }
	    paintSegment(g, wfseg[i], previousSample);
	 } // end of loop
     } // end of synch
    }

// --------------------------------------------------------------------
/**
 * Paints one WFSegment in the WFPanel.
 */

 // This stuff is here in an effort to reduce declarations inside of heavily used
 // loops. I'm not sure it really matters.
   double pixelsPerInterval;
   int offsetX;
   int jump;
   int pointCount;
   int x[];
   int y[];
   float sample[];
   int x0, y0, xPixel;
   float maxAmp, minAmp;

// THIS IS ONE OF THE MOST TIME CONSUMING PARTS OF THIS CODE. OPTIMIZE IF POSSIBLE

    public void paintSegment (Graphics g, WFSegment wfseg, Sample previousSample) {

      if (wfseg == null) return;
/*
 * Recalculate 'pixelsPerInterval' because the sample interval (dt)
 * could be different for each segment.
 */
	// pixels between sample points
	pixelsPerInterval = pixelsPerSecond * wfseg.getSampleInterval();

     // if time scale is compressed (>4 samples/pixel), paint as bars
     if (1.0/pixelsPerInterval >= 4.0) {
        paintSegmentAsBars (g, wfseg, previousSample);
        return;
     }

	// for time alignment
	offsetX = pixelOfTime(wfseg.getEpochStart());

// Decimation factor is tied to scale such that we "over sample" by about x4.
// I arrived at this value by testing. Decimation speeds things up emmensely.

	jump = Math.max(1, (int) (wfseg.size()/getSize().width/2) );

	// count of points that will be in decimated plotting array
	pointCount = (wfseg.size() / jump);

	// instantiate the x-y ploting arrays for use in Graphics.drawPolyline()
	x = new int[pointCount];
	y = new int[pointCount];
        sample = wfseg.getTimeSeries();

// Convert seismogram to pixel coordinate system

	// interval w/ decimation
	pixelInterval = pixelsPerInterval * (double)jump;

	xPosition = (double) offsetX;

     int idx = 0;
	for (int i = 0 ; i < pointCount; i++) {

	  x[i] = (int) xPosition;

		//"inline" this method to optimize
          y[i] = pixelOfAmp(sample[idx]);

          //TODO: might round off error accumulate here?
	  xPosition += pixelInterval;	// inc. pixel by decimated interval
	  idx += jump;
	}

    // show start of WFSegment if showSegment is true
    // do it first so they are behind the waveform

	if (getShowSegments()) paintPacketSpans (g);

	// set plot color specific to component sh log /sys
	g.setColor(WFColor);

// plot line connecting segments if necessary
	if (previousSample != null) {

	    x0 = offsetX + (int) (-1 * pixelsPerInterval);
	    y0 = pixelOfAmp(previousSample.value);

	    g.drawLine(x0, y0, x[0], y[0]);
	}

// plot the seismogram, this is the major resource sink of the program

	g.drawPolyline (x, y, pointCount);

// mark individual samples if scale allows (> 4 pixels per sample)
 	if (pixelsPerInterval > 4.0) {
//	    if (showSamples) {
//              g.setColor(Color.red);
//              int center = pixelOfAmp(wfseg.getBias());
//              for (int i = 0 ; i < x.length ; i++) {
//                g.drawLine(x[i], y[i], x[i], center);
//              }
//	    }
//            } else
            if (showSamples) {
              g.setColor(Color.red);
              for (int i = 0 ; i < x.length ; i++) {
                g.drawLine(x[i], y[i]-2, x[i], y[i]+2);
              }
	    }
    }

} // end of paintSegment()

/**
 * Paints one WFSegment in the WFPanel using a shortcut technique that plot the
 * waveform by plotting a vertical bar, one pixel wide, that
 * represents the max/min range of the samples that lie within that pixel. This only
 * works if the time scale is compressed to the point that there is more than one sample
 * per pixel.  <p>
 * This speeds up plotting and makes the seismograms look better then simple decimation
 * without filtering.
 * Suggested by Paul Friberg.
 */
    public void paintSegmentAsBars (Graphics g, WFSegment wfseg, Sample previousSample) {
/*
 * Recalculate 'pixelsPerInterval' because the sample interval (dt)
 * could be different for each segment.
 */

      // data check
      if (!wfseg.hasTimeSeries()) return;

	// pixels between sample points
     double secPerPixel = 1.0/pixelsPerSecond;

	// for time alignment
	xPixel = pixelOfTime(wfseg.getEpochStart());
     sample = wfseg.getTimeSeries();

	if (showSegments) paintPacketSpans (g);

	// set plot color specific to component sh log /sys
	g.setColor(WFColor);

// Convert seismogram to pixel coordinate system

     maxAmp = sample[0];
     minAmp = maxAmp;

     double pixelEnd = wfseg.getEpochStart() + secPerPixel;
     double currentTime = wfseg.getEpochStart();

     for (int i = 0; i < sample.length; i++ ) {

         maxAmp = Math.max(maxAmp, sample[i]);
         minAmp = Math.min(minAmp, sample[i]);

         if ( currentTime >= pixelEnd ) {  // end of a pixel interval

           // draw the bar
           g.drawLine(xPixel, pixelOfAmp(maxAmp),
                      xPixel, pixelOfAmp(minAmp));

           // inc. stuff and reset for next sample interval
           xPixel++; // one pixel
           pixelEnd += secPerPixel;

           // this actually included the last sample in the next interval so
           // there are no gaps between bars
           maxAmp = sample[i];
           minAmp = maxAmp;
         }
       currentTime += wfseg.getSampleInterval();  // increment
     }
}

/**
 * Draw lines at the packet boundaries.
 */
public void paintPacketSpans(Graphics g) {
    int start;
    int end;

    Waveform wf = getWf();

    if (wf != null) {
      g.setColor (segEdgeColor);

      TimeSpan pkt[] = wf.getPacketSpans();

      if (pkt != null) {
       for (int i = 0; i< pkt.length; i++) {
	    start = pixelOfTime(pkt[i].getStart());
	    end   = pixelOfTime(pkt[i].getEnd());

	    g.drawLine (start, 0 , start, getHeight() );
	    g.drawLine (end,   0 , end,   getHeight() );
       }
      }
    }
}

/**
 * Plot the phases on the waveform
 */
//TODO: could set an optional mode that would show deleted phases
    public void paintPickFlags (Graphics g)
    {
        if ( wfv == null ) return;	// bail if no WFView (no data loaded)
	// get the phase list
	Phase ph[] = (Phase[]) wfv.phaseList.getArray();
	//	wfv.phaseList.toArray(ph);

	PickFlag pf;

	// Note that PickFlag.draw(g) will not draw 'deleted' phases
	for (int i = 0; i < ph.length; i++) {

       if (! ph[i].isDeleted()) {
	    pf = new PickFlag (this, ph[i]);
	    pf.setShowDescription(showPhaseDescriptions);
	    pf.draw(g);
       }
	}
    }
/** Turns on or off the display of phase descriptions in the "flag" part
* of the pick flags. */
public void showPhaseDescriptions (boolean tf) {
	 showPhaseDescriptions = tf;
}
/**
 * Plot the amplitudes on the Waveform
 */
    public void paintAmpFlags (Graphics g)
    {
        if ( wfv == null ) return;	// bail if no WFView (no data loaded)
	// get the phase list
	Amplitude amp[] = (Amplitude[]) wfv.ampList.getArray();

	AmpFlag flag;

	// Note that AmpFlag.draw(g) will not draw 'deleted' phases
	for (int i = 0; i < amp.length; i++) {

         if (! amp[i].isDeleted()) {
	      flag = new AmpFlag (this, amp[i]);
	      flag.draw(g);
         }
	}
    }
/**
 * Plot the codas on the waveform
 */
    public void paintCodaFlags (Graphics g)  {

        if ( wfv == null ) return;	// bail if no WFView (no data loaded)
	// get the phase list
	Coda coda[] = (Coda[]) wfv.codaList.getArray();

	CodaFlag flag;

	// Note that AmpFlag.draw(g) will not draw 'deleted' phases
	for (int i = 0; i < coda.length; i++) {
         if (! coda[i].isDeleted()
             && !(coda[i].getWindowTimeAmpPairs().size() == 0)) // DK 082802 added for codadurs without AvAmpWindows
         {
  	      flag = new CodaFlag (this, coda[i]);
	      flag.draw(g);
         }
	}
    }
/**
* Paint a graphical marker to show where P & S phases are expected to lie.
* Its just a colored rectangle.
*/
    public void paintPhaseCues(Graphics g) {

        if ( wfv == null ) return;	// bail if no WFView (no data loaded)
        Solution sol = wfv.mv.getSelectedSolution();
        if (sol == null || sol.datetime.isNull()) return;  // no location

        // note solution must be the first arg.
        double ptime = wfv.tt.getTTp( sol.getLatLonZ(), wfv.getChannelObj().latlonz);
        double stime = ptime * wfv.tt.getPSRatio();

        // change from traveltime to absolute epoch time
        ptime += sol.datetime.doubleValue();
        stime += sol.datetime.doubleValue();

        cue.set(this, ptime);
        cue.draw(g);
        cue.set(this, stime);
        cue.draw(g);

    }

/**
 * Plot the phases on the waveform
 */
//TODO: could set an optional mode that would show deleted phases
/*
    public void paintPickFlags (Graphics g)
    {

	if (pickFlagList == null) return;

	// get array of PickFlags
	PickFlag pf[] = new PickFlag[pickFlagList.size()];
	pickFlagList.toArray(pf);

	for (int i = 0; i < pf.length; i++) {
	    pf[i].draw(g);
	}
    }
*/
/**
 * Paint the time scale. Also paints at bounding box.
 *
 */
    public void paintScale (Graphics g)
    {

        Color scaleColor = ScaleColor;
// Hardwired test to indicating bad time with red time ticks
	if (wfv.getWaveform() != null &&
	    wfv.getWaveform().getTimeQuality().doubleValue() < mv.getClockQualityThreshold()) {

//	      System.out.println(wfv.getWaveform().toString() +"  "+
//	      wfv.getWaveform().getTimeQuality().doubleValue());

	  scaleColor = Color.red;

	}

	if (getShowTimeScale()) {

	    g.setColor (scaleColor);

	    // paint the bounding box
	    g.drawRect (0, 0, getSize().width, getSize().height);

	    //	g.drawLine (0, 0, getSize().width, 0);	//top horizontal line
	    //	g.drawLine (0, getSize().height-1,
	    //	    getSize().width, getSize().height-1); //bottom horizontal line

	    int smallTick = getSize().height / 30;
	    int tick;

	    double startTime =  panelBox.getStartTime();

	    double frac = startTime - (double) ((long) startTime);
	    double startSec = startTime - frac;
	    double sec;

	    for (int tsec=1; tsec < panelBox.getTimeSize(); tsec++)
		{
		    sec = startSec + (double)tsec;
		    tick = smallTick;
		    if (sec% 5 == 0) tick = smallTick * 2; // 5 sec
		    if (sec%10 == 0) tick = smallTick * 3;  // 10 sec
		    if (sec%60 == 0) tick = smallTick * 4;  // 60 sec
		    int x = pixelOfTime(sec);
		    g.drawLine (x, 0, x, tick);

		    if (showTimeLabel) {
			if (sec%60 == 0) {
			    String label = EpochTime.epochToString (sec, "HH:mm");
			    g.drawString(label, x+2, 12);
			}
		    }
		}
	}

	if (getShowAmpScale()) {
	  // not implemented
	}

    } // end of paintScale()

/**
 * Return a rectangle in pixels for the given WFSelectionBox.
 * This is a method of WFPanel because it needs the WFPanel
 * scaling information.
 */
public Rectangle rectOfBox(WFSelectionBox box)
{
    return new Rectangle (
	pixelOfTime(box.getStartTime()),	       // x
	pixelOfAmp (box.maxAmp),		       // y
	(int) (box.getTimeSize() * pixelsPerSecond),   // width
	(int) (box.getAmpSize() * pixelsPerCount) );   // height

}

/**
 * Return a rectangle in pixels for the given WFSelectionBox.
 * This is a method of WFPanel because it needs the WFPanel
 * scaling information.
 */
public WFSelectionBox boxOfRect(Rectangle rec)
{
    TimeSpan ts = new TimeSpan(dtOfPixel(rec.x),
			       dtOfPixel(rec.x + rec.width));

    return new WFSelectionBox (ts,
			       (int) ampOfPixel(rec.y),
			       (int) ampOfPixel(rec.y + rec.height));
}

/**
 *  Return dt Time of this x position in the panel
 */
    double dtOfPixel (Point pt)
    {
      return dtOfPixel(pt.x) ;
    }

    double dtOfPixel (int x)
    {
      return (panelBox.getStartTime() + ((double)x / pixelsPerSecond) ) ;
    }

/**
 * Calculate the pixel on the WFPanel that corrisponds to this time.
 */
    int pixelOfTime (double dt)
    {
      // round() finds closest pixel, returns long so cast to int
      return (int) Math.round( (dt - panelBox.getStartTime()) * pixelsPerSecond);
    }

/**
 * Return the amplitude of y position
 */
    double ampOfPixel (Point pt)
    {
      return ampOfPixel (pt.y);
    }
/**
 * Return the amplitude of y position
 */
    double ampOfPixel (int y)
    {
      return  ((double) -( (y / pixelsPerCount) - transY)) ;
    }
/**
 * Return the amplitude of y position
 */
    int pixelOfAmp (double amp)
    {
      return (int) Math.rint( (transY - amp) * pixelsPerCount) ;
    }
    /** Return a WFSelectionBox representing the part of this WFPanel that is
    * visible. Simply returns the panelBox. */
    public WFSelectionBox getVisibleBox() {
           return panelBox;
    }
    /** Return the point in (x,y) pixels of the center point of this panel. */
//    public Point getCenterPoint () {
//      // need center of scroll window!
//      return new Point (getWidth()/2, getHeight()/2);
//    }

/** **************************************************************************************
 * Scrollable interface methods
 * These methods determine the behavior of this component if it is placed in a JScrollPane.
 * @See: Scrollable
 */

    boolean trackViewportX = false;
    boolean trackViewportY = false;

    public Dimension getPreferredScrollableViewportSize()
    {
	return getPreferredSize();
    }

    public boolean getScrollableTracksViewportWidth()
    {
	return trackViewportX;
    }

    public boolean getScrollableTracksViewportHeight()
    {
	return trackViewportY;
    }
    public void setScrollableTracksViewportWidth(boolean tf)
    {
	trackViewportX = tf;;
    }
    public void setScrollableTracksViewportHeight(boolean tf)
    {
	trackViewportY = tf;
    }

/**
 * Determines scrolling increment when "track" is clicked. (Coarse scrolling)
 * Default is 1/10 of the WFPanel dimension. Override this method to change this
 * behavior.
 */
    public int getScrollableBlockIncrement (Rectangle visRec, int orient, int dir)
    {
	if (orient == SwingConstants.VERTICAL)
	{
	    return getSize().height/10;	    // jump 1/10 the height
	} else {
	    return getSize().width/10;	    // jump 1/10 the width
	}
    }

/**
 * Determines scrolling increment when arrow widgit is clicked. (Fine scrolling)
 * Default is 1 pixel. Override this method to change this behavior.
 */
//TODO: because of rounding, repeated scroll jumps cause cumulative error in whole.
// :. should calc. jump amt. to be explicit to next sample.
    public int getScrollableUnitIncrement (Rectangle visRec, int orient, int dir)
    {
	if (orient == SwingConstants.VERTICAL)
	{
	    return 5;	    // jump 1/10 the height
	} else {
	    return  Math.max(1, getWidth() / 100);
	}

    }

/*
 * End of Scrollable interface methods
 ************************************************************************************** */
// //////////////////////////////////////////////////////
/*
* This needs more work. The width does not adjust to the text within the panel.
*/
class WFRowHeader extends JPanel
{
//  ColumnLayout layout = new ColumnLayout();
  // Use of a BoxLayout here does not work! Can't control the size of the panel
  // with setPreferredSize() it sizes the box using the contents.
  //BoxLayout layout = new BoxLayout(this, BoxLayout.Y_AXIS);

  WFPanel wfp;
  int fontSize = 2;
  int widest = 50;

  JLabel htmlLabel = new JLabel();

  // constructor
    public WFRowHeader(WFPanel wfpanel) {
        wfp = wfpanel;
	setDefaultText() ;
        add(htmlLabel, JLabel.CENTER);
	setBorder(BorderFactory.createLineBorder(Color.black, 1));
	setMinimumSize(new Dimension(1,1));  // doesnt work
    }

    /** Set the text in tool tip and Panel row label */
    void setDefaultText() {
      if (wfp == null || wfp.wfv == null || wfp.wfv.chan == null) {
	setToolTipText("");
	htmlLabel.setText("");

      } else {

	String strtt = wfp.wfv.chan.getNet() +" "+ wfp.wfv.chan.getSta() +
	      " "+ wfp.wfv.chan.getSeedchan() +
	      " dist= "+df1.format(wfp.wfv.getDistance());

	if (wfp.getWf() != null) {
	  strtt +=
	      " amp min/max= "+wfp.getWf().getMinAmp()+"/"+wfp.getWf().getMaxAmp() +
	      " "+wfp.getWf().getAmpUnits()+
	      " bias= "+wfp.getWf().getBias();
	}
	setToolTipText(strtt);
	// labels can interpret HTML (Java v1.3 and higher)
	String str = "<html>"+
		     "<b>"+
		     wfp.wfv.chan.getNet()+"<p>"+
		     wfp.wfv.chan.getSta() + "<p>"+
		     wfp.wfv.chan.getSeedchan() +"<p>"+
		     "</b>"+
		    "</html>";
	htmlLabel.setText(str);
      }
//	setAlignmentX(JComponent.LEFT_ALIGNMENT);
//	setAlignmentY(JComponent.CENTER_ALIGNMENT);
//	htmlLabel.setAlignmentX(JComponent.LEFT_ALIGNMENT);
//	htmlLabel.setAlignmentY(JComponent.CENTER_ALIGNMENT);

    }
/*
    void setText() {
	setToolTipText(
	    wfp.wfv.chan.getNet() +" "+ wfp.wfv.chan.getSta() +
	    " "+ wfp.wfv.chan.getSeedchan() +
	    " dist= "+wfp.wfv.getDistance() +
	    " amp min/max= "+wfp.getWf().getMinAmp()+"/"+wfp.getWf().getMaxAmp() +
	    + " "+wfp.getWf().getAmpUnits()+
	    +" bias= "+wfp.getWf().getBias());

	label[0].setText( wfp.wfv.chan.getNet()+" "+ wfp.wfv.chan.getSta());
	// trying to keep track of widest label, doesn't work worth a shit.
	widest = Math.max(widest, (int)label[0].getPreferredSize().getWidth());
	label[1].setText( wfp.wfv.chan.getSeedchan() );
	label[2].setText( ""+wfp.wfv.getDistance());
	label[3].setText( wfp.getWf().getMinAmp()+"-"+wfp.getWf().getMaxAmp() );
    }
*/
/** Reports the preferred size of this component as equal to the height of
 *  the WFPanel it represents. This keeps the row header aligned with the
 *  WFPanels in the WFGroupPanel. */
    public Dimension getPreferredSize () {
	return new Dimension (40, wfp.getHeight());
    }

    /** WFRowHeader adjusts its font size a bit when resized. */
    public void paintComponent (Graphics g) {

      setSize(getPreferredSize());

      // make sure the rowheader text is up to date, something may have changed
      setDefaultText();

      // scale font size to size of viewport (within max/min of 8/14)
/*      int newFontSize = Math.max(8, getPreferredSize().height/15);
      newFontSize = Math.min(newFontSize, 14);

      if (newFontSize != fontSize) {
  	  fontSize = newFontSize;
	  Font font = new Font("Serif", Font.PLAIN, fontSize);
//	  Font font = new Font("Lucida Sans", Font.PLAIN, fontSize);
	  for (int i = 0; i<this.getComponentCount(); i++) {
	    this.getComponent(i).setFont(font);
	  }
      }
*/
      super.paintComponent(g);
    }

} // end of class WFRowHeader

} // end of WFPanel class

