package org.trinet.jasi;

import java.sql.Date; 
import java.sql.Connection;
import java.util.*;
import java.io.*;

import org.trinet.jdbc.datatypes.*;

import org.trinet.util.gazetteer.LatLonZ;
import org.trinet.util.gazetteer.GeoidalUnits;

import org.trinet.util.DateTime;
import org.trinet.util.BenchMark;

/**
 * Seed channel object. Contains a ChannelName member which includes
 * descriptive information for the channel like location
 * of the site (latlonz) and response info. <p>
 *
 * @See: org.trinet.jasi.ChannelName
 */

public abstract class Channel extends JasiObject
   implements Cloneable, AuthChannelIdIF, java.io.Serializable, Channelable
{
    /** Channel description identifier */
    protected ChannelName channelId = new ChannelName();

    /** Location of the sensor. Note that 'z' is in km with down negative. */
    public LatLonZ latlonz = new LatLonZ();

    /** Emplacement depth in kilometers. Zero unless this is a downhole sensor. */
    public DataDouble depth = new DataDouble(0.0);

    /** Distance of this channel from the last point for which calcDistance was called.
    * Usually it the associated solution in km. Used for distance sorting, etc. Defaults to
    * 9999.9 so channels with unknown distances show up at the END of a list. This is the true
    * distance taking into account depth and elevation. */
    public DataDouble  dist = new DataDouble(9999.9);

    /** Horizontal distance of this channel.*/
    public DataDouble hdist = new DataDouble(9999.9);

    /** Azimuth from the station/channel to the associated Solution. Degrees
        clockwise from north.*/
    public DataDouble azimuth = new DataDouble();

    /** Nominal sample rate (samples/second) expected for this channel.
     * The actual data rate of retreived time series may differ.*/
    public DataDouble sampleRate = new DataDouble();

    // Response response;
    /** The instrument gain.<br>
  Units are counts/cm/sec for velocity or counts/cm/sec^2 for acceleration.
    */
    public ChannelGain gain = new ChannelGain();

    /** ML correction. Static correction to be added to calculated ML. */
    public DataDouble mlCorr = new DataDouble();

    /** ME correction. Static correction to be added to calculated ME. */
    public DataDouble meCorr = new DataDouble();

    /** Mca correction. Static correction to be added to calculated Mca.
     * This is not currently read from the database. */
    public DataDouble mcaCorr = new DataDouble();

     /** If true only currently active channels will be added to the list. */
    static boolean getCurrent = true;

    protected static boolean debug = false;

// -- Concrete FACTORY METHODS ---
    /**
     * Factory Method: This is how you instantiate a jasi object. You do
     * NOT use a constructor. This is so that this "factory" method can create
     * the type of object that is appropriate for your site's database.
     * If no type is specifed it creates a Channel object of the DEFAULT type.<br>
     * @See: JasiObject
     */
     public static final Channel create() {
  return create(DEFAULT);
     }

    /**
     * Instantiate an object of this type. You do NOT use a constructor.
     * This "factory" method creates various concrete implementations.
     * The argument is an integer implementation type.<br>
     * @See: JasiObject
     */
     public static final Channel create(int schemaType) {
        return create(JasiFactory.suffix[schemaType]);
     }

    /**
     * Instantiate an object of this type. You do NOT use a constructor.
     * This "factory" method creates various concrete implementations.
     * The argument is as 2-char implementation suffix.<br>
     * @See: JasiObject
     */
     public static final Channel create(String suffix) {
  return (Channel) JasiObject.newInstance("org.trinet.jasi.Channel", suffix);
     }
// ////////////////////////////////////////////////////////////////////////////



  /** Set this channel's name. The components of the channel name are copied
   *  to the ChannelName object.
   *  Returns Channel so you can construct with call like:
   *  Channel chan = Channel.create().setChannelName("CI", "COK", "EHZ");*/
    public Channel setChannelName (ChannelName cname) {
  channelId = (ChannelName) cname.clone();
  return this;
    }

/**
 * Set these parts of the channel name. Unspecified parts are left null.
 */
    public Channel setChannelName (  String net, String sta, String comp){
  channelId = new ChannelName(net, sta, comp);
  return this;
    }

/**
 * Set these parts of the channel name. Unspecified parts are left null.
 */
  public Channel setChannelName (  String net,
        String sta,
        String comp,
        String seedchan)
 {
     channelId = new ChannelName(net, sta, comp, seedchan);
     return this;
 }
/**
 * Set these parts of the channel name. Unspecified parts are left null.
 */
  public Channel setChannelName (  String net,
        String sta,
        String comp,
        String auth,
        String subsource)
 {
     channelId = new ChannelName(net, sta, comp, auth, subsource);
     return this;
 }
/**
 * Set these parts of the channel name. Unspecified parts are left null.
 */
  public Channel setChannelName (  String net,
        String sta,
        String comp,
        String auth,
        String subsource,
        String channelsrc,
        String seedchan,
        String location)
 {
     channelId = new ChannelName(net, sta, comp, auth, subsource, channelsrc, seedchan, location);
     return this;
 }
 /** Return the ChannelName object of this Channel */
    public ChannelName getChannelName() {
        //return this;
        return channelId;
    }

    public AuthChannelIdIF getChannelId() {
        //return this;
        return channelId;
    }

// Implement AuthChannelIdIF
/** Return ASCII representation of the channel name with the fields delimited
 *  by the given char. */
    public String toDelimitedString(char delimiter) {
        return channelId.toDelimitedString(delimiter);
    }
/** Return ASCII representation of the channel name with the fields delimited
 *  by the given string. */
    public String toDelimitedString(String delimiter) {
        return channelId.toDelimitedString(delimiter);
    }
/** Return ASCII representation of the channel name with the fields delimited
 *  by "." */
    public String toDelimitedString() {
        return channelId.toDelimitedString();
    }
/** Return ASCII representation of the channel SEED name with the fields delimited
 *  by the given char. */
    public String toDelimitedSeedString(char delimiter) {
        return channelId.toDelimitedSeedString(delimiter);
    }
/** Return ASCII representation of the channel SEED name with the fields delimited
 *  by the given string. */
    public String toDelimitedSeedString(String delimiter) {
        return channelId.toDelimitedSeedString(delimiter);
    }
/** Return ASCII representation of the channel SEED name with the fields delimited
 *  by "." */
    public String toDelimitedSeedString() {
        return channelId.toDelimitedSeedString();
    }
    public String getSta() {
        return channelId.getSta();
    }
    public String getNet() {
        return channelId.getNet();
    }
    public String getSeedchan() {
        return channelId.getSeedchan();
    }
    public String getLocation() {
        return channelId.getLocation();
    }
    public String getChannel() {
        return channelId.getChannel();
    }
    public String getChannelsrc() {
        return channelId.getChannelsrc();
    }
    public String getAuth() {
        return channelId.getAuth();
    }
    public String getSubsource() {
        return channelId.getSubsource();
    }
    public void setSta(String sta) {
        channelId.setSta(sta);
    }
    public void setNet(String net) {
        channelId.setNet(net);
    }
    public void setLocation(String location) {
        channelId.setLocation(location);
    }
    public void setChannel(String channel) {
        channelId.setChannel(channel);
    }
    public void setSeedchan(String seedchan) {
        channelId.setSeedchan(seedchan);
    }
    public void setChannelsrc(String channelsrc) {
        channelId.setChannelsrc(channelsrc);
    }
    public void setAuth(String auth) {
        channelId.setAuth(auth);
    }
    public void setSubsource(String subsource) {
        channelId.setSubsource(subsource);
    }

    public double getSampleRate() {
        return sampleRate.doubleValue();
    }
    public void setSampleRate (double rate) {
        sampleRate.setValue(rate);
    }
// End of AuthChannelIdIF
    /**
     * Return 'TRUE' if the channel is a vertical component (V or Z)
     */
    public boolean isVertical() {
        String channel = channelId.getChannel();
  if (channel.length() >= 3) {		// guard against short components
      char comp = channel.charAt(2);	//get 3rd character (starts with 0)
      if (comp == 'V' || comp == 'Z') return true;
  }
  return false;
    }

    /** Return true if this channel is a low-gain component */
    public boolean isLowGain() {
        return getChannelName().getSeedchan().substring(1,2).equalsIgnoreCase("L");
    }
    /** Return true if this channel is a high-gain component */
    public boolean isHighGain() {
        return getChannelName().getSeedchan().substring(1,2).equalsIgnoreCase("H");
    }
  /** Return true if the channel is a broadband (SEED 'B__') */
    public boolean isBroadBand() {
        return getChannelName().getSeedchan().substring(1,1).equalsIgnoreCase("B");
    }
    /** Set flag that if true will cause only currently active channels to be
    * added to the static channel list.*/
    public static void setCurrentFlag (boolean tf) {
           getCurrent = tf;
    }

    /** Returns the value of the flag that if true will cause only currently
    * active channels to be added to the static channel list.*/
    public static boolean getCurrentFlag () {
           return getCurrent;
    }

    /** Calculate and set both epicentral (horizontal) distance and true (hypocentral)
     * distance of this channel from the given location in km. Returns hypocentral
     * distance. Also sets azimuth.
     * Note: this is better than setDistance() or setHorizontalDistance() because
     * it does both and insures they are consistent. If the channel's LatLonZ or
     * the LatLonZ in the argument is null, both distances are set = 9999.9*/
    static final double NULL_DIST = 9999.9;
    public double calcDistance (LatLonZ loc) {

        if (loc == null || loc.isNull() || !hasLatLonZ()) {  // lat/lon unknown
        setDistance(NULL_DIST);				// so they'll be at END of a sorted list
              setHorizontalDistance(NULL_DIST);
           // leave azimuth null
        } else {
        setDistance(this.latlonz.distanceFrom(loc));
        setHorizontalDistance(this.latlonz.horizontalDistanceFrom(loc));
              setAzimuth(this.latlonz.azimuthTo(loc));
        }
        return getDistance();
    }

    /** Set hypocentral distance ONLY. NOTE: calcDistance(LatLonZ) is prefered because
    * it sets both hypocentral and horizontal distance. */
    public void setDistance(double distance) {
      dist.setValue(distance);
    }
    /** Set horizontal distance ONLY. NOTE: calcDistance(LatLonZ) is prefered because
    * it sets both hypocentral and horizontal distance. */
    public void setHorizontalDistance(double distance) {
      hdist.setValue(distance);
    }


    public double getDistance() {
      return dist.doubleValue();
    }
    public double getHorizontalDistance() {
      return hdist.doubleValue();
    }

    public void setAzimuth(double az) {
      azimuth.setValue(az);
    }
    public double getAzimuth() {
      if (azimuth == null || azimuth.isNull() || azimuth.doubleValue() == Double.NaN) return 0.0;
      return azimuth.doubleValue();
    }

    /** Needed for Channelable interface. */
    // Can't call it getChannel() because that's used to return channel string
    public Channel getChannelObj() {
        return this;
    }
    public void setChannelObj(Channel chan) {
        this.copy(chan);
    }

     /** Return true if the channel has a vaild LatLonZ, false if not. */
     public boolean hasLatLonZ () {
        return  !(this.latlonz == null || this.latlonz.isNull()) ;
     }

     /** Return the index of this channel in this collection of Channelable objects.
     * Returns -1 if list is empty, it doesn't contain Channelable objects or this
     * channel is not in the list. */
     public int indexOf (Collection list) {

         if (list == null || list.isEmpty()) return -1;

         ArrayList alist = (ArrayList) list;

         if (!(alist.get(0) instanceof Channelable)) return -1;

         Channel channel;
         for (int idx = 0; idx < list.size(); idx++) {
            channel = ((Channelable) alist.get(idx)).getChannelObj();
            if (this.equalsIgnoreCase(channel)) return idx;
         }

         return -1;
     }

/**
 * Make a "deep" copy.
 */
    public Channel copy(Channel chan)
    {
  channelId = (ChannelName) chan.getChannelId().clone();

  latlonz.copy( chan.latlonz );

  depth   = new DataDouble(chan.depth);
  dist    = new DataDouble(chan.dist);
  azimuth = new DataDouble(chan.azimuth);
  gain    = new ChannelGain(chan.gain);
  mlCorr  = new DataDouble(chan.mlCorr);
  meCorr  = new DataDouble(chan.meCorr);

  return this;
    }

    /**
     * Return brief channel name with LatLonZ.
     */
    public String toString(){

  return channelId.toDelimitedString(' ') + " " + latlonz.toString() ;
    }
    /**
     * Complete object dump. For debugging. Overides ChannelName.toDumpString()
     * to include LatLonZ.
     */
    public String toDumpString(){
  String str =
      " net: " + channelId.getNet()+
      " sta: " + channelId.getSta() +
      " channel: " + channelId.getChannel() +
      " seedchan: " + channelId.getSeedchan() +
      " location: " + channelId.getLocation()+
      " channelsrc: " + channelId.getChannelsrc() +
      " LatLonZ= "+ latlonz.toString() +
      " gain= "+ gain.doubleValue() +
      " mlCorr= "+mlCorr+
      " meCorr= "+meCorr;
  return str;
    }
    /** Return a string with all the members of a Channel object, delimited
    * by the given string. The string must be a single character if you
    * will read it back with parseCompleteString(). It also, should not be
    * a character that would appear in any of the members (letters and numbers). */
    public String toCompleteString(String delim) {
       return  toCompleteString (this, delim);
    }

    /** Return a string with all the members of a Channel object, delimited
    * by the given string. The string must be a single character if you
    * will read it back with parseCompleteString(). It also, should not be
    * a character that would appear in any of the members (letters and numbers). */
    public static String toCompleteString(Channel chan, String delim) {
      return chan.channelId.toCompleteString(delim) +
      delim + chan.latlonz.toString(delim) +
      delim + chan.gain.doubleValue() +
      delim + chan.gain.getUnitsString() +
      delim + chan.mlCorr.doubleValue() +
      delim + chan.meCorr.toString() ;
         //.doubleValue();
    }

    public boolean binaryWrite (DataOutputStream out) {
       return binaryWrite(this, out);
    }
    public boolean binaryWrite (Channel chan, DataOutputStream out) {

      boolean status = true;
      try {
       out.writeUTF(chan.getNet());
       out.writeUTF(chan.getSta());
       out.writeUTF(chan.getChannel());
       out.writeUTF(chan.getSeedchan());
       out.writeUTF(chan.getChannelsrc());
       out.writeUTF(chan.getLocation());

       out.writeDouble(chan.latlonz.getLat());
       out.writeDouble(chan.latlonz.getLon());
       out.writeDouble(chan.latlonz.getZ());

       out.writeDouble(chan.gain.doubleValue());
       out.writeInt(chan.gain.getUnits());
       out.writeDouble(chan.mlCorr.doubleValue());
       out.writeDouble(chan.meCorr.doubleValue());
      } catch (IOException ex) {
            ex.printStackTrace();
            status = false ;
      }
      return status;
    }

    public static Channel binaryRead (DataInputStream in) {

       Channel chan = Channel.create();

       try {

       chan.setNet(in.readUTF());
       chan.setSta(in.readUTF());
       chan.setChannel(in.readUTF());
       chan.setSeedchan(in.readUTF());
       chan.setChannelsrc(in.readUTF());
       chan.setLocation(in.readUTF());

       chan.latlonz.set(in.readDouble(), in.readDouble(), in.readDouble());

       chan.gain.gain.setValue(in.readDouble());
       chan.gain.setUnits(in.readInt());

       chan.mlCorr.setValue(in.readDouble());
       chan.meCorr.setValue(in.readDouble());
      } catch (IOException ex) {
            ex.printStackTrace();
      }

       return chan;

    }

    /** Parse a string for the format created by toCompleteString(). */
    public static Channel parseCompleteString (String str, String delim) {

       StringTokenizer strTok = new StringTokenizer(str, delim);

       Channel chan = Channel.create();

       try {
      chan.channelId = ChannelName.parseCompleteString(strTok);
      chan.latlonz = LatLonZ.parse(strTok);
         chan.gain.gain.parseValue(strTok);
         chan.gain.setUnits(strTok.nextToken());
         chan.mlCorr.parseValue(strTok);
         chan.meCorr.parseValue(strTok);

       } catch (NoSuchElementException ex) {}

       return chan;
    }

// /////////////////////////////////////////////////////////////////////////////

    public Object clone() {
        Channel chan = null;
        try {
            chan = (Channel) super.clone();
        }
        catch (CloneNotSupportedException ex) {
            ex.printStackTrace();
        }
        chan.channelId = (ChannelName) channelId.clone();
  chan.latlonz = (LatLonZ) latlonz.clone();
  chan.depth   = (DataDouble) depth.clone();
  chan.dist    = (DataDouble) dist.clone();
  chan.azimuth = (DataDouble) azimuth.clone();
  chan.gain    = (ChannelGain) gain.clone();
  chan.mlCorr  = (DataDouble) mlCorr.clone();
  chan.meCorr  = (DataDouble) meCorr.clone();
        return chan;
    }

    /**
     * Case sensitive compare. Returns true if sta, net and seedchannel are
     * equal. Does not check the other fields.  The "extra" fields like
     * 'channel' and 'location' are populated in some data sources and not
     * others.
     */
    public boolean equals(Object x) {
        if (this == x) return true;
  else if ( x == null || ! (x instanceof Channel)) return false;
        Channel chan = (Channel) x;
  return(this.channelId.equals(chan.getChannelId()));
    }

    /**
     * Case insensitive compare. Returns true if sta, net and channel are
     * equal. Does not check the other fields.  Made necessary because not all
     * data sources populate all the fields.  Force a trim to remove trailing
     * blanks just in case the reader didn't.  ALSO returns true if sta & net
     * match and channel equals either the 'channel' OR <it> seedchan </it>
     * value. Does NOT check the value of 'location'. <p>
     * This includes a horrible kludge to get around the problem of renamed analog
     * channels. Considers "EHZ" == "VHZ"*/

    public boolean sameAs(Object x) {
  if ( !(x instanceof Channel) ) return false;
  Channel ch = (Channel) x;
  return ch.getChannelName().sameAs(channelId);
    }

    /** Returns true if the channels are from the same station. */
    public boolean sameStationAs(Object x) {
  if ( !(x instanceof Channel) ) return false;
  Channel ch = (Channel) x;
  return ch.getChannelName().sameStationAs(channelId);
    }

    /**
     * Case insensitive compare. Returns true if sta, net, and  channel or
     * 'seedchan' attributes are equal. <p>
     * ALSO returns true if sta & net match and  channel equals either the
     * 'channel' OR <it> seedchan </it> value. Does NOT check the value of 'location'. */
    public boolean equalsIgnoreCase(Object x) {
        if (this == x) return true;
  else if ( x == null || ! (x instanceof Channel)) return false;
        Channel chan = (Channel) x;
  return(this.channelId.equalsIgnoreCase(chan.getChannelId()));
    }
/*    public boolean equalsIgnoreCase(Object x) {
  if ( !(x instanceof Channel) ) return false;
  Channel ch = (Channel) x;
  return ch.getChannelName().equalsIgnoreCase(channelId);
    }
*/
/**
 * This uses just the essentials as in equals().
 */
    public int hashCode() {
        return channelId.hashCode();
    }
    protected void setHashValue() {
        channelId.setHashValue();
    }

/** Returns the result of comparing the strings generated by invoking toString() for this instance and the input object.
* Throws ClassCastException if input object is not an instanceof Channel.
* A return of 0 == value equals, <0 == value less than, >0 == value greater than, the string value of the input argument.
*/
    public int compareTo(Object x) {
  if (x == null ||  x.getClass() != getClass() )
     throw new ClassCastException("compareTo(object) argument must be a Channel class type: "
                                + x.getClass().getName());
  return channelId.toString().compareTo( ((Channel) x).getChannelId().toString());
    }
    /**
     * Look up the given Channel in the DataSource dbase and return a fully
     * populated Channel object. This is how you look up LatLonZ and response
     * info for a single channel. Note this finds the LATEST entry if more then
     * one entry is in the dbase (largest 'ondate'). But, it does NOT require that
     * the channel be "active". If no entry is found, this
     * method return the channel name that was passed in the arg as a Channel.
     */
    public static Channel lookUp (Channel chan) {
       return create().lookUpChannel(chan);
    }
// /////////////////////////////////////////////////////////////////////////////
// Abstract methods

    /**
     * Look up the given Channel in the DataSource dbase and return a fully
     * populated Channel object. This is how you look up LatLonZ and response
     * info for a single channel. Note this finds the LATEST entry if more then
     * one entry is in the dbase (largest 'ondate'). But, it does NOT require that
     * the channel be "active". If no entry is found, this
     * method return the channel name that was passed in the arg as a Channel.
     */
    public abstract Channel lookUpChannel (Channel chan) ;
/**
 * Return exhaustive Collection of Channels from the default DataSource.
 */
    public abstract ChannelList readList();
/**
 * Return Collection of Channels that were active on the given date.
 * Uses the default DataSource Connection.
 */
    public abstract ChannelList readList(java.sql.Date date);
/**
 * Return Collection of Channels that were active on the given date.
 * Uses the given DataSource Connection.
 */
    public abstract ChannelList readList(Connection conn, java.sql.Date date);
    /** Return a list of channels that were active on the given date and
    *  that match the list of component types given in
    * the array of strings. The comp is compared to the SEEDCHAN field in the NCDC
    * schema.<p>
    *
    * SQL wildcards are allowed as follows:<br>
    * "%" match any number of characters<br>
    * "_" match one character.<p>
    *
    * For example "H%" would match HHZ, HHN, HHE, HLZ, HLN & HLE. "H_" wouldn't
    * match any of these but "H__" would match them all. "_L_" would match
    * all components with "L" in the middle of three charcters (low gains).
    * The ANSI SQL wildcards [] and ^ are not supported by Oracle.*/
    public abstract ChannelList getByComponent(String[] compList,  java.sql.Date date) ;

    /** Return a list of currently acive channels that match the list of
     *  component types given in
    * the array of strings. The comp is compared to the SEEDCHAN field in the NCDC
    * schema.<p>
    *
    * SQL wildcards are allowed as follows:<br>
    * "%" match any number of characters<br>
    * "_" match one character.<p>
    *
    * For example "H%" would match HHZ, HHN, HHE, HLZ, HLN & HLE. "H_" wouldn't
    * match any of these but "H__" would match them all. "_L_" would match
    * all components with "L" in the middle of three charcters (low gains).
    * The ANSI SQL wildcards [] and ^ are not supported by Oracle.*/
    public abstract ChannelList getByComponent(String[] compList);

     /** Return count of all active channels in data source. */
    public abstract int getCurrentCount () ;

    /** Return count of all channels in data source that were active on this date. */
    public abstract int getCount (java.sql.Date date) ;

// ///////////////////////////////////////////////////////////////////////
/**
 * Main for testing:
 */
    public static void main (String args[]){

        System.out.println ("Making connection... ");
  DataSource dbase = new TestDataSource();
  System.out.println (dbase.toString());

    // look up one channel by name

//     System.out.println ("Current channel count = "+getCurrentCount());

  Channel cn = Channel.create().setChannelName("CI", "ALP", "BHE", "BHE");

  if (cn == null) {
      System.out.println ("cn is null.");
      System.exit(0);
  }

  System.out.println ("lookUp :"+cn.toDumpString());

  BenchMark bm = new BenchMark();

  Channel ch = cn.lookUp(cn);

  System.out.println ("found :"+ch.toDumpString());
  bm.print();

     System.out.println ("HypoFormat: /"+HypoFormat.toH2StationString(ch)+"/");
     String str = ch.toCompleteString(" ");
     System.out.println ("CompleteFmt: /"+str+"/");

     Channel newChan = Channel.parseCompleteString(str, " ");
     System.out.println ("CompletePrs: /"+newChan.toCompleteString(" ")+"/");


/*
//
  // NOTE: this demonstrates making a static Channel list.
        System.out.println ("Reading CURRENT channels...");
  bm.reset();
  //	Channel.setList (Channel.getAllList());
  Channel.setList (Channel.getCurrentList());

  //        System.out.println ("Reading channels with net=CI...");
  //	ArrayList chanList =
  //	    (ArrayList) Channel.readListWhere("Where net = 'CI'");

  System.out.println ("Found "+Channel.chanList.size()+" active channels.");
  bm.print();

  //
  Channel cn2 = Channel.create().setChannelName("NC", "PTQ", "VHZ", "EHZ");
  System.out.println ("Find "+cn2.toDumpString()+ " in list...");

  Channel found = Channel.findExactInList(cn2);

  if (found == cn2) {
      System.out.println ("No exact match found.");
  } else {
      System.out.println ("Found exact: " + found.toString());
  }

  found = Channel.findSimilarInList(cn2);
  if (found == cn2) {
      System.out.println ("No similar match found.");
  } else {
      System.out.println ("Found similar: " + found.toString());
  }

  //
  cn2 = Channel.create.setChannelName("CI", "HOD", "VHZ", "EHZ");
  System.out.println ("Find "+cn2.toDumpString()+ " in list...");

  found = Channel.findExactInList(cn2);

  if (found == cn2) {
      System.out.println ("No exact match found.");
  } else {
      System.out.println ("Found exact: " + found.toString());
  }

  found = Channel.findSimilarInList(cn2);
  if (found == cn2) {
      System.out.println ("No similar match found.");
  } else {
      System.out.println ("Found similar: " + found.toString());
  }

  // Test sorting

  Channel.distanceSort(new LatLonZ (34.35, -118.86, 8.0));

  // dump list
  Channel[] cha = getArray();
  String str="";

  int len = cha.length;
  //	int len = 100;

  if (cha.length == 0) {
      str = ("No channels in list.");
  } else {
      for (int i = 0; i < len; i++)
      {
    System.out.println (cha[i].toDumpString()+ " dist= "+cha[i].dist);
      }
  }


    // test serialization
    System.out.println ("Reading in current channel info ...");
    //Channel.setList (Channel.getCurrentList());
    Channel.setList (Channel.readListWhere(" Channel_Data.net = 'AZ'"));

    System.out.println ( "Write cache = " + Channel.writeCacheList() );
    System.out.println ("list length = "+Channel.listSize());
    System.out.println (Channel.currentListToString());

    Channel.chanList = null;
     System.out.println ("nuked it, list length = "+Channel.listSize());

    System.out.println ( "Read cache = " + Channel.readCacheList() );
    System.out.println ("list length = "+Channel.listSize());

    System.out.println (Channel.currentListToString());

    */

 }  // end of Main

}   // end of Channel class


