
//package org.trinet.util;
package org.trinet.eventmatch;

import java.math.*;
import java.text.*; 
import java.io.*;

/*******************************************************************
 * THIS WILL BE LOADED INTO ORACLE
 * MUST COMPILE UNDER JDK1.1.X AND USE ONLY LIBRARIES LOADED IN ORACLE
 * (this class duplicates the class in org.trinet.util and is included
 *  here so the whole util package does not need to be compiled in 
 *  JDK 1.1.x and loaded into Oracle)
 *******************************************************************/

/**
 * Geographical location in 3-D: latitude, longitude, depth
 * Northern and eastern hemispheres are positive.
 * Southern and western hemispheres are negative.
 * Depths are negative, elevations are positive. Units are ALWAYS km.
 * If you use a LatLonZ object for something like
 * as station location where meters are a more natural unit, make sure you 
 * convert to km when you set the value of depth. Otherwise, distance
 * calculations will be wrong! Distances are alway returned as km.
 */
public class LatLonZ
{
    static final double RADIUS = 6371.0;    //radius of the earth
    static final double FAC = 0.01745329;   //degrees to radians

    /** Latitude: southern hemisphere is negative */
    public double latitude;
    /** Longitude: western hemisphere is negative */
    public double longitude;

    /** Depth in km. Note if you use a LatLonZ object for something like
     as station location where meters are a more natural unit, make sure you 
     convert to km when you set the value of depth. Otherwise, distance
     calculations will be wrong!*/
    public double depth;

    String grid;

/**
 * Null constructor
 */
public LatLonZ () 
{
}
/**
 * Contruct from existing LatLonZ.
 */
public LatLonZ (LatLonZ llz) 
{
    copy(llz);
}
/** 
 * Decimal degrees 
 */
 public LatLonZ (double lat, double lon, double z)
{
    set (lat, lon, z);
}
/** 
 * Degrees, minutes
 */
 public LatLonZ (int lat, double latm, int lon, double lonm, double z)
{
    set (lat, latm, lon, lonm, z);
}

/**
 * Copy LatLonZ to this one.
 */
public void copy (LatLonZ llz)
{
    set (llz.latitude, llz.longitude, llz.depth);
}

/** 
 * Decimal, degrees
 */
public void set (double lat, double lon, double z)
{
    latitude = lat;
    longitude = lon;
    depth = z;
}
/** 
 * Degrees/minutes. Sign is preserved. If lat or lon is negitive it should only
 * be so in the minutes part.  Sign and units of the depth are preserved.  */
public void set (int lat, double latm, int lon, double lonm, double z)
{
    if (lat > 0) {
	latitude  = (double) lat + (double)latm/60.0;
    } else {
	latitude  = -( (double) Math.abs(lat) + (double)latm/60.0);
    }

    if (lon > 0) {
	longitude = (double) lon + (double)lonm/60.0;
    } else {
	longitude = -( (double) Math.abs(lon) + (double)lonm/60.0);

    }

    depth     = z;
}

public double getLatitude ()
{
    return latitude;
}
public double getLongitude()
{
    return longitude;
}
public double getDepth()
{
    return depth;
}

public static double distanceBetween (LatLonZ l1, LatLonZ l2)
{
    return l1.distanceFrom(l2);
}

/**
 * Calculate the distance from this location in km. 
 * Does include depth in the distance calculation. 
 */
public double distanceFrom(LatLonZ loc0) 
{
    double hdist = horizontalDistanceFrom(loc0);

    double dz    = depth - loc0.depth;		    // depth difference

    return  Math.sqrt(hdist*hdist + dz*dz) ;
}
/**
 * Calculate the horizontal distance from this location in km. 
 * Does NOT include depth in the distance calculation. This is only accurate
 * over smaller distances because it calculates the length of a chord on the
 * surface of a sphere.
 */
public double horizontalDistanceFrom(LatLonZ loc0)   // based on DDIST from CUSP libraries
{

    double dlat = (latitude - loc0.latitude) * FAC;

    double olat = loc0.latitude * FAC;	    // convert to radians
    double olon = loc0.longitude * FAC;

    double rlon = longitude * FAC;

    double y = RADIUS * dlat;
    double x = (rlon - olon) * RADIUS * Math.cos((dlat/2.0)+olat);

    //    System.out.println 
    //     ("dist: "+ this.toString() +" -> "+loc0.toString()+" = "+ Math.sqrt(x*x + y*y));

    return  Math.sqrt(x*x + y*y);
}

public String toString()
{
    DecimalFormat df = new DecimalFormat ( "###0.0000" );
    return    df.format(latitude) + " " + 
	      df.format(longitude) + "  " + 
	      df.format(depth) ;
}

// distance, azimuth, angle
}
