package org.trinet.util.graphics.task;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*; 
import javax.swing.event.*; 
//import org.trinet.util.*;

/**
  *  Extension of a JPanel containing a progress component for
  *  monitoring an AbstractMonitorableTask running in a daemon thread.
  *  Progress component details are defined by concrete subclasses.
  *  This base class contains a repeating timer which is needed for
  *  updating the progress component with task thread's progress values.
*/
public abstract class AbstractTaskMonitorPanel extends JPanel {

    protected EventListenerList listenerList;

    protected AbstractMonitorableTask task;
    protected Object progressMeter;
    protected Timer timer;
    protected boolean notifyResults = true;

    protected boolean confirmTaskStop;
    protected boolean logCompletion;
    protected boolean logProgress;
    protected boolean beepOn;

    public AbstractTaskMonitorPanel(AbstractMonitorableTask task ) {
        this.task = task;
        initPanel();
    }

/**
  * Subclasses must implement the progress component. 
*/
    protected abstract Object    createProgressMeterComponent();
/**
  * Subclasses must implement to update the progress component values. 
*/
    protected abstract void      updateProgressComponent() ;
/**
  * Subclasses must implement to reset the progress component values. 
*/
    protected abstract void      resetProgressComponent() ;
/**
  * Subclasses must implement the panel component layout. 
*/
    protected abstract void      initPanelLayout() ;

/**
  * Initialize AbstractTaskMonitorPanel component to include in a GUI. 
*/
    protected void initPanel() {
        initProgressComponent();
        initPanelLayout();
        initTimer();
    }

/**
  * Initialize progress monitor component of this panel. 
*/
    protected void initProgressComponent() {
        if (progressMeter != null && progressMeter instanceof Component ) remove((Component) progressMeter);
        progressMeter = createProgressMeterComponent();
    }


/**
  * Initialize the timer used to update the progress component. 
*/ 
    protected void initTimer() {
        if (timer != null) timer.stop();
        timer = new Timer(task.getProgressTimerDelay(), new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                updateProgressComponent();
                if (task.isTaskComplete()) {
                    doStopProcessing();
                }
                else if (isLogProgress()) {
                    System.out.println(task.getProgressMessage());
                }
            }
        });
    }

/**
  * Invoke when task thread ends.
  * Fires task completion event.
  * Beeps terminal console if isBeepOn() == true.
  * Logs completion time to System if isLogCompletion() == true.
  * @see #setBeepOn(boolean)
  * @see #setLogCompletion(boolean)
*/
    protected void notifyCompletion() {
        fireTaskCompleted(task.isTaskComplete());
        if (beepOn) Toolkit.getDefaultToolkit().beep();
        if (isLogCompletion())
            System.out.println("Task: " + task.getTaskName() + " completed.");
    }

/**
  * Resets the panel's progress component settings. 
*/
    protected void resetProgressGraphics() {
        resetProgressComponent();
    }

/**
  * If true, task progress updates are logged to System.
*/
    public boolean isLogProgress() {
        return logProgress;
    }

/**
  * Set true to log any implemented task progress updates to System.
*/
    public void setLogProgress(boolean value) {
        logProgress = value;
    }

/**
  * If true, console will beep when task ends. 
*/
    public boolean isBeepOn() {
        return beepOn;
    }

/**
  * Set true to have console beep when task ends. 
*/
    public void setBeepOn(boolean value) {
        beepOn = value;
    }

/**
  * If true, the task name and current time are logged to System when task ends. 
*/
    public boolean isLogCompletion() {
        return logCompletion;
    }

/**
  * Set true to have the task name and current time logged to System when task ends. 
*/
    public void setLogCompletion(boolean value) {
        logCompletion = value;
    }

/**
  * Resets the progress component, starts task thread and progress update timer. 
*/
    public synchronized void startTask() {
        if (task.isProcessing()) {
            System.out.println("Prior task still running ... press stop first");
            return;
        }
        resetProgressGraphics();
        notifyResults = true; // enable a single task completed notification
        task.startTask();
        if (timer.isRunning()) timer.restart();
        else timer.start();
        
    }

/**
  * Stop task, stop timer, update progress component and notify task completion. 
*/
    protected void doStopProcessing() {
        if (task.isProcessing()) task.stopTask();
        if (timer.isRunning()) timer.stop();
        updateProgressComponent();
        if (notifyResults) {
           notifyCompletion();
           // disable notification; avoids multiple notification events for single task invocation.
           notifyResults = false;
        }
    }

/**
  * Set true to allow a stopTask() invocation for a running task.
  * to create task termination confirmation dialog. 
*/
    public void setConfirmTaskStop(boolean value) { confirmTaskStop = value;}

/**
  * End task processing. Confirms stop of runngin task if setConfirmTaskStop(true). 
*/
    public synchronized void stopTask() {
        if (confirmTaskStop && ! isStopConfirmed()) return; 
        doStopProcessing();
    }

/**
  * Returns result from the completed AbstractMonitorableTask,
  * Invoking this method will block invoking thread until
  * daemon task thread stops.
*/
    public Object getTaskResult() {
        return task.getTaskResult();
    }

/**
  * Returns true from a confirmation dialog if the YES option is chosen. 
*/
    protected boolean isStopConfirmed() {

        if (! task.isProcessing() ) return true;
        String name = task.getTaskName();
        int status = JOptionPane.showConfirmDialog(getRootPane().getParent(),
                        "Stop " + name + " processing",
                         name,
                         JOptionPane.YES_NO_OPTION);

        return (status == JOptionPane.YES_OPTION);
    }

    public synchronized void addTaskCompletedListener(TaskCompletedListener  taskListener) {
        if (listenerList == null) listenerList = new EventListenerList();
        listenerList.add(TaskCompletedListener.class, taskListener);
    }

    public synchronized void removeTaskCompletedListener(TaskCompletedListener  taskListener) {
        if (listenerList == null) return;
        listenerList.remove(TaskCompletedListener.class, taskListener);
    }

/**
  * Fire TaskCompletedEvent with specified input completion status.
*/
    protected void fireTaskCompleted(boolean completionStatus) {
        if (listenerList == null) return;
        Object[] listeners = listenerList.getListenerList();
        TaskCompletedEvent tce = null;
        for (int index = listeners.length-2; index >= 0; index -= 2) {
            if (listeners[index] == TaskCompletedListener.class) {
                if (tce == null) tce = new TaskCompletedEvent(this, task.getTaskName(), completionStatus);
                ((TaskCompletedListener) listeners[index+1]).taskCompleted(tce);
            }
        }
    }

}
