package org.trinet.util;

import java.lang.*;
import java.lang.Math.*;
import java.text.*;
import java.text.SimpleDateFormat;
import java.util.*;

// AWW - 4/2000 cleaned up the methods and added a few for Date to string etc.
// AWW - Added Sarala's DateConversion exception trapping to this code plus
// default to GMT if no UTC timezone.

/**
 * Allow simple handling of epoch times. Epoch times are elapse seconds since
 * epoch start. Our epoch start time is Jan. 1, 1970 at 00:00. This class
 * provides static methods for conversion from strings to double and back.
 */

/*
  KNOWN BUG IN SimpleDateTimeFormat:

  A string with two decimals in the second will not parse correctly:

Current systemTime: January 11, 2000 01:12:26.41
Parse test of string: January 11, 2000 01:12:26.41
Parse result is     : January 11, 2000 01:12:26.04
                                          Error ^^

NOTE: the "S" format of SimpleDateFormat returns milliseconds
    * as an integer. This is prone to unexpected formatting results.
    * For example, if the seconds part of a time = 12.03456. the "ss.SS" format returns "12.34" and
    * "ss.SSSS" reuturns "12.0034". Only "ss.SSS" gives a correct result.
 */

public class EpochTime {

    public EpochTime() {}

    /*    NOTE: the "S" format of SimpleDateFormat returns milliseconds
    * as an integer. This is prone to unexpected formatting results.
    * For example, if the seconds part of a time = 12.03456. the "ss.SS" format returns "12.34" and
    * "ss.SSSS" reuturns "12.0034". Only "ss.SSS" gives a correct result. */
    static final String DEFAULT_FORMAT_PREFIX = "yyyy-MM-dd HH:mm:";
    static final String DEFAULT_FORMAT = DEFAULT_FORMAT_PREFIX + "ss.SSS z";
    //static final String DEFAULT_FORMAT = "MMMM dd, yyyy HH:mm:ss.SS";

    /** Returns a SimpleDataFormat with the specified Date string pattern for UTC time zone.
    */
    public static SimpleDateFormat getDateFormat(String pattern) {
        return getDateFormat(pattern, "UTC");
    }

    /** Returns a SimpleDataFormat with the specified Date string pattern for specified time zone.
    */
    public static SimpleDateFormat getDateFormat(String pattern, String zone) {
      TimeZone timeZone = TimeZone.getTimeZone(zone);
      if (timeZone == null) timeZone = TimeZone.getTimeZone("GMT");
      SimpleDateFormat df = new SimpleDateFormat(pattern);
      df.setTimeZone(timeZone);

      df.setLenient(true);
      return df;
    }

    /** Returns a date String formatted using the specified pattern for UTC Time zone. */
    public static String dateToString(Date date, String pattern) {
        return dateToString(date, pattern, "UTC");
    }

    /** Returns a date String formatted using the specified pattern and time zone parameters. */
    public static String dateToString(Date date, String pattern, String zone) {
      final String fmtStringShort = "MMMM dd, yyyy HH:mm:ss";
      final Format df2 = new Format("%02d");
      String cstr = null;

        if (date == null) return "null";
	if (pattern == null  || pattern.length() == 0) pattern = DEFAULT_FORMAT;

//        if (pattern == DEFAULT_FORMAT) {
//          // DateTime format for seconds is ALWAYS "xx.xxx"
//          // I want "xx.xx" so have to do this.
//          DateTime datetime = new DateTime(date);
//          double epochsecs = datetime.getEpochSeconds();
//          // tack on fractional part of seconds. Add 0.005 for rounding
//          cstr = datetime.toDateString(DEFAULT_FORMAT_PREFIX)
//                 + datetime.getSecondsStringToPrecisionOf(2);
//    //               + "."+ df2.form((int)((DateTime.fracOf(epochsecs)+.005)*100.0));
//        } else {

          try {
            cstr = getDateFormat(pattern, zone).format(date);
          }
          catch (IllegalArgumentException ie){
            System.err.println("EpochTime - dateToString : Requested Pattern is not available");
          }
          catch (NullPointerException ne){
            System.err.println("EpochTime - dateToString: NullPointerException converting the Date");
          }
//        }
          return cstr;
    }

    /**
     * Returns dateToString(...) using the DEFAULT_FORMAT pattern, UTC time zone:
     * "yyyy-MM-dd HH:mm:ss.SSS z"
     * @see #dateToString(Date, String)
     */
    public static String toString(Date date) {
	return dateToString(date, DEFAULT_FORMAT);
    }

    /**
     * Same as dateToString(...), returns the String date using specified pattern, UTC time zone.
     * @see #dateToString(Date, String)
     */
    public static String toString(Date date, String pattern) {
	return dateToString(date, pattern);
    }

    /**
     * Same as epochToString(...), returns the epoch time as a string for DEFAULT format:
     * "yyyy-MM-dd HH:mm:ss.SSS z"
     */
    public static String toString(double dateTime) {
	return epochToString(dateTime, DEFAULT_FORMAT);
    }

    public static final String YY_NO_ZONE_FORMAT   =  "yy-MM-dd HH:mm:ss.SSS";
    public static final String YYYY_NO_ZONE_FORMAT =  "yyyy-MM-dd HH:mm:ss.SSS";
    public static String toNoZoneYYString(double dateTime) {
	return epochToString(dateTime, YY_NO_ZONE_FORMAT);
    }
    public static String toNoZoneYYYYString(double dateTime) {
	return epochToString(dateTime, YYYY_NO_ZONE_FORMAT);
    }

    /**
     * Returns the epoch time as a string for format:
     * "yyyy-MM-dd HH:mm:ss.SSS z"
     */
    public static String epochToString(double dateTime) {
	return epochToString(dateTime, DEFAULT_FORMAT);
    }

    /**
     * Returns the epoch time as a string with a SimpleDateTime format given in
     * 'pattern' for UTC time zone. If 'pattern' is null or zero length string is returned as
     * "yyyy-MM-dd HH:mm:ss.SSS z". @see java.text.SimpleDateFormat
     */
    public static String epochToString(double dateTime, String pattern ) {
	return dateToString(new java.util.Date((long) Math.round(dateTime*1000.)), pattern );
    }

    /**
     * Returns the epoch time as a string with a SimpleDateTime format given in
     * 'pattern' for specified input time zone. If 'pattern' is null or zero length string is returned as
     * "yyyy-MM-dd HH:mm:ss.SSS z". @see java.text.SimpleDateFormat.
     */
    public static String epochToString(double dateTime, String pattern, String zone ) {
	return dateToString(new java.util.Date((long) Math.round(dateTime*1000.)), pattern, zone );
    }

    /**
     * Returns the epoch time as a string with a SimpleDateTime format given in
     * 'pattern' for PST time zone. If 'pattern' is null or zero length string
     * is returned as "yyyy-MM-dd HH:mm:ss.SSS z". @see java.text.SimpleDateFormat
     */
    public static String epochToPST(double dateTime, String pattern ) {
	return epochToString(dateTime, pattern, "PST");
    }

    /**
     * Returns the epoch time as a string with a SimpleDateTime format given in
     * default pattern for PST time zone. If 'pattern' is null or zero length string
     * is returned as "yyyy-MM-dd HH:mm:ss.SSS z". @see java.text.SimpleDateFormat
     */
    public static String epochToPST(double dateTime) {
	return epochToString(dateTime, DEFAULT_FORMAT, "PST");
    }

    /**
     * Parse a string of format "yyyy-MM-dd HH:mm:ss.SSS" and return epoch seconds.
     */
    public static double stringToEpoch(String UTC) {
	return stringToEpoch(UTC, DEFAULT_FORMAT);
    }

    /**
     * Parse a string of format given by 'pattern' and return epoch seconds.
     */
    public static double stringToEpoch(String UTC, String pattern) {
	Date myDate = stringToDate(UTC, pattern);
        return (myDate == null) ? 0.0 : (double) myDate.getTime()/1000.;
    }

    /**
     * Parse a string of format "yyyy-MM-dd HH:mm:ss.SSS" and return a Date.
     */
    public static Date stringToDate(String UTC) {
	return stringToDate(UTC, DEFAULT_FORMAT);
    }

    /**
     * Parse a string of format given by 'pattern' and return a Date.
     */
    public static Date stringToDate(String UTC, String pattern) {
	return getDateFormat(pattern).parse(UTC, new ParsePosition(0));
    }

    /**
     * Convert an java.util.Date to epoch double.
     */
    public static double dateToEpoch(Date date) {
	return (double) date.getTime() / 1000. ;
    }

    /**
     * Convert an epoch time double to a java.util.Date
     */
    public static Date epochToDate(double epochTime) {
	return new Date(Math.round(epochTime * 1000.));
    }

/** Return absolute value of seconds between end and start times */
    public static double elapsedSeconds(java.sql.Timestamp tsStart, java.sql.Timestamp tsEnd) {
	double startMillis = (double) tsStart.getTime() + (double) tsStart.getNanos()/1000000.;
	double endMillis = (double) tsEnd.getTime() + (double) tsEnd.getNanos()/1000000.;
	return Math.abs(endMillis - startMillis)/1000.;
    }
/** Return absolute value of seconds between end and start times */
    public static double elapsedSeconds(Calendar calendarStart, Calendar calendarEnd) {
	return Math.abs((double) calendarEnd.getTime().getTime() - (double) calendarStart.getTime().getTime())/1000.;
    }
/** Return absolute value of seconds between end and start times */
    public static double elapsedSeconds(Date dateStart, Date dateEnd) {
	return Math.abs((double) dateEnd.getTime() - (double) dateStart.getTime())/1000.;
    }

	protected static final int SECS_PER_MIN  = 60;
	protected static final int MINS_PER_HOUR = 60;
	protected static final int HOURS_PER_DAY = 24;
	protected static final int SECS_PER_HOUR = SECS_PER_MIN * MINS_PER_HOUR;
	protected static final int SECS_PER_DAY  = SECS_PER_HOUR * HOURS_PER_DAY;

    /**
     * Return epoch time string with format like 0d 3h 25m 45s. This is used
     * primarily for formatting elapse time or time differences.
     */
    public static String elapsedTimeToText(Number isecs){
	return elapsedTimeToText(isecs.intValue());
    }
    /**
     * Return epoch time string with format like 0d 3h 25m 45s. This is used
     * primarily for formatting elapse time or time differences.
     */
    public static String elapsedTimeToText(double secs){
	return elapsedTimeToText((int) secs);
    }
    /**
     * Return epoch time string with format like 0d 3h 25m 45s. This is used
     * primarily for formatting elapse time or time differences.
     */
    public static String elapsedTimeToText(int interval){
	int days = interval/SECS_PER_DAY;
	interval -= days * SECS_PER_DAY;
	int hours = interval/SECS_PER_HOUR;
	interval -= hours * SECS_PER_HOUR;
	int minutes = interval/SECS_PER_MIN;
	int seconds = interval - minutes * SECS_PER_MIN;
	String retVal = days + "d " + hours + "h " + minutes + "m " + seconds + "s";
	return retVal;
    }
    // /////////////////////////////////////////////////////////////////////

    public static void main(String args []) {
        long millis =  System.currentTimeMillis();
	System.out.println("Current systemTime: " + millis);
        System.out.println("java.sql.Date.toString: \"" + new java.sql.Date(millis).toString() + "\"");
        System.out.println("java.sql.Time.toString: \"" + new java.sql.Time(millis).toString() + "\"");
        System.out.println("java.sql.Timestamp.toString: \"" + new java.sql.Timestamp(millis).toString() + "\"");
        System.out.println("Current time dateToString(date,pattern) \"" + dateToString(new Date(millis), "yyyy-MM-dd") + "\"");

	double secs = ((double) millis)/1000.0;
	String nowString = EpochTime.epochToString(secs);
	System.out.println("Current systemTime: " + nowString + " length: " + nowString.length());
	System.out.println("Date result is    : " + EpochTime.toString(EpochTime.stringToDate(nowString, DEFAULT_FORMAT)));
	nowString = EpochTime.epochToPST(secs);
	System.out.println("Current PST time: " + nowString + " length: " + nowString.length());
	nowString = EpochTime.toNoZoneYYString(secs);
	System.out.println("Current short time: " + nowString + " length: " + nowString.length());


	System.out.println("Elapsed time (12345): " + EpochTime.elapsedTimeToText(new Integer(12345)));
	System.out.println("MaxInteger time     : " + EpochTime.epochToString((double) Float.MAX_VALUE));

        nowString = "1981-01-01 00:00:00.000";
        System.out.println("Test time: " + nowString + " length: " + nowString.length());
	System.out.println("Date result is    : " + EpochTime.toString(EpochTime.stringToDate(nowString, DEFAULT_FORMAT)));


	System.out.println("\n*** Demonstrate ERROR in parsing of ss.ss format ***");

	// test custom formatting and parsing
	String pat = "MMMM dd, yyyy HH:mm:ss.SSS";
	nowString = epochToString(secs, pat);
	System.out.println("Current systemTime : " + nowString + " length: " + nowString.length());
	System.out.println("Date result        : " + EpochTime.stringToDate(nowString, pat).toString());

	double dt = EpochTime.stringToEpoch(nowString, pat);
	System.out.println("Parse test of string: " + nowString);
	System.out.println("Parse result is     : " + EpochTime.epochToString(dt, pat));

	// test custom formatting and parsing
	pat = "MMMM dd, yyyy HH:mm:ss.SS";
	System.out.println("Test Pattern "+pat);
	nowString = epochToString(secs, pat);
	System.out.println("Current systemTime: " + nowString);


        dt = 0.009;
        pat = EpochTime.DEFAULT_FORMAT;
	System.out.println("Test Pattern "+pat);
	System.out.println("Default format : " + EpochTime.epochToString(dt, pat));
	System.out.println("Default format : " + EpochTime.epochToString(dt));
    }
}
