package org.trinet.jasi;

import java.sql.Connection;

/**
 * Define the default data source for all JASI classes.  This data source will
 * be used in calls and in other methods when no data source is given.  */
public abstract class GenericDataSource  {

    public static final int UNKNOWN = 0;
    public static final int DBASE   = 1;
    public static final int FILE    = 2;

    /** Value that tells if we are using a "DATABASE" or a "FILE" */
    protected int source = UNKNOWN;

    /** The GenericDataSource connection */
    protected Connection conn = null;

    // filename - if working with a file rather then a dbase
    String filename = "";

    /** The DbaseConnectionDescription object
     *  @See: DbaseConnectionDescription*/
    protected DbaseConnectionDescription desc = new DbaseConnectionDescription();

    /** Flag that controls ability to write back to the data source. This flag
        just controls if a call to commit() will attempt to write to the data
        source. It can be used to make applications READ-ONLY. By default it is
        'true'. */
    /* Note that this is NOT the same as the Connection.isReadOnly()
     * condition. It also no longer sets org.trinet.jdbc classes to make table
     * row locks (using SelectForUpdate). Such locks caused unexpected and
     * apparently unrelated tables to become locked until commit() or rollback.
     * Now this flag just controls if a call to commit() will attempt to write.*/
    //static boolean writeBackEnabled = false;
    boolean writeBackEnabled = true;

  /* Constructor  -- DOES NOTHING!  You must use GenericDataSource.CreateDataSource() */
    public GenericDataSource ()
		{
    /* do nothing here -- on purpose */
    }


  /* Real Construction Method */
  public static GenericDataSource CreateDataSource(String sClassName)
  {
            GenericDataSource newDataSource = null;

        try {
            newDataSource =  (org.trinet.jasi.GenericDataSource)Class.forName(sClassName).newInstance();
        }
        catch (ClassNotFoundException ex) {
            ex.printStackTrace();
        }
        catch (InstantiationException ex) {
            ex.printStackTrace();
        }
        catch (IllegalAccessException ex) {
            ex.printStackTrace();
        }
        return newDataSource;
  }

    /** Configure a GenericDataSource using an existing jdbc connection to a dbase. Data
     * will be accessed READONLY. That is writeBackEnabled = false */
    public void Configure (Connection conn)
    {
	Configure (conn, false);
    }
    /** Configure a GenericDataSource using a jdbc connection. If 'allowWriteBack' is
        true, caller will be allowed to update, delete, and insert data. The
        default is 'false'. */
    public void Configure (Connection conn, boolean allowWriteBack)
    {
	set(conn);
	setWriteBackEnabled(allowWriteBack);
    }
    /**
     * Configure a GenericDataSource by opening a jdbc connection. Data will be accessed
     * READONLY. That is writeBackEnabled = false.
     *
     * The 'dbaseURL' is a verbose JDBC style URL
     * of the form jdbc:<subprotocol>:<subname>:@<IP-address>:<port>:<dbasename>.
     * Ex: "jdbc:oracle:thin:@makalu.gps.caltech.edu:1521:makaludb"*/
    public void Configure (String dbaseURL,
		       String driver,
		       String username,
		       String passwd)   {

	Configure ( dbaseURL, driver, username, passwd, false);

    }
    /**
     * Configure a GenericDataSource by opening a jdbc connection. Data will be access
     * READONLY if 'allowWriteBack' is false, caller will not be allowd to update,
     * delete, and insert data.  The 'dbaseURL' is a verbose JDBC style URL
     * of the form jdbc:<subprotocol>:<subname>:@<IP-address>:<port>:<dbasename>.
     * Ex: "jdbc:oracle:thin:@makalu.gps.caltech.edu:1521:makaludb" */
    public void Configure (String dbaseURL,
		       String driver,
		       String username,
		       String passwd,
		       boolean allowWriteBack)   {

	set( dbaseURL, driver, username, passwd);
	setWriteBackEnabled(allowWriteBack);
    }

    /** Returns a GenericDataSource instatiated with this DbaseConnectionDescription.
     *  Sets the GenericDataSource to this object so calls to getXXX() return
     *  this object.
     *  @see: DbaseConnectionDescription */
    public void Configure (DbaseConnectionDescription desc) {
	Configure(desc.getURL(), desc.getDriver(),
	     desc.getUsername(), desc.getPassword(), true);
    }

    /** Return et the current connection to the default data source. This is a
     *  singlton. */
     public Connection getConnection () {
	return conn;
    }

    /** Set flag true/false to allow/disallow writing back to the data
        source. In some implementations setting this flag may lock data records
        that are read until GenericDataSource.commit() is called. The actual locking
        behavior is dependent on the specific schema implementation */
    public void setWriteBackEnabled (boolean tf) {

	writeBackEnabled = tf;

	// this makes subsequent uses of ExecuteSQL methods do select-for-update
	// which allows writeback later. Rows are LOCKED until a commit is done.
	// ExecuteSQL.setSelectForUpdate(tf);
    }

    /** Return true if you can write results back to the data source */
    public boolean isWriteBackEnabled () {

	return writeBackEnabled;
    }


    /** Set/reset the GenericDataSource to this DbaseConnectionDescription. Establishes a new
     *  default connection to this source.
     *  @see: DbaseConnectionDescription */
    public boolean set (DbaseConnectionDescription descript) {
      // only act if its a change
      if (!desc.equals(descript)) {
        desc = descript;
        source = DBASE;
	conn = getNewConnect();
        return (getConnection() != null);
      }
      return true;   // no change
    }
    /** Set/reset connection.
     *  Lookup parameter data about a connection. Does not connect, assumes
     *  the connection is valid. */
    public void setConnectionInfo (Connection connection){
       set(connection);
    }

    public DbaseConnectionDescription getDbaseConnectionDescription() {
      return desc;
    }

    /** Return dbase host name. */
    public String getHostName () {
	return desc.getHost();
    }
    /** Return dbase host IP address like we.cannot.edu . */
    public String getIPAddress () {
	return desc.getIPAddress();
    }
    /** Return the connection's username. */
    public String getUsername () {
           return desc.getUsername();
    }

    /** Retrun the port number. */
    public String getPort () {
	return desc.getPort();
    }
    /** Return the database name. */
    public String getDbaseName () {
	return desc.dbasename;
    }

    /** Return a string describing the GenericDataSource */
    public String toString() {

	if (source == DBASE) {

	    return "GenericDataSource is a dbase:"+
		" URL= "+desc.getURL()+
				" domain = "+desc.getDomain()+
				" host = "+desc.getHost()+
				" port = "+desc.getPort()+
		" driver = "+desc.getDriver()+
		" username = "+ desc.getUsername()+
		" writable= "+writeBackEnabled;
	} else if (source == FILE) {
	    return "GenericDataSource is a file:"+
		" filename= "+filename+
		" writable= "+writeBackEnabled;
	}
	return "GenericDataSource is unknown.";
    }

    /** Can't make toString() static, so this.*/
    public String toDumpString() {

	if (source == DBASE) {

	    return "dbase:"+
		" URL= "+desc.getURL()+
		" user= "+ desc.getUsername();
	} else if (source == FILE) {
	    return "file:"+filename;
	}
	return "GenericDataSource is unknown.";
    }
    public String toBlockString () {
   	    return
		" URL = "+ desc.getURL()+ "\n"+
	        " dbase = " + desc.getDbasename() + "\n"+
		" driver = "+ desc.getDriver()+ "\n"+
		" username = "+ desc.getUsername();
    }

    /** Set/reset connection.
     *  Lookup parameter data about a connection. Does not connect, assumes
     *  the connection is valid. */
    public abstract boolean set (Connection connection);


    /** Set/reset the connection.
     *  Sets the GenericDataSource to this object so calls to getXXX() return
     *  this object. */
    public abstract void set(String url,
		       String driver,
		       String username,
		       String passwd);

    /** Set the parameters of the default connection */
/*    public void setDefaultParameters(DbaseConnectionDescription descript)
    {
	source = DBASE;
	desc = new DbaseConnectionDescription(descript);
	dbaseURL = desc.getDbaseURL();
    }
*/
    /** Get a new connection to the default data source. This is usefull when
        you need multiple, simultaneous dbase connections. */
    public abstract Connection getNewConnect ();

    public abstract String getStatus ();

    /** Return true if you cannot write results back to the data source */
    public abstract boolean isReadOnly ();
    /**
     * Actually commit any transactions on this connect to the dbase.
     * If this is not called changes will not take effect.  */
    public abstract void commit (Connection connection);

    /**
     * Actually commit any transactions on the default connect to the dbase.
     * If this is not called changes will not take effect. */
    public void commit () {
        commit(conn);
      /*
	try {
	    conn.commit();
	} catch (SQLException ex)
	    {
		System.err.println(ex);
		ex.printStackTrace();
	    }
          */
    }
    /**
     * Rollback to the dbase. If this is not called changes will not take
     * effect.  */
    public abstract void rollback ();

    /*
    public void reset (String filename)
    {
	source = FILE;
	this.filename = filename;

	// not implemented
    }
    */

    /** Close the connection */
    public abstract void close();




    // TEST
    public void main (String args[])
    {

	String driver = "oracle.jdbc.driver.OracleDriver";
	String url    = "jdbc:oracle:thin:@k2.gps.caltech.edu:1521:k2db";
	String user   = "trinetdb";
	String passwd = "calgs";

        System.out.println ("Making connection... "+url);

	//       JDBConn jc = new JDBConn(url, driver, user, passwd);    // make connection
        GenericDataSource jc = GenericDataSource.CreateDataSource("org.trinet.jasi.JDBCDataSource");
        jc.Configure(url, driver, user, passwd);    // make connection

	System.out.println ("Connection = "+jc.toString());

	System.out.println ("Host  ="+jc.getHostName());
	System.out.println ("Dbase ="+jc.getDbaseName());
	System.out.println ("Port  ="+jc.getPort());

    }

} // GenericDataSource
