package org.trinet.util.velocitymodel;
import java.util.*;
import org.trinet.util.gazetteer.*;

public abstract class UniformFlatLayerVelocityModel implements UniformFlatLayerVelocityModelIF {
    public static final UniformFlatLayerVelocityModel SOCAL_DEFAULT = new HK_SoCalVelocityModel();

    protected Collection layerList;
    protected double psRatio;

    class ConstantVelocityLayer {
         double depth;
         double velocity;
         ConstantVelocityLayer(double depth, double velocity) {
             this.depth = depth;
             this.velocity = velocity;
         }
    }

    protected UniformFlatLayerVelocityModel(Collection layerList) {
        this.layerList = layerList;
    }
    protected UniformFlatLayerVelocityModel(double [] depths, double [] velocities, double psRatio) {
        this(depths.length, depths, velocities, psRatio) ;
    }
    protected UniformFlatLayerVelocityModel(int layerCount, double [] depths, double [] velocities, double psRatio) {
        if (layerCount > depths.length || layerCount > velocities.length)
            throw new IllegalArgumentException("layerCount > length of input velocity/depth array.");
        layerList = new ArrayList(layerCount);
        for (int idx = 0; idx < layerCount; idx++) {
            layerList.add(new ConstantVelocityLayer(depths[idx],velocities[idx]));
        }
        this.psRatio = psRatio;
    }

    public double [] getVelocities() {
        if (layerList == null) return null;
        int size = layerList.size();
        double [] velocities = new double[size];
        if (size == 0) return velocities;
        ArrayList list = (ArrayList) layerList;
        for (int idx = 0; idx < size; idx++) {
            velocities[idx] = ((ConstantVelocityLayer) list.get(idx)).velocity;
        }
        return velocities;
    }

    public double [] getBoundaryDepths() {
        if (layerList == null) return null;
        int size = layerList.size();
        double [] depths = new double[size];
        if (size == 0) return depths;
        ArrayList list = (ArrayList) layerList;
        for (int idx = 0; idx < size; idx++) {
            depths[idx] = ((ConstantVelocityLayer) list.get(idx)).depth;
        }
        return depths;
    }

    public double getVelocityAt(Geoidal geoidal) {
        return getVelocityAt(geoidal.getLat(), geoidal.getLon(), geoidal.getZ()) ;
    }
    public double getVelocityAt(double lat, double lon, double z) {
        return getVelocityAt(z);
    }
    public double getVelocityAt(double z) {
        double velocity = 0.d;
        int nLayers = layerList.size();
        if ( nLayers ==  0) return velocity;
        ArrayList list = (ArrayList) layerList;
        for (int idx = 0; idx < nLayers; idx++) {
            ConstantVelocityLayer cvl = (ConstantVelocityLayer) list.get(idx);
            if (z < cvl.depth) break;
            velocity = cvl.velocity;
        }
        return velocity;
    }

    public double getVelocity(int layerNumber) {
        if (layerNumber < 1 || layerNumber > layerList.size())
            throw new IllegalArgumentException("layerNumber > layers in model");
        ArrayList list = (ArrayList) layerList;
        return ((ConstantVelocityLayer) list.get(layerNumber - 1)).velocity;
    }

    public double getBoundaryDepth(int layerNumber) {
        if (layerNumber < 1 || layerNumber > layerList.size())
            throw new IllegalArgumentException("layerNumber > layers in model");
        ArrayList list = (ArrayList) layerList;
        return ((ConstantVelocityLayer) list.get(layerNumber - 1)).depth;
    }

    public int getLayerCount() {
        return layerList.size();
    }

    public double getPSRatio() {
        return psRatio;
    }

    public void setPSRatio(double psRatio) {
        this.psRatio = psRatio;
    }
}// end of class UniformFlatLayerVelocityModel 
