package org.trinet.jiggle;

import java.util.*;
import org.trinet.jdbc.*;
import org.trinet.jasi.*;
import org.trinet.jasi.TravelTime;
import org.trinet.jasi.coda.*;
import org.trinet.util.*;
import org.trinet.util.gazetteer.LatLonZ;
/**
 * This is a container that holds a Waveform and bounds info for a
 * particular channel/time-window.
 * The viewSpan (time bounds) of the view may be set arbitrarily by the caller
 * and may be independent of the data that may actually be available.
 * Therefore, the viewSpan time window may
 * have no time series, be incomplete, or may clip a Waveform.
 * A WFView may have associated Phases, Amplitudes, etc. even if
 * there is no time series. <p>

 <tt>
    WFView
    viewSpan.start					     viewSpan.end
    |	|dataSpan.start					|dataSpan.end	|
    v   v                                               v               v
    +===WFView==========================================================+
    |   +---Waveform------------------------------------+               |
    |	+-----------+		+-----------------------+		|
    |	| WFSegment | time-tear	|      WFSegment	|		|
    |	+-----------+		+-----------------------+		|
    |   tstart      tend        tstart			tend		|
    |   +-----------------------------------------------+               |
    +===================================================================+
</tt>
*/

public class WFView implements Channelable {
    /** Channel description and info */
    // Note: a Waveform object also has a Channel's but this one is used for
    // sorting, etc.
    public Channel chan;

    /** Time series */
    public Waveform wf = null;

    /** List of original data packet start/stop times if data was
     * in a packetized form when read in. */
    public TimeSpan packetSpans[];

    /** Distance (km) from current active origin */
//    public double dist;

    /** The time window of the virtural "view".
     *  It is independent of the actual data (dataSpan), especially if there is
     *  no waveform in the view.*/
    TimeSpan viewSpan = new TimeSpan();

    /** The span of the actual Waveform data. May contain time-tears. */
    TimeSpan dataSpan = new TimeSpan();

    /** List of Phases that are in this view */
    // This list is dynamically maintained by the update method in ActiveWFPanel.
    PhaseList phaseList = new PhaseList();

    /** List of Amplitudes in this view. */
    AmpList ampList = new AmpList();

    /** List of Amplitudes in this view. */
    CodaList codaList = new CodaList();

    /** One reference to the MasterView shared by all WFViews */
    // this looks suspect to me 1/17/02
    static MasterView mv;

    /** Travel time for P and S identification. */
    TravelTime tt = TravelTime.getInstance();
/**
 * Construct a nul view window.
 */
public WFView ()
{
}

/**
 * Construct an empty view window.
 * ViewSpan and DataSpan will be empty.
 */
public WFView (Channelable ch) {
    chan = Channel.create();
    chan.setChannelObj(ch.getChannelObj());
}
/**
 * Construct view window with defined viewSpan set to these time bounds.
 * DataSpan will be empty.
 */
 public WFView (Channelable ch, double t0, double tn) {
     this(ch);
     viewSpan.set(t0, tn);
 }
/**
 * Construct view window with defined viewSpan time bounds.
 * DataSpan will be empty.
 */
 public WFView (Channelable ch, TimeSpan ts) {
     this(ch, ts.getStart(), ts.getEnd());
 }
/**
 * Construct a view window containing the given WFSegment.
 * Set viewSpan time window to segment data time span.
 */
public WFView (Waveform wf) {
    setWaveform (wf);
}

/**
 * Give ALL WFViews a reference to the MasterView.
 */
    public static void setMasterView(MasterView masterView) {
	mv = masterView;
    }

/**
 * Set the WFSegment vector for this WFView to the one passed as arg.
 * Set viewSpan time window to include data time span.
 */
public void setWaveform(Waveform wf) {

    this.wf = wf;
    chan = Channel.create();
    chan.setChannelObj(wf.getChannelObj());

    dataSpan = new TimeSpan(wf.getTimeSpan());
    viewSpan.include(dataSpan);
    //    dataSpan.include(wf.timeStart.doubleValue(), wf.timeEnd.doubleValue());
    //    viewSpan = new TimeSpan (dataSpan);
}

/**
 * Set the viewSpan to the given Timespan. This is used to align traces by
 * setting all their viewSpans equal.
 */
public void setViewSpan (TimeSpan ts) {
    viewSpan = new TimeSpan(ts);
}

public void setViewSpanStart (double start) {
    viewSpan.setStart(start);
}

public void setViewSpanEnd (double end) {
    viewSpan.setStart(end);
}

/**
 * Make sure the viewSpan includes the given TimeSpan.
 */
public void viewSpanInclude (TimeSpan ts) {
    viewSpan.include(ts);
}
/**
 * Make sure the viewSpan includes the given time.
 */
public void viewSpanInclude (double t0) {
    viewSpan.include(t0);
}

/**
 * Return the Waveform object. Will return 'null' if there is no waveform.
 */
public Waveform getWaveform() {

       return wf;
}

/**
 * Returns 'true' if there is waveform header data in this WFView. The actual time-series
 * may or may not be loaded. Use hasTimeSeries() to check for actual time-series.
 */
    public boolean hasWaveformHeader()  {
	if (getWaveform() == null) return false;
	return true;
    }
/**
 * Returns 'true' if there is actual time-series loaded for this WFView.
 */
    public boolean hasTimeSeries() {
	    if (wf == null) return false;
         return  wf.hasTimeSeries();

    }
/**
 * Attempt to load the time series for his WFView. There may be none to load. Do
 * NOT reload if hasTimeSeries() is true. Returns 'true' if time-series was/is loaded.  */
    public synchronized boolean loadTimeSeries() {

	if (hasTimeSeries()) return true;	// don't reload if we already have it

	return reloadTimeSeries();

    }

/**
 * Attempt to reload the time series for his WFView. There may be none to load.
 Returns 'true' if time-series was reloaded.  */
    public boolean reloadTimeSeries() {
//	if (wf == null) return false;
     if (!hasWaveformHeader()) return false;

	return wf.loadTimeSeries();

    }
/**
 * Attempt to unload the time series for his WFView. There may be none to load.
 Returns 'true' if time-series was unloaded.  */
    public synchronized boolean unloadTimeSeries() {
	if (wf == null) return false;

	return wf.unloadTimeSeries();

    }

/**
 * Returns an array of WFSegments that make up the Waveform in this WFView
 */
/*   public WFSegment [] getSegmentArray() {

       if (wf == null) return null;
       return WFSegment.getArray(wf.getSegmentList());

   }
*/
   // Implement the Channelable interface
    /** Set the Channel */
    public void setChannelObj(Channel chan){
      this.chan = chan;
    }
    /** Return the Channel. */
    public Channel getChannelObj() {
      return chan;
    }

    /** Calculate and set both epicentral (horizontal) distance and true (hypocentral)
     * distance of this channel from the given location in km.
     * Note: this is better than setDistance() or setHorizontalDistance() because
     * it does both and insures they are consistent. If the channel's LatLonZ or
     * the LatLonZ in the argument is null, both distances are set = 9999.9*/
    public double calcDistance (LatLonZ loc) {
        return chan.calcDistance(loc);
    }
    /** Set the distance. */
    public void setDistance(double distance) {
      chan.setDistance(distance);
    }
    /** Set the distance. */
//    public void setDistance(double distance, double horizDistance) {
//      chan.setDistance(distance, horizDistance);
//    }
    /** Set the distance. */
    public void setHorizontalDistance(double hdistance) {
      chan.setHorizontalDistance(hdistance);;
    }
    /** Return the distance. */
    public double getDistance(){
      return chan.getDistance();
    }
    /** Return the distance. */
    public double getHorizontalDistance(){
      return chan.getHorizontalDistance();
    }

     public void setAzimuth(double az) {
        getChannelObj().setAzimuth(az);
     }
     public double getAzimuth () {
        return getChannelObj().getAzimuth();
     }

/**
 * Given a Collection of Channels (i.e. a station list), find the one that
 * matches this WFViews's current Channel. Replace the WFView's Channel object
 * with the "found" Channel which has complete Channel location & response info.
 * This is necessary for distance calcs, sorting, magnitudes, etc.  Return
 * 'true' if match was found, else return 'false'.  */
public boolean copyChannel(Collection chanList)
    {
	if (chanList == null) return false;

	Channel ch[] = new Channel[chanList.size()];
	chanList.toArray(ch);

	return copyChannel(ch);

    }
/**
 * Given an array of Channels (i.e. a station list), find the one that matches
 * this WFViews's current Channel. Replace the WFView's Channel object with the
 * "found" Channel which has complete Channel location & response info.  This is
 * necessary for distance calcs, sorting, magnitudes, etc.  Return 'true' if
 * match was found, else return 'false'.  */
public boolean copyChannel(Channel[] ch)
    {

	// VH_/EH_ kludge
	String origChan = getChannelObj().getChannel();

	Channel chan = getChannelObj();

// No efficient way to do this unless we put Channel list in hash table
	for (int i = 0; i<ch.length; i++)
	{
	    if (chan.equalsIgnoreCase(ch[i])) {

		setChannelObj(ch[i]);		// substitute

		// rest of the VH_/EH_ kludge
		//		System.out.println ("1) "+chan.toString());
		chan.setChannel(origChan);
		//		System.out.println ("2) "+chan.toString());

		return true;
	    }
	}
	return false;
    }
    /** Update the local phaseList by scanning the one in the MasterView. This
	* local list is used to more efficiently plot the phases. Returns the
	* number of phases in the phaselist for this WFView. */
    public int updatePhaseList() {
       return updatePhaseList(mv.getAllPhases());
    }
    /** Update the local phaseList by scanning the one passed. This
	* local list is used to more efficiently plot the phases. Returns the
	* number of phases in the phaselist for this WFView. */
    public int updatePhaseList(PhaseList list) {
     phaseList.clear();
     phaseList.addAll(list.getAssociated(getChannelObj()));
	return phaseList.size();
    }
    /** Update the local ampList by scanning the one in the MasterView. This
	* local list is used to more efficiently plotting. Returns the
	* number of amps in the list for this WFView. */
    public int updateAmpList() {
       return updateAmpList(mv.getAllAmps());
    }
    /** Update the local ampList by scanning the one in the MasterView. This
	* local list is used to more efficiently plotting. Returns the
	* number of amps in the list for this WFView. */
    public int updateAmpList(AmpList list) {

     ampList.clear();
	ampList.addAll(list.getAssociated(getChannelObj()));

	return ampList.size();
    }
    /** Update the local codaList by scanning the one in the MasterView. This
	* local list is used to more efficiently plotting. Returns the
	* number of codas in the list for this WFView. */
    public int updateCodaList() {
      return updateCodaList(mv.getAllCodas());
    }
    /** Update the local codaList by scanning the one in the MasterView. This
	* local list is used to more efficiently plotting. Returns the
	* number of codas in the list for this WFView. */
    public int updateCodaList(CodaList list) {

     codaList.clear();
	codaList.addAll(list.getAssociated(getChannelObj()));

	return codaList.size();
    }

    /** Update the local codaList by scanning the one in the MasterView. This
	* local list is used to more efficiently plotting. Returns the
	* number of codas in the list for this WFView. */
/*    public int updateJasiReadingList(JasiReadingList list) {

	list = (JasiReadingList) list.getAssociatedByChannel(chan);

	return list.size();
    }
*/
/** Return true if there are phases */
    public boolean hasPhases() {
	return !phaseList.isEmpty();
    }
/** Return true if there are amps */
    public boolean hasAmps() {
	return !ampList.isEmpty();
    }
/** Return true if there are codas */
    public boolean hasCodas() {
	return !codaList.isEmpty();
    }

/**
 * Returns a partial Phase description for this WFView.
 * Only populates the Channel part of the Phase.
 * Leaves specific variables null, e.g. datetime, iphase, etc.
 */
    public Phase getEmptyPhase() {
	Phase ph = Phase.create();
	ph.setChannelObj((Channel) chan.clone());
	return ph;
    }

/**
 * Takes its best shot at describing the phase pick at the given time.
 * This first hack is very crude.
 */
//TODO: add estimate of quality, and phase (based on ttime to selected origin)
    public Phase describePickAt (double dtime, Solution selSol)  {

	Phase ph = getEmptyPhase();
	ph.datetime.setValue(dtime);
//	ph.description.fm = "..";
	double diff = 0.0;

	// figure first motion it there's a waveform
	if (wf != null) {
	    Sample samp1 = wf.closestSample(dtime);
	    //	Sample samp2 = wf.closestSample(dtime + getSampleInterval() );
	    Sample samp2 =
		wf.closestSample(dtime + wf.getSampleInterval());

	    diff = samp1.value - samp2.value;

	    // <> First motion
	    if	  (diff < 0.0) {
		ph.description.fm = "c.";
	    } else if (diff > 0.0) {
		ph.description.fm = "d.";
	    } else  {
		ph.description.fm = "..";
         }
	}

// <> Phase type, could be clever and check traveltimes
	double ttP = tt.getTTp(getHorizontalDistance(), mv.getSelectedSolution().getLatLonZ().getZ());
	double ttS = ttP*tt.getPSRatio();
	double sMinusP = ttS - ttP;

	if (selSol == null) {
	    ph.description.iphase = "P";	// guess P if no solution
	} else {
	    double ot = selSol.datetime.doubleValue();

	    double ttObs = dtime - ot;

	// Guess "S" if time is closer to expected S than expected P
	// i.e. if time is greater then P-time + 1/2 S-P
	    if (ttObs >= (ttP+ (0.5 *sMinusP) ) ) {
            ph.description.iphase = "S";
         } else {
	       ph.description.iphase = "P";
         }
	}

// <> "qual" (aka onset) i, e, w
// "i" if first diff is 10% of the full range, "e" if 5% (this is a kludge)
// but looks reasonable upon inspection.
//TODO: refine this
	double firstDiffRatio =
	    (double) Math.abs(diff) /(double) wf.rangeAmp();

	if        (firstDiffRatio >= 0.10) { ph.description.ei = "i";
	} else if (firstDiffRatio >= 0.05) { ph.description.ei = "e";
	} else {			     ph.description.ei = "w"; }
	/*
	System.out.println ("firstDiffRatio = "+ firstDiffRatio +
			    " Math.abs(diff)= "+ Math.abs(diff) +
			    " wf.rangeAmp()= "+wf.rangeAmp());
	*/
// <> "quality" (aka weight)
	ph.description.setQuality(1.0);

// <> association, could be clever and check traveltimes
//	if (selSol != null) ph.associate(selSol);

	return ph;

    }

/**
 * Return a WFSelectionBox the is based on the extremes of the data, including
 * waveforms and phases. If there is no waveform max/min amps cannot be set
 * based on the data so arbitrary values of +- 100 will be usex1d.  */
public WFSelectionBox getViewBox () {

      if (wf != null && wf.hasTimeSeries())  {
         return new WFSelectionBox(viewSpan, wf.getMaxAmp(), wf.getMinAmp());
      } else {
      // box must have non-zero dimension else /0 error happen elsewhere
         return new WFSelectionBox(viewSpan, 1, 0);
      }
}

/** Return the TimeSpan of the view. */
public TimeSpan getViewSpan () {
       return viewSpan;
}

/**
 * Return a WFSelectionBox the is based on the extremes of the data, including
 * waveforms and phases. If there is no waveform max/min amps cannot be set
 * based on the data so arbitrary values of +- 100 will be used.  */
public WFSelectionBox getDataBox () {

      if (wf != null && wf.hasTimeSeries())  {
         return new WFSelectionBox(dataSpan, wf.getMaxAmp(), wf.getMinAmp());
      } else {
      // box must have non-zero dimension else /0 error happens elsewhere
      // can't use dataSpan because it depends on wf bounds
         return new WFSelectionBox(viewSpan, 1, 0);
      }
}
/**
 * Set the WFView's 'dataSpan' value to match the actual data in the Waveform.
 */

public void autoSetDataSpan() {
    dataSpan = wf.getTimeSpan();
}

/**
 * Dump some info about the view for debugging
 */
public void dump ()
{
    System.out.println (dumpToString());
}

/**
 * Dump some info about the view for debugging
 */
public String dumpToString ()
{
    String str = "WFView: "+ getChannelObj().toString() +
	" dist = "+ getDistance();

    if (wf != null) {
	str +=
	    " nsegs = " + wf.getSegmentList().size() +"\n"+
	    " nsamples = "+ wf.samplesInMemory() +
	    " maxamp = " + wf.getMaxAmp() +
	    " minamp = " +  wf.getMinAmp() +
	    " nphases= "+ phaseList.size()+"\n";
    } else {
	str += " -- no waveform available --";
    }

    str += "\n   viewSpan = " + viewSpan.toString() + "\n";
    str += "   dataSpan = " + dataSpan.toString() + "\n";

    //    str += wf.toString();

// dump info about all the segments
    /*    WFSegment seg[] = WFSegment.getArray(segVector);

    for (int i = 0; i< segmentCount(); i ++)
    {
	str += "....\n";
	str += seg[i].dumpToString();
    }
    */

    return str;
}

} // end of class
