package org.trinet.jasi;

//import java.lang.*;
//import java.io.File;
import java.util.*;

import org.trinet.jdbc.datatypes.*;

import org.trinet.util.*;
import org.trinet.util.gazetteer.*;

/**
   A JasiTimeSeries is a collection of Segments. The type of segment will depend
   on the type of time series.
*/
public abstract class JasiTimeSeries extends JasiTimeBoundedData
           implements Channelable {

// statics
   /** Pseudo null value for primatives. */
    private static final int NULL = (int) Math.pow(2.0, 31.0);

    // Enumeration of data types. These are from the NCDC Schema v1.5 doc
    public static final int UNKNOWN          = 0;
    public static final int RAW_FORMAT		= 1;
    public static final int SEED_FORMAT		= 2;
    public static final int PASSCAL_SEGY_FORMAT = 3;
    public static final int UCSB_SEGY_FORMAT	= 4;

    /** Data format as defined in NCDN schema */
    //public DataLong    format		= new DataLong();
    public DataLong    format		= new DataLong(SEED_FORMAT);

    /** The sample rate in units of samples/second. */
    protected  DataDouble samplesPerSecond = new DataDouble();

    double bytesPerSample;

    /** Channel information. */
    protected Channel chan  = Channel.create();

    /** Collection of Time-series Segments containing data for this channel/window */
    protected ArrayList segList = new ArrayList();  // must instantiate so we can use size()

    /** Array of TimeSpans containing the original data packet boundaries */
    TimeSpan[] packetSpans;

    /** Rectified background noise level with bias removed. It may be derived
    * from some outside source (say a longterm average from the external datasource.)
    * If calculated by this class it is
    * the median of the peaks between zero-crossings for a rectified time-series
    * with the bias removed. The time window for which it is calculated is set by
    * a call to scanForNoiseLevel(start, end). */
    float noiseLevel = Float.NaN;

    /** True if timeseries has been filtered. */
    boolean isFiltered = false;

    // Channelable interface
    /**
     * Return the channel object associated with this reading.
     */
    public Channel getChannelObj () {
  return chan;
    }
    public void setChannelObj(Channel chan) {
     this.chan = chan;
    }

    public void setDistance(double distance) {
      getChannelObj().dist.setValue(distance);
    }
    public void setDistance(double distance, double horizontalDistance) {
      setDistance(distance);
      setHorizontalDistance(distance);
    }
    public void setHorizontalDistance(double distance) {
      getChannelObj().hdist.setValue(distance);
    }

    /** Calculate and set both epicentral (horizontal) distance and true (hypocentral)
     * distance of this channel from the given location in km.
     * Note: this is better than setDistance() or setHorizontalDistance() because
     * it does both and insures they are consistent. */
    public double calcDistance (LatLonZ loc) {
      return chan.calcDistance(loc);
    }

    public double getDistance() {
      return getChannelObj().dist.doubleValue();
    }
    public double getHorizontalDistance() {
      return getChannelObj().hdist.doubleValue();
    }
    public void setAzimuth(double az) {
        getChannelObj().setAzimuth(az);
    }
    public double getAzimuth () {
        return getChannelObj().getAzimuth();
    }
    /** Set the timeseries sample rate in samples/second. */
    public void setSampleRate (double samplesPerSecond) {
       this.samplesPerSecond.setValue(samplesPerSecond);

    }
    /** Return the timeseries sample rate in samples/second. */
    public double getSampleRate () {
       return samplesPerSecond.doubleValue();
    }

    /** Save the start & stop times of the segment boundaries.
     *  These are sometimes useful in understanding the original data. */
    void savePacketSpans() {
        packetSpans = new TimeSpan[segList.size()];
        for (int i = 0; i < packetSpans.length ; i++) {
            packetSpans[i] = ((TimeSeriesSegment)segList.get(i)).getTimeSpan();
        }
    }
    /** Return the start & stop times of the segment boundaries.
     *  These are sometimes useful in understanding the original data. */
    public TimeSpan[] getPacketSpans() {
           return packetSpans;
    }
    public void setPacketSpans ( TimeSpan[] spanArray ) {
           packetSpans = spanArray;
    }
  /** Return the number of segments in the timeseries.*/
    public int getSegmentCount() {
           return segList.size();
    }
    /** Return the collection of WFSegments. */
    public Collection getSegmentList() {
           return segList;
    }
    /** Sets the list of WFSegments.
     *  Resets the PacketSpans to match the new list.*/
    public void setSegmentList(Collection list) {
           segList = (ArrayList) list;
     savePacketSpans();
     noiseLevel = Float.NaN;
     scanSegListForTimeQuality();
    }

  /** Scan the timeQuality of each segment and set the whole waveform's
   *  timeQuality equal to the WORST (least value) in the list.  */
  public void scanSegListForTimeQuality() {
        if (segList.size() < 1) return;
        DataDouble worst = ((TimeSeriesSegment)segList.get(0)).getTimeQuality();
  DataDouble next;
        for (int i = 1; i < segList.size() ; i++) {
            next = ((TimeSeriesSegment)segList.get(i)).getTimeQuality();
      if (worst.compareTo(next) == 1) {
        worst = next;
      }
        }
  setTimeQuality(worst);
  }
  /** Set value of the noise level. */
  public void setNoiseLevel (float value) {
    noiseLevel = value;
  }
  /** Return value of the noise level. Returns Float.NaN if it is unknown. */
  public float getNoiseLevel () {
    if (noiseLevel == Float.NaN) scanForNoiseLevel();
    return noiseLevel;
  }

  // ABSTRACT METHODS
  /** Scan time series for the noise level. Different time series may calculate
   *  this in different ways. */
  public abstract float scanForNoiseLevel();

  public void setIsFiltered (boolean tf) {
    isFiltered = tf;
  }

  public boolean isFiltered () {
    return isFiltered;
  }
}
