/*
 *   THIS FILE IS UNDER RCS - DO NOT MODIFY UNLESS YOU HAVE
 *   CHECKED IT OUT USING THE COMMAND CHECKOUT.
 *
 *    $Id: rw_mag.c,v 1.13 2004/04/12 22:11:47 dietz Exp $
 *
 *    Revision history:
 *     $Log: rw_mag.c,v $
 *     Revision 1.13  2004/04/12 22:11:47  dietz
 *     included stdlib.h
 *
 *     Revision 1.12  2002/09/10 17:20:59  dhanych
 *     Stable scaffold
 *
 *     Revision 1.11  2002/06/28 21:03:57  lucky
 *     Fixed reading of mag phases -- it would not properly read float
 *     values for amplitude and period
 *
 *     Revision 1.10  2001/08/07 16:54:17  lucky
 *     Pre v6.0 checkin
 *
 *     Revision 1.9  2001/06/07 20:51:49  lucky
 *     Modified wr_mag so that it checks for NULL strings. Before, if qid was null,
 *     wr_mag would produce an invalid TYPE_MAGNITUDE message which could not
 *     be parsed by rd_mag. This is not good, given that the qid is not known
 *     in the review system.
 *
 *     Revision 1.8  2001/06/07 19:48:28  lucky
 *     rd_chan_mag_lucky renamed to rd_chan_mag. The old rd_chan_mag is still
 *     available as rd_chan_mag_Pete.
 *
 *     Revision 1.7  2001/05/31 21:09:02  lucky
 *     Addedd rd_chan_mag_lucky; I don't think that rd_chan_mag in its original form
 *     works. There are numerical comparisons of pointers, and other visible problems.
 *
 *     Revision 1.6  2001/05/26 21:04:59  lombard
 *     Changed ML_INFO struct to MAG_CHAN_INFO struct to make it more
 *     generic; now this struct should work with most mag types.
 *     Fixed a number of obvious bugs and cleaned up the code.
 *     But the parsers still have not been tested and this may not work.
 *
 *     Revision 1.5  2001/05/02 20:58:45  alex
 *     *** empty log message ***
 *
 *     Revision 1.4  2001/05/01 22:38:48  davidk
 *     moved the MagNames table out of the header file and into rw_mag.c
 *
 *     Revision 1.3  2001/05/01 20:26:02  davidk
 *     Modified several new functions added by Alex, so that they compile and do not impede the
 *     release.  The new functions have been added, but have not properly been tested and debugged,
 *     at all.
 *
 *     Revision 1.2  2001/04/29 00:15:29  alex
 *     Alex: changes for mag type int and string, and added alternate parsing routines.
 *
 *     Revision 1.1  2001/03/31 00:44:58  lombard
 *     Initial revision
 *
 *
 *
 */

/* rw_mag.c
 *
 * Contains functions in that convert from a
 * TYPE_MAGNITUDE message to a structure and visa versa.
 *
 * written by Pete Lombard   February, 2001
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <earthworm.h>
#include <rw_mag.h>

/* Internal Function Prototypes */
static int strappend( char *s1, int s1max, char *s2 );
static int tokenlength( char *begtok, char c );

/********************************************************************
 * rd_mag()                                                         *
 * Reads an ascii TYPE_MAGNITUDE message and fills in a MAG_INFO    *
 * structure which must be allocated by the caller                  *
 * If, in addition, the caller allocates space for auxiliary        *
 * structures at pMag->pMagAux (number of bytes in pMag->size_aux), *
 * then rd_mag will call the appropriate function to decode the     *
 * remainder of the message into appropriate structures.            *
 * Currently supported magnitude types are: Ml                      *
 * Returns 0 on success                                             *
 *        -1 on parsing errors                                      *
 *        -2 on insufficient space in pMagAux                       *
 ********************************************************************/
int rd_mag( char *msg, int msgLen, MAG_INFO *pMag)
{
  size_t size_aux;
  char *pMagAux;

  size_aux = pMag->size_aux;
  pMagAux = (char *)pMag->pMagAux;
  memset(pMag, 0, sizeof(MAG_INFO));

  switch ( sscanf( msg
                 , "%s MAG %d %s %lf %s %d %d %lf %lf %lf %d %s %d %d"
                 ,  pMag->qid
                 , &pMag->imagtype
                 ,  pMag->szmagtype
                 , &pMag->mag
                 ,  pMag->algorithm
                 , &pMag->nstations
                 , &pMag->nchannels
                 , &pMag->error
                 , &pMag->quality
                 , &pMag->mindist
                 , &pMag->azimuth
                 ,  pMag->qauthor
                 , &pMag->origin_version
                 , &pMag->qdds_version
         )       )
  {
    case 12:  /* old-style, no origin version, qdds version numbers */
         pMag->origin_version = 0;
         pMag->qdds_version = 0;
         break;
    case 13:  /* new-style, with origin version, no qdds version */
         pMag->qdds_version = 0;
         break;
    case 14:  /* new-style, with both number */
         break;
    default:
         return( -1 );
  }
  
  if (pMagAux && size_aux > 0)
  {
    pMag->size_aux = size_aux;
    pMag->pMagAux = pMagAux;
    
    /* Read the rest of a Mag message */
    if (pMag->pMagAux != NULL)
	{
         return( rd_chan_mag(msg, msgLen, (MAG_CHAN_INFO *)pMag->pMagAux,
                          pMag->size_aux) );
	}
  }
  
  return( 0 );
}

/*****************************************************************
 * rd_chan_mag()                                                 *
 * Reads an ascii TYPE_MAGNITUDE Ml message and fills in         *
 * MAG_CHAN_INFO structures.                                     *
 * This function can be called by rd_mag if sufficient space is  *
 * allocated by its caller. Or it may be called after rd_mag     *
 * when the caller has allocated space for `nchannels' of        *
 * MAG_CHAN_INFO structures.                                     *
 * Returns 0 on success                                          *
 *        -1 on parsing errors                                   *
 *        -2 on insufficient space in pMagAux                    *
 *****************************************************************/
int rd_chan_mag ( char *msg, int msglen, MAG_CHAN_INFO *pMci, int size_ml)
{
  char     *line;
  char     *begtok;
  char     *tmpMsg;     /* local copy of msg */
  int       len;       /* length (bytes) of the next token */
  int       nchan;     /* number of channels read from msg so far */
  int       nML;       /* number of MAG_CHAN_INFO structures we can use */
  int       rc;
  int		index;
  
  memset(pMci, 0, size_ml);
  nML = size_ml / sizeof(MAG_CHAN_INFO);

  tmpMsg = (char *) malloc (msglen);
  strcpy (tmpMsg, msg);

  nchan = 0;

  /* Skip the first line */
  len = tokenlength (tmpMsg, '\n'); 

  index = len + 1;

  /* Read next line from the message into working buffer */
  while(index < (msglen-1))
  {
	/* Process next line */
	line = tmpMsg + index;
    len = tokenlength(line, '\n' ); 
	line[len] = '\0';
    index = index + len + 1;

    begtok = line;
    while( begtok[0] == ' ' && begtok[0] != '\0') begtok++;    /* go to 1st non-blank char */
    if( begtok[0] == '\0' ) return( -1 );
    len = tokenlength( begtok, '.' );
    if( len >= TRACE_STA_LEN ) return( -1 );
    strncpy( pMci[nchan].sta, begtok, len );
	pMci[nchan].sta[len] = '\0';
    begtok += len + 1;

    len = tokenlength( begtok, '.' );
    if( len >= TRACE_CHAN_LEN ) return( -1 );
    strncpy( pMci[nchan].comp, begtok, len );
	pMci[nchan].comp[len] = '\0';
    begtok += len + 1;

    len = tokenlength( begtok, ' ' );
    if( len >= TRACE_NET_LEN ) return( -1 );
    strncpy( pMci[nchan].net, begtok, len );
	pMci[nchan].net[len] = '\0';
    begtok += len + 1;

    while( begtok[0] == ' ' && begtok[0] != '\0' ) begtok++;    /* go to next non-blank char */
    if ( (rc = sscanf(begtok, "%lf %lf %lf %E %lf %f %E %lf %f",
                      &(pMci[nchan].mag), &(pMci[nchan].dist), &(pMci[nchan].corr),
                      &(pMci[nchan].Amp1), &(pMci[nchan].Time1), &(pMci[nchan].Period1),
                      &(pMci[nchan].Amp2), &(pMci[nchan].Time2), 
                      &(pMci[nchan].Period2))) != 9)
    {
      if (rc == 6)
      {
        pMci[nchan].Amp2 = pMci[nchan].Time2 = pMci[nchan].Period2 = -1.0;
        continue;
      }
      return( -1 );
    }

    nchan = nchan + 1;

  }

  return 0;
}



/*******************************************************************
 * wr_mag()                                                        *
 * Reads a MAG_INFO structure and writes an ascii TYPE_MAGNITUDE   *
 * message (null terminated)                                       *
 * Returns 0 on success, -1 on failure (potential buffer overflow) *
 *******************************************************************/
int wr_mag( MAG_INFO *pMag, char *buf, int buflen )
{
  char     tmp[256]; /* working buffer */
  char     qid[256]; 
  char     szmagtype[256]; 
  char     algorithm[256]; 
  char     qauthor[256]; 

  memset( buf, 0, (size_t)buflen );    /* zero output buffer */

  /* 
   * We should check for null strings, if for nothing else than to 
   * avoid mis-formatted message trickling down the system. LV 6/2001
   */
	if (strlen (pMag->qid) == 0)
		strcpy (qid, "UNKNOWN");
	else
		strcpy (qid, pMag->qid);
 
	if (strlen (pMag->szmagtype) == 0)
		strcpy (szmagtype, "UNKNOWN");
	else
		strcpy (szmagtype, pMag->szmagtype);
 
	if (strlen (pMag->algorithm) == 0)
		strcpy (algorithm, "UNKNOWN");
	else
		strcpy (algorithm, pMag->algorithm);
 
	if (strlen (pMag->qauthor) == 0)
		strcpy (qauthor, "UNKNOWN");
	else
		strcpy (qauthor, pMag->qauthor);
 
  sprintf(tmp, "%s MAG %d %s %0.2f %s %d %d %0.2f %0.2f %0.2f %d %s %d %d\n",
          qid, pMag->imagtype, szmagtype, pMag->mag, algorithm, 
          pMag->nstations, pMag->nchannels, pMag->error, pMag->quality,
          pMag->mindist, pMag->azimuth, qauthor, pMag->origin_version
          , pMag->qdds_version
          );
   
  if ( strappend( buf, buflen, tmp ) ) return( -1 );
   
  if (pMag->pMagAux != NULL)
  {
      return( wr_chan_mag( (MAG_CHAN_INFO *)pMag->pMagAux, pMag->nchannels, 
                           buf, buflen) );
  }
   
  return( 0 );
}

/*******************************************************************
 * wr_chan_mag()                                                   *
 * Reads MAG_CHAN_INFO structures and appends to an ascii          *
 * TYPE_MAGNITUDE message (null terminated)                        *
 * Normally this would be called by wr_mag().                      *
 * Returns 0 on success, -1 on failure (potential buffer overflow) *
 *******************************************************************/
int wr_chan_mag( MAG_CHAN_INFO *pMci, int nchannels, char *buf, int buflen )
{
  int i;
  char line[256];
  
  for (i = 0; i < nchannels; i++)
  {
    sprintf(line, 
            "%s.%s.%s %.2f %.2f %.2f %9.2E %9.2f %9.2f %9.2E %9.2f %9.2f\n",
            /*s  c  n mag  dist corr Amp1 Time1 Period1 Amp2 Time2 Period2 */
            pMci[i].sta, pMci[i].comp, pMci[i].net, 
			pMci[i].mag, pMci[i].dist, pMci[i].corr, 
            pMci[i].Amp1,
            ( (pMci[i].Time1 < 0.0) ? -1.0 : pMci[i].Time1),
            ( (pMci[i].Period1 < 0.0) ? -1.0 : pMci[i].Period1),
            pMci[i].Amp2,
            ( (pMci[i].Time2 < 0.0) ? -1.0 : pMci[i].Time2),
            ( (pMci[i].Period2 < 0.0) ? -1.0 : pMci[i].Period2));
    
      if ( strappend(buf, buflen, line))
        return( -1 );
  }
  return( 0 );
}

 
/********************************************************************
 * strappend() append second null-terminated character string to    *
 * the first as long as there's enough room in the target buffer    * 
 * for both strings an the null-byte                                *
 ********************************************************************/
static int strappend( char *s1, int s1max, char *s2 )
{
  if( (int)strlen(s1)+(int)strlen(s2)+1 > s1max ) return( -1 );
  strcat( s1, s2 );
  return( 0 );
}

/********************************************************************
 * tokenlength() given a null-terminated character string and a     *
 * character that delimits the end of a token, tokenlength returns  * 
 * the length (in bytes) of the next token. If the character wasn't * 
 * found, tokenlength returns the length of the string.             *
 ********************************************************************/
static int tokenlength( char *begtok, char c )
{
  char    *endtok;   /* points to the end of this token */

  endtok = strchr( begtok, c );
  if( endtok == NULL ) return( (int)strlen(begtok) );
  return( (int)(endtok-begtok) );
}
