package org.trinet.util.magnitudeengines;
import java.util.*;
import org.trinet.jasi.*;
import org.trinet.jasi.coda.*;
import org.trinet.pcs.*;
import org.trinet.util.*;
import org.trinet.util.velocitymodel.*;
import org.trinet.util.gazetteer.*;

//  Subclass implementations create the codaList and magnitude members of org.trinet.jasi.Solution
public abstract class CodaMagnitudeGenerator
    implements CodaMagnitudeGeneratorIF, CommitableIF, ChannelFilterIF, Runnable {
    //protected org.trinet.util.Concat fmt;

    protected static final String P_TYPE = "P";
    protected static final String S_TYPE = "S";
    protected static final int    DEFAULT_MIN_CODA_VALUES                  = 3;
    protected static final double DEFAULT_CODA_START_TIME_OFFSET           = 1.;
    protected static final double DEFAULT_PRE_P_TIME_OFFSET                = 2.;
    protected static final long   DEFAULT_WAVEFORM_LOAD_WAIT_MILLIS        = 2000l;
    protected static final int    DEFAULT_MAX_LOADING_WAIT_ITERATIONS      = 15;
    protected static final int    DEFAULT_WAVEFORM_CACHE_SIZE              = 100;

    protected int minCodaCountForSummaryMag           = DEFAULT_MIN_CODA_VALUES;
    protected double codaStartOffsetSecs              = DEFAULT_CODA_START_TIME_OFFSET;
    protected double codaPrePTimeBiasOffsetSecs       = DEFAULT_PRE_P_TIME_OFFSET;
    protected long maxSleepTimeMillis                 = DEFAULT_WAVEFORM_LOAD_WAIT_MILLIS;
    protected int  maxSleepLoadingIterations          = DEFAULT_MAX_LOADING_WAIT_ITERATIONS;
    protected int  maxWaveformCacheSize               = DEFAULT_WAVEFORM_CACHE_SIZE;
    //protected int  solWaveformCacheSize = DEFAULT_WAVEFORM_CACHE_SIZE;

    protected CodaSolutionProcessingMode DEFAULT_SOLUTION_PROCESSING_MODE = CodaSolutionProcessingMode.CALC_CODA_AND_SUMMARY_MAG;
    protected CodaSolutionProcessingMode processingMode = DEFAULT_SOLUTION_PROCESSING_MODE;

    protected Phase defaultPhase = Phase.create();
    protected CodaCalibration defaultCalibration = new McaCalibrationTN();
    //    = CodaCalibration.create("org.trinet.util.magnitudeengines.McaCalibrationTN");  // necessary evil for now
    protected CodaGeneratorParms codaGeneratorParms;
    protected PrimarySecondaryWaveTravelTimeGeneratorIF ttGenerator;
    protected CodaGeneratorIF codaGenerator;

    protected ChannelList channelList;
    protected CodaCalibrationListIF calibrList;
    protected String channelCacheFileName;

    protected int calcCount;  // calculated coda counter for debug use
    protected static boolean debug                = false;
    protected boolean logCodaCalc                 = false;
    protected boolean logMagResiduals             = false;
    protected boolean dbEnabled                   = true;
    protected boolean waveformCacheLoadingEnabled = true;
    protected boolean verbose                     = false;
    protected boolean autoCommit                  = false;
    protected boolean magAssociationDisabled      = false;

// Instance objects utilized by methods for caching the input/output data of single solution
    protected Thread loadingThread;
    protected Solution currentSol;
    protected long currentSolId = -1l;
    protected String resultsMessage;

    protected org.trinet.util.gazetteer.LatLonZ  solutionLatLonZ;  // coupled to setSolution, Solution does not implement Geoidal.
    protected List   solCodaWaveformDataList;
    protected int    solWaveformLoadedIndex    = -1;

    protected int    solWaveformToProcessIndex = -1;
    protected int    eligibleCount ;
    protected String progressMessageChannelName;


    protected String   sourceDevice;
    protected String   defaultSourceDevice;
    protected Magnitude summaryMag;
    protected MagDataStatistics summaryMagStats;

    public class MagDataStatistics {
        public int    count;
        public double value;
        public double mean;
        public double median;
        public double weightedMedian;
        public double stdDev;

        public void init() {
            count           = 0;
            value           = 0.;
            mean            = 0.;
            median          = 0.;
            weightedMedian  = 0.;
            stdDev          = 0.;
        }

        // Need a more robust statistical algorithm to allow for different channel data counts
        // i.e. small data sets
        public double calcValue(double [] magValues, double [] weights, int calibratedCount) {

            count           = calibratedCount;
            mean            = Stats.mean(magValues, count);
            median          = Stats.median(magValues, count);
            weightedMedian  = WeightedMedian.calcMedian(count, magValues, weights);
            stdDev          = Stats.standardDeviation(magValues, count);

            value = median;
            return value;
        }

        public double getError() {
            return stdDev;
        }

        public String toString() {
            StringBuffer sb = new StringBuffer(132);
            //if (fmt == null) fmt = new org.trinet.util.Concat();
            sb.append(" weightedMedian:");
            Concatenate.format(sb, weightedMedian, 3, 2);
            sb.append(" median:");
            Concatenate.format(sb, median, 3, 2);
            sb.append(" mean:");
            Concatenate.format(sb,mean, 3, 2);
            sb.append(" stdDev:");
            Concatenate.format(sb, stdDev, 3, 2);
            sb.append(" count:");
            Concatenate.format(sb, count, 3);
            return sb.toString();
        }
    }

    protected class CodaWaveformData {
        Waveform wave;
        double pTime;
        double sCodaStartTime;
        boolean selfLoadedWaveTimeSeries;
        boolean rejected;

        protected CodaWaveformData(Waveform wave, double pTime, double sCodaStartTime) {
            this.wave = wave;
            this.pTime = pTime;
            this.sCodaStartTime = sCodaStartTime;
        }

        public String toString() {
            StringBuffer sb = new StringBuffer(512);
            sb.append(wave.toString());
            sb.append(" pTime: ").append(pTime);
            sb.append(" sCodaStartTime:").append(sCodaStartTime);
            sb.append(" selfLoad: ").append(selfLoadedWaveTimeSeries);
            sb.append(" reject: ").append(rejected);
            return sb.toString();
        }
    }

    private static void sortWaveformsByDistance(List list) {
        Collections.sort(list, new Comparator () {
             public final boolean equals(Object obj) {
                return (obj != null && obj.getClass() == this.getClass());
             }
             public final int compare(Object obj1, Object obj2) {
                double delta = ((CodaWaveformData) obj2).wave.getDistance() -
                               ((CodaWaveformData) obj1).wave.getDistance();
                if (delta == 0.) return 0;
                return (delta < 0.) ?  1 : -1;
             }
        });
    }

    protected class CalibratedCoda {
        Coda coda;
        CodaCalibrParms calibr;

        protected CalibratedCoda(Coda coda, CodaCalibrParms calibr) {
            this.coda = coda;
            this.calibr = calibr;
        }

    }

    public CodaMagnitudeGenerator() {}

    public final boolean hasMagAssociationDisabled() {
        return magAssociationDisabled;
    }

    public final void setMagAssociationDisabled(boolean value) {
        magAssociationDisabled = value;
    }

    public final void setWaveformCacheLoading(boolean value) {
        waveformCacheLoadingEnabled = value;
    }

    public final boolean isWaveformCacheLoadingEnabled() {
        return waveformCacheLoadingEnabled;
    }

    public final void setVerbose(boolean value) {
        verbose = value;
        //if (fmt == null) fmt = new org.trinet.util.Concat();
    }

    protected static void setDebug(boolean value) {
        debug = value;
    }

    public final void setDbEnabled(boolean value) {
        dbEnabled = value;
    }

    public final boolean getDbEnabled() {
        return dbEnabled;
    }

    public void setSolution(Solution sol) {
        resultsMessage = "No results";
        currentSol = sol;
        currentSolId = currentSol.id.longValue();
        solutionLatLonZ = null;
        summaryMag = null;
        if (summaryMagStats == null) summaryMagStats = new MagDataStatistics();
        else summaryMagStats.init();

        if (defaultSourceDevice != null) sourceDevice = defaultSourceDevice;
        else if (! currentSol.source.isNull()) sourceDevice = currentSol.source.toString();
        else sourceDevice = null;

        ttGenerator.setSource(currentSol);

        if (currentSol != null) {
            solutionLatLonZ = currentSol.getLatLonZ();
            if (currentSol.phaseList == null || currentSol.phaseList.size() == 0) {
                currentSol.loadPhaseList();
                currentSol.setStale(false);
            }
            //if (debug && currentSol.phaseList != null) currentSol.phaseList.dump();
        }
    }

    public final Solution getSolution() {
        return currentSol;
    }

    protected Phase setDefaultPhase(Channel chan, String type, Solution sol) {
            defaultPhase.setChannelObj(chan);
            defaultPhase.description.set(type, "", "");
            defaultPhase.sol = sol;
            return defaultPhase;
    }

    protected double getPTravelTime(Channel chan) {
        return getTravelTime(chan, P_TYPE) ;
    }

    protected double getSTravelTime(Channel chan) {
        return getTravelTime(chan, S_TYPE) ;
    }

    public double getPhaseTravelTime(Channel chan, String type) {
        double tt = 0.;
        // hasSPhaseTimed = false;
        if (currentSol.phaseList.size() > 0) {
            Phase phase = setDefaultPhase(chan, type, currentSol);
            phase = currentSol.phaseList.getPhaseSame(phase);
            if (phase != null) {
                tt = phase.getTime();
                if (tt > 0.) {
                    tt -= currentSol.datetime.doubleValue();   // convert to traveltime by removing origin time
                   //if (S_TYPE.equals(type)) hasSPhaseTimed = true;
                }
            }
        }
        return tt;
    }

    protected double getTravelTime(Channel chan, String type) {
        double tt = getPhaseTravelTime(chan, type);
        return (tt > 0.) ? tt : calcTravelTime(chan, type);
    }

    public Channel getChannelFromList(Channel inputChannel) {
        if (! hasChannelList()) {
            resultsMessage =
                "CodaMagnitudeGenerator.getChannelFromList null/empty list, invoke setChannelList(List) or initChannelList()";
            System.out.println(resultsMessage);
            return null;
        }
        //Channel chan = channelList.findSimilarInList(inputChannel);
        Channel chan = channelList.findSimilarInMap(inputChannel);
        if (chan == null) {
            if (verbose)
                System.out.println(
                   "CodaMagnitudeGenerator.getChannelFromList Unable to find channel:" + getStnChlNameString(inputChannel)
                );
        }
        return chan;
    }

    public double getChannelToSolutionDistance(Channel chan) {
        if (chan.dist.isNull()) {
            chan.dist.setValue(GeoidalConvert.horizontalDistanceKmBetween(chan.latlonz, solutionLatLonZ));
        }
        return chan.dist.doubleValue();
    }

    public double getChannelToSolutionAzimuth(Channel chan) {
        if (chan.azimuth.isNull())
            chan.azimuth.setValue(GeoidalConvert.azimuthDegreesTo(chan.latlonz, solutionLatLonZ));
        return chan.azimuth.doubleValue();
    }

    protected double calcTravelTime(Channel chan, String type) {
        double range = getChannelToSolutionDistance(chan);
        //if (debug) System.out.println("   calcTT:" + getStnChlNameString(chan) + " range:" + range);
        return (type == P_TYPE) ? ttGenerator.pTravelTime(range) : ttGenerator.sTravelTime(range);
    }

    public int calcChannelMagByTime(double start, double end) {
        return calcChannelMag(Solution.create().getByTime(start, end));
    }

    public int calcChannelMag(SolutionList solList) {
        return calcChannelMag((Solution []) solList.toArray(new Solution[solList.size()]));
    }

    public int calcChannelMag(Solution [] solutions) {
        int size = (solutions == null) ?  0 : solutions.length;
        if (size < 1) return 0;
        int successCount = 0;
        for (int index = 0; index < size; index++) {
            if (calcChannelMag(solutions[index]) > 0) successCount++;
        }
        return successCount;
    }

    // Returns number of solution waveforms with good codas
    public int calcChannelMagForEvid(long evid) {
        return calcChannelMag(Solution.create().getValidById(evid)); // create a Solution object using prefor of evid
    }

    protected Collection getSolutionWaveformList() {
        if (currentSol == null) throw new NullPointerException("Current solution is null");
        Collection wfList = currentSol.waveformList;
        if (wfList == null || wfList.size() < 1) {
            wfList = Waveform.create().getBySolution(currentSol); // associate all event waveforms with chosen origin,
            currentSol.addWaveforms(wfList);
        }
        return wfList;
    }

    public boolean initChannelList() {
        if (channelCacheFileName != null) channelList = loadChannelList(channelCacheFileName);
        else if (dbEnabled) channelList = loadChannelListFromDb(EpochTime.epochToDate(currentSol.datetime.doubleValue()));
        if (! hasChannelList()) {
            resultsMessage = "CodaMagnitudeGenerator warning loaded channel list is null or empty!";
            System.out.println(resultsMessage);
            return false;
        }
        channelList.createLookupMap();
        //channelList.toString();
        return true;
    }

    public final boolean hasChannelList() {
        return (channelList != null && ! channelList.isEmpty());
    }

    public ChannelList getChannelList() {
        return channelList;
    }

    public void setChannelList(ChannelList list) {
        channelList = list;
    }

    public abstract ChannelList loadChannelList(String filename);
    public abstract ChannelList loadChannelListFromDb(java.util.Date date);
    public abstract boolean storeChannelList(String filename);

    public abstract void initCalibrList(java.util.Date date) ;

    // Returns number of solution waveforms with good codas creates a solution waveform list.
    public int calcChannelMag(Solution sol)  {
        if (sol == null) {
            System.out.println("CodaMagnitudeGenerator calcChannelMag null solution input");
            return 0;
        }
        else setSolution(sol); // must set the active solution i.e. currentSol for this instance
        return calcChannelMag(sol, getSolutionWaveformList());
    }

    // Returns number of solution waveforms with good codas, uses passed parameter waveform list.
    public int calcChannelMag(Solution sol, Collection wfList)  {
        if (sol == null) {
            System.out.println("CodaMagnitudeGenerator calcChannelMag null solution input");
            return 0;
        }
        else setSolution(sol); // must set the active solution i.e. currentSol for this instance
        // clear out any preexisting coda list  entries before recalculation
        sol.getCodaList().clear();
        //Collection wfList = getSolutionWaveformList();  // changed to humor doug here, see above,  no default load, 01/01/18 aww
        if (wfList == null ) {
            if(verbose)
                 System.out.println("CodaMagnitudeGenerator calcChannelMag: null waveform list found, solution id:"
                 + currentSolId);
            return 0;
        }
        if (wfList.size() < 1) {
            if (verbose)
                 System.out.println("CodaMagnitudeGenerator calcChannelMag: No waveforms in list, solution id:"
                 + currentSolId);
            return 0;
        }

        if (! hasChannelList()) {
            /*DEBUG
                System.out.println("channel list = null? :" + (channelList == null));
                if (channelList != null) System.out.println("channel list.isEmpty()? :" + (channelList.isEmpty()));
                System.out.println("has invalid channel list, loading new one ..." );
            */
            if (! initChannelList() ) {
                System.out.println("CodaMagnitudeGenerator failure constructing valid station channel list.");
                return 0;
            }
        }

        if (calibrList == null)  initCalibrList(EpochTime.epochToDate(currentSol.datetime.doubleValue()));

        if (verbose) {
            logHeader();
            System.out.println("    Total waveforms in list:" + wfList.size());
        }

        // prescan all waveform headers for coda time span eligibility then process derived waveform list
        solCodaWaveformDataList = getCodaWaveformData(wfList) ;
        eligibleCount = solCodaWaveformDataList.size(); // maxProgressValue
        if (verbose) System.out.println("    Total waveforms timespan eligible:" + eligibleCount + "\n");
        if (eligibleCount <= 0) return 0;

        calcCount = 0;  // reset counter total per solution
        int validCount = processCodaWaveformData(solCodaWaveformDataList);
        if (verbose) {
            System.out.println("    Total solution channel codas calculated: " + calcCount);
            System.out.println("    Total solution channel codas validated : " + validCount);
        }
        try {
           if (validCount > 0 && isAutoCommit()) commitCoda();
        } catch (JasiCommitException ex) {}

        return validCount;
    }

    protected Channel getWaveformChannel(Waveform wave) {
        if (wave == null) {
            if (verbose) System.out.println("getWaveformChannel(Wave) wave null, solution id:" + currentSolId);
            return null;
        }
        if (debug) System.out.println("\nCodaMagnitudeGenerator.getWaveformChannel wave:\n" + wave.toString());

        Channel chan = getChannelFromList(wave.getChannelObj());
        if (chan == null) {
            System.out.println("CodaMagnitudeGenerator.getWaveformChannel(Wave) channel not in list "
                               +  wave.getChannelObj().getChannelName().toDelimitedString('_')
                               + "  solution id:" + currentSolId);
        }
        return chan;
    }


    private CodaWaveformData getTimeSeriesData(int index, List codaWaveformDataList) {
        solWaveformToProcessIndex = index; // currentProgressValue
        CodaWaveformData data  = (CodaWaveformData) codaWaveformDataList.get(index);
        progressMessageChannelName = data.wave.getChannelObj().getChannelName().toDelimitedString('_');
        if (waveformCacheLoadingEnabled) {
            if (index <= solWaveformLoadedIndex) return data;
            if (loadingThread == null) {
                loadingThread  = new Thread(this);
                loadingThread.setDaemon(true);
                loadingThread.start();
            }
            try {
                for (int count = 0; count < maxSleepLoadingIterations; count++) {
                    if ( ! loadingThread.isAlive() || loadingThread.isInterrupted()) break;
                    Thread.currentThread().sleep(maxSleepTimeMillis);
                    if (index <= solWaveformLoadedIndex) return data;
                }
            }
            catch (InterruptedException ex) {
               ex.printStackTrace();
            }
            resultsMessage = "CodaMagnitudeGenerator.getTimeSeriesData unable to load time series data";
            System.out.println(resultsMessage);
            return null;
        }
        else {
            loadWaveformData(data); // single thread load here
            solWaveformLoadedIndex = index;
            return data;
        }
    }

    private void initializeWaveformListProcessingIndices() {
        solWaveformToProcessIndex = -1;
        solWaveformLoadedIndex = -1;
    }

    protected final int processCodaWaveformData(List codaWaveformDataList) {
        initializeWaveformListProcessingIndices();
        int size = (codaWaveformDataList == null) ? 0 : codaWaveformDataList.size();
        if (size < 1) return 0;
        if (debug) System.out.println(" **** processCodaWaveformData input list size: " + size);
        int validCount = 0;
        sortWaveformsByDistance(codaWaveformDataList);  // add sort by distance here
        if (logCodaCalc) logCodaOutputHeader();
        for (int index = 0; index < size; index++) {
            CodaWaveformData data = getTimeSeriesData(index, codaWaveformDataList);
            if (data == null) {
                System.out.println(" WARNING - timeout loading time series, skipped further loading of coda waveform data");
                System.out.println("    processCodaWaveformData indices current,loaded: " +index+ " " + solWaveformLoadedIndex);
                break; // aborts all further processing alternative is just to continue to next index ????
            }
            if (debug) {
                System.out.println("  processCodaWaveformData indices current,loaded: " +index+ " " + solWaveformLoadedIndex);
                // System.out.println(data.toString());
            }

            if (calcChannelMag(data)) {
                Coda coda = createChannelCoda(data.wave.getChannelObj(), data.pTime);
                if (isValidCoda(coda)) {         // check of windowcount, other values ?
                    resultsMessage = "Calculated valid channel coda, associated with solution";
                    coda.associate(currentSol);  // stores in list used for later summary processing
                    validCount++;
                }
            }
            Thread.currentThread().yield();
        }
        return validCount;
    }

    private Channel initWaveformChannel(Waveform wave) {
        Channel chan = getWaveformChannel(wave);
        if (chan == null) return null;
        if (! acceptChannelType(chan.getSeedchan())) {
            if (debug) {
                System.out.println("    initWaveformChannel not accepting channel of type : " + chan.getSeedchan());
            }
            return null;
        }
        Channel channel = (Channel) chan.clone(); // copy lat, lon
        channel.azimuth.setNull(true);
           // note: do not use Channel.calcDistance(LatLonZ) need horizontal range for ttraveltime generator;
        channel.dist.setValue(GeoidalConvert.horizontalDistanceKmBetween(channel.latlonz, solutionLatLonZ));
        wave.setChannelObj(channel);
        return channel;
    }

    protected final List getCodaWaveformData(Collection waveList) {
        int size = (waveList == null) ? 0 : waveList.size();
        if (size < 1) {
            resultsMessage = "CodaMagnitudeGenerator.getCodaWaveformData found empty waveform Collection";
            return null;
        }
        List codaWaveList = new ArrayList(size);
        Waveform [] wave = (Waveform []) waveList.toArray(new Waveform [size]);
        double originTime  = currentSol.datetime.doubleValue();
        if (debug) {
            System.out.println("    **** getcodaWaveformData input list size: " + size);
            System.out.println("    Origintime:" + EpochTime.toNoZoneYYString(originTime));
        }

        for (int idx = 0; idx < size; idx++) {
            CodaWaveformData codaWaveformData = createCodaWaveformData(wave[idx], originTime);
            if (codaWaveformData != null) codaWaveList.add(codaWaveformData);
        }
        //System.out.println("DEBUG waveforms in output:" + codaWaveList.size());
        return codaWaveList;
    }

    public final Coda calcChannelCoda(Waveform wave, double originTime) {
        CodaWaveformData data = createCodaWaveformData(wave, originTime);
        loadWaveformData(data);
        return (calcChannelMag(data)) ? createChannelCoda(data.wave.getChannelObj(), data.pTime) : null;
    }

    //protected boolean hasSPhaseTimed = false;  // only if observed phase time is known to be good.
    protected final double getCodaStartOffsetSecs(double smpTime) {
        //if (hasSPhaseTimed) return codaStartOffsetSecs;
        return (codaStartOffsetSecs > 0.) ? codaStartOffsetSecs : smpTime * 0.1; // offset inside S arrival
    }

    protected final CodaWaveformData createCodaWaveformData(Waveform wave, double originTime ) {
        Channel chan = initWaveformChannel(wave);
        if (chan == null) return null;
        double pTT = getPTravelTime(chan);
        double pTime = originTime + pTT;
        double sTT = getSTravelTime(chan);
        double sCodaStartTime = originTime + sTT + getCodaStartOffsetSecs(sTT-pTT) ; // offset inside S arrival
        double waveStartTime = wave.getEpochStart();
        double waveEndTime  = wave.getEpochEnd();
        if (debug) System.out.println("    createCodaWaveformData " +
             getStnChlNameString(chan) + " pTime:" + EpochTime.toNoZoneYYString(pTime) +
             " sTravelTime:" + EpochTime.toNoZoneYYString(sTT) +
             " codaStartTime:" + EpochTime.toNoZoneYYString(sCodaStartTime));
        if (pTime > waveEndTime || sCodaStartTime > waveEndTime || sCodaStartTime < waveStartTime) {
            if (debug) System.out.println("    createCodaWaveformData wave time out of bounds for id:" + currentSolId);
            return null;
        }
        return new CodaWaveformData(wave, pTime, sCodaStartTime);
    }

// invoke in runnable thread:
    public final void run() {
        if (debug) System.out.println("*** Loading thread started *** ");
        loadWaveformData(solCodaWaveformDataList);
        loadingThread  = null;
        if (debug) System.out.println("*** Loading thread ended *** ");
    }

// invoke in runnable thread:
    protected final void loadWaveformData(CodaWaveformData codaWaveData) {
        codaWaveData.selfLoadedWaveTimeSeries = false;
        if (! codaWaveData.wave.hasTimeSeries()) { // load time series only if not preloaded
            codaWaveData.selfLoadedWaveTimeSeries = codaWaveData.wave.loadTimeSeries();
            if (! codaWaveData.selfLoadedWaveTimeSeries) {
                if (debug) System.out.println("loadWaveformData wave timeseries not loaded:" + currentSolId);
                codaWaveData.rejected = true;
            }
        }
    }

// invoke in runnable thread:
    protected final void loadWaveformData(List codaWaveData) {
        int size = (codaWaveData == null) ? 0 : codaWaveData.size();
        if (size < 1) return;
        for (int idx = solWaveformLoadedIndex+1; idx < size; idx++) {
            if (solWaveformLoadedIndex - solWaveformToProcessIndex > maxWaveformCacheSize) {
                if (debug) System.out.println("   Max waveforms cached: " +maxWaveformCacheSize+ " loading thread stopping.");
                break;
            }
            loadWaveformData((CodaWaveformData) codaWaveData.get(idx));
            solWaveformLoadedIndex = idx;
            Thread.currentThread().yield();
        }
    }

    /*
    Does not load the timeseries if Channel data out of range,
    sets start preP to recover timeseries needed to calculate pre-event noise LTA and S wave coda.
    Waveform objects must contain valid Channel data objects
    */
    protected boolean calcChannelMag(CodaWaveformData codaWaveData) {
        if (codaWaveData.rejected) return false;
        Waveform wave = codaWaveData.wave;
        boolean selfLoadedWaveTimeSeries = codaWaveData.selfLoadedWaveTimeSeries;
        double waveStartTime =  wave.getEpochStart();
        double waveEndTime   = wave.getEpochEnd();

        /* patch here temporary for .001 jigger at end of data after collapse
                WFSegment waveSegment = (WFSegment) wave.getSegmentList().get(wave.getSegmentList().size()-1);
                waveEndTime = waveSegment.getEpochEnd();
        */
        // end of patch aww 12/7/00

        double preEventStartTime =
            codaWaveData.pTime - codaGenerator.getNoiseBiasSamples()/wave.getSampleRate() - codaPrePTimeBiasOffsetSecs;

            if (debug) {
                System.out.println(
                    "calcChannelMag " + wave.getChannelObj().getChannelName().toDelimitedString('_') +
                    " prePStart:" + EpochTime.toNoZoneYYString(preEventStartTime) +
                    " waveStart:" + EpochTime.toNoZoneYYString(waveStartTime) +
                    " waveEnd:" + EpochTime.toNoZoneYYString(waveEndTime));
            }
        if (preEventStartTime < waveStartTime) preEventStartTime = waveStartTime;
        if (! codaGenerator.isResetOnClipping()) {
            double desiredWaveEndTime = codaWaveData.sCodaStartTime + codaGenerator.getMaxCodaDurationSecs();
            if (debug) System.out.println("  desired waveEndTime:" + EpochTime.toNoZoneYYString(desiredWaveEndTime));
            waveEndTime = Math.min(waveEndTime, desiredWaveEndTime);
        }

        if (waveEndTime < waveStartTime) {
            if (debug) {
                System.out.println("    wave number of segments: " + wave.getSegmentList().size());
                System.out.println("    input waveform timeStart: " + EpochTime.toNoZoneYYString(wave.getEpochStart()) );
                System.out.println("    input waveform   timeEnd: " + EpochTime.toNoZoneYYString(wave.getEpochEnd()) );
            }

            System.out.println("CodaMagnitudeGenerator.calcChannelMag waveEndTime < waveStartTime "
                + getStnChlNameString(wave.getChannelObj()));
            if (selfLoadedWaveTimeSeries) wave.unloadTimeSeries();
            return false;
        }

        DateRange codaTimeSpan = new DateRange(preEventStartTime, waveEndTime);
            if (debug) {
                System.out.println("  codaTimeSpan: " + codaTimeSpan.toString());
                if (wave.hasTimeTears()) System.out.println(" waveform has time tears");
            }

        WFSegment wfs = window(wave, codaTimeSpan); // aka WFSegment.collapse(Vector)
        if (wfs == null) {
            if (debug)
                System.out.println("calcChannelMag requested Swave coda segment out of bounds: " +
                                    wave.getChannelObj().getChannelName().toDelimitedString('_'));
            if (selfLoadedWaveTimeSeries) wave.unloadTimeSeries();
            return false;
        }

        // scan coda from startingIdx to end.
        double wfSampleInterval = wfs.getSampleInterval();
        int startingIdx = (int) Math.round((codaWaveData.sCodaStartTime - wfs.getEpochStart())/wfSampleInterval);
        int wfSamplesExpected = wfs.samplesExpected;
        if (startingIdx < 0 || startingIdx >= wfSamplesExpected) {
            if (debug) System.out.println("calcChannelMag requested Swave coda startindex out of bounds");
            if (selfLoadedWaveTimeSeries) wave.unloadTimeSeries();
            return false;
        }

        // calculate the coda fit parameters for station channel
        double smpSecsToAdd = codaWaveData.sCodaStartTime - codaWaveData.pTime;        // S-P time to add to S-coda to get P-coda
          boolean status = calcCoda(wfSampleInterval, smpSecsToAdd, wfs.getTimeSeries(), startingIdx,
                                    wfSamplesExpected, wave.getChannelObj(), 0.);
        if (logCodaCalc) logCodaResult(wave.getChannelObj());
        if (selfLoadedWaveTimeSeries) wave.unloadTimeSeries();
        calcCount++;
        return status;
    }

    public final double getSummaryMagnitudeValue() {
        return (summaryMag == null) ? -1. : summaryMag.value.doubleValue();
    }

    public final Magnitude getSummaryMagnitude() {
        return summaryMag;
    }

    protected final WFSegment getWaveformSegment(Waveform wave, DateRange codaTimeSpan) {
        List waveformSegments = (List) wave.getSegmentList();
        if (waveformSegments == null)  {
            System.out.println("ERROR - waveform segments list NULL");
            return null;
        }
        int size = waveformSegments.size(); // should be 1 if contiguous
        if (debug) System.out.println("    getWaveformSegment total waveform segments: " + size);
        for (int idx = 0; idx < size; idx++) {
            WFSegment wfs = (WFSegment) waveformSegments.get(idx);
            DateRange ts = new DateRange(wfs.getEpochStart(), wfs.getEpochEnd());
            if (debug) {
               System.out.println("      getWaveformSegment segment timespan: " + ts.toString());
            }
            // if (ts.contains(codaTimeSpan)) return wfs;
            if (ts.contains(codaTimeSpan)) return new WFSegment(wfs);
        }
        // resultsMessage = "time tear in requested waveform window";
        return null;
    }

    protected final WFSegment window(Waveform wave, DateRange codaTimeSpan) {
        double windowStartTime = codaTimeSpan.getMinEpochSecs();
        double windowEndTime =   codaTimeSpan.getMaxEpochSecs();
        if (windowStartTime >= windowEndTime) return null;

        WFSegment wfs = getWaveformSegment(wave, codaTimeSpan);
        if (wfs == null ) {
            if (debug) System.out.println("    CodaMagnitudeGenerator.window coda time not contiguous; spans wave segments");
            return null;
        }
        int wfSamplesExpected = wfs.samplesExpected;
        int wfTimeSeriesSize = wfs.size();
        double wfEndTime = wfs.getEpochEnd();
        double wfStartTime = wfs.getEpochStart();
        if (debug) {
            System.out.println("    wfSamplesExpected: " + wfSamplesExpected + " size: " + wfTimeSeriesSize);
            System.out.println("    wfStartTime: " + EpochTime.toNoZoneYYString(wfStartTime)
                                + " wfEndTIme: " + EpochTime.toNoZoneYYString(wfEndTime));
            System.out.println("    windowStartTime: " + EpochTime.toNoZoneYYString(windowStartTime)
                                + " windowEndTime: " + EpochTime.toNoZoneYYString(windowEndTime));
        }
        if (wfTimeSeriesSize < 1 || windowStartTime > wfEndTime  || windowEndTime < wfStartTime) return null;
        if (windowStartTime < wfStartTime) windowStartTime = wfStartTime;
        if (windowEndTime > wfEndTime) windowEndTime = wfEndTime;
        double wfSampleInterval = wfs.getSampleInterval();
        int sampleStartOffset = (int) Math.round((windowStartTime - wfStartTime )/wfSampleInterval);
        int sampleCount = (int) Math.round((windowEndTime - windowStartTime)/wfSampleInterval) + 1; // include end of range sample
        if ((sampleCount + sampleStartOffset) > wfTimeSeriesSize) {
            sampleCount = wfTimeSeriesSize - sampleStartOffset;
        }
        //if (debug) System.out.println("    window wfs.ts.sampleCount: " + sampleCount);
        float[] sample = new float[sampleCount];
        float[] wfsTimeSeries = wfs.getTimeSeries();
        for (int idx = 0; idx < sampleCount; idx++) {
            sample[idx] = wfsTimeSeries[idx + sampleStartOffset];
        }

        wfs.setStart(windowStartTime);
        wfs.setEnd(windowEndTime);
        wfs.samplesExpected = sampleCount;
        wfs.setTimeSeries(sample, false);
        //deprecated wfs.length      = wfs.bytesPerSample*wfs.sampleCount;
        //deprecated wfs.lenSecs     = wfs.getEpochEnd() - wfs.getEpochStart();
        return wfs;

    }

    protected boolean calcCoda(double secsPerSample, double smpSecsToAdd, float [] timeSeries,
                           int startingIdx, int sampleCount, Channel chan, double lta) {

        CodaCalibrParms calibrParms = getCodaCalibrParms(chan);

        double codaCutoffAmp = 0.;
        double clippingAmp = 0.;

        if (calibrParms != null) {
            codaCutoffAmp = calibrParms.codaCutoffAmp;
            clippingAmp = calibrParms.clippingAmp;
        }

        String seedchan = chan.getSeedchan();
        if (codaGenerator.hasFilter()) {
            if (filterChannelType(seedchan)) {
                codaGenerator.enableFilter();
                codaGenerator.setFilter(seedchan);
            }
            else codaGenerator.disableFilter();
        }

        if (debug)
            System.out.println("CodaMagnitudeGenerator.calcCoda " + getStnChlNameString(chan) +
            " fit codaCutoffAmp:" + codaCutoffAmp + " clippingAmp:" + clippingAmp);

        boolean status = codaGenerator.calcCoda(secsPerSample, smpSecsToAdd,
                             timeSeries, startingIdx, sampleCount,
                             codaCutoffAmp, clippingAmp, lta) ;

        if (debug && ! status)
            System.out.println("CodaMagnitudeGenerator.calcCoda bad coda:" + " * exit:" + codaGenerator.getExitStatusString());

        return status;
    }

    public CodaCalibrParms getCodaCalibrParms(Channel chan) {
        if (chan == null) return null;

        CodaCalibration calibr = null;
        if (calibrList != null) calibr = calibrList.get(chan.getChannelId());

        return  (calibr == null)
                        ? defaultCalibration.getDefaultCodaCalibrParms(chan.getChannelId())
                        : calibr.getCodaCalibrParms();
    }

    protected Coda createChannelCoda(Channel chan, double pTime) {
        Coda coda = null;
        coda = codaGenerator.setCodaResults(coda); // set values: ngood, AFix, AFree, QFree, tau, and windowAmp
        coda.setChannelObj(chan);
        if (! org.trinet.jdbc.NullValueDb.isBlank(sourceDevice)) coda.source.setValue(sourceDevice);
        //coda.setHorizonatlDistance(getChannelToSolutionDistance(chan));
        coda.getChannelObj().calcDistance(solutionLatLonZ);

        coda.getChannelObj().azimuth.setValue(getChannelToSolutionAzimuth(chan));
        coda.datetime.setValue(pTime);
        return coda;
    }

    public boolean isValidSolution(Solution sol) {
        return (sol != null);
    }
    public abstract boolean acceptChannelType(String type) ;
    public abstract boolean filterChannelType(String type) ;
    public abstract boolean summaryChannelType(String type) ;
    public abstract boolean isValidCoda(Coda coda) ;

    public abstract double setChannelMag(Coda coda, CodaCalibrParms calibrParms) ;
    public abstract Magnitude createSummaryMag(int count, double value, double err, double minRange, double maxGap) ;
    public abstract Magnitude createSummaryMag(Coda [] codaArray);

    protected boolean removeSummaryMagAssociation() {
        if (summaryMag != null) summaryMag.unassociate();
        return currentSol.getAlternateMagnitudes().remove(summaryMag);
    }

    protected void associateSummaryMag() {
        if (summaryMag == null || hasMagAssociationDisabled()) return;
        if (! currentSol.getAlternateMagnitudes().contains(summaryMag)) currentSol.addAlternateMagnitude(summaryMag);
        //currentSol.setPreferredMagnitude(summaryMag);
        summaryMag.associate(currentSol);
    }

   //Calculate summary magnitude and coda statistics using collection of Coda table row objects
    protected boolean calcSummaryMag(Coda [] codaArray) {
        summaryMag = null;
        if (codaArray == null)  {
            resultsMessage = " ERROR - calcSummaryMag(Coda[] input parameter array null";
            System.out.println(resultsMessage);
            return false;
        }
        if (currentSol == null) throw new NullPointerException("Current solution is null, see setSolution(Solution)");

        if (codaArray.length < minCodaCountForSummaryMag) {
            resultsMessage = "codaArray.length < minCodaCountForSummaryMag";
            return false;
        }

        summaryMag = createSummaryMag(codaArray);
        if (summaryMag == null) return false;

        associateSummaryMag();
        try {
          return (isAutoCommit()) ? commitMag() : true;
        } catch (JasiCommitException ex) {
          return false;
        }
    }

    protected CalibratedCoda [] getCalibratedCoda(Coda [] codaArray) {
        int size = (codaArray == null) ? 0 : codaArray.length;
        ArrayList list = new ArrayList(size);
        int calibratedCodaCount = 0;
        for (int idx = 0; idx < size; idx++) {
            Solution sol = codaArray[idx].getAssociatedSolution();
            if (sol != currentSol)
                throw new AssociatedSolutionMismatchException("Current solution != associated solution.");
            Channel chan = codaArray[idx].getChannelObj();
            CodaCalibrParms calibrParms = getCodaCalibrParms(chan);
            boolean isSummaryChannelType = summaryChannelType(chan.getSeedchan());
            if (! isSummaryChannelType) {
                if (debug) System.out.println("    calcSummaryMag not summary channel type: " +  getStnChlNameString(chan));
                continue;
            }
            if (! calibrParms.hasMagCorr()) {
                if (isSummaryChannelType) {
                    if (logCodaCalc) System.out.println("    calcSummaryMag no magCorr found for: " + getStnChlNameString(chan));
                }
                continue;
            }
            //calibratedCoda[calibratedCodaCount].coda = codaArray[idx];
            //calibratedCoda[calibratedCodaCount].calibr = calibrParms;
            list.add(new CalibratedCoda(codaArray[idx], calibrParms));
            calibratedCodaCount++;
        }
        if (calibratedCodaCount < 1) {
            calibrList = null; // reset to re-initialize for next event ??
        }
        return (CalibratedCoda []) list.toArray(new CalibratedCoda[list.size()]);
    }

    protected Magnitude createSummaryMag(CalibratedCoda [] calibratedCoda, boolean verbose) {
        int calibratedCodaCount = calibratedCoda.length;
        if (calibratedCodaCount < minCodaCountForSummaryMag) {
            resultsMessage = "calibratedCodaCount < minCodaCountForSummaryMag";
            return null;
        }

        double [] magValues = new double[calibratedCodaCount];
        double [] range     = new double[calibratedCodaCount];
        double [] azimuth   = new double[calibratedCodaCount];
        double [] weights   = new double[calibratedCodaCount];

        for (int idx = 0; idx < calibratedCodaCount; idx++) {
            Channel chan   = calibratedCoda[idx].coda.getChannelObj();
            range[idx]     = getChannelToSolutionDistance(chan);
            azimuth[idx]   = getChannelToSolutionAzimuth(chan);
            magValues[idx] = setChannelMag(calibratedCoda[idx].coda, calibratedCoda[idx].calibr);
            weights[idx]   = calibratedCoda[idx].coda.weightIn.doubleValue();
        }

        double sumMagValue = summaryMagStats.calcValue(magValues, weights, calibratedCodaCount);

        // report summary statistics.
        if (verbose) System.out.println("Summary coda magnitude fit solution id: " + currentSolId + summaryMagStats.toString());

        Arrays.sort(range,   0, calibratedCodaCount); // ascending, first is closest to solution
        Arrays.sort(azimuth, 0, calibratedCodaCount);
        double gap = getMaxGap(azimuth, calibratedCodaCount);

        Magnitude mag = createSummaryMag(summaryMagStats.count, summaryMagStats.value, summaryMagStats.getError(), range[0], gap);
        //moved to MCA class invocation of updateCodaMagResiduals(mag, calibratedCoda, logMagResiduals);
        resultsMessage = "CodaMagnitudeGenerator success - created summary magnitude";
        return mag;
    }

    protected void updateCodaMagResiduals(Magnitude mag, CalibratedCoda [] codaArray, boolean verbose) {
        if (mag == null || codaArray == null) throw new NullPointerException("input argument null");
        int size = codaArray.length;
        if (size < 1 ) return;

        double magValue = mag.value.doubleValue();
        if (verbose) System.out.println("  channel       dist     mag   resid      wt segments");
        for (int idx = 0; idx < size; idx++) {
            Coda coda = codaArray[idx].coda;
            // System.out.println("fit updateCodaMag result [" + idx + "]:" + coda.toString());

            ChannelMag chanMag = coda.getChannelMag();
            if (! chanMag.value.isNull()) {
                chanMag.residual.setValue(chanMag.value.doubleValue() - magValue);
                chanMag.weight.setValue(1.);
            }

            coda.associateMag(mag);

            if (verbose) {
                StringBuffer sb = new StringBuffer(132);
                sb.append("  ");
                sb.append(getStnChlNameString(coda.getChannelObj()));
                Concatenate.format(sb, coda.getHorizontalDistance(), 4,1).append(" ");
                if (debug) sb.append(codaArray[idx].calibr.toString()).append(" ");
                Concatenate.format(sb, chanMag.value.doubleValue(),4,2).append(" ");
                Concatenate.format(sb, chanMag.residual.doubleValue(),4,2).append(" ");
                Concatenate.format(sb, coda.weightIn.doubleValue(),4,2).append(" ");
                Concatenate.format(sb, coda.windowCount.longValue(),4);
                System.out.println(sb.toString());
            }
        }
        if (verbose) System.out.println();
    }

    protected final double getMaxGap(double[] azimuth, int length) {
        int size = (azimuth == null) ? 0 : length;
        if (size < 2 ) return 360.;

        int maxIdx = size - 1;
        double deltaAz = 0.;
        double maxGap  = 0.;
        for (int idx = 0; idx < maxIdx; idx++) {
            deltaAz = Math.abs(azimuth[idx+1] - azimuth[idx]);
            if (deltaAz > maxGap) maxGap = deltaAz;
        }
        deltaAz = 360. - Math.abs(azimuth[maxIdx]) + Math.abs(azimuth[0]);
        return (deltaAz > maxGap) ? deltaAz : maxGap;
    }

    public boolean calcSummaryMagForEvid(long id, boolean useExisting) {
        Solution sol = Solution.create().getValidById(id);
        if (sol == null) {
            resultsMessage = "CodaMagnitudeGenerator.calcSummaryMagForEvid error - no solution for input id:" + id;
            if (verbose) System.out.println(resultsMessage);
            return false;
        }
        return calcSummaryMag(sol, useExisting);
    }

    public int calcSummaryMagByTime(double start, double end, boolean useExisting) {
        return calcSummaryMag(Solution.create().getByTime(start, end), useExisting);
    }

    public int calcSummaryMag(SolutionList solList, boolean useExisting) {
        return calcSummaryMag((Solution []) solList.toArray(new Solution[solList.size()]), useExisting);
    }

    public int calcSummaryMag(Solution [] solutions, boolean useExisting) {
        int size = (solutions == null) ? 0 : solutions.length;
        if (size < 1) return 0;
        int successCount = 0;
        for (int index = 0; index < size; index++) {
            if (calcSummaryMag(solutions[index], useExisting)) successCount++;
        }
        return successCount;
    }

    public boolean calcSummaryMag(Solution sol, boolean useExisting) {
        if (sol == null) {
            System.out.println("CodaMagnitudeGenerator error - null solution input");
            return false;
        }
        Coda [] codaArray = null;
        if (useExisting) {
            Collection codaList = Coda.create().getBySolution(sol);
            if (codaList != null) codaArray = (Coda []) codaList.toArray(new Coda [codaList.size()]);
            setSolution(sol);
        }
        else {
             calcChannelMag(sol);
             codaArray = currentSol.codaList.getArray();
        }
        if (codaArray == null || codaArray.length == 0) {
            if (verbose)
                System.out.println("CodaMagnitudeGenerator - null or zero length codaArray, solution id: " + currentSolId);
            resultsMessage = "coda count < minCountForSummaryMag";
            return false;
        }
        return calcSummaryMag(codaArray);
    }

    public abstract boolean commitMag() throws JasiCommitException ;
    public abstract boolean commitCoda() throws JasiCommitException ;

    public boolean commit() throws JasiCommitException {
        return (commitCoda()) ? commitMag() : false;
    }

    public final void setAutoCommit(boolean value) {
        autoCommit = value;
    }

    public final boolean isAutoCommit() {
        return autoCommit;
    }

    protected void logHeader() {
        System.out.println();
        System.out.println("-----------------------------------------------------------------");
        System.out.println(currentSol.toFingerFormat());
        System.out.println("-----------------------------------------------------------------");
        System.out.println();
    }

    protected void logCodaOutputHeader() {
        System.out.println(
            "            range  phase qual   tau startAmp    endAmp good bad 1st   rms  QFix  QFree AFree AFix Filter   Status"
        );
    }

    public String getResultsMessage() {
        return resultsMessage;
    }

    protected String getStnChlNameString(Channel chan) {
        StringBuffer sb = new StringBuffer(32);
        sb.append(chan.getChannelName().toDelimitedSeedString('_'));
        sb.append("  ");
        return sb.substring(0,12);
    }

    protected void logCodaResult(Channel chan) {
        if (debug) logCodaOutputHeader();
        StringBuffer sb = new StringBuffer(160);
        sb.append(getStnChlNameString(chan));
        Concatenate.format(sb, chan.dist.doubleValue(), 4, 1);
        sb.append(" ");
        sb.append(codaGenerator.outputToString());
        System.out.println(sb.toString());
    }

    protected void setDefaultSource(String sourceDevice) {
        defaultSourceDevice = sourceDevice;
    }

// JasiSolutionProcessorIF
/*
    //public static final int CALC_CODA_ONLY                     = 0;
    //public static final int CALC_CODA_AND_SUMMARY              = 1;
    //public static final int CALC_SUMMARY_MAG_USE_EXISTING_CODA = 2;

    //public static final int UNKNOWN_PROCESSING_MODE = -1;
    //public static final int NULL_INPUT = -2;
    //public static final int INPUT_NOOP = 0;
    //public static final int UNIT_SUCCESS = 1;

    public int processSolution(Solution sol)  {
        switch (getSolutionProcessingMode().getIdCode()) {
            case CALC_CODA_ONLY:
                return calcChannelMag(Solution sol)  ;
            case CALC_CODA_AND_SUMMARY:
                return calcSummaryMag(sol, false) ;
            case CALC_SUMMARY_MAG_USE_EXISTING_CODA:
                return calcSummaryMag(sol, true) ;
        }
        return UNKNOWN_PROCESSING_MODE;
    }
*/

    public int processSolution(long evid) {
        return processSolution(Solution.create().getValidById(evid)); // create a Solution object using prefor of evid
    }

    public int processSolution(Solution sol)  {
        if (sol == null) return ProcessingResult.NULL_INPUT.getIdCode();
        CodaSolutionProcessingMode mode = (CodaSolutionProcessingMode) getSolutionProcessingMode();
        if (mode == CodaSolutionProcessingMode.CALC_CODA_ONLY) {
            return calcChannelMag(sol);
        }
        else if (mode == CodaSolutionProcessingMode.CALC_CODA_AND_SUMMARY_MAG) {
            return (calcSummaryMag(sol, false)) ?
                ProcessingResult.UNIT_SUCCESS.getIdCode() : ProcessingResult.FAILURE.getIdCode();
        }
        else if (mode == CodaSolutionProcessingMode.CALC_SUMMARY_MAG_USE_EXISTING_CODA) {
           return (calcSummaryMag(sol, true)) ?
                ProcessingResult.UNIT_SUCCESS.getIdCode() : ProcessingResult.FAILURE.getIdCode();
        }
        else return ProcessingResult.UNKNOWN_PROCESSING_MODE.getIdCode();
    }

    public int processSolution(Solution [] solutions) {
        if (solutions == null) return ProcessingResult.NULL_INPUT.getIdCode();
        int size = solutions.length;
        if (size < 1 ) return ProcessingResult.INPUT_NOOP.getIdCode();
        int resultCode = ProcessingResult.UNIT_SUCCESS.getIdCode();
        for (int index = 0; index < size; index++) {
            resultCode = processSolution(solutions [index]);
            if (resultCode != ProcessingResult.UNIT_SUCCESS.getIdCode()) break;
        }
        return resultCode;
    }

    public int processSolution(SolutionList solList) {
        return processSolution(solList.getArray()) ;
    }

    public void setSolutionProcessingMode(SolutionProcessingMode processingMode) {
        this.processingMode = (CodaSolutionProcessingMode) processingMode;
    }

    public SolutionProcessingMode getSolutionProcessingMode() {
        return processingMode;
    }

    public boolean isValidId(long id) { return (id > 0); }

} // end of CodaMagnitudeGenerator class
