package org.trinet.jiggle;

import java.util.*;
import java.awt.event.*;
//import java.sql.*;

import org.trinet.util.*;
import org.trinet.jdbc.*;
import org.trinet.jasi.*;
import org.trinet.jasi.coda.*;
import org.trinet.util.gazetteer.LatLonZ;

import javax.swing.*;	    // JFC "swing" library

/**
 * This class is the Big Data View.
 * It is primarily a list of WFViews that define a
 * "view" of network data. Note that this is a REQUESTED view;
 * some requested channels or time windows may contain no data.
 * This also contains a list of Solutions that are associated with phases
 * that appear in the view.
 * The MasterView timeSpan will be expanded to include any added objects.<p>
 *                                     6
 * The WFViews that make up the MasterView are determined by the ChannelTimeWindowModel.
 * The default model is DataSourceChannelTimeModel which gets its list of channels
 * and time windows from those associated with the event at the DataSource.
 *
 *  @see: ChannelTimeWindowModel()
 *  @see: DataSourceChannelTimeModel()
 *
 * @author Doug Given
 */
/*
    Removed all notions of what's "selected" from this class to keep it pure. What
    Solution, WFView, Phase, etc. is currently selected is meaningless for many
    uses of MasterView and, when needed, is maintained by SelectionModel classes.
 */

public class MasterView {

    /** The model for which WFPanel and :. which channel is selected. */
    public SelectedWFViewModel  masterWFViewModel = new SelectedWFViewModel();
    /** The model for the zoomed waveform window that is selected. */
    public SelectedWFWindowModel masterWFWindowModel = new SelectedWFWindowModel();
    /** The model for the scrolling window that is selected. */
//    public SelectedWFWindowModel masterWFScrollModel = new SelectedWFWindowModel();

    /** The total time span bounding this view */
    public TimeSpan timeSpan = new TimeSpan();

    /** List of solution in this view */
    //    public SolutionList solList = new SolutionList();

    /** MVC model for list and selected solution */
//    public ActiveSolutionList solList = new ActiveSolutionList();
    public SolutionList solList = new SolutionList();

    /** Phase list. Includes ALL phases from all solutions in the MasterView,
    * both associated and unassociated. */
//x//    public PhaseList phaseList = new PhaseList();

    /** Set true if you want to retreive all phases */
    boolean phaseRetrieveFlag = true;

    /** Default value for stripping residuals. */
    double residualStripValue = 1.0;

    /** Paint WFPanel time scale red if clock quality is worse than this value,
     *  0.0 -> 1.0 */
    double clockQualityThreshold = 0.5;

    /** Amplitudes list. Includes ALL amplitudes in the MasterView. */
//x//    public AmpList ampList = new AmpList();

    /** Set true if you want to retreive all amplitudes */
    boolean ampRetrieveFlag = true;

    /** Coda list. Includes ALL codas in the MasterView. */
//x//    public CodaList codaList = new CodaList();

    /** Set true if you want to retreive all amplitudes */
    boolean codaRetrieveFlag = true;

    /** List of WFViews */
    public WFViewList wfvList = new WFViewList();

    protected ChannelTimeWindowModel defaultChannelTimeWindowModel =
					  new DataSourceChannelTimeModel();

    /** Keep individual WFView lists in synch with the master lists. */
    JasiReadingListModel jasiReadingListModel = new JasiReadingListModel(wfvList);

    static final int NULL = (int) Math.pow(2.0, 31.0);

/** Set true if you want to retreive all waveforms */
    boolean wfRetrieveFlag = true;

/** If true, load Channel data (lat/lon) for each WFView */
    boolean loadChannelData = true;

    /* Enumerate load mode options */
    public static final int LoadAllInForeground = 0;
    public static final int LoadAllInBackground = 1;
    public static final int LoadNone = 2;
    public static final int Cache = 3;
    public static final int LimitNumber = 4;
    public static final int ModeCount = 4;

    /** Set waveform default load mode */
    int waveformLoadMode = LoadAllInBackground;

/** Set true if you want to line up the waveforms by time. If true the viewBox of
 *  all the WFView's will be set to the max/min times of the whole data set */
//    boolean timeAlign = true;

    public static final int AlignOnTime = 0;
//    public static final int AlignOnP    = 1;
//    public static final int AlignOnS    = 2;
    int alignmentMode = AlignOnTime;
    int AlignmentModeCount = 1;

    /** Include only WFViews that have phase picks */
    boolean includeOnlyWithPhases = false;

/** */
    StatusPanel statusPanel = null;

    /** Set max. waveforms to load */
    int waveformLoadLimit = Integer.MAX_VALUE;

    // output verbosity flag
       static boolean debug = true;
   // static boolean debug = false;

    // access runtime information
    //    Runtime runtime = Runtime.getRuntime();

/**
 * Default constructor
 */
  public MasterView() {

      // give all WFViews a reference to the MasterView via a static member
      WFView.setMasterView(this);

  }

 /**
 * Default constructor with statusPanel reference.
 *
 * @param ts - a StatusPanel where status updates will go
 * @see StatusPanel()
 */
  public MasterView(StatusPanel sp) {

      this();

      setStatusPanel(sp);

  }

/**
 * Constructor sets initial size of timeSpan
 *
 * @param ts - a TimeSpan object
 * @see TimeSpan
 */
  public MasterView(TimeSpan ts ) {

      this();

      timeSpan = ts;

  }

/**
 * Build a master view for this 'evid'. In other words, the view is based on one
 * event as found in the dbase.  The view's timeSpan will be defined to include
 * all the waveforms and phases associated with this evid.  Other phases in the
 * time span associated with another evid (or no evid) will not be included.  */
  public MasterView(long id) {

      this();

      defineByDataSource(id);
  }

/**
 * Build a master view for this Solution based on this event as found in the
 * DataSource.  The view's timeSpan will be defined to include all the waveforms
 * and phases associated with this evid.  Other phases in the time span
 * associated with another evid (or no evid) will not be included.  */
  public MasterView(Solution sol) {

      this();

      defineByDataSource(sol);
  }

    /**
     * Build a MasterView based on this solution containing stations within the
     * parameter bounds and with waveforms with the given timespan. The view is
     * NOT based on waveforms saved in the DataSource. Rather Waveform objects
     * are created based on the TimeSpan given. These are effectively 'requests'
     * that may or may not be successfully fulfilled when a call to
     * loadWaveforms() is made. The DataSource will be searced for the TimeSpan
     * specified to find phases, amps, etc.  <p>
     *
     * The list will end at either the  distance cutoff or the station count,
     * whichever comes first. For example, assume there are 50 channels within
     * 100 km of the epicenter; if distCutoff=100 & maxCount=20, only the
     * nearest 20 channels will be included. If distCutoff=100 & maxCount=200,
     * only the 50 stations within 100 km will be include. The station list
     * comes from the channel list in * the Channel class.
     *
     * @see: defineByParameters()*/

    public MasterView (Solution sol, double distCutoff, int maxCount, TimeSpan ts) {

      this();

      defineByParameters(sol, distCutoff,maxCount, ts);
    }

    /**
     * Build a MasterView based on this solution ID containing stations within the
     * parameter bounds and with waveforms with the given timespan. The view is
     * NOT based on waveforms saved in the DataSource. Rather Waveform objects
     * are created based on the TimeSpan given. These are effectively 'requests'
     * that may or may not be successfully fulfilled when a call to
     * loadWaveforms() is made. The DataSource will be searced for the TimeSpan
     * specified to find phases, amps, etc.  <p>
     *
     * The list will end at either the  distance cutoff or the station count,
     * whichever comes first. For example, assume there are 50 channels within
     * 100 km of the epicenter; if distCutoff=100 & maxCount=20, only the
     * nearest 20 channels will be included. If distCutoff=100 & maxCount=200,
     * only the 50 stations within 100 km will be include. The station list
     * comes from the channel list in * the Channel class.
     *
     * @see: defineByParameters()*/

    public MasterView (long id, double distCutoff, int maxCount, TimeSpan ts) {

	this();

	defineByParameters(id, distCutoff, maxCount, ts);

    }

    /** Create MasterView defining time window as 'preOT' seconds before origin time
     * and 'postOT' seconds after origin time. The Total window size is preOT+postOT. */
    public MasterView (Solution sol, double distCutoff, int maxCount,
		       double preOT, double postOT) {

	this();

	defineByParameters(sol, distCutoff, maxCount, preOT, postOT);

    }

    /** Create MasterView defining time window as 'preOT' seconds before origin time
     * and 'postOT' seconds after origin time. The Total window size is preOT+postOT. */
    public MasterView (long evid, double distCutoff, int maxCount,
		       double preOT, double postOT) {

	this();

	defineByParameters(evid, distCutoff, maxCount, new TimeSpan(preOT, postOT));

    }

/** Set the statusPanel where we should repor progress. */
  public void setStatusPanel (StatusPanel sp) {

       statusPanel = sp;
  }

/** Report to statusPanel */
  void report (String msg, int val) {
       if (statusPanel == null) return;

//doesn't work//       statusPanel.setProgressBarValue(val, msg);
  }
  void report (String msg) {
       report (msg, 0);   // sets status bar to min value
  }
  void report (int val) {
       if (statusPanel == null) return;
//doesn't work//           statusPanel.setProgressBarValue(val);
       statusPanel.validate();  // redundant???
  }

  /** Make a Solution from scratch and add it to the solList. Returns the
  * new solution. */
  public Solution createSolution() {

     Solution sol = Solution.create();

     addSolution(sol);

     return sol;
  }

  /** Add solution to the master view. Will not add again if it's already in the list.
  * Fires solList change event. Adds reading list change listeners. Returns true
  * if Solution was added, false if not.*/
  public boolean addSolution(Solution sol) {

	// adds sol to list in proper time order, notifies observers
	if (solList.add(sol)) {

        // add listeners to keep individual WFView lists in synch
        sol.phaseList.addChangeListener(jasiReadingListModel);
        sol.ampList.addChangeListener(jasiReadingListModel);
        sol.codaList.addChangeListener(jasiReadingListModel);
	return true;
     } else {
        return false;
     }
  }
/**
 * Remove this solution and all associated data from this MasterView. Does NOT
 * delete it from the data source. Does NOT set delete flags in the solution.
 */
public void removeSolution (Solution sol) {

    //Stop wf cache manager if this sol was selected. */
    if (sol == solList.getSelected()) {
       wfvList.stopCacheManager();
    }

    // remove listeners else you have dangling references
    sol.phaseList.removeChangeListener(jasiReadingListModel);
    sol.ampList.removeChangeListener(jasiReadingListModel);
    sol.codaList.removeChangeListener(jasiReadingListModel);

    // remove solution
    solList.remove(sol);

    // Re-make a local phase and amp lists for each WFView to reflect change.
    makeLocalLists();
}
/** Remove all solutions for the MasterView's solutionList. */
public void clearSolutionList() {

    // Stop wf cache manager if there is one running
    wfvList.stopCacheManager();

    Solution sol[] = solList.getArray();
    for (int i = 0; i<sol.length; i++) {
           // remove listeners else you have dangling references
          sol[i].phaseList.removeChangeListener(jasiReadingListModel);
          sol[i].ampList.removeChangeListener(jasiReadingListModel);
          sol[i].codaList.removeChangeListener(jasiReadingListModel);
     }

     solList = new SolutionList();

     // Re-make a local phase and amp lists for each WFView to reflect change.
     makeLocalLists();
}
  /** Return the selected solution. */
  public Solution getSelectedSolution() {
    return solList.getSelected();
  }

  /** Return the selected solution. */
  public boolean setSelectedSolution(Solution solution) {
    return solList.setSelected(solution);
  }
    /**
     * Build a MasterView based on this solution containing stations within the
     * parameter bounds and with waveforms with the given timespan. The view is
     * NOT based on waveforms saved in the DataSource. Rather Waveform objects
     * are created based on the TimeSpan given. These are effectively 'requests'
     * that may or may not be successfully fulfilled when a call to
     * loadWaveforms() is made. The DataSource will be searced for the TimeSpan
     * specified to find phases, amps, etc.  <p>
     *
     * The list will end at either the  distance cutoff or the station count,
     * whichever comes first. For example, assume there are 50 channels within
     * 100 km of the epicenter; if distCutoff=100 & maxCount=20, only the
     * nearest 20 channels will be included. If distCutoff=100 & maxCount=200,
     * only the 50 stations within 100 km will be include. The station list
     * comes from the c
     */
    public void defineByParameters(Solution sol, double distCutoff,
				               int maxCount, TimeSpan ts) {

	if (sol == null) return;

	// adds sol to list in proper time order, notifies observers
	addSolution(sol);
	setSelectedSolution(sol);

	makeWFViewList(distCutoff, maxCount, ts);

	// load Phases, Amps, * Codas
	loadReadings();

	// copy lat/lon info from Channel class to WFViews
 	loadChannelData();

	// calc distances and sort by dist from selected Solution
	distanceSort(sol);

	// Time align the waveforms by setting all their viewSpans equal to the
	// master View's timeSpan. Obviously must do AFTER the timeSpan is set
	// by all data in loop above.
	alignViews();

	// Get timeseries if flag is set
	loadWaveforms();

     }

     /**
     * Load all readings; Phases, Amps & Codas IF the loader flag of each is set
     */
     protected void loadReadings () {

	  report ("Getting Phase data.", 10);
       loadPhases();

       // load amps for magnitude of the sols
	  report ("Getting Amp data.", 20);
       loadAmplitudes();

       // load amps for magnitude of the sols
	  report ("Getting Coda data.", 30);
       loadCodas();

       // Make a local phase and amp lists for each WFView.
	  makeLocalLists();

    }



    /**
     * Build a MasterView based on this solution containing stations within the
     * parameter bounds and with waveforms with timespan defined as a window
     * 'preOT' seconds before and 'postOT' seconds after the origin time of the
     * solution. The view is NOT based on waveforms saved in the
     * DataSource. Rather Waveform objects are created based on the TimeSpan
     * given. These are effectively 'requests' that may or may not be
     * successfully fulfilled when a call to loadWaveforms() is made. The
     * DataSource will be searced for the TimeSpan specified to find phases,
     * amps, etc.  <p>
     *
     * The list will end at either the  distance cutoff or the station count,
     * whichever comes first. For example, assume there are 50 channels within
     * 100 km of the epicenter; if distCutoff=100 & maxCount=20, only the
     * nearest 20 channels will be included. If distCutoff=100 & maxCount=200,
     * only the 50 stations within 100 km will be include. The station list
     * comes from the Channel class. */
    public void defineByParameters(Solution sol, double distCutoff, int maxCount,
				   double preOT, double postOT) {

	if (sol != null ) {

	    double ot   = sol.datetime.doubleValue();
	    TimeSpan ts = new TimeSpan (ot - preOT, ot + postOT);

	    defineByParameters(sol, distCutoff, maxCount, ts);
	}
    }
    /**
     * Build a MasterView based on this solution ID containing stations within
     * the parameter bounds and with waveforms with this timespan. The view is
     * NOT based on waveforms saved in the DataSource. Rather Waveform objects
     * are created based on the TimeSpan given. These are effectively 'requests'
     * that may or may not be successfully fulfilled when a call to
     * loadWaveforms() is made. The DataSource will be searced for the TimeSpan
     * specified to find phases, amps, etc.  <p>
     *
     * The list will end at either the  distance cutoff or the station count,
     * whichever comes first. For example, assume there are 50 channels within
     * 100 km of the epicenter; if distCutoff=100 & maxCount=20, only the
     * nearest 20 channels will be included. If distCutoff=100 & maxCount=200,
     * only the 50 stations within 100 km will be include. The station list
     * comes from the Channel class. */

    public void defineByParameters(long evid, double distCutoff, int maxCount,
				   TimeSpan ts) {

	Solution sol = Solution.create().getById(evid);

	if (sol != null ) defineByParameters(sol, distCutoff, maxCount, ts);
    }


    /** Define MasterView to includ all channels with readings; phases, amps or
     * codas.
     * This is used to construct a MasterView that does
     * not depend on the waveforms windows that were assigned to this event by the
     * data source. It does NOT load time series. */

    public void defineByReadings (Solution sol,
				   double distCutoff, int maxCount,
				   TimeSpan ts) {

	if (sol == null) return;

	// adds sol to list in proper time order, notifies observers
	addSolution(sol);
	setSelectedSolution(sol);

	// Set ALL views + master view to the time window passed in the arg
	setAllViewSpans(ts);

	loadReadings();

	// create a WFView for each channel with a reading
	insureReadingsHaveViews(true);

	// match channels info
	loadChannelData();

	// calc distances and sort by dist from selected Solution
	distanceSort();

	// toss out-of-bounds views
	wfvList.trim(maxCount);
	wfvList.trim(distCutoff);

	// create waveform time windows from scratch NOT from the data source
     // (time-series are NOT loaded yet)
	WFView wfv[] = wfvList.getArray();

	for ( int i=0; i < wfv.length; i++) {

         Waveform wf = Waveform.create();
         wf.setChannelObj(wfv[i].chan);
         wf.setTimeSpan(ts);

	    wfv[i].setWaveform( wf );

	}

    }


    /** Define MasterView to includ all channels with phases.
     * This is used to construct a MasterView that does
     * not depend on the waveforms windows that were assigned to this event by the
     * data source. It does NOT load time series. */

    public void defineByPhaseList (Solution sol,
				   double distCutoff, int maxCount,
				   TimeSpan ts) {

	if (sol == null) return;

	// adds sol to list in proper time order, notifies observers
	addSolution(sol);
	setSelectedSolution(sol);

	// Set ALL views + master view to the time window passed in the arg
	setAllViewSpans(ts);

	loadPhases();

	// create a WFView for each channel with a reading
	insureReadingsHaveViews(true);

	// match channels info
	loadChannelData();

	// calc distances and sort by dist from selected Solution
	distanceSort();

	// toss out-of-bounds views
	wfvList.trim(maxCount);
	wfvList.trim(distCutoff);

	// create waveform time windows from scratch NOT from the data source
     // (time-series are NOT loaded yet)
	WFView wfv[] = wfvList.getArray();

	for ( int i=0; i < wfv.length; i++) {

         Waveform wf = Waveform.create();
         wf.setChannelObj(wfv[i].chan);
         wf.setTimeSpan(ts);

	    wfv[i].setWaveform( wf );

	}

    }


/**
 * Create a MasterView based on the Waveforms associated with this Solution in
 * the DataSource. Load the phase, amp, etc.  */
/* public boolean defineByChannelTimeWindowModel(ChannelTimeWindowModel model,
                                              double distCutoff, int maxCount) {
    if (defineByChannelTimeWindowModel(model)) {

	// toss out-of-bounds views
	wfvList.trim(maxCount);
	wfvList.trim(distCutoff);
     return true;
    }
    return false;
}
*/

/**
 * Set MasterView's default ChannelTimeWindowModel.
 * */
public void setChannelTimeWindowModel(ChannelTimeWindowModel model) {
    defaultChannelTimeWindowModel = model;
//    defaultChannelTimeWindowModel.setChannelList(MasterChannelList.get());
    // DDG 10/11/02 test
    defaultChannelTimeWindowModel.setChannelList(null);
}
/**
 * Get MasterView's default ChannelTimeWindowModel.
 * */
public ChannelTimeWindowModel getChannelTimeWindowModel() {
    return defaultChannelTimeWindowModel;
}
/**
 * Create a MasterView based the default ChannelTimeWindowModel and the currently
 * selected Solution. Returns true on success. */
public boolean defineByChannelTimeWindowModel() {
    if (this.getSelectedSolution() == null) return false;
//    defaultChannelTimeWindowModel.setChannelList(MasterChannelList.get());
    // DDG 10/11/02 test
    defaultChannelTimeWindowModel.setChannelList(null);
    return defineByChannelTimeWindowModel(defaultChannelTimeWindowModel,
			                  this.getSelectedSolution());
}
/**
 * Create a MasterView based this model and event ID number. Loads the phases, amps, etc.
 * The Solution created for this 'evid' will be added to the MasterView and set
 * as the selected Solution.
 * Returns true on success.  */
public boolean defineByChannelTimeWindowModel(ChannelTimeWindowModel model, long evid) {
    Solution sol = Solution.create();
    sol.getById(evid);
    if (sol == null) return false;
    addSolution(sol);
    setSelectedSolution(sol);
    return defineByChannelTimeWindowModel(model, sol);
}
/**
 * Create a MasterView based this model and Solution. Loads the phases, amps, etc.
 * Returns true on success. */
public boolean defineByChannelTimeWindowModel(ChannelTimeWindowModel model, Solution sol) {
    model.setSolution(sol);
    return defineByChannelTimeWindowModel(model);
}

/**
 * Create a MasterView based this model. Loads the phases, amps, etc.  Note that
 * the solution must be set for the model.
 * The currently selected solution of the MasterView will be used as the model's
 * Solution.
 * Returns true on success. */
public boolean defineByChannelTimeWindowModel(ChannelTimeWindowModel model) {

    Solution sol = getSelectedSolution();
    // Gotta' have a Solution
    if ( sol == null ) return false;

    report("Getting waveform headers.", 20);
    // Get Waveforms Header Info (time series is NOT loaded yet)

    ArrayList wfa = (ArrayList) model.getWaveformList(sol);

    if (debug) System.out.println ("Number of available waveforms = "+wfa.size());

    // Create a WFView for each waveform found for this event
    if (wfa.size() > 0) {

	report("Making Waveform views.", 0);

	// start with a clean WFViewList. Can't just abandon the old reference and
     // instatiate a new one because the Waveform listeners would still have
     // a reference and never garbage collect.
	wfvList.clear();

	// Add these WFViews to this MasterView (would getting an array be faster here?)
	int counter = 0;
	int inc = 100/wfa.size();
	int done = 0;

	for (int i = 0; i<wfa.size(); i++)  {
	   this.addWFView(new WFView( (Waveform) wfa.get(i)));

	    report (done);
	    done += inc;
	}
    } else {
      	report("No Waveform views defined.", 0);
    }

    // loads phases, amps and codas
    loadReadings();

	// create a WFView for each channel with a reading
	insureReadingsHaveViews(true);

    // copy lat/lon info from Channel class to WFViews
     report ("Loading channel data.", 0);
	loadChannelData();

	// calc distances and sort by dist from selected Solution
	distanceSort(sol);

	alignViews();

	// Get timeseries if flag is set
	loadWaveforms();

	return true;
  }


/**
 * Create a MasterView based on a ChannelTimeWindow List. */
public boolean defineByChannelTimeWindowList(Collection ctwList) {

   // Create a WFView for each waveform found for this event
    if (ctwList.size() > 0) {

     // Get Waveforms Header Info (time series is NOT loaded yet)
     ChannelTimeWindow ctw[] = (ChannelTimeWindow[]) ctwList.toArray(new ChannelTimeWindow[0]);

	report("Making Waveform views.", 0);

	// start with a clean WFViewList. Used to just make a new make a new wfvList
	// but that killed the Waveform listeners that inform when they are loaded
	// to repaint WFPanel.
	wfvList.clear();

	// Add these WFViews to this MasterView (would getting an array be faster here?)
	int counter = 0;
	int inc = 100/ctw.length;
	int done = 0;

	WFView wfv;
        TimeSpan ts;

	for (int i = 0; i<ctw.length; i++)  {

        ts = new TimeSpan(ctw[i].getStart(), ctw[i].getEnd());
        wfv = new WFView(ctw[i].getChannelObj(), ts);

//	   wfv.setWaveform(new Waveform (ctw[i].chan, ts));

         Waveform wf = Waveform.create();
         wf.setChannelObj(ctw[i].getChannelObj());
         wf.setTimeSpan(ts);

	    wfv.setWaveform( wf );

	   addWFView(wfv);

	   report (done);
	   done += inc;
	}
    }

     loadChannelData();

     // Make a local phase and amp lists for each WFView.
	makeLocalLists();

	alignViews();

	// Get timeseries if flag is set
	loadWaveforms();

	return true;
  }

/**
 * Create a MasterView based on TriggerChannelTimeWindows list. Trigger times will create
 * phases. */
public boolean defineByTriggerChannelTimeWindowList(Collection ctwList) {

   // Create a WFView for each waveform found for this event
    if (ctwList.size() > 0) {

     // Get Waveforms Header Info (time series is NOT loaded yet)
     TriggerChannelTimeWindow ctw[] =
           (TriggerChannelTimeWindow[]) ctwList.toArray(new TriggerChannelTimeWindow[0]);

	report("Making Waveform views.", 0);

	// start with a clean WFViewList. Used to just make a new make a new wfvList
	// but that killed the Waveform listeners that inform when they are loaded
	// to repaint WFPanels.
	wfvList.clear();

	// Add these WFViews to this MasterView (would getting an array be faster here?)
	int counter = 0;
	int inc = 100/ctw.length;
	int done = 0;

	WFView wfv;
        TimeSpan ts;

	for (int i = 0; i<ctw.length; i++)  {

        ts = new TimeSpan(ctw[i].getStart(), ctw[i].getEnd());
        wfv = new WFView(ctw[i].getChannelObj(), ts);

//	   wfv.setWaveform(new Waveform (ctw[i].chan, ts));
         Waveform wf = Waveform.create();
         wf.setChannelObj(ctw[i].getChannelObj());
         wf.setTimeSpan(ts);

	    wfv.setWaveform( wf );

	    addWFView(wfv);

//  Interpret the trigger time as a P-phase so a pick marker will be plotted

        if (ctw[i].triggerTime != 0.0) {
          Phase ph = Phase.create();
          ph.setChannelObj(ctw[i].getChannelObj());
          ph.setTime(ctw[i].triggerTime);
          ph.description = new PhaseDescription("T", "x", " ", 4);
          addPhaseToCurrentSolution(ph);
        }

	   report (done);
	   done += inc;
	}
    }

     loadChannelData();

     // Make a local phase and amp lists for each WFView.
	makeLocalLists();

	alignViews();

	// Get timeseries if flag is set
	loadWaveforms();

	return true;
  }



    /**
     * Build a WFViewList based containing channels within the parameters with
     * waveforms with the given timespan. The list will end at either the
     * distance cutoff or the station count, whichever comes first. For example,
     * assume there are 50 channels within 100 km of the epicenter; if
     * distCutoff=100 & maxCount=20, only the nearest 20 channels will be
     * included. If distCutoff=100 & maxCount=200, only the 50 stations within
     * 100 km will be include.  Returns the number of WFViews created.<p>
     *
     * The station list comes from the static channel
     * list in the Channel class. If it is not already loaded it will be loaded with
     * all currently active channels.
     *
@param channelList - a collection of Channel objects that is the set from which
the WFViewList will be made.
@see org.trinet.jasi.Channel()

@param distCutoff - (km) the maximum distance allowed to a station
@param maxCount - the maximum number of channels to return in the list
@param ts - the time-span from which define the WFViews and Waveforms
 */

public int makeWFViewList(double distCutoff, int maxCount, TimeSpan ts) {

    ChannelList chanList = new ChannelList();

    if (MasterChannelList.get().isEmpty()) {
// load if you must
      MasterChannelList.set(ChannelList.smartLoad());
    }

    // sort the list by distance from the hypocenter
     MasterChannelList.get().distanceSort(solList.getSelected());

//    return makeWFViewList(Channel.getList(), distCutoff, maxCount, ts);
    return makeWFViewList(MasterChannelList.get(), distCutoff, maxCount, ts);
}

    /**
     * Build a WFViewList based containing channels within the parameters with
     * waveforms with the given timespan. The list will end at either the
     * distance cutoff or the station count, whichever comes first. For example,
     * assume there are 50 channels within 100 km of the epicenter; if
     * distCutoff=100 & maxCount=20, only the nearest 20 channels will be
     * included. If distCutoff=100 & maxCount=200, only the 50 stations within
     * 100 km will be include.  Returns the number of
     * WFViews created.<p>
     *
     * You pass in a channelList from which the WFViewList will be selected.

@param channelist - a collection of Channel objects that is the set from which
the WFViewList will be made.
@see org.trinet.jasi.Channel()

@param distCutoff - (km) the maximum distance allowed to a station
@param maxCount - the maximum number of channels to return in the list
@param ts - the time-span from which define the WFViews and Waveforms

    */

public int makeWFViewList(Collection channelList, double distCutoff,
			  int maxCount, TimeSpan ts) {

    // get channel array
    Channel chan [] = new Channel[channelList.size()];
    channelList.toArray(chan);

    // create WFViews for channels within cutoff and maxCount
    WFView wfv;
    int count = 0;
    for ( int i=0; i < chan.length; i++) {

	if (chan[i].dist.doubleValue() > distCutoff) break;		// too far, bail

	wfv = new WFView( chan[i], ts );

//	wfv.setWaveform(new Waveform (chan[i], ts));
         Waveform wf = Waveform.create();
         wf.setChannelObj(chan[i]);
         wf.setTimeSpan(ts);

	    wfv.setWaveform( wf );

	addWFView(wfv);

	if (count++ >= maxCount) break;			// got enough, bail
    }

    return count;
}
/*  // I think this would cause evilness. If all lists shared references to the
// master list, a sort would scramble the secondary lists' channels!
public void matchChannelLists () {

       phaseList.matchChannelsWithList(MasterChannelList.get());
       ampList.matchChannelsWithList(MasterChannelList.get());
       codaList.matchChannelsWithList(MasterChannelList.get());
}
*/
/**
 * Create a MasterView and load all the date for a single evid in the dbase.
 * Returns 'true' if successful.
 */
    //TODO: test "error" conditions: no WF's, no phases, etc.
public boolean defineByDataSource(long evid) {
    // Get Solution

    Solution sol = Solution.create().getById(evid);

    if (sol == null) {
	if (debug) System.out.println ("No Solution matches id# :" +evid);
	return false;
    }
    this.addSolution(sol);
    this.setSelectedSolution(sol);

    return defineByChannelTimeWindowModel(getChannelTimeWindowModel());
    //TEST//return defineByDataSource(sol);

}
/**
 * Create a MasterView and load all the date for a single evid using the current model.
 * Returns 'true' if successful.
 * @see: setChannelTimeWindowModel()
 */
    //TODO: test "error" conditions: no WF's, no phases, etc.
public boolean defineByCurrentModel(long evid) {
    // Get Solution

    Solution sol = Solution.create().getById(evid);

    if (sol == null) {
	if (debug) System.out.println ("No Solution matches id# :" +evid);
	return false;
    }

    // add to MV and set selected
    this.addSolution(sol);
    this.setSelectedSolution(sol);

    return defineByChannelTimeWindowModel(getChannelTimeWindowModel());
    //TEST//return defineByDataSource(sol);

}
/**
 * Create a MasterView based on the Waveforms associated with this Solution in
 * the DataSource. Load the phase, amp, etc.  */
public boolean defineByDataSource(Solution sol) {

    // TODO: read in other event that fall in the window

    if (sol == null) return false;

    // adds sol to list in proper time order, notifies observers
    addSolution(sol);
    solList.setSelected(sol);

    report("Getting waveform headers.", 20);

// [WAVEFORMS] Get Waveforms Header Info (time series is NOT loaded yet)
    ArrayList wfa = (ArrayList) Waveform.create().getBySolution(sol);

    if (debug) System.out.println ("Number of available waveforms = "+wfa.size());

    // Create a WFView for each waveform found for this event
    if (wfa.size() > 0) {

	report("Making Waveform views.", 0);

	// start with a clean WFViewList. Used to just make a new make a new wfvList
	// but that killed the Waveform listeners that inform when they are loaded
	// to repaint WFPanels.
	wfvList.clear();

	// Add these WFViews to this MasterView (would getting an array be faster here?)
	int counter = 0;
	int inc = 100/wfa.size();
	int done = 0;

	for (int i = 0; i<wfa.size(); i++)  {

	    this.addWFView(new WFView( (Waveform) wfa.get(i)));

	    report (done);
	    done += inc;
	}
    }

    // load amps, phases, codas
    report("Loading parameters...", 0);
    loadReadings();

    // Make sure TimeSpan of WFViews are big enough to show phases.
    // The 'true' means create WFViews for any phases (channels) that don't have them.
    // There can be phases with no waveforms.

    insureReadingsHaveViews(true);

    // copy lat/lon info from Channel class to WFViews
    report ("Loading channel data.", 0);
    loadChannelData();

	// calc distances and sort by dist from selected Solution
	distanceSort(sol);

// [TIMESERIES] Get timeseries if flag is set, needs wfviews and sort
//  else wf's won't load in sorted order
	loadWaveforms();

	alignViews();

	return true;
  }

    /**
     * Load the Channel object of each WFView with location and response info.
     * If there is a static Channel list in memory use that. If not, look up the
     * info  for each channel in the WFView list at the DataSource (dbase).
     */
    public void loadChannelData() {

	int efficencyCrossover = 100;	// if more then this many channels, load all
//	int efficencyCrossover = 0;	// disable per-channel reading of channels

	if (!loadChannelData) return;

	// bail if no WFViews
	if (wfvList.size() == 0) return;

	WFView wfv[] = wfvList.getArray();
	int done = 0;
	int inc = 100/wfvList.size();

	// its more efficient to load the whole list if there are more then 'n'
	// channels. Takes about 20 seconds to load current list.
	if (MasterChannelList.isEmpty() && wfvList.size() > efficencyCrossover) {
            MasterChannelList.set(ChannelList.readCurrentList());
        }

	// channel info is already loaded
	if (!MasterChannelList.isEmpty()) {

        Channel ch[] = MasterChannelList.get().getArray();

	    if (debug) System.out.println ("Matching channel info to each WFView...");
	    report ("Matching channel info to each WFView...");

	    for (int i = 0; i<wfv.length; i++)  {
		// find and copy latlonz info to Channel object in each WFView
		// if its in the channel list (ch)
		if (wfv[i].copyChannel(ch)) {
                  // also point the Waveform at it
                  if (wfv[i].wf != null) wfv[i].wf.setChannelObj(wfv[i].chan);
                } else {
		  if (debug) System.out.println ("No Match: "+wfv[i].chan.toString());
		  // Try individual channel lookup -- channel may not be in "current" list
		  wfv[i].chan = Channel.lookUp(wfv[i].chan);
                  MasterChannelList.get().add(wfv[i].chan); // add to channel list
                  // also point the Waveform at it
                  if (wfv[i].wf != null) wfv[i].wf.setChannelObj(wfv[i].chan);
                }
              report (done);
              done += inc;
	    }
	} else {		// no list loaded, look up each in dbase

         // we will be making a channel list as we go
         if (MasterChannelList.get() == null)
                  MasterChannelList.set(new ChannelList());

	    BenchMark bm = new BenchMark();
	    if (debug)  System.out.println ("Looking up channel info for each WFView...");
	    report ("Looking up channel info.");

	    // copy Channel info for each WFView
	    for (int i = 0; i<wfv.length; i++)  {

		wfv[i].chan = Channel.lookUp(wfv[i].chan);

                MasterChannelList.get().add(wfv[i].chan); // add to channel list

                // also point the Waveform at it
                if (wfv[i].wf != null) wfv[i].wf.setChannelObj(wfv[i].chan);

                report (done);
                done += inc;
	    }
	    if (debug) bm.print("Looked up: "+ wfv.length+" channels in ");
	}

    }

    /**
     * Make a local phase and amp lists for each WFView. This speeds some functions.
     */
    public void makeLocalLists () {

	WFView wfv[] = wfvList.getArray();

	for (int i = 0; i<wfv.length; i++)  {
	    wfv[i].updatePhaseList();
	    wfv[i].updateAmpList();
         wfv[i].updateCodaList();
     }
    }
/** Scan the reading lists for each solution and make sure there is a WFView
 for each reading and that it has a TimeSpan big enough to
include the reading. This insures that each WFView's viewSpan is
big enough to display its readings. Otherwise, they could land before or after available
waveform data. If 'createNewWFViews' is true, create a new WFView for any reading
with no matching WFView.  Thus, there can be phases, amps, etc with no waveforms.<p>
*
* If the masterView's timeSpan is not 'null' the Channel WFView's TimeSpan is
* set to that value. */

void insureReadingsHaveViews(boolean createNewWFViews) {

    Solution sol[] = solList.getArray();
    for (int i = 0; i<sol.length; i++) {
       insureReadingsHaveViews(sol[i], sol[i].phaseList, createNewWFViews);
       insureReadingsHaveViews(sol[i], sol[i].ampList,   createNewWFViews);
       insureReadingsHaveViews(sol[i], sol[i].codaList,  createNewWFViews);
    }
    makeLocalLists();
}
/** Scan this reading list and make sure there is a WFView
 for each reading and that it has a TimeSpan big enough to
include the readin. This insures that each WFView's viewSpan is
big enough to display its readings. Otherwise, they could land before or after available
waveform data. If 'createNewWFViews' is true, create a new WFView for any reading
with no matching WFView.  Thus, there can be phases, amps, etc with no waveforms.<p>
*
* If the masterView's timeSpan is not 'null' the Channel WFView's TimeSpan is
* set to that value. */
void insureReadingsHaveViews(Solution solution, JasiReadingList list, boolean createNewWFViews) {

     JasiReading[] jr = (JasiReading[]) list.toArray(new JasiReading[0]);
     WFView wfv;
     boolean match = false;
     final double minWindowSize = 10.0; // seconds
     final double halfWindow = minWindowSize/2.0;

     for (int i = 1; i < jr.length; i++) {

        wfv = wfvList.get(jr[i].getChannelObj());

        if (wfv != null) {    // is there already a matching view?
		wfv.viewSpanInclude(jr[i].getTime()); // make sure it includes reading time
        } else {
 //         if (false) {
         if (createNewWFViews) {
	              // show at least minWindowSize
 	       TimeSpan ts = new TimeSpan (jr[i].getTime() - halfWindow,
					   jr[i].getTime() + halfWindow);
//	       this.addWFView(new WFView(jr[i], getViewSpan() ));  // set viewspan to the masterview's
	       wfv = new WFView(jr[i], ts );
	       wfv.viewSpan.set(getViewSpan().getStart(), getViewSpan().getEnd());
	       wfv.dataSpan.set(jr[i].getTime(), jr[i].getTime());
	       this.addWFView(new WFView(jr[i], ts ));

          }
        }
     }
}
/** Scan the phase list and make sure each WFView has a TimeSpan big enough to
include any reading for that channel. This insures that each WFView's viewSpan is
big enough to include its phases. They could land before or after available
waveform data. If 'createNewWFViews' is true, create a new WFView for any phase
with no matching WFView.  Thus, there can be phases with no waveforms.<p>
*
* If timeSpan is not 'null' the WFView's TimeSpan is set to that value. */
/*
void insurePhasesHaveViews(PhaseList phaseList, boolean createNewWFViews) {

    Phase  ph[]  = (Phase[]) phaseList.getArray();
    WFView wfv[] = wfvList.getArray();
    boolean match = false;
    Channel chan;

    // a phase-only WFView needs a window length which is phase time +- 5 sec.
    TimeSpan ts = new TimeSpan();
    double phTime = 0.0;

    for (int i = 0; i < ph.length; i++)  {

	chan = ph[i].chan;	// for clarity

	// calc the minimum window around a phase (+- 5 seconds)

	phTime = ph[i].datetime.doubleValue();
	ts = new TimeSpan (phTime - 5.0, phTime + 5.0);

	match = false;

	for (int j = 0; j < wfv.length; j++)  {

	    // must use 'sameAs()' not 'equals()' because phases don't have
	    // SeedChan, Location, etc.
	    if (chan.sameAs(wfv[j].chan)) {

		// make sure WFView TimeSpan includes phase
		wfv[j].viewSpanInclude(ts);
		match = true;
		break;		// stop looking
	    }
	}

	if (!match && createNewWFViews) {

	    this.addWFView(new WFView(chan, ts ));

	    // must refresh array to reflect NEW list with this addition
	    wfv = wfvList.getArray();
	}

    }

}
*/

/**
 * Load the amplitudes for this MasterView. Loads all amps associated with
 * the preferred magnitude of all solutions in view.
 */
 /* This should be done for a time window but they've fucked up the dbase by
 * writing multiple RT system's data into it with no indication of what is primary.
 * I am ignoring "network amplitudes" for now.
 */

public void loadAmplitudes() {

    if (ampRetrieveFlag) {

       Solution sol[] = solList.getArray();
       for (int i = 0; i<sol.length; i++) {

           // add amps to the sol's mag's list...
           sol[i].addAmps (Amplitude.create().getByMagnitude(sol[i].magnitude) );
       }
    }
}
/**
 * Load the codas for this MasterView. Loads all codas associated with
 * the preferred magnitude of all solutions in view.
 */

public void loadCodas() {

    if (codaRetrieveFlag) {

       Solution sol[] = solList.getArray();
       for (int i = 0; i<sol.length; i++) {
          sol[i].addCodas( Coda.create().getByMagnitude(sol[i].magnitude) );
       }
    }
}
/**
 * Load the phases for this MasterView. Connect a list to each solution
 * and add all phases to the MasterView's list.
 */

public void loadPhases() {

    if (phaseRetrieveFlag) {

       Solution sol[] = solList.getArray();
       for (int i = 0; i<sol.length; i++) {

           sol[i].addPhases( Phase.create().getBySolution(sol[i]) );

//           phaseList.addAll( sol[i].phaseList );

       }
    }
}

/**
 * Load the waveforms for this MasterView using the mode defined by
 * MasterView.waveformLoadMode.  */
public void loadWaveforms() {
    // Get timeseries if flag is set

    System.out.println ("MV: loadWaveforms mode= "+waveformLoadMode);

    if (waveformLoadMode == LoadAllInForeground) {

	wfvList.loadWaveformsNow();

    } else if (waveformLoadMode == LoadAllInBackground) {

	wfvList.loadWaveformsInBackground();

    } else if (waveformLoadMode == Cache) {

	// start the CacheMgr
	System.out.println ("MV: loadWaveformsInCache ");
	wfvList.loadWaveformsInCache();
    }
}

/**
 * Return Collection (ArrayList) of waveforms in this masterView. Not all
 * WFViews will necessarily have a waveform so you can't assume the returned
 * collection will have the same size or order as the WFView list.
 * Returns null if there are no
 * waveforms. */
public Collection getWaveformList () {

    WFView wfv[] = wfvList.getArray();

    ArrayList wfa = new ArrayList();

    for (int i = 0; i<wfv.length; i++)  {
	if (wfv[i].wf != null) wfa.add(wfv[i].wf);
    }

    return wfa;

}

public int getWFSegmentCount() {
    WFView wfv[] = wfvList.getArray();

    Waveform wf[] = new Waveform[wfv.length];
    int knt = 0;
    for (int i = 0; i<wfv.length; i++)  {
        knt = wfv[i].wf.getSegmentCount();
    }

    return knt;
}
/**
 * Return array of waveforms in this masterView. Not all
 * WFViews will necessarily have a waveform. Returns empty array if there are no
 * waveforms. */
public Waveform[] getWaveformArray () {

    WFView wfv[] = wfvList.getArray();

    Waveform wf[] = new Waveform[wfv.length];

    for (int i = 0; i<wfv.length; i++)  {
	  wf[i] = wfv[i].wf;
    }

    return wf;

}

/**
 * Add a WFView to the masterView
 */
public void addWFView (WFView wfv) {

    wfvList.add(wfv);

    this.timeSpan.include(wfv.viewSpan);

}

/** Associate this phase with the currently selected Solution and add the phase
to the Solutions's phase list and the MasterView's phase list. Enforces the rule that
only one phase of each type is allowed in each solution. */
public void addPhaseToCurrentSolution(Phase ph) {

       Solution sol = getSelectedSolution();

         // associate phase with current solution
         ph.associate(sol);

         // add it to the Solution phase list
	    Phase phres = sol.phaseList.addOrReplacePhase(ph);

         // if the method returns the original phase that phase was ADDED
         // to the list, otherwise it replaced and existing phase.
/*         if (phres ==  ph)
         // add it to the MasterView phase list
         phaseList.addOrReplacePhase(ph);
*/

}

/**
 * Return the number of WFView in this MasterView
 */
public int getWFViewCount() {
    return wfvList.size();
}
public TimeSpan getViewSpan () {
    return timeSpan;
}

public PhaseList getAllPhases() {
    PhaseList bigList = new PhaseList();
    Solution sol[] = solList.getArray();
    for (int i = 0; i<sol.length; i++) {
       bigList.addAll(sol[i].phaseList);
    }
    return bigList;
}
/** Return the count of phases in ALL solutions. */
public int getPhaseCount() {
    int knt = 0;
    Solution sol[] = solList.getArray();
    for (int i = 0; i<sol.length; i++) {
       knt += sol[i].phaseList.size();
    }
    return knt;
}
public AmpList getAllAmps() {
    AmpList bigList = new AmpList();
    Solution sol[] = solList.getArray();
    for (int i = 0; i<sol.length; i++) {
       bigList.addAll(sol[i].ampList);
    }
    return bigList;
}
/** Return the count of amps in ALL solutions. */
public int getAmpCount() {
    int knt = 0;
    Solution sol[] = solList.getArray();
    for (int i = 0; i<sol.length; i++) {
       knt += sol[i].ampList.size();
    }
    return knt;
}
public CodaList getAllCodas() {
    CodaList bigList = new CodaList();
    Solution sol[] = solList.getArray();
    for (int i = 0; i<sol.length; i++) {
       bigList.addAll(sol[i].codaList);
    }
    return bigList;
}
/** Return the count of codas in ALL solutions. */
public int getCodaCount() {
    int knt = 0;
    Solution sol[] = solList.getArray();
    for (int i = 0; i<sol.length; i++) {
       knt += sol[i].codaList.size();
    }
    return knt;
}
/** If arg is true, only WFViews with phases will be included in the master view. */
public void includeOnlyWithPhases (boolean tf) {

    includeOnlyWithPhases = tf;

}
/** Returns true if only WFViews with phases will be included in the master view. */
public boolean getIncludeOnlyWithPhases () {
    return includeOnlyWithPhases;
}

/** Set the alignment mode. Currently only time alignment is supported. Returns
* false if an invalid mode is specified. */
    public boolean setAlignmentMode (int mode) {
      if (mode > -1 && mode < AlignmentModeCount) {
         alignmentMode = mode;
         return true;
      } else {
         return false;
      }
    }

    /** Return the alignment mode. */
    public int getAlignmentMode() {
      return  alignmentMode;
    }

    /** Align views on whatever is defined by getAlignmentMode() */
    public void alignViews() {
       if (getAlignmentMode() == AlignOnTime) {
         timeAlignViews();
       } /* else if (getAlignmentMode() == AlignOnP) {
         pAlignViews();
       } else if (getAlignmentMode() == AlignOnS) {
         sAlignViews();
       }  */
    }
/**
 * Time align the WFViews by setting all the viewSpans equal to the
 * MasterView's viewSpan
 */
    public void timeAlignViews () {

	WFView wfv[] = wfvList.getArray();

	for (int i = 0; i<wfv.length; i++)  {
	    wfv[i].setViewSpan(this.timeSpan);
	}
    }
/**
 * Align the WFViews on the P-arrival by adding a move-out to all the viewSpans
 */
 ////
 //// THIS WAS NEVER REALLY WORKING CORRECTLY
/*
    public void pAlignViews () {

	WFView wfv[] = wfvList.getArray();

     if (wfv.length < 1) return;

	TravelTime tt = new TravelTime();

     // OT - 10 sec
     double ot = getSelectedSolution().datetime.doubleValue() - 10.0;
     double baseTime = wfvList.getEarliestSampleTime();
     double offset;
     // add P=wave travetime to each
	for (int i = 0; i<wfv.length; i++)  {
         offset = wfv[i].getViewSpan().getStart() - baseTime + tt.getTTp(wfv[i].dist);
         wfv[i].getViewSpan().setStart(baseTime + tt.getTTp(wfv[i].dist));
	}
    }
*/
/**
 * Align the WFViews on the S-arrival by adding a move-out to all the viewSpans
 */
/*
    public void sAlignViews () {

	WFView wfv[] = wfvList.getArray();

     if (wfv.length < 1) return;

	TravelTime tt = new TravelTime();

     // OT - 10 sec
     double baseTime = getSelectedSolution().datetime.doubleValue() - 10.0;

     // add P=wave travetime to each
	for (int i = 0; i<wfv.length; i++)  {
         wfv[i].getViewSpan().setStart(baseTime + tt.getTTs(wfv[i].dist));
	}
    }
*/
/**
 * Set the viewSpan of the MasterView and ALL WFViews to this timeSpan.
 */
    public void setAllViewSpans (TimeSpan ts) {

	this.timeSpan = ts;

	alignViews();
    }

/**
 * Set flag to time align the WFViews by setting all the viewSpans equal to
 * the MasterView's viewSpan
 */
/*    public void setTimeAlign (boolean tf) {

	timeAlign = tf;
    }
*/
/**
 * Set max number of waveforms to load. Used to avoid memory overflow
 */
    public void setWaveformLoadLimit (int limit) {

	waveformLoadLimit = limit;
    }

/**
 * Return max number of waveforms to load. Used to avoid memory overflow
 */
    public int getWaveformLoadLimit () {

	return waveformLoadLimit;
    }

/**
 * Sort/resort the WFView list by distance from the current selected Solution.
 */
    public void distanceSort() {
	    distanceSort(solList.getSelected());
    }

/**
 * Sort/resort the WFView list by distance from the given Solution.
*/
public void distanceSort(Solution sol) {
	distanceSort(sol.getLatLonZ());
}

/**
 * Sort/resort the WFView list by distance from the given LatLonZ.
*/
public void distanceSort (LatLonZ latlonz) {
	wfvList.distanceSort(latlonz);
     if (getSelectedSolution() != null)
         getSelectedSolution().sortReadingLists();
}
/**
* Delete all Amps associated with this solution that are greater
* than or equal to this dist.  Returns the number of deleted items.
*/
  public int stripAmpsByDistance(double cutDist, Solution sol) {

     int knt = 0;

     Amplitude am[] = sol.ampList.getArray();

     for (int i = 0;i<am.length; i++)  {
        // must delete from Solutions's list *NOT* WFView's list because
        // the MV's list is Active and will notify observers of change
        // This, in turn, will update the WFView.phaseLists
        if (am[i].getDistance() > cutDist) {
           sol.deleteAmplitude(am[i]);
           knt++;
        }
     }

     return knt;
  }
/**
* Delete all Codas associated with this solution that are greater
* than or equal to this dist.  Returns the number of deleted items.
*/
  public int stripCodasByDistance(double cutDist, Solution sol) {

     int knt = 0;

     Coda coda[] = sol.codaList.getArray();

     for (int i = 0;i<coda.length; i++)  {
        // must delete from Solutions's list *NOT* WFView's list because
        // the MV's list is Active and will notify observers of change
        // This, in turn, will update the WFView.phaseLists
        if (coda[i].getDistance() > cutDist) {
           sol.deleteCoda(coda[i]);
           knt++;
        }
     }

     return knt;
  }
/**
* Delete all phases associated with this solution that are in WFViews greater
* than or equal to this dist. (This function is here rather than in PhaseList or
* Solution because the WFViews are the only components garenteed to have the
* distance set) . Returns the number of deleted phases.
*/
  public int stripPhasesByDistance(double cutDist, Solution sol) {

     int knt = 0;

     Phase ph[] = sol.phaseList.getArray();

     for (int i = 0;i<ph.length; i++)  {
        // must delete from Soltions's list *NOT* WFView's list because
        // the MV's list is Active and will notify observers of change
        // This, in turn, will update the WFView.phaseLists
        if (ph[i].getDistance() > cutDist) {
           sol.deletePhase(ph[i]);
           knt++;
        }
     }

     return knt;
  }
  public void setResidualStripValue(double value) {
    residualStripValue = value;
  }

  public double getResidualStripValue() {
    return residualStripValue;
  }

  public void setClockQualityThreshold (double value) {
    clockQualityThreshold = value;
  }

  public double getClockQualityThreshold() {
    return clockQualityThreshold;
  }

/**
* Delete all phases associated with the selected solution that have a residual greater
* than or equal to the value set by setResidualStripValue().
* Returns the number of deleted phases.
*/
  public int stripPhasesByResidual() {
     return  stripPhasesByResidual(getSelectedSolution());
  }
/**
* Delete all phases associated with this solution that have a residual greater
* than or equal to the value set by setResidualStripValue().
* Returns the number of deleted phases.
*/
  public int stripPhasesByResidual(Solution sol) {
     return  stripPhasesByResidual(getResidualStripValue(), sol);
  }
/**
* Delete all phases associated with this solution that have a residual greater
* than or equal to this value. Returns the number of deleted phases.
*/
  public int stripPhasesByResidual(double val, Solution sol) {

     return sol.phaseList.stripByResidual(val, sol);
 /*
     int knt = 0;

     System.out.println ("Stripping by residule = "+val+"\n"+sol.toString());

     Phase ph[] = sol.phaseList.getArray();

	for (int i = 0; i<ph.length; i++)  {
         if (ph[i].residual.doubleValue() >= val) {

             // see deletePhase() for more info
             sol.phaseList.delete(ph[i]);
             System.out.println ("Deleting: "+ph[i].toString());
         }
	}
     return knt;
 */
  }
  /** This is here because we must delete phases from two lists; the MasterView's and
  * the Solution's. These should be coordinated in a more clever way but that's for
  * the TODO list.... The MasterView's list is Active and will notify observers of change
  * This, in turn, will update the WFView.phaseLists. */
/*
  public void deletePhase (Phase ph) {
//      ph.sol.phaseList.delete(ph);       // just sets the delete flag doens't remove from list
      this.phaseList.delete(ph);
  }
*/
/** Locate the given event. The PropertyList must have values for:
 * locationEngineAddress & locationEnginePort which identify a socket based
 * location engine. @See: org.trinet.jiggle.LocationEngine */

public boolean locate (GenericPropertyList props, Solution sol) {

    String locEngAddr = props.getProperty("locationEngineAddress");
    int locEngPort    = props.getInt("locationEnginePort");

    LocationEngine locEng = LocationEngine.CreateLocationEngine(props.getProperty("LocationEngine",
	                                                    "org.trinet.jiggle.LocalEngineHypoInverse"));
    locEng.setServer(locEngAddr, locEngPort);


    return locEng.solve(sol, sol.phaseList);
}

/**
 * Locate the selected event.
 */
public boolean locate (GenericPropertyList props) {

    return locate (props, solList.getSelected());
}

/**
 * Create a new masterView containing only the first 'maxViews' components in 'mv'.
 * This will generally be used to create a more managable MasterView for events
 * with lots of components. If the original MasterView has 'maxViews' WFViews or
 * fewer it is returned without modification.
 */

public static MasterView subset (MasterView mv, int maxViews) {

    if (mv.getWFViewCount() <= maxViews) return mv;

    MasterView mvNew = new MasterView();

    // copy the solution list
    mvNew.solList = new SolutionList(mv.solList);

    WFView wfv[] = mv.wfvList.getArray();

    // select the proper WFView's
    for ( int i=0; i < wfv.length; i++)
	{
	    mvNew.addWFView(wfv[i]);
	    if (i == maxViews-1) break;		// got enough
	}

	mvNew.alignViews();

	return mvNew;
}

/**
 * Create a new masterView containing only vertical components. It will contain no more
 * then 'maxViews' WFViews.
 */
// Is there a more general way to do this sort of thing?

public static MasterView vertOnly(MasterView mv, int maxViews) {

    MasterView mvNew = new MasterView();

    // copy the solution list
//    mvNew.solList = new ActiveSolutionList(mv.solList);
    mvNew.solList = new SolutionList(mv.solList);

    WFView wfv[] = mv.wfvList.getArray();

    // select the proper WFView's
    for ( int i=0; i < wfv.length; i++)
	{
	    if (wfv[i].chan.isVertical()) mvNew.addWFView(wfv[i]);
	    if (i == maxViews) break;		// got enough
	}

	mvNew.alignViews();

	return mvNew;
}
/**
 * Create a new masterView containing only components that have phase picks.
 */
public static MasterView withPhasesOnly(MasterView mv) {

    return withPhasesOnly (mv, Integer.MAX_VALUE);
}

/**
 * Create a new masterView containing only components that have phase picks.
 * It will contain no more then 'maxViews' WFViews.
 */
// Is there a more general way to do this sort of thing?

public static MasterView withPhasesOnly(MasterView mv, int maxViews) {

    MasterView mvNew = new MasterView();

    // copy the solution list
    mvNew.solList = new SolutionList(mv.solList);

    WFView wfv[] = mv.wfvList.getArray();

    // select the proper WFView's
    for ( int i=0; i < wfv.length; i++)	{
	    if (wfv[i].hasPhases()) mvNew.addWFView(wfv[i]);
	    if (i == maxViews) break;		// got enough
	}

	mvNew.alignViews();

	return mvNew;
}

/**
 * Calculate the true and horizontal distance from location in the argument for
 * each site in the list. Requires that loadChannelData be called first.
 */

public void calcDistances (LatLonZ loc) {
    wfvList.calcDistances(loc);
}


/**
 * Calculate the distance from the given location
 * Requires that loadChannelData be called first.
 */

public void calcDistances (float lat, float lon, float z)
{
    calcDistances(new LatLonZ (lat, lon, z));
}

/**
 * Calculate the distance from the location of the selected Solution.
 * Requires that loadChannelData be called first.
 */

public void calcDistances ()
{
    calcDistances(solList.getSelected().getLatLonZ());
}

/**
 * If 'true' waveforms will be retreived. Deprecated.
 */
 /*
public void setWfRetrieveFlag (boolean tf)
{
    if (tf) {
	waveformLoadMode = LoadAllInForeground;
    } else {
	waveformLoadMode = LoadNone;
    }
}
*/
/**
 * Depricated
 */
 /*
public boolean getWfRetrieveFlag ()
{
    return wfRetrieveFlag;
}
*/
//* Set the waveform loading method. Returns 'false' if the mode is invalid.
public boolean setWaveFormLoadMode (int mode) {

    // check for valid mode type
    if (mode > ModeCount) return false;

    waveformLoadMode = mode;

    if (waveformLoadMode == Cache) {
	wfvList.setCachingEnabled(true);
    }

    return true;
}

public int getWaveFormLoadMode () {
    return waveformLoadMode;
}

/** Pass-thru method */
public void setCacheSize (int above, int below) {
    if (getWaveFormLoadMode() == Cache) {
	wfvList.setCacheSize(above, below);
    }
}

public void setLoadChannelData(boolean tf) {
    loadChannelData = tf;
}

public boolean getLoadChannelData() {
    return loadChannelData;
}


/** Cleanup stuff before this mv is destroyed. Otherwise, dangling references
* cause a memory leak. Stops the cache manager. */
public void destroy() {

       solList.clearChangeListeners();

       masterWFViewModel.clearChangeListeners();
       masterWFWindowModel.clearChangeListeners();

       wfvList.stopCacheManager();
       wfvList.clearChangeListeners();

       // remove solutions and delete listeners
       clearSolutionList();
}

/**
 * Dump some info about the views for debugging
 */
public void dump () {
    System.out.println (dumpToString());
}

/** Returns a string containing a line describing each solution followed
* by a list of amplitudes associated with that solution. */
public String ampsToString() {
       String str="";
       Solution sol[] = solList.getArray();

          for (int i = 0; i<sol.length; i++) {
             str += sol[i].toString() + "\n";
             if (sol[i].magnitude == null) {
                str += " * No magnitude for this solution *\n";
             } else {
                str += sol[i].magnitude.toDumpString() + "\n";
//                str += sol[i].magnitude.ampList.dumpToString()+"\n\n";
             }
             str += sol[i].ampList.dumpToString()+"\n\n";
          }
       return str;

     }

/** Returns a string containing a line describing each solution followed
* by a list of phases associated with that solution. */
public String phasesToString() {

       String str="";
       Solution sol[] = solList.getArray();
       for (int i = 0; i<sol.length; i++) {
          str += sol[i].toString() + "\n";
          // This is not just the raw phaseList to weed out virtually deleted events
          str += sol[i].phaseList.getAssociated(sol[i]).dumpToArcString()+"\n\n";
//          str += sol[i].phaseList.dumpToArcString()+"\n\n";
       }
       return str;
}



public String dumpToString()
{
    String str;

    str = solList.size() + " origins\n";
    str += solList.dumpToString() + "\n";

    str += wfvList.size() + " waveforms\n";

    // show info about WFViews
    WFView wfv[] = wfvList.getArray();

    for ( int i=0; i < wfv.length; i++)
	{
	    str += "\n";
	    str += wfv[i].dumpToString()+"\n";
	    if (wfv[i].wf == null) {
		str += " (no waveform)";
	    } else {
		str += wfv[i].wf.toString() +"\n";
	    }
	}

    return str;
}

/**
 * Dump the MasterView to the given JTextArea
*/
public void dumpToJTextArea(JTextArea jtext)
{
    jtext.setText(dumpToString());
}



/// --------------------------------------------------------------------------
/**
 * Main for testing
 */

    public static void main (String args[])
    {
	int evid;

	if (args.length <= 0)	// no args
	{
	  System.out.println ("Usage: java MasterView [evid])");

	  evid = 9723237;

	  System.out.println ("Using evid "+ evid+" as a test...");

	} else {

	  Integer val = Integer.valueOf(args[0]);    // convert arg String to 'double'
	  evid = (int) val.intValue();
	}

        System.out.println ("Making connection...");
	DataSource db = new TestDataSource();    // make connection
        System.out.println ("DataSource = "+db.toString());

	System.out.println ("Reading in current channel info...");
        String compList[] = {"EH_", "HH_", "HL_", "AS_"};
    	ChannelList chanList = ChannelList.getByComponent(compList);
	MasterChannelList.set(chanList);

///	DataSourceChannelTimeModel model = new DataSourceChannelTimeModel();

	Solution sol    = Solution.create().getById(evid);
//	model.setSolution(sol);
	System.out.println (sol.toSummaryString());
        System.out.println ("Making MasterView for evid = "+evid);

      	MasterView mv = new MasterView();

	mv.addSolution(sol);
	mv.setSelectedSolution(sol);
        mv.setWaveFormLoadMode(MasterView.LoadNone);
	mv.defineByChannelTimeWindowModel();

	mv.dump();
//	mv.phaseList.dump();
        System.out.println (mv.phasesToString());
     /*
	Solution sol[] = mv.solList.getArray();
	PhaseList pl = mv.phaseList.getAssociated(sol[0]);
	pl.dump();
       */
	while (mv.wfvList.wfLoaderThread.thread.isAlive()) {

	    System.out.println("Waiting for wfloader to finish...");

	    try{
		Thread.sleep(1000);
	    } catch (InterruptedException e){}

	    // test killing loader thread
	    System.out.println("### Killing loader thread...");
	    mv.wfvList.stopLoadWaveformsInBackground();
	}


    }
}   // end of class

