package org.trinet.jasi.coda.TN;

import java.util.*;

import org.trinet.jasi.coda.*;
import org.trinet.jasi.*;
import org.trinet.util.*;
import org.trinet.jasi.TN.*;
/** Base class for calculating time series coda data and get results in a CodaTN object.
* Concrete subclasses set parameters and override default methods.
*/
public abstract class CodaGeneratorTN implements CodaGeneratorIF {

    public static final String  LOW_AMP_CODA_DESC          = "lowStartAmp";
    public static final String  BELOW_CUTOFF_AMP_CODA_DESC = "cutoffAmp";
    public static final String  COUNT_CODA_DESC            = "count";
    public static final String  TRUNCATED_CODA_DESC        = "truncated";
    public static final String  NOISY_CODA_DESC            = "noisy";
    public static final String  HIGH_AMP_CODA_DESC         = "highAmp";

    public static final String  UNITS                      = "c";

    protected static boolean       debug = false;

    //protected Concat        fmt;                   // string buffer formater for output
    protected FilterIF      timeSeriesFilter;
    protected boolean       filterEnabled = false;

    protected String        algorithm;            // output to coda object
    protected int           goodWindowCount;      // output to coda object or total windows?
    protected double        tauFinal;             // output to coda object
    protected double        aveAbsAmpAtEndOfCoda; // output to coda object good or not windows
    protected double        qFree;                // output to coda object
    protected double        aFree;                // output to coda object
    protected double        aFix;                 // output to coda object
    protected double        rms;                  // output to coda object
    protected double        quality;              // output to coda object
    protected String        codaDesc;             // output to coda object
    protected String        codaPhase = "S";      // output to coda object

    protected int     firstGoodWindow;       // optional output
    protected double  firstWindowAveAbsAmp;  // optional output
    protected int     badWindowCount;        // optional output
    protected int     exitStatus;            // output via method
    protected String  exitStatusString;

    protected double[] goodWindowAmp;        // output to coda object
    protected double[] goodWindowTau;        // output to coda object

    protected double[] windowWeight;
    protected double[] log10WindowTime; // shared buffer used to cache window log10 time
    protected double[] log10WindowAmp;  // shared buffer used to cache window log10 amps

    protected int      noiseBiasSamples;
    protected int      maxGoodWindowCount;
    protected int      minGoodWindowsToEndCalcOnClipping;

    protected double   codaStartSignalToNoiseRatio;
    protected double   codaCutoffSignalToNoiseRatio;
    protected double   passThruNoiseDecayAmpRatio;

    protected double   qFix;

    protected boolean  resetOnClipping;

    protected double  windowSize;                // shared set from method input
    protected double  secsPerSample;             // shared set from method input
    protected double  secsPastPTimeAtStartingIdx;// shared set from method input
    protected int     lastSampleIdxInTimeSeries; // shared set from method input
    protected double  minCodaStartingAmp;        // shared set from method input
    protected double  maxCodaNoiseCutoffAmp;     // shared set from method input

    protected int     windowCount;               // shared calc counter could be output
    protected boolean isFirstWindow;             // shared calc flag
    protected double  aveAbsAmpOfLastGoodWindow; // shared calc value

    protected CodaGeneratorTN() {}

    public static void setDebug(boolean value) {
        debug = value;
    }

    protected void initBuffers(int maxCount) {
        this.maxGoodWindowCount = maxCount;
        windowWeight     = new double[maxCount];
        goodWindowTau    = new double[maxCount]; // output
        log10WindowTime  = new double[maxCount]; // shared buffer used to cache window log10 time
        goodWindowAmp    = new double[maxCount]; // output to coda object
        log10WindowAmp   = new double[maxCount]; // shared buffer used to cache window log10 amps
    }

    private void initOutputData() {

        codaDesc                  = "";
        qFree                     = 0.;
        aFree                     = 0.;
        aFix                      = 0.;
        rms                       = 0.;
        tauFinal                  = 0.;
        aveAbsAmpAtEndOfCoda      = 1.;
        firstWindowAveAbsAmp      = 1.;

        firstGoodWindow           = 0;
        badWindowCount            = 0;
        windowCount               = 0;
        goodWindowCount           = 0;
        quality                   = 0.;
        exitStatus                = 0;
    }

    protected TimeAmp [] getGoodWindowTimeAmps() {
        TimeAmp [] timeAmps = new TimeAmp [goodWindowCount];
        for (int idx = 0; idx < goodWindowCount; idx++) {
            timeAmps[idx] = new TimeAmp(goodWindowTau[idx], goodWindowAmp[idx]);
        }
        return timeAmps;
    }

    protected double getWindowStartingSampleSecsOffset(int windowCount) {
        return windowCount*getWindowSize();
    }

    protected int getWindowStartingSampleIndexOffset(int windowCount, double secsPerSample) {
        return (int) Math.round(getWindowStartingSampleSecsOffset(windowCount)/secsPerSample);
    }

// Coda object also needs to have its channel desc, datetime (Ptime) set by the invoking class
    private CodaTN resultsToCoda(CodaTN coda) {
        CodaTN resultsCoda = coda;
        if (resultsCoda == null) resultsCoda = new CodaTN();

        resultsCoda.changeDescriptor(new CodaPhaseDescriptorTN(codaPhase));

        resultsCoda.algorithm.setValue(algorithm);
        resultsCoda.ampUnits.setValue(UNITS);
        resultsCoda.windowSize.setValue(windowSize);
        resultsCoda.timeS.setValue(secsPastPTimeAtStartingIdx);

        resultsCoda.weightIn.setValue(quality);
        resultsCoda.tau.setValue(tauFinal);

        timeAmpsToResults(resultsCoda);

        if (goodWindowCount > 0) {
            resultsCoda.windowCount.setValue(goodWindowCount);
            resultsCoda.aFix.setValue(aFix);
            resultsCoda.aFree.setValue(aFree);
            resultsCoda.qFix.setValue(qFix);
            resultsCoda.qFree.setValue(qFree);
            resultsCoda.residual.setValue(rms);
         }

        return resultsCoda;
    }

    protected CodaTN timeAmpsToResults(CodaTN resultsCoda) {
        Collection timeAmpPairs = resultsCoda.getWindowTimeAmpPairs();
        if (debug) System.out.print("  CodaGeneratorTN.timeAmpsToResults goodWindowCount: " + goodWindowCount);
        if (goodWindowCount > 0) {
            TimeAmp [] timeAmps = getGoodWindowTimeAmps();
            int size = timeAmps.length;
            if (size > 6) size = 6;
            for (int idx = 0; idx < size; idx++) {
                timeAmpPairs.add(timeAmps[idx]);
                if (debug) System.out.print(" " + timeAmps[idx].toString() + " WT: " + windowWeight[idx]);
            }
        }
        else {
            if (debug) System.out.print(" zero good windows final timeamp:" + tauFinal + " " + aveAbsAmpAtEndOfCoda);
            timeAmpPairs.add(new TimeAmp(tauFinal, aveAbsAmpAtEndOfCoda));
        }
        if (debug) System.out.println();
        return resultsCoda;
    }

// Method to collect coda window data along time series;
// adapted from  CUSP CODAT_TM_R4_SUB.FOR
// Add method to use waveform object and Coda object???
    public boolean calcCoda(Coda result, double secsPerSample, double secsAfterPTime,
                            float [] timeSeriesAmps, int startingIdxOffset, double responseNormalizedCodaCutoffAmp,
                            double responseClippingAmp, double inLTA) {

        boolean status = calcCoda(secsPerSample, secsAfterPTime,
                            timeSeriesAmps, startingIdxOffset, responseNormalizedCodaCutoffAmp,
                            responseClippingAmp, inLTA);

        if (status) setCodaResults(result);
        return status;
    }

    public boolean calcCoda(double secsPerSample, double secsAfterPTime,
                            float [] timeSeriesAmps, int startingIdxOffset, double responseNormalizedCodaCutoffAmp,
                            double responseClippingAmp, double inLTA) {

        int numberOfSamples = timeSeriesAmps.length - startingIdxOffset;
        return calcCoda(secsPerSample, secsAfterPTime, timeSeriesAmps, startingIdxOffset, numberOfSamples,
                               responseNormalizedCodaCutoffAmp, responseClippingAmp, inLTA);
    }

    public boolean calcCoda( double secsPerSample, double secsAfterPTime,
                             float [] timeSeriesAmps, int startingIdxOffset, int timeSeriesSamples,
                             double responseNormalizedCodaCutoffAmp, double responseClippingAmp,
                             double inLTA
                           ) {

        initOutputData();

        if (timeSeriesSamples < 0 ) throw new IllegalArgumentException("input number of samples < 0");

        this.secsPerSample = secsPerSample;
        this.secsPastPTimeAtStartingIdx   = secsAfterPTime;

        int firstSampleIdxInTimeSeries = startingIdxOffset;

        if (timeSeriesSamples > timeSeriesAmps.length) timeSeriesSamples = timeSeriesAmps.length;
        if (firstSampleIdxInTimeSeries > 0 ) timeSeriesSamples -= firstSampleIdxInTimeSeries;
        if (timeSeriesSamples < 1) {
            System.out.println("  CodaGeneratorTN calcCoda input starting index beyond times series end.");
            return false;
        }
        this.lastSampleIdxInTimeSeries = firstSampleIdxInTimeSeries + timeSeriesSamples - 1;
        if (debug) System.out.println("  CodaGeneratorTN.calcCoda.secsPastPTimeAtStartingIdx: " +
            secsPastPTimeAtStartingIdx + " startingIdxOffset: " + startingIdxOffset +
            " lastSampleIdx: " + lastSampleIdxInTimeSeries);

        boolean hasResponseClippingAmp = false;
        if (responseClippingAmp > 0.) hasResponseClippingAmp = true;

        aveAbsAmpOfLastGoodWindow = 1.;
        isFirstWindow = true;

        int samplesPerWindow = (int) Math.rint(getWindowSize()/secsPerSample);

        int lowAmpWindowCount = 0;
        int validStartingAmpWindowCount = 0;       // count for amplitude check on first 2 windows

        int windowStartingSampleIdx = 0;
        int windowEndingSampleIdx = 0;
        int currentWindowSamples = 0;

        double averagedAbsAmp = 0.;

        float [] filteredAmps = filterTimeSeries(timeSeriesAmps);

        int nsamples = Math.min(timeSeriesSamples, noiseBiasSamples);
        double wfBias = calcBias(filteredAmps, 0, nsamples); // option change to pre-calculated LTA and bias for waveform input
        double longTermAverage =
             (inLTA == 0.) ? calcLongTermAbsAverage(filteredAmps, 0, nsamples, wfBias, startingIdxOffset): Math.abs(inLTA);


        minCodaStartingAmp = codaStartSignalToNoiseRatio * longTermAverage ;
        maxCodaNoiseCutoffAmp = codaCutoffSignalToNoiseRatio * longTermAverage;

        int iterations = (timeSeriesSamples + samplesPerWindow - 1)/samplesPerWindow;
        int iterCount = 0;

        if (debug) {
            System.out.println("  CodaGeneratorTN longTermAve: " + longTermAverage +
            " minCodaStartingAmp: " +  minCodaStartingAmp  + " maxCodaNoiseCutoffAmp: " + maxCodaNoiseCutoffAmp);
            System.out.println("  CodaGeneratorTN calcCoda bias:" + wfBias + " iterations:" + iterations);
            System.out.println("  CodaGeneratorTN calcCoda:" + inputToString());
        }
        MainLoop: while (iterCount < iterations) {
            iterCount++;
            CheckStatus: {   // break out of this block to check coda termination return conditions

                windowStartingSampleIdx = firstSampleIdxInTimeSeries +
                                              getWindowStartingSampleIndexOffset(windowCount, secsPerSample);
                windowEndingSampleIdx = windowStartingSampleIdx + samplesPerWindow - 1;       // end of current window

                if (windowStartingSampleIdx < 0) {
                    System.out.println("  CodaGeneratorTN starting sample index < 0" );
                    windowCount++;
                    badWindowCount++;
                    break CheckStatus; // before start of input time series data do next window
                }

                if (windowStartingSampleIdx >= lastSampleIdxInTimeSeries) {               // end of data
                    if (debug) System.out.println("  CodaGeneratorTN starting sample index at end of series" );
                    currentWindowSamples = lastSampleIdxInTimeSeries + samplesPerWindow - windowStartingSampleIdx;
                    endOfCoda(windowEndingSampleIdx, currentWindowSamples, averagedAbsAmp);
                    return doCodaStatistics();
                }

                if (windowEndingSampleIdx > lastSampleIdxInTimeSeries) {
                    windowEndingSampleIdx = lastSampleIdxInTimeSeries; // a truncated window
                }

                currentWindowSamples = windowEndingSampleIdx - windowStartingSampleIdx + 1;   // length of current window
                if (2 * currentWindowSamples < samplesPerWindow) {  // don't calc ave amp just add this length to coda duration
                    //System.out.println("  CodaGeneratorTN window samples < window size" );
                    currentWindowSamples = lastSampleIdxInTimeSeries - windowStartingSampleIdx;
                    endOfCoda(windowEndingSampleIdx, currentWindowSamples, averagedAbsAmp);
                    return doCodaStatistics();
                }

                // Usual path for coda calcuation, determines averaged absolute amplitude of window samples;
                boolean hasNoClippedSamples = true;
                for (int idx = windowStartingSampleIdx; idx <= windowEndingSampleIdx; idx++) {
                    //if( hasResponseClippingAmp && filteredAmps[idx] >= responseClippingAmp) hasNoClippedSamples = false;
                    averagedAbsAmp += Math.abs((double) filteredAmps[idx] - wfBias);
                }
                averagedAbsAmp = averagedAbsAmp/currentWindowSamples;
                if( hasResponseClippingAmp && (averagedAbsAmp >= responseClippingAmp)) hasNoClippedSamples = false;

                if (debug) {
                    System.out.print("  CodaGeneratorTN.calcCoda windowStartIdx: " +
                        windowStartingSampleIdx + " endIdx: " + windowEndingSampleIdx);
                    System.out.println(" clipAmp: " + responseClippingAmp +  " " +
                        " unClipped: " +  hasNoClippedSamples + " aveAmp: " + averagedAbsAmp);
                }
                if (averagedAbsAmp < 1.0) averagedAbsAmp = 1.0;

                validStartingAmpWindowCount++;
                boolean hasLowAmplitude = false;               // not yet finished
                int startOfCodaWindowCount = lowAmpWindowCount;// low-amplitude for 1st two windows
                if (validStartingAmpWindowCount <= 2) {        // execute only on first 2 windows
                    if (validStartingAmpWindowCount == 1) firstWindowAveAbsAmp = averagedAbsAmp;
                    if (firstWindowAveAbsAmp < responseNormalizedCodaCutoffAmp) lowAmpWindowCount++;
                    if (averagedAbsAmp < minCodaStartingAmp) lowAmpWindowCount++;
                    if (lowAmpWindowCount == 1) {
                        aveAbsAmpAtEndOfCoda = Math.rint(minCodaStartingAmp);
                        quality = 0.1;
                    }
                    else aveAbsAmpAtEndOfCoda = averagedAbsAmp;
                    if (lowAmpWindowCount > startOfCodaWindowCount) hasLowAmplitude = true; // terminate and do coda statistics
                }

                if (lowAmpWindowCount > 1 && validStartingAmpWindowCount > 2) {
                    quality = 0.0;
                    exitStatus = -1;
                    exitStatusString = "lowStartingAmp";
                    codaDesc = LOW_AMP_CODA_DESC;
                    return false;
                }

                if (! hasLowAmplitude) lowAmpWindowCount = 0;

                if (lowAmpWindowCount > 0) {       // bad coda window
                    windowCount++;
                    badWindowCount++;
                    break CheckStatus;            // do next window
                }

                // normal "60mv" coda (1 window less than total windows calculated);
                if (averagedAbsAmp < responseNormalizedCodaCutoffAmp) {
                    tauFinal = getWindowTauSeconds(windowCount);
                    aveAbsAmpAtEndOfCoda = aveAbsAmpOfLastGoodWindow;
                    codaDesc = BELOW_CUTOFF_AMP_CODA_DESC;
                    exitStatusString = "responseCutoffAmp";
                    exitStatus = 3;
                    return doCodaStatistics();
                }

                // high noise coda, background noise greater than cutoff counts;
                else if (averagedAbsAmp < maxCodaNoiseCutoffAmp) {
                    tauFinal = getWindowTauSeconds(windowCount);
                    aveAbsAmpAtEndOfCoda = aveAbsAmpOfLastGoodWindow;
                    codaDesc = NOISY_CODA_DESC;
                    exitStatusString = "noiseCutoffAmp";
                    exitStatus = 4;
                    return doCodaStatistics();
                }

                // Check for clipped amp;
                if (goodWindowCount > 0) {        // check window average amp for clipping
                    if (hasNoClippedSamples && averagedAbsAmp < responseClippingAmp &&
                        averagedAbsAmp < aveAbsAmpOfLastGoodWindow*passThruNoiseDecayAmpRatio ) {
                        if (endOfCoda(windowEndingSampleIdx, currentWindowSamples, averagedAbsAmp)) return doCodaStatistics();
                        else break CheckStatus;
                    }
                    else {                       // reset processing of coda windows
                        codaDesc = HIGH_AMP_CODA_DESC;
                        if (goodWindowCount >= minGoodWindowsToEndCalcOnClipping) {   // end of processing
                            aveAbsAmpAtEndOfCoda = aveAbsAmpOfLastGoodWindow;
                            tauFinal = getWindowTauSeconds(windowCount);
                            exitStatusString = "minGoodClipped";
                            exitStatus = 1;
                            return doCodaStatistics();
                        }
                        else if (resetOnClipping) {                  // reset counts and start again - WHY should this be done?
                            badWindowCount += goodWindowCount + 1;
                            windowCount = badWindowCount;
                            goodWindowCount = 0;
                            isFirstWindow = true;
                            break CheckStatus; // do next window
                        }
                        else return doCodaStatistics();       // added this logic aww 9/00
                    }
                }
                else {   // no valid windows yet
                    if (hasNoClippedSamples && averagedAbsAmp < responseClippingAmp) {   // got a good one
                        if (endOfCoda(windowEndingSampleIdx, currentWindowSamples, averagedAbsAmp)) {
                            return doCodaStatistics();
                        }
                        else break CheckStatus;
                    }
                    else { // discard current coda windows;
                        windowCount++;
                        badWindowCount++;
                        break CheckStatus; // do next window
                    }
                }
            } // end of CheckStatus block

            if (goodWindowCount >= maxGoodWindowCount) {  // terminate prematurely may still have unanalyzed time series data
                    tauFinal = getWindowTauSeconds(windowCount);
                    aveAbsAmpAtEndOfCoda = aveAbsAmpOfLastGoodWindow;
                    codaDesc = COUNT_CODA_DESC;
                    exitStatusString = "maxGoodCount";
                    exitStatus = 5;
                    return doCodaStatistics();
            }

            if (windowEndingSampleIdx == lastSampleIdxInTimeSeries &&  windowCount > 0) { // series ended at last window done
                currentWindowSamples = lastSampleIdxInTimeSeries - windowStartingSampleIdx + 1;
                endOfCoda(windowEndingSampleIdx, currentWindowSamples, averagedAbsAmp);
                return doCodaStatistics();
            }


        } // end of iteration while loop
        return false;  // abnormal shouldn't get here, ran off end of data
    }

    public boolean isFilterEnabled() { return filterEnabled;}
    public boolean hasFilter() { return (timeSeriesFilter != null);}
    public void enableFilter() { filterEnabled = true; }
    public void disableFilter() { filterEnabled = false; }
    public void setFilter(String seedchan) {
        if (timeSeriesFilter != null) setFilter(timeSeriesFilter.getFilter(seedchan));
        else System.out.println("Warning : CodaGeneratorTN.setFilter(String) timeSeriesFilter is null");
    }

/** Sets the filter to apply to input waveform time series. */
    public void setFilter(FilterIF filter) {
        timeSeriesFilter = filter;
    }

    protected float [] filterTimeSeries(float [] timeSeriesAmps) { // de-spiking then bandpass would be nice
        if (timeSeriesFilter == null ) return timeSeriesAmps;
        return (filterEnabled) ? (float []) timeSeriesFilter.filter(timeSeriesAmps) : timeSeriesAmps;
    }

    protected double getWindowTauSeconds(int windowCount) {
        return secsPastPTimeAtStartingIdx + ((double) windowCount + .5)*windowSize;
    }

// Method invoked when time series scan is ended because of amplitude tests or end of data
    private boolean endOfCoda(int windowEndingSampleIdx, int currentWindowSamples, double averagedAbsAmp) {
        // Test for end of data coda time series scan
        if (windowEndingSampleIdx >= lastSampleIdxInTimeSeries || goodWindowCount >= maxGoodWindowCount) {
              tauFinal = secsPastPTimeAtStartingIdx
                         + getWindowStartingSampleSecsOffset(windowCount)
                         + currentWindowSamples*secsPerSample;
            aveAbsAmpAtEndOfCoda = aveAbsAmpOfLastGoodWindow;       // note scalar fudge factor below -aww
            if (windowEndingSampleIdx >= lastSampleIdxInTimeSeries) codaDesc = TRUNCATED_CODA_DESC;
            else codaDesc = COUNT_CODA_DESC;
            exitStatusString = "endOfSeries";
            exitStatus = 2;
            return true; // do coda statistics
        }

        // not at coda end thus set values for current iteration
        if (averagedAbsAmp <= 0.) {
            System.err.println("Coda error bad averagedAbsAmp:" + averagedAbsAmp + " ; reset to 1. count.");
            averagedAbsAmp = 1.;
        }
        aveAbsAmpOfLastGoodWindow = averagedAbsAmp;

        double tauCurrentWindow = getWindowTauSeconds(windowCount);
        if (tauCurrentWindow <= 0.) {
            System.err.println("Coda error bad tauCurrentWindow:"+ tauCurrentWindow+ " ; reset to 1.0 seconds.");
            tauCurrentWindow = 1.;
        }

        windowCount++;
        goodWindowCount++;

        saveBufferValues(tauCurrentWindow, averagedAbsAmp);

        if (isFirstWindow) {
            firstGoodWindow = windowCount;
            isFirstWindow = false;
        }
        return false;
    }

    protected void saveBufferValues(double tauCurrentWindow, double averagedAbsAmp) {
        int idx = goodWindowCount - 1;
        windowWeight[idx]    = getWindowWeight(tauCurrentWindow, averagedAbsAmp);
        log10WindowAmp[idx]  =  windowWeight[idx]*MathTN.log10(averagedAbsAmp);
        log10WindowTime[idx] = -windowWeight[idx]*MathTN.log10(tauCurrentWindow);
        goodWindowTau[idx]   = tauCurrentWindow;
        goodWindowAmp[idx]   = averagedAbsAmp;
        if (debug) System.out.println("  CodaGeneratorTN saveBufferValues fit goodWindowCount: " +
            goodWindowCount + " " + tauCurrentWindow + " " +  averagedAbsAmp + " Wt: " + windowWeight[idx]);
    }

    protected double getWindowWeight(double tauCurrentWindow, double averagedAbsAmp) {
        return 1.0;
    }

// Method is invoked at end coda data collection for each time series.
    private boolean doCodaStatistics() {

        tauFinal             = Math.abs(tauFinal);  // when is it less than zero if starting idx is negative?
        aveAbsAmpAtEndOfCoda = Math.rint(aveAbsAmpAtEndOfCoda);

        if (goodWindowCount < 1) {    // no valid coda windows
            exitStatusString = "noGoodWindows";
            exitStatus = -2;
            return false;
        }
        else if (qFix < .01) {       // why, if ever true?
            exitStatusString = "badQFix";
            exitStatus = -3;
            return false;
        }
        calcL1Norm();
        return true;
    }

// l1 norm fit of data to determine Q and A parameters
    protected void calcL1Norm() {
        double smallestResidualSum = 999999.;
        for (int idx = 0; idx < goodWindowCount; idx++) {  // solve for Afix using Qfix
            double Q0 = qFix;
            double A0 = (log10WindowAmp[idx] - log10WindowTime[idx] * Q0)/windowWeight[idx];
            double sum = 0.;

            for (int idx3 = 0; idx3 < goodWindowCount; idx3++) {
                double residual = windowWeight[idx3]*A0 + log10WindowTime[idx3]*Q0 - log10WindowAmp[idx3];
                sum += Math.abs(residual);
            }

            if (sum <= smallestResidualSum) {
                aFix = A0;
                smallestResidualSum = sum;
            }
        }

        if (goodWindowCount >= 2) { // solve for Afree and Qfree

            smallestResidualSum = 999999.;
            for (int idx = 1; idx < goodWindowCount; idx++ ) {
                for (int idx2 = 0; idx2 < idx; idx2++) {
                    double numerator = log10WindowAmp[idx]*log10WindowTime[idx2] - log10WindowAmp[idx2]*log10WindowTime[idx];
                    double denominator = windowWeight[idx]*log10WindowTime[idx2] - windowWeight[idx2]*log10WindowTime[idx];
                    if (denominator < 0.001) denominator = 0.001;
                    double A0 = numerator/denominator;
                    double Q0 = (log10WindowAmp[idx] - windowWeight[idx]*A0)/log10WindowTime[idx];
                    double sum = 0.;
                    for (int idx3 = 0; idx3 < goodWindowCount; idx3++) {
                        double residual = windowWeight[idx3]*A0 + log10WindowTime[idx3]*Q0 - log10WindowAmp[idx3];
                        sum += Math.abs(residual);
                    }
                    if (sum <= smallestResidualSum) {
                        aFree = A0;
                        qFree = Q0;
                        smallestResidualSum = sum;
                    }
                }
            }
        }

        if (goodWindowCount > 2) {
            rms = smallestResidualSum/(goodWindowCount-2);       // save l1-fit as rms for qfree and afree
            quality = 1.;
        }
        else {
            if (goodWindowCount == 2) {
                rms = smallestResidualSum;
                quality = .5;
            }
            else {
                rms = 0.0;
                quality = .25;
            }
        }
    }

/** Print input setting to System.out*/
    public void printInputSettings() {
        System.out.println(inputToString());
    }

/** Print last results to System.out*/
    public void printResults() {
        System.out.println(outputToString());
    }

/** Print good time amp pair values to System.out*/
    public void printTimeAmps() {
        System.out.println(timeAmpsToString());
    }

/** Return String of any good time amp values.*/
    public String timeAmpsToString() {
        TimeAmp [] ta = getGoodWindowTimeAmps();
        int size = ta.length;
        if (size <= 0) return null;
        StringBuffer sb = new StringBuffer(512);
        for (int idx = 0; idx < size; idx++) {
            sb.append(" ");
            sb.append(ta[idx].toString(','));
            if (idx % 5 == 0) sb.append("\n");
        }
        return sb.toString();
    }

    public String inputToString() {
        StringBuffer sb = new StringBuffer(256);
        sb.append(" minWin: ").append(minGoodWindowsToEndCalcOnClipping);
        sb.append(" maxWin: ").append(maxGoodWindowCount);
        sb.append(" winSize: ").append(windowSize);
        sb.append(" nsamp: ").append(lastSampleIdxInTimeSeries+1);
        sb.append(" sps: ").append(secsPerSample);
        sb.append(" SmP: ").append(secsPastPTimeAtStartingIdx);
        sb.append("\n    startSNR: ").append(codaStartSignalToNoiseRatio);
        sb.append(" endSNR: ").append(codaCutoffSignalToNoiseRatio);
        sb.append(" passNSR: ").append(passThruNoiseDecayAmpRatio);
        sb.append(" clipReset: ").append(resetOnClipping);
        return sb.toString();
    }

    public String outputToString() {
        StringBuffer sb = new StringBuffer(256);
        //if (fmt == null) fmt = new Concat();
        sb.append("    ").append(codaPhase).append(" ");
        Concatenate.format(sb, quality, 2, 1);
        Concatenate.format(sb, tauFinal, 4, 1);
        Concatenate.format(sb, firstWindowAveAbsAmp, 9, 0);
        Concatenate.format(sb, aveAbsAmpAtEndOfCoda, 9, 0);
        Concatenate.format(sb, goodWindowCount, 4);
        Concatenate.format(sb, badWindowCount,  4);
        Concatenate.format(sb, firstGoodWindow, 4);
        Concatenate.format(sb, rms, 3, 2);
        Concatenate.format(sb, qFix, 3, 2);
        Concatenate.format(sb, qFree, 3, 2);
        Concatenate.format(sb, aFree, 3, 2);
        Concatenate.format(sb, aFix, 3, 2);
        sb.append(" ");
        String filterStr = (isFilterEnabled() && hasFilter()) ?  "true    " : "false   ";
        sb.append(filterStr);
        sb.append(getExitStatusString());
        return sb.toString();
    }
    public String outputToLabeledString() {
        StringBuffer sb = new StringBuffer(256);
        sb.append(" status: ").append(exitStatus);
        sb.append(" phase: ").append(codaPhase);
        sb.append(" qual: ").append(quality);
        sb.append(" tau: ").append(tauFinal);
        sb.append(" startAmp: ").append(firstWindowAveAbsAmp);
        sb.append(" endAmp: ").append(aveAbsAmpAtEndOfCoda);
        sb.append(" ngood: ").append(goodWindowCount);
        sb.append(" nbad: ").append(badWindowCount);
        sb.append(" 1stGoodAt: ").append(firstGoodWindow);
        sb.append(" rms: ").append(rms);
        sb.append(" QFix: ").append(qFix);
        sb.append(" QFree: ").append(qFree);
        sb.append(" AFree: ").append(aFree);
        sb.append(" AFix: ").append(aFix);
        sb.append(" desc: ").append(getExitStatusString());
        return sb.toString();
    }
    public String outputHeaderString() {
        StringBuffer sb = new StringBuffer(132);
        sb.append("            phase qual   tau startAmp    endAmp good bad 1st   rms  QFix  QFree AFree AFix Filter   Status");
        return sb.toString();
    }

// Methods to  initialize global data values
/** Returns last results in input  Coda object */
    public Coda setCodaResults(Coda results) {
        return resultsToCoda((CodaTN) results) ;
    }

/** Returns maximum value allowed for aveAbsAmpCurrentCodaWindow/aveAbsAmpLastGoodCodaDecayWindow.
* If value is exceeded (e.g. a noise burst or new phase arrival) the timeseries scan of coda is terminated.
 */
    public double getPassThruNoiseDecayAmpRatio() {
        return passThruNoiseDecayAmpRatio;
    }

/** Sets maximum value allowed for aveAbsAmpCurrentCodaWindow/aveAbsAmpLastGoodCodaDecayWindow. */
    public void setPassThruNoiseDecayAmpRatio(double passThruNoiseDecayAmpRatio) {
        if (passThruNoiseDecayAmpRatio > 0.) this.passThruNoiseDecayAmpRatio = passThruNoiseDecayAmpRatio;
    }

/** Returns Qfix */
    public double getQFix() {
        return qFix;
    }

/** Sets QFix value. */
    public void setQFix(double qFix) {
        if (qFix > 0.) this.qFix = qFix;
    }

    public void setResetOnClipping(boolean value) {
        resetOnClipping = value;
    }

    public boolean isResetOnClipping() {
        return resetOnClipping;
    }

/** Returns the minimum number of good coda time series windows needed to terminate the coda calculation
*   if the current window amplitude is above the clipping level.
*/
    public int getMinGoodWindows() {
        return minGoodWindowsToEndCalcOnClipping;
    }

    public void setMinGoodWindows(int numberOfWindows) {
        this.minGoodWindowsToEndCalcOnClipping = numberOfWindows;
    }

/** Returns maximum number of valid coda time series windows allowed before coda calculation is terminated.*/
    public int getMaxGoodWindows() {
        return maxGoodWindowCount;
    }

/** Sets maximum number of valid coda time series windows allowed before coda calculation is terminated.*/
    public void setMaxGoodWindows(int maxCount) {
        if (maxCount > this.maxGoodWindowCount) initBuffers(maxCount);
    }

/** Gets maximum number coda seconds duration for coda cutoff */
    public double getMaxCodaDurationSecs() {
        return getWindowStartingSampleSecsOffset(maxGoodWindowCount) + windowSize;
    }

/** Returns size in seconds of the time series window over which absolute amplitude values are averaged.*/
    public double getWindowSize() {
        return windowSize;
    }

/** Sets size in seconds of the time series window over which to average the absolute amplitude values.*/
    public void setWindowSize(double windowSize) {
        this.windowSize = windowSize;
    }

/** Valid coda window counting begins if averagedAbsoluteAmpofWindow/averagedAbsoluteValueOfNoise > snrToStart.  */
    public void setCodaStartSignalToNoiseRatio(double snrToStart) {
        codaStartSignalToNoiseRatio = snrToStart;
    }

    public double getCodaStartSignalToNoiseRatio() {
        return codaStartSignalToNoiseRatio ;
    }

/** Valid coda window counting ends if averagedAbsoluteAmpofWindow/averagedAbsoluteValueOfNoise < snrToEnd.  */
    public void setCodaCutoffSignalToNoiseRatio(double snrToEnd) {
        codaCutoffSignalToNoiseRatio = snrToEnd;
    }

    public double getCodaCutoffSignalToNoiseRatio() {
        return codaCutoffSignalToNoiseRatio ;
    }

/** Returns the average of the absolute demeaned values of the time series amplitudes over the specified range. */
    public static double calcLongTermAbsAverage(float [] timeSeriesAmps, int startingSampleIdx, int numberOfSamples,
                 double bias, int codaStartingIdx) {
        int nSamples = startingSampleIdx + numberOfSamples;
        if (nSamples > codaStartingIdx) nSamples = codaStartingIdx - startingSampleIdx;
        if (startingSampleIdx < 0 || nSamples > timeSeriesAmps.length || numberOfSamples < 0 )
            throw new IllegalArgumentException("starting or ending index is outside of input array bounds");
        double sum = 0.;
        for (int idx = startingSampleIdx; idx < nSamples; idx++) {
            sum += Math.abs((double) timeSeriesAmps[idx] - bias);
        }
        return sum/numberOfSamples;
    }

/** Returns the mean of the values of the time series amplitudes over the specified range. */
    public static double calcBias(float [] timeSeriesAmps, int startingSampleIdx, int numberOfSamples) {
        int nSamples = startingSampleIdx + numberOfSamples;
        if (startingSampleIdx < 0 || nSamples > timeSeriesAmps.length || numberOfSamples < 0 )
            throw new IllegalArgumentException("starting or ending index is outside of input array bounds");
        double sum = 0.;
        for (int idx = startingSampleIdx; idx < nSamples; idx++) {
            sum += (double) timeSeriesAmps[idx];
        }
        return sum/numberOfSamples;
    }

/** Returns bias window size in samples. */
    public double getNoiseBiasSamples() {
        return noiseBiasSamples;
    }

/** Returns bias window size in samples. */
    public void setNoiseBiasSamples(int noiseBiasSamples) {
        this.noiseBiasSamples = noiseBiasSamples;
    }

/** Returns a status value set by the coda calculations methods. */
    public int getExitStatus() {
        return exitStatus;
    }

/** Returns the String value of the exit code set by the coda calculations methods. */
    public String getExitStatusString() {
        return exitStatusString;
    }

}
