package org.trinet.jasi;

import java.text.*;
import java.util.*;
import java.io.*;
import org.trinet.util.*;

/**
 * Generic selection properties list.
 * You must have defined the environmental variable "JasiHome".
 * Default properties are in the file "$JasiHome/<fileName>" (UNIX).
 * User specific properties are in "$HOME/.jasi/<fileName>" (UNIX).
 * On PC's paths are equivalent.
 *
 * Use 'getProperty()' to retrieve properties from this list.
 */


public class JasiPropertyList extends GenericPropertyList
{

// Property path & file names
    String jasiHome;

    String userHome;

    // get system specific file separator so we can build a path string
    String fileSep  = System.getProperty("file.separator");  // "/" on UNIX, "\" on PC's ;  // "/" on UNIX, "\" on PC's

    // "$JasiHome/<fileName>"
    String defaultPropertiesFile;

    // "$HOME/.jasi/<fileName>"
    String userPropertiesFile;

    String filename;

    String filetype = "jasi";

/**
 *  Construtor: make empty property list. Use setFilename() and reset() to load it.
 */
    public JasiPropertyList()
    {
    }

/**
 *  Construtor: Makes a COPY of a property list. Doesn't read the files.
 */
    public JasiPropertyList(JasiPropertyList props)
    {
	super(props);
	setFilename(props.getFilename());	// remember the filename
    }

/**
 *  Construtor: Makes a COPY of a property list then read the files.
 */
    public JasiPropertyList(Properties props, String fn)
    {
	super(props);
	setFilename(fn);
	reset();
    }
/**
 *  Construtor: reads caller supplied property files.
 *  (String filename)
 */
    public JasiPropertyList(String fn)
    {
	setFilename(fn);
	reset();
    }

   public String getFilename() {

          return filename;
   }

   public String getUserPropertiesFile() {
      return userPropertiesFile;
   }

   public void setFiletype (String type) {
     filetype = type;
   }

   public String getFiletype () {
     return filetype;
   }
/**
 * Set file and path names
 */
    public void setFilename(String fn)
    {
      filename = fn;

	// "$JasiHome/properties"
	defaultPropertiesFile = getDefaultFilePath() + fileSep + filename;

	// "$HOME/.jasi/properties"
	userPropertiesFile    =
	    getUserFilePath() + fileSep+ "."+getFiletype() + fileSep + filename;
    }

    public String getUserFilePath() {

	if(System.getProperty("JIGGLE_USER_HOMEDIR") != null)
          return(System.getProperty("JIGGLE_USER_HOMEDIR"));
        else
          return(System.getProperty("user.home","."));


    }

    public String getDefaultFilePath() {
    // Look-up Jasi path environmental variable, system independently.
    // Default is ".", current directory environment
    // The variable "JASI_HOME"is passed with -D on java command line.
    // Native machine variables don't seem to be accessible.

	return System.getProperty ("JASI_HOME", ".");

    }

/**
 * Reset all property values to file defaults. If a property value is NOT in a file or
 * the RequiredProperties list it is NOT reset.
 */
    public void reset ()
    {
	setRequiredProperties();
	readDefaultProperties();
	readUserProperties();
    }

/**
 * Set required default properties so things will still work even if they are not set
 * in the properties files or the files are not present. This must be overridden by subclasses
 * set set their private essential property values.
 */
    public void setRequiredProperties()
    {
    }

// If other classes use getProperty() they will get a string and they often want an
// int, float, etc. So we'll do the type conversion here.

/**
 * Return property as an integer.
 * Returns 0 if there is no such property defined.
 */
public int getInt(String property)
{
    String str = getProperty(property);
    if (str == null) return 0;
    return Integer.valueOf(str).intValue();
}

/**
 * Return property as a float
 * Returns 0.0 if there is no such property defined.
 */
public float getFloat(String property)
{
    String str = getProperty(property);
    if (str == null) return (float) 0.0;
    return Float.valueOf(str).floatValue();
}
/**
 * Return property as a double
 * Returns 0.0 if there is no such property defined.
 */
public double getDouble(String property)
{
    String str = getProperty(property);
    if (str == null) return 0.0;
    return Double.valueOf(str).doubleValue();
}
/**
 * Parse the given property as a date/time string and return a DateTime object.
 * Returns current time if there is no such property defined.
 * Value of "*" will set to current time.
 */
public DateTime getDateTime(String property)
{
    String str = getProperty(property);

    if (str == null || str.equals("*")) return new DateTime();
    // expects EpochTime format string
    return new DateTime (EpochTime.stringToEpoch(getProperty(property)));    // constructor that parses date/time string
    //return new DateTime (getProperty(property));    // constructor that parses date/time string

}
/**
 * Set a date/time property. Insures string format is parseable.
 * Writes property string in EpochTime format. */
public void setDateTime(String property, DateTime dt)
{
//    put(property, dt.toString());
    // expects EpochTime format string
    put(property, EpochTime.epochToString(dt.getEpochSeconds()));
}
/**
 * Set a date/time property. Only allows "*".
 */
public void setDateTime(String property, String str)
{
    if (str.equals("*")) put(property, str);
}
/**
 * Parse the given property as a range (pair of doubles).
 * Returns null if no such property.
 */
public RangeDouble getRangeDouble(String property)
{

    String str = getProperty(property);
    if (str == null) return null;

    StringTokenizer tok = new StringTokenizer(str);

    double v1 = Double.valueOf(tok.nextToken()).doubleValue();
    double v2 = Double.valueOf(tok.nextToken()).doubleValue();

    return new RangeDouble(v1, v2);

}
/**
 * Parse the given property as a range (pair of ints).
 * Returns null if no such property.
 *
 * @depricated
 */
/*public RangeInt getRangeInt(String property)
{

    String str = getProperty(property);

    StringTokenizer tok = new StringTokenizer(str);

    int v1 = Integer.valueOf(tok.nextToken()).intValue();
    int v2 = Integer.valueOf(tok.nextToken()).intValue();

    return new RangeInt(v1, v2);

}*/
/**
 * Parse the given property as a range (pair of ints).
 * Returns null if no such property.
 */
public IntegerRange getIntegerRange(String property)
{

    String str = getProperty(property);
    if (str == null) return null;

    StringTokenizer tok = new StringTokenizer(str);

    int v1 = Integer.valueOf(tok.nextToken()).intValue();
    int v2 = Integer.valueOf(tok.nextToken()).intValue();

    return new IntegerRange(v1, v2);

}

/**
 * Read properties from system default file in $JasiHome/<filename>
 */
    public void readDefaultProperties()
    {
// create and load default properties
      try {
	  FileInputStream in = new FileInputStream(defaultPropertiesFile);

	  this.load(in);

          // DK Added 10202002
// already done by Given in Jiggle         String sDefaultJASISuffix = this.getProperty("JASISuffix");
          // already done by Givenif(sDefaultJASISuffix != null)
            // already done by GivenJasiObject.setDefaultBySuffix(sDefaultJASISuffix);
          // END DK Added 10202002

	  in.close();
      } catch (Exception e) {
	System.out.println ("Warning: no system default properties file: "+defaultPropertiesFile);
      }
    }

/**
 * Read properties from the user's file in $HOME/.jasi/<filename>
 */
    public void readUserProperties()
    {
// now load properties from last invocation
      try {
              FileInputStream in = new FileInputStream(userPropertiesFile);
              this.load(in);
              in.close();
      } catch (Exception e) {
	System.out.println ("Warning: no user properties file: "+userPropertiesFile);
      }

    }

/**
 * Save the current set of properties
 */
    public void saveProperties ()
    {
	//        saveProperties ("--- Jasi properties --- " + filename);
        saveProperties ("--- Jasi properties --- [" + userPropertiesFile+"]");
    }

    public void saveProperties (String header)
    {
      try {
	FileOutputStream out = new FileOutputStream(userPropertiesFile);

	//        this.save(out, header);	    // depricated w/ v1.2
        this.store(out, header);	    // new as of v1.2

        out.close();

      } catch (Exception e) {System.err.println(e);}

    }

    public void dumpProperties ()
    {
	this.list(System.out);
    }
    /**
     * Prints this property list out to the string. Based on method in parent class
     * Property.
     * This method is useful for debugging.
     */
    public String listToString() {

     String str = "";
	for (Enumeration e = propertyNames() ; e.hasMoreElements() ;) {
	    String key = (String)e.nextElement();
	    String val = (String) getProperty(key, "-not defined-");
	    str += key + "=" + val + "\n";
	}
     return str;
    }
} // end of class

