package org.trinet.jasi;

import java.io.*;
import java.util.*;
//import java.text.*;
import java.text.SimpleDateFormat;
import java.lang.*;

import org.trinet.util.*;
import org.trinet.util.gazetteer.*;
import org.trinet.jdbc.*;
import org.trinet.util.Format;		// CoreJava printf-like Format class

import org.trinet.jdbc.datatypes.DataDouble;
import org.trinet.jdbc.datatypes.DataLong;
import org.trinet.jdbc.datatypes.DataString;

/**
 * Create and parse Hypoinverse formatted lines
 */

public class HypoFormat
{

    static final int ARC_STRING_SIZE = 110;

/**
 * Null constructor
 */
public void HypoFormat()
{
}

/**
 * Make a string in .arc format (new hypo2000 version) from the contents of Phase.
 */

/*
    1         2         3         4         5         6         7         8         9        10        11
012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
STATNnt cCMP pr  yearmodyhrmnseconPresPwtS-secSr sSrespek2pekmmSwtPdlySdlyDistAngADPerRCodaAzmDurMagPimpSimpSFX
MCV  NC VVHZ  PU0199806262007 4310  -9110    0   0   0      0 0  0   0   0  1616400  0  159297404  0 502   0WD
HMT  CI VVHZ     199811102059    0   0  0 5632IS 0 -27    151 0  0   0   0 708 9000  0    0228  0  0   0
RDM  CI HHHZ     199902101612             1584iS 0
HMT  CI VVLN     200004071235        0  0 1285 S0 TPO   G  P V EP+2199811120254 3783  90  0    0   0   0      0 0  0   0   01334 7000  0    0154  0  0   0   0
QAL   G  P V    0199811120254    0   0  0 5709ES 2 301      0 0  0   0   01359 7000  0    0174  0  0   0   0
*/

public static String toArchiveString (Phase ph) {

    StringBuffer str = new StringBuffer(ARC_STRING_SIZE);	// make a space filled string buffer
    for (int i = 0; i < ARC_STRING_SIZE; i++) str.append(' ');

    Format fmt1 = new Format("%1s");	    // for efficiency

    Channel chan = ph.getChannelObj();

    str = overlay(str, 0, new Format("%-5s").form(chan.getSta()) );
    str = overlay(str, 5, new Format("%-2s ").form(chan.getNet()) );
    str = overlay(str, 8, fmt1.form(chan.getChannel().substring(0,1) ) );   // 1st character of component
//    str = overlay(str, 9, new Format("%3s ").form(chan.getChannel()));
 // changed to SEED component names
    str = overlay(str, 9, new Format("%3s ").form(chan.getSeedchan()));
/*
       1         2         3         4         5         6         7         8         9        10        11
012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
STATNnt cCMP pr  yearmodyhrmnseconPresPwtS-secSr sSrespek2pekmmSwtPdlySdlyDistAngADPerRCodaAzmDurMagPimpSimpSFX
MCV  NC VVHZ  PU0199806262007 4310  -9110    0   0   0      0 0  0   0   0  1616400  0  159297404  0 502   0WD
SIO  CI HHLE  P0 200001130934 2647 704 75
DTP  CI  EHZ    0200209122043    0   0  0 4582 S 3  -2      0 0 54   0   0 504 9000  0    0148  0  0   0 141
CLC  CI  HHZ  PU1200209122043 3985   2163    0   0   0      0 0  0   0   0 515 9000  0    0 69  0  0 362   0
WWP  CI VEHZ  PD2200209122043 3308  -1  0                                  106
WWP  CI VEHZ     200209122043             3458 S 1  -1           0         106
*/
// <P> wave Phase description, example: "IPU1"
    if (ph.description.iphase.equals("P")) {

      str = overlay(str, 13, ph.description.toMediumString());

    // phase time
      str = overlay(str, 17,  arcTimeString(ph.datetime.doubleValue()) );	    // "YEARMODYHRMNSS.SS"

      if (!ph.residual.isNull())
     str = overlay(str, 34, new Format("%4d").form((int) (ph.residual.doubleValue()*100) ));

      if (!ph.weightOut.isNull())
     str = overlay(str, 38, new Format("%3d").form((int) (ph.weightOut.doubleValue()*100) ));
    }

/*
    1         2         3         4         5         6         7         8         9        10        11
012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
STATNnt cCMP pr  yearmodyhrmnseconPresPwtS-secSr sSrespek2pekmmSwtPdlySdlyDistAngADPerRCodaAzmDurMagPimpSimpSFX
QAL   G  P V    0199811120254    0   0  0 5709ES 2 301      0 0  0   0   01359 7000  0    0174  0  0   0   0
Note: a station that is not "setup" will return a line as follows:
LVA2 CI HHHN     199909072229        0  0        1018iSc1                    0
*/
// <S> wave
    else if (ph.description.iphase.equals("S")) {

      // Must set P-Weight to bogus = 4
      str = overlay(str, 16, Integer.toString(4) );

    // phase time, S arrival time description is split in the format
      str = overlay(str, 17, arcYMDHMString(ph.datetime.doubleValue()) );	    // "YEARMODYHRMN"
    // field actually starts at 41, but we are doing 4 digits max so use 42.
      str = overlay(str, 42, arcSecString(ph.datetime.doubleValue()) );	    // "SSss"= SS.ss*100

      //      str = overlay(str, 46,  ph.description.toShortString());
      str = overlay(str, 46,  ph.description.toMediumString());

      if (!ph.residual.isNull())
     str = overlay(str, 50, new Format("%4d").form((int) (ph.residual.doubleValue()*100) ));

      if (!ph.weightOut.isNull())
     str = overlay(str, 63, new Format("%3d").form((int) (ph.weightOut.doubleValue()*100) ));

    } else {

  System.out.println ("* Phase type that Hypoinverse can't handle: " +
          str + " "+ph.description.iphase);
  return (str.toString());
    }

    // AssocArO parts
    // weight "actually used"
    if (!ph.weightIn.isNull())
  str = overlay(str, 38, new Format("%3d").form((int) (ph.weightIn.doubleValue()*100) ));

    if (!ph.getChannelObj().dist.isNull())
  str = overlay(str, 74, new Format("%4d").form((int) (ph.getDistance()*10.0) ) ) ;

    // Emergence angle (is using incidence here bogus?)
    if (!ph.incidence.isNull())
  str = overlay(str, 78, new Format("%3d").form((int) ph.incidence.doubleValue()) ) ;

    if (!(ph.getAzimuth() == 0.0))
  str = overlay(str, 91, new Format("%3d").form((int) ph.getAzimuth()) ) ;

/* DK 08/24/02
 *   if(ph.coda != null)
 *   {
 *	str = overlay(str, 87, new Format("%4d").form( ph.coda.tau.intValue()) ) ;
 *	str = overlay(str, 82, new Format("%1d").form( Coda.Weight_Jiggle2HI(ph.coda.weightIn.doubleValue())) ) ;
 *   }
 */
    return (str.toString());
    //    return (str.toString().trim());
}

/**
 * The StringBuffer doesn't really have a simple overlay method. Its insert()
 * method shoves bytes to the right as it inserts.
 */
private static StringBuffer overlay (StringBuffer sb, int start, String str) {

    int end = start + str.length();

    return sb.replace(start, end, str);
}

/*
    1         2         3         4         5         6         7         8         9        10        11
012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
STATNnt cCMP pr  yearmodyhrmnseconPresPwtS-secSr sSrespek2pekmmSwtPdlySdlyDistAngADPerRCodaAzmDurMagPimpSimpSFX
MCV  NC VVHZ  PU0199806262007 4310  -9110    0   0   0      0 0  0   0   0  1616400  0  159297404  0 502   0WD
HMT  CI VVHZ     199811102059    0   0  0 5632IS 0 -27    151 0  0   0   0 708 9000  0    0228  0  0   0
WOF   G  P V IPD0199811120254 2436   7172    0   0   0      0 0  0   0   0 49610100  0    0165  0  0 174   0

GTM  CI  EHZ  P00199912080004 4174  -9155    0   0   0      0 0  0   0   0 264 9000  0    0201  0  0 741   0

Note: a station that is not "setup" will return a line as follows:
LVA2 CI HHHN     199909072229        0  0        1018iSc1                    0
HDB  CI EEHZ  P3 200001130933 4050 312 25
*/


// TODO: Need to handle P and S on a single card

/**
 * Parse .arc format station line into new Phase object.
 * <tt>
01234567890
DTP  CI  EHZ    0200209122043    0   0  0 4582 S 3  -2      0 0 54   0   0 504 9000  0    0148  0  0   0 141
CLC  CI  HHZ  PU1200209122043 3985   2163    0   0   0      0 0  0   0   0 515 9000  0    0 69  0  0 362   0
WWP  CI VEHZ  PD2200209122043 3308  -1  0                                  106
WWP  CI VEHZ     200209122043             3458 S 1  -1           0         106
WOR  CI VEHZ  PU1200209122043 3330  -1  0                                  118
 * </tt>
 */

  public static Phase parseArcToPhase (String str) {

    double val; // general variable

    if (str.startsWith("    ")) return null;	// not an arrival line

    Phase ph = Phase.create();

    // Channel name, Note that it is incomplete. Note .trim() to remove trailing blanks.
    String sta  = str.substring(0,5).trim();
    String net  = str.substring(5,7).trim();
    String comp = str.substring(9,12).trim();

    // note putting channel in both 'channel' & 'seedchan'
    ph.setChannelObj(Channel.create().setChannelName(net, sta, comp, comp));

    // Phase description, example: "IPU1"

    /* WARNING: it is possible for Hypoinverse to combine p and S readings on a
    single card.  If so, this method will miss the S on a P card.

      Also, using str.substring(46,47).equals("S") seemed to return 'false' even
    when it wasn't but str.substring(46,47).equalsIgnoreCase("S") seems to work.  */

    // Read P info
    if (str.substring(14,15).equalsIgnoreCase("P"))	    // its a P-wave
    {
      String desc = str.substring(13, 17);
      ph.description.parseHypoinverseString(desc);

      //    	ph.weightIn.setValue(parseVal(str, 38, 41) );
      //	ph.weightIn.setValue(ph.description.quality);		// operator assigned quality

      //	ph.residual.setValue(parseVal(str, 34, 38) / 100.0) ;	// residual
      // *******************************************************
      // there's an error in the dbase constraints that won't allow negatives!
      // double residual = Math.abs(parseVal(str, 34, 38) / 100.0);
      // CONSTRAINT NOW CORRECTED

      // *******************************************************
      ph.residual.setValue(parseVal(str, 34, 38) / 100.0) ;	// residual

      // Hypoinverse Weights (from the program) seem to be in the range 0.0 -> 2.0
      // To fit into dbase constraint of 0.0 -> 1.0 I just /10.
      val = parseVal(str, 38, 41)/100;
      ph.weightOut.setValue(scaleWeight(val));

      ph.datetime.setValue(parseArcPTime(str));
    }

    // DK CODE CHANGE  121802
    // The string was checked for length > 41.  This doesn't
    // seem to be a valid check (atleast based on hi 2000 at MP) becauese
    // the distance(9999) always gets included on the end of lines that are not
    // in the station file, or do not have a valid location, or for some reason
    // cannot be associated with the location.
    // So we want to check one of the "association values" to make sure it is on
    // the line, before we go plotting blindly ahead trying to parse numbers out
    // of blank spaces or parsing passed the end of the string.
    // check that string is long enough to test
    //    if (str.length() > 41) {
    // check that there is atleast one "association based value" calculated
    if (parseNumber(str, 91, 94)!= null) {
      // Read S info
      if (str.substring(47,48).equalsIgnoreCase("S"))	    // its a S-wave
      {
        String desc = str.substring(46,50);

        ph.description.parseHypoinverseString(desc);
        //		ph.weightIn.setValue(ph.description.quality);	 // operator assigned quality

        double residual = parseVal(str, 50, 54) / 100.0;
        // *******************************************************
        // there was an error in the dbase constraints that won't allow negatives!
        //		double residual = Math.abs(parseVal(str, 50, 54) / 100.0);
        // constraint if now fixed
        // *******************************************************

        ph.residual.setValue(residual);	// residual

        // Hypoinverse Weights (from the program) seem to be in the range 0.0 -> 2.0
        // To fit into dbase constraint of 0.0 -> 1.0 I just /10.
        val = parseVal(str, 63, 66)/100 ;
        ph.weightOut.setValue(scaleWeight(val));

        ph.datetime.setValue(parseArcSTime(str));
      }

      /*	double val = parseVal(str, 78, 81);	 // emergence angle
      if (val != -1.0) ph.incidence.setValue(val);

        val = parseVal(str, 74, 78);
        if (val != -1.0) ph.distance.setValue(val / 10.0);

          val = parseVal(str, 91, 94);
          if (val != -1.0) ph.azimuth.setValue(val) ;	 // station azimuth
      */

      ph.incidence.setValue(parseNumber(str, 78, 81) ) ;	 // emergence angle
      Number num = parseNumber(str, 74, 78);                  // must check nullness
      if (num != null)
        ph.setDistance(num.doubleValue() / 10.0);  // before /10
      ph.setAzimuth(parseNumber(str, 91, 94).doubleValue()) ;		 // station azimuth

    }
    else  // DK CODE CHANGE 121802
    {
      // DK CODE CHANGE 121802
      // We didn't get association information, do we still want the phase????
    }
    /*    82,1  Dur Mag Weight Code
    87,4  Coda Dur (seconds)
    94,3,2Dur Mag value
    */
    Number num;
    if((num = parseNumber(str, 87, 91)) != null && (num.intValue() != 0))
    {
      Coda newCoda = Coda.create();
      newCoda.tau = new DataDouble(num.intValue());
      newCoda.tCodaTerm = new DataDouble(newCoda.tau.doubleValue() +
        ph.datetime.doubleValue());

      if((num = parseNumber(str, 82, 83)) != null)
        newCoda.weightIn = new DataDouble(Coda.Weight_HI2Jiggle(num.intValue()));
      if((num = parseNumber(str, 94, 97)) != null)
      {
        ChannelMag newChanMag = new ChannelMag();
        newChanMag.set(num.doubleValue()/100);
        newChanMag.weight = newCoda.weightIn;
        newChanMag.importance = newCoda.weightIn;
        newCoda.setChannelMag(newChanMag);
      }
      newCoda.setChannelObj(ph.getChannelObj());
      // ph.coda = newCoda;  // this line requires Md mod to coda
    }
    return (ph);
  }

/** Hypoinverse weights (from the program) seem to be in the range 0.0 -> 2.0
 * So scale to fit into dbase constraint of 0.0 -> 1.0 by dividing by 2.0. */
 public static double scaleWeight (double val) {

        double newVal = val / 2.0;
        if (newVal > 1.0) newVal = 1.0;    // insure no constrain violations
        if (newVal < 0.0) newVal = 0.0;

        return newVal;
 }

/**
 * Return a time string of the form: "YEARMODYHRMN SSSS". Seconds are to 2 decimals with no "."
 */
 public static String arcTimeString (double dt) {
  return ( arcYMDHMString(dt) + " " + arcSecString(dt) );
 }

/**
 * Return a time string of the form: "YEARMODYHRMN"
 */
 public static String arcYMDHMString (double dt)  {

/*
  SimpleDateFormat fmt
          = new SimpleDateFormat ("yyyyMMddHHmm");

// * 1000.0 for millisecs
  Date date =  new Date( (long) (dt * 1000.0) );

  String stime = fmt.format(date);
      return ( stime );
*/
     DateTime datetime = new DateTime(dt);
     return datetime.toDateString("yyyyMMddHHmm");

   }


/** Return a string for the epoch time with only hour/minute/second as required
 *  in the Hypoinverse terminator format. Ex: "HHmmssss". */
  public static String arcTerminatorTime(double dt) {

     String str = arcYMDHMString(dt) + arcSecString(dt);  // "yyyyMMddHHmmssss"
     return str.substring(8);
  }


/**
 * Return a time string of the form: "SSSS" with no decimal point.
 */
 public static String arcSecString (double dt) {

  return to42String( (new DateTime (dt)).getDoubleSecond() );

   }


/**
 * Make a Hypoinverse archive terminator line from a Solution object.
 * Note that the format uses Fortran style floating point format in some fields.
 * For example an "f4.2" field will appear like "3247" and be interpreted as
 * "32.47". <p>

 <tt>
   1         2         3         4         5         6         7
1234567890123456789012345678901234567890123456789012345678901234567890
      i2i2f4.2i2 f4.2i3  f4.2 f5.2                            i10
      hrmnsecsLT minsLON minszzzzz                            IDnumberxx
      0218152533 3050117 4512  600                            9876543
 </tt>
 */
    static public String toTerminatorString (Solution sol) {

    return toTerminatorString(sol, false, false, false);
     }

/**
 * Make a Hypoinverse archive terminator line from a Solution object.
 * If 'fixZ' is true the depth is fixed and will not be changed by the location process.
 * If 'fixLoc' is true both the location and depth are fixed but the origin time is
 * solved for.
 * If both are false, all parameters are solved for.
 * If 'trialLoc' is true the Solution (lat, lon, z and origin time) is used as
 * a trial (beginning) location by the location process. <p>
<tt>
   1         2         3         4         5         6         7
1234567890123456789012345678901234567890123456789012345678901234567890xxi2i2f4.2i2 f4.2i3  f4.2 f5.2                            i10
^^^^xxhrmnsecsLT minsLON minszzzzz                            IDnumberxx
      0218152533 3050117 4512  600                            9876543

Examples:
fixLoc = false, fixZ = false
                                                              9223372
fixLoc = true, fixZ = true
              33 4015116 5965-0.03X                           9223372
fixLoc = true, fixZ = false
              33 4015116 5965 0.03X                           9223372
fixLoc = false, fixZ = true
                             -0.03                            9223372
</tt>
 */
 static public String toTerminatorString (Solution sol,
                                          boolean fixLoc,
                                          boolean fixZ,
            boolean trialLoc) {

    StringBuffer sb = new StringBuffer(73);

    sb = sb.append ("      ");

    if (sol == null) return sb.toString();        //

    Format d2  = new Format("%2d");
    Format d3  = new Format("%3d");
    Format f52 = new Format("%5.2f");
    Format f42 = new Format("%4.2f");

//    sb = sb.append ("        ");  // trial time would go here (not used)
    sb = sb.append(arcTerminatorTime(sol.datetime.doubleValue()));
    LatLonZ latlonz = sol.getLatLonZ();

      double depth = Math.abs(latlonz.getZ());  // returns 0.0 if null
      if (fixZ) {
          depth = -depth;              // "-" indicates fixed
          if (depth == 0.0) depth = -0.01;  // Hypo no comprende -0.0
      }

      if ((fixLoc || trialLoc) && !latlonz.isNull()) {              // fix loc
         sb = sb.append(d2.form(latlonz.getLatDeg())+" ");
         sb = sb.append(to42String(latlonz.getLatMin()));
         sb = sb.append(d3.form(Math.abs(latlonz.getLonDeg()))+" ");
         sb = sb.append(to42String(latlonz.getLonMin()));
         sb = sb.append(f52.form(depth));
      } else {
         sb = sb.append("               ");
         if (fixZ) {                                  // fix Z only
           sb = sb.append(f52.form(depth));
         } else {                                     // don't fix nuthin'
           sb = sb.append ("     ");
         }
      }

      if (fixLoc && !latlonz.isNull()) {
         sb = sb.append ("X");    // any non-blank char here means fix location
      } else {
         sb = sb.append (" ");
      }

    if (!sol.id.isNull())
          sb = sb.append("                           "+sol.id.toString());

    return sb.toString();
 }

 /** Format seconds in Hypoinverse F4.2 style with no decimal point.
  *  For example: "12.8364" = "1284"*/
 protected static String to42String(double secs) {

    Format d2  = new Format("%02d");

    int   whole = (int) secs;               // casting truncates
    double frac = (secs - whole) * 100.0;

    return d2.form(whole) + d2.form( (int) Math.rint(frac));

 }
/**
 * Make a Hypoinverse2000 "summary header" line from a Solution object <p>
 <tt>
    1         2         3         4         5         6         7         8         9        10        11        12        13
01234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
199903080408538932 4205120  700    3  0 54329197 309223879900 43 29900  0   2659 -  098939893 71   0   0  0  0HAD      71    0  0   0  0         0   0   0   0   0
1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
yearmodyhrmnseccLT minsLON minszzzzzMagNphGapDmnRmssAziDpSize

199902101612100333 4015116 5965    3  0 31155  3  2316273 10134516  58  0     28 #  2  56  97 37   0   0  0  0HAD      39    0  0   0  0  13761428   0   0   0   0
</tt>
 */
    static public String toSummaryString (Solution sol) {

  if (sol == null) return new String("    ");
  return "      ";	    /// obviously not done yet here
    }

/**
 * Format the channel as a Hypoinverse #2 station data format string.
         |         |         |         |         |         |         |         |
12345678901234567890123456789012345678901234567890123456789012345678901234567890
                                      xxxx3.1
GLA   CI  HHZ  33  0.0520N114  0.8270E   00.0  P  0.00  0.00  0.00  0.00 0  0.00
BAR   CI  HLE  32 40.8000 116 40.3300  496.0 A 0.00 0.00 0.00 0.00 1 0.00
BAR   CI  HLN  32 40.8000 116 40.3300  496.0 A 0.00 0.00 0.00 0.00 1 0.00
BAR   CI  HLZ  32 40.8000 116 40.3300  496.0 A 0.00 0.00 0.00 0.00 1 0.00
BAR2  CI  EHZ  32 40.8000 116 40.3300  496.0 A 0.00 0.00 0.00 0.00 1 0.00
BAR2  CI  VHZ  32 40.8000 116 40.3300  496.0 A 0.00 0.00 0.00 0.00 1 0.00
 */
public static String toH2StationString(Channel chan) {

    StringBuffer sb = new StringBuffer(82);

    Format d2 = new Format("%2d");
    Format d3 = new Format("%3d");
    Format f52 = new Format("%5.2f");
    Format f62 = new Format("%6.2f");
    Format f74 = new Format("%7.4f");

    ChannelName cn = chan.getChannelName();

    String str = cn.getSta();
    if (str.length() > 5) str = str.substring(0,5);
    sb = sb.append ( new Format("%-5s ").form(str) );

    str = cn.getNet();
    if (str.length() > 2) str = str.substring(0,2);
    sb = sb.append ( new Format("%-2s  ").form(str) );

    str = cn.getSeedchan();
    if (str.length() > 3) str = str.substring(0,3);
    sb = sb.append ( new Format("%-3s  ").form(str) );

/*
    try {
      sb = sb.append(cn.getSta().substring(0,5)+" ");
    }
    catch  (StringIndexOutOfBoundsException e) {}

    try {
      sb = sb.append(cn.getNet().substring(0,2)+" ");
    }
    catch  (StringIndexOutOfBoundsException e) {}
    sb = sb.append(' ');          // optional 1-letter comp code
    try {
      sb = sb.append(cn.getChannel().substring(0,3)+" ");
    }
    catch  (StringIndexOutOfBoundsException e) {}

    sb = sb.append(' ');          // optional sta weight code code
*/

// lat/lon

    sb = sb.append(d2.form(Math.abs(chan.latlonz.getLatDeg())) + " ");
    sb = sb.append(f74.form(chan.latlonz.getLatMin()));
    if (chan.latlonz.getLat() > 0) {
       sb = sb.append('N');          // North
    } else {
       sb = sb.append('S');          // South
    }

    sb = sb.append(d3.form(Math.abs(chan.latlonz.getLonDeg())) + " ");
    sb = sb.append(f74.form(chan.latlonz.getLonMin()));
    if (chan.latlonz.getLon() > 0) {
       sb = sb.append('E');
    } else {
       sb = sb.append('W');
    }

    sb = sb.append( new Format("%4d").form( (int) (chan.latlonz.getZ()*1000.0))  );   // elevation, km-> m (not used)

    sb = sb.append( new Format("%3.1f").form(0.0) );   // period

    sb = sb.append("  P ");   // space, primary model, remark

    sb = sb.append( f52.form(0.0) + " ");   // P-delay #1
    sb = sb.append( f52.form(0.0) + " ");   // P-delay #2

    sb = sb.append( f52.form(0.0) + " ");   // Amp corr & weight

    sb = sb.append( f52.form(0.0) + " ");   // mag corr. & wt

    sb = sb.append( "0" );   // inst type code

    sb = sb.append( f62.form(0.0) + "  ");   // Cal factor & comp extension

    return sb.toString();
}

/**
 * Parse the  Hypoinverse2000 "summary header" line and and return a new Solution object <p>
 */
public static Solution parseArcSummary (String str)
{
    Solution sol = Solution.create();
    parseArcSummary (sol, str);

    return sol;
}
/**
 * Parse the  Hypoinverse2000 "summary header" line into an existing Solution object. <p>
 * This will only replace those values in the Solution that are returned by Hypoinverse. In
 * particular the orid, evid, commid, Event and Magnitude objects of the Solution are left
 * unchanged. Other fields are not modified because that info is not in the Hypoinverse
 * output, e.g. authority, source, quality, type, etype, errorLat, errorLon, etc.
 */

public static Solution parseArcSummary (Solution sol, String str) {

    final double f100 = (float) 100.0;
    final double f60  = (float) 60.0;

// Only use the evid from Hypoinverse if there isn't one already
    if (sol.id.longValue() <= 0) sol.id.setValue(parseInt(str, 136, 146) );

// Origin time
    sol.datetime.setValue(parseArcSummaryTime(str));

// Latitude
    double deg = parseVal(str, 16, 18);
    double min = (double) parseInt(str, 19, 23) / f100;	// f4.2 :. /100
    double xlat = deg + ( min / f60);
    String hemi = str.substring(18,19);  // "S" for south, BLANK otherwise!
    if (hemi.equalsIgnoreCase("S")) xlat = -xlat;

// Longitude
    deg = parseVal(str, 23, 26);
    min = (double) parseInt(str, 27, 31) / f100;		// f4.2 :. /100
    // WARNING: Hypoinverse does NOT follow its own rules. The summary line returns
    // positive longitude and does NOT put "E" in the line. :. must kludge here.
    double xlon = -(deg + ( min / f60));
    hemi = str.substring(26,27);         // "E" for south, BLANK otherwise!
    if (hemi.equalsIgnoreCase("E")) xlon = -xlon;

// Z
    double xz = parseInt(str, 31, 36) / f100;	// f5.2, in Schema positive depth is down.

    sol.setLatLonZ(xlat, xlon, xz);

// Location parameters
    sol.usedReadings.setValue(parseInt(str, 39, 42));	// P & S w/ weights > 0.1
    //    sol.totalReadings.setValue(parseInt(str, 39, 42));	// P & S w/ weights > 0.1
    sol.totalReadings.setValue(sol.phaseList.size());	// total associated phases
    sol.gap.setValue(parseVal(str, 42, 45));
    sol.distance.setValue(parseVal(str, 45, 48));
    sol.rms.setValue(parseVal(str, 48, 52) / f100);

// TODO: parse more from OriginError table?

    sol.sReadings.setValue(parseInt(str, 82, 86));
    sol.errorHoriz.setValue(parseVal(str, 85, 89) / f100);
    sol.errorVert.setValue(parseVal(str, 89, 93) / f100);
    sol.firstMotions.setValue(parseInt(str, 93, 96));

    // EventQuality requires errorVert & errorHoriz
    sol.quality.setValue(EventQuality.getValue(sol));

    sol.crustModel.setValue(str.substring(110, 113));

    // These parameters are not available from the Hypoinverse format
    // Authority, subsource
    // Other problems:
    // Crustal model is char[3], we expect int
    // "last authority" is one char, we expect FDSN char[2]

    // magnitude info not read yet
    //  grab duration magnitude data and build new Md mag

/*
    Number num;
    if((num = parseNumber(str, 70, 73)) != null)
    {
       Magnitude mag = Magnitude.create();
       mag.subScript = new DataString("d");
       mag.value = new DataDouble(num.doubleValue()/100);

       if((num = parseNumber(str, 100, 105)) != null)
       {
         mag.usedStations = new DataLong((num.intValue()+9.9)/10);  // approx
       }

       // mad (Median Absolute Difference)  -- could be used for quality
       if((num = parseNumber(str, 107, 110)) != null)
       {
         mag.error = new DataDouble(num.doubleValue() / 100.0);  // approx
       }
       mag.sol = sol;
       mag.codaList = sol.codaList;
       mag.source = sol.source;
       sol.magnitude = mag;
    }
*/

    return sol;

}

/**
 * Parse the P-wave time in the station Archive string
 */
public static double parseArcPTime (String str)
{
    double epochSecs = HypoFormat.parseArcYMDHM(str.substring(17,29));

    int sec  = (int) parseVal (str, 29, 32);      // whole seconds
    int frac = (int) parseVal (str, 32, 34);     // fractional 0.01 seconds

    return epochSecs + (double) sec + (double)frac/100.0;     // convert to epoch time
}

/**
 * Parse the S-wave time in the station Archive string
 */
public static double parseArcSTime (String str)
{
    double epochSecs = HypoFormat.parseArcYMDHM(str.substring(17,29));

    int sec  = (int) parseVal (str, 42, 44);      // whole seconds
    int frac = (int) parseVal (str, 44, 46);     // fractional 0.01 seconds

    return epochSecs + (double) sec + (double)frac/100.0;     // convert to epoch time
}

/**
 * Return epoch second of time string up to the minute. Does NOT include the seconds part
 * which varies with context. For station lines you need to pass <str>.substring(17,29).
 * This is the "base minute" of the "card".
 */
public static double parseArcYMDHM(String str) {

    int yr = parseInt (str,  0,  4);
    int mo = parseInt (str,  4,  6) - 1;  // in Java-land January = 0, etc.
    int dy = parseInt (str,  6,  8);
    int hr = parseInt (str,  8, 10);
    int mn = parseInt (str, 10, 12);

// Y2K note: Date expects 'yr' to be "years since 1900", but Calendar expects 4 digit year.
// January is month '0'
// If you try to use GMT time zone here, Java will kindly "correct" to local time later when
// you use SimpleDateFormat and subtract 7 or 8 hours. THIS ISN'T WHAT WE WANT. So, just use
// default and it should be OK.

//    TimeZone gmt = TimeZone.getTimeZone("GMT");
/*
    TimeZone gmt = TimeZone.getDefault();
    GregorianCalendar cal = new GregorianCalendar(gmt);

    cal.set(yr, mo-1, dy, hr, mn, 0);                  // convert to Java time base

    long epochMillis = cal.getTime().getTime();
*/
    DateTime dt = new DateTime(yr, mo, dy, hr, mn, 0.0);

    return dt.getEpochSeconds();

}

/**
 * Parse the Archive summary line format string. Note: its slightly different from the
 * time in the station lines.
 * <tt>
 * "199902101612100333 4015116 5965    3  0 31155  3  2316273 10134516  58  0     28 #  2  56  97 37   0   0  0  0HAD      39    0  0   0  0  13761428   0   0   0   0"
 </tt>
 *
 */
public static double parseArcSummaryTime (String str)
{
    double epochSecs = HypoFormat.parseArcYMDHM(str);
    int sec  = parseInt (str, 12, 14);	// whole seconds (f4.2) Note:sta fmt is (f5.2)
    int frac = parseInt (str, 14, 16);	// fractional seconds

    return epochSecs + (double) sec + (double)frac/100.0;     // convert to epoch time
}

/**
 * Convert the specified field in the string to a value. First field is 0.
 */
/* Note: "Double" is a class & "double" is a type, thus
 *        this weirdness
 */
   public static double parseVal (String str, int start, int stop)
    {
  String substr = "";

  try
  {
      substr = str.substring(start, stop);

      Double val = Double.valueOf(substr.trim());
      return (val.doubleValue());
  }
  catch (NumberFormatException exc)
  {
      System.out.println ("Double format conversion error: /" + substr+"/");
  } catch  (StringIndexOutOfBoundsException e) {
      System.out.println ("String out of bounds error: /" + str+"/");
  }
  return -1;
    }
/**
 * Convert the specified field in the string to a Number object. This allows a
 * return of 'null' if the string is blank.  */
   public static Number parseNumber (String str, int start, int stop)
    {
  String substr = "";

  try
  {
      substr = str.substring(start, stop);

      Double val = Double.valueOf(substr.trim());
      return (Number) val;
  }
  catch (NumberFormatException exc)
  {
      System.out.println ("Number format conversion error: /" + substr+"/");
  } catch  (StringIndexOutOfBoundsException e) {
      System.out.println ("String out of bounds error: /" + str+"/");
  }
  return null;
    }
/**
 * Convert the specified field in the string to an int value. First field is 0.
 */
   public static int parseInt (String str, int start, int stop)
    {
  String substr = "";

  try
  {
      substr = str.substring(start, stop);

      Integer val = Integer.valueOf(substr.trim());
      return (val.intValue());
  }
  catch (NumberFormatException exc)
  {
      System.out.println ("Integer format conversion error: /"+substr+"/");
  } catch (StringIndexOutOfBoundsException e) {
      System.out.println ("String out of bounds error: " + str);
  }
  return -1;  // well, ya gotta return somethin'
    }

/**
 * Parse a station list line in Hypoinverse format into a Channel object.
 * Returns null if there's a problem.
 */
/*
0         1         2         3         4         5         6         7         8
012345678901234567890123456789012345678901234567890123456789012345678901234567890
ABL  34 50.91N119 13.50W1975        0.0                            GSP VHZ 2252355    30  1700

 */

  public static Channel parseStationLine (String str)
  {
  double deg, min, lat, lon, z;

  Channel ch = Channel.create();

//TODO: make this general!!

        ch.setNet(EnvironmentInfo.getNetworkCode());     // must assume
  ch.setSta(str.substring(0, 5).trim());	// trim blanks
  ch.setChannel(str.substring(71, 74).trim());

  ch.setSeedchan(ch.getChannel());	// kludge

// cull out obvious crap
  if ( ch.getSta().equals("") ) return null;
  if ( ch.getChannel().equals("???") ) return null;

  deg = parseVal (str, 5, 7);
  min = parseVal (str, 8, 13);
  lat  = deg + (min/60.0);
  if (str.substring(13, 14).equalsIgnoreCase("S")) lat *= -1.0;    // change sign

  deg = parseVal (str, 14, 17);
  min = parseVal (str, 18, 23);
  lon = deg + (min/60.0);
  if (str.substring(23, 24).equalsIgnoreCase("W")) lon *= -1.0;    // change sign

  z = parseVal (str, 24, 28);
  z = z/1000.0;			// m -> km

  // guard against bogus lines
  if (lat == 0.0 && lon == 0.0) return null;

  ch.latlonz.set(lat, lon, z);

  return (ch);
  }


/**
 * Parse a station list line in Hypoinverse archive format into a Channel object.
 * Returns null if there's a problem. */
/*
0         1         2         3         4         5         6         7         8
012345678901234567890123456789012345678901234567890123456789012345678901234567890
ABL   CI  EHZ  34 50.9100 119 13.5000 1975.0 A 0.00 0.00 0.00 0.00 1 0.00

 */

  public static Channel parseStationArcLine (String str)
  {
  double deg, min, lat, lon, z;

  Channel ch = Channel.create();

  ch.setNet (str.substring(6, 8).trim());
  ch.setSta(str.substring(0, 5).trim());	// trim blanks
  ch.setChannel(str.substring(10, 13).trim());

  ch.setSeedchan(ch.getChannel());	// kludge

// cull out obvious crap
  if ( ch.getSta().equals("") ) return null;
  if ( ch.getChannel().equals("???") ) return null;

  deg = parseVal (str, 15, 17);
  min = parseVal (str, 18, 25);
  lat  = deg + (min/60.0);

  deg = parseVal (str, 26, 29);
  min = parseVal (str, 30, 37);
  lon = deg + (min/60.0);
// kludge !!!!! Longitudes in the file are positive and have no hemisphere letter.
  lon *= -1.0;    // change sign

  z = parseVal (str, 38, 44);
  z = z/1000.0;			// m -> km

  // guard against bogus lines
  if (lat == 0.0 && lon == 0.0) return null;

  ch.latlonz.set(lat, lon, z);

  return (ch);

  }

// ---------------------------------------------------------------------------------------
// Main for testing
// ---------------------------------------------------------------------------------------

public static void main (String[] arg) {

/*
       1         2         3         4         5         6         7         8         9        10        11
012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
STATNnt cCMP pr  yearmodyhrmnseconPresPwtS-secSr sSrespek2pekmmSwtPdlySdlyDistAngADPerRCodaAzmDurMagPimpSimpSFX
QAL  CI VVHZ     199811120254    0   0  0 5709ES 2 301      0 0  0   0   01359 7000  0    0174  0  0   0   0
QAL  CI VVHZ     199811120254      301 50 5709ES 2                        1359              70
*/
    //String testarr = "MCV  NC VVHZ  PU0199806262007 4310  -9110    0   0   0      0 0  0   0   0  1616400  0  159297404  0 502   0WD ";

String testarr =
"QAL  CI VVHZ     199811120254    0   0  0 5709ES 2 301      0 0  0   0   01359 7000  0    0174  0  0   0   0   ";

String testorg =
"199902101612100333 4015116 5965    3  0 31155  3  2316273 10134516  58  0     28 #  2  56  97 37   0   0  0  0HAD      39    0  0   0  0  13761428   0   0   0   0";

// parse the string
    Phase ph = parseArcToPhase(testarr);

// reformat

    System.out.println (" in: |"+testarr);
    System.out.println ("out: |"+toArchiveString (ph));
    System.out.println (ph.toString());

    System.out.println("");

// solution summary
    Solution sol = parseArcSummary(testorg);

    System.out.println (" in: |"+testorg);
    System.out.println ("out: |"+toSummaryString(sol));
    System.out.println (sol.toSummaryString());

    System.out.println("");

// terminator variants
    System.out.println (HypoFormat.toTerminatorString(sol));
    System.out.println (HypoFormat.toTerminatorString(sol, true, true, false));
    System.out.println (HypoFormat.toTerminatorString(sol, true, false, false));
    System.out.println (HypoFormat.toTerminatorString(sol, false, true, false));
    System.out.println (HypoFormat.toTerminatorString(sol, false, false, true));
}


} // end of class

