package org.trinet.jasi;

import java.net.*;
import java.io.*;
import java.util.*;

/**
 * Description of a JDBC dbase connection.
 */

public class DbaseConnectionDescription  {

   // Default port number ofor JDBC connections
    final static String DefaultPort = "1521";
    final static String DefaultDriver = "oracle.jdbc.driver.OracleDriver";

    /** JDBC driver name. Default = "oracle.jdbc.driver.OracleDriver" */
    String driver = DefaultDriver;
    String username;
    String password;

    /** Computer name without domain information.
     *  A full IP-style address would be <domain>.<host>*/
    String host;
    /** Domain name. Example: gps.caltech.edu
     *  A full IP-style address would be <domain>.<host>*/
    String domain ="gps.caltech.edu";
    String dbasename;

    /** Port number. Default =  1521*/
    String port = DefaultPort;

    public DbaseConnectionDescription() {

    }

    /** No port given, use default. */
    public DbaseConnectionDescription( String host,
				       String domain,
				       String dbasename,
				       String driver,
				       String username,
				       String password ) {

	this(host, domain, dbasename, DefaultPort, driver, username, password);

    }
/** Full specification. */
    public DbaseConnectionDescription( String host,
				       String domain,
				       String dbasename,
				       String port,
				       String driver,
				       String username,
				       String password ) {
	setURL (host, domain, dbasename, port);
	setDriver(driver);
	setUsername(username);
	setPassword(password);

    }

    public DbaseConnectionDescription(DbaseConnectionDescription dis) {
        setURL (dis.host, dis.domain, dis.dbasename, dis.port);
        setDriver(dis.driver);
        setUsername(dis.username);
        setPassword(dis.password);
    }

/** Return a new copy of this DbaseConnectionDescription*/
    public  DbaseConnectionDescription copy (DbaseConnectionDescription dis) {
        return new DbaseConnectionDescription(dis);

    }

    public void setURL (String host, String domain, String dbasename, String port) {
	setHost(host);
	setDomain(domain);
	setDbasename(dbasename);
	setPort(port);
    }

    public void setURL (String host, String dbasename, String port) {
	setHost(host);
	setDbasename(dbasename);
	setPort(port);
    }

    public void setURL (String host, String dbasename, int portnumber) {
	setURL(host, dbasename, ""+portnumber);
    }

    public String getDriver() {
	return driver;
    }
    public void setDriver(String driver) {
        this.driver = new String(driver);
    }

    public String getUsername() {
	return username;
    }
    public void setUsername(String username) {
        this.username = new String(username);
    }

    public String getPassword() {
	return password ;
    }
    public void setPassword(String password) {
	this.password = new String(password);
    }

    public String getDomain() {
	return domain;
    }
    public void setDomain(String domain) {
	this.domain = new String(domain);
    }

    public String getHost() {
	return host;
    }
    public void setHost(String host) {
	this.host = new String(host);
    }

    public String getDbasename() {
	return dbasename;
    }
    public void setDbasename(String dbasename) {
	this.dbasename = new String(dbasename);
    }

    public String getPort() {
	return port;
    }
    public void setPort(String port) {
	this.port = new String(port);
    }
    public void setPort(int port) {
	this.port = ""+port;
    }
    public void setURL(String dburl) {
	// must parse port, domain, host, and dbase from the URL
 	//dbaseURL = dburl;
	setPort(parsePortFromURL(dburl));
	setDbasename(parseDbaseNameFromURL(dburl));
	setHost(parseHostNameFromURL(dburl));
	setDomain(parseDomainNameFromURL(dburl));
    }
    /** Return the fully specified name of the JDBC URL for the dbase. It is
     * composed three properties: dbaseHost, dbasePort and dbaseName. It has a
     * form like: jdbc:oracle:thin:@quake.gps.caltech.edu:1521:quakedb*/
    public String getURL () {

	 return "jdbc:oracle:thin:@"+ getHost()+"."+getDomain()+
	     //"gps.caltech.edu"
	        ":"+getPort()+":"+getDbasename();
    }
    /** Return the IP address formed by <domain>.<host> */
    public String getIPAddress () {
       return getHost() + "." + getDomain();
    }

    public void readFromPropertyfile (String filename) {

	Properties props = new Properties();

	try {
	  FileInputStream in = new FileInputStream(filename);

	  props.load(in);

	  in.close();
	} catch (Exception e) {
	    System.out.println ("Warning: no such properties file: "+ filename);
	}
    }

    public void writeToPropertyfile (String filename) {

	Properties props = new Properties();
	props.put ("dbaseHost", host);
	props.put ("dbaseDomain", domain);
	props.put ("dbaseName", dbasename);
	props.put ("dbaseUsername", username);
	props.put ("dbasePassword", password);

	try {
	    FileOutputStream out = new FileOutputStream(filename);

	    props.store(out, "Dbase connection description");	    // new as of v1.2

	    out.close();

	} catch (Exception e) {System.err.println(e);}
    }

    public boolean equals (Object obj) {
      if (!(obj instanceof DbaseConnectionDescription)) return false;
      DbaseConnectionDescription other = (DbaseConnectionDescription) obj;
      return getURL().equals(other.getURL()) &&
	     getUsername().equals(other.getUsername()) &&
	     getPassword().equals(other.getPassword());
    }

    public String toString() {

	return driver + "  " + getURL() + " " + getUsername() + " " + getPassword();
    }


    /** Return dbase host name: assumes URL format like
        "jdbc:oracle:thin:@quake.gps.caltech.edu:1521:quakedb" */
    public static String parseHostNameFromURL (String URLstring) {
	String str = getURLAfterAt(URLstring);
	//int i2 = str.indexOf(':');
	int i2 = str.indexOf('.');

	return str.substring(0, i2);

    }
    /** Return domain name: assumes URL format like
     *  <tt>
     *  "jdbc:oracle:thin:@quake.gps.caltech.edu:1521:quakedb"
     *  returns            ^^^^^^^^^^^^^^^^
     *  </tt>
     */
    private static String parseDomainNameFromURL (String URLstring) {
	String str = getURLAfterAt(URLstring);
	int i1 = str.indexOf('.');        // 1st "."
	int i2 = str.indexOf(':', i1+1);  // 2nd ":"

	return str.substring(i1+1, i2);

    }

    /** Parse the port number out of the URL string.assumes URL format like
     *  <tt>
     *  "jdbc:oracle:thin:@quake.gps.caltech.edu:1521:quakedb"
     *  returns                                  ^^^^
     *  </tt>
     *  */
    private static String parsePortFromURL (String URLstring) {
	String str = getURLAfterAt(URLstring);
	int i1 = str.indexOf(':');        // 1st ":"
	int i2 = str.indexOf(':', i1+1);  // 2nd ":"

	return str.substring(i1+1, i2);
    }
    /** Return dbase name:assumes URL format like
     *  <tt>
     *  "jdbc:oracle:thin:@quake.gps.caltech.edu:1521:quakedb"
     *  returns                                       ^^^^^^^
     *  </tt>*/
    private static String parseDbaseNameFromURL (String URLstring) {
	String str = getURLAfterAt(URLstring);
	int i1 = str.indexOf(':');        // 1st ":"
 	int i2 = str.indexOf(':', i1+1);  // 2nd ":"
	return str.substring(i2+1);
    }
    /** Helper for parse methods: assumes URL format like
     *  <tt>
     *  "jdbc:oracle:thin:@quake.gps.caltech.edu:1521:quakedb"
     *  returns            ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
     *  </tt>
     *  */
    private static String getURLAfterAt(String URLstring) {
	int idx = URLstring.lastIndexOf("@");
        return (idx != -1) ? URLstring.substring(idx+1) : "";
    }
} // DbaseConnectionDescription
