package org.trinet.jasi;

import java.util.*;
import org.trinet.util.gazetteer.LatLonZ;

/** An ActiveArrayList of Channelable objects. Allows calculation of distance from
* a Solution or LatLonZ and sorting of the list by distance/component. */
public class ChannelableList extends ActiveArrayList {

     static final double NULL_DIST = 9999.9;

     public ChannelableList() {
     }

     public ChannelableList (int initCapacity) {
            super(initCapacity);
     }

     public ChannelableList(Collection col) {
	       super(col);
     }

/**
 * Calculate the horizontal distance and azimuth from location in the argument
 * for each object in the list.
 * If the LatLonZ of the object is null the distance is set to 9999.9.*/
public void calcDistances (LatLonZ loc) {

     Channelable ch[] = getChannelableArray();

	for ( int i=0; i < ch.length; i++) {
         ch[i].calcDistance(loc);        // does azimuth, too
	}
}

/** Given a list of Channelable's (which have azimuth values) find the
* maximum azmuthal gap. Returns the gap in degrees. */
public double getMaximumAzimuthalGap () {

     if (size() < 2) return 360.0;  // only one point

     Channelable ch[] = getChannelableArray();

     double az[]   = new double[size()];

     for (int i = 0; i < size(); i++) {
            az[i]   = ch[i].getAzimuth();
     }

     // sort in ascending order, first is smallest
     Arrays.sort(az, 0, az.length);

     double diff;

     // start with gap from last to first azimuth
     // this angle would span the 0 position in the circle
     double maxgap = (az[0] + 360.0) - az[az.length-1];

     // go clockwise around the circle
	for ( int i=0; i < az.length-2; i++) {
          diff = az[i+1] - az[i];
          if (diff > maxgap && diff != Double.NaN) maxgap = diff;
	}
     return maxgap;
}

/**
 * Calculate distances and sort the list by distance/channel from the given location.
*/
    public void distanceSort(LatLonZ latlonz) {
       calcDistances (latlonz);
	  Collections.sort(this, new ChannelSorter());
    }

    /**
     * Trim any objects with distance greater then 'dist'. Returns true if objects
     * were removed.
     */
    public boolean trim (double dist) {

      boolean trimmed = false;

	 Channelable ch[] = getChannelableArray();

      // Most efficient to do in reverse, eliminates repacking array
      for ( int i = ch.length-1; i >= 0; i--) {
	  if (ch[i].getDistance() > dist) {
          remove(i);                      // each remove fires an event
          trimmed = true;
       }
      }
      return trimmed;
    }

    /**
     * Trim any WFViewList to this many WFViews by removing those with indexes > count.
     * Returns true if objects were removed.
     */
    public boolean trim (int count) {

     if (count < 0) return false;
	if (count >= (this.size()-1) ) return false;	// nothing to trim

	// guard against out-of-bounds errors
	int from = count;
	if (from < 0) from = 0;

     // Removes from this List all of the elements whose index is between
     // fromIndex, inclusive and toIndex, exclusive.
	removeRange(from, this.size());

     return true;
    }
    /**
     * Return this list (collection) as an array of Channel objects.
     * Returns a 0 length array if the ChannelableList is null.
     */
     public Channelable[] getChannelableArray () {

	//Channelable ch[] = new Channelable[this.size()];

	return (Channelable[]) this.toArray(new Channelable[0]);
    }

    /** Return the first JasiReading in the list that matches the
     * given Channel. Returns null if no match. */

    public Channelable getByChannel (Channel chan) {

	Channelable ch[] = getChannelableArray();
	for (int i = 0; i<size(); i++) {

	    if ( ch[i].getChannelObj().sameAs(chan) ) return ch[i];
	}
	return null;
    }
    /** Return the index in the JasiReading of the object that matches the
     * given Channel. Returns -1 if no match. */

    public int getIndexOf(Channel chan) {

	Channelable ch[] = getChannelableArray();
	for (int i = 0; i<size(); i++) {

	    if ( ch[i].getChannelObj().sameAs(chan) ) return i;
	}
	return -1;
    }

    /** Given a list of Channelable objects, return a count of how many unique
    * stations are represented in the list. A "station" is a unique "net", "sta".
    * A station may have multiple channels. */
    public int getStationCount () {

         if (isEmpty()) return 0;

         int count = size();

         Channelable ch[] = getChannelableArray();

         // for each entry, scan forward in list & only count if no repeats ahead
	    for (int i = 0; i<size()-1; i++) {
             for (int k = i+1; k<size(); k++) {
	          if ( ch[i].getChannelObj().sameStationAs(ch[k].getChannelObj()) ) {
                  count--;
                  break;
               }
             }
         }
         return count;
    }

}