package org.trinet.util.magnitudeengines;

import java.util.*;

import org.trinet.util.gazetteer.LatLonZ;
import org.trinet.util.*;
import org.trinet.jasi.*;

//import org.trinet.filters.GenericFilter;

/**
 * Magnitude calculation engine. Given a magnitude calculation method object and an
 * amplitude or a list of amplitudes calculate a magnitude.
 *
 * @See: Amplitude
 * @See: MagnitudeMethod
 * @See: ChannelMag
 *
 *
 * Created: Fri Jun  2 08:48:39 2000
 *
 * @author Doug Given
 * @version
 */

public class LocalMagnitudeEngine extends MagnitudeEngine {

    /** The magnitude calculation method. Defaults to generic local magnitude
        (Richter) ML. */
    public TrinetMagnitudeMethod magMethod; //0 =
//        (TrinetMagnitudeMethod) MagnitudeMethod.CreateMagnitudeMethod("org.trinet.util.magnitudeengines.ML");

    /** Set true to reuse old amps rather then rescan the time-series. */
    boolean reuseOldAmps = false;

    /** If 'true' only that part of a waveform that is likely to contain seismic
    * energy will be scanned. If 'false' the whole available waveform is scanned.<p>
    *
    * If 'true' window scanned begins 1 sec before expected P-wave onset and
    * ends at P-wave onset plus 2x the S-P time.
    *
    */
    boolean scanEnergyWindow = true;

    /** Channel list for looking up lat/lon, gain, corrections, etc. */
    ChannelList chanList = null;

    static boolean debug = true;
    //static boolean debug = false;

    /** */
    public LocalMagnitudeEngine() {

    }

  /** abstract from MagnitudeEngine
   **************************************/
  public void ConfigureMagnitudeEngine()
  {
  }

  /** abstract from MagnitudeEngine
   **************************************/

  public void ConfigureMagnitudeEngine(int iConfigurationSource,
                                String sConfigurationLocation,
                                String sConfigurationSection,
                                ChannelList MasterChannelList
                               )
  {
  }



  /** Configuration method that derives its configuration information from
       a MagnitudeMethod object.
   *********************************************************************/
//  public void ConfigureMagnitudeEngine(MagnitudeMethod magMeth)
//  {
//     magMeth = (TrinetMagnitudeMethod)magMeth;
//     bMagnitudeEngineObjectIsValid = true;  // now we're valid
//  }
  /**
   * Set the magnitude calculation method.
   * @See: MagnitudeMethod
   */
  // Override to cast as TrinetMagnitudeMethod
  public void setMagMethod (MagnitudeMethod magMeth) {
    this.magMethod = (TrinetMagnitudeMethod) magMeth;
    bMagnitudeEngineObjectIsValid = true;
  }
  /**
   * Return the magnitude calculation method.
   * @See: MagnitudeMethod
   */
  // Needed because of override
  public MagnitudeMethod getMagMethod() {
    return magMethod;
  }
  /** Calculation methods */

    /** Calculate the given Magnitude using the amps in its ampList. Sets all
     * the Amplitude.ChannelMag values. Sets the Solution.magnitude
     * values. Returns reference to Solution.magnitude. Follows all the rules of
     * the MagnitudeMethod and checks the channel rejection criteria like
     * maxDistance.  Does NOT assume list is in any particular order, so if
     * maxChannels is set only the first 'maxChannels" amps in the list will
     * contribute to the Magnitude. */
    public Magnitude solve (Solution sol)
    {
       return solve (sol, sol.ampList);
    }

//    /**
//    * Set the magnitude calculation method.
//    * @See: MagnitudeMethod
//    */
//    public void setMagMethod (MagnitudeMethod magMeth) {
//      this.magMethod = (TrinetMagnitudeMethod)magMeth;
//    }
//    /**
//    * Return the magnitude calculation method.
//    * @See: MagnitudeMethod
//    */
//    public MagnitudeMethod getMagMethod() {
//  return magMethod;
//    }

    /** Set true to reuse old amps rather then rescan the time-series. */
    public void setReuseOldAmps(boolean tf) {
       reuseOldAmps = tf;
    }

    /**
    * If 'true' the waveform window scanned begins 1 sec before expected
    * P-wave onset and ends at P-wave onset plus 2x the S-P time.
    * If 'false' the whole available waveform is scanned.
    */
    public void setScanEnergyWindow  (boolean tf) {
       scanEnergyWindow = tf;
    }
    /** Set the channel list for looking up lat/lon, gain, corrections, etc. */
    public void setChannelList (ChannelList list) {
       chanList = list;
    }

    /**
     * Calculate the magnitude for one amplitude. Put the result into the
     * ChannelMag instance of the amplitude object.  The Amplitude must be of the
     * correct type for the MagnitudeMethod.
     * The amp value must be in the units expected by the current
     * mag method. No corrections are applied. Returns the magnitude value.  */
    public double calcChannelMag (double dist, double val) {

      return magMethod.getValue(dist, val);
    }

    /**
     * Calculate the magnitude for one amplitude. Put the result into the
     * ChannelMag instance of the amplitude object. The Amplitude must be of the
     * correct type for the MagnitudeMethod, if not, ChannelMag is unchanged and
     * this method returns the original Amplitude unchanged.  Amplitude must
     * have a value and a distance. Returns the Amplitude that was passed as an
     * argument.<p> The distance cutoff and channel rejection criteria like
     * maxdistance, are NOT checked.  */
    public Amplitude calcChannelMag (Solution sol, Amplitude amp) {

      // make sure channel is "set up" and distance is calculated
      if (!amp.getChannelObj().hasLatLonZ() || amp.getChannelObj().gain.isNull())
        amp.setChannelObj(Channel.create().lookUp(amp.getChannelObj()));

      if (amp.getChannelObj().getDistance() == 0.0 )
        amp.getChannelObj().calcDistance(sol.getLatLonZ());

      // calc answer
      double magValue = 0.0;
      try {

        magValue = magMethod.getValue(amp);

      } catch (WrongAmpTypeException ex) {
        System.out.println(ex.getMessage());
        amp.delete();
        return amp;
      }

      if (debug) System.out.println ("calcChannelMag: "+
                                     amp.getChannelObj().toDelimitedSeedString(' ')+
                                     " amp= "+amp.value.floatValue()+
                                     " mag= "+magValue);

      // success, put it in the amplitude's channelmag value
      amp.channelMag.set( magValue );

      // set the mag correction value if it's not null
      // there's a bug in DataObject.setValue() that makes this needful
      if (!amp.getChannelObj().mlCorr.isNull()) {
        amp.channelMag.correction.setValue(amp.getChannelObj().mlCorr);
      }

// already comes from channel	amp.setDistance(amp.getChannel().getDistance());

      // amp.channelMag.residual not known yet

//     amp.setQuality(1.0);
      amp.setWeight(1.0);

      amp.channelMag.subScript.setValue(magMethod.getSubScript());

      return amp;
    }
    /** Return a new magnitude with basic attributes set mag type. */
     protected Magnitude getNewMag (Solution sol, AmpList ampList ) {

       Magnitude newMag = getNewMag(sol);
       newMag.addAmps(ampList);

       return newMag;
     }
    /** Return a new magnitude with basic attributes set mag type. */
     protected Magnitude getNewMag (Solution sol) {

       Magnitude newMag = Magnitude.create();

       newMag.authority.setValue(EnvironmentInfo.getNetworkCode());
       newMag.source.setValue(EnvironmentInfo.getApplicationName());
       newMag.method.setValue(magMethod.getName());
       newMag.subScript.setValue(magMethod.getSubScript());

       newMag.associate(sol);     // this also sets mag's orid to sol's

       return newMag;
     }
    /** Calculate the Magnitude using the amps associated in this list.
     * Sets all the Amplitude.ChannelMag values. Sets the new
     * magnitude as the prefered one and the old prefered is retained in the
     * Solution's list of alternate mags.  Returns a reference to
     * Solution.magnitude. Follows all the rules of the MagnitudeMethod and
     * checks the channel rejection criteria like maxDistance.  Does NOT assume
     * the Amplitude list is in any particular order, so if maxChannels is set
     * only the first 'maxChannels" amps in the list will contribute to the
     * Magnitude. <p> It is up to the caller to set the resulting Magnitude as
     * the Solution's magnitude if it wants to. <p> The values of 'authority'
     * and 'source' are set to the values in EnvironmentInfo.
     *
     * @See: EnvironmentInfo*/

    public Magnitude solve (Magnitude mag) {

  // are there amps?
  if (mag.ampList.isEmpty()) return mag;

  // !!! have distances? Corrections?
  channelLookup(mag.ampList);

  calcDistances(mag.ampList, mag.sol.getLatLonZ());

  // add amps to this mag
//     mag.addAmps(ampList);

  // do the summary mag, amps are added to the mag
  mag = calcSummaryMag (mag);

    // Magnitude.addAmp() automatically sets this true, so undo it
  mag.setStale(false);

  // make the Mag the primary one for the solution (saves the old one)
  mag.sol.setPreferredMagnitude(mag);

  return mag;
    }

/** Magnitude Calculator
    Calculates a Magnitude using only the channels that have codas and
    amplitudes associated with "mag".  It may use phases and waveforms from the
    Solution, for use in calculating station magnitudes, but only in conjunction
    with existing amplitudes/codas from "mag".
    I don't know that this makes a whole lot of since, but it is spec'd behavior.
    DaveKr 050802
 */
  public Magnitude solve(Magnitude mag, Solution sol, Collection Waveforms)
  {
    // DK CLEANUP this should be better implemented
   return solveFromWaveforms ( sol, Waveforms);
   //return solve(mag);
  }

    /** Calculate a Magnitude using the amps in its ampList. Sets all
     * the Amplitude.ChannelMag values. Sets the Solution.magnitude
     * values. Returns reference to Solution.magnitude. Follows all the rules of
     * the MagnitudeMethod and checks the channel rejection criteria like
     * maxDistance.  Does NOT assume list is in any particular order, so if
     * maxChannels is set only the first 'maxChannels" amps in the list will
     * contribute to the Magnitude. */
    public Magnitude solve (Solution sol, AmpList ampList) {

      Magnitude newMag = getNewMag(sol, ampList);
      return solve (newMag);

    }
    /** Given a Magnitude with an collection of Amplitudes, calculate the median
     * magnitude using only non-zero-weight readings.
     * Does NOT do weighting otherwise, just finds median on non-zero weight
     * amps. Returns a Magnitude object with propert summary values set. */
    public Magnitude calcMedianMag (Magnitude mag) {

  Amplitude amp[] = mag.ampList.getArray();

  double goodList[] = new double[amp.length];

  int nused = -1;

  // for closest station search
  double closest = Double.MAX_VALUE;

  // Accumulate usable amps
  for (int i=0; i<amp.length; i++) {
      if ( amp[i].getWeight() != 0.0) {
      goodList[++nused] = amp[i].channelMag.value.doubleValue();
      closest = Math.min( closest, amp[i].getDistance());
         }
  }

  nused++;     	// change from index to count

     // bail if none used
  if (nused > 0)  {
    // calculate revised summary mag NOTE: nused is needed here rather than
    // goodList.size() because the list may not be full.
    mag.value.setValue( Stats.median(goodList, nused) );
    mag.error.setValue( Stats.standardDeviation(goodList, nused) );
    mag.quality.setValue(1.0);          // real meaning of quality is undefined
    mag.distance.setValue(closest);
  } else {
    mag.value.setValue( 0.0 );      // should be null?
    mag.error.setValue( 0.0 );
    mag.distance.setValue( 0.0 );
    mag.quality.setValue(0.0);
  }

     // mag.usedReadings.setValue(nused);
     // use "stations" not channels used
//     mag.usedReadings.setValue(mag.ampList.getStationUsedCount());

     // calc residuals, needed for 2nd pass of trimming
     setResiduals(mag);

     mag.setStale(false);

     return mag;
    }

    /** Given an array of Amplitudes calculate a magnitude. Don't use 0-weights.
     * Does NOT do weighting otherwise, just finds median on non-zero weight
     * amps and returns a Magnitude object. Sets residuals for ALL amps
     * regardless of weight. */
    public Magnitude calcSummaryMag (Magnitude mag) {

    // pre-filter the list
     magMethod.preScanAmpList(mag);

     // do the channel mags
     Amplitude amp[] = mag.ampList.getArray();
     for (int i=0; i<amp.length; i++) {
       amp[i] = calcChannelMag(mag.sol, amp[i]);
     }

      // keep calulating and trimming until its stable
      int knt = 0;
      mag.setStale(true);  // do following loop at least once

      while (mag.isStale()) {

         calcMedianMag(mag);
         if (debug) System.out.println ("Pass "+knt+" : "+mag.toNeatString());

         // post-filter the list, set mag stale if it removes any amps
         // causing this 'while loop' to repeat
         if (magMethod.postScanAmpList(mag)) mag.setStale(true);
      }
      if (debug) System.out.println ("Final  : "+mag.toNeatString());

     mag.usedStations.setValue(mag.ampList.getStationUsedCount());
     mag.gap.setValue(mag.ampList.getMaximumAzimuthalGap());

  return mag;
    }

    /** */
    public void setResiduals (Magnitude mag) {

       Amplitude amp[]  = mag.ampList.getArray();

    for (int i=0; i<amp.length; i++) {
      amp[i].channelMag.setResidual( mag );
    }
    }


    /** Calculate the Magnitude using this Solution and waveforms. Creates a
     * new Solution.ampList and sets all
     * the Amplitude.ChannelMag values. Sets the new magnitude as the prefered one
     * and the old prefered is retained in the Solution's list of alternate mags.
     * Returns a reference to Solution.magnitude. Follows all the rules of
     * the MagnitudeMethod and checks the channel rejection criteria like maxDistance.
     * Does NOT assume the Waveform list is in any particular order, so if
     * maxChannels is set
     * only the first 'maxChannels" amps in the list will contribute to the
     * Magnitude. */
    public Magnitude solveFromWaveforms (Solution sol, Collection wfList) {

/*




This uses a WFlist but the pre and post filters of MagEng use an amp list in the
mag.





*/
  // make a fresh, blank magnitude object
  Magnitude newMag = getNewMag(sol);

     // pre-filter the list  NO GOOD, MAG HAS NO AMPS!
//     magMethod.preScanAmpList(newMag);

  // are there amps?
  if (wfList.isEmpty()) return newMag;

     // kill old amps
     //if (!reuseOldAmps) sol.ampList = new AmpList();

  // !!! have distances? Corrections?
  System.out.println("wflist.size() = "+ wfList.size());

  channelLookup(wfList);

  calcDistances(wfList, sol.getLatLonZ());

  Waveform wf[] = new Waveform[wfList.size()];
  wfList.toArray(wf);

  // make an array to hold the results for statistics later
  double magList[] = new double[wf.length];

     // inforce maxChannels
     int chanKnt = wf.length;
     if (chanKnt > magMethod.getMaxChannels()) {
         chanKnt = magMethod.getMaxChannels();  //  break out of loop
         if (debug) System.out.println ("Will trim to maxChannels limit = "+chanKnt);
     }

// Scan'em Dan-o

  Amplitude amp;

  for (int i=0; i<chanKnt; i++) {

// REJECT some channels up-front to save computation
         // check maxDistance & maxChannels criteria
         if ( wf[i].getDistance() > magMethod.getMaxDistance() ) {
            if (debug) System.out.println
              ("Hit maxDistance limit, skip channels beyond " + magMethod.getMaxDistance());
            break;  //  break out of loop
         }

         if ( !useChannel(wf[i].getChannelObj()) )  continue;  //  skip rest of loop

      // scan the time-series for an amp
   // ----
         amp = calcAmpFromWaveform(sol, wf[i]);
   // ----

         // save it if it was good
         if (amp != null && amp.channelMag != null) {

       // reject amp with low SNR if value is set
       if (magMethod.getMinSNR() != Double.MIN_VALUE &&
        (!amp.snr.isNull() &&
         amp.snr.doubleValue() < magMethod.getMinSNR())) {

       if (debug) System.out.println("Rejecting low SNR: "+
         amp.getChannelObj().toDelimitedSeedString(" ")+
         " snr= " + amp.snr);
       } else {

             // this assoc's amp with mag AND solution
             newMag.associate(amp);
          }

      }

  } // end of for loop

  // no contributing amps
  if (newMag.ampList.size() < 1) {
      newMag.setStale(false);
      return newMag;
  }

  //
  newMag = solve (newMag);

  // make the newMag the primary one for the solution (saves the old one)
  // This also associates newMag with the sol.
  sol.setPreferredMagnitude(newMag);

  return newMag;
    }

    /** Return 'true' if the channel should be scanned. */
    public boolean useChannel (Channel chan) {
        if (!magMethod.isIncludedComponent(chan)) {
           return false;
        } else {
           return true;
        }
    }
    /**
    * Scan the time-series in the waveform for a peak amplitude, then use that peak
    * to calculate the channel magnitude. If the MagnitudeMethod has a filter it
    * will be applied to the wavefrom before it is scanned. Returns null on failure,
    * e.g. no time-series.
    */
    public Amplitude calcAmpFromWaveform (Solution sol, Waveform wf) {

       boolean localLoad = false;

       if (wf == null) return null;

       // if there is no time-series, we load it here and unload it when done.
       if (!wf.hasTimeSeries())  {
          if (wf.loadTimeSeries()) {     // try to load it
            localLoad = true;
          } else {
            return null ;                // nothing loaded
          }
       }

       // convert/filter if appropreate
//       Waveform filteredWf;
//       if (magMethod.hasWaveformFilter()) {
//          filteredWf = magMethod.getWaveformFilter().filter(wf);
//       } else {
//          filteredWf = wf;
//       }
// Make a DEEP copy of the waveform so original unfiltered is preserved.
       Waveform fwf = Waveform.copy(wf);
       if (magMethod.hasWaveformFilter()) {
          //fwf = magMethod.wfFilter.filter(fwf);
          fwf = magMethod.getWaveformFilter().filter(fwf);
//       } else {
 //         filteredWf = wf;
       }

       // failed
       if (fwf == null) return null;

       Amplitude amp;

       // get peak in energy window
       if (scanEnergyWindow) {
			    // DK - OLD CODE REFERENCED "fwf" here instead of "wf"
          TimeSpan ts = wf.getEnergyTimeSpan(sol.datetime.doubleValue(),
                                             wf.getDistance() );
          amp = fwf.getPeakAmplitude(ts);
          if (debug) System.out.println ("Scan: "+fwf.getChannelObj().toDelimitedSeedString(" ")+
                               " windowsize= "+ts.getDuration());

       } else {         // get peak in whole waveform
          amp = fwf.getPeakAmplitude();
       }

       // amp can be 'null' if the energy window is outside the available time series
       if (amp == null) return null;

       // calc SNR
       TimeSpan tsp = fwf.getPreEnergyTimeSpan( sol.datetime.doubleValue(),
                                               fwf.getDistance() );
       float noiseLevel = fwf.scanForNoiseLevel(tsp);

       if (noiseLevel != Float.NaN) {
          amp.snr.setValue( Math.abs( amp.value.doubleValue() ) / noiseLevel );
       }

       // TODO: should check for lop-sided waveform

       // unload original time-series
       if (localLoad) wf.unloadTimeSeries();

       // destroy object
       fwf = null;

       return amp;
    }

    /**
     * Lookup channel info for the amps.
     */
/*    public void ampChannelLookup(Collection ampList) {

  Amplitude amp[] = new Amplitude[ampList.size()];
  ampList.toArray(amp);

     for (int i = 0; i<amp.length; i++) {
    amp[i].chan = channelLookup(amp[i].chan);  // go to dbase for each one
        }

    }
*/
    /**
     * Lookup channel info for the Waveforms.
     */
 /*   public void wfChannelLookup(Collection wfList) {

  Waveform wf[] = new Waveform[wfList.size()];
  wfList.toArray(wf);

     for (int i = 0; i<wf.length; i++) {
    wf[i].chan = channelLookup(wf[i].chan);
        }

    }
*/
    /**
     * Lookup channel info for the Waveforms.
     */
    public void channelLookup(Collection list) {

    if (list == null || list.isEmpty()) return;

    // Load channel list if that would be more efficient
    if (list.size() > 100 && (chanList == null || chanList.isEmpty()) ) {
    // TODO: this should be generalized, could use a list of components
    // that are specified to contribute to Mag but parsing wildcards is a problem.
       String compList[] = {"HH_", "E%", "HL_"};
        setChannelList(ChannelList.getByComponent(compList));

//       if (debug)
          System.out.println ("Channel list got "+chanList.size());
    }

    Object thing = ((ArrayList)list).get(0);

    if (thing instanceof Channelable) {
  Channelable ob[] = new Channelable[list.size()];
  list.toArray(ob);

     for (int i = 0; i<ob.length; i++) {
          if (ob[i] != null && !ob[i].getChannelObj().hasLatLonZ()) {
      Channel ch = ob[i].getChannelObj();
      ch = channelLookup(ch);
      ob[i].setChannelObj(ch);
          }
        }
     }

    }

    /**
     * Lookup channel info for the Waveforms.
     */
    public Channel channelLookup(Channel chan) {

     if (chanList != null) {                 // use local list if available
       Channel ch = chanList.lookUp(chan);
       if (ch != null) return ch;            // return it if sucessfull
     }                                       // if not fall thru to direct lookup

     return Channel.create().lookUp(chan);    // go to the data source
    }
    /**
     * Calculate distance for each channel to the event hypocenter. The distance
     * value is put in Amplitude.chan.distance.
     */
/*    public void calcDistances (Collection ampList, LatLonZ latLonZ) {
  Amplitude amp[] = new Amplitude[ampList.size()];
  ampList.toArray(amp);

  for (int i = 0; i<amp.length; i++) {
    amp[i].chan.calcDistance(latLonZ);
     }

    }
 */
    /**
     * Calculate distance for each channel to the event hypocenter. The objects
     * in the list must implement the Channelable interface. The distance
     * value is put in Object.chan.distance.
     */
    public void calcDistances (Collection list, double lat, double lon, double z) {

      calcDistances (list, new LatLonZ (lat, lon, z));

    }

    /**
     * Lookup channel info for the Waveforms.
     */
    public void calcDistances(Collection list, LatLonZ latLonZ) {

    Object thing = ((ArrayList)list).get(0);
    Channel chan;

    if (thing instanceof Channelable) {
      Channelable ob[] = new Channelable[list.size()];
      list.toArray(ob);

     for (int i = 0; i<ob.length; i++) {

          chan = ob[i].getChannelObj();
          // look up latlonz if there is none
          if (!chan.hasLatLonZ()) ob[i].setChannelObj(channelLookup(chan));
 //		ob[i].setChannel(channelLookup(ob[i].getChannelObj()));
          ob[i].setDistance(ob[i].getChannelObj().calcDistance(latLonZ));

        }
     }

    }

    // ///////////////////////////////////////////////////////////////////
    // test
    public static void main (String args[])
    {

     String host = "makalu";
//     String host = "k2";

     EnvironmentInfo.setNetworkCode("CI");
  EnvironmentInfo.setApplicationName("MagEng");
     EnvironmentInfo.setAutomatic(false);

  long evid = 0;

  if (args.length <= 0)	// no args
  {
    System.out.println ("Usage: java LocalMagnitudeEngine <evid> [<dbase>])");
    System.exit(0);

  } else {

    Integer val = Integer.valueOf(args[0]);	    // convert arg String
    evid = (int) val.intValue();

    if (args.length > 1) host = args[1];	    // dbase

  }

        System.out.println ("Making connection...");
	DataSource init = new TestDataSource();  // make connection

	init.setWriteBackEnabled(true);

     if (init == null) {
       System.exit(0);
     }

  System.out.println (" Getting evid = "+evid);

  Solution sol = Solution.create().getById(evid);

  System.out.println (" Getting amps for evid = "+evid);
  // Read amps for this event from the dbase
  //	ArrayList amp = (ArrayList)Amplitude.create().getBySolution(sol);
  //sol.magnitude.getAmps();
     String list[] = {"WAU" } ;   // get only W-A uncorrected amps
     sol.magnitude.addAmps(Amplitude.create().getByMagnitude(sol.magnitude, list));

  System.out.println ("Amp count = " + sol.magnitude.getAmpCount());
     System.out.println ("Sta count = "+ sol.magnitude.ampList.getStationCount());
  /*
  for (int i = 0; i<sol.ampList.size(); i++)
        {
      System.err.println (sol.ampList.get(i).toString());
  }
  */
  System.out.println (" Original mag for: "+sol.toString());
  System.out.println (sol.magnitude.toDumpString()) ;
  System.out.println ("") ;
  System.out.println (sol.magnitude.ampList.toNeatString());
  System.out.println ("") ;

  // Calc SoCal mag.
  System.out.println (" Calculating mag (SoCalML) for: "+evid);

  String DefaultMagMethodClass = "org.trinet.util.magnitudeengines.SoCalML";
  String MLmagMethodClass = DefaultMagMethodClass;
  String DefaultMagnitudeEngineClass = "org.trinet.util.magnitudeengines.LocalMagnitudeEngine";
  String magEngineClass = DefaultMagnitudeEngineClass;

  MagnitudeEngine magEng = MagnitudeEngine.CreateMagnitudeEngine(magEngineClass);
  magEng.ConfigureMagnitudeEngine(MagnitudeMethod.CreateMagnitudeMethod(MLmagMethodClass));

//	magEng.solve(sol.magnitude);
  sol.magnitude = magEng.solve(sol);

  System.out.println (sol.magnitude.toDumpString()) ;
  System.out.println ("") ;
  System.out.println (sol.magnitude.ampList.toNeatString());
/*
     System.out.println ("sta count = "+ sol.magnitude.ampList.getStationCount());
     System.out.println ("usedcount = "+ sol.magnitude.ampList.getStationUsedCount());
     System.out.println ("chn count = "+ sol.magnitude.ampList.size());
     System.out.println ("usedcount = "+ sol.magnitude.ampList.getChannelUsedCount());
*/
     // commit the changes

//.//     sol.magnitude.commit();

  DataSource.close();

    }

} // LocalMagnitudeEngine
