package org.trinet.util.gazetteer;
import java.io.*;
import java.util.Collection;

/** Utility used by client applications to obtain the closest geographic types relative to a specified reference point.
*  Example usage:
<PRE>
	.
	.
	.

	WhereIsEngine wicf = WhereIsEngine.CreateWhereIsEngine("org.trinet.util.gazetteer.WhereIsClosestFrom");

	wicf.setReference(34.4801,-119.5104, 0.); // need to set source point to measure to relative to database data.

	wicf.setDistanceUnits(GeoidalUnits.MILES); // sets these units for only for where methods.
	System.out.println(wicf.where(34.4801,-119.5104));
	System.out.print(wicf.whereType(34.3103, -117.6218, 0., "town"));

	WhereSummaryItem wsi= wicf.getClosestTown();
	System.out.println(wsi.fromWhereString(GeoidalUnits.KILOMETERS, false));
	DistanceAzimuthElevation dae = wsi.getDistanceAzimuthElevation();
	System.out.println("distance: " + dae.getDistance(GeoidalUnits.MILES) + " miles.");

	.
	.
	.

	wicf.closeStatements(); // release database resource when done.
</PRE>
*/
public abstract class WhereIsEngine
{

  protected static String sWhereEngineClassName = null;


  /* Real Construction Method */
  public static WhereIsEngine CreateWhereIsEngine(String sClassName)
  {
            WhereIsEngine newWhereIsEngine = null;

        try {
            newWhereIsEngine =  (WhereIsEngine)Class.forName(sClassName).newInstance();
  		    sWhereEngineClassName = new String(sClassName);
        }
        catch (ClassNotFoundException ex) {
            ex.printStackTrace();
        }
        catch (InstantiationException ex) {
            ex.printStackTrace();
        }
        catch (IllegalAccessException ex) {
            ex.printStackTrace();
        }

        return newWhereIsEngine;
  }


  public static WhereIsEngine CreateWhereIsEngine()
  {
    if(sWhereEngineClassName == null)
	  return(WhereIsEngine.CreateWhereIsEngine("(Unknown)"));
	else
	  return(WhereIsEngine.CreateWhereIsEngine(sWhereEngineClassName));
  }

/** GeoidalUnits.MILES == distances are reported in miles in the where(...) method output text strings.
* GeoidalUnits.KILOMETERS (default) == distances reported in kilometers in the output text strings. */
    public abstract void setDistanceUnits(GeoidalUnits units) ;

    public void setDistanceUnitsKm() {
	setDistanceUnits(GeoidalUnits.KILOMETERS);
    }
    public void setDistanceUnitsMiles() {
	setDistanceUnits(GeoidalUnits.MILES);
    }

/** Returns String describing where the specified WhereIs is relative to closest database entry for all known database types. */
    public String where(double lat, double lon) {
	return where(lat, lon, 0.);
    }
/** Returns String describing where the specified WhereIs (int degrees and decimal minutes) is relative to the closest
* database entry for all known types.  Uses z to determine elevation.
*/
    public String where(int lat, double latmin, int lon, double lonmin, double  z) {
	return where(GeoidalConvert.toDecimalDegrees(lat, latmin), GeoidalConvert.toDecimalDegrees(lon, lonmin),
		z);
    }
/** Returns String describing where the specified WhereIs is relative to closest database entry for all known database types.
* Uses z to determine elevation.
*/
    public abstract String where(double lat, double lon, double z);

/** Returns String describing where the specified WhereIs is relative to the closest database entry
* for the specified input database type. Uses z to determine elevation.
*/
    public abstract String whereType(double lat, double lon, double z, String type);

/** Returns String describing where the specified WhereIs is relative to closest database entry for all known database types.
* Uses z to determine elevation.
*/
    public abstract String where();

/** Returns String describing where the specified WhereIs is relative to the closest database entry
* for the specified input database type. Uses z to determine elevation.
*/
    public abstract String whereType(String type);

/** Returns a Collection of WhereSummarys describing closest database entries for all known database types,
 *  from the current point of reference
*/
    public abstract Collection WhereSummary();

/** Returns a WhereSummary describing the closest database entry for the given type,
 *  from the current point of reference.
*/
    public abstract WhereSummary WhereSummaryType(String type);


/** Sets this object's geographic reference source point.
*/
    public void setReference(Geoidal latLonZ)  {
	setReference(latLonZ.getLat(), latLonZ.getLon(), latLonZ.getZ());
    }
/** Sets this object's geographic reference source point.
*/
    public void setReference(double lat, double lon)  {
	setReference(lat, lon, 0.);
    }
/** Sets this object's geographic reference source point.
*   Distance, azimuth and elevation are calculated from the Gazetteer database points to this reference point.
*/
    public abstract void setReference(double lat, double lon, double z);

/** Closes all the JDBC callable statements used by this object.*/
    public void closeStatements() {
   }

}
