package org.trinet.jiggle;

import java.applet.*;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import java.text.*;
import javax.swing.border.*;
import javax.swing.event.*;

import org.trinet.util.*;
import org.trinet.jasi.Waveform;

/**
 * This is a panel that supports zooming and scrolling interface functions. The
 * waveform is displayed in a ZoomableWFPanel in a BombSightViewport in a
 * JScrollPane.  There is also a button panel with scaling buttons and a label
 * panel with channel and status info.  This component does not support phase
 * picking. The PickingPanel class extends this class for that purpose. <p>
 *
 * If the view object has focus this component responds to the default keyboard actions
 * for JScrollPane as defined in the Java Look and Feel:<p>
 <tt>
JScrollPane (Java L&F) (Navigate refers to focus)
Navigate out forward      |  Tab
Navigate out backward     |  Shift+Tab
Move up/down              |  Up, Down
Move left/right           |  Left, Right
Move to start/end of data |  Ctrl+Home, Ctrl+End
Block move up/down        |  PgUp, PgDn
Block move right          |  Ctrl+PgDn
Block move left           |  Ctrl+PgUp
</tt>

/*
 The ZoomPanel contains several other nested graphical components:

 ZoomPanel

	labelBox  (JPanel)

	scrollZoom (JScrollPane)

		vport (BombSightViewport)

			wfp  (ZoomableWFPanel)

	buttonPanel (JPanel)
*/

public class ZoomPanel extends JPanel {

    JLabel staLabel;

    public int width;
    public int height;
    public int UsableHeight;
    public int UsableWidth;
    public int BorderLeft;
    public int BorderRight;
    public int BorderTop;
    public int BorderBottom;

    Color WFColor         = Color.blue;
    Color TextColor       = Color.black;
    Color BackgroundColor = Color.white;

    Color BombSightColor  = Color.gray;

    /** The ZoomableWFPanel that shows through the viewport. */
    public ZoomableWFPanel zwfp;
    TimeColumnHeader timeColumnHeader;

    /** The master view of what's in the GUI */
    public MasterView mv;

    /** This extends JViewport and overrides its paint method to draw a
        bombsight for picking. */
    BombSightViewport vport = new BombSightViewport(BombSightColor) ;
    JScrollPane scrollZoom;

    /** Model of MVC for cursorLocLabel */
    CursorLocModel cursorLocModel = new CursorLocModel();
    CursorLocPanel cursorPanel = new CursorLocPanel(cursorLocModel);

    /** Lable for channel and status info */
    JPanel labelBox = new JPanel();  //top panel with sta label and cursor info

// event handling stuff
    private Object objSelected;     // the menu, list or button item selected
    private String EventName;	    // String label of the event (e.g. "Exit")
    private String ItemName;

    /** If true, allow ZoomPanel scroll bars to scroll the window */
    protected boolean scrollable = true;
    protected boolean filterEnabled = false;

    /** Format for distance in channel label: "####0.0"*/
    DecimalFormat df = new DecimalFormat("####0.0");
    DecimalFormat di = new DecimalFormat("#######0");
    DecimalFormat dsci = new DecimalFormat("0.##E0");

    /** Handles async. loading of waveforms */
    ActiveWFLoadListener activeWFLoadListener = new ActiveWFLoadListener();

    /** Debug verbosity flag */
//    boolean debug = true;
    boolean debug = false;

    ImageIcon filterIcon = null;
    ImageIcon unfilterIcon = null;

    JToggleButton filterButton;

//-----------------------------------------------------------------
 public ZoomPanel () {		 //constructor
 }

    /*
     * Create a ZoomPanel based on information in this MasterView. This is only
     * needed to support the red up/down buttons. They need the WFViewList in
     * the master view to know the previous/next WFView.  */
public ZoomPanel (MasterView masterView) {		 //constructor
    mv = masterView;
    buildPanel();
}

/** Builds the whole panel. */

protected void buildPanel() {

   // listener updates the channel label when the selected WFView changes.
   //NOTE: must be added BEFORE makeScrollZoom() makes the ZoomableWFPanel
   // so it will be called AFTER ZoomableWFPanel reacts to WFViewModel changes
   // (listeners are called in the opposite order they are added)
    mv.masterWFViewModel.addChangeListener(new WFViewListener());

// icons
    try {
      filterIcon =
          new ImageIcon(IconImage.getImage("filter16.gif"));
      unfilterIcon =
          new ImageIcon(IconImage.getImage("unfilter16.gif"));
    } catch (Exception ex) {
      System.out.println ("ZoomPanel could not find icon image files.");
    }

    setOpaque(true);

    setLayout( new BorderLayout() );

// create the "North" main label

    labelBox = makeInfoBar();

    add("North", labelBox);		// add it to the ZoomPanel


    add("Center", makeScrollZoom());

// <> create the button panel

    add("South", makeButtonPanel());		// add it to the ZoomPanel

    // arrows to move to next/previous time-series
    //add("West", makeArrowPanel());
    add("East", makeArrowPanel());

    //addKeyboardActions ();

//    this.addKeyListener(new KeyHandler());

} // end of builder

public boolean isFocusTraversable() { return true;}

/** Arrow keys were not used because they are used to shift focus between components.
 *  (And because I couldn't get them to work.) */
/* public void addKeyboardActions () {

//  final int RIGHTKEY = KeyEvent.VK_RIGHT;
//  final int LEFTKEY  = KeyEvent.VK_LEFT;
//  final int RIGHTKEY = KeyEvent.VK_KP_RIGHT;
//  final int LEFTKEY  = KeyEvent.VK_KP_LEFT;
  final int RIGHTKEY = KeyEvent.VK_PERIOD;
  final int LEFTKEY  = KeyEvent.VK_COMMA;

// < KEY
  this.registerKeyboardAction(
    new ActionListener (){
       public void actionPerformed (ActionEvent e) {
	  JScrollBar bar = scrollZoom.getHorizontalScrollBar();
	  int inc = zwfp.getScrollableUnitIncrement(vport.getViewRect(),
				bar.getOrientation(), SwingConstants.LEFT);
          int val = Math.max(bar.getMinimum(), bar.getValue() - inc);
	  bar.setValue(val);
       }
    },
    KeyStroke.getKeyStroke(LEFTKEY, 0, false),
    JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
// shift-<
  this.registerKeyboardAction(
    new ActionListener (){
       public void actionPerformed (ActionEvent e) {
	  JScrollBar bar = scrollZoom.getHorizontalScrollBar();
	  int inc = zwfp.getScrollableBlockIncrement(vport.getViewRect(),
				bar.getOrientation(), SwingConstants.LEFT);
          int val = Math.max(bar.getMinimum(), bar.getValue() - inc);
	  bar.setValue(val);
       }
    },
    KeyStroke.getKeyStroke(LEFTKEY,  InputEvent.SHIFT_MASK, false),
    JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);

// > KEY
// left arrow
  this.registerKeyboardAction(
    new ActionListener (){
       public void actionPerformed (ActionEvent e) {
	  JScrollBar bar = scrollZoom.getHorizontalScrollBar();
	  int inc = zwfp.getScrollableUnitIncrement(vport.getViewRect(),
				bar.getOrientation(), SwingConstants.RIGHT);
          int val = Math.min(bar.getMaximum(), bar.getValue() + inc);
	  bar.setValue(val);
       }
    },
    KeyStroke.getKeyStroke(RIGHTKEY, 0, false),
    JComponent.WHEN_IN_FOCUSED_WINDOW);

// shift->
  this.registerKeyboardAction(
    new ActionListener (){
       public void actionPerformed (ActionEvent e) {
	  JScrollBar bar = scrollZoom.getHorizontalScrollBar();
	  int inc = zwfp.getScrollableBlockIncrement(vport.getViewRect(),
				bar.getOrientation(), SwingConstants.RIGHT);
          int val = Math.min(bar.getMaximum(), bar.getValue() + inc);
	  bar.setValue(val);
       }
    },
    KeyStroke.getKeyStroke(RIGHTKEY, InputEvent.SHIFT_MASK, false),
    JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
}
*/

/** Construct the JScrollPane and the underlying ZoomableWFPanel that will
    show through the viewport. */
// This is split into two steps to allow customizing by extending classes
    protected JScrollPane makeScrollZoom () {

       	zwfp = new ZoomableWFPanel (makeScrollZoom1(), mv);
          return setupZoomableWFPanel();
    }

    /**
     * First part of making the JScrollPane.
     */
    protected JScrollPane makeScrollZoom1 () {

// <> create the ScrollPane
	scrollZoom = new JScrollPane(
			 ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
			 ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);

	// Replace the default viewport of the scrollPane with ours
	scrollZoom.setViewport(vport);
        // deprecated in v1.3
	//vport.setBackingStoreEnabled(true);   // speeds scrolling

     // put filter button in lower right corner of the scroll panel
     filterButton = new JToggleButton(filterIcon);
     filterButton.setPressedIcon(unfilterIcon);
     filterButton.setToolTipText("Toggle filter on/off");
     filterButton.addActionListener(new FilterButtonHandler());
     filterButton.setEnabled(getFilterEnabled());

     scrollZoom.add(filterButton, ScrollPaneConstants.LOWER_RIGHT_CORNER);

     return scrollZoom;

     }

	// Set characteristics of the ZoomableWFPanel
	// THIS MUST BE DONE AFTER THE 'ScrollZoom.setViewport(vport)' ABOVE
	////zwfp = new ZoomableWFPanel (scrollZoom, mv);

     protected JScrollPane setupZoomableWFPanel() {

	zwfp.setCursorLocModel(cursorLocModel);

	 // set the default filter
	 zwfp.setFilter(new GeneralButterworthFilter(GeneralButterworthFilter.HIGHPASS, true));

	 zwfp.setJScrollPane(scrollZoom);

	    // set scrolling time scale in column header
	 timeColumnHeader = new TimeColumnHeader(zwfp) ;
	 scrollZoom.setColumnHeaderView(timeColumnHeader);

	// ** Had turned this off because of recursion problem with scroller
        if (scrollable) {	// control some of the scroll pane's behavior

     	// set scrolling increments
	  JScrollBar vBar = scrollZoom.getVerticalScrollBar();
	  JScrollBar hBar = scrollZoom.getHorizontalScrollBar();

	// Add  adjustment listener to both
//     if (true) {
        vBar.addAdjustmentListener(new VerticalScrollListener());
        hBar.addAdjustmentListener(new HorizontalScrollListener());
/*     } else {
        ScrollListener scrollListener = new ScrollListener();
        vBar.addAdjustmentListener(scrollListener);
        hBar.addAdjustmentListener(scrollListener);
     }
*/
     }
     return scrollZoom;
    }

    /**
     * Make the panel with the up/down buttons for selecting next/prevous
     * WFPanel.
     */

    protected Box makeArrowPanel () {

        JButton btn;

	   Box box = Box.createVerticalBox();

	   box.add(Box.createGlue());

	   Image image = IconImage.getImage("RedUp3.gif");

        // arrow
        if (image == null) {       // handle situation when .gif file can't be found
 	     	 btn = new JButton("^");
         } else {
	      	 btn = new JButton(new ImageIcon(image));
         }
	       btn.addActionListener (new ButtonHandler(ButtonHandler.TOP) );
	       btn.setToolTipText("Select top waveform");
	       btn.setMargin(new Insets(0, 0, 0, 0));
	       box.add(btn);
	       box.add(Box.createVerticalStrut(5));

     // arrow
	image = IconImage.getImage("RedUp2.gif");
     if (image == null) {
 	   btn = new JButton("^");
     } else {
   	  btn = new JButton(new ImageIcon(image));
     }
	btn.addActionListener (new ButtonHandler(ButtonHandler.PAGEUP) );
	btn.setToolTipText("Jump up a page");
	btn.setMargin(new Insets(0, 0, 0, 0));
	box.add(btn);

	box.add(Box.createVerticalStrut(5));

     // arrow
	image = IconImage.getImage("RedUp.gif");
     if (image == null) {
 	  btn = new JButton("^");
     } else {
   	  btn = new JButton(new ImageIcon(image));
     }
	btn.addActionListener (new ButtonHandler(ButtonHandler.UP) );
	btn.setToolTipText("Select previous waveform");
	btn.setMargin(new Insets(0, 0, 0, 0));
	box.add(btn);

	box.add(Box.createVerticalStrut(5));

    // arrow
	image = IconImage.getImage("RedDown.gif");
     if (image == null) {
 	  btn = new JButton("v");
     } else {
   	  btn = new JButton(new ImageIcon(image));
     }
	btn.addActionListener (new ButtonHandler(ButtonHandler.DOWN) );
	btn.setToolTipText("Select next waveform");
	box.add(btn);
	btn.setMargin(new Insets(0, 0, 0, 0));

	box.add(Box.createVerticalStrut(5));

     // arrow
	image = IconImage.getImage("RedDown2.gif");
     if (image == null) {
 	  btn = new JButton("v");
     } else {
   	  btn = new JButton(new ImageIcon(image));
     }
	btn.addActionListener (new ButtonHandler(ButtonHandler.PAGEDOWN) );
	btn.setToolTipText("Jump down a page");
	box.add(btn);
	btn.setMargin(new Insets(0, 0, 0, 0));
	box.add(Box.createVerticalStrut(5));

     // arrow
	image = IconImage.getImage("RedDown3.gif");
     if (image == null) {
 		 btn = new JButton("v");
     } else {
   	      btn = new JButton(new ImageIcon(image));
     }
	btn.addActionListener (new ButtonHandler(ButtonHandler.BOTTOM) );
	btn.setToolTipText("Select bottom waveform");
	box.add(btn);
	btn.setMargin(new Insets(0, 0, 0, 0));
	box.add(Box.createGlue());

	return box;
    }

    /**
    *  Define motion buttons. Can't do in ButtonHandler because you only do
    * statics in top level classes.
    */
    interface MotionButtons {

	static final int TOP    = 0;
	static final int UP     = 1;
	static final int PAGEUP = 2;
	static final int CENTER = 3;
	static final int PAGEDOWN = 4;
	static final int DOWN   = 5;
	static final int BOTTOM = 6;
    }
    /**
     * Handle next/prevous buttons actions. This allows control of the selected WFView.
     */
    class ButtonHandler implements ActionListener, MotionButtons {


	int action;		// defines what action this listener should take.

	/**
	 * Create the button listener and define what action this listener should take
	 * based on the value of 'type'. Thus, on class can service all actions.
	 */
	public ButtonHandler (int type) {

	    action = type;

	}

	public void actionPerformed (ActionEvent evt) {

	    Object source = evt.getSource();

	    switch (action) {

	    case TOP:
  		mv.masterWFViewModel.selectFirst(mv.wfvList);
          mv.masterWFWindowModel.setFullAmp(zwfp.getWf());
		break;

	    case PAGEUP:
  		mv.masterWFViewModel.pageUp(mv.wfvList, false);
          mv.masterWFWindowModel.setFullAmp(zwfp.getWf());
		break;

	    case UP:
		mv.masterWFViewModel.selectPrevious(mv.wfvList);
          mv.masterWFWindowModel.setFullAmp(zwfp.getWf());
		break;

	    case CENTER:
		break;

	    case DOWN:
		mv.masterWFViewModel.selectNext(mv.wfvList);
          mv.masterWFWindowModel.setFullAmp(zwfp.getWf());
		break;

	    case PAGEDOWN:
  		mv.masterWFViewModel.pageDown(mv.wfvList, false);
          mv.masterWFWindowModel.setFullAmp(zwfp.getWf());
		break;

	    case BOTTOM:
   		mv.masterWFViewModel.selectLast(mv.wfvList);
          mv.masterWFWindowModel.setFullAmp(zwfp.getWf());
		break;

	    }

	}

    }
    /**
     * Build the "north" bar of the panel display.
     */
    protected JPanel makeInfoBar() {

	JPanel jp = new JPanel();

	GridBagLayout layout = new GridBagLayout();
	jp.setLayout(layout);
	GridBagConstraints GB  = new GridBagConstraints();
	// define a generic layout object that we'll reuse for diff. comps^M
	GB.weighty = 100;
	GB.fill    = GridBagConstraints.BOTH;
	GB.anchor  = GridBagConstraints.NORTHWEST;
	GB.gridx   = GridBagConstraints.RELATIVE;
	GB.gridy   = 0;

	// The staLabel has file scope because it is changed by another method
	staLabel = new JLabel("Station Info here...");
	staLabel.setBackground(Color.black);
	staLabel.setForeground(Color.blue);

	//
	GB.weightx = 60;		// 60% of bar
	layout.setConstraints(staLabel, GB);
	jp.add (staLabel);

	// cursor position labels
	GB.weightx = 40;			// 40% of bar
	layout.setConstraints(labelBox, GB);
	jp.add(cursorPanel);

	return jp;
}

//-----------------------------------------------------------------
// Make the button panel
    protected JPanel makeButtonPanel () {

	JPanel bp = new JPanel();
	// JToolBar bp = new JToolBar();


// parameters describing the Button panel's grid layout
     final int GridRows = 1;  // either rows or columns
	final int GridCols = 0;  // should be = 0
     final int Vgap = 10;
	final int Hgap = 10;

    // define the grid layout object
	GridLayout LayOut = new GridLayout(GridRows, GridCols, Vgap, Hgap);

	bp.setLayout(LayOut);

// 1st row
	//    addButton(bp, new JButton("Next"), "Jump to next waveform");
	//    addButton(bp, new JButton("Previous"), "Jump to Previous waveform");
      addButton(bp, new JButton("<-->"), "Expand time scale x2");
      addButton(bp, new JButton("-><-"), "Compress time scale x2");
      addButton(bp, new JButton("Full View"), "Return to full seismogram view");

// 2nd row
      addButton(bp, new JButton("v^"), "Expand amplitude scale x2");
      addButton(bp, new JButton("X"), "Compress amplitude scale x2");

// <> create a zoom viewport scale combobox. Values are seconds for width of viewport.

      bp.add(new JLabel("Sec/view->"));
      bp.add(new ZoomScaleComboBox(mv));

      return bp;
    }

/*-----------------------------------------------------------------
 *  method to streamline adding of buttons
 */
     void addButton (Container cont, JButton btn)
    {
      btn.addActionListener (new ActionHandler() );
      cont.add(btn);
    }
/*-----------------------------------------------------------------
 *  method to stream line adding of buttons, with tooltips
 */
     JButton addButton (Container cont, JButton btn, String tip) {
      btn.addActionListener (new ActionHandler() );
      btn.setToolTipText(tip);

      cont.add(btn);

      return btn;
    }

    /** Set Filtering enabled. Enables button in lower-right corner of the
    * scroller that toggles waveform filtering on/off. By default a Butterworth
    * highpass filter is used. A different filter can be set by calling
    * zwfp.setFilter(FilterIF). */
    public void setFilterEnabled (boolean tf) {

       filterEnabled = tf;
       if (filterButton != null) filterButton.setEnabled(filterEnabled);
    }

    /** Returns true if the filter button is enabled. */
    public boolean getFilterEnabled () {
       return filterEnabled;
    }

/** Handle filter button actions. */
class FilterButtonHandler implements ActionListener {

	public void actionPerformed (ActionEvent evt) {

         // toggle state
         zwfp.setShowAlternateWf(!zwfp.getShowAlternateWf());

         zwfp.repaint();
         repaint();

         makeChannelLabel();

     }

}

/**
 * Make the Label describing the channel.
 */
  void makeChannelLabel () {

      String header = "";

       if (zwfp == null || zwfp.wfv == null) {
         header = "No View Data.";
       } else {

         header = zwfp.wfv.getChannelObj().toDelimitedString(' ');

         if (zwfp.wfv.hasTimeSeries()) {
           Waveform wfx = zwfp.getWf();  // get selected waveform

           DecimalFormat fmt = di;
           if (Math.abs(wfx.getMaxAmp()) < 1.0) fmt = dsci;  // Sci. Notatation for small numbers

           header +=
               " bias=" + fmt.format((int) wfx.getBias()) +
               " max= " + fmt.format(wfx.getMaxAmp()) +
               " min= " + fmt.format(wfx.getMinAmp()) +
               " dist= "+ df.format(zwfp.wfv.getDistance());
         } else {
           header += " No time series.";
         }
       }

       staLabel.setText(header);   //should repaint automatically

    }

////
    /** INNER CLASS: Change events are expected from Waveform objects when the
     *  timeseries is loaded. */
    class ActiveWFLoadListener implements ChangeListener {

	public void stateChanged (ChangeEvent changeEvent) {

         setWFView(mv.masterWFViewModel.get());
	    //makeChannelLabel();

	}
    }
////

    /** Return the ZoomableWFPanel */
    public ZoomableWFPanel getActiveWFPanel() {
	return  zwfp;
    }

    /** */
    void setWFView (WFView wfv) {

         makeChannelLabel();

         // set the cursor location panel's amp style
         if (wfv == null) return;

         Waveform wf = zwfp.getWf();

         if (wf != null) {
            synchronized (wf) {
	         cursorPanel.setAmpFormat(wf.getAmpUnits(), wf.getMaxAmp());

	         if (!wf.hasTimeSeries())  {
                // remove old one then add again, else might register multi. times
                wf.removeChangeListener(activeWFLoadListener);
                wf.addChangeListener (activeWFLoadListener);
              }
            }
         }

    }

/** INNER CLASS: Handle changes to the selected WFView. */
      class WFViewListener implements ChangeListener {

         public void stateChanged (ChangeEvent changeEvent) {

	    WFView wfv =  mv.masterWFViewModel.get();
	    setWFView (wfv);

	    }
      }  // end of WFViewListener

/** INNER CLASS: Handle changes to the selected time/amp window. */
      class WFWindowListener implements ChangeListener {
     	public void stateChanged (ChangeEvent changeEvent) {
	    // noop
          }
      }

/**
 * Adjustment events are triggered by BOTH mouse down and mouse up.
 * They are also fired by a frame resize but the 'value' is 0.
 * The object here is to keep the masterWFViewWindowModel in synch with
 * the viewport area.
 */
 class VerticalScrollListener implements AdjustmentListener {

       WFSelectionBox viewBox = new WFSelectionBox();   //viewport box
       Rectangle vrect;
       int oldValue = 0;
       int newValue = 0;

       public void adjustmentValueChanged (AdjustmentEvent e) {

	   if (zwfp.wfv == null) return;	// empty panel

	   newValue = e.getValue();	// the "change" value (= pixels scrolled)
	   if (newValue == 0) return;	// ignore resize events

        // the following prevents recursive thrash
        // rounding error in conversion from amp to pixels can cause a difference
        // of one pixel in the value even if there was no change
        if (Math.abs(newValue - oldValue) <= 1) return;   // ignore redundant events
//        if (newValue == oldValue) return;   // ignore redundant events

        oldValue = newValue;

	   // get current JViewport rectangle
        // NOTE: the viewport has already been adjusted by the underlying component!
        // So just synch masterWFWindowModel with current viewport.
	   vrect = vport.getViewRect();

        double centerAmp = zwfp.ampOfPixel(vrect.y + (vrect.height/2));
//        double centerAmp = zwfp.ampOfPixel(vrect.height/2);
// THIS FIRES WFV EVENT!!!!!!!!!!!!!!!!!!!!!!!!!!!!1
        mv.masterWFWindowModel.setCenterAmp(zwfp.getWf(), (int) Math.round(centerAmp) );
        //mv.masterWFWindowModel.setCenterAmp( (int) centerAmp);

       }

 }  // end of VerticalScrollListener

/*
 class ScrollListener implements AdjustmentListener {

       WFSelectionBox vpBox = new WFSelectionBox();   //viewport box

       public void adjustmentValueChanged (AdjustmentEvent e) {

              vpBox = zwfp.getVisibleBox();

              mv.masterWFWindowModel.set(vpBox);
       }
 }
*/
/**
 * Adjustment events are triggered by BOTH mouse down and mouse up.
 * They are also fired by a frame resize but the 'value' is 0.
 */
 class HorizontalScrollListener implements AdjustmentListener {

       WFSelectionBox viewBox = new WFSelectionBox();   //viewport box
       Rectangle vrect;
       int oldValue = 0;
       int newValue = 0;

       public void adjustmentValueChanged (AdjustmentEvent e) {

	   if (zwfp.wfv == null) return;	// empty panel

	   int newValue = e.getValue();	// the "change" value (= pixels scrolled)
	   if (newValue == 0) return;	// ignore resize events

        // the following prevents recursive thrash
        if (newValue == oldValue) return;

        oldValue = newValue;

	   // get current JViewport rectangle
	   vrect = vport.getViewRect();

        double centerTime = zwfp.dtOfPixel(vrect.x + (vrect.width/2));

        mv.masterWFWindowModel.setCenterTime(centerTime);
       }

 }  // end of HorizontalScrollListener

//-----------------------------------------------------------------
// Event handling
//-----------------------------------------------------------------

// Handle button and menu events
class ActionHandler implements ActionListener {
	private int EventNo;	// a unique # for each item (not used)

	ActionHandler (int number)	//constructor
	{
	  EventNo = number;
	}

	ActionHandler ()			// alternate constructor
	{
	  EventNo = 0;
	}
//------------------------------------------------------------------
    // actual event handling happens here (we're using the Java 1.1 model)
	public void actionPerformed (ActionEvent evt) {
        // save and make public the string of the menu item selected
          objSelected = evt;
	  EventName =  evt.getActionCommand();

// *** if button push
	if (EventName == "<-->")
	{
	    zwfp.setScale(2.0, 1.0);
	}
	if (EventName == "-><-")
	{
	    zwfp.setScale(0.5, 1.0);
	}

	if (EventName == "v^")
	{
	    zwfp.setScale(1.0, 2.0);
	}
	if (EventName == "X")
	{
	    zwfp.setScale(1.0, 0.5);
	}
	if (EventName == "Full View")
	{
	    // notify selection model that the panelbox changed
	    mv.masterWFWindowModel.set(zwfp.getWf(), zwfp.dataBox);
	}

// repaint the panel to update labels, etc. Otherwise, you get PARTIAL repaint
// when you zoom.

//	System.out.println ("repaint from ZoomPanel.actionPerformed...");

// DK/JRR 032603 bug - remove repaint --       repaint();

	}	    // end of actionPerformed

    }	// end of ActionHandler class

} // end of ZoomPanel class


