package org.trinet.jasi.TN;

import org.trinet.jasi.*;

import java.util.*;
import java.sql.*;

import org.trinet.jdbc.*;
import org.trinet.jdbc.datatypes.*;
import org.trinet.jdbc.table.*;
import org.trinet.util.*;

/**
 * A concrete Amplitude object represents an peak amplitude reading made by any
 * number of techniques.
 *
 *
 * Created: Thu May  4 17:05:15 2000
 *
 * @author Doug Given
 * @version */

public class AmplitudeTN extends Amplitude {

    /** the orid of association */
    protected DataLong orid   	= new DataLong();

    /** the ampid */
    protected DataLong ampid   	= new DataLong();

    /** True if this Amp was orginally read from or has been written to the dbase.
     Used to know if a commit requires an 'insert' or an 'update' */
    boolean fromDbase = false;

    /** True if this Amp's association was orginally read from the dbase. Used to know if
  a commit requires an 'insert' or an 'update' */
    boolean assocFromDbase = false;

    /** The standard start to most SQL queries, to get amps associated with an
        event, joins Amp & AssocAmO tables */
    static String sqlPrefixByEvent = "Select "+
  Amp.QUALIFIED_COLUMN_NAMES+ ","+
  AssocAmO.QUALIFIED_COLUMN_NAMES+
  " from Amp, AssocAmO where (Amp.ampid = AssocAmO.ampid) ";

    /** The standard start to most SQL queries, to get amps associated with a
        magnitude, joins Amp & AssocAmM tables */
    static String sqlPrefixByMag = "Select "+
  Amp.QUALIFIED_COLUMN_NAMES+ ","+
  AssocAmM.QUALIFIED_COLUMN_NAMES+
  " from Amp, AssocAmM where (Amp.ampid = AssocAmM.ampid) ";

    /** The standard start to most SQL queries, to get amps in a time window */
    static String sqlPrefixByTime = "Select "+
  Amp.QUALIFIED_COLUMN_NAMES+
  " from Amp ";

    /** order so channels appear together in the list. */
    static String sqlOrderSuffix = " order by net, sta ";

    static final int Type_AssocAmO = 1;
    static final int Type_AssocAmM = 2;

    /** Holds value describing if this is amp lookup is done with a join to the
     * AssocAm or AssocAmM table */
    static int joinType;

    boolean debug = false;
    //boolean debug = true;

    public AmplitudeTN() {

    }

// Concrete methods from Amplitude ---------------------------------

    /**
     * Extract from DataSource all amps associated with this Magnitude. This
     * does a join using magid and the AssocAmM link table.  */
    public AmpList getByMagnitude (Magnitude mag) {
           return getByMagnitude (mag, null);
    }

    /**
     * Extract from DataSource all amps associated with this Magnitude. This
     * does a join using magid and the AssocAmM link table.  */
    public AmpList getByMagnitude (Magnitude mag, String[] ampTypeList) {

  String sql = sqlPrefixByMag +
      " and (AssocAmM.magid = "+ mag.magid.toString()+")";

        sql += makeTypeClause(ampTypeList) + sqlOrderSuffix;

  joinType = Type_AssocAmM;
  AmpList lst = (AmpList) AmplitudeTN.getBySQL(sql);

  //System.out.println ("getByMagnitude: amps found ="+lst.size());

  // associate all found amps with this solution
  for (int i = 0; i<lst.size(); i++) {
      ((Amplitude)lst.get(i)).associateMag(mag);
  }

  return lst;
    }
    /**
     * Extract from DataSource all amps associated with the Magnitude of this
     * event, described by an id number. Returns null if there is no such
     * id. This does a join using the AssocAmM link table.  */
    public AmpList getByMagnitude (long id) {
           return getByMagnitude (id, null);
    }

    /**
     * Extract from DataSource all amps associated with the Magnitude of this
     * event, described by an id number. Returns null if there is no such
     * id. This does a join using the AssocAmM link table.  */
    public AmpList getByMagnitude (long id, String[] ampTypeList) {

  // must first  retreive the Solution, then look up prefor for this id
  Solution tsol = Solution.create().getById(id);

  if (tsol == null) return null;	// no such solution in DataSource

  return getByMagnitude(tsol.magnitude, ampTypeList);
    }

    /**
     * Extract from DataSource all amps associated with this Solution. This does
     * a join using the AssocAmO link table.  */
    public AmpList getBySolution (Solution tsol) {
        return getBySolution(tsol, null);
    }
    /**
     * Extract from DataSource all amps associated with this Solution. This does
     * a join using the AssocAmO link table.  */

    public AmpList getBySolution (Solution tsol, String[] ampTypeList) {

  long orid = ((SolutionTN)tsol).getOrid();
  String sql = sqlPrefixByEvent +
      " and (AssocAmO.orid = "+ orid +")";

        sql += makeTypeClause(ampTypeList) + sqlOrderSuffix;

  joinType = Type_AssocAmO;
  AmpList lst = AmplitudeTN.getBySQL(sql);

  // associate all found amps with this solution
  for (int i = 0; i<lst.size(); i++) {
      ((Amplitude)lst.get(i)).associate(tsol);
  }

  return lst;
    }

    /** Return the SQL clause to get amps only of the types in the list.
    * For example given String list[] = {"PGA", "PGD" }  this will return
    * " and ((amptype like 'PGA') or (amptype like 'PGD') )*/
    String makeTypeClause (String[] ampTypeList) {

           String str = "";
           if (ampTypeList == null || ampTypeList.length == 0) return str;
           str += " and (";
           for (int i = 0; i < ampTypeList.length; i++) {
               if (i > 0) str += " or ";
               str += "(Amp.amptype like '" + ampTypeList[i] + "')";
           }
           str += ")";

 //          System.out.println ( "makeTypeQuery /"+str+"/");

           return str;

    }
    /**
     * Extract from DataSource all amps associated with this Solution ID. This
     * does a join using the AssocAmO link table.  */
    public AmpList getBySolution (long id) {
          return getBySolution (id, null);
    }
    /**
     * Extract from DataSource all amps associated with this Solution ID. This
     * does a join using the AssocAmO link table.
     * This does NOT get amps associated with the preferred magnitude. */
    public AmpList getBySolution (long id, String[] ampTypeList) {

  // must first  retreive the Solution, then look up prefor for this id
  Solution tsol = Solution.create().getById(id);

  if (tsol == null) return null;	// no such solution in DataSource

  return getBySolution(tsol, ampTypeList);
    }

    /**
     * Returns array of Amplitudes within this time window using default DataSource
     * connection. Times are seconds in
     * UNIX epoch time. Returns null if amplitudes are found. <br>
     * (NOTE: in TriNet this gets amps from ALL sources (RT1, RT2) so there will be duplicate
     * readings.) */
     // This is quite SLOW, need dbase indexing?
     public AmpList getByTime(double start, double stop) {

  return getByTime(DataSource.getConnection(), start, stop);
    }

    /**
     * Returns array of Amplitudes within this time window. Times are seconds in
     * UNIX epoch time.  Returns null if no amplitudes are found.<br>
     * (NOTE: in TriNet this gets amps from ALL sources (RT1, RT2) so there will be duplicate
     * readings.)  */
     // This is quite SLOW, need dbase indexing?
     public AmpList getByTime(Connection conn, double start, double stop) {

           String sql = sqlPrefixByTime + " where DATETIME BETWEEN " +
               StringSQL.valueOf(start) + " AND " + StringSQL.valueOf(stop) +
               " order by Datetime";

        return getBySQL(conn, sql);

    }

    /**
     * Returns array of Amps based on the results of the SQL query to an
     * NCDCv1.5 data base. It must be of a form that returns the full set
     * of columns.  Returns null if no data is found.  Uses default connection
     * created by DataSource.  */
    protected static AmpList getBySQL(String sql)
    {
  if (DataSource.getConnection() == null)
      {
    System.err.println ("* No DataSource is open.");
    return null;
      }
  return getBySQL(DataSource.getConnection(), sql);
    }

    /** Return an Amplitude with the given unique ID number. Returns null
     *  if there is no such amplitude. */
    public Amplitude getById(long id) {

        String sql = sqlPrefixByTime + " where ampid = "+id;
        AmpList ampList = getBySQL(DataSource.getConnection(), sql);

  if (ampList.isEmpty()) return null;

  return (Amplitude) ampList.get(0);
    }

    /**
     * Returns array of Amps based on the results of the SQL query to an
     * NCDCv1.5 data base. It must be of a form that returns the full set
     * of columns.  Returns null if no data is found.  Uses default connection
     * created by DataSource.  */
    protected static AmpList getBySQL(Connection conn, String sql)
    {
  AmpList ampList = new AmpList();

  // Debug
//	System.out.println ("SQL: "+sql);

  try {
      if ( conn.isClosed() )	// check that valid connection exists
    {
        System.err.println ("* DataSource connection is closed");
        return null;
    }

      Statement sm = conn.createStatement();

      // NOTE: if ExecuteSQL.setSelectForUpdate(true) was set in DataSource
      // this will lock the selected rows until commit() or close() are called
      ResultSetDb rsdb = new ResultSetDb(ExecuteSQL.rowQuery(sm, sql));

      if (rsdb == null) return null;	// nothing found

         // note: this can't tell if it its adding an amp a second time
         // if its not the same object instance
      while ( rsdb.getResultSet().next() ) {

    ampList.add(AmplitudeTN.parseResultSet(rsdb));
      }

      sm.close();

  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();
  }

  return ampList;
    }

    /**
     * Parse a resultset row that contains a concatinated Arrival/AssocArO
     */
    protected static Amplitude parseResultSet (ResultSetDb rsdb) {

  int offset = 0;

  // get jdbc objects from the result set
  Amp amp = new Amp();
  amp = (Amp) amp.parseOneRow(rsdb, offset);

  // build up the Amplitude from the jdbc objects
  AmplitudeTN ampNew = new AmplitudeTN();

  ampNew.parseAmp(amp);

  offset += Amp.MAX_FIELDS;

  // Parse depends on whether you did joint with AssocAmO or AssocAmM table.
  if (joinType == Type_AssocAmO) {

      AssocAmO  assoc = new AssocAmO();
      assoc = (AssocAmO) assoc.parseOneRow(rsdb, offset);
      ampNew.parseAssocAmO(assoc);
      if (DataSource.isWriteBackEnabled()) assoc.setMutable(true);

  } else if (joinType == Type_AssocAmM) {
      AssocAmM  assoc = new AssocAmM();
      assoc = (AssocAmM) assoc.parseOneRow(rsdb, offset);
      ampNew.parseAssocAmM(assoc);
      if (DataSource.isWriteBackEnabled()) assoc.setMutable(true);
  }

  // allow changes to this DataTableRow. Need to set this because we are
  // not using our own rather than org.trinet.jdbc parsing methods
  if (DataSource.isWriteBackEnabled()) amp.setMutable(true);

  // remember that we got this from the dbase
  ampNew.fromDbase = true;

  return (Amplitude) ampNew;
    }

    /**
     * Copy DataObjects out of a jdbc.Amp object into this Amplitude.
     */

    protected void parseAmp(Amp amp) {

     // call setChannel to synch with master channel list
  setChannelObj(ChannelTN.parseNameFromDataTableRow(amp));

        this.authority    = (DataString) amp.getDataObject(Amp.AUTH);
        this.source       = (DataString) amp.getDataObject(Amp.SUBSOURCE);

  this.ampid        = (DataLong)   amp.getDataObject(Amp.AMPID);

  this.datetime	   = (DataDouble) amp.getDataObject(Amp.DATETIME);

  this.processingState = (DataString) amp.getDataObject(Amp.RFLAG);

  this.phaseType    = (DataString) amp.getDataObject(Amp.IPHASE);

  this.value	   = (DataDouble) amp.getDataObject(Amp.AMPLITUDE);

  // Set AmpType value
  String ampType  = amp.getDataObject(Amp.AMPTYPE).toString();
  this.type       = AmpType.getInt(ampType);

  // if zero-to-peak = 1 , if peak-to-peak = 0
  String half  = amp.getDataObject(Amp.AMPMEAS).toString();
  if (half.equals("0")) this.halfAmp=false;

  // Set units
  String unitstr  = amp.getDataObject(Amp.UNITS).toString();
  this.units = Units.getInt(unitstr);

  this.uncertainty  = (DataDouble) amp.getDataObject(Amp.ERAMP);

  // IGNORING 'FLAGAMP', 'TAU'

  this.period  = (DataDouble) amp.getDataObject(Amp.PER);
  this.snr     = (DataDouble) amp.getDataObject(Amp.SNR);
  this.quality = (DataDouble) amp.getDataObject(Amp.QUALITY);

  String clip  = amp.getDataObject(Amp.CFLAG).toString();
  if      (clip.equalsIgnoreCase("OS")) { this.clipped =  ON_SCALE; }   //on scale
  else if (clip.equalsIgnoreCase("BN")) { this.clipped =  CLIPPED; }   //clipped
  else if (clip.equalsIgnoreCase("CL")) { this.clipped =  BELOW_NOISE; }   //below noise

  this.windowStart	  = (DataDouble) amp.getDataObject(Amp.WSTART);
  this.windowDuration	  = (DataDouble) amp.getDataObject(Amp.DURATION);

  return;
    }

/**
 * Copy DataObjects out of a jdbc.AssocAmO object into a Amplitude.
 */

protected void parseAssocAmO(AssocAmO assoc) {

    this.orid         = (DataLong)   assoc.getDataObject(AssocAmO.ORID);
    this.chan.dist    = (DataDouble) assoc.getDataObject(AssocAmO.DELTA);
    this.chan.azimuth = (DataDouble) assoc.getDataObject(AssocAmO.SEAZ);

    return;
}

/**
 * Copy DataObjects out of a jdbc.AssocAmO object into a Amplitude.
 */
protected void parseAssocAmM(AssocAmM assoc) {

    channelMag.value      = (DataDouble) assoc.getDataObject(AssocAmM.MAG);
    channelMag.residual   = (DataDouble) assoc.getDataObject(AssocAmM.MAGRES);

//    channelMag.importance = (DataDouble) assoc.getDataObject(AssocAmM.IMPORTANCE);

    channelMag.correction = (DataDouble) assoc.getDataObject(AssocAmM.MAGCORR);
    channelMag.weight     = (DataDouble) assoc.getDataObject(AssocAmM.WEIGHT);

    return;
}

    /**
     * Copy contents of this amplitude into a jdbc.Amp row object
     */

    protected org.trinet.jdbc.table.Amp toAmpRow() {

  // Always make a new row, get unique key #
  long newId = SeqIds.getNextSeq("ampseq");
  ampid.setValue(newId);			// set ID for this.Phase

  Amp ampRow = new Amp(newId);

  stuffChannel(ampRow, getChannelObj());

  // set flag to enable processing
  ampRow.setUpdate(true);

  ampRow.setValue(Amp.DATETIME, datetime);
     // "rectify" - peak amp must be > 0.0
     double ampval = Math.abs(value.doubleValue());
  ampRow.setValue(Amp.AMPLITUDE, ampval);
  ampRow.setValue(Amp.AMPTYPE, AmpType.getString(type));

  // if zero-to-peak = 1 , if peak-to-peak = 0
  if (halfAmp) {
      ampRow.setValue(Amp.AMPMEAS, 1);
  } else {
      ampRow.setValue(Amp.AMPMEAS, 0);
  }

  ampRow.setValue(Amp.UNITS, Units.getString(units));

  // the following can be null

  if (!authority.isNull())	     ampRow.setValue(Amp.AUTH, authority);

  if (!processingState.isNull())  ampRow.setValue(Amp.RFLAG, processingState);

  if (!source.isNull())		ampRow.setValue(Amp.SUBSOURCE, source);

  if (!phaseType.isNull())	     ampRow.setValue(Amp.IPHASE, phaseType);

  if (!uncertainty.isNull())	ampRow.setValue(Amp.ERAMP, uncertainty);

  if (!period.isNull())		ampRow.setValue(Amp.PER, period);
  if (!snr.isNull())		     ampRow.setValue(Amp.SNR, snr);
  if (!quality.isNull())		ampRow.setValue(Amp.QUALITY, quality);

  // IGNORING 'FLAGAMP', 'TAU'

  if (clipped ==  0) ampRow.setValue(Amp.CFLAG, "OS");
  if (clipped == -1) ampRow.setValue(Amp.CFLAG, "BN");
  if (clipped ==  1) ampRow.setValue(Amp.CFLAG, "CL");

  if (!windowStart.isNull())	ampRow.setValue(Amp.WSTART,windowStart );
  if (!windowDuration.isNull())	ampRow.setValue(Amp.DURATION, windowDuration);

  //	if (!.isNull())		ampRow.setValue(Amp., );

  return ampRow;
    }

/** Put the channel name attributes in the Arrival DataTableRow */
protected void stuffChannel (Amp dtrow, Channel cn) {

    // Constrained "NOT NULL" in schema
    dtrow.setValue(Amp.STA, cn.getSta());

    if (cn.getNet().length() > 0)        dtrow.setValue(Amp.NET, cn.getNet());
    if (cn.getAuth().length() > 0)       dtrow.setValue(Amp.AUTH, cn.getAuth());
    if (cn.getSubsource().length() > 0)  dtrow.setValue(Amp.SUBSOURCE, cn.getSubsource());
    if (cn.getChannel().length() > 0)    dtrow.setValue(Amp.CHANNEL, cn.getChannel());
    if (cn.getChannelsrc().length() > 0) dtrow.setValue(Amp.CHANNELSRC, cn.getChannelsrc());
    if (cn.getSeedchan().length() > 0)   dtrow.setValue(Amp.SEEDCHAN, cn.getSeedchan());
    if (cn.getLocation().length() > 0)   dtrow.setValue(Amp.LOCATION, cn.getLocation());
}

/**
 * Copy DataObjects out of a jdbc.AssocAmM object into a Amplitude.
 */
protected org.trinet.jdbc.table.AssocAmM toAssocAmMRow() {
    // keys are ampid and magid
    AssocAmM assoc = new AssocAmM();

    // Constrained "NOT NULL" in schema
    assoc.setValue(AssocAmM.MAGID, magnitude.magid);
    assoc.setValue(AssocAmM.AMPID, ampid);
    assoc.setValue(AssocAmM.AUTH,  authority);

    // ALLOWED TO BE NULL
    if (!source.isNull())	           assoc.setValue(AssocAmM.SUBSOURCE, source);
    if (!channelMag.value.isNull())	 assoc.setValue(AssocAmM.MAG, channelMag.value);
    if (!channelMag.residual.isNull())	 assoc.setValue(AssocAmM.MAGRES, channelMag.residual);
//    if (!channelMag.importance.isNull()) assoc.setValue(AssocAmM.IMPORTANCE, channelMag.importance);
    if (!channelMag.correction.isNull()) assoc.setValue(AssocAmM.MAGCORR, channelMag.correction);
    if (!channelMag.weight.isNull())	 assoc.setValue(AssocAmM.WEIGHT, channelMag.weight);
    if (!processingState.isNull())       assoc.setValue(AssocAmM.RFLAG, processingState);

    return assoc;
}

/**
 * Copy DataObjects out of a jdbc.AssocAmO object into a Amplitude.
 */
protected org.trinet.jdbc.table.AssocAmO toAssocAmORow() {
    // keys are ampid and magid
    AssocAmO assoc = new AssocAmO();

    // Constrained "NOT NULL" in schema
    long orid = ((SolutionTN)sol).getOrid();
    assoc.setValue(AssocAmO.ORID,  orid);
    assoc.setValue(AssocAmO.AMPID, ampid);

    // ALLOWED TO BE NULL
    assoc.setValue(AssocAmO.AUTH,  authority);
    if (!source.isNull())	        assoc.setValue(AssocAmO.SUBSOURCE, source);
    if (!chan.dist.isNull())	        assoc.setValue(AssocAmO.DELTA, getDistance());
    if (!chan.azimuth.isNull())       assoc.setValue(AssocAmO.SEAZ, chan.azimuth);
    if (!processingState.isNull())    assoc.setValue(AssocAmO.RFLAG, processingState);

    return assoc;
}

    /** Commit changes or delete to underlying DataSource. The action taken will
        depend on the state of the amplitude. It may or may not be originally
        from the data source, it may be deleted, unassociated, modified, or
        unchanged.  To write to the DataSource, the flag
        DataSource.isWriteBackEnabled() must be true. This must be set with
        DataSource.setWriteBackEnabled(true) BEFORE data is read.*/

/*
  We will never really delete an amp row in the dbase because this would alter
  a mag/amp data-set created by someone else. That would be rude. We only write
  an NEW NetMag, AssocAmM rows connecting existing or new Amps to the NetMag.
*/
public boolean commit() {

    // you can't write to the dbase
    if (!DataSource.isWriteBackEnabled()) return false;

    // debug
    if (debug) System.out.println ("AmplitudeTN.commit() : fromDbase= "+ fromDbase+
           "  isDeleted()= "+isDeleted());

    // <DELETE>
    // Phase is deleted or not associated! So don't write new Amp or AssocAmM.
    // Note we NEVER actually delete a dbase row, just don't write it out if its
    // new or don't make association if the amp came from the dbase.
    if (isDeleted()) return true;

    // Must be associated with a Solution to save it.
    if (!isAssociated()) return true;

//    long magid = sol.magnitude.magid.longValue();

    // "existing" amp (from the dbase)
    if (fromDbase) {

  if (hasChanged()) {

     // Old amp had value or time modified, this shouldn't happen.
      if (debug) System.out.println ("(changed amp) INSERT and assoc");
         // change this flag to allow insert as a new amp
         fromDbase = false;
      return dbaseInsert();		// insert & associate

  } else {

      // if no changes to amp, just make new association
         if (dbaseAssociate()) {
            setUpdate(false);        // mark as up-to-date
            if (debug) System.out.println ("(amp unchanged) assoc only");
            return true;
         } else {
            return false;
         }
  }

    // virgin amp (not from dbase) <INSERT>
    } else {

  if (debug) System.out.println ("(new amp) INSERT and assoc");

  return dbaseInsert();	// insert & associate
    }

}

/**
 * NEVER delete an amp. Never remove association with another NetMag.  If
you don't like it just don't assoc it with your new NetMag.  */
protected boolean dbaseDelete () {

    if (fromDbase) {

  return true;

    } else {		// not from dbase :. nothing to delete

  return true;
    }
}

protected boolean dbaseAssociate () {
    return dbaseAssociateWithOrigin() &&
           dbaseAssociateWithMag();
}
/**
 * Make AssocAmO row for this Amp
 */
protected boolean dbaseAssociateWithOrigin () {

    boolean status = false;

    // assoc'ed with Solution
    if (isAssociated() ) {
            AssocAmO assocORow = toAssocAmORow();
            assocORow.setProcessing(DataTableRowStates.INSERT);
            status = (assocORow.insertRow(DataSource.getConnection()) > 0);
    }

    return status;

}
/**
 * Make AssocAmM row for this Amp
 */
protected boolean dbaseAssociateWithMag () {

    boolean status = false;

    // assoc'ed with Mag
    if (isAssociatedWithMag()) {
       // Check that 'magid' has been set, it can't be null
       if (getAssociatedMag() == null || getAssociatedMag().magid.isNull()) {

      if (debug) System.out.println ("Associated Magnitude is null or has no magid.");
       } else {
            AssocAmM assocMRow = toAssocAmMRow();
            assocMRow.setProcessing(DataTableRowStates.INSERT);
            status = (assocMRow.insertRow(DataSource.getConnection()) > 0);
       }
    }

    return status;

}
/**
 * Insert a new row in the dbase. This will also write association to the
 * event and the magnitude.
 */
protected boolean dbaseInsert () {

    if (fromDbase) return false;       // nothing to update

    boolean status = true;

    if (debug) System.out.println ("dbaseInsert: "+this.toString());

    Amp ampRow = toAmpRow();
    ampRow.setProcessing(DataTableRowStates.INSERT);

    // write arrival row
    status = (ampRow.insertRow(DataSource.getConnection()) > 0);

    if (status) {		// successful insert

  status = dbaseAssociate();	// write association row
  // now its "from" the dbase
  setUpdate(false);        // mark as up-to-date (fromDbase=true)
    }

    return status;
}

/**
 * Return true if the amp has changed from what was
 * read in from the dbase. Returns true if it was not originally from the
 * dbase.  */
protected boolean hasChanged () {

    if (!fromDbase ||
  value.isUpdate() ||
  datetime.isUpdate()
  ) return true;

    return false;
}
/**
 * Set the isUpdate() flag for all data dbase members the given boolean value.  */
protected void setUpdate (boolean tf) {

    fromDbase = !tf;

    value.setUpdate(tf);
    datetime.setUpdate(tf);
}

// ///////////////////////////////////////////////////////

    public static void main (String args[])
    {

     EnvironmentInfo.setApplicationName("AmpTest");

  long evid = 0;

  if (args.length <= 0)	// no args
  {
    System.out.println ("Usage: java AmplitudeTN [evid])");

    System.out.println ("Using evid "+ evid+" as a test...");

  } else {

    Integer val = Integer.valueOf(args[0]);	    // convert arg String
    evid = (int) val.intValue();
  }

        System.out.println ("Making connection...");
  DataSource ds = new TestDataSource();

     // get the solution
     Solution sol = Solution.create().getById(evid);


// Test getting amps in time window
 /*
 // This works but is slow, and gets amps from RT1 & RT2, + amps that were not
 // used in the mag.
  System.out.println (" Getting amps in window around event = "+evid);

     double start = sol.datetime.doubleValue() - 60.0;
     double stop  = sol.datetime.doubleValue() + 60.0;
  ArrayList amps = (ArrayList) Amplitude.create().getByTime(start, stop);
     if (amps.size() == 0) {
        System.out.println ("No amps in time window.");
     } else {
    for (int i = 0; i<amps.size(); i++)
        {
      System.err.println (amps.get(i).toString());
      }
     }
 */
  System.out.println ("\n Getting amps for magnitude of evid = "+evid);

     // Read amps for this event from the dbase
  sol.magnitude.fetchAmps();

  System.out.println ("Amp count = " + sol.magnitude.ampList.size());

     if (sol.magnitude.ampList.size() <= 0) {
        System.out.println ("No amps for this event.");
//        System.exit(0);
     } else {
    System.out.println("Raw magnitude from dbase...");
          System.out.println(sol.magnitude.neatDump() + "\n");
  }

  // Calc mag.

  MagnitudeMethod ml = MagnitudeMethod.CreateMagnitudeMethod("org.trinet.util.magnitudeengines.SoCalML");
  ml.ConfigureMagnitudeMethod();

  MagnitudeEngine magEng = MagnitudeEngine.CreateMagnitudeEngine("org.trinet.util.magnitudeengines.LocalMagnitudeEngine");
  magEng.ConfigureMagnitudeEngine(ml);

  Magnitude newMag = magEng.solve(sol.magnitude);

  System.out.println ("Relocation...") ;
     System.out.println(sol.magnitude.neatDump() + "\n");


// test changing and writing a new mag with at least one modified amp
// get the last amp
     Amplitude testAmp = (Amplitude)sol.magnitude.ampList.get(sol.magnitude.ampList.size()-1);

     // dump some stuff
     System.out.println (" Some stuff...\n"+
                        " mag.fromDbase = " + ((MagnitudeTN)sol.magnitude).fromDbase +
                        " testAmp.hasChanged = " + ((AmplitudeTN)testAmp).hasChanged() +
                        "");
     System.out.println ("Commiting magnitude to dbase...");

     try {
        sol.magnitude.commit();
     } catch (JasiCommitException ex) {}

    // NOTE *******************************************************************
    //       There's no DataSource.commit() done here so the results are NOT
    //	     Actually written to the dbase.
    // ************************************************************************

    // if (status) DataSource.commit();

  DataSource.close();
    }

} // AmplitudeTN
