package org.trinet.jasi;

import java.text.*;
import java.util.*;
import java.net.URL;
import java.sql.Connection;

import org.trinet.jasi.*;
import org.trinet.util.IndexSort;
import org.trinet.util.graphics.ColorList;
import java.awt.Color;

/**
 * A list of solutions that make a catalog.  It is constrained by ranges or
 * limits to parameters like start-end time, magnitude range, region, etc. that
 * are set in an instance of EventSelectionProperties.  (Only box regional
 * selection is implemented).  Specific Solutions objects can be added or
 * removed.<p> Uses an existing JASI DataSource.  */


public class SolutionList extends ActiveArrayList {

    /* An array list is similar to a Vector but is consistent with the new
     * Collection interface.  Sun strongly recommends use of ArrayList because
     * that makes it trivial to substitute another Collection type later if
     * necessary. You can't do that with vector because it has a bunch of
     * "extra" methods that aren't in the Collection interface.  */
    EventSelectionProperties props;

    // the currently selected event. This is the nexus of the MVC model for
    // event selection. NOTE: id rather than Solution object is used because
    // Object can be replaced but ID is unique.
    //private static Solution selected = null;
    private static long selectedId = -1;

    private static final int InitCapacity = 50;

    /** Connection that will be used for dbase access */
    protected Connection conn; // DK
    // DK protected Connection conn = DataSource.getConnection();

    /** hash table to keep track of the unique color of each solution in the list */
//    Hashtable colorTable = new Hashtable(10);  // initial capacity = 10
    Hashtable colorTable = new Hashtable();  // initial capacity = 11
    int maxColor = 0;

/** Make an empty SolutionList */
public SolutionList ()
{
    super(InitCapacity);
}

    /** Create a SolutionList based on this property description */
public SolutionList (EventSelectionProperties properties)
{
    super(InitCapacity);

    props = properties;

    fetchByProperties();

}

    /** Create a SolutionList containing this solution */
public SolutionList (long evid)
{
    super(InitCapacity);

    fetchById(evid);

}
    /** Create a SolutionList for this time window */
public SolutionList (double start, double stop)
{
    super(InitCapacity);

    fetchByTime(start, stop);

}

/** Copy a SolutionList */
public SolutionList (Collection coll)
{
    super(InitCapacity);
    addAll(coll);
    // copy 'selected' solution if arg is a SolutionList
    if (coll instanceof SolutionList) {
	setSelected( ((SolutionList)coll).getSelected(), true);
    }
}

/**
 * Define the connection
 */
    public void setConnection (Connection conn) {
	this.conn = conn;
    }

/**
 * Define the properties
 */
    public void setProperties (EventSelectionProperties object) {
	this.props = object;
    }

/**
 * Return the event selection properties
 */
    public EventSelectionProperties getProperties () {
	return this.props;
    }

/**
 * Returns the connection
 */
    public Connection getConnection () {
	return conn;
    }
/**
 * Returns the currently selected event
 */
public Solution getSelected() {
    if (getSelectedId() <= -1) return null;
    int idx = getSelectedIndex();
    if (idx < 0) return null;
    return (Solution) get(idx);
}
public long getSelectedId() {
   return selectedId;
}
/**
 * Returns the index of the currently selected event
 */
public int getSelectedIndex() {
    return getIndex(getSelectedId());
}
/**
 * Set the selected event. Only works if the event is already in the list.
 * Returns true if the Solution is in the list and was set as the selected Solution.
 * Notifies listeners.
 */
public boolean setSelected(Solution sel) {
       return setSelected (sel , true);
}
/**
 * Set the selected event. Only works if the event is already in the list.
 * Returns true if the Solution is in the list and was set as the selected Solution.
 * Only notify listeners if 'notify' = true.
 */
public boolean setSelected(Solution sel, boolean notify) {

//  System.out.println ("SolList.setSelected: notify = "+notify);

    if (sel == null ||
        sel == getSelected()) return true;   // prevent infinite loops, don't RE-select
    return setSelected(sel.getId().longValue(), notify);
}
/**
 * Set the selected event by this id. Only works if the event is already in the list.
 * Returns true if the Solution is in the list and was set as the selected Solution.
 * Notify listeners.
 */
public boolean setSelected(long id) {
    return setSelected(id, true);
}
/**
 * Set the selected event by this id. Only works if the event is already in the list.
 * Returns true if the Solution is in the list and was set as the selected Solution.
 * Only notify listeners if selected ID changes and 'notify' is true.
 */
public boolean setSelected(long id, boolean notify) {
    // already selected
    if (id == getSelectedId()) return true;

    if (this.contains(id)) {
       selectedId = id;
       if (notify) fireStateChanged(getSelected());
       return true;
    } else {
       return false;
    }
}
/**
 * Thin wrapper just to do the cast from Object to Solution.
 */
/*
 * Tried to call it 'get' and got error:

The method get(int) cannot override the method of the same signature declared
in class java.util.ArrayList.  They must have the same return type.

 */
public Solution getSolution(int index)
{
    return (Solution) get(index);
}

/**
 * Return a new SolutionList that contains all the same Solutions but is sorted
 * by time. The arg is either SolutionList.ASCENDING or SolutionList.DESCENDING
 * where ASCENDING in chronological order and DESCENDING is reverse.
 */

public static final int ASCENDING = 0;
public static final int DESCENDING = 1;

public SolutionList getTimeSortedList(int order) {

    Solution sol[] = this.getArray();

    // make the time array that is needed by IndexSort
      double tArray[] = new double[sol.length];
      for ( int i = 0; i < sol.length; i++)
      {
	tArray[i] = sol[i].datetime.doubleValue();
      }

    // sort by time
      IndexSort indexSort = new IndexSort();
      int idx[] = indexSort.getSortedIndexes(tArray);

      SolutionList sl = new SolutionList();

      if (order == ASCENDING) {
	  for (int i = 0; i < sol.length; i++)
	      {
		  sl.add( sol[idx[i]] );	    // add sols in sorted order
	      }
      } else {
	  for (int i = sol.length-1; i >=0; i--)
	      {
		  sl.add( sol[idx[i]] );     // add sols in reverse sorted order
	      }
      }

      return sl;
}

/**
 * Populate this list with Solutions that meet the EventSelectionProperties given.
 */


    public void fetchByProperties(EventSelectionProperties props)
    {

     addSolutions(Solution.create().getByProperties(props));

    }
/**
 * Populate this list with Solutions that meet the EventSelectionProperties
 * already set with the setProperties() method. Does nothing if instance properties
 * are null.
*/
    public void fetchByProperties()
    {

     if (props != null) fetchByProperties(props);

    }

/**
 * Create list with one entry for this evid.
 */
    public void fetchById(long evid)
    {
	if (conn == null) {
	    this.add( Solution.create().getById(evid) );
	} else {
	    this.add( Solution.create().getById(conn, evid) );
	}
    }

/**
 * Create list of Solutions for this time window.
 */
   public void fetchValidByTime(double start, double stop)
    {
	Solution sol[] = null;
	if (conn == null) {
	     sol = Solution.create().getValidByTime(start, stop);
	} else {
	     sol = Solution.create().getValidByTime(conn, start, stop);
	}
	addSolutions(sol);
    }

/**
 * Create list of Solutions for this time window.
 */
   public void fetchByTime(double start, double stop)
    {
	Solution sol[] = null;
	if (conn == null) {
	     sol = Solution.create().getByTime(start, stop);
	} else {
	     sol = Solution.create().getByTime(conn, start, stop);
	}
	addSolutions(sol);
    }

/** Add an array of Solution objects to this list.*/
    public void addSolutions(Solution [] sol) {
	if (sol == null) {
		System.out.println("No data; solution array null");
		return;
	}
	// may be a better way to do this, addByTime(Solution) instead ??? aww
     this.ensureCapacity(size()+sol.length);
	for (int i=0; i<sol.length;i++) {
		this.add(sol[i]);
     }
    }
/**
 * Add an object. Will not add object if its already in the list.
 * Returns 'true' if object was added.
 */
    public boolean add(Object obj){
       return add(obj, true);
    }
/**
 * Add an object. Will not add object if its already in the list.
 * Returns 'true' if object was added. If notify = false, change event will not be
 * fired.
 */
    public boolean add(Object obj, boolean notify) {

    // must set color BEFORE firing the event, else listeners get the wrong color!
       if (super.add(obj, false)) {
         setColor((Solution) obj);     //sets unique event color
         if (notify) this.fireStateChanged(obj);
         return true;
       } else {
         return false;
       }
    }
 /**
 * Insert an object at this index. Will not add object if its already in the list.
 * Overrides ArrayList.add() to inforce no-duplicates rule.
 */
    public void add(int index, Object obj)  {

        super.add(index, obj);  // this piece-of-crap doesn't return a boolean!
        setColor((Solution) obj);

    }
/**
 * Add a solution to the list, maintain sorting by time.
 */

public void addByTime (Solution newSol) {

    Solution sol[] = this.getArray();
    double newDateTime = newSol.datetime.doubleValue();

    for (int i = 0; i< sol.length; i++) {
      if (sol[i].datetime.doubleValue() > newDateTime) {
		this.add(i, newSol);
		return;
      }
    }
    //System.out.println ("Inserting at end "+sol.length);
    add(newSol);	// add to end
}
    /**
     * Actually deletes the Solution from the list (rather then doing a virtual delete).
     * If the deleted event is the selected event, the next event in the list will be
     * selected. If there are no more events in the list the selected event will become null.
     */
    public boolean delete (Object sol) {   // must be Object to override ActiveArrayList.delete()

      Solution zombie = (Solution) sol;
    // must look-up next sol BEFORE deletion, else you can't ref deleted sol's position
      Solution nextSol = null;
      if (zombie.getId().longValue() == this.getSelectedId()) {
	nextSol = getNext((Solution) sol);
      }

      if (super.delete(sol)) {
         if (nextSol != null) setSelected(nextSol);
         return true;
      } else {
         return false;
      }
    }

/**
 * Count the wfid entries in the wvevidassoc table for each Solution.
 * This is done seperately because its slow buy not always necessary.
 */
    public void countWaveforms() {

	Solution sol[] = this.getArray();

	for (int i = 0; i < sol.length; i++) {
	    sol[i].countWaveforms();
	}
    }

/**
 * Return a Solution[]
 */
	// This weirdness is necessary because the 'List' stores Objects
	// So we must both cast and use the arg to tell the List class what the
	// object type really is.
public Solution[] getArray() {

     return (Solution[]) this.toArray(new Solution[0]);
}
/**
 * Return true if this id is in the solution list.
 */
public boolean contains (long id) {
	Solution sol[] = this.getArray();

	for (int i = 0; i < sol.length; i++) {
	    if (sol[i].id.longValue() == id) return true;
	}

	return false;
}

/**
 * Extract a Solution from this list with this id (evid). Return null if not found.
 */
public Solution getById(long id) {

	Solution sol[] = this.getArray();

	for (int i = 0; i < sol.length; i++) {
	    if (sol[i].id.longValue() == id) return sol[i];
	}

	return null;
}
/**
 * Return the index in the list for the Solution with this id (evid).
 * Return -1 if not found.
 */
public int getIndex(long id) {

	Solution sol[] = this.getArray();

	for (int i = 0; i < sol.length; i++) {
	    if (sol[i].id.longValue() == id) return i;
	}

	return -1;
}
/**
 * Return the index in the list for this Solution.
 * Return -1 if not found.
 */
public int getIndex(Solution sol) {

	return getIndex(sol.id.longValue());
}
/**
 * Return the Solution following the selected Solution. If it is the last in the
 * list this will "wrap" and return the first Solution in the list.
 * Return null if the list is empty.  Same as getNext(getSelected()).
 */
public Solution getNext() {
    return getNext(getSelected());
}
/**
 * Return the Solution following this id (evid). If the given id is the last in the
 * list this will "wrap" and return the first id in the list.
 * Return null if the list is empty. If the id is not found the first Solution in the
 * list is returned.
 */
public Solution getNext(long id) {

       if (isEmpty()) return null;

       int idx = getIndex(id);

       if (idx < 0) {
          return (Solution) get(0);
       } else if (idx >= size()-1) {   // last in list, wrap
          return (Solution) get(0);
       } else {
          return (Solution) get(idx+1);
       }

}

/**
 * Return the Solution following this one. Return null if list is empty.
 * If the given Solution is the last in the
 * list or the Solution is not found the first Solution in the list is returned.
 */
public Solution getNext(Solution sol) {

    long id = -1;             // this will cause 1st to select if sol is bogus
    if (sol != null && !sol.id.isNull()) id = sol.id.longValue();

    return getNext(id);
}

/**
 * Extract a Solution from this list with this ORID. Return null if not found.
 */
public Solution getByOrid (long orid)
{
	Solution sol[] = this.getArray();

	if (sol != null) {

	    for (int i = 0; i < sol.length; i++)
		{
		    if ((sol[i]).getOrid() == orid) return sol[i]; 
		}
	}

	return null;
}

/**
 * Dump the list contents in Solution.toSummaryString() format
 */
public void dump() {

    System.out.println (dumpToString());

}
/**
 * Dump the list contents in Solution.toSummaryString() format
 */
public String dumpToString() {
    String str = "";
    Solution sol[] = this.getArray();
    if (sol != null) {
	for (int i = 0; i < sol.length; i++)
	    {
		str += sol[i].toSummaryString()+"\n";
	    }
    }
    return str;
}


/**
 * Main for testing: % SolutionList [hours-back]  (default = 1)
 */
    public static void main (String args[]) {

	double hoursBack;
	final int secondsPerHour = 60*60;

	if (args.length > 0)	// translate epoch second on command-line value
	{
	  Double val = Double.valueOf(args[0]);	    // convert arg String to 'double'
	  hoursBack = (double) val.doubleValue();
	} else {
	  System.out.println ("Usage: java SolutionList [hours-back]  (default=24)");
	  hoursBack = 24;	// default
	}

        System.out.println ("Making connection...");
	DataSource ds = new TestDataSource();

//	catView.setLatRange (25.0, 45.0);
//	catView.setLonRange (-140.0, -110.0);

        Calendar cal = Calendar.getInstance();

// must distinguish between 'java.util.Date'  and 'java.sql.Date'
	java.util.Date date = cal.getTime();	// current epoch millisec
	long now = date.getTime()/1000;	// current epoch sec (millisecs -> seconds)

	long then = now - (long) (secondsPerHour * hoursBack);	// convert to seconds
    // add 1000 to now just to be sure we get everything

	// read in saved eventProperties file
     EventSelectionProperties props = new EventSelectionProperties("eventProperties");

	// set time properties
//	props.setDateTime("startTime", new DateTime(then));
//	props.setDateTime("endTime", new DateTime(now));

     props.setTimeSpan (then, now);

	props.setProperty("validFlag", "TRUE");
//	props.setProperty("validFlag", "FALSE");
//	props.setProperty("dummyFlag", "TRUE");
	props.setProperty("dummyFlag", "FALSE");

     props.setProperty(EventSelectionProperties.prefix+"local", "TRUE");
     props.setProperty(EventSelectionProperties.prefix+"human", "TRUE");
     props.setProperty(EventSelectionProperties.prefix+"automatic", "TRUE");

	System.out.println ("-- Event Selection Properties --");

     props.dumpProperties();

	System.out.println ("Fetching last "+hoursBack+" hours...");

// Get a catalogview
	SolutionList catList = new SolutionList(props);

	Solution sol[] = catList.getArray();

	//	System.out.println ("found ="+sol.length);

	if (sol.length == 0) System.out.println (" * No events found.");

	for (int i = 0; i < sol.length; i++)
	{
	    System.out.println (sol[i].toSummaryString() +" "+
				sol[i].source.toString() );
	}

	// test addition by time
/*
	Solution solx = Solution.create().getById(sol[0].id.longValue());

	System.out.println (" Test insertion of: \n"+solx.toSummaryString());
	System.out.println (" ");

	catList.addByTime(solx);
	//	catList.add(solx);
	//	catList.add(1, solx);

        Solution sol2[] = catList.getArray();
	for (int i = 0; i < sol2.length; i++)
	{
	    System.out.println (sol2[i].toSummaryString() +" "+
				sol2[i].waveRecords.toString());
	}
*/
    }

// Support event color coding

    /**
     * Return a unique color for this solution. The SolutionList keeps track
     * of a color for each solution in the list. Colors are assigned as solutions are
     * added to the list according t the scheme in org.trinet.util.ColorList.
     * The colors are used by GUI's to distingush between data like phases and amps
     * associated with the solutions
     */
    public Color getColorOf (Solution sol) {

     if (sol == null) return ColorList.UNASSOC;
	return (Color) colorTable.get(sol.id);	      //sol.id is a DataObject
    }

    /** Set the unique color for this solution. The SolutionList keeps track
     * of a color for each solution in the list. Colors are assigned as solutions are
     * added to the list according t the scheme in org.trinet.util.ColorList.
     * The colors are used by GUI's to distingush between data like phases and amps
     * associated with the solutions. */
    void setColor (Solution sol) {
	colorTable.put(sol.id, ColorList.getColor(maxColor++));
    }
    /** Set unique colors for for all solutions in list. The SolutionList keeps track
     * of a color for each solution in the list. Colors are assigned as solutions are
     * added to the list according t the scheme in org.trinet.util.ColorList.
     * The colors are used by GUI's to distingush between data like phases and amps
     * associated with the solutions. */
    void setColors () {

	Solution sol[] = getArray();

	for (int i = 0; i < sol.length; i++) {

	    setColor(sol[i]);
	}
    }
} // end of class


