package org.trinet.jasi;

/**
 * Model to calculate the peak ground acceleration or velocity from a given
 * magnitude at a given distance from an earthquake. Distance and depth is in
 * km. If the depth of the event is unknown use a reasonable value like 6.0 km.<p>
 * 
 * This class is derived from a perl script developed for ShakeMap by Vince Quitoriano 
 * and Dave Wald.<p>

 Regression from Boore, Joyner and Fumal, 1997, SRL, Volume 68, p. 128.
 with the form: 
<tt>
        ln (PGA,PSV) = B1 + B2(M-6) + B3(M-6)**2 - B5*ln(R) - Bv*ln(Vs/Va),

        where R = sqrt(Rjb**2 + h**2)
<\tt> 
 PGV is from Joyner and Boore, 1988, Proc. Earthq. Eng. & Soil Dyn. II,  
 Park City, Utah, 1988, in the form:
<tt>
        ln (PGV) = a + b(M-6) + c(M-6)**2 - dlog(R) + k*R + s

<\tt> 
 PSA, PGA are in "g". PGV in cm/s. PSA is 5% damped pseudo-acceleration. 
 Random horizontal component on rock. Distance is JB definition (see
 SRl V68, No. 1, p10.) This is horizontal distance (km) from the nearest projection
 of the fault rupture to the surface.
*/

public class JBModel  {

    static double Vs = 620.0;		// S-wave velocity in rock
    //    double Vs = 310.0;		// S-wave velocity in soil


    static double B1, B2, B3, B5, Bv, Va, depth;
    /* B3 was always 0.0 so I removed it from all calcs. */

    /** Scale acceleration in "g" to %g, and multiply by 1.15 to get maximum
        rather than random comp.*/
    final static double g_m  = 9.81;		// m/sec^2 per g
    final static double g_cm = g_m/100.0;	// cm/sec^2 per g

    // Magic number form the original perl script
    final static double scale = 1.15/g_m;

    /** Private constructor */
    private JBModel() {

    }

// Acceleration
    /** Return the peak ground acceleration (g's) for this magnitude at this distance. */
    public static double getPeakGs(double mag, double depth, double horizdist) {

	setAccelParams();

	return getPseudoAccel(mag, horizdist, depth);	// scale to %g
    }
    static void setAccelParams() {
	B1=4.037;
	B2=0.572;
	B5=-1.757;
	Bv=-0.473;
	Va=760.0;
    }
    /** Return the peak ground acceleration (cm/sec^2) for this magnitude at
        this distance. Depth is assumed to be 6.0 km.*/
    public static double getPeakAccel(double mag, double depth, double horizdist) {

	return getPeakGs(mag, horizdist, depth) * g_cm;	// scale to cm/sec^2
    }

    /** Return the peak ground acceleration (%g) for this magnitude at this
        distance. Depth is assumed to be 6.0 km.*/
    public static double getPeakPercentG(double mag, double depth, double horizdist) {

	return gToPercentG(getPeakGs(mag, horizdist, depth) );	// scale to %g
    }

    static double percentGToG(double percentG) {
	return percentG / scale;
    }

    static double gToPercentG(double g) {
	return g * scale;
    }

// Velocity
    /** Return the peak ground velocity (cm/sec) for an event of this magnitude
	at this horizontal distance and depth. */
    public static double getPeakVelocity(double mag, double depth, double horizdist) {

	setVelParams();

	return getPseudoAccel(mag, horizdist, depth);
    }

    static void setVelParams() {
	B1=2.223;
	B2=0.740;
	B5=-1.387;
	Bv=-0.669;
	Va=760.0;
    }

    /** Return the distance at which the velocity will be this value for this magnitude
     * event. */
    public static double getDistanceOfVelocity(double mag, double depth, double velocity) {
	setVelParams();
	return getHorizDistanceOfPseudoAccel(mag, depth, velocity);
    }
    /** Return the distance at which the acceleration will be this value for this magnitude
     * event. Acceleration is in g's.*/
    public static double getDistanceOfAccel(double mag, double depth, double accel) {
	setAccelParams();
	return getHorizDistanceOfPseudoAccel(mag, depth, accel);
    }

    /** Return 5% damped pseudo-acceleration. */
    // This has been checked against results from the perl script and is correct.
    public static double getPseudoAccel(double mag, double depth, double horizdist) {

	// true diagonal distance
	double dist = Math.sqrt(horizdist*horizdist + depth*depth);

	return Math.pow( 10.0, (B1 + B2*(mag-6.0) + B5*log10(dist) + Bv*log10(Vs/Va)) );

    }

    /** Return horizonatal distance at which the 5% damped pseudo-acceleration (peak) is
     * this value for an event of this magnitude and depth. */
    public static double getHorizDistanceOfPseudoAccel
		(double mag,  double depth, double peak) {

	double dist = Math.pow( 10.0, (log10(peak) - B1 - B2*(mag-6.0) - Bv*log10(Vs/Va))/B5 );

	// return horizonatal distance
	return Math.sqrt(dist*dist - depth*depth);

    }
    
    /**
     * Java does not provide a function for log(base10). This is how you
     * do it. (see Que, Using Java 1.1, pg. 506).
     */
    public static double log10 (double val) {

        final double logOf10 = Math.log(10.0);

        if (val > 0) {
            return Math.log(val) / logOf10;
        } else {
            return 0.0;         // should throw exception?
        }
    }

    // Test values to compare to perl script, also see if dist is extractable.
    // All works (there's a tiny precision error)
    public static void main (String args[]) {

	double mag = 0.0;

/*        if (args.length > 0)
        {
          Double val = Double.valueOf(args[0]);     // convert arg String to 'double'
          mag = (double) val.doubleValue();
	}
 */
	double depth = 6.0;
	double pgv = 0.0;
	double pga = 0.0;


	for (int dist = 10; dist <= 300; dist = dist + 10) {

	    pga = JBModel.getPeakPercentG(mag, depth, dist);
	    pgv = JBModel.getPeakVelocity(mag, depth, dist);

	    System.out.println (dist +"  "+ JBModel.getPeakAccel(mag, depth, dist)+
				"   "  + pga +
				"   "  + pgv+
				"  test-> v: "+(int)JBModel.getDistanceOfVelocity(mag, depth, pgv) + 
				"  a: "+(int)JBModel.getDistanceOfAccel(mag, depth, percentGToG(pga)) 
				);
	}

    }

} // JBModel
