package org.trinet.util.magnitudeengines;

import java.util.*;
import org.trinet.filters.*;
import org.trinet.util.MathTN;

/**
 * Static methods for ML (local magnitude) calculation.<p>
 *
 * Formula is defined by Richter in Elementry Seismology, pg. 340: <p>
 *
 *			ML = log A + Cd + Cs<p>
 * Where:<br>
 *		A  = half amp, in mm from a 2,800x Wood-Anderson torsion seismometer<br>
 *		Cd = Southern California distance correction<br>
 *		Cs = emperical station correction<p>
 *
 * Over-rides the distance correction of the Richter ML. <br>
 * This distance correction is based on an emperical relationship for southern
 * California derived by Hiroo Kanamori. Assumes HYPOCENTRAL distance not
 * epicentral distance. <p>
 *
 * The method can use either corrected or uncorrected Wood-Anderson amplitudes
 * (types WAC and WAU)
 * @see AmpType()
 *
 * @see ML()
 * @author Doug Given
 * @version */

public class SoCalML extends ML {

    public SoCalML()
    {
      ConfigureMagnitudeMethod();
    }

    public void ConfigureMagnitudeMethod() {
           // Magnitude type subscript. For example: for "Ml" this would equal "l".
         subScript = "l";

         // String describing the magnitude method
         name = "SoCalMl";

        // set the waveform filter to Synthetic Wood-Anderson
        setWaveformFilter(new WAFilter());

        // cutoff dist
        setMinDistance(30.0);
        setMaxDistance(600.0);

        setRequireCorrection(true);
        setDefaultParams();
    }

    public void ConfigureMagnitudeMethod(int iConfigurationSource,
                                         String sConfigurationLocation,
                                         String sConfigurationSection
                                        )
    {
      ConfigureMagnitudeMethod();
    }


    void setDefaultParams() {
      // set characteristics (should get from properties)
      this.setTrimResidual(1.0);
      this.setMinSNR(8.0);  // Per Kate Hutton 4/2002
      this.setRequireCorrection(true);
      this.setUseLowGains(false);
    }
    /** Over-ride the distance correction of the Richter ML. <br>
  This is based on an empirical relationship for southern California
  derived by Hiroo Kanamori. Assumes HYPOCENTRAL distance not
  epicentral distance. <p>

  Correction = -log10( 0.3173 * exp(-0.00505 * dist) * pow(dist, -1.14))

  */
    public double distanceCorrection (double distance) {

  double magic1 = -0.00505;
  double magic2 =  0.3173;
  double magic3 = -1.14;

  double c = Math.exp (magic1 * distance);
  c = magic2 * c * Math.pow(distance, magic3);
  c = -(MathTN.log10(c));

  //	if (debug) System.out.println ("SoCalMl Dist. Corr = "+c);
  return c;
    }

    /**
     * Return the distance cutoff appropriate for the method. Amplitudes from
     * stations father then this will not be included in the summary magnitude.
     * Because it depends on the magnitude the cutoff can only be applied in a
     * second pass through the amplitude readings. In the
     * southern California magnitude scheme the distance cutoff is an empirical
     * relationship derived by Kanamori. It is given by:<p>
     *<tt>
     * cutoff = 170.0 * magnitude - 205.0 (all distances are in km)
     *</tt>
     * The formula can yields 0.0 at a magnitude of 1.21, so a minimum cutoff is
     * enforced. This method will never yield a value less then 30.0km.
     */
    // mag  cut
    // 1.0  -35
    // 2.0  135
    // 3.0  305
    // 4.0  475
    public double getDistanceCutoff (double magnitude) {

  double dist = Math.max( getMinDistance(), (170.0 * magnitude) - 205.0 ) ;

     return Math.min(dist, getMaxDistance() );

    }

} // ML





