package org.trinet.jasi.TN;

import org.trinet.jasi.*;

import java.sql.*;
import java.util.*;
import java.io.*;

import org.trinet.jdbc.*;
import org.trinet.jdbc.table.*;
import org.trinet.jdbc.datatypes.*;

import org.trinet.util.gazetteer.LatLonZ;
import org.trinet.util.gazetteer.GeoidalUnits;

import org.trinet.util.DateTime;
import org.trinet.util.BenchMark;

/**
 * JASI - Java Abstract Seismic Interface
 * This is a schema specific implimentation of Channel that acts as the
 * interface to the Channel_Data table of the NCDC v1.5 schema.
 * Contains a ChannelName member which includes
 * descriptive information for the channel like location
 * of the site (latlonz) and response info. <p>
 *
 * @See: org.trinet.jasi.ChannelName
 */

public class ChannelTN extends Channel {

  public ChannelTN() {
  }
// Concrete instances of Channel abstract methods
    /**
     * Look up the given Channel in the DataSource dbase and return a fully
     * populated Channel object. This is how you look up LatLonZ and response
     * info for a single channel. Note this finds the LATEST entry if more then
     * one entry is in the dbase (largest 'ondate'). But, it does NOT require that
     * the channel be "active". If no entry is found, this
     * method return the channel name that was passed in the arg as a Channel.
     */
    public Channel lookUpChannel (Channel chan) {

        ChannelIdIF chanId = chan.getChannelId();
  String whereClause =
      " (Channel_Data.net = '"+ chanId.getNet() +"' and "+
      "Channel_Data.sta = '"+ chanId.getSta() +"' and "+
      "Channel_Data.seedchan = '"+ chanId.getSeedchan() +"' "+
      ") order by Channel_Data.ondate DESC";	// descending order so latest is first

  ArrayList list = (ArrayList) getWhere (DataSource.getConnection(), whereClause );

  // got something
  if (list != null && list.size() > 0) {

      Channel ch = (Channel) list.get(0);	// return latest entry only

      return (Channel) ch;	// return first entry only

  }

  return chan;	// return original entry

    }

  /** Return a list of currently active channels that match the list of
    * component types given in
    * the array of strings. The comp is compared to the SEEDCHAN field in the NCDC
    * schema.<p>
    *
    * SQL wildcards are allowed as follows:<br>
    * "%" match any number of characters<br>
    * "_" match one character.<p>
    *
    * For example "H%" would match HHZ, HHN, HHE, HLZ, HLN & HLE. "H_" wouldn't
    * match any of these but "H__" would match them all. "_L_" would match
    * all components with "L" in the middle of three charcters (low gains).
    * The ANSI SQL wildcards [] and ^ are not supported by Oracle.*/
    public ChannelList getByComponent(String[] compList) {

           String whereClause = "";

           whereClause = getNowWhereClause() + " and ";

           if (compList.length > 0) {
            whereClause += " (";

            for (int i = 0; i < compList.length; i++) {
               if (i > 0) whereClause += " or ";
               whereClause += "Channel_Data.seedchan like '"+compList[i]+"'";
            }
            whereClause += ") order by Channel_Data.net,Channel_Data.sta,Channel_Data.seedchan";
           }

           return getWhere(DataSource.getConnection(), whereClause);
    }
    /** Return a list of channels that were active on the given date and
    *  that match the list of component types given in
    * the array of strings. The comp is compared to the SEEDCHAN field in the NCDC
    * schema.<p>
    *
    * SQL wildcards are allowed as follows:<br>
    * "%" match any number of characters<br>
    * "_" match one character.<p>
    *
    * For example "H%" would match HHZ, HHN, HHE, HLZ, HLN & HLE. "H_" wouldn't
    * match any of these but "H__" would match them all. "_L_" would match
    * all components with "L" in the middle of three charcters (low gains).
    * The ANSI SQL wildcards [] and ^ are not supported by Oracle.*/
    public ChannelList getByComponent(String[] compList,  java.sql.Date date) {

           String whereClause = "";

           whereClause = getDateWhereClause(date) + " and ";

           if (compList.length > 0) {
            whereClause += " (";

            for (int i = 0; i < compList.length; i++) {
               if (i > 0) whereClause += " or ";
               whereClause += "Channel_Data.seedchan like '"+compList[i]+"'";
            }
            whereClause += ") order by Channel_Data.net,Channel_Data.sta,Channel_Data.seedchan";
           }

           return getWhere(DataSource.getConnection(), whereClause);
    }

/** Return count of all currently active channels in data source. */
    public int getCurrentCount () {

  // use NOW
  java.sql.Date currentTime =
      new java.sql.Date(new DateTime().getMilliSeconds());

       return getCount (currentTime);

    }

    /** Return count of all channels in data source at the given time. */
    public int getCount (java.sql.Date date) {

       return getCountBySQL (getDateWhereClause(date));

    }
/**
 * Get a count of sql rows given an SQL query.
 */
    static int getCountBySQL(String whereClause){
       return getCountBySQL(DataSource.getConnection(), whereClause);
    }
/**
 * Get a count of sql rows given an SQL query.
 */
    static int getCountBySQL(Connection conn, String whereClause){

     int count = 0;

     String sql = "SELECT count(*) from Channel_Data where "+whereClause;
  // Debug
  if (debug) System.out.println ("SQL: "+sql);

  try {
      if ( conn.isClosed() )	// check that valid connection exists
    {
        System.err.println ("* ChannelList connection is closed. *");
        return 0;
    }

      Statement sm = conn.createStatement();

      ResultSet rs = org.trinet.jdbc.table.ExecuteSQL.rowQuery(sm, sql);

      if (rs == null) return 0;	// nothing found

         rs.next();

         count = rs.getInt(1);

      sm.close();

  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();
  }

  return count;
    }

    /**
     * Appends this SQL where clause to the statement
     * "Select * from Channel_Data ". Example: "where net = 'CI'"
     */
    public  ChannelList getListWhere(String whereClause) {

  return getWhere(DataSource.getConnection(), whereClause);
    }

/**
 * Return exhaustive Collection of Channels from the default DataSource.
 */
    public ChannelList readList() {
  return getWhere("");
    }

    public ChannelList readList(java.sql.Date date) {
      return readList(DataSource.getConnection(), date);
    }

    public ChannelList readList(Connection conn, java.sql.Date date) {
  return Channel.create().readList(conn, date);
//	return Channel.create().readList(Channel.getDateWhereClause(date);

    }


// Internal methods ////////////
    /** Return a where clause to select channels that were active on this date. */
    static String getDateWhereClause ( java.sql.Date date) {
  return " (Channel_Data.OFFDATE >= " + StringSQL.valueOf(date) +
                        " OR Channel_Data.OFFDATE = NULL) and "+
                        "(Channel_Data.ONDATE <= " + StringSQL.valueOf(date) +
                        " OR Channel_Data.ONDATE = NULL) ";
    }

    /** Return a where clause to select channels are were active NOW. */
    static String getDateWhereClause () {
      return getNowWhereClause();
    }

    /** Return a where clause to select channels that are currently active.
     *  This uses the database's concept of "now" just in case there is a
     *  difference in time bases between the dbase and the client. */
    static String getNowWhereClause () {
  return " (Channel_Data.OFFDATE >= SYSDATE" +
                        " OR Channel_Data.OFFDATE = NULL) and "+
                        "(Channel_Data.ONDATE <= SYSDATE" +
                        " OR Channel_Data.ONDATE = NULL) ";
    }

    /**
     * Appends this SQL where clause to the statement
     * "Select * from Channel_Data ". Example: "where net = 'CI'"
     */
    ChannelList readListWhere(String whereClause) {

  return (ChannelList) getWhere(DataSource.getConnection(), whereClause);
    }
/**
 * Get an array of this object given an SQL query.
 */
    ChannelList getBySQL(String sql)
    {
  return getBySQL(DataSource.getConnection(), sql);
    }
/**
 * Get an array of this object given an SQL query.
 */
    ChannelList getBySQL(Connection conn, String sql)
    {

  //ArrayList chanList = new ArrayList();
  ChannelList chanList = new ChannelList();

  // Debug
  if (debug) System.out.println ("SQL: "+sql);

  try {
      if ( conn.isClosed() )	// check that valid connection exists
    {
        System.err.println ("* Connection is closed");
        return null;
    }

      Statement sm = conn.createStatement();

      ResultSet rs = org.trinet.jdbc.table.ExecuteSQL.rowQuery(sm, sql);

      if (rs == null) return null;	// nothing found

      // parse and add rows one-at-a-time
      while ( rs.next() ) {

    chanList.add(parseResultSet(rs));
      }

      sm.close();

  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();
  }

  // convert vector to array
  //	if (wfList.size() == 0) return null;

  return chanList;
    }

/**
 * Parse a resultset row that contains the results of a join + some
 * stored procedures
 */
  static ChannelData cdInst = new ChannelData();

    Channel parseResultSet (ResultSet rs){
  int offset = 0;

  // parse the first part of the return that is just a Waveform table row
  ResultSetDb rsdb = new ResultSetDb(rs);

  Channel ch =
         parseChannelRow((ChannelData) cdInst.parseOneRow(rsdb, offset));

  // Now, pick up the "extra" columns specified by the SQL query.
  // See SelectJoinString for a list of fields
  offset += cdInst.MAX_FIELDS;

  try {
      // must get as objects because they might be null
      // If I used getDouble an null field returns 0.0!
      ch.gain.setValue (rs.getObject(++offset));
      ch.mlCorr.setValue(rs.getObject(++offset));

         /* Check state of correction
          * F = frozen   mature(?)
          * C =          still being updated
          * D =          "default" seem all to be 0.0
          * If the corr is unknown it is still set to 0.0 in the dbase but the
          * corr_flag = "D" for default :. must check for that flag.
          */
      DataString corrFlag = new DataString();
            corrFlag.setValue(rs.getObject(++offset));
         if (corrFlag.isNull() ||
             corrFlag.toString().equalsIgnoreCase("D") ) ch.mlCorr.setNull(true);

         // can't figure out a querry to get ML and Me at the same time
//	    ch.meCorr.setValue (rs.getObject(++offset));
  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();
  }

  return ch;
    }

   /** Return the SQL join string for current channel data. */
    public  String getSelectJoinString () {

       return getSelectJoinString("Sysdate");
    }

    /** Return the SQL join string for channel data at the time described by
    * this java.sql.Date . */
    public  String getSelectJoinString (java.sql.Date date) {
       return getSelectJoinString(StringSQL.valueOf(date));
    }

    /** Return the SQL join string for channel data at the time described by
    * this SQL formatted date string. Only ML correction is read, not Me, etc.*/
    /* NOTES:
       o Join: note that we must do an "outer" join by using (+). Otherwise, an
       channel that lacks RT_param info will NOT be seen. (see: Oracel8
       Bible, McCullough-Dieter, pg. 217)
       o Must qualify ALL tables with on/off dates else the outerjoin will create
       a resultset row for every response or correction entry. */
    public  String getSelectJoinString (String dateStr) {

    return "Select "+
      org.trinet.jdbc.table.ChannelData.QUALIFIED_COLUMN_NAMES+","+
     " SIMPLERESP.gain, staCorrections.corr, staCorrections.corr_flag "+
     " from channel_data, staCorrections, SIMPLERESP "+
     " Where exists( select * from Channel_Data where "+
     " (Channel_Data.OFFDATE >= "+dateStr+" and Channel_Data.ONDATE <= "+dateStr+")) and "+
     " (  Channel_Data.net = staCorrections.net(+) and "+
     " Channel_Data.sta = staCorrections.sta(+) and "+
     " Channel_Data.seedchan = staCorrections.seedchan(+) and "+
     " (staCorrections.corr_type(+) = 'ml' and "+               // ml correction
     " staCorrections.OFFDATE(+) >= "+dateStr+" and staCorrections.ONDATE(+) <= "+dateStr+") ) and "+
     " (Channel_Data.net = SIMPLERESP.net(+) and "+
     " Channel_Data.sta = SIMPLERESP.sta(+) and "+
     " Channel_Data.seedchan = SIMPLERESP.seedchan(+) and "+
     " SIMPLERESP.OFFDATE(+) >= "+dateStr+" and SIMPLERESP.ONDATE(+) <= "+dateStr+") " ;
     }
/**
 * Return Collection of ALL Channels in the DataSource that satisfy the where clause.
 * Uses the given Connection.
 */
    ChannelList getWhere (String whereClause)
    {
      return getWhere (DataSource.getConnection(), whereClause);
    }
/**
 * Return Collection of ALL Channels in the DataSource that satisfy the where clause.
 * Uses the given Connection.
 */
    ChannelList getWhere (Connection conn, String whereClause)
    {
     String sql = getSelectJoinString();
     if (whereClause.trim().length() > 0) {
     sql += " and " + whereClause;
      }

  return getBySQL (conn, sql);

    }
/**
 * Get a ChannelName description from a Channel_Data DataTableRow object.
 * Use a DataTableRow because we an use this to parse ANY row type with a
 * channel in it.
 */
    // Must use Column names and not indexes here because every DataTableRow type
    // has different indexes.

    static Channel parseChannelRow (ChannelData cdr) {

  Channel ch = Channel.create();
  ch.setChannelObj(ChannelTN.parseNameFromDataTableRow((DataTableRow)cdr));

  DataDouble lat = (DataDouble) cdr.getDataObject(cdr.LAT);
  DataDouble lon = (DataDouble) cdr.getDataObject(cdr.LON);
  DataDouble z   = (DataDouble) cdr.getDataObject(cdr.ELEV);	//km
        DataDouble sampleRate = (DataDouble) cdr.getDataObject(cdr.SAMPRATE);

        if (sampleRate != null) ch.setSampleRate(sampleRate.doubleValue());

  if (lat != null && lon != null) {

      ch.latlonz.setLat(lat.doubleValue());
      ch.latlonz.setLon(lon.doubleValue());
     }

     if (z != null) {
  // convert elevation to km, which is what LatLonZ uses.

      double km = z.doubleValue()/1000.0;

         ch.latlonz.setZ(km);

      // This is the default but be explicite
      ch.latlonz.setZUnits(GeoidalUnits.KILOMETERS);
  }

  return ch;
    }

/**
 * Get a Channelname description from a DataTableRow object.
 * Use a DataTableRow because we an use this to parse ANY row type with a
 * channel in it.
 */
    // Must use Column names and not indexes here because every DataTableRow type
    // has different indexes.
    public static Channel parseNameFromDataTableRow (DataTableRow dtr) {
     Channel chan = Channel.create();
  try {
      chan.setNet(dtr.getStringValue("NET"));
      chan.setSta(dtr.getStringValue("STA"));
      chan.setChannel(dtr.getStringValue("CHANNEL"));
      // the Channel_Data table does NOT have these two columns
      //	    channelId.setAuth(dtr.getStringValue("AUTH"));
      //	    channelId.setSubsource(dtr.getStringValue("SUBSOURCE"));
      chan.setChannelsrc(dtr.getStringValue("CHANNELSRC"));
      chan.setSeedchan(dtr.getStringValue("SEEDCHAN"));
      chan.setLocation(dtr.getStringValue("LOCATION"));
  } catch (NoSuchFieldException ex)
      {
    System.err.println(ex);
    ex.printStackTrace();
      }

  return chan;
    }

}