package org.trinet.jasi;

import org.trinet.jdbc.datatypes.*;
import org.trinet.util.DateTime;
import org.trinet.util.gazetteer.LatLonZ;

/**
 * Abstract class that represents some sort of per channel reading like a phase
 * pick, amplitude or coda measurement. This class encapsulates the behaviors
 * that these things have in common.
 *
 *
 * Created: Wed Jun  7 16:34:43 2000
 *
 * @author Doug Given
 * @version
 *
 * @See: Arrival
 * @See: Amplitude
 */

public abstract class JasiReading extends JasiObject
           implements Channelable {

    /** Channel information including name, location, reponse info and distance */
    protected Channel chan		= Channel.create();
    protected ChannelList channelList;

    //* Date and time of the reading. Seconds from epoch 1/1/1970. *.
    public DataDouble  datetime	= new DataDouble();

    /** Data authority. The 2-character FDSN network code from which this
        reading came. Default to network code set in EnvironmentInfo.
        @See: EnvironmentInfo.getNetworkCode */
    public DataString  authority  = new DataString(EnvironmentInfo.getNetworkCode());

    /** Data source string. Optional site defined source string.
        Default to application name set in EnvironmentInfo.
        @See: EnvironmentInfo.getApplicationName */
    public DataString  source	  = new DataString(EnvironmentInfo.getApplicationName());

    /** Distance of this reading from the associated solution in km. Used for
        distance sorting, etc.*/
//    public DataDouble distance = new DataDouble();

    /** Difference (error) between actual and expected reading value.
        Residual = observed - expected. If the significant value is a
        time, positive values are too late and negative values are too
        early.*/
    public DataDouble  residual = new DataDouble();

    /** Free format text comment. */
    public DataString comment = new DataString();

    /** Azimuth from the reading to the associated Solution. Degrees
        clockwise from north.*/
//    DK3 should be using Channel.azimuth since this is a channelable class
//    public DataDouble azimuth = new DataDouble();

    /** Solution with which this reading is associated. */
    public Solution sol = null;

    /** True if this reading has been virtually deleted */
    boolean deleteFlag = false;

    /** */
//    public boolean fromDbase = false;

    /** State of processing. 'A'=automatic, 'H'=human */
    public DataString  processingState	=
           new DataString(EnvironmentInfo.getAutoString());	// rflag

    // ABSTRACT METHODS ///////////////////////////////////////////////////////

    /** Commit any additions, changes or deletions to the data source. */
    abstract public boolean commit();

    /** A value between 0.0 and 1.0 used to indicate problem or bad readings in
    * a generic way. Gives a "grade" to a reading
    * that is used by components that want to highlight "bad" readings.
    * 0.0 is best, 1.0 is worst.
    */
    public double getWarningLevel() {
      return Math.min(Math.abs(this.residual.doubleValue()), 1.0);
    }

    /** Return true if the reading was used in the summary solution, mag, etc. */
    abstract public boolean wasUsed();

    /** Return a string representing the object. */
    abstract public String toString() ;

    /** Return a neatly formatted string representing the object. */
    abstract public String toNeatString() ;

    /** Return header string describing the format of toNeatString(). */
    static public String getNeatStringHeader() {return "";};

    // CONCRETE METHODS ///////////////////////////////////////////////////////

    // Channelable interface
    /**
     * Return the channel object associated with this reading.
     */
    public Channel getChannelObj () {
  return chan;
    }

    /** Set the Channel. The ChannelList will be searched for a
    * matching channel. If found, this Channel object is replaced by a reference to
    * the matching Channel object in this list. If no match is found, the
    * original Channel object is retained. This is done so that the readings
    * get the full channel description, lat/lon/z, response info, calibration,
    * distance from epicenter, etc. <p>
    * */
    public void setChannelObj(Channel channel, ChannelList chanList) {

     chan = channel;     // match the channel to the master list

     if (chanList != null) setChannelFromList(chanList);

    }
    /** Set the Channel. If defined, this object's chanList will be scanned for a matching
    * channel. If not defined, the MasterChannelList will be searched.
    * If a match is found, this Channel object is replaced by a reference to
    * the matching Channel object in the list. If no match is found, the
    * original Channel object is retained. This is done so that the readings
    * get the full channel description, lat/lon/z, response info, calibration,
    * distance from epicenter, etc. <p>
    * */
    public void setChannelObj(Channel channel) {

     // match the channel to the master list
     if (channelList != null) {
        setChannelObj(channel, channelList);
     } else {
        setChannelObj(channel, MasterChannelList.get());
     }

    }

    /** Set the master ChannelList. Any call to setChannel() will examine this list
    * for a matching channel. If one is found, this reading's Channel object will be
    * replaced by a reference to
    * the matching Channel object in the list. If no match is found, the
    * original Channel object is retained. This is done so that the readings
    * get the full channel description, lat/lon/z, response info, calibration,
    * distance from epicenter, etc. <p>
    * If no ChannelList is set the MasterChannelList will be used by default.
    */
    public void setChannelList(ChannelList channelList) {
       this.channelList = channelList;
    }
    /** Match this reading's channel with one in this list. If a match is found
     * replace the readings channel object with a reference to the one in the list.
     * This is used to match objects to a more complete description of a channel then
     * they probably have otherwise. Gives access to lat/lon/z, response info, calibration,
     * distance from epicenter, etc. */
    public boolean setChannelFromList(ChannelList channelList) {

        if (channelList == null || channelList.isEmpty()) return false;

        Channel chanMatch = channelList.findSimilarInMap(getChannelObj());
        if (chanMatch == null) {
//               System.out.println("setChannelFromList can't find channel: "
//                   + getChannel().toString());
            return false;
        } else {
            chan = chanMatch;      // DON'T CALL setChannel() - recursion error
            return true;
        }
    }
    /** Calculate and set both epicentral (horizontal) distance and true (hypocentral)
     * distance of this channel from the given location in km.
     * Note: this is better than setDistance() or setHorizontalDistance() because
     * it does both and insures they are consistent. */
    public double calcDistance (LatLonZ loc) {
      return chan.calcDistance(loc);
    }

    /** Set hypocentral distance. */
    public void setDistance(double distance) {
      getChannelObj().setDistance(distance);
    }

    /** Set horizontal distance. */
    public void setHorizontalDistance(double distance) {
      getChannelObj().setHorizontalDistance(distance);
    }

    public double getDistance() {
      return getChannelObj().getDistance();
    }
    public double getHorizontalDistance() {
      return getChannelObj().getHorizontalDistance();
    }

     public void setAzimuth(double az) {
        getChannelObj().setAzimuth(az);
     }
     public double getAzimuth () {
        return getChannelObj().getAzimuth();
     }
    /**
     * Set time of pick as a double.
     */
    public void setTime(double dt) {
  datetime.setValue(dt);
    }

    /**
     * Set time of pick as a DateTime object.
     */
    public void setTime(DateTime dt) {
  datetime.setValue(new DateTime(dt));
    }

    /**
     * Return epoch time of pick as a double. Returns 0.0 if datetime is null.
     */
    public double getTime() {
  if (datetime.isNull()) return 0.0;
  return datetime.doubleValue();
    }
    /**
     * Return epoch time of pick as a DateTime object. If datetime is null
     * returns DateTime object set to epoch start.
     */
    public DateTime getDateTime() {
  if (datetime.isNull()) return new DateTime(0.0);
  return new DateTime(datetime.doubleValue());
    }

    /**
     * Associate this reading with this Solution.
     */
    public void associate(Solution sol)
    {
  this.sol = sol;
    }

    /**
     * Unassociate, that is, set Solution to null.
     */
    public void unassociate()
    {
  this.sol = null;
    }

    /** Return the associated solution. Returns null if unassociated. */
    public Solution getAssociatedSolution () {
  return sol;
    }

    /** Return 'true' if this is associated with a Solution */
    public boolean isAssociated() {
  if (sol == null) return false;
  return true;
    }

    /** Return 'true' if this is associated with this Solution */
    public boolean isAssociatedWith(Solution compSol) {
  if (sol == compSol) return true;
  return false;
    }

    /**
     * Delete this reading. This is a "virtual delete".
     * It is  not actually removed from memory or the dbase.
     * Other classes must decide how they want to handle deleted readings when
     * commit() is called.
     * Override this method to add additional behavior. Deleted readings are
     * automatically unassociated.
     */
    public boolean delete() {
  deleteFlag = true;
  return true;
    }

    /** Returns true if this phase has been virtually deleted */
    public boolean isDeleted() {
  return deleteFlag;
    }

} // JasiReading
