package org.trinet.util.magnitudeengines;

import java.io.*;
import java.sql.*;
import java.util.*;
import org.trinet.jasi.*;
import org.trinet.jasi.coda.TN.*;
import org.trinet.jdbc.table.*;
import org.trinet.util.*;

public class McaCalDb {

    public static final String SQL_CODA_QUERY_STRING =
        "select coda.COID,coda.STA,coda.NET,coda.AUTH,coda.SUBSOURCE,coda.SEEDCHAN,coda.LOCATION," +
        "coda.AFIX,coda.AFREE,coda.QFIX,coda.QFREE,coda.NSAMPLE,coda.DATETIME," +
        "origin.EVID,assoccoo.ORID,netmag.MAGID,netmag.MAGNITUDE " +
        "FROM coda,assoccoo,origin,netmag WHERE " +
        "coda.ALGORITHM='MCA'AND coda.COID=assoccoo.COID AND origin.orid=assoccoo.ORID AND netmag.ORID=assoccoo.ORID AND " +
        "netmag.MAGTYPE='l' AND (NET=? AND STA=? AND SEEDCHAN=?) ";

    public static final String SQL_CODA_ORDER_BY_STRING = " ORDER BY netmag.MAGNITUDE";

    public static final String SQL_STN_QUERY_STRING = "SELECT DISTINCT STA,NET,SEEDCHAN FROM CODA WHERE ";
    public static final String SQL_STN_ORDER_BY_STRING  = " ORDER BY coda.net,coda.STA,coda.SEEDCHAN";

    protected boolean rejectDisabled = true;
    protected boolean verbose = true;

    //protected Concat cc = new Concat();
    protected PrintStream log;
    protected Statement preparedDbStatement;
    protected int totalChannelCount;
    protected int processedChannelCount;
    protected int acceptedChannelCount;

    protected ArrayList stnChlList = new ArrayList(1024);

    protected class OutputStats {
        DataStnChl stnChl;
        int    codaCount;
        double medianQFree;
        double medianQFreeDeviation;
        double medianAFree;
        double medianAFreeDeviation;
        double medianAFix;
        double medianAFixDeviation;
        double calibrFitSlope;
        double calibrFitAFix;
        String statusString;

        protected void reset() {
            stnChl               = null;
            codaCount            = 0;
            medianQFree          = 0.;
            medianQFreeDeviation = 0.;
            medianAFree          = 0.;
            medianAFreeDeviation = 0.;
            medianAFix           = 0.;
            medianAFixDeviation  = 0.;
            calibrFitSlope       = 0.;
            calibrFitAFix        = 0.;
            statusString         = "";
        }

        public String toString() {
            StringBuffer sb = new StringBuffer(132);
            sb.append(stnChl.toAbbrvStnChlString()).append(" ");
            Concatenate.format(sb, codaCount, 4).append(" ");
            Concatenate.format(sb, medianAFix, 5, 2).append(" ");
            Concatenate.format(sb, medianAFixDeviation, 5, 2).append(" ");
            Concatenate.format(sb, medianAFree, 5, 2).append(" ");
            Concatenate.format(sb, medianAFreeDeviation, 5, 2).append(" ");
            Concatenate.format(sb, medianQFree, 5, 2).append(" ");
            Concatenate.format(sb, medianQFreeDeviation, 5, 2).append(" ");
            Concatenate.format(sb, calibrFitSlope, 5, 2).append(" ");
            Concatenate.format(sb, calibrFitAFix, 5, 2).append("   ");
            sb.append(statusString);
            return sb.toString();
        }

        public String getHeaderString() {
            //return "STN                 CNT    AFIX      STD    AFREE      STD    QFREE      STD    SLOPE  INTERCEPT";
            return "STN         CNT     AFIX      STD    AFREE      STD    QFREE      STD    SLOPE  INTERCEPT";
        }

    }

    protected DateRange   currentInputDateRange;
    protected DateRange   currentCalibrDateRange;
    protected OutputStats currentStats;
    protected Collection  currentInputDataCollection = new ArrayList(512);

    protected class InputData {
        DataStnChl stnChl = new DataStnChl();
        double  aFix;
        double  qFix;
        double  aFree;
        double  qFree;
        int     windows;
        double  time;

        long    coid;
        long    evid;
        long    orid;

        long    magid;
        double  ml;

        public String getHeader() {
            return "staChl             afix qfix afree qfree nc coid evid orid magid ml";
        }

        public String toString() {
            StringBuffer sb = new StringBuffer(132);
            sb.append(stnChl.toStnChlString()).append(" ");
            sb.append(aFix).append(" ");
            sb.append(qFix).append(" ");
            sb.append(aFree).append(" ");
            sb.append(qFree).append(" ");
            sb.append(windows).append(" ");
            sb.append(coid).append(" ");
            sb.append(evid).append(" ");
            sb.append(orid).append(" ");
            sb.append(magid).append(" ");
            sb.append(ml).append(" ");
            sb.append(EpochTime.toNoZoneYYYYString(time));
            return sb.toString();
        }

    }

    public McaCalDb() { }

    public McaCalDb(java.util.Date startDate, java.util.Date endDate) {
        setDateRange(startDate, endDate);
    }

    public void setVerbose(boolean value) {
        verbose = value;
    }

    public void enableReject() {
        rejectDisabled = false;
    }

    public void setDateRange(DateRange dateRange) {
        if (dateRange == null) currentInputDateRange = new DateRange();
        else this.currentInputDateRange = dateRange;
    }

    public void setDateRange(java.util.Date startDate, java.util.Date endDate) {
        if (currentInputDateRange == null) currentInputDateRange = new DateRange(startDate, endDate);
        else {
            currentInputDateRange.setMin(startDate);
            currentInputDateRange.setMax(endDate);
        }
    }

    private void closeIO() {
        if (preparedDbStatement != null) {
            try {
               preparedDbStatement.close();
            }
            catch (SQLException ex) {
               ex.printStackTrace();
            }
        }
        log.flush();
        log.close();
    }

    protected int loadChannelListFromDb() {
        return loadChannelListFromDb(DataSource.getConnection());
    }

    private String createSQLStnQueryString() {
        StringBuffer sb = new StringBuffer(132);
        sb.append(SQL_STN_QUERY_STRING);
        if (currentInputDateRange.isLimited()) {
            sb.append(org.trinet.jdbc.table.Coda.toDateTimeConstraintSQLWhereClause("CODA",
                currentInputDateRange.getMinDate(), currentInputDateRange.getMaxDate()));
        }
        sb.append(SQL_STN_ORDER_BY_STRING);
        return sb.toString();
    }

    protected int loadChannelListFromDb(Connection conn) {
        stnChlList.clear(); // reuse existing collection

        if (conn == null) {
            System.err.println ("McaCalDb no connection is open.");
            return 0;
        }
        try {
            if (conn.isClosed()) {
                System.err.println ("McaCalDb DataSource connection is closed");
                return 0;
             }
            Statement sm = conn.createStatement();


            ResultSet rsdb = sm.executeQuery(createSQLStnQueryString());
            if (rsdb == null) return 0;        // no data found
            while (rsdb.next()) {
                DataStnChl dsc = new DataStnChl();
                dsc.setSta(rsdb.getString(1));
                dsc.setNet(rsdb.getString(2));
                dsc.setSeedchan(rsdb.getString(3));
                stnChlList.add(dsc);
            }
            rsdb.close();
            sm.close();
        }
        catch (SQLException ex) {
            System.err.println(ex);
            ex.printStackTrace();
        }
        return stnChlList.size();
    }

    protected boolean initializeIO() throws IOException {
        log =  new PrintStream(new BufferedOutputStream(new FileOutputStream("mcacaldb.log")));
        System.setOut(log);
        System.setErr(log);

        currentStats               = new OutputStats(); // channel fit results data stored in instance of OutputStats

        currentInputDataCollection.clear();

        processedChannelCount      = 0;
        acceptedChannelCount       = 0;

        //channelList              = loadChannelListFromDb().getArray();

        totalChannelCount          = loadChannelListFromDb();
        log.println("Input channel count: " + totalChannelCount);

        if (currentInputDateRange == null || ! currentInputDateRange.hasMinLimit())
            currentCalibrDateRange = new DateRange(new java.util.Date(System.currentTimeMillis()), null);
        else currentCalibrDateRange = currentInputDateRange;

        return true;
    }

/*
    protected Channel [] channelList;
    protected DataStnChl currentInputDataStnChl = new DataStnChl();
    protected DataStnChl getCurrentStnChl(int idx) {
        currentInputDataStnChl.setUpdateAllValues(false);
        currentInputDataStnChl.setNullAllValues(true);
        currentInputDataStnChl.setValue(DataStnChl.STA, channelList[idx].getSta());
        currentInputDataStnChl.setValue(DataStnChl.NET, channelList[idx].getNet());
        currentInputDataStnChl.setValue(DataStnChl.SEEDCHAN, channelList[idx].getSeedchan());
        return currentInputDataStnChl;
    }

*/
    protected DataStnChl getCurrentStnChl(int idx) {
        return (DataStnChl) stnChlList.get(idx);
    }

    /** Input database station channel Coda data and generate the database MCA calibration data. */
    public boolean calibrate() throws IOException {
        if (! initializeIO())  return false;
        boolean status = false;
        for (int idx = 0; idx < totalChannelCount; idx++) {
            currentStats.reset();
            int codaCount = getDataFromDb(getCurrentStnChl(idx));
            if (codaCount <= 0) continue;
            InputData [] inputDataList = (InputData []) currentInputDataCollection.toArray(new InputData[codaCount]);
            status = processInputData(inputDataList);
            if (! status) break;
        }
        log.println("Number of channels processed: " + processedChannelCount);
        log.println("Number of channels accepted : " + acceptedChannelCount);
        closeIO();
        return status;
    }

    private boolean processInputData(InputData [] inputDataList) {
        boolean status = calcCurrentChannelStats(inputDataList);
        if (verbose) {
            printChannelStats();
            printChannelMagResiduals(inputDataList);
        }
        if (status || rejectDisabled) {
            if (! writeChannelStatsToDb()) return false;
            acceptedChannelCount++;
        }
        processedChannelCount++;
        //if (processedChannelCount == 5) return false; // TEST DEBUG UNCOMMENT FOR LIMITED CHANNEL OUTPUT
        return true;
    }

    private boolean calcCurrentChannelStats(InputData [] inputDataList) {

        int currentCodaCount = inputDataList.length;
        if (currentCodaCount == 0) return true;

        calcL1Norm(inputDataList);

        currentStats.stnChl = inputDataList[0].stnChl;
        currentStats.codaCount = currentCodaCount;

        double [] x = new double[currentCodaCount];

        for (int idx = 0; idx < currentCodaCount; idx++) {
            x[idx] = inputDataList[idx].aFix - inputDataList[idx].ml;
        }
        currentStats.medianAFix          = Stats.median(x, currentCodaCount);
        currentStats.medianAFixDeviation = Stats.meanDeviation(x, currentCodaCount);

        for (int idx = 0; idx < currentCodaCount; idx++) {
            x[idx] = inputDataList[idx].aFree - inputDataList[idx].ml;
        }
        currentStats.medianAFree          = Stats.median(x, currentCodaCount);
        currentStats.medianAFreeDeviation = Stats.meanDeviation(x, currentCodaCount);


        for (int idx = 0; idx < currentCodaCount; idx++) {
            x[idx] = inputDataList[idx].qFree;
        }
        currentStats.medianQFree = Stats.median(x, currentCodaCount);
        currentStats.medianQFreeDeviation = Stats.meanDeviation(x, currentCodaCount);

        return evaluateChannelStats(currentStats);
    }

    private boolean evaluateChannelStats(OutputStats stats) {
        boolean status = true;
        StringBuffer sb = new StringBuffer(80);
        if (! rejectDisabled) {
            // double qdif = Math.abs(stats.medianQFree - McaCalibrationTN.QFIX_DEFAULT)
            if (stats.codaCount < 10) {
                sb.append("ncoda<10; ");
                status = false;
            }
            if ( stats.medianQFree < 0.2) {
                sb.append("qfree<.2; ");
                status = false;
            }
            if (Math.abs(stats.calibrFitAFix - stats.medianAFix) > 0.5) {
                sb.append("adiff>.5; ");
                status = false;
            }
            if (stats.calibrFitSlope < 1.0) {
                sb.append("slope<1.; ");
                status = false;
            }
        }
        if (status) sb.append("ACCEPT");
        else sb.append("REJECT");
        stats.statusString = sb.toString();
        return status;
    }

    protected boolean writeChannelStatsToDb() {
        String seedchan = currentStats.stnChl.getSeedchan();
        long clipAmp    = CodaCalibrationTN.getDefaultClipAmp(seedchan);
        long cutoffAmp  = CodaCalibrationTN.getDefaultCutoffAmp(seedchan);

        McaCalibrationTN  mcaCalibr = new McaCalibrationTN(currentStats.stnChl, currentCalibrDateRange,
                                      currentStats.codaCount, currentStats.medianAFix, McaCalibrationTN.QFIX_DEFAULT,
                                      currentStats.medianAFree, currentStats.medianQFree,
                                      currentStats.calibrFitSlope, clipAmp, cutoffAmp);
        boolean status = mcaCalibr.store();
        if (! status ) log.println("Error writing to db : " + currentStats.stnChl.toAbbrvStnChlString() );
        return status;
    }

    private void printChannelStats() {
        log.println();
        log.println(currentStats.getHeaderString());
        log.println(currentStats.toString());
    }

    private void printChannelMagResidualsHeader() {
        log.println("\n IDX     ID   DEV    NW      AFX      AFR      QFR         ML       MC      RMS       - MAGRES +");
    }

    private void printChannelMagResiduals(InputData [] inputDataList) {
        char [] cres = new char[21];
        Arrays.fill(cres, ' ');
        cres[10] = '|';

        printChannelMagResidualsHeader();

        int currentCodaCount = inputDataList.length;

        for (int idx = 0; idx < currentCodaCount; idx++) {
            StringBuffer sb = new StringBuffer(132);
            Concatenate.format(sb, idx, 4).append(" ");
            Concatenate.format(sb, inputDataList[idx].evid, 8).append(" ");
            sb.append(inputDataList[idx].stnChl.getSubsource()).append(" :");
            Concatenate.format(sb, inputDataList[idx].windows, 4).append(" ");
            Concatenate.format(sb, inputDataList[idx].aFix, 5, 2).append(" ");
            Concatenate.format(sb, inputDataList[idx].aFree, 5, 2).append(" ");
            Concatenate.format(sb, inputDataList[idx].qFree, 5, 2).append(" : ");
            Concatenate.format(sb, inputDataList[idx].ml, 5, 2).append(" ");

            double calibratedMagnitude = inputDataList[idx].aFix - currentStats.medianAFix;
            double residual = calibratedMagnitude - inputDataList[idx].ml;
            int intRes = (int) ((residual + 1.05)/0.1);
            if (intRes < 0)  intRes = 0;
            else if (intRes > 20) intRes = 20;
            double value = Math.floor(inputDataList[idx].ml);
            if (value < 0.) value = 0.;
            else if (value > 6.) value = 6.;
            char chr = Character.forDigit((int) value, 10);
            cres[intRes] = chr;

            Concatenate.format(sb, calibratedMagnitude, 5, 2).append(" ");
            Concatenate.format(sb, residual, 5, 2).append(" |");
            sb.append(new String(cres)).append("|");
            log.println(sb.toString());

            cres[intRes] = ' ';
            cres[10] = '|';
        }
    }

    // l1 norm fit of data to determine slope and intercept parameters
    private void calcL1Norm(InputData [] inputDataList) {
        double smallestResidualSum = 999999.;
        double bestFitLineIntercept = 0.;
        double bestFitLineSlope = 0.;

        int currentCodaCount = inputDataList.length;
        double [] x = new double[currentCodaCount];

        for (int idx =0; idx < currentCodaCount; idx++) {
            x[idx] = inputDataList[idx].qFree;
        }
        int [] is = IndexSort.getSortedIndexes(x);

        double [] y = new double[currentCodaCount];
        // DEBUG log.println(" idx sidx       id dev   cnt     afix    afree    qfree        mag");
        for (int idx =0; idx < currentCodaCount; idx++) {

            x[idx] = inputDataList[is[idx]].qFree;
            y[idx] = inputDataList[is[idx]].aFree - inputDataList[is[idx]].ml;

/* DEBUG OUTPUT BELOW HERE
            StringBuffer sb = new StringBuffer(132);
            Concatenate.format(sb, idx, 4).append(" ");
            Concatenate.format(sb, is[idx], 4).append(" ");
            Concatenate.format(sb, inputDataList[is[idx]].evid, 8).append(" ");
            sb.append(inputDataList[is[idx]].stnChl.getSubsource()).append(" :");
            Concatenate.format(sb, inputDataList[is[idx]].windows, 4).append(" ");
            Concatenate.format(sb, inputDataList[is[idx]].aFix, 5, 2).append(" ");
            Concatenate.format(sb, inputDataList[is[idx]].aFree, 5, 2).append(" ");
            Concatenate.format(sb, inputDataList[is[idx]].qFree, 5, 2).append(" : ");
            Concatenate.format(sb, inputDataList[is[idx]].ml, 5, 2).append(" ");
            sb.append(inputDataList[is[idx]].stnChl.toStnChlString());
            log.println(sb.toString());
*/
        }

        for (int idx = 1; idx < currentCodaCount; idx++ ) {
            for (int idx2 = 0; idx2 < idx; idx2++) {
                double numerator   = y[idx2]*x[idx] - y[idx]*x[idx2];
                double denominator = x[idx]  - x[idx2];
                if (denominator < 0.001) denominator = 0.001;
                double trialIntercept = numerator/denominator;
                //double trialSlope = (trialIntercept - y[idx])/x[idx];
                double trialSlope = (y[idx] - trialIntercept)/x[idx];
                double sum = 0.;
                for (int idx3 = 0; idx3 < currentCodaCount; idx3++) {
                    double residual = trialIntercept + x[idx3]*trialSlope - y[idx3];
                    sum += Math.abs(residual);
                }
                if (sum <= smallestResidualSum) {
                    bestFitLineIntercept  = trialIntercept;
                    bestFitLineSlope = trialSlope;
                    smallestResidualSum = sum;
                }
            }
        }

        currentStats.calibrFitSlope = bestFitLineSlope;
        currentStats.calibrFitAFix  = bestFitLineIntercept + bestFitLineSlope * McaCalibrationTN.QFIX_DEFAULT;
    }

    protected int getDataFromDb(DataStnChl stnChl) {
        return getDataFromDb(DataSource.getConnection(), stnChl);
    }

    protected int getDataFromDb(Connection conn, DataStnChl stnChl) {
        currentInputDataCollection.clear(); // reuse existing collection
        if (conn == null) {
                System.err.println ("McaCalDb no connection is open.");
                return 0;
        }
        try {
            if (conn.isClosed()) {
                System.err.println ("McaCalDb DataSource connection is closed");
                return 0;
            }
            ResultSet rsdb = executeQuery(getJDBCStatement(conn), stnChl);
            if (rsdb == null) return 0;        // no data found
            while (rsdb.next()) {
                currentInputDataCollection.add(parseResultSet(rsdb));
            }
            rsdb.close();
        }
        catch (SQLException ex) {
            System.err.println(ex);
            ex.printStackTrace();
        }
        return currentInputDataCollection.size();
    }


    private String createSQLCodaQueryString() {
        StringBuffer sb = new StringBuffer(512);
        sb.append(SQL_CODA_QUERY_STRING);
        if (currentInputDateRange.isLimited()) {
            sb.append(" AND ");
            sb.append(org.trinet.jdbc.table.Coda.toDateTimeConstraintSQLWhereClause("CODA",
                currentInputDateRange.getMinDate(), currentInputDateRange.getMaxDate()));
        }
        sb.append(SQL_CODA_ORDER_BY_STRING);
        if (processedChannelCount < 1) log.println ("getDataFromDb SQL: "+sb.toString()); // DEBUG
        return sb.toString();
    }

    private Statement getJDBCStatement(Connection conn) {
        try {
            if (preparedDbStatement == null) {
                preparedDbStatement = conn.prepareStatement(createSQLCodaQueryString());
            }
        }
        catch (SQLException ex) {
            System.err.println(ex);
            ex.printStackTrace();
        }
        return preparedDbStatement;
    }

    private ResultSet executeQuery(Statement statement, DataStnChl stnChl) {
        ResultSet resultSet = null;
        PreparedStatement sm = (PreparedStatement) statement;
        try {
            sm.setString(1,stnChl.getNet());
            sm.setString(2,stnChl.getSta());
            sm.setString(3,stnChl.getSeedchan());
            resultSet = sm.executeQuery();
        }
        catch (SQLException ex) {
            System.err.println(ex);
            ex.printStackTrace();
        }
        return resultSet;
    }

    protected InputData parseResultSet(ResultSet rsdb) throws java.sql.SQLException{

        InputData inputData = new InputData();

        inputData.coid      = rsdb.getLong(1);    // coda.COID

        inputData.stnChl.setSta(      rsdb.getString(2));  // coda.STA
        inputData.stnChl.setNet(      rsdb.getString(3));  // coda.NET
        inputData.stnChl.setAuth(     rsdb.getString(4));  // coda.AUTH
        inputData.stnChl.setSubsource(rsdb.getString(5));  // coda.SUBSOURCE
        inputData.stnChl.setSeedchan(  rsdb.getString(6));  // coda.SEEDCHAN
        inputData.stnChl.setLocation( rsdb.getString(7));  // coda.LOCATION

        inputData.aFix      = rsdb.getDouble(8);  // coda.AFIX
        inputData.aFree     = rsdb.getDouble(9);  // coda.AFREE
        inputData.qFix      = rsdb.getDouble(10); // coda.QFIX
        inputData.qFree     = rsdb.getDouble(11); // coda.QFREE
        inputData.windows   = rsdb.getInt(12);    // coda.NSAMPLE
        inputData.time      = rsdb.getDouble(13); // coda.DATETIME
        inputData.evid      = rsdb.getLong(14);   // origin.EVID
        inputData.orid      = rsdb.getLong(15);   // assoccoo.ORID
        inputData.magid     = rsdb.getLong(16);   // netmag.MAGNITUDE
        inputData.ml        = rsdb.getDouble(17); // netmag.MAGNITUDE
        return inputData;
    }

    public static void main(String [] args) {
        // alternative is configure  with properties/resource file for debug settings, database info etc.
        int nargs = args.length;
        if (nargs < 1) {
             System.err.println("syntax: McaCalDb hostName [startDateStr endDateStr]");
             System.err.println("example: McaCalDb  k2 2000/10/01 2000/12/31");
             System.exit(0);
        }
        String hostName = args[0].trim();
        String dbName = hostName + "db";
        String dbConnect =  "jdbc:oracle:thin:@" + hostName + ".gps.caltech.edu:1521:" + dbName;
        System.out.println("Connection url: " + dbConnect);

        DataSource ds = new DataSource(
            dbConnect,
            "oracle.jdbc.driver.OracleDriver",
            "trinetdb",
            "calgs",
             true
        );
        EnvironmentInfo.setApplicationName("McaCalDb");
        EnvironmentInfo.setNetworkCode("CI");

        boolean status = false;
        java.util.Date startDate = (nargs > 1) ? EpochTime.stringToDate(args[1], "yyyy/MM/dd") : null;
        java.util.Date endDate   = (nargs > 2) ? EpochTime.stringToDate(args[2], "yyyy/MM/dd") : null;
        try {
            McaCalDb calibrator = new McaCalDb(startDate, endDate);
            status = calibrator.calibrate();
        }
        catch (IOException ex) {
           ex.printStackTrace();
        }
        catch (Exception ex) {
           ex.printStackTrace();
        }
        if (status) DataSource.commit();
        else {
            try {
                DataSource.getConnection().rollback();
            }
            catch (java.sql.SQLException sqlEx) {
                sqlEx.printStackTrace();
            }
        }
        DataSource.close();
        System.exit(0);
    }
}
