package org.trinet.jasi.seed;

import java.io.*;
import java.sql.*;
import org.trinet.jasi.*;
/** Gets waveform timeseries from the data source by using a stored procedure
 *  that returns the data as a blob (Binary Large OBject). The blob contains
 *  miniSeed packets. This allows us to use the dbase server as the waveserver. <p>
 *  The WAVE.GET_WAVEFORM_BLOB(?,?,?,?,?) stored procedure could be implemented
 *  on any database for any waveform data set. */

// AWW 12/2002

public class DbBlobReader {
  CallableStatement csStmt = null;
  byte [] aBuffer = new byte [SeedReader.DEFAULTSEEDBLOCKSIZE];

  /** Read the timeseries for this waveform, decode the miniSeed and put the
   *  resulting timeseries in the Waveform. Returns the size of the resulting
   *  Waveform segment list.*/
  public int getDataFromDataSource(Waveform waveform) {
    return this.getDataFromDataSource(waveform,waveform.getStart().doubleValue(),waveform.getEnd().doubleValue());
  }

  /** Read the timeseries for this waveform, decode the miniSeed and put the
   *  resulting timeseries in the Waveform. Returns the size of the resulting
   *  Waveform segment list.*/
  public int getDataFromDataSource(Waveform waveform, double startTime, double endTime) {
    Waveform wf = waveform;
    double wfStartTime = wf.getStart().doubleValue();
    double wfEndTime   = wf.getEnd().doubleValue();
    if(startTime == 0.) startTime = wfStartTime;
    if(endTime   == 0.) endTime = wfEndTime;
    if (wfStartTime > endTime || wfEndTime < startTime) return 0;
    wf.setAmpUnits(Units.COUNTS);
    String filename  = wf.getPathFilename();
    int traceoff = wf.getFileOffset();
    int nbytes   = wf.getByteCount();
    if (filename == null)  return 0; // could try to get db file info here -aww ??
    if (traceoff < 0 || nbytes <= 0) return 0;
    int status = 0;
    try {
        if (csStmt == null) {
          Connection conn = DataSource.getConnection();
          if (conn == null) throw new SQLException("Null DataSource connection");
          csStmt = conn.prepareCall("{ ? = call WAVE.GET_WAVEFORM_BLOB(?,?,?,?,?) }"); // time range extraction
          csStmt.registerOutParameter(1, Types.BLOB);
        }
        csStmt.setString(2,filename);
        csStmt.setInt(3,traceoff);
        csStmt.setInt(4,nbytes);
        csStmt.setDouble(5,startTime);
        csStmt.setDouble(6,endTime);

        boolean rs = csStmt.execute();
//	if (rs) {
      java.sql.Blob blob = (java.sql.Blob) csStmt.getBlob(1);

      if (blob == null) return 0;
      byte [] wfBytes = blob.getBytes(1,(int) blob.length());
      blob = null;

      if (wfBytes == null || wfBytes.length == 0) return 0;
      status = getDataPacketBytes(wf, wfBytes);
      wfBytes = null;
//	}
      }
    catch (SQLException ex) {
      System.out.println("DbBlobReader error: sta = "+ wf.getChannelObj().toDelimitedSeedString());
      ex.printStackTrace();
    }
    return status;
  }
  /** Decode the miniSeed in the byte array using SeedReader and put the
   *  resulting timeseries in the Waveform. Returns the size of the resulting
   *  Waveform segment list.*/

  public int getDataPacketBytes(Waveform wf, byte [] bytes) {
    try {

      ByteArrayInputStream inBufferStream = new ByteArrayInputStream (bytes);

      while (inBufferStream.available() > 0) {

  // read enough to get header
        inBufferStream.read(aBuffer, 0, SeedReader.FrameSize);

        WFSegment wfseg = SeedReader.createWFSegment(aBuffer);
  SeedHeader hdr = SeedReader.getHeader();

  // you only know the SEED block size and therefore data size
  // AFTER reading the header
  // NOTE: can't assume byte.length will tell you because the array may have
  //       been dimensioned arbitrarily large.
  int dataSize = hdr.seedBlockSize-SeedReader.FrameSize;
        inBufferStream.read(aBuffer, SeedReader.FrameSize, dataSize);

  // suck data from one SEED record
        if ( hdr.isData() ) {
            wfseg.setTimeSeries( SeedReader.decompress (hdr, aBuffer) );
            wf.getSegmentList().add(wfseg);
        }
      } // end of while loop

      inBufferStream.close();
    }
    catch (IOException ex) {
      System.out.println ("IO error: " + ex.toString());
    }
    catch (Exception ex) {
      System.out.println ("General exception: " + ex.toString());
    }
    return wf.getSegmentList().size();
  }

}
