package org.trinet.jasi.TN;

import org.trinet.jasi.*;

import java.text.*;
import java.util.*;

import org.trinet.jdbc.*;
import org.trinet.util.DateTime;
import org.trinet.util.RangeDouble;
//import org.trinet.util.LatLonZ;
import org.trinet.util.gazetteer.LatLonZ;
import org.trinet.util.Format;		// CoreJava printf-like Format class
import org.trinet.util.TimeSpan;

import org.trinet.jdbc.table.*;
import org.trinet.jdbc.table.SeqIds;
import org.trinet.jdbc.datatypes.*;
import java.sql.*;

import org.trinet.jasi.*;
import org.trinet.jasi.coda.*;

/**
 * Component of the Seismic Abstraction Layer. This implementation interfaces
 * to the Berkeley/Caltech/USGS schema in Oracle. <p>
 *
 * All communication with the data source is via the mechanism specified in
 * DataSource.  Therefore, DataSource must be used before any JASI components
 * can read/write to data sources. <p>
 *
 * All data members are DataObjects rather than primative types so they can
 * handle the concepts of "nullness", "mutability", and "change-state".<p>
 *
 * NULLNESS = a field can be null, i.e. unknown or no value. The traditional
 * approach has been to insert zero or some bogus value. In ASCII
 * representations sometimes a field was just left blank. For example: RMS
 * 'distance' might simply be unknown. To give it a value of 0.0km could foul up
 * other calculations. When you write data out to a database you want to leave
 * null fields null and not write a bunch of zeros that were used as an
 * expedient inside of an application.<p>
 *
 * MUTABLITY = some fields should NEVER be explicity manipulated by an
 * application.  These are usually "key" fields that would break the database or
 * cause constraint errors. They would be set immutable. Think of these as
 * "read-only".<p>
 *
 * CHANGE-STATE = this is primarily an efficiency issue. If you read in 1000
 * records, change 10 of them, then save your work, you only want to write
 * ('update') the changed records to the database.<p>
 *
 * The database can contain null values for
 * certain fields. Also applications may not want to write a "bogus" value like 0
 * into the database when when the real value is not known. Data primatives can't
 * be null and are initiallize to zero.  */

/* NCDC Schema map

  [EVENT]--[Origin]-+------------[[[AssocAmO]]]--+
                    |                            +--[[[Amp]]
                    +--[NetMag]--[[[AssocAmM]]]--+
                    |
        +------------[[[AssocArO]]]-----[[[Arrival]]]
 */

public class SolutionTN extends Solution {
    /*
      Here are protected data members used to support this particular
      implimentation of the abstract layer */

    // Schema classes that map to tables. Keep these for update, delete, etc.
    protected Event  eventRow;
    protected Origin originRow;
    // foreign keys
    /*
      The fields prefmag, prefmec & commid exist in BOTH the Event and
      Origin tables. Which to use??  Made decision to use those in Origin.  */
    protected DataLong prefor   	= new DataLong();
    protected DataLong prefmag   	= new DataLong();
    protected DataLong prefmec   	= new DataLong();
    protected DataLong commid   	= new DataLong();

    // flags
    protected DataLong bogusFlag   	= new DataLong();
    protected DataLong selectFlag   	= new DataLong();

    /** True if this Solution was orginally read from the dbase. Used to know if
  a commit requires an 'insert' or an 'update' */
    boolean fromDbase = false;

    // Dbase connection objects
    /** Database connection to be used by an instance of Solution */
    // NOTE: static calls must pass connection as arg OR use default DataSource connection
//    protected static Connection conn;
    protected static PreparedStatement psCount = null;	// Waveform counter prepared statement

    // The basic event/origin/netmag join
    //    protected static final String sqlPrefix =
    //	"Select event.*, origin.*, netmag.* from event, origin, netmag ";

    protected static final String sqlPrefix =
  "Select "+
  Event.QUALIFIED_COLUMN_NAMES+","+
  Origin.QUALIFIED_COLUMN_NAMES+","+
  NetMag.QUALIFIED_COLUMN_NAMES+
  " from event, origin, netmag ";

    /* Join: note that we must do an "outer" join by using (+). Otherwise, an
       event that lacks an 'prefor' or 'prefmag' will NOT be seen. (see: Oracel8
       Bible, McCullough-Dieter, pg. 217) */
    protected static final String sqlJoin = sqlPrefix +
  " WHERE (event.prefor = origin.orid(+)) and (event.prefmag = netmag.magid(+)) ";

    // debug output control flag
    boolean debug = true;
    //boolean debug = false;

    /** Collection containing alternate (non-preferred) solutions */
    //    ArrayList altSols = new ArrayList();

    /** Collection containing alternate (non-preferred) magnitudes */
    //    ArrayList altMags = new ArrayList();

    /**
     *  Create a Solution object.
     * A new solution must have a unique id at the time it is created.
     */

    public SolutionTN () {
  //	setConnection(DataSource.conn);	// use default connection
    }

    /**
     *  Create a Solution object. DataSource must be stablished first.
     */
/*
    public SolutionTN (Connection conn) {
  setConnection(conn);
    }
*/
    /**
     * Explicitely set or change the dbase connection to be used by a solution object.
     */
/*    public static void setConnection (Connection connection) {
  conn = connection;
  prepCountStatement(conn);
    }
*/
    /** Return the connection object. If none was explicitly set returns the
    * DataSource default connection. If none return null. */
    public static Connection  getConnection() {
//      if (conn == null) setConnection(DataSource.getConnection());
      return DataSource.getConnection();
//      return conn;
    }
    /**
     * Make a prepared statement for counting waveforms for a solution.
     * This is for efficiency. Use the default DataSource connection.
     */
    protected static void prepCountStatement()  {
      prepCountStatement(getConnection());
    }
    /**
     * Make a prepared statement for counting waveforms for a solution.
     * This is for efficiency.
     */
    protected static void prepCountStatement(Connection conn)  {

  if (conn == null) return;

  try {
      psCount =
      conn.prepareStatement("Select COUNT(wfid) from AssocWaE where evid = ?" );
  } catch	 (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();
  }

    }

    /**
     * Get the next valid solution (event) sequence number (evseq).
     * Returns the unique value. Returns 0 if there is an error.
     */
    public long setUniqueId() {
  long newId = SeqIds.getNextSeq("evseq");
  id.setValue(newId);
  return newId;
    }

    /**
     * Set the internal copy of the Event table object for later use.
     */
    protected void setEventRow (Event evt) {
  eventRow = evt;
    }

    /**
     * Set the internal copy of the Origin table object for later use.
     */
    protected void setOriginRow (Origin org)  {
  originRow = org;
    }

/**
 * Set value of 'waveRecords' for this Solution by examining the datasource,
 * if available.
 */
    public int countWaveforms(){
  // We can't do this at instatiation time because the Connection is not created yet.

  if (psCount == null) prepCountStatement(getConnection());

// make the query
  try
  {
      psCount.setLong(1, this.id.longValue());

      ResultSet rset = psCount.executeQuery();

      if (!rset.next()) return 0;	// no resultset

      int wcount = rset.getInt(1);

      waveRecords.setValue(wcount);

      rset.close();

      return wcount;
   }
   catch (SQLException ex) {
            System.err.println("countWaveforms SQLException: " + ex.getMessage());
            System.err.println(ex);
      return 0;
         }
    }

    /** Set the event type to a valid jasi eventType. If the type passed in 'type'
     * is not valid 'false' is returned and the eventType is not changed.
     * @see org.trinet.jasi.EventTypeMap */
    public boolean setEventType (String type) {

      return setEventType (EventTypeMap.getIntOfJasiCode(type));

    }

     /** Set the event type. If the type passed in 'type'
     * is not valid 'false' is returned and the eventType is not changed.
     * @see org.trinet.jasi.EventTypeMap */
    public boolean setEventType (int type) {

     if (!EventTypeMap.isValid(type)) return false;

     eventType.setValue(EventTypeMap.get(type));

     return true;
    }

    /** Return the local implentation's event type string.
     * @see org.trinet.jasi.EventTypeMap */
    public String getEventTypeString () {

       return eventType.toString();

    }

    /**
     * Returns array of Solutions based on the results of the SQL query to an
     *		NCDCv1.5  data base. It must be of the form:<p> "Select
     *		event., origin., netmag. from event, origin, netmag " <p>
     *		because you must do a join of Event/Origin/NetMag to get all the
     *		info we need.  Returns null if no event is found.<p> * Uses
     *		default connection created by DataSource.  */
    public static SolutionTN[] getBySQL(String sql)
    {
  if (getConnection() == null)  {
    System.err.println ("* No DataSource is open.");
    return null;
      }
  return getBySQL(getConnection(), sql);
    }

    /**
     * Returns array of Solutions based on the results of the SQL query to an
     *		NCDCv1.5 * data base. It must be of the form:<p> "Select
     *		event.*, origin.*, netmag.* from event, origin, netmag " <p> *
     *		because you must do a join of Event/Origin/NetMag to get all the
     *		info we need.  * Returns null if no event is found.  */
    protected static SolutionTN[] getBySQL(Connection connection, String sql)  {

  Vector vec = new Vector();

  // Debug
  //System.out.println ("getBYSQL(Connection, String) SQL\n: "+sql);

  try {
      if ( getConnection() == null || getConnection().isClosed() )	// check that valid connection exists
    {
        System.err.println ("* DataSource connection is closed");
        return null;
    }

      // ** Removed: don't use Oracle locking
      // do "select for update" if writeBack is enabled
      //	    ExecuteSQL.setSelectForUpdate(DataSource.isWriteBackEnabled());

      Statement sm = getConnection().createStatement();

      ResultSetDb rsdb = new ResultSetDb(ExecuteSQL.rowQuery(sm, sql));

      if (rsdb == null) {
  //	System.out.println("getBySQL(Connection, String) rsdb is null");
    return null;	// nothing found
      }

      while ( rsdb.getResultSet().next() ) {

    vec.add(SolutionTN.parseResultSet(rsdb));
      }
      sm.close();

  }
  catch (SQLException ex) {
      System.out.println ("getBYSQL(Connection, String) SQL=\n"+sql);
      System.err.println(ex);
      ex.printStackTrace();
  }

  // convert vector to array
  if (vec.size() == 0) return null;

  SolutionTN sol[] = new SolutionTN[vec.size()];
  vec.copyInto(sol);

  dbFetchComments(sol);

  return sol;
    }
    /**
     * Returns the Solution with this ID number from the data source.
     * Uses default DataSource.
     * Returns null if no event is found.
     * For the NCDC v1.5 schema ID is the 'evid' not the 'orid'.
     */
    public Solution getById(long id){
  return getById(getConnection(), id);
    }
    /**
     * Returns the Solution with this ID number from the data source.
     * Returns null if no event is found.
     * For the NCDC v1.5 schema ID is the 'evid' not the 'orid'.
     */
    public Solution getById(Connection conn, long id){
  // must do a join of Event/Origin/NetMag to get all the info we need
  String sql = sqlJoin + " and (event.evid = " + id +")" ;

  Solution sol[] = SolutionTN.getBySQL(conn, sql);

  if (sol == null) return null;

  return sol[0];

    }
    /**
     * Returns the VALID Solution with this ID number from the data source.
     * Uses default DataSource.
     * Returns null if no event is found.
     * For the NCDC v1.5 schema ID is the 'evid' not the 'orid'.
     */
    public Solution getValidById(long id){
  return getValidById(getConnection(), id);
    }
    /**
     * Returns the VALID Solution with this ID number from the data source.
     * Returns null if no event is found.
     * For the NCDC v1.5 schema ID is the 'evid' not the 'orid'.
     */
    public Solution getValidById(Connection conn, long id) {
  // must do a join of Event/Origin/NetMag to get all the info we need
  String sql = sqlJoin + " and (event.evid = " + id +
            ") and (selectflag = 1) and (bogusflag = 0) ";

  Solution sol[] = SolutionTN.getBySQL(conn, sql);

  if (sol == null) return null;

  return sol[0];

    }

    /**
     * Returns the Solution with this ID number from the data source.
     * Uses default DataSource.
     * Returns null if no event is found.
     * For the NCDC v1.5 schema ID is the 'evid' not the 'orid'.
     */
    public static SolutionTN getByOrid(long orid){
  return getByOrid(getConnection(), orid);
    }
    /**
     * Returns the Solution with this ID number from the data source.
     * Returns null if no event is found.
     * For the NCDC v1.5 schema ID is the 'evid' not the 'orid'.
     */
    public static SolutionTN getByOrid(Connection conn, long orid) {
  // must do a join of Event/Origin/NetMag to get all the info we need
  String sql = sqlJoin + " and (event.prefor = " + orid +")" ;

  SolutionTN sol[] = SolutionTN.getBySQL(conn, sql);

  if (sol == null) return null;

  return sol[0];

    }

    /**
     * Returns array of Solutions within this time window.
     * Returns null if no event is found.
     */
    public Solution[] getByTime(double start, double stop) {
  return getByTime (getConnection(), start, stop);
    }

    /**
     * Returns array of Solutions within this time window.
     * Returns null if no event is found.
     */
    public Solution[] getByTime(Connection conn, double start, double stop) {
  // must do a join of Event/Origin/NetMag to get all the info we need
//	String sql = sqlJoin + "and Origin.DATETIME BETWEEN " +
//	    StringSQL.valueOf(start) + " AND " + StringSQL.valueOf(stop) +

  String sql = sqlJoin + getTimeSpanSQL(start, stop) +
      " order by Origin.Datetime";

  return getBySQL(conn, sql);

    }

    /**
     * Returns array of"valid" Solutions within this time window.
     * Some data sets contain solutions that have duplicates, interim, or bogus entries.
     * This method filters those out and returns only VALID solutions.
     */
    public Solution[] getValidByTime (double start, double stop) {
  return getValidByTime (getConnection(), start, stop);
    }
    /**
     * Returns array of"valid" Solutions within this time window.
     * Some data sets contain solutions that have duplicates, interim, or bogus entries.
     * This method filters those out and returns only VALID solutions.
     */
    public Solution[] getValidByTime (Connection conn, double start, double stop) {
  String sql = sqlJoin + " and "+getTimeSpanSQL(start, stop) +
     //" and (Origin.DATETIME BETWEEN " +
  //    StringSQL.valueOf(start) + " AND " + StringSQL.valueOf(stop) + ") "+
      " and (selectflag = 1) "+
      " and (bogusflag = 0)" +
      " order by Origin.Datetime";

  //NOTE: bogusflag is wrongly set true for all data as of 10/13/99

  return getBySQL(conn, sql);
    }

    /** Return the part of an sql query that specifies a time window.
     *  Note that "BETWEEN" is much faster that ">" */
    static String getTimeSpanSQL (double start, double stop) {
    return " Origin.DATETIME BETWEEN " +
      StringSQL.valueOf(start) + " AND " + StringSQL.valueOf(stop);
    }

    /**
     * Return array of solutions that match the properties defined in the
     * EventSelectionProperties object */
    public Solution[] getByProperties (EventSelectionProperties props) {

  return getByProperties (getConnection(), props);
    }

    /**
     * Return array of solutions that match the properties defined in the
     * EventSelectionProperties object */

    // TODO: this only uses time and valid and dummy flags !!!
    public Solution[] getByProperties (Connection conn,
               EventSelectionProperties props) {

     String str;

     // Get time from properties. This will handle both "absolute" and "relative"
     // time definitions

     TimeSpan span = props.getTimeSpan();
     double start = span.getStart();
     double stop  = span.getEnd();

//	double start = props.getDateTime("startTime").getEpochSeconds() ;
//	double stop  = props.getDateTime("endTime").getEpochSeconds() ;

     // time bounds
     String where = getTimeSpanSQL(start, stop);

     // validFlag
     str = props.getProperty("validFlag");
     if (str != null) {
    if (str.equalsIgnoreCase("TRUE")) {

         where += " and (Event.selectFlag = 1) ";
    } else {
         where += " and (Event.selectFlag = 0) ";
       }
  }

     // dummyFlag
     str = props.getProperty("dummyFlag");
     if (str != null) {
    if (props.getProperty("dummyFlag").equalsIgnoreCase("TRUE")) {

        where += " and (Origin.bogusFlag = 1) ";
    } else {
        where += " and (Origin.bogusFlag = 0) ";
       }
  }

     // Event types (e.g. "local", "sonic", etc.) - these must be OR'ed
     // These properties have the form:  SelectAttribute_trigger = FALSE
     // There may not be a property for each type, thus the check for != null.
      String typeChoice[] = EventTypeMap.getEventTypeArray();

       EventTypeMap typeMap = EventTypeMap.create();

       String subStr = "";

       // explicite includes, logic is OR = (property = TRUE)
       for (int i = 0; i< typeChoice.length; i++) {
           str = props.getProperty(EventSelectionProperties.prefix+typeChoice[i]);
           if (str != null && str.equalsIgnoreCase("TRUE")) {
             if (!subStr.equals("")) subStr += " or ";    // need this if more then one
             subStr += " Event.ETYPE = '"+typeMap.toLocalCode(typeChoice[i])+"' ";
           }
       }

       if (!subStr.equals("")) where += " and (" + subStr + ") ";

       // explicite excludes, logic is AND != (property = FALSE)
       subStr = "";
       for (int i = 0; i< typeChoice.length; i++) {
           str = props.getProperty(EventSelectionProperties.prefix+typeChoice[i]);
           if (str != null && str.equalsIgnoreCase("FALSE")) {
             if (!subStr.equals("")) subStr += " and ";    // need this if more then one
             subStr += " Event.ETYPE != '"+typeMap.toLocalCode(typeChoice[i])+"' ";
           }
       }

       if (!subStr.equals("")) where += " and (" + subStr + ") ";

       // processing states - these must be OR'ed
       subStr = "";

       // get list of possible processing states (e.g. "A", "H", etc.)
          String label[] = ProcessingState.getLabelArray();
          String tag[]   = ProcessingState.getTagArray();

          for (int i = 0; i< label.length; i++)
          {
           // if property exists set initial state of check box
           str = props.getProperty(EventSelectionProperties.prefix+label[i]);
           if (str != null && str.equals("TRUE")) {
           if (!subStr.equals("")) subStr += " or ";    // need this if more then one
             subStr += " Origin.rflag = '"+tag[i]+"' ";
           }
          }

       if (!subStr.equals("")) where += " and (" + subStr + ")";

     // sort by time
  where += " order by Origin.Datetime";

  return (Solution[]) SolutionTN.getWhere(where);

    }

    /** Get using this SQL WHERE clause */
    public static SolutionTN[] getWhere (String whereClause) {

  String sql = sqlJoin+ " and " + whereClause;

     // debug
//	System.out.println (sql);

  return getBySQL(sql);
    }

    /** Get the comment from the dbase if 'includeComment' flag is set and
    * the Event has one. */
    static protected void dbFetchComment(SolutionTN sol) {
     if (sol.includeComment && !sol.commid.isNull()) {
        Remark rmk = new Remark(getConnection());
        Remark rmks[] = rmk.getRowsByCommId(sol.commid.longValue());
        String newStr = "";
        for (int i = 0; i < rmks.length; i++) {
           newStr += rmks[i].toStringRemark();
        }
        sol.comment.setValue(newStr);  // this sets isUpdate(true)
        sol.comment.setUpdate(false);  // it shouldn't be if its from the dbase
     }
    }
    /** Get the comment from the dbase if 'includeComment' flag is set and
    * the Event has one. */
    static protected void dbFetchComments(SolutionTN[] sol) {
        for (int i = 0; i < sol.length; i++) {
          dbFetchComment(sol[i]);
        }
    }
    /**
     * Parse a resultset row that contains a concatinated Event/Origin/NetMag
     */
    protected static SolutionTN parseResultSet (ResultSetDb rsdb)  {

  SolutionTN sol = new SolutionTN();

  // note: these are local instances
  Event  eventRow  = new Event();
  Origin originRow = new Origin();

  NetMag netMagRow = new NetMag();

  int offset = 0;

  // note we are using the class instances of event/orgin/netmag

  //	System.out.println ("parsing event...");
  // copy data from resultset to DataTableRow object
        eventRow = (Event) eventRow.parseOneRow(rsdb, offset);
  eventRow.setProcessing(Event.NONE);

  // allow future changes to this DataTableRow
  eventRow.setMutable(true);

  offset += Event.MAX_FIELDS;

  // 	System.out.println ("parsing origin...");
  originRow = (Origin) originRow.parseOneRow(rsdb, offset);
  originRow.setProcessing(Origin.NONE);

  offset += Origin.MAX_FIELDS;

  // don't keep a copy of NetMag in Solution, there will be one in Magnitude
  //	System.out.println ("parsing mag...");
  netMagRow = (NetMag) netMagRow.parseOneRow(rsdb, offset);
  netMagRow.setProcessing(NetMag.NONE);

  sol.fromDbase = true;

  sol.parseEventRow(eventRow);
  sol.parseOriginRow(originRow);

  // allow changes to this DataTableRow. Need to set this because we are
  // not using our own rather than org.trinet.jdbc parsing methods
  if (DataSource.isWriteBackEnabled()) {
      eventRow.setMutable(true);
      originRow.setMutable(true);
      netMagRow.setMutable(true);

      // allow modification of the EVENT table
      eventRow.setSelectForUpdate(true);

  }

  // a new magnitude object
  sol.magnitude = Magnitude.create();

  ((MagnitudeTN)sol.magnitude).parseNetMagRow(netMagRow);

  sol.magnitude.associate(sol);

  // remember DataTableRows for later writeback
  sol.eventRow  = eventRow;
  sol.originRow = originRow;

  // calculate the priority
  SolutionTN.calcPriority(sol);

  return sol;
    }

    /** Calculate the priority of this solution. */
    static void calcPriority(Solution sol) {

  //sol.priority.setValue(EventPriorityModelTN.getPriority(conn, sol));
      sol.priority.setValue(EventPriorityModelTN.getPriority(sol));

    }

    /**
     * Suck contents out of an Event TableRow to populate this Solution.
     */
    // Note that this copies DataObjects not references.

    protected SolutionTN parseEventRow(Event evt) {

  this.id = (DataLong) evt.getDataObject(Event.EVID);
  id.setUpdate(false);	// DataTableRow weirdness: sets key field isUpdate = true

     // parent id is not currently stored in the dbase
     setParentId(id.longValue());
     parentId.setUpdate(false);

  // foreign keys (PROTECTED)
  this.prefor  = (DataLong) evt.getDataObject(Event.PREFOR);

  /*
    The fields prefmag, prefmec & commid exist in BOTH the Event and
    Origin tables Which to use??  Made decision to use those in Origin.  */
  //	this.prefmag = (DataLong) evt.getDataObject(Event.PREFMAG);
  //	this.prefmec = (DataLong) evt.getDataObject(Event.PREFMEC);
  //	this.commid  = (DataLong) evt.getDataObject(Event.COMMID);

     // This gets a NATIVE event type, must translate to generic (jasi) type
  String estr = ((DataString) evt.getDataObject(Event.ETYPE)).toString();
  this.setEventType(EventTypeMap.toJasiCode(estr));
  this.eventType.setUpdate(false);     // don't mark as changed by program

     // Event auth and source are distinct from Origin's
  this.eventAuthority	    = (DataString) evt.getDataObject(Event.AUTH);
  this.eventSource	    = (DataString) evt.getDataObject(Event.SUBSOURCE);

  this.validFlag = (DataLong)   evt.getDataObject(Event.SELECTFLAG);

  return this;
    }

    /**
     * Put contents of this Solution back into the original Event TableRow object.
     * If there was none, make one.
     * This is the reverse of parseEvent().
     */

    // Note that this copies DataObjects not references.

    protected void toEventRow () {

  // new evid could have been assigned already
  if (id.isNull()) id.setValue(setUniqueId());	// set if not assigned

  // create an Event object if there wasn't one we read from originally
  if (eventRow == null) {

    eventRow = new Event(id.longValue());
    eventRow.setValue(Event.AUTH, getEventAuthority());
    eventRow.setValue(Event.SUBSOURCE, getEventSource());
  }

  // allow updating the row
  eventRow.setUpdate(true);

  /*
    The fields prefmag & prefmec exist in BOTH the Event & Origin tables
    Which to use?? Made decision to use those in Origin.
   */

  // MAKE SURE "NOT NULL" COLUMNS HAVE VALUES
  if (id.isUpdate())	  eventRow.setValue(Event.EVID, id);

     /* Do NOT change Event.subsource and Event.AUTH if it was originally from
         the dbase, else you'll overwrite the original source of a relocated event.
         See: toOriginRow().
         However, a brand new event must populate these attributes. */
//     if (!fromDbase) {
//	  eventRow.setValue(Event.AUTH, getEventAuthority());
//	  eventRow.setValue(Event.SUBSOURCE, getEventSource());
//     }

  if (eventAuthority.isUpdate()) eventRow.setValue(Event.AUTH, getEventAuthority());
  if (eventSource.isUpdate())    eventRow.setValue(Event.SUBSOURCE, getEventSource());

    // foreign keys (PROTECTED)
  if (commid.isUpdate())    eventRow.setValue(Event.COMMID, commid);

  if (prefmag.isUpdate())   eventRow.setValue(Event.PREFMAG, prefmag);
  if (prefmec.isUpdate())   eventRow.setValue(Event.PREFMEC, prefmec);
  if (prefor.isUpdate())    eventRow.setValue(Event.PREFOR, prefor);

     // must convert to TriNet style 2-char event type
  if (eventType.isUpdate())
         eventRow.setValue(Event.ETYPE, EventTypeMap.create().toLocalCode(eventType.toString()));

  // note: validFlag is written to both Event.selectflag and Origin.bogusflag
  if (validFlag.isUpdate()) eventRow.setValue(Event.SELECTFLAG, validFlag);
    }

    /**
     * Suck contents out of an Origin (TableRow) object to populate
     * DataObjects of this Solution.
     */
    // Note that this gets COPIES of DataObjects not references.

    protected SolutionTN parseOriginRow(Origin org) {

  // use EVID we got from EVENT table if avail
  if (id.isNull()) {
      this.id = (DataLong) org.getDataObject(Origin.EVID);
      id.setUpdate(false);	// DataTableRow weirdness: sets key field isUpdate = true
  }

  // foreign keys (PROTECTED)
  this.prefmag       = (DataLong) org.getDataObject(Origin.PREFMAG);
  this.prefmec       = (DataLong) org.getDataObject(Origin.PREFMEC);
  this.commid        = (DataLong) org.getDataObject(Origin.COMMID);

  this.datetime	    = (DataDouble) org.getDataObject(Origin.DATETIME);
  this.lat           = (DataDouble) org.getDataObject(Origin.LAT);
  this.lon           = (DataDouble) org.getDataObject(Origin.LON);
  this.depth	    = (DataDouble) org.getDataObject(Origin.DEPTH);

  this.horizDatum    = (DataString) org.getDataObject(Origin.DATUMHOR);
  this.vertDatum	    = (DataString) org.getDataObject(Origin.DATUMVER);

  this.type	         = (DataString) org.getDataObject(Origin.TYPE);
  this.method	    = (DataString) org.getDataObject(Origin.ALGORITHM);
// NCDC schema cmodel & vmodel are numbers not strings.
// For now just turn the number into a string.
// TODO: Look in a data dictionary for a translation?
  DataLong modelId   = (DataLong)   org.getDataObject(Origin.CMODELID);
  this.crustModel.setValue(modelId.toString());
  this.crustModel.setUpdate(false);            // don't mark as changed by program

  DataLong velModel  = (DataLong)   org.getDataObject(Origin.VMODELID);
  this.velModel.setValue(velModel.toString());
  this.velModel.setUpdate(false);              // don't mark as changed by program

  this.authority	    = (DataString) org.getDataObject(Origin.AUTH);
  this.source	    = (DataString) org.getDataObject(Origin.SUBSOURCE);

  this.gap           = (DataDouble) org.getDataObject(Origin.GAP);
  this.distance	    = (DataDouble) org.getDataObject(Origin.DISTANCE);
  this.rms           = (DataDouble) org.getDataObject(Origin.WRMS);
  this.errorTime	    = (DataDouble) org.getDataObject(Origin.STIME) ;
  this.errorHoriz    = (DataDouble) org.getDataObject(Origin.ERHOR) ;
  this.errorVert	    = (DataDouble) org.getDataObject(Origin.SDEP);
  this.errorLat      = (DataDouble) org.getDataObject(Origin.ERLAT) ;
  this.errorLon      = (DataDouble) org.getDataObject(Origin.ERLON) ;

  this.totalReadings = (DataLong)  org.getDataObject(Origin.TOTALARR) ;
  this.usedReadings  = (DataLong)  org.getDataObject(Origin.NDEF) ;
  this.sReadings	    = (DataLong)  org.getDataObject(Origin.NBS) ;
  this.firstMotions  = (DataLong)  org.getDataObject(Origin.NBFM) ;
  this.externalId    = (DataString) org.getDataObject(Origin.LOCEVID);
  this.quality       = (DataDouble) org.getDataObject(Origin.QUALITY);

  this.dummyFlag     = (DataLong) org.getDataObject(Origin.BOGUSFLAG);
  this.processingState = (DataString) org.getDataObject(Origin.RFLAG);

        // fixed flags: default is "false" and "null" so if null or NOT="Y" they're left false.
        String fix = ((DataString) org.getDataObject(Origin.FDEPTH)).toString();
  if (fix.equalsIgnoreCase("y")) depthFixed = new DataBoolean(true);

  fix = ((DataString) org.getDataObject(Origin.FEPI)).toString();
  if (fix.equalsIgnoreCase("y")) locationFixed = new DataBoolean(true);

  fix = ((DataString) org.getDataObject(Origin.FTIME)).toString();
  if (fix.equalsIgnoreCase("y")) timeFixed = new DataBoolean(true);

  return this;

    }

/**
 * Return true if any Event or Origin field is different from what's in the dbase.
 * Either, 1) its been changed and not saved or 2) it is newly created and not saved.
 * Does NOT check the magnitude. If you want the status of the magnitude you
 * need to check that with magnitude.hasChanged() */
public boolean hasChanged () {

          return  eventHasChanged() || originHasChanged();
}
/**
 * Set the isUpdate() flag for all data dbase members the given boolean value.  */
public void setUpdate (boolean tf) {

       fromDbase = !tf;
       eventSetUpdate(tf);
       originSetUpdate(tf);
}
/**
 * Return true if any Event field has changed from what was read in from the
 * dbase. Returns true if phase was not originally from the dbase.  */
protected boolean eventHasChanged () {

  if (!fromDbase )           return true;

  if (id.isUpdate())	       return true;
  if (eventAuthority.isUpdate())  return true;
  if (eventSource.isUpdate())     return true;

  if (commid.isUpdate())     return true;
  if (comment.isUpdate())     return true;
  if (prefmag.isUpdate())    return true;
  if (prefmec.isUpdate())    return true;
  if (prefor.isUpdate())     return true;

  if (eventType.isUpdate())  return true;
  if (validFlag.isUpdate())  return true;    // appears in Origin, too.

     return false;
}

/**
 * Set the setUpdate() flag for all data dbase members the given boolean value.  */
protected void eventSetUpdate (boolean tf) {

  id.setUpdate(tf);
  eventAuthority.setUpdate(tf);
  eventSource.setUpdate(tf);

  commid.setUpdate(tf);
  prefmag.setUpdate(tf);
  prefmec.setUpdate(tf);
  prefor.setUpdate(tf);

  eventType.setUpdate(tf);
  validFlag.setUpdate(tf);
}

/**
 * Return true if any field has changed from what was read in from the
 * dbase. Returns true if phase was not originally from the dbase.  */
protected boolean originHasChanged () {

  if (!fromDbase)          return true;

  if (datetime.isUpdate())   return true;
  if (lat.isUpdate())        return true;
  if (lon.isUpdate())        return true;
  if (depth.isUpdate())	  return true;
  if (horizDatum.isUpdate()) return true;
  if (vertDatum.isUpdate())  return true;

  if (type.isUpdate())       return true;
  if (method.isUpdate())     return true;
  if (crustModel.isUpdate()) return true;
  if (velModel.isUpdate())   return true;
  if (authority.isUpdate())  return true;
  if (source.isUpdate())     return true;

  if (gap.isUpdate())        return true;
  if (distance.isUpdate())   return true;
  if (rms.isUpdate())        return true;
  if (errorTime.isUpdate())  return true;
  if (errorHoriz.isUpdate()) return true;
  if (errorVert.isUpdate())  return true;
  if (errorLat.isUpdate())   return true;
  if (errorLon.isUpdate())   return true;

  if (totalReadings.isUpdate()) return true;
  if (usedReadings.isUpdate()) return true;
  if (sReadings.isUpdate())  return true;
  if (firstMotions.isUpdate()) return true;
  if (externalId.isUpdate()) return true;
  if (quality.isUpdate())    return true;

  if (processingState.isUpdate()) return true;
  if (validFlag.isUpdate()) return true;    // appears in Event, too.
  if (dummyFlag.isUpdate()) return true;

  if (depthFixed.isUpdate()) return true;
  if (locationFixed.isUpdate()) return true;
  if (timeFixed.isUpdate())  return true;

    return false;
}


/**
 * Set the setUpdate() flag for all data dbase members the given boolean value.  */
protected void originSetUpdate (boolean tf) {

  datetime.setUpdate(tf)   ;
  lat.setUpdate(tf)        ;
  lon.setUpdate(tf)        ;
  depth.setUpdate(tf)	  ;
  horizDatum.setUpdate(tf) ;
  vertDatum.setUpdate(tf)  ;

  type.setUpdate(tf)       ;
  method.setUpdate(tf)     ;
  crustModel.setUpdate(tf) ;
  velModel.setUpdate(tf)   ;
  authority.setUpdate(tf)  ;
  source.setUpdate(tf)     ;

  gap.setUpdate(tf)        ;
  distance.setUpdate(tf)   ;
  rms.setUpdate(tf)        ;
  errorTime.setUpdate(tf)  ;
  errorHoriz.setUpdate(tf) ;
  errorVert.setUpdate(tf)  ;
  errorLat.setUpdate(tf)   ;
  errorLon.setUpdate(tf)   ;

  totalReadings.setUpdate(tf) ;
  usedReadings.setUpdate(tf) ;
  sReadings.setUpdate(tf)  ;
  firstMotions.setUpdate(tf) ;
  externalId.setUpdate(tf) ;
  quality.setUpdate(tf)    ;

  processingState.setUpdate(tf) ;
  validFlag.setUpdate(tf) ;
  dummyFlag.setUpdate(tf) ;

  depthFixed.setUpdate(tf) ;
  locationFixed.setUpdate(tf) ;
  timeFixed.setUpdate(tf)  ;
}
    /**
     * Stuff contents of this Solution into a NEW Origin (TableRow) object.
     */

    protected void toOriginRow () {
        long newOrid = getOrid();  // it will be null so a new one will be seq'ed

  // make a NEW Origin table row object
  originRow = new Origin(newOrid,
             id.longValue(),
             getAuthority());

  // allow updating the row
  originRow.setUpdate(true);

  // set NOT NULL values
  //(already done above)// originRow.setValue(Origin.AUTH, getAuthority());

  // set NOT NULL values (these were NOT set in constructor)
  if (datetime.isNull()) {
      originRow.setValue(Origin.DATETIME, 0);
  } else {
      originRow.setValue(Origin.DATETIME, datetime);
  }
  if (lat.isNull()) {
      originRow.setValue(Origin.LAT, 0.0);
  } else {
      originRow.setValue(Origin.LAT, lat);
  }
  if (lon.isNull()) {
      originRow.setValue(Origin.LON, 0.0);
  } else {
      originRow.setValue(Origin.LON, lon);
  }

  // foreign keys (PROTECTED)
  //WARNING:
  // This will cause an UPDATE even if nothing has changed !!!
  if (!commid.isNull())     originRow.setValue(Origin.COMMID, commid);
  if (!prefmag.isNull())    originRow.setValue(Origin.PREFMAG, prefmag);
  if (!prefmec.isNull())    originRow.setValue(Origin.PREFMEC, prefmec);

  if (!depth.isNull())	 originRow.setValue(Origin.DEPTH, depth);
  if (!horizDatum.isNull()) originRow.setValue(Origin.DATUMHOR, horizDatum);
  if (!vertDatum.isNull())  originRow.setValue(Origin.DATUMVER, vertDatum);
  if (!type.isNull())       originRow.setValue(Origin.TYPE, type);
  if (!method.isNull())     originRow.setValue(Origin.ALGORITHM, method);
     // note: crustModel is a String, dbase expects an int!
/*
     int cmodelId = getCrustalModelId(crustModel);
     int vmodelId = getVelocityModelId(velModel);
  if (!crustModel.isNull()) originRow.setValue(Origin.CMODELID, cmodelId);
  if (!velModel.isNull())   originRow.setValue(Origin.VMODELID, vmodelId);
*/
  if (!source.isNull())     originRow.setValue(Origin.SUBSOURCE, getSource());
  if (!gap.isNull())        originRow.setValue(Origin.GAP, gap);
  if (!distance.isNull())   originRow.setValue(Origin.DISTANCE, distance);
  if (!rms.isNull())        originRow.setValue(Origin.WRMS, rms);
  if (!errorTime.isNull())  originRow.setValue(Origin.STIME, errorTime);
  if (!errorHoriz.isNull()) originRow.setValue(Origin.ERHOR, errorHoriz);
  if (!errorVert.isNull())  originRow.setValue(Origin.SDEP, errorVert);
  if (!errorLat.isNull())   originRow.setValue(Origin.ERLAT, errorLat);
  if (!errorLon.isNull())   originRow.setValue(Origin.ERLON, errorLon);

  if (!totalReadings.isNull()) originRow.setValue(Origin.TOTALARR, totalReadings);
  if (!usedReadings.isNull())  originRow.setValue(Origin.NDEF, usedReadings);
  if (!sReadings.isNull())     originRow.setValue(Origin.NBS,sReadings );
  if (!firstMotions.isNull())  originRow.setValue(Origin.NBFM,firstMotions );
  if (!externalId.isNull())    originRow.setValue(Origin.LOCEVID, externalId);
  if (!quality.isNull())       originRow.setValue(Origin.QUALITY, quality);

  if (!dummyFlag.isNull())     originRow.setValue(Origin.BOGUSFLAG, dummyFlag);

  if (!processingState.isNull())
                               originRow .setValue(Origin.RFLAG, processingState);

     // translate fixed flags
  if (!depthFixed.isNull()) {
        if (depthFixed.booleanValue()) {
           originRow.setValue(Origin.FDEPTH, "y");
        } else {
           originRow.setValue(Origin.FDEPTH, "n");
        }
     }

  if (!locationFixed.isNull()) {
        if (locationFixed.booleanValue()) {
           originRow.setValue(Origin.FEPI, "y");
        } else {
           originRow.setValue(Origin.FEPI, "n");
        }
     }

  if (!timeFixed.isNull()) {
        if (timeFixed.booleanValue()) {
           originRow.setValue(Origin.FTIME, "y");
        } else {
           originRow.setValue(Origin.FTIME, "n");
        }
     }

    }

    /** Schema constraint for size of AUTH attribute */
    static final int MaxAuthSize = 15;
    /** Schema constraint for size of SUBSOURCE attribute */
    static final int MaxSubsourceSize = 8;

    /** get ORIGIN.auth
    * Enforce schema length and notnull constraints. Truncate if necessary.
    * AUTH is a 'NOT NULL' attribute so if the internal value is null
    * default to "??". That shouldn't happen because the Solution constructor
    * always sets it to "?". */
    String getAuthority() {
        String str = authority.toString();
        if (authority.isNull()) str = "??";
        int end = Math.min(str.length(), MaxAuthSize) ;
        return str.substring(0, end);
    }
    /** Get ORIGIN.subsource.
    * Enforce schema length and notnull constraints. Truncate if necessary. If it is null
    * will default to EnvironmentInfo.getApplicationName()*/
    String getSource() {
        String str = source.toString();
        if (source.isNull()) str = EnvironmentInfo.getApplicationName();
        int end = Math.min(str.length(), MaxSubsourceSize) ;
        return str.substring(0, end);
    }
    /** Get EVENT.auth.
    * Enforce schema length and notnull constraints. Truncate if necessary.
    * AUTH is a 'NOT NULL' attribute so if the internal value is null
    * default to "??". That shouldn't happen because the Solution constructor
    * always sets it to "?". */
    String getEventAuthority() {
        String str = eventAuthority.toString();
        if (eventAuthority.isNull()) str = "??";
        int end = Math.min(str.length(), MaxAuthSize) ;
        return str.substring(0, end);
    }
    /** Get EVENT.subsource.
    * Enforce schema length and notnull constraints. Truncate if necessary. If it is null
    * will default to EnvironmentInfo.getApplicationName()*/
    String getEventSource() {
        String str = eventSource.toString();
        if (eventSource.isNull()) str = EnvironmentInfo.getApplicationName();
        int end = Math.min(str.length(), MaxSubsourceSize) ;
        return str.substring(0, end);
    }
    // TODO: must translate from model string to model int
    private int getCrustalModelId(String crustModel) {

            return 0;
    }
    // TODO: must translate from model string to model int
    private int getVelocityModelId(String velModel) {
            return 0;
    }
    // TODO: must translate from model string to model int
    private String getCrustalModelString(int crustModelId) {

            return "";
    }
    // TODO: must translate from model string to model int
    private String getVelocityModelString(int velModelId) {
            return "";
    }

    /*
      STATES:
      o  The class has a boolean deleteFlag.
      o  The fromDbase boolean tells us if we will 'insert' or 'update' on commit.
      o  If the individual DataObject.isUpdate() = true,  an update is necessary.

      Commit action matrix:

      fromDbase		=	true		false
      deleteFlag=true           dbaseDelete	noop
      isUpdate()=true		update		insert
      isUpdate()=false		noop		insert

     */

    /** Commit changes or delete to underlying DataSource. To write to the
        DataSource, the flag DataSource.isWriteBackEnabled() must be true. This
        must be set with DataSource.setWriteBackEnabled(true) BEFORE data is
        read. <p>
        Behavior:<p>
        1) If event was not originally read from the dbase, new EVENT, ORIGIN and NETMAG rows are inserted.<p>
        2) If the event was read from the dbase and:<br>
           a) no changes were made, nothing is done to the dbase.<br>
           b) changes were made,  new ORIGIN and NETMAG rows are inserted, and the original EVENT row's <it>prefor</it> and <it> prefmag </it> entries are pointed at the new rows.<br> (Note that a new NETMAG row is written event if it has not changed. if the ORIGIN changes the NETMAG row <b>should</b> change. This rule would need to be enforced by the application.

        If there is a problem with this commit it will do a rollback. This means that any
        transactions done before this one can be lost if DataSource.commit() is not called
        first.

    */
    public boolean commit() throws JasiCommitException {

        boolean status = false;
  commitStatus = "";
  long evid = id.longValue();

  // you can't write to the dbase
  if (!DataSource.isWriteBackEnabled()) {

         commitStatus = "Database is NOT write enabled.";
         if (debug) System.out.println(commitStatus);
         throw new JasiCommitException(commitStatus);
      //return false;
  }

  if (debug) {
      System.out.println ("SolutionTN.commit() : fromDbase= "+ fromDbase);
      System.out.println (" isDeleted() = "+ isDeleted());
      System.out.println (" hasChanged()= "+ hasChanged());
  }

// ////////////////////////////////////////////////////////////////////////
// Event/Origin DELETE
// ////////////////////////////////////////////////////////////////////////
// <delete from dbase>
  try {
  if (isDeleted()) {

         return commitDelete();

// ////////////////////////////////////////////////////////////////////////
// Event/Origin UPDATE/INSERT
// ////////////////////////////////////////////////////////////////////////
  } else if (hasChanged()) {

         // insert comment early so commid gets written with Event row
         dbaseInsertComment();

        // update if Solution was originally from the dbase
      if (fromDbase) {
    if (debug) System.out.println ("Update existing event...");

    status = dbaseUpdate();

         // insert if this is a new Solution
      } else {
    if (debug) System.out.println ("Insert new event...");

    status = dbaseInsert();

      }
      }
    } catch (Exception ex)  {   // SQL or JasiCommitException
    System.err.println(ex);
    ex.printStackTrace();
          rollback();
          commitStatus = "Event/Origin insert/update failed for event "+evid;
          throw new JasiCommitException(commitStatus);
    }

// Do intermediate commit of summary info, rollback and bail if problems
   if (status) {   // something got written :. commit

        try {
        if (debug) System.out.println ("DataSource.commit event >>> ");
        getConnection().commit();
        } catch (SQLException ex)  {
    System.err.println(ex);
    ex.printStackTrace();
          rollback();
          commitStatus += "\nEvent/Origin commit failed.";
          throw new JasiCommitException(commitStatus);
        }
        commitStatus += "\nEvent/Origin commit succeeded." +evid;
   } else {
        commitStatus = "No change to Event/Origin.";
     if (debug) System.out.println (commitStatus);
   }

// ////////////////////////////////////////////////////////////////////////
// Magnitude
// ////////////////////////////////////////////////////////////////////////
// MAGNITUDE:  It may have changed even if Solution didn't.
// Magnitude.commit() will do more checks and do the right thing
     if (magnitude != null) {

        try {

          status = magnitude.commit();

        } catch (Exception ex)  {
    System.err.println(ex);
    ex.printStackTrace();
          rollback();
          status = false;
          commitStatus += "\nWrite of magnitude failed.";
// NON-FATAL
//          throw new JasiCommitException("Magnitude database write failed.");
        }

        if (status) { // Do intermediate commit of mag info

              commitStatus += "\nWrite of magnitude succeeded.";
              try {
                 if (debug) System.out.println ("DataSource.commit mag >>> ");

              getConnection().commit();

                 commitStatus += "\nCommit of magnitude succeeded.";

              } catch (SQLException ex)  {
          System.err.println(ex);
          ex.printStackTrace();
                commitStatus += "\nCommit of magnitude failed.";
// NON-FATAL                return false;
              } // end of try/catch

         }  else {
              commitStatus += "\nNo change to Magnitude.";
              if (debug) System.out.println ("No change to magnitude.");
         }

     } // end of "if (mag..."

// ////////////////////////////////////////////////////////////////////////
// Alternate Magnitudes
// ////////////////////////////////////////////////////////////////////////
//
// Removed this because it is nasty.
//
// 1) Old mags become bogus once a new solution is created.
//
/*
     if (commitAltMagList()) {
         try {
         if (debug) System.out.println ("DataSource.commit alt mags >>> ");
         getConnection().commit();
            commitStatus += "\nCommit of alternate magnitudes succeeded.";
         } catch (SQLException ex)  {
         System.err.println(ex);
         ex.printStackTrace();
            commitStatus += "\nWrite of alternate magnitudes failed.";
         }
     } else {
       rollback();
     }
*/
     // Now that all mags are saved, let the dbase logic decide which mag to use
     // This does a dbase commit!
     //MagnitudeTN.dbaseAutoSetPrefMag(getConnection(), id.longValue());

// ////////////////////////////////////////////////////////////////////////
// WAVEFORMS       <AssocWaE> write waveform associations
// ////////////////////////////////////////////////////////////////////////
     if (commitWaveformList()) {
         try {
         if (debug) System.out.println ("DataSource.commit waveforms >>> ");
         getConnection().commit();
            commitStatus += "\nCommit of wavforms succeeded.";
         } catch (SQLException ex)  {
         System.err.println(ex);
         ex.printStackTrace();
            commitStatus += "\nWrite of waveform list failed.";
         }
     } else {
       rollback();
     }

     setNeedsCommit(false);
  return status;

    }   // end of commit()

    /** Rollback the last set of dbase transactions. Catch exceptions and print them.
    * Returns false on exception. */
    protected boolean rollback () {
        try {
              getConnection().rollback();
        } catch (SQLException ex)  {
    System.err.println(ex);
    ex.printStackTrace();
          return false;
        }
        return true;

    }
/** Do everything necessary to delete event from dbase. */
    protected boolean commitDelete() throws JasiCommitException {

         long evid = getId().longValue();
         boolean status;

      if (debug) System.out.println ("Deleting "+evid );

      // No reason to continue with mag, etc. if its a delete
         // Note: commit is done in dbaseDelete()
         try {
           commitStatus = "Event/Origin delete failed for event "+evid;

           status = dbaseDelete();

           if (status) commitStatus = "Event/Origin delete succeeded for event "+evid;

         } catch (Exception ex)  {   // SQL or JasiCommitException
        System.err.println(ex);
     ex.printStackTrace();
           rollback();
           commitStatus = "Event/Origin delete failed for event "+evid;
           throw new JasiCommitException(commitStatus);
         }

         setNeedsCommit(false);
         return status;
    }
    /** Commit any additions, changes or deletions to the data source plus
    * take any site-specific actions for a final solution. */
    public boolean finalCommit() throws JasiCommitException {

       boolean status = true;

       if (getNeedsCommit()) status = commit();

       // commit succeeded or unnecessary, finalize
       //if (status)  status = dbaseFinalize();

       // finalize if no exception was thrown
       status = dbaseFinalize();

       return status;
    }

/**
 * Delete event from the database. This calls a stored procedure, so the
 * actual meaning of "delete" is defined there. (This is a virtual delete that sets
 * selectflag = 0). The change is commited in this method.
 */
protected boolean dbaseDelete () throws JasiCommitException {

    if (fromDbase) {

  // Call stored procedure that knows rules for deleting an event
  // The procedure does a commit so you don't have to.
  String sql = "call jasi.Delete_Event("+id.longValue()+")";
  return doSql(sql);

    } else {		// not from dbase :. nothing to delete

//	if (debug) System.out.println ("Nothing to delete");

  return true;
    }
}

/**
 * Finalize event in the database. This calls a stored procedure, so the
 * actual meaning of "finalize" is defined there.
 */
protected boolean dbaseFinalize () throws JasiCommitException {

  // Call stored procedure that knows rules for finalizing an event
  String sql = "call jasi.Finalize_Event("+id.longValue()+")";

     System.out.println (sql);

  return doSql(sql);
}

/** Execute the sql string, return false on error */
protected boolean doSql (String sql) throws JasiCommitException {

    /// make sure there's a connection
    if (getConnection() == null) {
      throw new JasiCommitException("No connection to data source.");
      //return false;
    }
    try {
  Statement sm = getConnection().createStatement();

  sm.execute(sql);
    } catch (SQLException ex) {
  System.out.println ("SQL: "+sql);
  System.err.println(ex);
  ex.printStackTrace();

     // rollback on error
        try {
              getConnection().rollback();
        } catch (SQLException ex2)  {          // rollback failed
    System.err.println("Rollback failed:\n"+ex2);
    ex2.printStackTrace();
          return false;
        } // end of try/catch

  return false;
    }

    return true;

}

/**
 * Insert a new row in the dbase
 */
    // To avoid key constraint errors you must:
    // 1) First set Event row's foreign keys = null
    // 2) Then insert the Event row
    // 3) Then insert the Origin, Mec, NetMag, Comment rows
    // 4) Then set key fields & update the Event and Origin table rows

protected boolean dbaseInsert () {

    boolean status = true;

    // This will set an 'evid' if there is none (eventRow.id)
    toEventRow();
    eventRow.setProcessing(DataTableRowStates.INSERT);

    System.out.println (eventRow.toString());

    // 1)  set Event row's foreign keys = null
    eventRow.setValue(Event.PREFOR,  null);
    eventRow.setValue(Event.PREFMAG, null);
    eventRow.setValue(Event.PREFMEC, null);
    eventRow.setValue(Event.COMMID,  null);

    // 2) insert the Event row (w/ null foreign keys)
    if ( eventRow.insertRow(getConnection()) < 1) return false;

    // 3a) insert the Origin row (w/ null foreign keys)

    status = dbaseInsertOrigin();

    // would do prefmec, and commid here...

    status |= setForeignKeys();

    if (status) setUpdate(false);    // now there's a dbase row

    return status;
}
/** Insert the event comment if it is new. Do this BEFORE committing the Event row
* so the 'commid' will get written. */
 public boolean dbaseInsertComment() {

    if (!comment.isUpdate() || comment.isNull()) return false;

    // new comment ID #
    long newId = SeqIds.getNextSeq(Remark.SEQUENCE_NAME);

    // Make a dbase table object
    final long rowNum = 1;
    Remark remarkRow = new Remark(newId, rowNum, comment.toString());
    remarkRow.setProcessing(Remark.INSERT);

    if ( remarkRow.insertRow(getConnection()) < 1) {
      return false;
    } else {
      commid.setValue(newId);
      return true;
    }

 }
    /**
     * Insert the new Origin and everything connected to it (e.g. Netmag,
     * Arrivals, Amps)
     */

    // THIS IS WHERE THE REAL WORK HAPPENS

    protected boolean dbaseInsertOrigin () {

        if (originHasChanged() ) {
          // <ORIGIN> insert the Origin row (w/ null foreign keys)
          prefor.setNull(true);   // this will force an new orid to be assigned
          toOriginRow();                // will assign 'orid' if none
          originRow.setProcessing(DataTableRowStates.INSERT);

          ////// make sure foreign keys are null, else constraint error
/*
       originRow.setValue(Origin.PREFMAG, null);
       originRow.setValue(Origin.PREFMEC, null);
       originRow.setValue(Origin.COMMID, null);
*/
          // debug
          if (debug) System.out.println (originRow.toString());

          // No point in continuing if Origin row didn't go
          if ( originRow.insertRow(getConnection()) < 1) return false;

          // make sure Magnitude has the new Orid
          MagnitudeTN magtn = (MagnitudeTN) magnitude;
          if (magtn != null &&
              !magtn.isNull() &&
              magtn.getOrid() != getOrid()    // don't set if already equal
              ) magtn.setOrid(getOrid());     // because it causes hasChanged() true

          // would do prefmec, and commid here...
          phaseList.commit(this);

          // <AMPLITUDE & AssocAmO>
          // Can't use ampList.commit() because it trys to write
          // AssocAmM rows and NetMag row may not be written yet
          // so this is ONLY done on Magnitude.commit()
          commitOriginAmpList();
        }

        return true;
    }

    /** Set the Magnitude object of this Solution. Overrides
    * Solution.setPreferredMagnitude in order to update the value of prefmag. */
    public void setPreferredMagnitude (Magnitude mg)
    {
     super.setPreferredMagnitude(mg);

     prefmag = mg.magid;

    }

    /**
     * Update the Event and Origin rows.
     * Set foreign key fields in the Event and Origin table rows. This must
     * be done AFTER the Origin and NetMag rows referenced by the keys have been
     * written or you get constraint errors.
     */

    protected boolean setForeignKeys() {

  boolean status = true;

  if (debug) System.out.println ("adjusting keys: ");
        if (debug) System.out.println (eventRow.toString());

     status = dbaseSetPrefs(prefor, prefmag);

  if (debug) System.out.println ("updated origin status = "+ status);

  if (debug) System.out.println (originRow.toString());

  return status;
}

/** Set the prefor and prefmag of this solution. */
    protected boolean dbaseSetPrefs(DataObject prefOrid, DataObject prefMagid) {

  boolean status = true;

  // Set foreign keys to point at preferred origin and mag.
     // prefOr
  eventRow.setValue(Event.PREFOR, prefOrid);
     // prefMag
  if (!prefMagid.isNull()) eventRow.setValue(Event.PREFMAG, prefMagid);
     // comment
     if (!commid.isNull() && commid.isUpdate()) eventRow.setValue(Event.COMMID, commid);
  eventRow.getDataObject(Event.EVID).setUpdate(true);

  eventRow.setProcessing(DataTableRowStates.UPDATE);

  status = (eventRow.updateRow(getConnection()) > 0);

     // NOTE: this updates the Origin even if evid & prefmag didn't really change

     originRow.setValue(Origin.EVID,  id);
     if (!prefMagid.isNull()) originRow.setValue(Origin.PREFMAG, prefMagid);
  originRow.setProcessing(DataTableRowStates.UPDATE);

  status &= (originRow.updateRow(getConnection()) > 0);

  return status;
}


/** Set the prefor of this solution. */
    protected boolean dbaseSetPrefOr(DataObject prefOrid) {

     if (prefOrid.isNull()) return false;

  boolean status = true;
     this.prefor = (DataLong) prefOrid;

  // Set foreign keys to point at preferred origin and mag.
  eventRow.setValue(Event.PREFOR, prefOrid);
  eventRow.getDataObject(Event.EVID).setUpdate(true);
  eventRow.setProcessing(DataTableRowStates.UPDATE);

  status = (eventRow.updateRow(getConnection()) > 0);

     // Origin points back to the event
  originRow.setValue(Origin.EVID,  id);
  originRow.setProcessing(DataTableRowStates.UPDATE);

  status = (originRow.updateRow(getConnection()) > 0);

     if (status) prefor.setUpdate(false);

  return status;
}
/** Set the prefmag of this solution using the dbase internal logic.
* Note that this method causes a datasource commit. */
    protected boolean dbaseAutoSetPrefMag() {
       long rtn = ((MagnitudeTN)magnitude).dbaseAutoSetPrefMag( getConnection(), getId().longValue());
         return rtn > 0;
    }

/** Set the prefmag of this solution to its Magnitude object.
* Does not do a datasource.commit */
    protected boolean dbaseSetPrefMag() {
         return dbaseSetPrefMag(magnitude);
    }

/** Set the prefmag of this solution to the given Magnitude.
* Does not do a datasource.commit */
    protected boolean dbaseSetPrefMag(Magnitude prefMag) {

     DataObject prefMagid = ((MagnitudeTN)prefMag).getMagid();

     if (prefMagid.isNull())  return false;

  boolean status = true;

     //this.prefmag = (DataLong) prefMagid;

  // Assumes eventRow is already setup. Set foreign keys to point at preferred mag.
  eventRow.setValue(Event.PREFMAG, prefMagid);
     // this is a DataTableRow weirdness that must be undone below.
  eventRow.getDataObject(Event.EVID).setUpdate(true);   //!
  eventRow.setProcessing(DataTableRowStates.UPDATE);

  status = (eventRow.updateRow(getConnection()) > 0);

  originRow.setValue(Origin.PREFMAG, prefMagid);
  originRow.setProcessing(DataTableRowStates.UPDATE);

  status = (originRow.updateRow(getConnection()) > 0);

     if (status) {
     eventRow.getDataObject(Event.EVID).setUpdate(false);
        //prefmag.setUpdate(false);
     }
  return status;
}

    /**
     * Write AssocWaE rows for each Waveform in the waveformList that associates
     * them with 'this' solution (Event)
     */
/*    protected int commitWaveformList () {

  if (waveformList == null) return 0;

  Waveform wf[] = new Waveform[waveformList.size()];
  waveformList.toArray(wf);

  int knt = 0;

  for (int i = 0; i<wf.length; i++) {
      if (wf[i].commit(this)) knt++  ;
  }
  return knt;
    }
 */

    /**
    * Commits the amp list elements associtated with a new Origin.
    * Only call this for once per new origin, usually
    * right after it is committed.
    * If Amp is new (not from dbase) an Amp row will be written.
    * An AssocAmO row will be written.
    * AssocAmM rows will NOT be written because we can't tell if they have been
    * previously written or not. That is the job of the Magnitude.commit() method.
    */
    protected int commitOriginAmpList() {
        int successfulCommitCount = 0;
        if (getAmpList() != null && ! getAmpList().isEmpty()) {
            Amplitude amp[] = getAmpList().getArray();

            AmplitudeTN amptn;
            for (int i = 0; i < amp.length; i++) {

                amptn = (AmplitudeTN) amp[i];   // one cast
                if ( ! amptn.fromDbase ) {     // insert in dbase if new amp
                   // insert and assoc
                   if (amptn.dbaseInsert() ) {
                      if (amptn.dbaseAssociateWithOrigin()) successfulCommitCount++;
                   }
                } else {
                  // associate only
                  if (amptn.dbaseAssociateWithOrigin()) successfulCommitCount++;
                }
            }
        }
        return successfulCommitCount;
    }

    /**
     * Copy all the AssocWaE rows for the oldEvid to this evid. This is much
     * faster then actually writing them one-by-one but only works if this is a
     * clone and you want ALL waveforms associated. Returns true on success,
     * false on failure or noop. Uses Oracle store procedure.
     */
    protected boolean commitWaveformList() throws JasiCommitException {

      if (id.longValue() == parentId.longValue()) return false; // not a clone, don't copy self

      // The "0" means DON'T COMMIT
   String sql = "call JASI.CloneAssocWaE("+ parentId.longValue() +", "+
                                               id.longValue()+", 0)";

   return doSql(sql) ;
    }

    /**
     * Write AssocAmO rows for each Amp in the ampList that associates
     * them with 'this' solution (Origin)
     */
    /*
    protected int commitAmpList () {

  if (ampList == null) return 0;

  Amp am[] = new Amp[ampList.size()];
  ampList.toArray(am);

  int knt = 0;

  for (int i = 0; i<am.length; i++)
  {
      if (am[i].commit(this)) knt++  ;
  }
  return knt;
    }
    */

    /**
     * Copy all the AssocAmO rows for the oldOrid to this orid. This is much
     * faster then actually writing them one-by-one but only works if you
     * want ALL waveforms associated.
     */
/*
    protected int commitAmpList (long oldOrid) {

  String sql = " call jasi.CloneAssocAmO("+ oldOrid +
      ", "+getOrid() +", 0)";

  boolean status = doSql(sql);

  if (status) return 1;
  return 0;
    }
*/
    /**
    *  Commits the coda list elements, creating database Coda and AssocCoO rows.
    */
/*   USE COMMIT() IN AmpList classs
    protected int commitAmpList() {
        int successfulCommitCount = 0;
        if (ampList != null) {
            int count = ampList.size();
            if (count > 0) {
                Amplitude amp[] = (Amplitude[]) ampList.toArray(new Amplitude[count]);
                for (int index = 0; index < count; index++) {
                    if (amp[index].commit()) successfulCommitCount++;
                }
            }
        }
        return successfulCommitCount;
    }
*/
    /*
    *  Commits the coda list elements, creating database Coda and AssocCoO rows.
    */
/*    USE COMMIT() IN CodaList classs
    protected int commitCodaList() {
        int successfulCommitCount = 0;
        if (codaList != null) {
            int count = codaList.size();
            if (count > 0) {
                CodaTN coda[] = (CodaTN []) codaList.toArray(new CodaTN[count]);
                for (int index = 0; index < count; index++) {
                    if (coda[index].commit()) successfulCommitCount++;
                }
            }
        }
        return successfulCommitCount;
    }
*/
    /*
    * Commits the alternate magnitude list elements, creating the database association rows.
    */
    protected boolean commitAltMagList() {
        int successfulCommitCount = 0;
        if (altMagList != null) {
            int count = altMagList.size();
            if (count > 0) {
                MagnitudeTN mag[] = (MagnitudeTN []) altMagList.toArray(new MagnitudeTN[count]);
                try {
                  for (int index = 0; index < count; index++) {
                     if (mag[index].commit()) successfulCommitCount++;
                  }
                } catch (JasiCommitException ex) {
                  // noop
                }
            }
        }
        return (successfulCommitCount > 0);
    }

/**
 * Update Event row in the dbase. Returns true if there was any change to the
 * dbase. Note: this write everything from Origin on down anew and only
 * 'updates' Event */
protected boolean dbaseUpdate () {

    // 3a) insert the NEW Origin row if it has changed (w/ null foreign keys)
    boolean status = dbaseInsertOrigin();

    // This will set any changed values in Event, etype, select, etc.
    // It will be written to the dbase by setForeignKeys()
    toEventRow();

    // would do prefmec, and commid here...
    status |= setForeignKeys();

    if (status) this.setUpdate(false);

    return status;

}

/**
 * Return the orid set in prefor (not the orid in the eventrow or originrow).
 * If there is none, one will be assigned from the dbase sequence.
 */
    public long getOrid() {

       if (prefor.isNull()) prefor.setValue(SeqIds.getNextSeq("orseq"));

    return prefor.longValue();
    }

public long getNextID()
{
   //   TN version should look like:
   return(SeqIds.getNextSeq("EVSEQ"));
}

// /////////////////////////////////////////////////////////////////////////////
/**
 * Main for testing: % CatalogView [hours-back]  (default = 1)
 */
    public static void main (String args[]) {

     System.out.println ("Starting");

  //DataSource init = new DataSource (url, driver, user, passwd);    // make connection
  DataSource init = new TestDataSource ();    // make connection
        init.setWriteBackEnabled(true);
  // get one by evid


  //	int evid =  9526860;
  //int evid =  9616101;
//	int evid =  9629593;
//     	int evid =  9559044;
          int evid = 9720601;

  System.out.println ("Get one entry by evid = "+ evid);

     /////////////////////
/*
     if (false) {
    // Test commit with NO CHANGES
    System.out.println ("========== Testing delete ... ");

       // MUST PUT A DELETEABLE EVID HERE TO TEST!
       Solution sol2 = Solution.create().getById(11558792);

       if (sol2 != null) {

          sol2.delete();

          try {
       sol2.commit();                //CodaTN
          } catch (JasiCommitException ex) {}
          System.out.println (sol2.getCommitStatus());
       }
     }
*/
     //Solution sol1 = Solution.create("TN").getById(evid);
     Solution sol1 = Solution.create().getById(evid);
     //Solution sol1 = Solution.create(JasiObject.TRINET).getById(evid);

     //sol1.loadAmpList();         // origin assoc amps
     sol1.loadMagAmpList();      // mag assoc amps
     sol1.loadPhaseList();
     sol1.loadCodaList();

//	System.out.println (sol1.toSummaryString()+" source="+sol1.source.toString());
  System.out.println (sol1.toNeatString());

     System.out.println (sol1.getPhaseList().toNeatString());
     System.out.println (sol1.getAmpList().toNeatString());
     System.out.println (sol1.magnitude.ampList.toNeatString());
     System.out.println (sol1.getCodaList().toNeatString());

/*
     if (true) {
    // Test commit with NO CHANGES
    System.out.println ("========== Testing commit with NO CHANGES... ");
       try {
     sol1.commit();                //CodaTN
       } catch (JasiCommitException ex) {}
       System.out.println (sol1.getCommitStatus());
     }

     if (true) {
  // Test changing an Event attribute
    System.out.println ("========== Change Event value (eventType)");

       sol1.setEventType(EventTypeMap.UNKNOWN);

    System.out.println (sol1.toSummaryString());
       try {
       sol1.commit();
       } catch (JasiCommitException ex) {}
       System.out.println (sol1.getCommitStatus());
     }
*/
/*
     if (true) {
  // Test changing an Origin attribute
    System.out.println ("========== Change Origin value (lat)");

    sol1.lat.setValue(37.89);

    System.out.println (sol1.toSummaryString());
       try {
       sol1.commit();
       } catch (JasiCommitException ex) {}
       System.out.println (sol1.getCommitStatus());
     }
 */
     if (true) {
  // Test setting new Mag
    System.out.println ("========== Recalc. Mag ");

        EnvironmentInfo.setNetworkCode("CI");
     EnvironmentInfo.setApplicationName("Test");
        EnvironmentInfo.setAutomatic(false);

//       newMag.value.setValue(1.11);
//       newMag.subScript.setValue("n");
//	  newMag.authority.setValue("CI");

       System.out.println ("Old Mag = "+ sol1.magnitude.toNeatString());

  System.out.println (" Calculating mag (SoCalML) for: "+evid);

  MagnitudeMethod ml = MagnitudeMethod.CreateMagnitudeMethod("org.trinet.util.magnitudeengines.SoCalML");
  ml.ConfigureMagnitudeMethod();

  MagnitudeEngine magEng = MagnitudeEngine.CreateMagnitudeEngine("org.trinet.util.magnitudeengines.LocalMagnitudeEngine");
  magEng.ConfigureMagnitudeEngine(ml);

  Magnitude newMag = magEng.solve(sol1);

       System.out.println ("New Mag = "+ sol1.magnitude.toNeatString());

/*
       try {
       sol1.commit();
       } catch (JasiCommitException ex) {}
*/
         System.out.println (sol1.getCommitStatus());

         // Test changing old Mag
         System.out.println ("========== sol.commit error ");
       }

    }  // end of Main

} // end of class


