package org.trinet.jasi.TN;

import org.trinet.jasi.*;

import java.util.*;
import java.sql.*;

import org.trinet.jdbc.*;
import org.trinet.jdbc.datatypes.*;
import org.trinet.jdbc.table.*;
import org.trinet.util.*;

/**
 * PhaseTN.java
 *
 * Concrete schema-specific implementation
 *
 * Created: Fri Oct 29 12:22:36 1999
 *
 * @author Doug Given
 * @version
 */

/*

  There are two types of phases (distingushed by 'fromDbase' = true/false):
  1) those that were read from the dbase
    - these would never be deleted, updated, or inserted
    - you would never delete or update an AssocArO
    - you may create a new AssocArO associating them with an new origin

  2) those created by the application
    - there's nothing in the dbase to update or delete
          - they must be inserted in the dbase to save them
    - you MUST create a new AssocArO for them if they are to be meaningful
 */
public class PhaseTN extends Phase {


    // Keep references to the original row objects for when we write back
    //    Arrival  arrivalRow = new Arrival();
    //    AssocArO assocRow   = new AssocArO();
    Arrival  arrivalRow;
    AssocArO assocRow;

    /** the orid of association */
    protected DataLong orid   	= new DataLong();
    /** the arid */
    protected DataLong arid   	= new DataLong();

    /** True if this Solution was orginally read from the dbase. Used to know if
  a commit requires an 'insert' or an 'update' */
    boolean fromDbase = false;

    // The standard start to most SQL queries
    static String sqlPrefix = "Select "+
  Arrival.QUALIFIED_COLUMN_NAMES+","+
  AssocArO.QUALIFIED_COLUMN_NAMES+
  " from Arrival, AssocArO where (Arrival.arid = AssocArO.arid) ";

    boolean debug = false;
    //boolean debug = true;

/**
 * Constructor
 */
public PhaseTN ()
{
}

    /** Copy the phase in the arg's info into this phase. */

    // NOTE: use assignment NOT setValue() because setValue() sets isNull flag false
    // even if the new value is in fact null.
public boolean replace(Phase ph) {

    // use .setValue() to keep DataObject flags right
    setChannelObj( (Channel) ph.getChannelObj().clone() );

    // DataObjects
    this.datetime.setValue(ph.datetime.doubleValue()) ;

    /** Descripton of the phase. @See org.trinet.jasi.PhaseDescription */
    this.description = ph.description;

    this.processingState = ph.processingState;	// rflag

    /** Solution which which this phase is associated. Is null if unassociated */
    this.sol = ph.sol ;

    // Things relating to the associated Solution

    // ***** WARNING *****
    // The setValue() method will set the isNull() flag false
    // even if the value set is in fact null!

    // use .setValue() to keep DataObject flags right
    /*
    this.distance.setValue(ph.distance.doubleValue()) ;
    this.azimuth.setValue(ph.azimuth.doubleValue()) ;
    this.incidence.setValue(ph.incidence.doubleValue()) ;
    this.weightIn.setValue(ph.weightIn.doubleValue()) ;
    this.weightOut.setValue(ph.weightOut.doubleValue()) ;
    this.residual.setValue(ph.residual.doubleValue()) ;
    */

    setDistance(ph.getDistance());
    setHorizontalDistance(ph.getHorizontalDistance());
    this.getChannelObj().azimuth   = ph.getChannelObj().azimuth ;
    this.incidence = ph.incidence;
    this.weightIn  = ph.weightIn ;
    this.weightOut = ph.weightOut ;
    this.residual  = ph.residual ;

    return true;
}

    /** Commit changes or delete to underlying DataSource. The action taken will
        depend on the state of the phase. It may or may not be originally
        from the data source, it may be deleted, unassociated, modified, or
        unchanged.  To write to the DataSource, the flag
        DataSource.isWriteBackEnabled() must be true. This must be set with
        DataSource.setWriteBackEnabled(true) BEFORE data is read.*/

public boolean commit() {

    // you can't write to the dbase
    if (!DataSource.isWriteBackEnabled()) return false;

    // set the 'rflag' value to "H" for human checked
    //!! This might not be right here! What if this class is used by an auto program?
    //    processingState.setValue("H");

    // debug
    if (debug) System.out.println ("PhaseTN.commit() : fromDbase= "+ fromDbase+
           "  isDeleted()= "+isDeleted());

    // <DELETE>
    // Phase is deleted or not associated! So don't write or associate it.
    if (isDeleted() || !isAssociated()) return true;

    // Check that 'orid' of associated event has been set
    long orid = ((SolutionTN)sol).getOrid();
    if ( orid == 0 ) {
  if (debug) System.out.println ("Associated Solution has no orid.");
  return false;
    }

    // "existing" phase has changed insert a new table row (with new number)
    // and a new association
    if (fromDbase) {

  if (hasChanged()) {

      if (debug) System.out.println ("(changed phase) INSERT and assoc");
      return dbaseInsert();		// insert & associate
  } else {
      if (debug) System.out.println ("(phase unchanged) assoc only");
      // if no changes to arrival row, don't change just make new assoc
      if (dbaseAssociate()) {
            setUpdate(false);    // now there's a dbase row
            return true;
         } else {
            return false;
         }
  }

    // <INSERT>
    } else {	// not from dbase

  if (debug) System.out.println ("(new phase) INSERT and assoc");

  return dbaseInsert();	// insert & associate
    }

}

/**
 * NEVER delete an arrival. Never remove association with another origin.  If
you don't like it just don't assoc it with your new origin.  */
protected boolean dbaseDelete () {

    if (fromDbase) {

  return true;

    } else {		// not from dbase :. nothing to delete

  return true;
    }
}


/**
 * Make AssocArO row for this Arrival
 */
protected boolean dbaseAssociate () {

    if ( !isAssociated() ) return false;

    AssocArO assocRow = toAssocArORow();
    assocRow.setProcessing(DataTableRowStates.INSERT);
    return (assocRow.insertRow(DataSource.getConnection()) > 0);

}

/**
 * Insert a new row in the dbase
 */
protected boolean dbaseInsert () {

    boolean status = true;

    if (debug) System.out.println ("dbaseInsert: "+this.toString());

    Arrival arrivalRow = toArrivalRow();
    arrivalRow.setProcessing(DataTableRowStates.INSERT);

    // write arrival row
    status = (arrivalRow.insertRow(DataSource.getConnection()) > 0);

    if (status) {		// success
     setUpdate(false);             // set flag to indicate there's a dbase row
  status = dbaseAssociate();	// write association row
    }

    return status;
}

/**
 * Return the 'arid' of the row (if it came from the database). If it did not
 * come from the data base 0 is returned.
 */
protected long getArid () {

    if (arid.isNull()) return 0;
    return arid.longValue();
}

/**
 * Return true if the phasedescription or time has changed from what was
 * read in from the dbase. Returns true if phase was not originally from the
 * dbase.  */
protected boolean hasChanged () {

    if (!fromDbase ||
  description.hasChanged() ||
  //	     chanName.isUpdate() ||
  datetime.isUpdate()
  ) return true;

    return false;
}
/**
 * Set the isUpdate() flag for all data dbase members the given boolean value.  */
protected void setUpdate (boolean tf) {

          fromDbase = !tf;
          description.remember(description);
          datetime.setUpdate(tf);
}
/**
 * Extract from DataSource all phases associated with this Solution.
 */

// Don't call getBySolutionId() because it has the expensive step of looking up
// the prefor which is already known to the Solution.
public Collection getBySolution (Solution tsol) {

    String sql = sqlPrefix + " and (AssocArO.orid = "+((SolutionTN)tsol).prefor.toString()+")";

    ArrayList ph = (ArrayList) PhaseTN.getBySQL(sql);

    // set associated solution to the one in the arg
    for (int i = 0; i<ph.size(); i++) {
  ((Phase)ph.get(i)).associate(tsol);
    }

    return ph;
}
/**
 * Extract from DataSource all phases associated with this Solution ID.
 */
public Collection getBySolution (long id) {

    // must first  retreive the Solution, then look up prefor for this id
    Solution tsol = Solution.create().getById(id);

    if (tsol == null) return null;	// no such solution in DataSource

    return getBySolution(tsol);
}

/**
 * Extract from DataSource all phases for this time window. The SolutionList
 * is used to match associated phases with solutions in the list. If a
 * Phase is associated with a Solution not in the list that Solution will be
 * added to the list.<p>
 * If you pass in an empty SolutionList, it will be populated with the Solutions
 * that match the Phases found in the time window.
 */
/*
  There is a problem here. Know the ORID they're associated with but don't
  know have a Solution object.
  Don't want to look them up but rather draw from an extant Sol list.
*/
public static Collection getByTime (double start, double end, SolutionList sl) {

    ArrayList phList = (ArrayList) Phase.create().getByTime(start, end);
    PhaseTN ph;

    if (phList.size() != 0) {

    // Match up with associated solutions. Get from dbase is necessary.

  Solution tsol;
  for (int i = 0; i < phList.size(); i++) {

      ph = (PhaseTN) phList.get(i);		//pull phase from collection

      // is it in the solution list?
      tsol = sl.getByOrid(ph.orid.longValue());	// lookup sol by ORID of phase

      if (tsol != null) {		// a match
    ph.associate(tsol);
      } else {			// no match, look in dbase for Solution

    tsol = SolutionTN.getByOrid(ph.orid.longValue());

    if (tsol != null){
        ph.associate(tsol);
        sl.addByTime(tsol);	// add to the Solution list
    }
      }

  }
    }

    return phList;
}


public Collection getByTime (double start, double end) {

    String sql = sqlPrefix + " and Arrival.datetime between "+
     start+" and "+ end;

    ArrayList phList = (ArrayList) PhaseTN.getBySQL(sql);
    Phase ph;

    return phList;
}

/**
 * Extract from DataSource all phases associated with this Solution ID.
 */
public static Collection getByTime (TimeSpan ts, SolutionList sl) {

    return getByTime (ts.getStart(),  ts.getEnd(), sl);

}

/**
 * Return a phase for the given Arid in the NCDC schema. Does not get assoc info.
 */
protected static Phase getByArid (long arid) {

    String sql = sqlPrefix + " and AssocArO.arid = "+ arid;

    ArrayList phList = (ArrayList) PhaseTN.getBySQL(sql);
    Phase ph;

    // only returns one Phase
    if (phList.size() > 0) return (Phase) phList.get(0);
    return null;
}

    /**
     * Returns array of Phases based on the results of the SQL query to an
     * NCDCv1.5 data base. It must be of the form:<p> * Select Arrival.*,
     * AssocArO.* from Arrival, AssocArO * where (Arrival.arid = AssocArO.arid)
     * <p> because you must do a join of Arrival and AssocArO to get all the
     * info we need.  Returns null if no data is found.  Uses default connection
     * created by DataSource.  */
    protected static Collection getBySQL(String sql)
    {
  if (DataSource.getConnection() == null)
      {
    System.err.println ("* No DataSource is open.");
    return null;
      }
  return getBySQL(DataSource.getConnection(), sql);
    }

    /**
     * Returns array of Phases based on the results of the SQL query to an
     *	      NCDCv1.5 * data base. It must be of the form:<p> * Select
     *	      Arrival.*, AssocArO.* from Arrival, AssocArO where (Arrival.arid =
     *	      AssocArO.arid) <p> * because you must do a join of Arrival and
     *	      AssocArO to get all the info we need.  * Returns null if no data
     *	      is found.  */
    protected static Collection getBySQL(Connection conn, String sql)
    {
  //	Vector vec = new Vector();

  ArrayList phList = new ArrayList();

  // Debug
  //	System.out.println ("SQL: "+sql);

  try {
      if ( conn.isClosed() )	// check that valid connection exists
    {
        System.err.println ("* DataSource connection is closed");
        return null;
    }

      Statement sm = conn.createStatement();

      // NOTE: if ExecuteSQL.setSelectForUpdate(true) was set in DataSource
      // this will lock the selected rows until commit() or close() are called
      ResultSetDb rsdb = new ResultSetDb(ExecuteSQL.rowQuery(sm, sql));

      if (rsdb == null) return null;	// nothing found

      while ( rsdb.getResultSet().next() ) {

    phList.add(PhaseTN.parseResultSet(rsdb));
      }

      sm.close();

  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();
  }

  return (Collection) phList;
    }

    /**
     * Parse a resultset row that contains a concatinated Event/Origin/NetMag
     */
    protected static Phase parseResultSet (ResultSetDb rsdb) {

  Arrival arr    = new Arrival();	// jdbc object
  AssocArO assoc = new AssocArO();

  int offset = 0;

  // get jdbc objects from the result set
  arr = (Arrival) arr.parseOneRow(rsdb, offset);

  offset += Arrival.MAX_FIELDS;

  assoc = (AssocArO) assoc.parseOneRow(rsdb, offset);

  // allow changes to this DataTableRow. Need to set this because we are
  // not using our own rather than org.trinet.jdbc parsing methods
  if (DataSource.isWriteBackEnabled()) {
      arr.setMutable(true);
      assoc.setMutable(true);
  }

  // build up the Phase from the jdbc objects
  PhaseTN ph = new PhaseTN();
  ph.parseArrival(arr);
  ph.parseAssocArO(assoc);

  // remember that we got this from the dbase
  ph.fromDbase = true;

  return (Phase) ph;
    }

    /**
     * Copy DataObjects out of a jdbc.Arrival object into a Phase.
     * This method can be used to build up a complete
     * phase by parsing rows from different tables (like AssocArO).
     */

    protected void parseArrival(Arrival arr) {

    // defaults
  String phaseType  = " ";
  String ie	  = " ";
  double decimalWt  = 4;
  String fm  = " ";

     // use setChannel to synch with master channel list
  setChannelObj(ChannelTN.parseNameFromDataTableRow(arr));

  this.arid	  = (DataLong) arr.getDataObject(Arrival.ARID);

  this.datetime	  = (DataDouble) arr.getDataObject(Arrival.DATETIME);

  this.processingState = (DataString) arr.getDataObject(Arrival.RFLAG);
  this.incidence    = (DataDouble) arr.getDataObject(Arrival.EMA);
     this.authority    = (DataString) arr.getDataObject(Arrival.AUTH);
     this.source       = (DataString) arr.getDataObject(Arrival.SUBSOURCE);

  // Create phaseDescription based on stuff in dbase
  DataString dataStr  = (DataString) arr.getDataObject(Arrival.IPHASE);
  if (!dataStr.isNull()) phaseType  = dataStr.toString();
  dataStr  = (DataString) arr.getDataObject(Arrival.QUAL);
  if (!dataStr.isNull()) ie  = dataStr.toString();
  dataStr  = (DataString) arr.getDataObject(Arrival.FM);
  if (!dataStr.isNull()) fm  = dataStr.toString();

     DataDouble quality = (DataDouble) arr.getDataObject(Arrival.QUALITY);
  if (!quality.isNull()) decimalWt  = quality.doubleValue();

  this.description = new PhaseDescription (phaseType, ie, fm, decimalWt);

  // Save the arrivalRow for future use (commit)
  arrivalRow = arr;

  return;
    }

/**
 * Stuff contents from this Phase object into a new Arrival (TableRow) object. Gets a
 * new 'arid' from "arseq" in the dbase.
 * @See: org.trinet.jdbc.Arrival().
 */
protected Arrival toArrivalRow () {

    // Always make a new row, get unique key #
    //    arid.setValue(SeqIds.getNextSeq("arseq"));
    //    Arrival arrivalRow = new Arrival(arid.longValue());
//    if (arid.isNull()) {
      long newId = SeqIds.getNextSeq("arseq");
      arid.setValue(newId);			// set ID for this.Phase
//    }
    Arrival arrivalRow = new Arrival(arid.longValue());	// set ID for arrival row

    if (debug) System.out.println ("toArrivalRow new id: " + arrivalRow.toString());

    // Stuff contents of channel name.
    stuffChannel(arrivalRow, getChannelObj());

    // set flag to enable processing
    arrivalRow.setUpdate(true);

    arrivalRow.setValue(Arrival.DATETIME, datetime);

    // the following can be null

    if (!authority.isNull())   arrivalRow.setValue(Arrival.AUTH, authority);

    if (!processingState.isNull())
                         arrivalRow.setValue(Arrival.RFLAG, processingState);

    if (!source.isNull())     arrivalRow.setValue(Arrival.SUBSOURCE, source);

    // phase description.
    //    if (description.hasChanged()) {
  arrivalRow.setValue(Arrival.IPHASE, description.iphase);
  arrivalRow.setValue(Arrival.QUAL, description.ei);
  arrivalRow.setValue(Arrival.QUALITY, description.getQuality());
  arrivalRow.setValue(Arrival.FM, description.fm);
  //    }

    // all the other fields are not used by us
    if (debug) {
  System.out.println ("phase     : "+this.toString());
  System.out.println ("arrivalRow: "+arrivalRow.toString());
    }

    return arrivalRow;
}
/** Put the channel name attributes in the Arrival DataTableRow */
protected void stuffChannel (Arrival arrivalRow, Channel cn) {

    // this one can't be null
    arrivalRow.setValue(Arrival.STA, cn.getSta());

    // by default all fields are "" rather then null strings: see ChannelName
    if (cn.getNet().length() > 0)        arrivalRow.setValue(Arrival.NET, cn.getNet());
    if (cn.getAuth().length() > 0)       arrivalRow.setValue(Arrival.AUTH, cn.getAuth());
    if (cn.getSubsource().length() > 0)  arrivalRow.setValue(Arrival.SUBSOURCE, cn.getSubsource());
    if (cn.getChannel().length() > 0)    arrivalRow.setValue(Arrival.CHANNEL, cn.getChannel());
    if (cn.getChannelsrc().length() > 0) arrivalRow.setValue(Arrival.CHANNELSRC, cn.getChannelsrc());
    if (cn.getSeedchan().length() > 0)   arrivalRow.setValue(Arrival.SEEDCHAN, cn.getSeedchan());
    if (cn.getLocation().length() > 0)   arrivalRow.setValue(Arrival.LOCATION, cn.getLocation());
}


/**
 * Copy DataObjects out of a jdbc.AssocArO object into a Phase.
 * This method can be used to build up a complete
 * phase by parsing rows from different tables (like Arrival).
 */

protected void parseAssocArO(AssocArO assoc)
{
    this.orid      = (DataLong)   assoc.getDataObject(AssocArO.ORID);
    chan.dist      = (DataDouble) assoc.getDataObject(AssocArO.DELTA);
//    setDistance( ((DataDouble) assoc.getDataObject(AssocArO.DELTA)).doubleValue());

    this.getChannelObj().azimuth   = (DataDouble) assoc.getDataObject(AssocArO.SEAZ);
    //	this.incidence = (DataDouble) assoc.getDataObject(AssocArO.EMA);
    this.weightIn  = (DataDouble) assoc.getDataObject(AssocArO.IN_WGT);
    this.weightOut = (DataDouble) assoc.getDataObject(AssocArO.WGT);
    this.residual  = (DataDouble) assoc.getDataObject(AssocArO.TIMERES);

    // Save the assocRow for future writeBack
    assocRow = assoc;

    return;
}

/**
 * Stuff contents of this Phase/association into an AssocArO (TableRow) object.
 */
protected AssocArO toAssocArORow () {

    // always create a new one
    assocRow = new AssocArO();

    // if no association...
    if (!isAssociated() ) return null;
    long assOrid = ((SolutionTN)sol).getOrid();
    if (assOrid == 0) return null;

    if (arid.isNull()) return null;	// no arid!

    // set flag to enable processing
    assocRow.setUpdate(true);

    // (KEY #1) assumes ARID has been set for this phase
    assocRow.setValue(AssocArO.ARID, arid);

    // (KEY #2) set ORID to orid of associated Solution
    //    if (debug) System.out.println ("assoc Sol: "+sol.toString()+ "  orid="+assOrid);
    //mutable?
    assocRow.setValue(AssocArO.ORID, assOrid);

    // the following can be null

    if (!authority.isNull())  assocRow.setValue(AssocArO.AUTH, authority);
    if (!source.isNull())     assocRow.setValue(AssocArO.SUBSOURCE, source);

    // set flag to allow changes
    //    assocRow.setMutable(true);

    //    if (orid.isNull())       assocRow.setValue(AssocArO.ORID,    orid);
    if (!chan.dist.isNull())  assocRow.setValue(AssocArO.DELTA,   getDistance());
    if (!getChannelObj().azimuth.isNull())    assocRow.setValue(AssocArO.SEAZ,    getChannelObj().azimuth);
    if (!weightIn.isNull())   assocRow.setValue(AssocArO.IN_WGT,  weightIn);
    if (!weightOut.isNull())  assocRow.setValue(AssocArO.WGT,     weightOut);
    if (!residual.isNull())   assocRow.setValue(AssocArO.TIMERES, residual);

    // phase description. Note: this is redundant with Arrival.IPHASE
    assocRow.setValue(AssocArO.IPHASE,  description.iphase);

    if (!processingState.isNull())
                        assocRow.setValue(AssocArO.RFLAG, processingState);

    if (debug) {
  System.out.println ("phase   : "+this.toString());
  System.out.println ("assocRow: "+assocRow.toString());
    }

    return assocRow;
}

public static void doWriteBackTest (Solution sol, Phase phx) {

  // test phase descp

  System.out.println ("ExecuteSQL.isSelectForUpdate = "+
           ExecuteSQL.isSelectForUpdate());

  boolean status = true;
  long evid = 9691944;
  Solution solx = Solution.create().getById(evid);

  // NOTE: you would NEVER unassociate phases from an origin in the dbase;
  // that would alter the origin.
  // You would only NOT associate them with the new origin.

  System.out.println ("\n----- New associate test ----- old id = "+
          sol.id.toString()+"   new evid= "+evid);


  phx.associate(solx);
  status = phx.commit();

  if (status) {
      long arid = ((PhaseTN) phx).getArid();
      System.out.println ("<><><><> commit (insert) status = "+status+
        "(arid = "+arid+" )");

      // readback check
      System.out.println ("Readback from dbase...");
      Phase cph = (Phase) PhaseTN.getByArid(arid);

      System.out.println (cph.toString());


  } else {
      System.out.println ("<><><><> commit FAILED (insert) status = "+status);
  }


  System.out.println ("\n----- Insert new phase test -----");

  // Make an new phase
  Phase newPh = Phase.create();

  java.util.Calendar cal = java.util.Calendar.getInstance();
  java.util.Date date = cal.getTime();	// current epoch millisec
  long now = date.getTime()/1000;	// current epoch sec (millisecs -> seconds)
  newPh.datetime.setValue(now);
  newPh.authority.setValue(EnvironmentInfo.getNetworkCode());
  //	newPh.source.setValue("RT1");
  newPh.source.setValue("Test");

  // need to associate with a solution
  newPh.associate(sol);

  newPh.description = new PhaseDescription("P", "i", "c.", 0);
  newPh.setChannelObj( Channel.create().setChannelName("CI", "XYZ", "BHZ"));
  System.out.println (newPh.toString());

  status = newPh.commit();

  if (status) {
      long arid = ((PhaseTN) newPh).getArid();
      System.out.println ("<><><><> commit (insert) status = "+status+
        "(arid = "+arid+" )");

      // readback check
      System.out.println ("Readback from dbase...");
      Phase cph = (Phase) PhaseTN.getByArid(arid);

      System.out.println (cph.toString());


  } else {
      System.out.println ("<><><><> commit (insert) status = "+status);
  }



  System.out.println ("\n----- No Change test -----");
  // committ again with no changes made, nothing should happen
  status = newPh.commit();
  if (status) {
      long arid = ((PhaseTN) newPh).getArid();
      System.out.println ("<><><><> commit (insert) status = "+status+
        "(arid = "+arid+" )");

      // readback check
      System.out.println ("Readback from dbase...");
      Phase cph = (Phase) PhaseTN.getByArid(arid);

      System.out.println (cph.toString());


  } else {
      System.out.println ("<><><><> commit (insert) status = "+status);
  }

  //	DataSource.commit();

}


// ///////////////////////////////////////////////////////

    public static void main (String args[])
    {

        System.err.println ("Making DataSource connection...");
        new TestDataSource("k2");	// make connection

  System.err.println ("Setting ExecuteSQL.setSelectForUpdate = true");

  System.out.println ("DataSource.isReadonly = "+
          DataSource.isReadOnly());

  System.out.println ("DataSource.isWriteBackEnabled = "+
          DataSource.isWriteBackEnabled());

  System.out.println ("ExecuteSQL.isSelectForUpdate()= "+
          ExecuteSQL.isSelectForUpdate());

  //	long evid = 9526705;
  // long evid = 9691940;
  long evid = 9629677;

  System.out.println (" Getting phases for evid = "+evid);

  Solution sol = Solution.create().getById(evid);

     PhaseList ph = new PhaseList (Phase.create().getBySolution(sol));

      System.out.println ("Phase count = " + ph.size());
     System.out.println (ph.toNeatString());

  //
    //	doWriteBackTest(sol, (Phase) ph.get(0));

  DataSource.close();

    }


} // PhaseTN
