package org.trinet.jasi;

import java.sql.Date;
import java.sql.Connection;
import java.util.*;
import java.io.*;

import org.trinet.util.*;
import org.trinet.util.gazetteer.LatLonZ;
import org.trinet.util.gazetteer.GeoidalUnits;

import org.trinet.util.DateTime;

//public class ChannelList extends ArrayList {
public class ChannelList extends ChannelableList {

    static boolean debug = false;
    static String cacheFilename = null;
    static String asciiFilename = null;
    static String binaryFilename = null;
    static final String FileSep  =
     System.getProperty("file.separator");  // "/" on UNIX, "\" on PC's
/**
   A list of channel objects that can be sorted, etc.<p>
   This class also supports local caching of a channel list which is usually
   faster then reading from a database, especially over slow links. The cache
   contains serialized Channel objects. The smartLoad() method
   attempts to read from a local cache. If it fails it reads from the DataSource
   then writes a cache file for later use.
*/

  public ChannelList() {
      this(getCacheFilename());
  }

  public ChannelList(int initCapacity) {
      super(initCapacity);
  }

  public ChannelList(String cacheFilename) {
      this.
      setCacheFilename(cacheFilename);
  }
/**
 * Add an object. Will not add object if its already in the list.
 * Returns 'true' if object was added.
 */
    public boolean add(Object obj) {

  if ( contains(obj) ) return false;

     return super.add(obj) ;
    }

    public boolean addWithoutCheck(Object obj) {

     return super.add(obj) ;
    }

    /** Create a HashMap of the channel list to speed lookups. */
    protected static final int  DEFAULT_CAPACITY = 5759;
    protected static final float  DEFAULT_LOAD_FACTOR = 0.9f;
    protected HashMap lookupMap;

    public void createLookupMap() {
        lookupMap = new HashMap(DEFAULT_CAPACITY, DEFAULT_LOAD_FACTOR);
        int size = this.size();
        for (int idx = 0; idx < size; idx++) {
            Channel channel = (Channel) get(idx);
            lookupMap.put(channel.getChannelName(), channel);
        }
    }

    /** Find a "similar" channel using a HashMap of the channel list to speed lookups.
     * "Similar" means match the net, sta, and seedchan strings. No other fields are compared.
     * @see: Channel.equals() */
    public Channel findSimilarInMap(Channel chan) {
        if (chan == null) return null;
        Channel channel = null;
        if (lookupMap == null) createLookupMap();
        if (! lookupMap.isEmpty()) {
            channel = (Channel) lookupMap.get(chan.getChannelName());
        }
        if (debug) System.out.println("   findSimilar listChan: " +channel+ " inputchan: " + chan + " hash: " + chan.hashCode());
        return channel;
    }

/**
 * Calculate the distance from location  */
public void calcDistancesFrom (LatLonZ loc)
{
    Channel chan [] = getArray();

    for ( int i=0; i < chan.length; i++) {
  chan[i].calcDistance(loc);
    }

}
/** Note: can't override to change returned type */
public Channel[] getArray() {
   return (Channel[]) toArray(new Channel[size()]);
}

/** Sort/resort the Channel list by distance from the given LatLonZ.  <p>

 1) An in-place sort<br>
 2) Order of components for a given station is not explicitely handled.<br>
*/

    public void distanceSort(Solution sol)
    {
        distanceSort(sol.getLatLonZ());
    }
/**
 * Sort/resort the Channel list by distance from the given Solution.  <p>

 1) An in-place sort<br>
 2) Order of components for a given station is not explicitely handled.<br>
*/

    public void distanceSort(LatLonZ latlonz)
    {
  if (debug) System.out.println ("Distance sorting channel list....");
  // calc. distances from solution to stations
  calcDistancesFrom(latlonz);

  // very weird syntax! to avoid compile error
  Collections.sort(this, new ChannelList(). new DistanceSorter());

    }

/*  FILE READ/WRITE NOTES:

    There are three types of files that you can read/write channel list to/from.
    1) ASCII
    2) Binary
    3) Objects

    These were done to compare speeds and space used.
    Results: there is no dramatic speed difference but objects are a bit faster
    to read. There was a scatter of up to 20% in different runs.

    Test read/write of 2876 channels:

    Source          Read    Write    File
                    (sec)   (sec)    Size
    dbase           34.0     ---     ---     (over 10Mbit link)
    ASCII           20.5     4.5     224k
    Binary          20.7     4.7     228k
    Objects         14.8     5.9     650k
*/

/**
* Write serialized version of the channel list to a file.
* The list will be written to a file called "channelList.cache" in the user's
* home directory.
*/
    public boolean writeToCache() {
      return writeToCache(getCacheFilename());
    }

/**
* Write serialized version of the channel list to a file.
* The filename should include the path.
*/
   public boolean writeToCache (String filename) {
      boolean status = true;
      try {
        FileOutputStream  ostream = new FileOutputStream(filename);
        ObjectOutputStream stream = new ObjectOutputStream(ostream);

        stream.writeObject( (Object) this );

        stream.flush();
        stream.close();
      } catch (IOException ex) {
            ex.printStackTrace();
            status = false ;
      } finally {
        return status;
      }
    }
/**
* Write serialized version of the channel list to a file in background thread.
* The list will be written to a file called "channelList.cache" in the user's
* home directory.
*/
    public boolean writeToCacheInBackground() {
      new CacheWriterThread(this);
      return true;
    }
/** Reads channel list from the data source and writes it to the cache file
 *  from a background thread. This keeps the cache current. */
    public void refreshCache () {
      new CacheRefreshThread();
    }
//////// INNER CLASS
/**
 * This is a one-shot thread that writes cache file in background.
 */
class CacheRefreshThread implements Runnable {
    public CacheRefreshThread () {     // constructor - also starts thread
  Thread thread = new Thread(this);
  thread.start();
    }
    public void run () {
     BenchMark bm = new BenchMark();
           ChannelList chanList = ChannelList.readCurrentList();
           chanList.writeToCache();
     bm.print("CacheRefreshThread finished");
    }
} // end of CacheRefreshThread class

/**
 * This is a one-shot thread that writes cache file in background.
 */
class CacheWriterThread implements Runnable {

    ChannelList list;
//    public Thread thread;

    // constructor - also starts thread
    public CacheWriterThread (ChannelList list ) {
  this.list = list;
  Thread thread = new Thread(this);
  thread.start();
    }

    public void run () {
       list.writeToCache();
    }
} // end of CacheWriterThread class



/**
* Write ASCII version of the channel list to default file.
*/
    public boolean writeAsciiFile() {
      return writeAsciiFile(getAsciiFilename());
    }
/**
* Write ASCII version of the channel list to a file.
* The filename should include the path.
*/
   public boolean writeAsciiFile (String filename) {
      boolean status = true;

      try {

        PrintWriter writer = new PrintWriter(
          new FileOutputStream(filename)
        );

        Channel chan[] = getArray();

          for (int i = 0; i<chan.length; i++) {
            writer.println(chan[i].toCompleteString(" "));
          }
        writer.flush();
        writer.close();
      } catch (IOException ex) {
            ex.printStackTrace();
            status = false ;
      } finally {
        return status;
      }
    }

    public boolean writeBinaryFile () {
      return writeBinaryFile(getFullDefaultBinaryFilename());
    }

    public boolean writeBinaryFile (String filename) {

      boolean status = true;
      try {
       DataOutputStream out = new DataOutputStream(
          new FileOutputStream(filename)
       );

        Channel chan[] = getArray();

          for (int i = 0; i<chan.length; i++) {
            chan[i].binaryWrite(out);
          }
        out.flush();
        out.close();
      } catch (IOException ex) {
            ex.printStackTrace();
            status = false ;
      } finally {
        return status;
      }
    }

    public boolean readBinaryFile () {
      return readBinaryFile(getFullDefaultBinaryFilename());
    }

/**
* Read Binary version of the channel list from a file.
* The filename should include the path.
*/
   public boolean readBinaryFile (String filename) {
      boolean status = true;

      try {

       DataInputStream in = new DataInputStream(
          new FileInputStream(filename)
       );

        while ( in.available() > 0) {
          // add new Channel objects to list
          add(Channel.binaryRead(in));
        }
        in.close();
      } catch (IOException ex) {
            ex.printStackTrace();
            status = false ;
      } finally {
        return status;
      }
    }


/**
* Read ASCII version of the channel list from a file.
* The filename should include the path.
*/
   public boolean readAsciiFile () {
     return readAsciiFile(getAsciiFilename());
   }
/**
* Read ASCII version of the channel list from a file.
* The filename should include the path.
*/
   public boolean readAsciiFile (String filename) {
      boolean status = true;

      try {

        BufferedReader reader = new BufferedReader(
          new FileReader(filename)
        );

        String line;
        while ( (line = reader.readLine()) != null) {
          // add new Channel objects to list
          add(Channel.parseCompleteString(line, " "));
        }
        reader.close();
      } catch (IOException ex) {
            ex.printStackTrace();
            status = false ;
      } finally {
        return status;
      }
    }

/**
* Read serialized version of the channel list from a file.
* The list will be written to a file called "channelList.cache" in the user's
* home directory.  Returns empty ChannelList if it fails.<p>
* The Java loader checks "version" number of the class so if the ChannelList class
* has been recompiled since the cache was written it will throw an exception and
* not load it.
*/
    public static ChannelList readFromCache () {
      return readFromCache(getCacheFilename());
    }

/**
* Read serialized version of the channel list from a file.
* The filename should include the path.
* Returns empty ChannelList if it fails. <p>
*
* The Java loader checks "version" number of the class so if the ChannelList class
* has been recompiled since the cache was written it will throw an exception and
* not load it.
*/
    public static ChannelList readFromCache (String filename) {

      Object obj;

       setCacheFilename(filename);

       try {
        FileInputStream instream = new FileInputStream(filename);
        ObjectInputStream stream = new ObjectInputStream(instream);

        obj = stream.readObject() ;

        stream.close();

      } catch (FileNotFoundException ex) {
            return new ChannelList();

      // This happens if ChannelList has been recompiled since last write
      } catch (InvalidClassException ex) {
              System.err.println ("Version of cached ChannelList class does not match code.");
              return new ChannelList();
      } catch (Exception ex) {

            ex.printStackTrace();
            return new ChannelList();
      }

      return (ChannelList) obj;
    }

    /** Returns the default ChannelList file name. */
    public static String getCacheFilename() {

  if (cacheFilename == null) setCacheFilename(getFullDefaultCacheFilename());

  return cacheFilename;

    }
    /** Set filename of cache file. The path remains the default. */
    public static void setCacheFilename(String filename) {
  cacheFilename = filename;
    }
    /** Set the path and filename of the cache file. */
    public static void setCacheFilename(String path, String filename) {
  setCacheFilename(path + FileSep + filename);
    }
    /** Returns the default ChannelList file name. */
    public static String getAsciiFilename() {

  if (asciiFilename == null) setAsciiFilename(getFullDefaultAsciiFilename());

  return asciiFilename;

    }
    public static void setAsciiFilename(String filename) {
  asciiFilename = filename;
    }

    /** Returns the default file path name. */
    public static String getDefaultPath() {

  if(System.getProperty("JIGGLE_USER_HOMEDIR") != null)
          return(System.getProperty("JIGGLE_USER_HOMEDIR"));
        else
          return(System.getProperty("user.home","."));
    }
    /** Returns the default ChannelList file name. */
    public static String getDefaultCacheFilename() {

      return "channelList.cache";

    }
    /** Returns the default ChannelList path and file name. */
    public static String getFullDefaultCacheFilename() {

      return getDefaultPath() + FileSep + "channelList.cache";

    }

    /** Returns the default ChannelList file name. */
    public static String getFullDefaultAsciiFilename() {

      return getDefaultPath() + FileSep + "channelList.list";

    }

    /** Returns the default ChannelList file name. */
    public static String getFullDefaultBinaryFilename() {

      return getDefaultPath() + FileSep + "channelList.bin";

    }

/**
 * Return Collection of currently active Channels from the default
 * DataSource Connection..
 */
    public static ChannelList readCurrentList() {
  // use NOW
    return Channel.create().readList();
    }
/**
 * Return Collection of Channels that were active on the given date.
 * Uses the default DataSource Connection.
 */
    public static ChannelList readList(java.sql.Date date) {
  return Channel.create().readList(date);
    }
/**
 * Return an array of Channels that were active on the given date.
 * Uses the default DataSource Connection.
 */
    /* NOTE on timing: reading in 2215 rows takes about 20 sec. on an USparc
     * over the network :. the rate is about 100 rows/sec.
     * When looking up each channel in a seperate query you get about 9 chans/sec
     * :.  the crossover point is about 180 channels. I.e. its more efficient to do
     * individual lookups if its a one time deal and there are less then 180 chans.
     */

    public static ChannelList readList(Connection conn, java.sql.Date date) {
  return Channel.create().readList(conn, date);
//	return Channel.create().readList(Channel.getDateWhereClause(date);

    }

     /** Return count of all channels in data source. */
    public static int getCurrentCount () {
      return Channel.create().getCurrentCount();
    }

    /** Return count of all channels in data source. */
    public static int getCount (java.sql.Date date) {

       return Channel.create().getCount(date);

    }
/**
 * Return Collection of ALL Channels that were in the DataSource. There may be
 * multiple entries for a channel that represent changes through time.
 * Uses the default DataSource Connection.
 */
    public static ChannelList readAllList()
    {
  return readAllList (DataSource.getConnection());
    }

    public static ChannelList readAllList(Connection conn)
    {
  return Channel.create().readList();
    }

    /** Return a list of currently acive channels that match the list of
     *  component types given in
    * the array of strings. The comp is compared to the SEEDCHAN field in the NCDC
    * schema.<p>
    *
    * SQL wildcards are allowed as follows:<br>
    * "%" match any number of characters<br>
    * "_" match one character.<p>
    *
    * For example "H%" would match HHZ, HHN, HHE, HLZ, HLN & HLE. "H_" wouldn't
    * match any of these but "H__" would match them all. "_L_" would match
    * all components with "L" in the middle of three charcters (low gains).
    * The ANSI SQL wildcards [] and ^ are not supported by Oracle.*/
    public static ChannelList getByComponent(String[] compList) {
        return Channel.create().getByComponent(compList);
    }

    /** Return a list of channels that were active on the given date and
    *  that match the list of component types given in
    * the array of strings. The comp is compared to the SEEDCHAN field in the NCDC
    * schema.<p>
    *
    * SQL wildcards are allowed as follows:<br>
    * "%" match any number of characters<br>
    * "_" match one character.<p>
    *
    * For example "H%" would match HHZ, HHN, HHE, HLZ, HLN & HLE. "H_" wouldn't
    * match any of these but "H__" would match them all. "_L_" would match
    * all components with "L" in the middle of three charcters (low gains).
    * The ANSI SQL wildcards [] and ^ are not supported by Oracle.*/
    public static ChannelList getByComponent(String[] compList,  java.sql.Date date) {

        return Channel.create().getByComponent(compList, date);
    }

    /**
     * Find the given Channel in the static channel list. If no entry is
     * found, this method returns the channel name that was passed in the arg as
     * a ChannelNamel.  */
    public Channel findExactInList (Channel chan) {

  Channel ch[] = getArray();

// No efficient way to do this unless we put Channel list in hash table
  for (int i = 0; i < ch.length; i++)
  {
            ChannelName chanName = chan.getChannelName();
      if (chanName.equalsIgnoreCase(ch[i].getChannelName())) {
    return ch[i];		// substitute
      }
  }
  return (Channel) chan;
    }

    /**
     * Find the given Channel in this channel list. If no entry is
     * found, this method returns the channel that was passed in the arg as
     * a Channel.  */
    public Channel lookUp (Channel chan) {
        return findSimilarInList(chan);
    }
    /**
     * Find the given Channel in this channel list. If no entry is
     * found, this method returns the channel name that was passed in the arg as
     * a Channel.  */
    public Channel findSimilarInList (Channel chan) {

  Channel ch[] = getArray();

// No efficient way to do this unless we put Channel list in hash table
  for (int i = 0; i < ch.length; i++)
  {
            ChannelName chanName = chan.getChannelName();
      if (chanName.sameAs(ch[i].getChannelName())) {
    return ch[i];		// substitute
      }
  }
  return (Channel) chan;
    }

    /** Try reading the local cached channel list, if that fails
    read the data source and create a cache list. */
    public static ChannelList smartLoad() {

      ChannelList chanList = ChannelList.readFromCache();

        // cache read failed, read from dbase and write to cache for future use
        if (chanList.isEmpty()) {
           chanList = ChannelList.readCurrentList();
           chanList.writeToCache();
        }

     return chanList;

    }

public String toString() {

  String str="";

  Channel[] ch = getArray();

  if (ch.length == 0) {
      str = ("No channels found.");
  } else {
      for (int i = 0; i < ch.length; i++)
      {
    str += ch[i].toString() + "\n";
      }
  }
  return str;
    }

    public String toDumpString() {

  String str="";

  Channel[] ch = getArray();

  if (ch.length == 0) {
      str = ("No channels found.");
  } else {
      for (int i = 0; i < ch.length; i++)
      {
    str += ch[i].toDumpString() + "\n";
      }
  }
  return str;
    }
/** Dump the Channel List directly to standard output. */
    public void dump() {
  Channel[] ch = getArray();

  if (ch.length == 0) {
      System.out.println("No channels found.");
  } else {
      for (int i = 0; i < ch.length; i++)
      {
    System.out.println(ch[i].toDumpString());
      }
  }
    }

// Inner class to perform Comparator methods for sorting by distance from epicenter

class DistanceSorter implements Comparator {

    public int compare (Object o1, Object o2) {

  Channel c1 = (Channel) o1;
  Channel c2 = (Channel) o2;

  double diff = c1.dist.doubleValue() - c2.dist.doubleValue();
        if (diff < 0.0) return -1;
        if (diff > 0.0) return  1;
  return 0;

    }

} // end of DistanceSorter inner class

/**
 * Main for testing:
 */
    public static void main (String args[])
    {

        System.out.println ("Making connection... ");

        DataSource db = new TestDataSource();

        System.out.println (db.toString());

/*
  // NOTE: this demonstrates making a static Channel list.
        System.out.println ("Reading CURRENT channels...");
      bm.reset();
  //	Channel.setList (Channel.getAllList());
      Channel.setList (Channel.getCurrentList());

  //        System.out.println ("Reading channels with net=CI...");
  //	ArrayList chanList =
  //	    (ArrayList) Channel.readListWhere("Where net = 'CI'");

  System.out.println ("Found "+Channel.chanList.size()+" active channels.");
  bm.print();

  //
  Channel cn2 = new Channel("NC", "PTQ", "VHZ", "EHZ");
  System.out.println ("Find "+cn2.toDumpString()+ " in list...");

  Channel found = Channel.findExactInList(cn2);

  if (found == cn2) {
      System.out.println ("No exact match found.");
  } else {
      System.out.println ("Found exact: " + found.toString());
  }

  found = Channel.findSimilarInList(cn2);
  if (found == cn2) {
      System.out.println ("No similar match found.");
  } else {
      System.out.println ("Found similar: " + found.toString());
  }

  //
  cn2 = new Channel("CI", "HOD", "VHZ", "EHZ");
  System.out.println ("Find "+cn2.toDumpString()+ " in list...");

  found = Channel.findExactInList(cn2);

  if (found == cn2) {
      System.out.println ("No exact match found.");
  } else {
      System.out.println ("Found exact: " + found.toString());
  }

  found = Channel.findSimilarInList(cn2);
  if (found == cn2) {
      System.out.println ("No similar match found.");
  } else {
      System.out.println ("Found similar: " + found.toString());
  }

  // Test sorting

  Channel.distanceSort(new LatLonZ (34.35, -118.86, 8.0));

  // dump list
  Channel[] cha = getArray();
  String str="";

  int len = cha.length;
  //	int len = 100;

  if (cha.length == 0) {
      str = ("No channels in list.");
  } else {
      for (int i = 0; i < len; i++)
      {
    System.out.println (cha[i].toDumpString()+ " dist= "+cha[i].dist);
      }
  }
*/
  String complist[] = {"HH_", "HL_", "EH_", "BH_"};

         ChannelList list = Channel.create().getByComponent(complist);
    // test serialization

    System.out.println ("Reading in current channel info ...");
    BenchMark bm  = new BenchMark();
    bm.print("Channels loaded = "+list.size());
    bm.reset();

    list.calcDistancesFrom(new LatLonZ(33.2, -116.5, 0.0));
    list.getMaximumAzimuthalGap();



//    boolean status = list.writeAsciiFile("channelList.test");
    boolean status = list.writeAsciiFile();

    bm.print( "Write ascii = "+status);
    bm.reset();

    ChannelList newList = new ChannelList();
    status = newList.readAsciiFile(ChannelList.getAsciiFilename());

    bm.print("Read ascii = "+status+ " size = "+newList.size());
    bm.reset();

    System.out.println ( "Write cache = " + list.writeToCache() );
    bm.print ("Write cache = "+list.size());
    bm.reset();

    list = new ChannelList();
//    System.out.println ("nuked it, list length = "+list.size());

    list = ChannelList.readFromCache();
    bm.print ( "Read cache = "+ list.size());
    bm.reset();

    System.out.println ( "Write binary = " + list.writeBinaryFile() );
    bm.print ("Write binary = "+list.size());
    bm.reset();

/// test cache
/*
    System.out.println ("Reading in current channel info from cache...");
    //ChannelList list2 = ChannelList.readFromCache();
    ChannelList list2 = ChannelList.smartLoad();

    System.out.println ( "Read cache, list length = "+
          list2.size());

//    System.out.println (list2.toString());
    System.out.println ( "Sort test...");

      list2.distanceSort(new LatLonZ(33., -117., 0.1));
      // dump list
  Channel cha[] = list2.getArray();
  String str="";

     int end = Math.min(cha.length, 30);
  if (cha.length == 0) {
      str = ("No channels in list.");
  } else {
      for (int i = 0; i < end; i++)
      {
    System.out.println (cha[i].toDumpString()+ " dist= "+cha[i].dist);
      }
  }
*/

 }  // end of Main



}
