package org.trinet.jiggle;

import java.text.*;
import java.awt.*;
import java.util.*;
import javax.swing.event.*;
import java.awt.event.*;
import javax.swing.*;

import org.trinet.jdbc.*;
import org.trinet.jasi.*;
import org.trinet.util.graphics.NumberChooser;

/**
 * ScrollPane that contains the MultiWFPanels. Is a 'view' in the MVC group for
 * selectionModel so that it can keep the selected WFPanel visible in the
 * ViewPort. <p>
 *
 * If the view object has focus this component responds to the default keyboard actions
 * for JScrollPane as defined in the Java Look and Feel:<p>
 <tt>
JScrollPane (Java L&F) (Navigate refers to focus)
Navigate out forward      |  Tab
Navigate out backward     |  Shift+Tab
Move up/down              |  Up, Down
Move left/right           |  Left, Right
Move to start/end of data |  Ctrl+Home, Ctrl+End
Block move up/down        |  PgUp, PgDn
Block move right          |  Ctrl+PgDn
Block move left           |  Ctrl+PgUp
</tt>
*/

/*
*** RUN-AWAY SCROLL BAR BEHAVIOR ****

If you press and hold the vertical scroll arrow the scroller runs-away. It
appears that holding the arrow causes many scroll events to be queued and the
scroller scrolls until they are all serviced. This seems to be a general bug in
the JScrollPane. It can happen wiht the JTable in the CatalogPanel also.
I don't know how to fix this weirdness. DDG 11/19/99 */

public class WFScroller extends JScrollPane {

    MasterView   mv;	    // the master view

    /** The underlying panel that shows thru the viewport. */
    public WFGroupPanel groupPanel;

    /** How many panels are visible in the viewport */
    static final int DEFAULT_PANEL_COUNT = 10;
    int panelsInView = DEFAULT_PANEL_COUNT;

    double secondsInViewport = 90.0;
    boolean showFullTime = false;    // flag for "show all"
    boolean showRowHeader = false;

// listen for changes to selected channel or time/amp window
    WFWindowListener wfwChangeListener = new WFWindowListener();
    WFViewListener   wfvChangeListener = new WFViewListener();
    VerticalScrollListener vertScrollListener = new VerticalScrollListener();

    /** Null constructor */
    public WFScroller () {
    }

   /**
     * Constructor with placeholder string. This is used to create a valid WFScroller
     * when there is no data loaded yet.
     */
    public WFScroller(String str) {
  add(new JLabel(str));
    }

    /**
     * Create a WFScroller containing a WFGroupPanel that has one passive
     * WFPanel for each WFView in the MasterView. By default 10 WFPanels will be
     * visible in the viewport.  */
    public WFScroller (MasterView mview) {
  this(mview, DEFAULT_PANEL_COUNT, false);
    }

    /**
     * Create a WFScroller containing a WFGroupPanel that has one WFPanel
     * for each WFView in the MasterView. If the second argument is 'true'
     * the WFPanels will be active and will respond to mouse events and
     * notify the MasterWFVModel. By default 10 WFPanels will be visible in
     * the viewport.
     */
    public WFScroller (MasterView mview, boolean activePanels) {

  this(mview, DEFAULT_PANEL_COUNT, activePanels);
    }
    /**
     * Create a WFScroller containing a WFGroupPanel that has one WFPanel
     * for each WFView in the MasterView. The second argument is the number of
     * WFPanels visible in the scroller's viewport. If the third argument is 'true'
     * the WFPanels will be active and will respond to mouse events and
     * notify the MasterWFVModel.<p>
     * Note that a fraction of a WFPanel may be visible at the bottom of the view.
     * This is because the viewport size (in pixels)  may not be perfectly
     * divisible by the value of 'Jiggle.props.tracesPerPage' and the "remainder" is extra pixels
     * that show at the bottom as the next panel.
     */
    public WFScroller (MasterView mview, int panelsVisible, boolean activePanels) {

  this.mv = mview;

        addListeners();

  // make a panel with all the WFPanels in it
  // this is what will be scrolled and show thru the viewport
  groupPanel = new WFGroupPanel(mv, activePanels);

  // define the component being scrolled
  this.setViewportView(groupPanel);

  //  speeds scrolling mucho
        // deprecated in v1.3
  //getViewport().setBackingStoreEnabled(true);

  // scales the groupPanel to fit right number of WFPanels in the viewport
  setPanelsInView(panelsVisible);

/*  // Not a good idea because the lower-right corner DISAPPEARS when there is no
    // horizontal scrollbar!
     JToggleButton cornerButton = new JToggleButton();
//     cornerButton.setPressedIcon(unfilterIcon);
     cornerButton.setToolTipText("Toggle fixed/auto timescale");
     cornerButton.addActionListener(new CornerButtonHandler());

     add(cornerButton, ScrollPaneConstants.LOWER_RIGHT_CORNER);
*/
    // show row header?
      if (getShowRowHeader())
      setRowHeaderView(getWFGroupPanel().getRowHeaderPanel());

    }

    /** Scales the groupPanel to fit right number of WFPanels in the viewport */
    public void setPanelsInView (int panelsVisible) {

  // sanity checks

  // don't exceed max panels
  if (panelsVisible > groupPanel.getPanelCount()) {
      panelsVisible = groupPanel.getPanelCount();
  }

  if (panelsVisible < 1) return;	// positive values only!

  panelsInView = panelsVisible;
        // tell the model so it can jump pages
        mv.masterWFViewModel.setViewsPerPage(panelsInView);

  int topIndex = getTopIndex();

  // use viewport size to set dimensions of ONE wfpanel in the group panel
  JViewport vp = getViewport();
  int height   = vp.getHeight() / panelsInView;
  int width    = vp.getWidth();
  // this means: just repaint area newly scrolled into view
  // should improve performance
// v1.3 only	vp.setScrollMode(JViewport.BLIT_SCROLL_MODE);

       // handle auto vs fixed time scale
       if (getSecondsInViewport() > 0.0) {
   width =  (int)(((double)width / getSecondsInViewport())* mv.getViewSpan().getDuration());
       }
  // This will scale the whole WFGroupPanel
  groupPanel.setSinglePanelSize(width, height);

  // keep same top panel
  setTopIndex(topIndex);

  validate();
    }

    /** Add model listeners */
    protected void addListeners() {
       addListeners (mv);
    }

    /** Add model listeners to this MasterView's models. */
    protected void addListeners (MasterView mview) {

     // add this panel as an observer of the MVC model of the MasterView
     // to which the WFView belongs
     if (mview != null) {

        // scroll listener to manage waveform cache
        this.getVerticalScrollBar().addAdjustmentListener(vertScrollListener);

       // listener for selected panel changes
         if (mview.masterWFViewModel != null) {
             wfvChangeListener = new WFViewListener();
             mview.masterWFViewModel.addChangeListener(wfvChangeListener);
         }
       // listener for selected time window changes
         if (mview.masterWFWindowModel != null) {
           mview.masterWFWindowModel.addChangeListener(wfwChangeListener);
         }

     }
    }
    /** Remove model listeners to this MasterView's models. */
    protected void removeListeners (MasterView mv) {

     // add this panel as an observer of the MVC model of the MasterView
     // to which the WFView belongs
     if (mv != null) {

       this.getVerticalScrollBar().removeAdjustmentListener(vertScrollListener);

       // listener for selected panel changes
         if (mv.masterWFViewModel != null) {
             mv.masterWFViewModel.removeChangeListener(wfvChangeListener);
         }
       // listener for selected time window changes
         if (mv.masterWFWindowModel != null) {
           mv.masterWFWindowModel.removeChangeListener(wfwChangeListener);
         }

     }
    }
    /** Clean up this object before destruction. Removes listeners from other
    * components, otherwise dangling references will not allow this object to
    * be garbage collected. */
/*
    public void destroy () {
         removeListeners(mv);
    }
*/
    /** Return the value, in pixels, to be used as the vertical scrolling unit
  increment. Jump one trace */
    public int getUnitHeight() {

  return groupPanel.singlePanelSize.height;

    }

    /** Return the value, in pixels, to be used as the vertical scrolling block
  increment. Jump one screen minus one trace */
    public int getBlockHeight() {

  return getViewport().getHeight() - groupPanel.singlePanelSize.height;

    }
    public void setWFGroupPanel(WFGroupPanel panel) {
        groupPanel = panel;
    }
    public  WFGroupPanel  getWFGroupPanel() {
       return groupPanel;
    }
/** Override paint() so that vertical scrollbar increments will be reset if the
    panel is resized */
    public void paintComponent (Graphics g) {

  JScrollBar vbar = getVerticalScrollBar();
  vbar.setUnitIncrement(getUnitHeight());
  vbar.setBlockIncrement(getBlockHeight());

  // adjust the waveform cache (if there is one)
// This stopped working in SDK1.3, so I added the VerticalScrollListener
//	mv.wfvList.setCacheIndex(getTopIndex());

  // resize wfpanels here
  setPanelsInView(panelsInView);

  // now do normal repaint
  super.paintComponent(g);
    }

    /**
     * Return the index of the wfPanel at the top of the viewport
     */
    public int getTopIndex() {

  if (getScrollPixelsPerPanel() <= 0) {
          return 0;
  } else {
    return (int) Math.ceil((double)getVerticalScrollBar().getValue() / (double)getScrollPixelsPerPanel());
        }
    }

    /**
     * Set the top visible WFPanel to this index value of the WFGroupPanel. The move is
     * done via the scrollbar so that the scroll bar stays in synch.
     */
    public void setTopIndex (int index) {

  int pixValue = index * getScrollPixelsPerPanel();

  getVerticalScrollBar().setValue(pixValue);
    }

    /**
     * Set the center visible WFPanel to this index value of the
     * WFGroupPanel. The move is done via the scrollbar so that the scroll bar
     * stays in synch.  */
    public void setCenterIndex (int index) {

  int idx = index - (panelsInView/2);
  if (idx < 0) idx = 0;
  setTopIndex(idx);
    }


    /** Return how many scroll bar pixels represent on WFPanel in the viewport */
    int getScrollPixelsPerPanel() {
  JScrollBar vbar = getVerticalScrollBar();
  return (int) ( (double)vbar.getMaximum() /
           (double)groupPanel.getComponentCount());
    }

/**
 * Make this WFPanel visible in the viewPort. If it's already visible do nothing.
 */
 public void makePanelVisible(WFPanel wfp) {

     if (wfp == null) return;
     if (panelIsVisible(wfp)) return;		// only move is necessary

     Dimension size = wfp.getSize();
     wfp.scrollRectToVisible(new Rectangle(0,0, size.width, size.height));
 }

 /** Set the width of the viewport in seconds. If 'secs' <= 0.0 then the width
  *  is set to the legth of the data. Note that if the value set here is greater
  *  than the duration of the data in the view getSecondsInViewport() will return
  *  the data duration rather than this value. This prevents a "short" view with
  *  gray space at the end. */
 public void setSecondsInViewport(double secs) {
   if (secs <= 0.0) {
     setShowFullTime(true);
   } else {
     setShowFullTime(false);
     secondsInViewport = secs;
   }
 }
/** Return the width of the Viewport in seconds. This value is set with
setSecondsInViewport(). If the value passed to setSecondsInViewport() is <= 0
this method returns the width of the MasterView's viewspan.
The returned value will never be greater than the duration of the data even if
setSecondsInViewport() is used to set it to a larger value. */
 public double getSecondsInViewport() {
    if (getShowFullTime()) return mv.getViewSpan().getDuration();

    return Math.min(secondsInViewport, mv.getViewSpan().getDuration());
 }
 /** Set true to show the fill time series in the window. */
 public void setShowFullTime(boolean tf) {
     showFullTime = tf;
 }
 /** Returns true if flag is set to show the fill time series in the window. */
 public boolean getShowFullTime() {
     return showFullTime;
 }
 /** Set true to show the WFPanel row headers in the scroller. */
 public void setShowRowHeader(boolean tf) {

     if (getShowRowHeader() == tf) return;    // no change

     showRowHeader = tf;
     if (getShowRowHeader()) {
      setRowHeaderView(getWFGroupPanel().getRowHeaderPanel());
     } else {
      setRowHeaderView(null);
     }
 }
 /** Returns true if flag is set to show the WFPanel row headers in the scroller. */
 public boolean getShowRowHeader() {
     return showRowHeader;
 }
 /**
 * Center this WFPanel in the viewPort. Sets as the selected view.
 */

 public void centerViewportOnPanel(WFPanel wfp) {

     // set selected
 // causes infinite loop!    mv.masterWFViewModel.set(wfp.wfv);

     Point viewXY = getViewport().getViewPosition();	// get ViewPort origin
     // only change Y for now
     Point newPoint = new Point(viewXY.x, wfp.getY());

     getViewport().setViewPosition (newPoint);

     validate();	// insure complete repaint, it may have done its 'update' before we did
 }

 /**
 * Center this Channel's WFPanel in the viewPort.
 */

 public void centerViewportOnPanel(Channel chan) {

     WFView wfview = mv.wfvList.get(chan);

     if (wfview == null) return;

     WFPanel wfp = groupPanel.wfpList.get(wfview);

     if (wfp == null) return;

     centerViewportOnPanel(wfp);

 }

 /** Return true if the given WFPanel is visible in the viewport */
     public boolean panelIsVisible(WFPanel wfp) {

       Rectangle viewRect = getViewport().getViewRect();		// whats showing?

       if ( viewRect.contains( wfp.getLocation() ) &&			// top corner
         viewRect.contains (wfp.getX(), wfp.getY()+ wfp.getHeight())	// bottom corner
    ) return true;

       return false;

     }

/** INNER CLASS: Handle changes to the selected WFView. */
      class WFViewListener implements ChangeListener {
      public void stateChanged (ChangeEvent changeEvent) {

            WFView newWFV = ((SelectedWFViewModel) changeEvent.getSource()).get();

            if (groupPanel != null &&  groupPanel.wfpList != null) {
              makePanelVisible(groupPanel.wfpList.get(newWFV));
           }

          }
      }  // end of WFViewListener

/** INNER CLASS: Handle changes to the selected time/amp window. */
      private class WFWindowListener implements ChangeListener {
      public void stateChanged (ChangeEvent changeEvent) {
              // NOOP yet, keep time window in view
          }
      }  // end of WFWindowListener

/** Handle filter button actions. */
/*
class CornerButtonHandler implements ActionListener {

     double viewSpan = secondsInViewport;

  public void actionPerformed (ActionEvent evt) {

         // toggle state
         setShowFullTime(!getShowFullTime());
         repaint();

         AbstractButton but = (AbstractButton) evt.getSource();
         if (getShowFullTime()) {
//             but.setBackground(Color.red);
           but.setText("X");
         } else {
//             but.setBackground(Color.gray);
           but.setText("O");
         }
     }
}
*/
/// --------------------------------------------------------------------------
/**
 * Main for testing
 */

    public static void main (String args[])  {

  int evid;

  if (args.length <= 0)	// no args
  {
    System.out.println ("Usage: java WFScroller [evid])");

    evid = 9655069;
    System.out.println ("Using evid "+ evid+" as a test...");

  } else {

    Integer val = Integer.valueOf(args[0]);	    // convert arg String to 'double'
    evid = (int) val.intValue();
  }

// make a frame
        JFrame frame = new JFrame("Main");
        frame.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {System.exit(0);}
        });

        System.out.println ("Making connection...");
  DataSource init = new TestDataSource();  // make connection

  // NOTE: this demonstrates making a static Channel list.
        System.out.println ("Reading in station list...");
  //	Channel.setList (Channel.getAllList());

        System.out.println ("Making MasterView for evid = "+evid);

  MasterView mv = new MasterView();
  mv.setWaveFormLoadMode(MasterView.Cache);
        mv.setAlignmentMode(MasterView.AlignOnTime);
  mv.defineByDataSource(evid);

  final WFScroller wfs = new WFScroller(mv) ;	    // make scroller

        NumberChooser numberChooser =
      new NumberChooser(5, 30, 5, wfs.panelsInView);

  Container contentPane = frame.getContentPane();

        contentPane.add(numberChooser, BorderLayout.NORTH);

        numberChooser.addActionListener(new ActionListener()
        {
          public void actionPerformed(ActionEvent e)
             {
              NumberChooser nc = (NumberChooser) e.getSource();
              int val = (int) nc.getSelectedValue();
              System.out.println (" set panels in view = " + val);
        wfs.setPanelsInView(val);
             }
         });

  frame.getContentPane().add( wfs );	    // add scroller to frame
  contentPane.add(wfs, BorderLayout.CENTER);

        frame.pack();
        frame.setVisible(true);

  frame.setSize(800, 800);	// must be done AFTER setVisible

    }
/**
 * Keep the waveform cache centered on the area in view. This was done via the
 * repaint() method until v1.3 when it stopped working! Seems someone optimized
 * the code to reduce repaints.
 * Note: Adjustment events are triggered by BOTH mouse down and mouse up.
 * So use isAdjusting()to reduce hyperactivity.
 * They are also fired by a frame resize but the 'value' is 0.
 */
 private class VerticalScrollListener implements AdjustmentListener {

 /** Keep WFViewList.cacheMgr, if there is one, in synch with the scroller. */
       public void adjustmentValueChanged (AdjustmentEvent e) {
          if ( ((JScrollBar)e.getSource()).getValueIsAdjusting() == false) {
       if (mv.wfvList.cacheMgr != null)
                       mv.wfvList.cacheMgr.setIndex(getTopIndex());
          }
       }
  }

} // end of class
