package org.trinet.jasi;

import java.lang.*;
import java.io.File;
import java.util.*;
import java.sql.Connection;


import org.trinet.jdbc.datatypes.*;
import org.trinet.util.*;
import org.trinet.util.WaveClient;
//import org.trinet.filters.GenericFilter;

/**
 * Represents the waveform time series for one channel for a time window (TimeSpan).
 * Because a time window may contain gaps or no data, the TimeSeries may contain
 * from zero to 'n' WFSegments. Each WFSegment is a CONTIGOUS time series.<p>
 *
 * Fires ChangeEvents when the time-series is loaded so that time-series can be
 * loaded asynchronously (in background thread) then notify a graphics object when
 * loading is complete.<p>
 *
 * Note that a waveform may be "empty", that is contain no time series. This may be
 * true before the loader thread has retreived the time-series. Check this
 * condition using the boolean method hasTimeSeries(). <p>
 *
 * <b>Note on synchronization: </b> Waveforms can be loaded in a background thread.
 * Therefore, care must be taken to make concrete methods that implement and extend
 * this class THREAD SAFE. This is best done by synchronizing on the specific
 * waveform instance ("this").
 * For example:
 <tt>
public class WaveformConcrete extends Waveform {

     public void someMethod() {
       synchronize (this) {
         // modify the waveform here
       }
     }
}

 <\tt>

 * Extensions of this class are also responsible for knowing how to read in
 * timeseries from various data sources.
 * @see: loadTimeSeries()
 * @see: loadTimeSeries(double startTime, double endTime)
 * <p>
 *  The wave source is set using
 *  @see: setWaveSource(Object source)
 */

public abstract class Waveform extends JasiTimeSeries {

    /** Time-series encoding format as defined in SEED v2.3, see SEED
     * Ref. Man. pg. 106
     *
     * @See: org.trinet.jasi.SeedEncodingFormat() */
    public DataLong    encoding		= new DataLong();

    /** "C" = continuous, "T" = triggered. */
    public DataString  dataMode		= new DataString();

    /** Optional unique ID number of a waveform. May not exist in all systems.
     * For NCDN schema data this would be the 'wfid'. */
    public DataLong    id		= new DataLong();

    /** Optional unique ID external number of a waveform. May not exist in all
        systems. */
    public DataString  externalId	= new DataString();

    /** If the waveform is associated with a particular event in the underlying
  data source, this is the unique ID of that event. */
    public DataString savingEvent = new DataString(); // event ID for which data was saved

// private, implimentation specific data members

    /** The maximum amplitude sample. */
    Sample maxSample;
    /** The mminimum amplitude sample. */
    Sample minSample;

    /** Units of the amplitude dimension.
    * @See: Units */
    int ampUnits = Units.UNKNOWN;

    /** The value of the bias (DC offset).*/
    protected float biasVal = 0.0f;

    /** Host where waveform file resides. */
    protected String host;
    /** Directory path to the waveform file. */
    protected String path;
    /** Filename of the waveform file. */
    protected String filename;
    /** Byte offset to data in file */
    protected int    fileOffset;
    /** Number of bytes of data in the file for this waveform. */
    protected int    byteCount;

    /* Enumerate time-series load source options */
    public static final int LoadFromDataSource = 0;
    public static final int LoadFromWaveServer = 1;

    /** Default source of time-series data. These are static so ALL waveforms must
     * be comming from the same data source. */
    static int waveDataSourceType = LoadFromDataSource;

    /* This will be either a DataSource (default) or a WaveClient */
    // The following blows up if default data source is not available
    //    static Object waveDataSource  = new DataSource();
    static Object waveDataSource;

    /** The travel-time model used to calculate the energy window. */
    //TravelTime ttModel = new TravelTime();
    TravelTime ttModel = TravelTime.getInstance();

// ////////////////////////////////////////////////////////////////////////////
    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. Creates a Solution of the DEFAULT type.
     * @See: JasiObject
     */
     public static final Waveform create() {
  return create(DEFAULT);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is an integer implementation type.
     * @See: JasiObject
     */
     public static final Waveform create(int schemaType) {
        return create(JasiFactory.suffix[schemaType]);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is as 2-char implementation suffix.
     * @See: JasiObject
     */
     public static final Waveform create(String suffix) {
  return (Waveform) JasiObject.newInstance("org.trinet.jasi.Waveform", suffix);
     }
// ////////////////////////////////////////////////////////////////////////////

    /** Do "deep" copy of the header, not the data, portion of the Waveform. */
    public void copyHeader(Waveform wf) {

       synchronized (wf) {           // make sure another thread doesn't change it under us
     setChannelObj((Channel)wf.getChannelObj().clone());
           //setChannelObj(Channel.create().setChannelObj(wf.getChannelObj()));

           format             = new DataLong(wf.format);
           encoding	      = new DataLong(wf.encoding);
           dataMode	      = new DataString(wf.dataMode);

           setSampleRate(wf.getSampleRate());
           id		      = new DataLong(wf.id);
           externalId	      = new DataString(wf.externalId);

           // 'primative' arg passed by value :. not passed by ref
           setStart(wf.getStart().doubleValue());
           setEnd(wf.getEnd().doubleValue());

           //timeStartActual	= new DataDouble(wf.timeStartActual);
           //timeEndActual	     = new DataDouble(wf.timeEndActual);

           savingEvent        = new DataString(wf.savingEvent);

           if (wf.host != null) host = new String(wf.host);
           if (wf.path != null) path = new String(wf.path);
           if (wf.filename != null) filename = new String(wf.filename);
           fileOffset         = wf.fileOffset ;
           byteCount          = wf.byteCount;

           waveDataSourceType = wf.waveDataSourceType;

           // Note: this copies the reference to this Object
           waveDataSource     = wf.waveDataSource;
       } // end of synch

    }  // end of copyHeader


    /** Do "deep" copy of the WFSegments in the Waveform. */
    public void copyTimeSeries(Waveform wf) {

           if (wf.hasTimeSeries()) {
              synchronized (wf) {
                  WFSegment seg[] = wf.getArray();

                  biasVal = getBias();

                  if (wf.getMaxSample() != null)
                    maxSample = wf.getMaxSample().copy();
                  if (wf.getMinSample() != null)
                    minSample = wf.getMinSample().copy();

               for (int i=0; i < seg.length; i++) {
                      WFSegment wfs = new WFSegment();
                      wfs.copy(seg[i]);
                      segList.add(wfs);
                  }
              } // end of synch
           }

    }

    //** Return a "deep" copy of the passed waveform. */
    public static Waveform copy (Waveform wf) {

           Waveform newWf = Waveform.create();

           newWf.copyHeader(wf);

           newWf.copyTimeSeries(wf);

           return newWf;

    } // end of copy()


    /** Set the source of the time-series data */
    public static boolean setWaveSource (Object source) {

  if (source instanceof DataSource) {

     // debug
        System.out.println ("Wavesource set to dbase.");

      waveDataSource = source;
      waveDataSourceType = LoadFromDataSource;
      return true;
  } else	if (source instanceof WaveClient) {

             System.out.println ("Wavesource set to waveserver.");

      waveDataSource = source;
      waveDataSourceType = LoadFromWaveServer;
      return true;
  }

  System.out.println ("Waveform: bad waveform data source.");

  return false;
    }

    public static Object getWaveSource () {
  return waveDataSource;
    }

    /** Return a string describing the data source. It its a WaveClient is
     *  returned. If its a dbase a string of the form "dbase@host" is returned.*/
    public static String getWaveSourceName () {
   if (getWaveSourceType() == LoadFromDataSource) {
      return ((DataSource)waveDataSource).getDbaseName()+
             "@"+((DataSource)waveDataSource).getHostName();
   } else if (getWaveSourceType() == LoadFromWaveServer) {
      return "WaveServer";
   } else {
     return "";
   }
    }
    public static int getWaveSourceType () {
  return waveDataSourceType;
    }

    /**
 * Return string with path/filename for this waveform
 */
    public String getPathFilename() {
  //	return path+"/"+filename;
  return path+filename;
    }

    /**
     * Set the Path string. Expects a trailing "/" in UNIX or
     *  "\" in WINTEL
     */
    public void setPath (String path) {
  this.path = path;
    }

    /**
     * Set the filename string.
     */
    public void setFilename (String filename) {
  this.filename = filename;
    }
/**
 *
 */
    public int getFileOffset() {
  return fileOffset;
    }
/**
 *
 */
    public void setFileOffset(int IN_iFileOffset) {
 fileOffset  = IN_iFileOffset;
    }
/**
 *
 */
    public int getByteCount() {
  return byteCount;
    }

/**
 * Returns the TimeSpan covered by the time-series. It may contain holes.
 */
    public TimeSpan getTimeSpan()
    {
  return new TimeSpan ( timeStart.doubleValue(), timeEnd.doubleValue());
    }

/**
 * Get one waveform from the DataSource by Waveform record ID #
 */
    public abstract Waveform getByWaveformId (long wfid);

/**
 * Get an array of waveforms from the DataSource by solution ID #
 */
    public abstract Collection getBySolutionId (long id);

/**
 * Get an array of waveforms from the DataSource by solution ID #
 */
    public abstract Collection getBySolution (Solution sol) ;

    /**
     * Returns 'true' if generic waveform path as returned by the dbase are local
     *  i.e. are on a file system where they can be opened either on the client
     * machine or NFS mounted.
     * Gets generic waveform file
     * root from the database and checks that is can be reached. Needs a
     * DataSource open connection.
     */
    public boolean filesAreLocal()
    {
  String root = getFileRoot();

  if (root == null) return false;

  //	System.out.println ("root: "+root);

  // check existance
  File file = new File(root);

  //	System.out.println ("file: "+file.toString());

  return file.isDirectory();
    }
    /** Returns true if this Waveform is accessible through some transport
     *  method. */
/*  DEPRECATED
    public abstract boolean isAccessible () ;
*/
    /**
     * Return the generic waveform file root.  Access to this path is tested
     * to determine if files can be read locally.
     */
    protected abstract String getFileRoot();

    /**
     * Return the waveform file root for a particular event id.
     * Access to this path is tested
     * to determine if files can be read locally.
     */
    protected abstract String getFileRoot(long eventId);

    /**
     * Create AssocWaE record for existing WaveForm and pointing at this Solution.
     * The underlying stored procedure commits, so you don't have to.
     * */

    public abstract boolean commit (Solution sol);

/**
 * Load the time series data for this Waveform. This is done as a separate step
 * because it is often more efficient to only load selected waveforms after
 * you have examined or sorted the Waveform objects by their header info.<p>
 *
 * The waveform is collapsed into the minimum number of WFSegments possible. More
 * then one WFSegment would only be necessary if there are time tears. The start-stop
 * times of the original data packets is preserved in the packetSpans array.<p>
 *
 * Fires ChangeEvents when the time-series is loaded so that time-series can be
 * loaded asynchronously (in background thread) then notify a graphics object when
 * loading is complete. <p>
 *
 * This method must synchronize on the waveform object for thread safty. Simply
 * synchronizing the method is not enough.
 */

    public abstract boolean loadTimeSeries() ;
    public abstract boolean loadTimeSeries(double startTime, double endTime) ;

    /**
     * Delete the time series for this waveform and allow the GC to reclaim
     * the memory space. This is used to manage memory and prevent overflows.  */
    public boolean unloadTimeSeries() {
     synchronized (this) {
    segList = new ArrayList() ;
    packetSpans = null;
     }
  return true;
    }

    /** Reduce waveform to the minimum possible number of WFSegments. Preserve
    * WFSegment boundaries in packetSpans array. */
    public void collapse() {
      // save the original packet timeSpans so we can plot packet boundaries later
      // also detects time tears
      WFSegment seg[] = WFSegment.getArray(getSegmentList());

      packetSpans = new TimeSpan[seg.length];
            // scan original packets to preserve pack boundaries for later display
      for (int i=0; i < seg.length; i++) {
        packetSpans[i] = new TimeSpan(seg[i].getEpochStart(),
                                                  seg[i].getEpochEnd());
            }

      // collapse multiple segements into minimum possible
      setSegmentList(WFSegment.collapse(getSegmentList()));
     }

    /** Return true if there are actual samples (time-series) loaded for this
        waveform. */
    public boolean hasTimeSeries() {
  if (segList.size() == 0)  return false;
  return true;
    }
 /**
 * Convert this Collection of WFSegments to an Array of WFSegments.
 */
    public WFSegment[] getArray() {
      if (segList.size() == 0) return (null);	    // no segments

      WFSegment segArray[] = new WFSegment[segList.size()];
      return (WFSegment[]) segList.toArray(segArray);
    }
    /**
     * Return total samples in memory for this Waveform. Note that they may not
     * be contiguous.
     */
     public int samplesInMemory() {
      if (!hasTimeSeries()) { return 0; }	// no time series

    int k = 0;
    WFSegment seg[] = WFSegment.getArray(segList);

    for (int i=0; i < seg.length; i++) {
      k += seg[i].ts.length;
    }
    return k;
    }

    /** Return the units of the amplitude dimension of this Waveform.
    @See: Units */
    public int getAmpUnits() {
        return ampUnits;
    }
    /** Set the units of the amplitude dimension of this Waveform.
    @See: Units */
    public void setAmpUnits(int units) {
      if (Units.isLegal(units)) ampUnits = units;
    }
    /** Set the units of the amplitude dimension of this Waveform.
    @See: Units */
    public void setAmpUnits(String units) {
      ampUnits = Units.getInt(units);
    }

    /** Return the Sample Interval (secs) for this Waveform */
    public double getSampleInterval() {
  if (samplesPerSecond.isNull()) return 0;
  return 1.0/samplesPerSecond.doubleValue();  //Samp/sec -> sec/Samp
    }

/**
 *  Return the time of where the sample closest to the given time would be.
 *  NOTE: there may be no sample there if there is a time tear or the time is beyond
 *  either end of the data! Times are raw epoch seconds.
 *  If there is no time series the original time is returned.
 */
    public double closestSampleInterval(double dtime) {
  if (!hasTimeSeries()) { return dtime; }	// no time series

  double dt0 = timeStart.doubleValue();
  double sps = samplesPerSecond.doubleValue();

  // integral number of samples to from startTime to dtime
  double nsamps = Math.rint((dtime - dt0) * sps);

  // offset in sec's to that sample
  return dt0 + (nsamps / sps);

    }

/**
 * Calculate the bias of this waveform by examining all waveform segments.
 * Sets bias = 0.0 if there is no time-series.
 * For efficiency this should only be called the first time a time-series
 * is loaded.
 */
  public void scanForBias () {
    synchronized (this) {
      if (!hasTimeSeries()) {
         biasVal = 0.0f;
         return;
      }	// no time series

      double sum = 0.0;
      double totalSamples = 0.0;

      WFSegment seg[] = WFSegment.getArray(segList);

      //Segments aren't equal in length :. weight the bias by sample count
      for (int i=0; i < seg.length; i++) {
        sum += seg[i].getBias() * seg[i].ts.length;
        totalSamples += seg[i].ts.length;
    }

      biasVal = (float) (sum / totalSamples);
    } // end of synch
  } // end of bias

  /** Remove the bias from this time series in place. */
  public void demean() {
      if (!hasTimeSeries()) return;

      scanForBias();
      WFSegment seg[] = WFSegment.getArray(segList);

      float val = - (getBias());  // must subtract :. "-"

      for (int i=0; i < seg.length; i++) {
        seg[i].addY(val);
      }

  }

  /** Return this waveform filtered. To reduce edge effects
  * the WFSegments are concatinated before they are filtered. */
  public Waveform filter(FilterIF filter) {
      if (hasTimeSeries())  {
  return (Waveform) filter.filter(this);
      } else {
  return this;
      }
  }

/**
* Scan the whole time-series for the recified peak. Returns the Sample of the
* peak which may have a negative value; the bias is NOT removed.
* Returns null if no timeseries.
*/
  public Sample scanForPeak () {

     return scanForPeak(getTimeSpan());
  }
/**
 * Return the Sample with the maximum amplitude in the waveform.
 * "Maximum" value could be
 * the largest excursion downward. Bias is NOT removed.
 * Because of large biases even "positive" peak
 * could be a negative number. Returns 'null' if no time-series.
 */
  public Sample scanForPeak (double startTime, double endTime) {
    return scanForPeak (new TimeSpan(startTime, endTime));
  }
/**
 * Return the Sample with the maximum amplitude in the waveform.
 * "Maximum" value could be
 * the largest excursion downward. Bias is NOT removed.
 * Because of large biases even "positive" peak
 * could be a negative number. Returns 'null' if no time-series.
 */
  public Sample scanForPeak (TimeSpan timeSpan) {

      if (!hasTimeSeries()) return null;	// no time series

    synchronized (this) {
      WFSegment seg[] = WFSegment.getArray(segList);

      Sample peakSample = seg[0].getPeakSample(timeSpan.getStart(), timeSpan.getEnd());
      Sample tmpSample  = null;

      for (int i=1; i < seg.length; i++) {

          tmpSample = seg[i].getPeakSample(timeSpan.getStart(), timeSpan.getEnd());

          peakSample = Sample.absMax(peakSample, tmpSample);

          // past end of time window, bail
          if (seg[i].getEnd().doubleValue() > timeSpan.getEnd()) break;
    }

      return peakSample;
     } // end of synch
    }

/**
* Calculate the background noise level. It is
* the median of the peaks between zero-crossings for a rectified time-series
* with the bias removed for the whole time-series.
* Returns Float.NaN if it cannot be calculated.
*/
  public float scanForNoiseLevel () {

     return scanForNoiseLevel(getEpochStart(), getEpochEnd());
  }


/**
* Calculate the background noise level. Sets value of 'noiseLevel'. It is
* the median of the peaks between zero-crossings for a rectified time-series
* with the bias removed for the time window given.
* Returns Float.NaN if it cannot be calculated.
*/
  public float scanForNoiseLevel (TimeSpan ts) {
         return scanForNoiseLevel(ts.getStart(), ts.getEnd());
  }

/**
* Calculate the background noise level. Sets value of 'noiseLevel'. It is
* the median of the peaks between zero-crossings for a rectified time-series
* with the bias removed for the time window given.
* Returns Float.NaN if it cannot be calculated.
*/
  public float scanForNoiseLevel (double startTime, double endTime) {

      if (!hasTimeSeries()) {
         noiseLevel = Float.NaN;
         return noiseLevel;
      }	// no time series

    synchronized (this) {
      WFSegment seg[] = WFSegment.getArray(segList);

      float segNoise;
      noiseLevel = 0.0f;
      //Segments aren't equal in length :. weight the bias by sample count
      for (int i=0; i < seg.length; i++) {

          segNoise = seg[i].scanForNoiseLevel(startTime, endTime);

          if (segNoise != Float.NaN) {
           noiseLevel = Math.max(noiseLevel, segNoise);
          }

          // past end of time window, bail
          if (seg[i].getEnd().doubleValue() > endTime) break;
    }
    } // end of synch

      return noiseLevel;
  }

/** Return true if the Waveform has time tears.
* Because waveforms are ALWAYS collapsed, > 1 segment means there are time tears.
*/
     public boolean hasTimeTears() {

     if (segList.size() > 1) {
        return true;
     } else {
        return false;
     }

    }

/**
 * Return the bias of this waveform.
 */
  public float getBias ()
  {
      return biasVal;
  }

/**
* Scans the given time window for a peak amplitude. Used to limit the scan for
* efficiency and to avoid energy from other events. If there is no time-series
* null is returned.
*/
  public Amplitude getPeakAmplitude (double startTime, double endTime) {
            return getPeakAmplitude(new TimeSpan(startTime, endTime));
  }
/**
* Scans the whole time series for a peak amplitude. If there is no time-series
* null is returned.
*/
  public Amplitude getPeakAmplitude () {
            return getPeakAmplitude(getTimeSpan());
  }
/**
* Scans the given time window for a peak amplitude. Used to limit the scan for
* efficiency and to avoid energy from other events. If there is no time-series
* null is returned.
*/
  public Amplitude getPeakAmplitude (TimeSpan timeSpan) {

     // get the peak
     Sample peak = scanForPeak(timeSpan);

     if (peak == null) {

        return null;

     } else {

        Amplitude amp = Amplitude.create();

        // zero-to-peak or peak-to-peak ?
        amp.halfAmp = true;

        // set the channel, value, type & time of the amp object
        amp.set(this, peak, AmpType.WAU, timeSpan);

        return amp;

     }

  }

/**
 * Find the minimum and maximum amplitudes of this waveform by examining all
 * waveform segments.
 */
  public void scanForAmps () {

     if (!hasTimeSeries()) return;	// no time series

     synchronized (this) {
      WFSegment seg[] = WFSegment.getArray(segList);

      maxSample = seg[0].getMaxSample();
      minSample = seg[0].getMinSample();

      for (int i=1; i < seg.length; i++) {
          if (seg[i].getMaxSample() != null &&
              seg[i].getMaxSample().value > maxSample.value)
                   maxSample = seg[i].getMaxSample();
          if (seg[i].getMinSample() != null &&
              seg[i].getMinSample().value < minSample.value)
                   minSample = seg[i].getMinSample();
      }
     } // end of synch
 }

/**
 * Return the Sample with the maximum amplitude of this waveform.
 * Return's null if no time-series. NOTE: bias is NOT removed.
 */
  public Sample getMaxSample () {
    if (maxSample == null) scanForAmps();
    return maxSample;
  }
/**
 * Return the maximum amplitude value of this waveform.
 * Return's 0 if no time-series.  NOTE: bias is NOT removed.
 */
  public double getMaxAmp () {
    if (maxSample == null) return 0;
    return maxSample.value;
  }

/**
 * Return the Sample with the minimum amplitude of this waveform.
 * Return's null if no time-series.  NOTE: bias is NOT removed.
 */
  public Sample getMinSample () {
    if (minSample == null) scanForAmps();
    return minSample;
  }
/**
 * Return the minimum amplitude of this waveform.
 * Returns 0 if no time-series.   NOTE: bias is NOT removed.
 */
  public double getMinAmp () {
    if (minSample == null) return 0;
    return minSample.value;
  }
/**
 * Return the total range of amplitudes in the trace
 */
  public double rangeAmp() {
    return getMaxAmp() - getMinAmp();
  }
/**
 * Return a Sample with the maximum <b>rectified</b> amplitude of this waveform with
 * the bias removed. Thus the Sample.value will always be positive.
 * Return's null if no time-series.
 */
  public Sample getPeakSample () {
    // bias is always subtracted (removed)
    double max = getMaxSample().value - getBias();
    double min = getMinSample().value - getBias();

    max = Math.abs(max);
    min = Math.abs(min);
    // compare absolute values to determine greatest deviation from mean
    if (min > max) {
      return new Sample(getMinSample().datetime, min);
    } else {
      return new Sample(getMaxSample().datetime, max);
    }
  }
    /**
     * Return string describing the Waveform object, primarily for debugging.<p>
     * Example:<br>
     * <tt>
   Waveform: id = 13356047, channel = CI AGA HLZ, timeStart = 2001-05-16 19:00:36.175 GMT+00:00, timeEnd = 2001-05-16 19:01:21.385 GMT+00:00
   samplesPerSecond  = 100.0, dataMode = T, bytes of data = 5632, samplesInMemory = 0, timeQual = NaN, format = 2
   savingEvent = 9653085, path/file = /archive/trig/2001/200105/9653085
     * </tt>
     */
    public String toString() {

      String idStr = "null";
      if (!id.isNull()) idStr =  id.toString();
  return
          "   Waveform: id = "  + idStr +
      ", channel = "	  + chan.toDelimitedSeedString() +

      ", timeStart = "	  + EpochTime.toString(timeStart.doubleValue()) +
      ", timeEnd = "	  + EpochTime.toString(timeEnd.doubleValue()) + "\n"+

      "   samplesPerSecond  = " +samplesPerSecond.toString()  +
      ", dataMode = "	  + dataMode.toString() +
      ", bytes of data = "  + byteCount +
      ", samplesInMemory = "+ samplesInMemory() +
      ", timeQual = "+ getTimeQuality().doubleValue() +
      //	    ", sampleCount = "	  + sampleCount.toString() +
      ", format = "	  + format.toString() + "\n"+
      "   savingEvent = "	  + savingEvent.toString() +
      ", path/file = "	  + path+filename ;

    }
    /**
     * Return string describing the Waveform object, primarily for debugging.<p>
     * Example:
<tt>
Waveform: id = 13356047
channel = CI AGA HLZ
timeStart = May 16, 2001 19:00:36.175
timeEnd   = May 16, 2001 19:01:21.385
samplesPerSecond  = 100.0
dataMode = T
encoding = 10
bytes of data = 5632
samplesInMemory = 0
timeQual = NaN
format = 2
savingEvent = 9653085
path/file = /archive/trig/2001/200105/9653085
<\tt>
     */
    public String toBlockString() {
      //final String fmtString = "MMMM dd, yyyy HH:mm:ss.SS";
      // "ss.SS" produces bad values
      final String fmtString = "MMMM dd, yyyy HH:mm:ss.SSS";

      String idStr = "null";
      if (!id.isNull()) idStr =  id.toString();
  return
          "Waveform: id = "  + idStr +
      "\nchannel = "	  + chan.toDelimitedSeedString() +

      "\ntimeStart = "	  + DateTime.toString(timeStart.doubleValue(), fmtString) +
      "\ntimeEnd   = "	  + DateTime.toString(timeEnd.doubleValue(), fmtString)+

      "\nsamplesPerSecond  = " +samplesPerSecond.toString()  +
      "\ndataMode = "	  + dataMode.toString() +
      "\nencoding = "       + SeedEncodingFormat.getTypeString(encoding.intValue())+
       " ("+encoding.toString()+")"+
      "\nbytes of data = "  + byteCount +
      "\nsamplesInMemory = "+ samplesInMemory() +
      "\ntimeQual = "       + getTimeQuality().doubleValue() +
      "\nformat = "	  + format.toString() +
      "\nsavingEvent = "	  + savingEvent.toString() +
      "\npath/file = "	  + path+filename ;

    }
     /** Set the travel-time model used to calculate the energy window. */
    public void setTravelTimeModel (TravelTime model) {
       ttModel = model;
    }
    /** Return the travel-time model used to calculate the energy window. */
    public TravelTime getTravelTimeModel () {
       return ttModel;
    }

     /** Return a TimeSpan representing a window where you are likely to find
    * the peak seismic energy.
    * The window begins 1 sec before the calculated P-wave onset and
    * ends at P-wave onset plus 2x the S-P time.<p>
    * Uses the standart TraveTime model.
    */
    public TimeSpan getEnergyTimeSpan (double originTime, double distance) {

           // define the window
           double ttp = ttModel.getTTp(distance);
           double tts = ttModel.getTTs(distance);
           double energyStart = originTime + ttp - 1.0;
           double energyEnd   = energyStart + 2.0*(tts-ttp);

           return new TimeSpan (energyStart, energyEnd);
    }
     /** Return a TimeSpan representing a window before the seismic energy.
    * The window begins at the start of the waveform and ends 1 sec before the
    * calculated P-wave onset.
    * Uses the standart TraveTime model.
    */
    public TimeSpan getPreEnergyTimeSpan (double originTime, double distance) {

       return new TimeSpan (getEpochStart(),
                            getEnergyTimeSpan(originTime, distance).getStart());
    }

/**
 *  Return the sample closest to the given time. Times are raw epoch seconds.
 *  Null is returns if  there is no time series.
 *  This is used to force operator picks to lie on a sample. We actually scan the data
 *  rather than calculate because there may be a time-tear. If the time is
 *  equidistant between samples the earlier sample will be returned.
 *  Also we want to look up the amp for the sample .
 */
    public Sample closestSample(double dtime) {

    synchronized (this) {

  if (!hasTimeSeries()) return null; 	// no time series

  // before time range
  if (dtime < timeStart.doubleValue()) return getFirstSample();
  // after time range
  if (dtime > timeEnd.doubleValue())   return getLastSample();

  //	must be in the time range
  Sample samp;
  Sample closeSamp = null;
  double dtNew;
  double dtOld = Double.MAX_VALUE;

  WFSegment seg[] = WFSegment.getArray(segList);

  for (int i=0; i < seg.length; i++) {

      samp = seg[i].closestSample(dtime);

      // closer?
      dtNew = Math.abs((samp.datetime - dtime));
      if (dtNew < dtOld) {
    dtOld = dtNew;
    closeSamp = samp;
      }

  }

  return closeSamp;
    } // end of synch
    }


    /** Return the first sample in the Waveform */
    public Sample getFirstSample() {

  if (!hasTimeSeries()) { return null; }	// no time series

  return ((WFSegment) segList.get(0)).getFirstSample();
    }

    /** Return the last sample in the Waveform */
    public Sample getLastSample() {
  if (segList.size() == 0) { return null; }	// no time series

  return ((WFSegment) segList.get(segList.size()-1)).getLastSample();
    }

    /**
    * Return a collection of WFSegments that contains the time-series between
    * the given times. Only one WFSegment will result if there are no time-tears.
    * If either the start or end time is beyond what is available in the waveform
    * the returned WFSegments will be limited by the available data. Returns an
    * empty Collection if there is no data within the time window.
    */

    public Collection getSubSegments (double startTime, double endTime) {

     synchronized (this) {

      ArrayList newSegList = new ArrayList();

      if (!hasTimeSeries()) return newSegList;	// no time series

      WFSegment seg[] = WFSegment.getArray(segList);

      WFSegment newSeg;
      for (int i=0; i < seg.length; i++) {
          newSeg = seg[i].getSubSegment(startTime, endTime);
          if (newSeg.size() > 0) {
             newSegList.add(newSeg);
          }
      }

      return newSegList;
     } // end of synch
    }


} // end of class
