package org.trinet.jasi.TN;

import org.trinet.jasi.*;

import java.lang.*;
import java.io.File;
import java.util.*;
import java.sql.*;

import org.trinet.jdbc.ResultSetDb;
import org.trinet.jdbc.datatypes.*;
import org.trinet.jasi.seed.*;

import org.trinet.util.TimeSpan;
import org.trinet.util.EpochTime;

import org.trinet.util.WaveClient;


/**
 * Represents the waveform time series for one channel for a time window (TimeSpan).
 * Because a time window may contain gaps or no data, the TimeSeries may contain
 * from zero to 'n' WFSegments. Each WFSegment is a CONTIGUOUS time series.<p>
 *
 * Fires ChangeEvents when the time-series is loaded so that time-series can be
 * loaded asynchronously (in background thread) then notify a graphics object when
 * loading is complete.<p>
 *
 * Note that a waveform may be "empty", that is it contain no time series. This may be
 * true before the loader thread has retreived the time-series. Check this
 * condition using the boolean method hasTimeSeries().<p>
 *
 * This class also knows how to read the waveforms from various data sources.
 * @see: loadTimeSeries().
 * <p>
 */
/*
1/15/02 Integrated AWW's DbBlobReader
*/
public class WaveformTN extends Waveform {

    static final String SelectJoinString =
  "SELECT "+
  org.trinet.jdbc.table.Waveform.QUALIFIED_COLUMN_NAMES+","+
  " AssocWaE.evid, "+
  " WavefilePath(Waveform.wfid), WavefileDfile(Waveform.wfid) "+
  " FROM Waveform, AssocWaE WHERE Waveform.wfid = AssocWaE.wfid  ";

// //////////////////////////////////////////////////////////////////
    /**
     * Empty constructor
     */
    public WaveformTN() {}
/**
 * Get an array of waveform headers given an SQL query.
 */
    protected static Collection getBySQL(String sql)
    {
  return getBySQL(DataSource.getConnection(), sql);
    }

/**
 * Get an array of waveform headers given an SQL query.
 */
    protected static Collection getBySQL(Connection conn, String sql)  {

  ArrayList wfList = new ArrayList();


  // Debug
  //	System.out.println ("SQL: "+sql);

  try {
      if ( conn.isClosed() )	// check that valid connection exists
    {
        System.err.println ("* Connection is closed");
        return null;
    }

      Statement sm = conn.createStatement();

      //	    ResultSetDb rsdb =
      //		new ResultSetDb(org.trinet.jdbc.table.ExecuteSQL.rowQuery(sm, sql));
      ResultSet rs = org.trinet.jdbc.table.ExecuteSQL.rowQuery(sm, sql);

      if (rs == null) return null;	// nothing found

      // parse and add rows one-at-a-time
      while ( rs.next() ) {

    wfList.add(parseResultSet(rs));
      }

      sm.close();

  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();
  }

  // convert vector to array
  //	if (wfList.size() == 0) return null;

  return (Collection) wfList;
    }

    /**
     * Parse a resultset row that contains the results of a join + some
     * stored procedures
     */
//    static Waveform parseResultSet (ResultSetDb rsdb)
     static Waveform parseResultSet (ResultSet rs) {
  // a jdbc.table Waveform
  org.trinet.jdbc.table.Waveform wfrow = new org.trinet.jdbc.table.Waveform();

  // a jsai Waveform
  WaveformTN wf = new WaveformTN();

  int offset = 0;

  // parse the first part of the return that is just a Waveform table row
  ResultSetDb rsdb = new ResultSetDb(rs);
  wfrow = (org.trinet.jdbc.table.Waveform) wfrow.parseOneRow(rsdb, offset);

  // transfer row contents to this Waveform
  wf.parseWaveformRow(wfrow);

  // Now, pick up the "extra" columns specified by the SQL query.
  offset += wfrow.MAX_FIELDS;

  try {
      // <AssocWaE.evid>
      //  event ID for which this event was saved
      int oid =  rs.getInt(++offset);   // get as int
      String str = "" + oid;
      wf.savingEvent.setValue(str);	// save as string

      // <WavefilePath(AssocWaE.evid)>
      // time-series data file PATH from stored procedure
      wf.path  = rs.getString(++offset);

      // <WavefileDfile(Waveform.wfid)>
      // time-series data filename from stored procedure
      wf.filename = rs.getString(++offset);
  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();
  }

  // keep a copy of the row
 //	wf.setWaveformRow(wfrow);

  return wf;
    }
    /**
     * Transfer the needed DataObject's from the Waveform row to this Waveform.
     */

    // Note that this gets DataObjects not values.
    // Note the use of 'wfr.EVID' for static constants rather then
    //    'org.trinet.jdbc.table.Waveform.EVID' for brevity.

    private Waveform parseWaveformRow(org.trinet.jdbc.table.Waveform wfr) {

  int i = 0;

  this.id = (DataLong) wfr.getDataObject(wfr.WFID);

  // suck out the Channel description
//	this.chan.parseNameFromDataTableRow(wfr);
  setChannelObj(ChannelTN.parseNameFromDataTableRow(wfr));

     // Should we do DateTime.trimToNanos(timeValue) here?
  this.timeStart	= (DataDouble) wfr.getDataObject(wfr.DATETIME_ON);
  this.timeEnd   = (DataDouble) wfr.getDataObject(wfr.DATETIME_OFF);

  this.samplesPerSecond=(DataDouble) wfr.getDataObject(wfr.SAMPRATE);
  this.dataMode	= (DataString) wfr.getDataObject(wfr.WAVETYPE);

  // sample count is unknown
  this.format	= (DataLong)   wfr.getDataObject(wfr.WAVE_FMT);
  this.encoding	= (DataLong)   wfr.getDataObject(wfr.FORMAT_ID);

  this.savingEvent= (DataString) wfr.getDataObject(wfr.LOCEVID);

  // file access stuff
  this.fileOffset = (int) wfr.getDataObject(wfr.FOFF).longValue();
  this.byteCount  = (int) wfr.getDataObject(wfr.NBYTES).longValue();
  //	this.fileOffset = wfr.getLong(wfr.FOFF);
  //	this.byteCount  = wfr.getLong(wfr.NBYTES);
  return this;

    }


/**
 * Get one waveform from the DataSource by Waveform record ID #
 * Returns empty Waveform object if no match is found.
 */
    public Waveform getByWaveformId (long wfid) {
  String sql = SelectJoinString +
      " and AssocWaE.wfid = " + wfid;

  ArrayList wf = (ArrayList) getBySQL(sql);

     if (wf.size() == 0) {
       return Waveform.create();
     } else {
    return (Waveform) wf.get(0);	// only one
     }

    }

/**
 * Get an array of waveforms from the DataSource by solution ID #
 */
    public Collection getBySolutionId (long id) {
  String sql = SelectJoinString +
      " and AssocWaE.evid = " + id;

  return getBySQL(sql);
    }

/**
 * Get an array of waveforms from the DataSource by solution ID #
 */
    public Collection getBySolution (Solution sol) {
  String sql = SelectJoinString +
      " and AssocWaE.evid = " + sol.id.toString();

  return getBySQL(sql);
    }

    /**
     * Create AssocWaE record for existing WaveForm and pointing at this Solution.
     * The underlying stored procedure commits, so you don't have to.
     * */

    public boolean commit (Solution sol) {

  String sql = "call jasi.AssociateWaveform("+
      id.longValue() + ", "+ sol.id.longValue()+ ")";

      return doSql(sql);

    }

/** Execute the sql string, return false on error */
protected boolean doSql (String sql) {

    try {
  Statement sm = DataSource.getConnection().createStatement();

  sm.execute(sql);
    } catch (SQLException ex) {
  System.out.println ("SQL: "+sql);
  System.err.println(ex);
  ex.printStackTrace();
  return false;
    }

    return true;

}

    /**
     * Gets generic waveform file root from the database. Needs a
     * JDBConn open connection. Note that this is an over simplification because
     * there may be multiple wavefile roots.
     */
    protected String getFileRoot() {

  String root = "";

  // get the data directory root from the data base
  String sql = "select wavefileroot() from dual";

  try {
      Statement sm = DataSource.getConnection().createStatement();
      ResultSet rs = sm.executeQuery(sql);
      rs.next();
      root  = rs.getString(1);
      sm.close();
  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();
  }

  return root;
    }
    /**
     * Return the waveform file root for a particular event id.
     * Access to this path is tested
     * to determine if files can be read locally.
     */
    protected String getFileRoot(long eventId) {

  String root = "";

  // get the data directory root from the data base
  String sql = "select wavefileRootByEvid(" + eventId + ") from dual";

  try {
      Statement sm = DataSource.getConnection().createStatement();
      ResultSet rs = sm.executeQuery(sql);
      rs.next();
      root  = rs.getString(1);
      sm.close();
  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();
  }

  return root;
    }
    /**
     * Returns 'true' if this waveform's files are local,
     * i.e. are on a file system where they can be opened either on the client
     * machine or NFS mounted.
     * Needs a DataSource open connection.
     */
    public boolean fileIsLocal() {

  // check existance
  File file = new File(getPathFilename());

  System.out.println ("file: "+file.toString());

  return file.exists();
    }

    /** Returns true if this Waveform is accessible through some transport
     *  method. */
/*  DEPRECATED
    public boolean isAccessible () {

       if (filesAreLocal()) return true;

       // try remote reader
       // NOTE: we assume the waveforms are reachable via FTP at the
       // same IP address as the DataSource
       String remoteHost = DataSource.getIPAddress();  // default data source;
       RemoteFileReader remoteReader = new RemoteFileReader(remoteHost);
       return remoteReader.isConnected();
    }
*/
/**
 * Load the time series data for this Waveform. This is done as a separate step
 * because it is often more efficient to only load selected waveforms after
 * you have examined or sorted the Waveform objects by their header info.<p>
 *
 * The waveform is collapsed into the minimum number of WFSegments possible. More
 * then one WFSegment would only be necessary if there are time tears. The start-stop
 * times of the original data packets is preserved in the packetSpans array.<p>
 *
 * Fires ChangeEvents when the time-series is loaded so that time-series can be
 * loaded asynchronously (in background thread) then notify a graphics object when
 * loading is complete.<p>
 *
 */
// Should method be synchronized so only one class will try to load waveform at a time?
//    public synchronized boolean loadTimeSeries() {
    public boolean loadTimeSeries() {
       return loadTimeSeries(getStart().doubleValue(), getEnd().doubleValue());
    }
    public boolean loadTimeSeries(double startTime, double endTime) {
      // debug
//      System.out.println("Loading: "+this.toString());

      if (hasTimeSeries() && (getStart().doubleValue() == startTime) &&
     (getEnd().doubleValue() == endTime) ) return true;
     // Note that synchronizing the Method does NOT provide thread safty because
     // other threads can modify the Waveform via other methods.
//     synchronized (this) {      // synchronize for thread safty
   // only SEED is currently implemented
   if (format.intValue() == SEED_FORMAT) {
            // read data from Seed file
            SeedReader.getData(this, startTime, endTime);
      // came up empty
      if (!hasTimeSeries()) return false;
            // concatinate contiguous segments
            collapse();
      // set max, min, bias values
      scanForAmps();
      scanForBias();
      // Notify any listeners that Waveform state has changed
      // (changeListener logic is in jasiObject())
      fireStateChanged();
      return true;	// success
   }
//     } // end of synch
  return false;
    }

    /**
     * Override Waveform.filesAreLocal() to always return false so dbase access of waveforms
     * will be used.
     */
    public boolean filesAreLocal()
    {
  return false;
    }

/**
 * Main for testing
 * NOTE!!! to run main from within a package use:
 *
 * java org.trinet.jdbc.Waveform
 */
    public static void main (String args[]) {

        System.out.println ("Making connection...");
  //DataSource ds = new TNDataSource("makalu");
  //DataSource ds = new TNDataSource("k2");
  DataSource ds = new TestDataSource("iron");

        int evid = 701559;

// TEST 0 -- test getting wiggle for latest waveform

//	int wfid = 82565122;        // 100 sps
  //int wfid = 13356047;        // 10 sec/sample DAN.VHZ
        //int wfid = 93588568;             // 1983, 50 sps converted CUSP data
        int wfid = 99403734;             // 1981, 50 sps converted CUSP data

  Waveform wfx = Waveform.create();
  wfx = wfx.getByWaveformId(wfid);

        System.err.println (wfx.toString());
       System.err.println (wfx.toBlockString());

  System.err.println ("Fetching waveform using DataSource: "+wfx.getChannelObj().toString());

  // force filename
//	    wfx.setFilename("2046133");
//	    wfx.setPath("/wavearchive2/trig/1992/199204/");
//	wfx.setFilename("701559");
//	wfx.setPath("/home/awwalter/");
        System.err.println (wfx.toString());

  if (wfx.loadTimeSeries()) {
      System.out.println ("Got waveform: nsegs =  "+ wfx.getSegmentList().size() );
      //	    System.out.println ("Expected samps = "+ wfx.sampleCount.toString() );
      System.out.println ("Actual samps   = "+ wfx.samplesInMemory() );
      System.out.println ("max = "+ wfx.getMaxAmp());
      System.out.println ("min = "+ wfx.getMinAmp());
      System.out.println ("bias= "+ wfx.getBias());

  } else {
      System.out.println ("Waveform retreival failed.");
  }

         // test noise scan

     wfx.scanForNoiseLevel();

     // Test subsetting of waveform
     if(wfx.getSegmentList().size() > 0)
     {
       WFSegment[] seglist = (WFSegment[])(wfx.getSegmentList().
                               toArray(new WFSegment[wfx.getSegmentList().size()]));
       for (int i = 0; i < wfx.getSegmentList().size(); i++) {
             WFSegment seg = seglist[i];
             System.out.println ( seg.toString() );
       }
     }

     TimeSpan ts = wfx.getTimeSpan();
     double start = ts.getStart() + 10.0;
     double end   = ts.getEnd() - 10.0;


     wfx.scanForNoiseLevel(start, end);

     ArrayList segList = (ArrayList) wfx.getSubSegments(start, end);
    System.out.println ("Trimmed waveform: nsegs =  "+ segList.size() );

          for (int i = 0; i < segList.size(); i++) {
              WFSegment seg = (WFSegment) segList.get(i);
             System.out.println ( seg.toString() );
          }

  // test wave client reader

  WaveClient waveClient = null;

  if (false) {
      // cheating
      String propFile = "/home/tpp/src/waveserver/rt/WaveClient.properties";

      try {
    // Make a WaveClient

    System.out.println ("Creating WaveClient using: "+propFile);

    waveClient = WaveClient.CreateWaveClient().ConfigureWaveClient(propFile); // property file name

    System.out.println (waveClient.toString());

    int nservers = waveClient.numberOfServers();
    if (nservers <= 0) {
        System.err.println("getDataFromWaveServer Error:"+
           " no data servers specified in input file: " +
           propFile);
        System.exit(-1);
    }

      }
      catch (Exception ex) {
    System.err.println(ex.toString());
    ex.printStackTrace();
      }
      finally {
    //	if (waveClient != null) waveClient.close();
      }

      // Get all time-series from a wave server
      Waveform.setWaveSource (waveClient);

      System.out.println ("***** Attempting to load time series from WaveServer *******");

      wfx.unloadTimeSeries();
      boolean status = wfx.loadTimeSeries();

      System.out.println ("Status = "+status);
  }


  if (false) {

// TEST 1
      WaveformTN wf = new WaveformTN();
         wf = (WaveformTN)wf.getByWaveformId(wfid);

      System.out.println ("filesAreLocal ="+ wf.filesAreLocal() );

      System.err.println (" One by wfid...");
      System.err.println (wf.toString());
      System.err.println ("Got waveform: nsegs = "+ wf.segList.size() );
  }

  if (false) {
// TEST 2 -- test retreival by evid
      System.err.println (" Many by evid...");

      System.err.println ("Fetching waveforms pointer info...");

      ArrayList wfarr = (ArrayList) Waveform.create().getBySolutionId(evid);
      Waveform wfa[] = new Waveform[wfarr.size()];
      wfarr.toArray(wfa);

      //	Waveform wfa[] = (Waveform[]) wfarr.toArray(new Waveform[0]);

      System.err.println ("Waveform count = " + wfa.length);

      for (int i = 0; i<wfa.length; i++)
    {
        System.err.println (wfa[i].toString());
    }
  }

    }

} // end of class
