package org.trinet.jasi;

import java.sql.Connection;
import java.text.*;
import java.util.*;
import org.trinet.jasi.coda.*;
import org.trinet.jdbc.datatypes.*;	// for DataObjects

import org.trinet.util.Format;		// CoreJava printf-like Format class

/**
 * Description of a calculated magnitude associated with a Solution. <p> This is
 * a component of the Java Seismic Abstraction Interface (jasi) which allows
 * applications to interact with different schemas tranparently. <p> All data
 * members are DataObjects rather than primative types so they can handle the
 * concept of "nullness" because databases can contain null values for certain
 * fields. Also, applications may not want to write a "bogus" value like 0 into
 * the database when the real value is not known. Data primatives can't be null
 * and are initiallize to zero.<p>
 *
 * @See: JasiObject */

public abstract class Magnitude extends JasiObject implements Cloneable
{
    /*
      Here's the list of public data members for a Magnitude. These need to be
      supported by any port of this abstract layer
     */

    /** A unique ID number for this magnitude. May be null. May be a dbase key. */
    public DataLong    magid		= new DataLong();

    /** The value of the magnitude */
    public DataDouble  value		= new DataDouble();
    /** The subScript of magnitude. Includes only the suffix part of the type without
     * "M".  For example: for an "Ms" subScript="s".
     * @See: MagnitudeMethod */
    public DataString  subScript		= new DataString();

    /** Data authority. The 2-character FDSN network code from which this
        reading came. Default to network code set in EnvironmentInfo.
        @See: EnvironmentInfo.getNetworkCode */
    public DataString  authority  = new DataString(EnvironmentInfo.getNetworkCode());

    /** Data source string. Optional site defined source string.
        Default to application name set in EnvironmentInfo.
        @See: EnvironmentInfo.getApplicationName */
    public DataString  source	  = new DataString(EnvironmentInfo.getApplicationName());

    /** The method used to calculate the magnitude. May be a program name or
        some other string that is meaningful to the underlying DataSource. */
    public DataString  method		= new DataString();

    /** Number of stations that contributed to this magnitude */
    /* Forced to do this rather than use getStationsUsed() because the RT system
    * does not give a weight to used amps. */
    public DataLong    usedStations	= new DataLong();

    /** The standard deviation of the magnitude */
    public DataDouble  error		= new DataDouble();
    /** Largest azmuthal gap in stations contributing to the magnitude */
    public DataDouble  gap		= new DataDouble();
    /** Distance in km to the nearest station contributing to the magnitude */
    public DataDouble  distance		= new DataDouble();
    /** Estimate of magnitude quality 0.0<=quality<=1.0, 1.0 is best */
    public DataDouble  quality		= new DataDouble();

    /** State of processing. 'A'=automatic, 'H'=human */
    public DataString  processingState	=
           new DataString(EnvironmentInfo.getAutoString());	// rflag

    /** List of Codas associated with this Magnitude */
    public CodaList codaList = new CodaList();  // AWW

    /** List of Amplitudes associated with this Magnitude */
    public AmpList ampList = new AmpList();

    /** Solution with which this magnitude is associated. */
    public Solution sol = null;

    /** Set true if this mag is virtually deleted. */
    boolean deleteFlag = false;

    /** True if amps list has changed which makes the magnitude invalid or stale. */
    boolean isStale = false;

// //////////////////////////////////////////////////////////////////////

// -- Concrete FACTORY METHODS ---
    /**
     * Factory Method: This is how you instantiate a magnitude object. You do
     * NOT use a constructor. This is so that this "factory" method can create
     * the type of magnitude that is appropriate for your site's database.
     */
// ////////////////////////////////////////////////////////////////////////////
    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. Creates a Solution of the DEFAULT type.
     * @See: JasiObject
     */
     public static final Magnitude create() {
	return create(DEFAULT);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is an integer implementation type.
     * @See: JasiObject
     */
     public static final Magnitude create(int schemaType) {
        return create(JasiFactory.suffix[schemaType]);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is as 2-char implementation suffix.
     * @See: JasiObject
     */
     public static final Magnitude create(String suffix) {
	return (Magnitude) JasiObject.newInstance("org.trinet.jasi.Magnitude", suffix);
     }
// ////////////////////////////////////////////////////////////////////////////
/** Clone the magnitude. Note that the codaList and ampList are not deep copied.*/
    public Object clone() {
        Magnitude mag = null;
        try {
            mag = (Magnitude) super.clone();
        }
        catch (CloneNotSupportedException ex) {
            ex.printStackTrace();
        }
        mag.magid        = (DataLong) magid.clone();
        mag.usedStations = (DataLong) usedStations.clone();
	mag.value        = (DataDouble) value.clone();
	mag.error        = (DataDouble) error.clone();
	mag.gap          = (DataDouble) gap.clone();
	mag.distance     = (DataDouble) distance.clone();
	mag.quality      = (DataDouble) quality.clone();

        mag.subScript       = (DataString) subScript.clone();
        mag.authority       = (DataString) authority.clone();
        mag.source          = (DataString) source.clone();
        mag.method          = (DataString) method.clone();
        mag.processingState = (DataString) processingState.clone();

// NOTE: these are not deep copied!
	mag.codaList = codaList;
	mag.ampList  = ampList;

        return mag;
    }

   /**
     * Returns the Magnitude for the event with this ID number from the data source.
     * Returns null if no mag is found.<p>
     * Uses the default DataSource.
     */
    abstract public Magnitude getBySolutionId (long id);

    /**
     * Returns the Magnitude for the event with this ID number from the data source.
     * Returns null if no mag is found.
     */
    abstract public Magnitude getBySolutionId (Connection conn, long id);

   /**
     * Returns ALL the Magnitudes associate with the event with this ID number
     * from the data source. This will return a Collection of alternate Magnitude
     * objects and will NOT contain the preferred mag.
     * Returns null if no mags are found.<p>
     * Uses the default DataSource.
     */
    abstract public Collection getAltBySolutionId (long id);

   /**
     * Returns ALL the Magnitudes associate with the event with this ID number
     * from the data source. Ust the passed connection to connect to the datasource.
     * This will return a Collection of alternate Magnitude
     * objects and will NOT contain the preferred mag.
     * Returns null if no mags are found.<p>
     * Uses the default DataSource.
     */
    abstract public Collection getAltBySolutionId (Connection conn, long id);

    /** Commit any changes or deletions to the data source */
    abstract public boolean commit() throws JasiCommitException;

    /**
     * Marks this object for delete from the data source.
     * The object is not actually deleted until commit() is called.
     */
    public boolean delete() {

	deleteFlag = true;

	return true;
    }

    /**
     * Return true if this magnitude has been virtually deleted.
     */
    public boolean isDeleted() {

	return deleteFlag;
    }

    /**
     *
     */
    public String toString()
    {
	return value.toString() + " "+getTypeString() ;
    }
    /**
     * Associate this reading with this Solution.
     */
    public void associate(Solution sol)
    {
	this.sol = sol;
    }

    /**
     * Unassociate, that is, set Solution to null.
     */
    public void unassociate()
    {
	this.sol = null;
        // shouldn't the codaList, ampList sol association for same solution be nulled AWW?
    }

    /** Return the associated solution. Returns null if unassociated. */
    public Solution getAssociatedSolution () {
	return sol;
    }

    /** Return 'true' if this is associated with a Solution */
    public boolean isAssociated()
    {
	return (sol != null) ;
    }

    /** Return 'true' if this is associated with this Solution */
    public boolean isAssociatedWith(Solution compSol)
    {
	return (sol == compSol) ;
    }

    public boolean isCodaMag() {

      if (subScript.isNull()) return false;

      return (subScript.toString().equalsIgnoreCase("c") ||
              subScript.toString().equalsIgnoreCase("ca")  ||
              subScript.toString().equalsIgnoreCase("d")  ||   // DK3 Added "Md" to list of CodaMags
              subScript.toString().equalsIgnoreCase("cd") );

    }

/**
 *  Add any amps that are associated in the DataSource to this
 *  Magnitude's ampList.  Note that references are used, the amps are
 *  not copied.  Returns a count of the number of amps in ampList. Sets
 *  staleMagnitude 'true' if any are added. */
    public int fetchAmps() {

	ampList = Amplitude.create().getByMagnitude(this);

	setStale(true);

	return ampList.size();

    }
/**
 * Given a Collection of Amps, add any that are associated with this
 *  Solution to its ampList.  Note that references are used, the amps are
 *  not copied.  Returns a count of the number that were added. Sets
 *  staleMagnitude 'true' if any are added */
    public int addAmps(Collection list) {
	Amplitude amp[] = new Amplitude[list.size()];
	list.toArray(amp);

	int knt = 0;

	for (int i = 0; i<amp.length; i++) {
	    if (addAmp(amp[i])) knt++  ;    // only adds if its for this Sol

	}
	return knt;
    }
    /**
     * Associates the amp with this magnitude.
     * deprecated: use associate()
     */
    boolean addAmp(Amplitude amp) {
        return associate(amp);
    }
    /**
     * Add an amplitude to this mag's list. Also associates the amp with this
     * magnitude.
     */
    public boolean associate(Amplitude amp) {

      if (ampList.add(amp)) {     // add to the mag's list
	  amp.magnitude = this;      // using amp.associateMag() would be recursive
       amp.associate(sol);        // add to the Sol's list

       setStale(true);

       return true;
      }

      return false;

    }
    /**
     * Remove this amp from the ampList. Returns 'true' if the amp was in the
     * list to start with. NOTE: this should NOT be confused with Amp.delete()
     * which marks a amp for deletion from the data source. Sets staleLocation
     * 'true' if amps are removed. */

    public boolean removeAmp(Amplitude amp)
    {
	if (ampList.remove(amp) ) {
	    amp.unassociateMag();
	    setStale(true);
	    return true;
	}

	return false;
    }

/**
 *  Add any codas that are associated in the DataSource to this
 *  Magnitude's coda list.  Note that references are used, the codas are
 *  not copied.  Returns a count of the number that were added. Sets
 *  staleMagnitude 'true' if any are added */
    public int getCodas()
    {
	return addCodas(Coda.create().getByMagnitude(this));

    }
/**
 * Given a Collection of Codas, add to codaList if associated Solution and Magnitude are the same.
 * Note that references are used, the codas are
 * not copied.  Returns a count of the number that were added. Sets
 * staleMagnitude 'true' if any are added */
    public int addCodas(Collection list)
    {
	Coda coda[] = new Coda[list.size()];
	list.toArray(coda);
	int knt = 0;
	for (int i = 0; i<coda.length; i++) {
	    if (coda[i].isAssociatedWith(this.sol)) {    // only adds if same solution assoc
                if (addCoda(coda[i])) knt++ ;            // only adds if same magnitude assoc
            }
	}
	return knt;
    }

    /**
     * Add an coda to this mag's list if coda associated with this magnitude.
     */
    public boolean addCoda(Coda coda) {
        if (codaList.contains(coda)) {
            System.out.println ("Magnitude: List already contains: "+coda.toString());
            return false;
        }
        if (coda.getAssociatedMag() == this) {
            codaList.add(coda);
            setStale(true);
            return true;
        }
        else return false;
    }

    /**
     * Remove this coda from the codaList. Returns 'true' if the coda was in the
     * list to start with. NOTE: this should NOT be confused with Coda.delete()
     * which marks a coda for deletion from the data source. Sets staleLocation
     * 'true' if codas are removed. */

    public boolean removeCoda(Coda coda)
    {
	if (codaList.remove(coda) ) {
	    coda.unassociateMag();
	    setStale(true);
	    return true;
	}
	return false;
    }

    /**
     * Return the number of amps associated with this magnitude
     */
    public int getCodaCount() {
	return codaList.size();
    }
    /** Return true if there are readings (amps or codas) connected to this
     *  magnitude. */
    public boolean hasReadings() {
       return (getAmpCount() > 0 || getCodaCount() > 0);
    }
    /** Return count of channels (not stations) used in summary. */
    public int getReadingsUsed() {
       return ampList.getChannelUsedCount();
    }
    /** Return count of stations (not channels) used in summary. */
    /* Because the damn RT system doesn't set weights = 1.0 we can't
    * count them and must read the 'nsta' value into usedStations. */
    public int getStationsUsed() {

       if (usedStations.isNull()) usedStations.setValue(ampList.getStationUsedCount());
       return usedStations.intValue();
    }

    /** Set value of isStale flag. */
    public void setStale (boolean tf) { isStale = tf; }

    /** True if changes have been made to the amp list and a recalc is
    needed.  */
    public boolean hasStaleMagnitude () { return isStale; };
    public boolean isStale () { return isStale; };

    /** Return true if this is the associated Solution's preferred magnitude. */
    public boolean isPreferred()  {
       return (sol.magnitude == this);
    }

    public boolean isNull() { return value.isNull(); }
    /**
     * Return true if any Magnitude field is different from what's in the DataSource.
     * Either, 1) its been changed and not saved or 2) it is newly created and not saved.
     */
     abstract public boolean hasChanged () ;
    /**
     * Return the number of amps associated with this magnitude
     */
    public int getAmpCount() {
	return ampList.size();
    }
    /**
     * Dump all the data members of this class.
     */
    public String toDumpString() {

	//Format df1 = new Format("%7.2f");
	Format df2 = new Format("%5.2f");

     String id;
     if (magid.isNull()) {
        id = "-none-";
     } else {
        id = magid.toString();
     }
	return "id= "+ id + " " +
	    "value= "+ df2.form(value.floatValue()) +" " +
	    "type = "+ getTypeString() +" " +
	    "authority= "+ authority.toString() +" "+
	    "source= "+ source.toString() +" "+
	    "method= "+ method.toString() +" "+
	    "usedAmps= "+ getReadingsUsed()+" "+
	    "usedStas= "+ getStationsUsed()+" "+
	    "error= "+ df2.form(error.floatValue()) +" "+
	    "gap= "+ df2.form(gap.floatValue()) +" "+
	    "distance= "+ df2.form(distance.floatValue()) +" "+
	    "quality= "+ quality.toString() +" "+
	    "state= "+ processingState.toString() ;

    }
    /*
      Return a fixed format string of the form:
<tt>
  Event ID     mag     type   source   method #sta   error     gap    dist    qual
dddddddddd ffff.ff ssssssss ssssssss ssssssss dddd ffff.ff ffff.ff ffff.ff ffff.ff
</tt>
You must call getNeatStringHeader() to get the header line shown above.

@see getNeatStringHeader()
    */
    public String toNeatString() {

	Format df1 = new Format("%7.2f");
	Format df2 = new Format("%9.4f");
	Format df3 = new Format("%8s");
	Format df4 = new Format("%4d");

	String idStr ;
	if (sol == null || sol.id.isNull()) {
	    idStr = "  -none-  ";
	} else {
	    Format df0 = new Format("%10d");
	    idStr = df0.form(sol.id.longValue());
	}

	String meth = method.toString();
	if (meth.length() > 8) meth = meth.substring(0,8);

	return idStr + " " +
	    df1.form(value.floatValue()) + " " +
	    df3.form(getTypeString()) + " "+
	    df3.form(source.toString()) +" "+
	    df3.form(meth) +" "+
	    df4.form(getStationsUsed()) +" "+
	    df1.form(error.floatValue()) +" "+
	    df1.form(gap.floatValue()) +" "+
	    df1.form(distance.floatValue()) +" "+
	    df1.form(quality.floatValue());

    }
    /*
      Return a fixed format header to match output from toNeatString(). Has the form:
<tt>
  Event ID     mag     type   source   method #sta   error     gap    dist    qual
dddddddddd ffff.ff ssssssss ssssssss ssssssss dddd ffff.ff ffff.ff ffff.ff ffff.ff

</tt>

@see: toNeatString()
    */
    public static String getNeatStringHeader() {
	return "  Event ID     mag     type   source   method #sta   error     gap    dist    qual";
    }

    /**
     * Return 2 character Magnitude type descriptive string of the form: "Ml", "Mb", etc.
     * If the type is null returns "M "
     */
    public String getTypeString()
    {
	if (subScript.isNull() || subScript.toString().equals("NULL")) return "M ";
	return "M"+ subScript.toString() ;
    }

    /** Return a string with the magnitude and the list of amplitudes or codas.*/
    public String neatDump () {

	String str = this.getNeatStringHeader()+"\n";
	str += this.toNeatString();

     if (!ampList.isEmpty()) {
         str += "\n"+ ampList.toNeatString();
     }
     if (!codaList.isEmpty()) str += "\n" + codaList.toNeatString();

     return str;
    }

} // end of class


