package org.trinet.jasi;


import javax.swing.event.*;
import java.util.*;
import org.trinet.util.gazetteer.LatLonZ;

/*
 * ActiveList.java
 *
 *
 * @author Doug Given
 * @version
 */

/**
 * This class extends ActiveList in order to support the special delete()
 * methods of JasiReading objects. Some jasi objects are not destroyed when they
 * are deleted but rather set some sort of internal flag so that they will be
 * deleted from the data source upon commit. <p>
 * Uses the commit() method of jasiObjects to commit the whole list.
 *
 * @See: ActiveList
 */

public class JasiReadingList extends ChannelableList {
    /** The master ChannelList. If set, any reading added to this list will be
    * matched to this list and its Channel object replaced by a reference to
    * the matching Channel object in this list. If no match is found, the
    * original Channel object is retained. This is done so that the readings
    * get the full channel description and the distance. */
    ChannelList chanList;

    public JasiReadingList() {

    }

    public JasiReadingList (Collection col) {

	  addAll(col);
    }

// **** Pass-thru methods to notify listeners ******

    /**
     * Overrides ActiveArrayList.delete() method.
     * Do a virtual delete of the object and notify observers. Note that this
     * sets the deleteFlag of the object true but does not remove it from
     * the list. Use Remove for that. If you do remove an object from the list,
     * all reference to it is lost and the dbase entry for that object will not
     * be changed when commit is done. */
//    public boolean delete (JasiReading obj) {  // bad signature on override!
/*    public boolean delete (Object obj) {    // must be Object to override ActiveArrayList.Delete(Object)

	boolean result;

	if (obj != null) {
	    result = ((JasiReading)obj).delete();          // special JasiReading.delete()
         if (result) fireStateChanged(obj);
	} else {
	    result = false;
	}
	return result;
    } */
    public boolean delete (Object obj) {    // must be Object to override ActiveArrayList.Delete(Object)

	boolean result;

	if (obj != null) {
	    result = ((JasiReading)obj).delete();          // special JasiReading.delete()
         if (result) {
            this.remove(obj);
            fireStateChanged(obj);
         }
	} else {
	    result = false;
	}
	return result;
    }

    /** Return a subset of this list containing the JasiReadings that are from the
     * given Channel. */

    public Collection getAssociatedByChannel(Channel chan) {

	JasiReadingList newList = new JasiReadingList();

	Channelable ch[] = (Channelable[]) toArray(new Channelable[0]);
	for (int i = 0; i<size(); i++) {

	    if ( ch[i].getChannelObj().sameAs(chan) ) {
	       newList.add(ch[i]);
	    }
	}
	return newList;
    }

    /** Return a list containing
    * the JasiReadings that are associated
    * with this Solution and are not virtually deleted.
    * Favor use of Reading-specific returns:<br>
    * @see PhaseList.getAssociated()
    * @see: AmpList.getAssociated()
    */
    public JasiReadingList getAssociatedWithSol(Solution sol) {

	JasiReadingList newList = new JasiReadingList();

	JasiReading obj[] = (JasiReading[]) toArray(new JasiReading[0]);
	for (int i = 0; i<obj.length; i++) {

	    if (obj[i].isAssociatedWith(sol) && !obj[i].isDeleted() ) {
	       newList.add(obj[i]);
	    }
	}
	return newList;
    }

   /**
   * Commit changes or delete to underlying DataSource
   */
    public boolean commit() {

	JasiReading obj[] = (JasiReading[]) toArray(new JasiReading[0]);
	for (int i = 0; i<obj.length; i++) {
	    obj[i].commit();
	}
	return true;
    }

    /** Match this all channels with ones in this list. If a match is found
     * replace the reading's channel object with a reference to the one in the list.
     * This is used to match objects to a more complete description of a channel then
     * they probably have otherwise. Gives access to lat/lon/z, response info, calibration,
     * distance from epicenter, etc. */
    public void matchChannelsWithList() {
     if (chanList != null) matchChannelsWithList(chanList);
    }
    /** Match this all channels with ones in this list. If a match is found
     * replace the reading's channel object with a reference to the one in the list.
     * This is used to match objects to a more complete description of a channel then
     * they probably have otherwise. Gives access to lat/lon/z, response info, calibration,
     * distance from epicenter, etc. */
    public void matchChannelsWithList(ChannelList channelList) {
	JasiReading obj[] = (JasiReading[]) toArray(new JasiReading[0]);
	for (int i = 0; i<obj.length; i++) {
	   obj[i].setChannelFromList(channelList);
	}
    }

   /** Commit changes or delete for only those JasiReadings that are associated with
   * this solution to underlying DataSource */
    public boolean commit(Solution sol) {
     int knt = 0;
	JasiReading obj[] = (JasiReading[]) toArray(new JasiReading[0]);
	for (int i = 0; i<obj.length; i++) {
	   if (obj[i].isAssociatedWith(sol)) {
           if (obj[i].commit()) knt++;
        }
	}

	return (knt > 0);
    }

/**
* Delete all readings associated with this solution that have a residual greater
* than or equal to this value. Returns the number of deleted readings.
*/
  public int stripByResidual(double val, Solution sol) {

     int knt = 0;

//     System.out.println ("Stripping by residule = "+val+"\n"+sol.toString());

     JasiReading jr[] = (JasiReading[]) getAssociatedWithSol(sol).toArray(new JasiReading[0]);

	for (int i = 0; i<jr.length; i++)  {
         if (Math.abs(jr[i].residual.doubleValue()) >= val) {

             // see deletePhase() for more info
             delete(jr[i]);
//             System.out.println ("Deleting: "+jr[i].toString());
         }
	}
     return knt;
  }
/**
 * Return the reading nearest to this time. Returns null if there is none.
 */
    public JasiReading getNearestToTime(double time) {

	if (isEmpty()) return null;

	JasiReading nearest = null;
	double nearestDiff = Double.MAX_VALUE;
	double diff;

	JasiReading jr[] = (JasiReading[]) toArray(new JasiReading[0]);
	for (int i = 0; i<jr.length; i++)	{

	    if (!jr[i].isDeleted()) {
		diff = Math.abs(jr[i].getTime() - time);

		if (diff < nearestDiff) {
		    nearestDiff = diff;
		    nearest = jr[i];
		}
	    }
	}

	return nearest;
    }
/**
 * Return the reading nearest to this time that matches this channel.
 * Returns null if there is none.
 */
    public JasiReading getNearestToTime(double time, Channel chan) {

	if (isEmpty()) return null;

	JasiReading nearest = null;
	double nearestDiff = Double.MAX_VALUE;
	double diff;

	JasiReading jr[] = (JasiReading[]) toArray(new JasiReading[0]);
	for (int i = 0; i<jr.length; i++)	{

	    if (!jr[i].isDeleted() &&
		    chan.equalsIgnoreCase(jr[i].getChannelObj())  ) {

		diff = Math.abs(jr[i].getTime() - time);

		if (diff < nearestDiff) {
		    nearestDiff = diff;
		    nearest = jr[i];
		}
	    }
	}

	return nearest;
    }

/**
 * Return the reading nearest to this time that is associated with this solution.
 * Returns null if there is none.
 */
    public JasiReading getNearestToTime(double time, Solution sol) {

	if (isEmpty()) return null;

	JasiReading nearest = null;
	double nearestDiff = Double.MAX_VALUE;
	double diff;

	JasiReading jr[] = (JasiReading[]) toArray(new JasiReading[0]);
	for (int i = 0; i<jr.length; i++)	{

	    if (!jr[i].isDeleted() && sol == jr[i].sol ) {

		diff = Math.abs(jr[i].getTime() - time);

		if (diff < nearestDiff) {
		    nearestDiff = diff;
		    nearest = jr[i];
		}
	    }
	}

	return nearest;
    }

    /** Return a string with a neat listing of this reading list. */
    public String toNeatString () {

	  JasiReading obj[] = new JasiReading[this.size()];
	  this.toArray(obj);

       if (obj.length == 0) return "No readings.";

       String str = obj[0].getNeatStringHeader() + "\n";

	  for (int i=0; i<obj.length; i++) {

	    str += obj[i].toNeatString() + "\n";
	  }
       return str;
    }
/**
 * Sort the JasiReading list by time.
*/
    public void timeSort() {
	  Collections.sort(this, new TimeSorter());
    }

// Inner class to perform Comparator methods for sorting by time
class TimeSorter implements Comparator {

    public int compare (Object o1, Object o2) {

	JasiReading c1 = (JasiReading) o1;
	JasiReading c2 = (JasiReading) o2;

	double diff = c1.getDateTime().getEpochSeconds() -
                   c2.getDateTime().getEpochSeconds();
        if (diff < 0.0) return -1;
        if (diff > 0.0) return  1;
	return 0;

    }
}

/** Keeps this list in synch with another by reacting to change events from
* the "master". */
/*
public class SynchListener implements ChangeListener {

     Channel chan = null;
     Solution sol = null;

     public SynchListener (Channel chan) {
       this.chan = chan;
     }
     public SynchListener (Solution sol) {
       this.sol = sol;
     }


     // Respond to a list change.
	public void stateChanged (ChangeEvent changeEvent) {

         Object obj = changeEvent.getSource();

         // a single reading
         if (obj instanceof JasiReading) {
           JasiReading src = (JasiReading) obj;      // cast to correct type
           if (sol != null) {
             if (src.isAssociatedWith(sol) {

             }

           } else if (chan != null) {
             if (src.getChannel().equals(chan)) {

             }
           }
         }

         // a list
         else if (obj instanceof JasiReadingList)  {

           JasiReadingList src = (JasiReadingList) obj;      // cast to correct type
           if (sol != null) {

           } else if (chan != null) {

           }

     }
  }
}  // end of SynchListener
*/

} // JasiReadingList
