package org.trinet.util.magnitudeengines;

import java.util.*;
import org.trinet.jdbc.table.*;
import org.trinet.jasi.coda.*;
import org.trinet.jasi.*;

/** List of Mca (coda amplitude) station channel coda calibration objects.
* Calibration objects contain the data needed to correct the calculated coda amplitude magnitudes.
*/
public class McaCalibrationList extends Object implements CodaCalibrationListIF, Cloneable {
    private HashMap calibrList;   // list defined by a HashMap

/**
* Adds input coda calibration to the list.
* Returns false is input is null.
* @throws ClassCastException input not an instance of McaCalibrationTN
*/
    public boolean add(CodaCalibration calibr) {
        return add((McaCalibrationTN) calibr);
    }

/**
* Adds input coda calibration to the list.
* Returns false is input is null.
*/
    public boolean add(McaCalibrationTN calibr) {
        if (calibr == null ) return false;
        Object key = calibr.getChannelId();
        if (key == null) throw new NullPointerException("Calibration ChannelIdentifier is null");
        if (calibrList == null) calibrList = new HashMap();
        calibrList.put(key, calibr);
        return true;
    }

/**
* Returns coda calibration object which has the matching input channel identifier.
* Returns null is list is empty.
*/
    public McaCalibrationTN getMca(ChannelIdIF id) {
        return (McaCalibrationTN) get(id);
    }

/**
* Returns coda calibration object which has the matching input channel identifier.
* Returns null is list is empty.
*/
    public CodaCalibration get(ChannelIdIF id) {
        return (calibrList == null) ? null : (McaCalibrationTN) calibrList.get(id);
    }

/**
* Returns true if list contains a calibration object whose channel identifier matches input parameter.
* Returns false if list is empty.
*/
    public boolean contains(ChannelIdIF id) {
        return (calibrList == null) ? false : calibrList.containsKey(id);
    }
/**
* Returns true if list contains a calibration object matching input parameter.
* Returns false if input is null or input not an instance of McaCalibrationTN.
*/
    public boolean contains(CodaCalibration calibr) {
        if (calibr == null || ! (calibr instanceof McaCalibrationTN) ) return false;
        ChannelIdIF id = calibr.getChannelId();
        if (id == null) throw new NullPointerException("Calibration ChannelIdentifier is null");
        return contains(id);
    }

/**
* Returns true is no calibrations are in the list.
*/
    public boolean isEmpty() {
        return (calibrList == null) ? true : calibrList.isEmpty();
    }

/**
* Returns number of calibrations in the list.
*/
    public int size() {
        return (calibrList == null) ?  0: calibrList.size();
    }

/**
* Clears the list calibration data.
*/
    public void clear() {
        if (calibrList != null) calibrList.clear();
    }

/**
* Writes the list calibration data to the default data archive.
*/
    public boolean store() {
        if (calibrList == null) return false;
        Iterator iter = calibrList.values().iterator();
        boolean status = true;
        while (iter.hasNext()) {
            status = ((McaCalibrationTN) iter.next()).store();
            if (! status) break;
        }
        return status;
    }

/**
*  Initializes the coda calibration list with data read from default data source.
*/
    public boolean load() {
        return load((java.util.Date) null);
    }

/**
*  Initializes the coda calibration list with data read from default data source.
*/
    public boolean load(java.util.Date date) {
        Collection newCollection = new McaCalibrationTN().loadAll(date);  // retrieve all calibration data from data source
        if (newCollection == null || newCollection.size() < 1) return false;
        calibrList = new HashMap(newCollection.size());
        Iterator iter = newCollection.iterator();
        while (iter.hasNext()) {
            add((McaCalibrationTN) iter.next());
        }
        return true;
    }

    public boolean equals(Object object) {
        if (object == this) return true;
        if (object == null || object.getClass() != this.getClass()) return false;
        McaCalibrationList list = (McaCalibrationList) object;
        if (calibrList == null) {
            if (list.calibrList == null) return true;
        }
        return calibrList.equals(list.calibrList);
    }

    public Object clone() {
        McaCalibrationList list = null;
        try {
            list = (McaCalibrationList) super.clone();
            if (calibrList != null) list.calibrList = (HashMap) calibrList.clone();
        }
        catch(CloneNotSupportedException ex) {
             ex.printStackTrace();
        }
        return list;
    }

    public int hashCode() {
        return (calibrList == null) ? 0 : calibrList.hashCode();
    }


    public void printValues() {
        if (calibrList == null || calibrList.isEmpty()) {
            System.out.println("McaCalibrationList null or empty");
            return;
        }
        Collection list = calibrList.values();
        Iterator iter = list.iterator();
        System.out.println("Channel Calibr      cutoff   clip  count    afix   qfix   afree   qfree   slope");
        while (iter.hasNext()) {
            System.out.println(((McaCalibrationTN) iter.next()).toString());
        }
    }
}   // end of McaCalibrationList class


