package org.trinet.jdbc.datatypes;
import java.util.*;
import org.trinet.jdbc.*;

/** Extends the base abstract DataObject class to implements a stateful boolean value.
*/
public class DataBoolean extends DataObject implements Comparable {
/** Data member holding value. */
    boolean value;

/** Constructor sets the object value to false and the object state flags to their default settings:
* (null:true, update:false, mutable:true).
*/
    public DataBoolean () {
	this.value = false;
    }
 
/** Constructor sets the object value to true if input argument != 0, else false.
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
*/
    public DataBoolean (boolean value) {
	setValue(value);
    }
/** Constructor sets the object value to true if input argument != 0, else false.
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
*/
    public DataBoolean (int value) {
	setValue(value);
    }
/** Constructor sets the object value to true if input argument != 0, else false.
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
*/
    public DataBoolean (long value) {
	setValue(value);
    }
/** Constructor sets the object value to true if input argument != 0, else false.
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
*/
    public DataBoolean (float value) {
	setValue(value);
    }
/** Constructor sets the object value to true if input argument != 0, else false.
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
*/
    public DataBoolean (double value) {
	setValue(value);
    }
/** Constructor sets the object value to true if argument is such that Double.parseDouble(inputString) != 0, else false.
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
*/
    public DataBoolean (String value) throws NumberFormatException {
	setValue(value);
    }

/** Copies the object value and state flag settings of the input argument object. 
*/ 
    public DataBoolean (DataBoolean object) {
	if (object == null) setNull(true);
	else {
	   this.value = object.value;
	   this.valueUpdate = object.valueUpdate;
	   this.valueNull = object.valueNull;
	   this.valueMutable = object.valueMutable;
	}
    }

/** Returns the String equivalent of the object value: "true" or "false". */
    public String toString() {
	return String.valueOf(value); 
    }

/** Returns the String equivalent of the object value or "NULL" is isNull() == true. */
    public String toStringSQL() {
	if (valueNull) return "NULL";
	return StringSQL.valueOf(String.valueOf(value));
    }

/** Returns a String of "label: value" pairs for the object value and its state flags. 
* "Value: " + value.toString() + " Null: " + isNull() + " Update: " + isUpdate() + " Mutable: " + isMutable()
*/
    public String classToString() {
	StringBuffer sb = new StringBuffer(128);
	sb.append("                                           ");
	sb.insert(0, "Value:");
	sb.insert(7, value);
	sb.insert(32, "Null:");
	sb.insert(37, valueNull);
	sb.insert(43, "Update:");
	sb.insert(50, valueUpdate);
	sb.insert(56, "Mutable:");
	sb.insert(64, valueMutable);
	return sb.toString().trim();
//	return  "Value: " + value + " Null: " + valueNull + " Update: " + valueUpdate + " Mutable: " + valueMutable;
    }

/** Returns 1231 if the object value == true; 1237 if object value == false. See java.lang.Boolea.
* @see java.lang.Boolean
*/
    public int hashCode() {
	if (value == true) return 1231;
	else return 1237;
    }

/** Returns true if the input object is of type Boolean or DataBoolean and object.booleanValue() == this.booleanValue(). */
    public boolean equalsValue(Object object) {
	if (object == null ) return false;
	if (object instanceof Boolean) {
	    if (value == ((Boolean) object).booleanValue()) return true;
	    else return false; 
	}
	else if (object instanceof DataBoolean) {
	    if (value == ((DataBoolean) object).value) return true;
	    else return false; 
	}
	else return false; 
    }

/** Returns true if the input object is of type DataBoolean, this.value == object.value, 
* and its state flags are equivalent to those of this object, else returns false.
*/
    public boolean equals(Object object) {
	if (object == null || ! (object instanceof DataBoolean)) return false;
	if (value == ((DataBoolean) object).value && 
	   valueUpdate == ((DataBoolean) object).valueUpdate && 
	   valueMutable == ((DataBoolean) object).valueMutable && 
	   valueNull == ((DataBoolean) object).valueNull) return true;
	else return false;
    }

/** Returns 0 if this.value == object.value, returns 1 if this.value == true and object.value == false.
* Returns -1 if this.value == false and object.value == true.
* Input object must be of type java.lang.Boolean or DataBoolean.
*/
    public int compareTo(Object object) throws ClassCastException {
	if (object instanceof Boolean) { 
	    if (this.value == ((Boolean) object).booleanValue()) return 0;
	    else if (this.value == true && ((Boolean) object).booleanValue() == false) return 1;
	    else return -1;
	}
	else if (object instanceof DataBoolean) { 
	    return compareTo((DataBoolean) object);
	}
	else throw new ClassCastException("compareTo(object) argument must be a Boolean or DataBoolean class type: "
			+ object.getClass().getName());
    }

/** Returns 0 if this.value == object.value, returns 1 if this.value == true and object.value == false.
* Returns -1 if this.value == false and object.value == true.
*/
    public int compareTo(DataBoolean object) {
	if (this.value == object.value) return 0;
	    else if (this.value == true && ((DataBoolean) object).value == false) return 1;
	else return -1;
    }


/** Returns the object boolean value. */
    public boolean booleanValue() {
	if (value == true) return true;
	else return false;
    }

/** Returns 1 if the object boolean value is true, else 0. */
    public int intValue() {
	if (value == true) return 1;
	else return 0;
    }

/** Returns 1 if the object boolean value is true, else 0l */
    public long longValue() {
	if (value == true) return 1l;
	else return 0l;
    }

/** Returns 1 if the object boolean value is true, else 0.f */
    public float floatValue() {
	if (value == true) return 1.f;
	else return 0.f;
    }

/** Returns 1 if the object boolean value is true, else 0.d */
    public double doubleValue() {
	if (value == true) return 1.;
	else return 0.;
    }

/** Sets the object boolean value to the boolean input argument.
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
* Does a no-op, if isMutable == false. 
*/
    public void setValue(boolean value) {
	if(! isMutable()) return;
	this.value = value;
	this.valueNull = false;
	this.valueUpdate = true;
	return;
    }

/** Sets the object boolean value to true if input argument != 0 .
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
* Does a no-op, if isMutable == false. 
*/
    public void setValue(int value) {
	if(! isMutable()) return;
	if (value != 0) this.value = true;
	this.valueNull = false;
	this.valueUpdate = true;
	return;
    }

/** Sets the object boolean value to true if input argument != 0 .
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
* Does a no-op, if isMutable == false. 
*/
    public void setValue(long value) {
	if(! isMutable()) return;
	if (value != 0l) this.value = true;
	this.valueNull = false;
	this.valueUpdate = true;
	return;
    }

/** Sets the object boolean value to true if input argument != 0 .
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
* Does a no-op, if isMutable == false. 
*/
    public void setValue(float value) {
	if(! isMutable()) return;
	if (value != 0.f) this.value = true;
	this.valueNull = false;
	this.valueUpdate = true;
	return;
    }

/** Sets the object boolean value to true if input argument != 0 .
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
* Does a no-op, if isMutable == false. 
*/
    public void setValue(double value) {
	if(! isMutable()) return;
	if (value != 0.) this.value = true;
	this.valueNull = false;
	this.valueUpdate = true;
	return;
    }

/** Sets the object boolean value to true if input argument is a DataObject, Number, or String object and its value parses != 0.
* Sets the object state flags: isNull() == false, and isUpdate() == true. 
* Does a no-op, if isMutable == false. 
* Throws a ClassCastException if the input object is not one of these types.
*/
    public void setValue(Object object) throws ClassCastException, NumberFormatException  {
	if(! isMutable()) return;
//	if (object == null)  throw new NullPointerException("DataBoolean setValue(Object) null input argument");
	if (object == null) {
	    setNull(true);
	}
	else if (Number.class.isInstance(object)) {
	    setValue(((Number) object).intValue());
	}
	else if (DataObject.class.isInstance(object)) {
	    setValue(((DataObject) object).intValue());
	}
	else if (String.class.isInstance(object)) {
	    setValue(Double.parseDouble((String) object));
	}
	else throw new ClassCastException("setValue(Object) invalid object argument class type: " + object.getClass().getName());
	return;
    }

/** Sets the object boolean value to false.
* Sets the object state flags: isNull() == true, and isUpdate() == true. */
    public DataObject setNull(boolean value) {
	this.value = false;
	this.valueNull = true;
	this.valueUpdate = true;
	return this;
    }

/**
* Returns true if a value can be parsed from input StringTokenizer.
* Value is true only if tokenizer.nextToken.equalsIgnoreCase("true"), else value is set false..
* Does not set value and returns false if tokenizer.hasMoreTokens() == false.
*/
    public boolean parseValue(StringTokenizer tokenizer) {
        if (! tokenizer.hasMoreTokens()) return false;
        setValue( Boolean.valueOf(tokenizer.nextToken()).booleanValue() );
        return true;
    }

}
