package org.trinet.jasi.seed;

import java.io.*;
import java.sql.CallableStatement;
import java.sql.SQLException;
import java.util.*;
import java.lang.*;
import java.math.*;

import org.trinet.jasi.*;
import org.trinet.jdbc.*;
import org.trinet.util.Bits;
import org.trinet.util.DateTime;
import org.trinet.util.TimeSpan;
import org.trinet.util.EpochTime;
import org.trinet.util.BenchMark;

import org.trinet.util.WaveClient;

// Changed to use Fissure decoder to get Steim2
// see: http://www.seis.sc.edu/software/Fissures/seed.html
import edu.iris.Fissures.seed.codec.Steim1;
import edu.iris.Fissures.seed.codec.Steim2;

/**
 * SeedReader. Read seed files and interpret them. All methods in the class are
 * static because we want only one instance of the RemoteFileReader. The class will
 * determine if Waveform files are local or remote at instantiation time.
 * If remote, the class reads from the data source using DbBlobReader which gets
 * timeseries using a stored procedure in the dbase. <p>
 *
 * This class also knows how to read the waveforms from various data sources.
 * See the various getDataXxxxxx() methods.

 */
/*
1/15/02 Integrated AWW's DbBlobReader
*/
public class SeedReader {
/* double Precision problems!
0.01 * 300 = 3.0
0.01 * 301 = 3.0100000000000002
0.01 * 302 = 3.02
0.01 * 303 = 3.0300000000000002
0.01 * 304 = 3.04
0.01 * 305 = 3.0500000000000003
*/

protected static CallableStatement csStmt;

// SEED header
  static SeedHeader header;

  // Header fields
  /*
  static int    sequenceNumber;
  static String stationName;
  static String locationName;
  static String componentName;
  static String networkName;
  static String headerType;

  static double startTime;		// converted to local epoch

  static int    sampleCount;		// count of samples in the Seed packet
  static double samplesPerSecond;
  static double sampleMultiplier;

  static byte activityFlag;
  static byte ioFlag;
  static byte qualityFlag;
  static int  headerTimeCorrection;

  static private int blockettesFollowing;

  static int dataOffset;               //byte offset to start of data, counting from byte 0
  static int blocketteOffset;          //start of 1st blockette, counting from byte 0

// Blockette 1000
  static int encodingFormat;            // from blockette 1000
  static int wordOrder;			//0=VAX/Intel, 1=SUN/Motorola
  static int dataRecordLength;
*/
    // Data encoding formats described in SEED v2.3,  see SEED Ref. Man. pg. 106
    // ONLY STEIM1 IS CURRENTLY SUPPORTED
//    public static final byte STEIM1	  = 10;
//    public static final byte STEIM2	  = 11;
//    public static final byte GEOSCOPE24   = 12;
//    public static final byte GEOSCOPE16_3 = 13;
//    public static final byte GEOSCOPE16_4 = 14;
//    public static final byte USNSN	  = 15;
//    public static final byte CDSN	  = 16;
//    public static final byte GRAEFENBERG  = 17;
//    public static final byte IGP	  = 18;

// Blockette 1001
  static int timeQuality;
  static int timeCorrection;
  static int reserved;
  static int framesInRecord;

// reuseable variable to reduce instantiations in methods :. speed up
  static int map[] = new int[16];	// 0 - 15;			// map of compression mask
//  static float pt[];			// ref. to decompressed samples
  static int imap = -1;
  static int firstValue;
  static int lastValue;

  static final int FrameSize = 64;	// in bytes

/** Bytes per read */
  final static int READBUFFERSIZE = 16384;

/** bytes per Seed packet work area, equal to maximum Seed packet size we can handle */
  final static int DEFAULTSEEDBLOCKSIZE  = 8192;    // bytes per Seed packet

// move this to file scope?
  static byte[] buff = new byte[DEFAULTSEEDBLOCKSIZE];

  static int seedBlockSize;	// may be set otherwise someday

  static TimeSpan fileSpan = new TimeSpan(); // earliest & latest times in file

  static Waveform wf;
  static WFSegment wfseg = new WFSegment();

    // Make the FTP reader, make one at instantiation time so the connection
    // will remain open rather then getting garbage collected.
    //  static RemoteFileReader remoteReader = new RemoteFileReader();
//  static RemoteFileReader remoteReader = null;
  static DbBlobReader dbBlobReader = null;

  static boolean accessKnown = false;		// is local/remote known?

/**
 * JDBConn must already be established.
 */

    public SeedReader ()
    {
    }
/** Decompresses MiniSEED packets.  Returns float[] of time-series samples if
* packet header indicates a data packet, else returns null.
*
* @exception java.lang.SeedReaderException input null, input packet <512 bytes, or i/o
* error parsing packet data.  */
    public static float [] getDataSamples(byte [] seedPacket) throws SeedReaderException {
        float [] samples = null;
        try {
           if (seedPacket == null)
               throw new NullPointerException("SeedReader.getDataSamples(byte[]) Null input parameter");
           if (seedPacket.length < 512)
               throw new IllegalArgumentException("SeedReader.getDataSamples(byte[]) Input byte array length:"
                                                   + seedPacket.length + " is too short for known packet sizes");

            WFSegment wfSegment = createWFSegment(seedPacket);
      header = getHeader();

            if (header.isData() ) { // decompress only data messages
                samples = decompress (header, seedPacket);
            }
        }
        catch (NullPointerException ex) {
            throw new SeedReaderException(ex.getMessage());
        }
        catch (IllegalArgumentException ex) {
            throw new SeedReaderException(ex.getMessage());
        }
        catch (Exception ex) {
            ex.printStackTrace();
            throw new SeedReaderException("getDataSamples(): " + ex.getMessage());
        }
        return samples;
    }

/** Decodes a Collection containing miniSEED time-series data packets as elements.
* Returns a Collection of jasi.WFSegments with the same number of elements.
* @exception java.lang.SeedReaderException input null, or i/o error parsing packet data.
*/
    public static Collection getWFSegments(Collection packetList) {
        if (packetList == null)
                throw new NullPointerException("getWFSegments() Null input collection of seed packets.");

        final int MAX_PACKET_SIZE = 8192;        // test efficiency, instead of dynamically creating new arrays for each packet.
        Collection wfSegmentCollection = new ArrayList(packetList.size());
        Iterator iter = packetList.iterator();

        try {
            while (iter.hasNext()) {
                byte [] seedPacket = (byte []) iter.next();
    WFSegment wfSegment = createWFSegment(seedPacket);
    header = getHeader();

                if ( header.isData() ) { // decompress only data messages
                   wfSegment.setTimeSeries(
                        decompress (header, seedPacket)
                   );
                }
                wfSegmentCollection.add(wfSegment);
            }
        }
        catch (NullPointerException ex) {
            throw new SeedReaderException(ex.getMessage());
        }
        catch (Exception ex) {
            ex.printStackTrace();
            throw new SeedReaderException("getWFSegments(): " + ex.getMessage());
        }
        return wfSegmentCollection;
    }

/**
 * Read a file in Seed format as described in a waveform object. Returns
 * the number of WFSegments read . The access method, local or remote,
 * is determined automatically.
 */

public static int getData (Waveform wf) {
    return getData(wf, wf.getStart().doubleValue(), wf.getEnd().doubleValue());
}

public static int getData (Waveform wf, double startTime, double endTime) {
    if (wf.getWaveSourceType() == Waveform.LoadFromDataSource) {
  // Try accessing locally first, if that fails use remote access.
  if (wf.filesAreLocal()) {
      return getDataLocal(wf);	// try local access
        }
        else {
      return getDataFromDataSource(wf, startTime, endTime); // try test DB access via blob - aww
        }
  //} else {
  //    return getDataRemote(wf);	// try remote ftp access
  //}
    } else if (wf.getWaveSourceType() == Waveform.LoadFromWaveServer) {
  return getDataFromWaveServer(wf);
    }
    return 0;
}

    //** Get timeseries from a wave server */
    public static int getDataFromWaveServer (Waveform waveform) {

       WaveClient waveClient = (WaveClient) Waveform.getWaveSource();

       //       System.out.println ("SeedReader: waveClient ="+waveClient);

       try {

     // Get the time-series
     waveClient.setTruncatetAtTimeGap(false);

        // not sure this is always true
        waveform.setAmpUnits(Units.COUNTS);

     return waveClient.getJasiWaveformDataRaw (waveform);

       }
       catch (Exception ex) {
     System.err.println(ex.toString());
     ex.printStackTrace();
       }
       finally {
     //	   if (waveClient != null) waveClient.close();
       }

       return 0;
    }


/**
 * Return a ChannelableList of Waveform objects extracted from a Local or NFS mounted file
 * containing Seed format.  Returns null on fatal error. */

public static ChannelableList getDataFromFile (String filename, int traceoff) {

    ChannelableList wfList = new ChannelableList();
    Waveform wf = null;

    FileInputStream seedStream;
    File file = new File(filename);

    long filesize = file.length();
    filesize -= traceoff;

    //    System.out.println ("SeedReader opening "+file+ "  "+traceoff+"  "+nbytes);

    // open the Seed File
    try {
//      seedStream = new FileInputStream(filename);
      seedStream = new FileInputStream(file);
    } catch (FileNotFoundException exc) {
  System.out.println ("File not found: " + filename);
  return null;
    } catch (IOException exc) {
  System.err.println ("IO error: " + exc.toString());
  return null;
    }

    BufferedInputStream inbuff =
  new BufferedInputStream (seedStream, READBUFFERSIZE);

    int buffOffset = 0;
    int dataSize = 0;
    int totalBytes = 0;
    int lastChannelStart = 0;

// skip the proper byte offset in the file

    if (traceoff > 0) {
  try {
    long bytesSkippped = inbuff.skip(traceoff);

    // catch non-exception problems
    if (bytesSkippped != traceoff) {
        System.err.println ("IO error: could only skip "+bytesSkippped+
          " wanted to skip " + traceoff);
        return null;
    }
      } catch (IOException exc) {
    System.err.println ("IO error skipping to data offset: " +
            exc.toString());
    return null;
      }
    }

// file read loop; while data is there and we haven't yet gotten all the bytes
    try {
     while (inbuff.available() > 0) {
     // && totalBytes < nbytes) {

  buffOffset = 0;
/*
 * NOTE: Seed packets can be of various sizes, although the size must be a power of 2.
 * You don't know what the size is until you read the header of the packet.
 */
// <1> read the header

//NOTE: DANGER HERE: THIS ASSUMES THE HEADER IS TOTALLY CONTAINED WITHIN THE
//      FIRST 64-BYTE FRAME. HEADERS MAY EXTEND INTO SUBSEQUENT FRAMES IF THE
//      BLOCKETTES FILL THE FIRST ONE (E.G. THE "V" HEADERS ARE > 64 BYTES)
  totalBytes +=
      inbuff.read(buff, buffOffset, FrameSize);	// read one Seed header (64 bytes)

  lastChannelStart =  totalBytes - FrameSize;   // beginning of a channel's data

  header = SeedHeader.parseSeedHeader(buff);

  if (header == null) {
     System.out.println("Skipping malformed SEED header");
     continue;  // continue while loop to read next frame
  } else if (!header.isData()) {
     System.out.println("Skipping non-data header of type "+ header.headerType);
     continue;  // continue while loop to read next frame
  }

  wfseg = SeedReader.createWFSegment(header);
  if (wfseg == null) return null;   // bad data

// <2> read the data part now that we know how big it is (=blockSize)

  dataSize = header.seedBlockSize - FrameSize;

  buffOffset = FrameSize;	    // start storing bytes in buffer at # 64

  totalBytes +=
      inbuff.read(buff, buffOffset, dataSize);	// append Seed data frames

  if ( header.isData() )	// a data "record"
  {

       // decompress the data into the WFSegment
    wfseg.setTimeSeries( decompress (header, buff) );

     // is it a channel we already have in our list
     wf = (Waveform) wfList.getByChannel(wfseg.getChannelObj());

     if (wf == null) {            // not in list :. a new channel/waveform
        wf = Waveform.create();

        // not sure this is always true
        wf.setAmpUnits(Units.COUNTS);
        wf.setFileOffset(lastChannelStart);
        wf.setFilename(filename);
        wf.setFileOffset(traceoff);
        wf.setChannelObj(wfseg.getChannelObj());
        wf.format.setValue(Waveform.SEED_FORMAT);
        wf.setSampleRate(1.0/wfseg.getSampleInterval());
        wf.encoding.setValue(header.encodingFormat);
        wf.setStart(wfseg.getStart());
    //           wf.id = evid;   // no meaningfull file!

        wfList.add(wf);
     }
       // add segment to the Waveform
    wf.getSegmentList().add(wfseg);
    wf.setEnd(wfseg.getEnd());       // push back end time

   } else {	// skip non-data blocks
    // noop
   }

      // progress
      System.out.print ("\rRead: " +(int) (((double)totalBytes/(double)filesize) * 100.0)+"%  "+
                       totalBytes+ "  / "+ filesize);

     } // end of while loop
      System.out.println ("");

      inbuff.close();
      seedStream.close();
    } catch (IOException exc) {
    System.err.println ("IO error: " + exc.toString());
    exc.printStackTrace();
    } catch (Exception exc) {
    System.err.println ("General exception: " + exc.toString());
    exc.printStackTrace();
    }

    // collapse multiple segements into minimum possible
    Waveform wfa[] = (Waveform[]) wfList.toArray(new Waveform[0]);
    for (int i = 0; i < wfa.length; i++) {

      if (wfa[i] != null ) {
        wfa[i].collapse();
        // set max, min, bias values
        wfa[i].scanForAmps();
        wfa[i].scanForBias();
      }
    }


    return wfList;

    }



/**
 * Read a Local or NFS mounted file containing Seed format as described in a
 * waveform object. Populates the Waveform's segVector vector with WFSegments
 * from the file. Returns the number of WFSegments in the vector.  */

    // NOTE: we pass the FileNotFoundException upward to the caller
public static int getDataLocal (Waveform waveform)
//    throws FileNotFoundException
{
    wf = waveform;

    // not sure this is always true
    wf.setAmpUnits(Units.COUNTS);

    // build file name. Path comes from 'algorithm'
    String file  = wf.getPathFilename();
    int traceoff = wf.getFileOffset();
    int nbytes   = wf.getByteCount();

    FileInputStream seedStream;

    // open the Seed file
    try {
      seedStream = new FileInputStream(file);
    } catch (FileNotFoundException exc) {
  System.out.println ("File not found: " + file);
  return 0;
    }
    catch (IOException exc)
    {
  System.err.println ("IO error: " + exc.toString());
  return 0;
    }

    BufferedInputStream inbuff =
  new BufferedInputStream (seedStream, READBUFFERSIZE);

    int buffOffset = 0;
    int dataSize = 0;
    int totalBytes = 0;

// skip the proper byte offset in the file

    if (traceoff > 0) {
  try
      {
    long bytesSkippped = inbuff.skip(traceoff);

    // catch non-exception problems
    if (bytesSkippped != traceoff) {
        System.err.println ("IO error: could only skip "+bytesSkippped+
          " wanted to skip " + traceoff);
        return 0;
    }
      }
  catch (IOException exc)
      {
    System.err.println ("IO error skipping to data offset: " +
            exc.toString());
    return 0;
      }
    }


// file read loop; while data is there and we haven't yet gotten all the bytes
    try {
     while (inbuff.available() > 0 && totalBytes < nbytes) {

  buffOffset = 0;

/*
 * NOTE: Seed packets can be of various sizes, although the size must be a power of 2.
 * You don't know what the size is until you read the header of the packet.
 */
// <1> read the header
  totalBytes +=
      inbuff.read(buff, buffOffset, FrameSize);	// read one Seed header

        wfseg = SeedReader.createWFSegment(buff);
  header = SeedReader.getHeader();

  //	System.out.println ("Parsed SEED header: " +
  //			    headerType+"\n"+ wfseg.dumpToString() );

// <2> read the data part now that we know how big it is (=blockSize)

//        System.out.println ("seedBlockSize = "+ seedBlockSize);	    // debug

  dataSize = header.seedBlockSize - FrameSize;

  buffOffset = FrameSize;	    // start storing bytes in buffer at # 64

  totalBytes +=
      inbuff.read(buff, buffOffset, dataSize);	// append Seed data frames

  if (header.isData())	// a data "record"
  {

// decompress the data into the WFSegment

    wfseg.setTimeSeries( decompress (header, buff) );

    //	System.out.println (" Samples count = " + wfseg.sample.length);

    wf.getSegmentList().add(wfseg);

   } else {
    // noop
   }
     } // end of while loop

      inbuff.close();
      seedStream.close();
    }
    catch (IOException exc)
      {
    System.err.println ("IO error: " + exc.toString());
    exc.printStackTrace();
      }
    catch (Exception exc)
      {
    System.err.println ("General exception: " + exc.toString());
    exc.printStackTrace();
      }

    return wf.getSegmentList().size();
}

/**
 * Read a remote file via FTP containing Seed format as described in a
 * waveform object. Populates the Waveform's segVector vector with WFSegments
 * from the file. Returns the number of WFSegments in the vector.
 */
/*
public static int getDataRemote (Waveform wf) {

    // not sure this is always true
    wf.setAmpUnits(Units.COUNTS);

    // get the remote host name from the DataSource
    //String remoteHost = DataSource.getHostName();
    //String remoteHost = DataSource.getIPAddress();
    String remoteHost = DataSource.getIPAddress();
// catch FTP exceptions
    try {

    if (remoteReader == null) remoteReader = new RemoteFileReader(remoteHost);

 // use the FTP reader that was created at instantiation time
    //    byte[] bytes = remoteReader.getBytes (wf);
    byte[] bytes = remoteReader.getBytes(wf.getPathFilename(),
           wf.getFileOffset(),
           wf.getByteCount());

    // check for no data
    if (bytes.length == 0) return 0;

    // System.out.println ("SeedReader got byte[] of size: "+bytes.length);

    // create a stream for reading
    ByteArrayInputStream inbuff = new ByteArrayInputStream (bytes);

    int buffOffset = 0;
    int dataSize = 0;

// file read loop =============================
     while (inbuff.available() > 0) {

  buffOffset = 0;

// <1> read the header
// NOTE: Seed packets can be of various sizes, although the size must be a
// power of 2. You don't know what the size is until you read the header
// of the packet.

  inbuff.read(buff, buffOffset, FrameSize);	// read one Seed header

  wfseg = parseSeedHeader (buff);

// <2> read the data part now that we know how big it is (=blockSize)

  dataSize = seedBlockSize - FrameSize;
  buffOffset = FrameSize;	    // start storing bytes in buffer at # 64

  inbuff.read(buff, buffOffset, dataSize);	// append Seed data frames

  if ( headerType.equalsIgnoreCase("D") )	// a data "record"
  {

// decompress the data into the WFSegment

    wfseg.setTimeSeries( decompress (wfseg.samplesExpected, buff) );

    wf.getSegmentList().add(wfseg);

   } else if (headerType.equalsIgnoreCase("V") )	// skip non-data blocks
   {
    // noop
   }
     } // end of while loop

      inbuff.close();
    }
    catch (IOException exc)
      {
  System.out.println ("IO error: " + exc.toString());
      }
    catch (Exception exc)
      {
  System.out.println ("General exception: " + exc.toString());
      }

    return wf.getSegmentList().size();

    }
*/
  /** Get the timeseries for this waveform from the datasource. */
  public static int getDataFromDataSource(Waveform waveform) {
    if (dbBlobReader == null) dbBlobReader = new DbBlobReader();
    return dbBlobReader.getDataFromDataSource(waveform,
      waveform.getStart().doubleValue(),waveform.getEnd().doubleValue());
  }
  /** Get the timeseries for this waveform and this start/stop time from the datasource. */
  public static int getDataFromDataSource(Waveform waveform, double startTime, double endTime) {
    if (endTime < startTime) throw new IllegalArgumentException("input endTime < startTime" );
    if (dbBlobReader == null) dbBlobReader = new DbBlobReader();
    return dbBlobReader.getDataFromDataSource(waveform, startTime, endTime);
  }
/**
 * Create a WFsegment using the information in this SEED header.
 * This method will not populate the data part of the WFsegment.
 * You must call decompress() to interpret and load timeseries.
 */
public static WFSegment createWFSegment (SeedHeader h)  {
    return h.createWFsegment();
}
/*
   Channel ch = Channel.create();
   ch.setChannelName(h.chan);
   WFSegment wfseg = new WFSegment (ch);

   wfseg.setStart(h.datetime);
   wfseg.samplesExpected = h.sampleCount;
         wfseg.setSampleInterval(1.0/h.samplesPerSecond);

    // object exists, assume a default format and zero length pathname
        if (wf == null) {                     // no valid Waveform
      //wfseg.fmt = STEIM1;
      wfseg.fmt = Waveform.SEED_FORMAT;
      wfseg.filename = "";
        }
        else {                               // use Waveform
//	        wfseg.fmt = h.encodingFormat;
      wfseg.fmt = wf.format.intValue(); // value from dbase
      wfseg.filename = wf.getPathFilename();
        }
  return wfseg;
    }
    */
/*
    READ QUANTERRA SEED "RECORDS" (PACKETS)

    Seed records are FIXED size and are always a power of 2; usually
    from 512 or 4096 bytes.

    SEED "records" are made up of 64 byte "frames".
    There is ALWAYS a 48 byte fixed data header. The header may be
    followed by any number of "blockettes" which are addition data
    and information about the time series in the record. Any extra space
    in the header contains garbage.
    If there are enough blockettes they may extend into the next frame or frames.
    Therefore, data may begin in the 2nd, 3rd or Nth frame. A non-data
    record may consist of ONLY a header and some blockettes and garbage in
    the time-series frames.

   512 byte fixed length records are divided into 8 64-byte "frames"
   The first frame contains the SEED data header and data blockettes.
   For non-time-series packets the remaining 7 frames contain more
   blockettes (usually 1 or 2 total) followed by garbage.
   For time-series packets the remaining 7 frames contain compressed data.

c       The FIXED DATA HEADER contains how many blockettes follow and each
c       blockette contains the byte offset to the next blocket.
c
c  SEED "D" RECORD STRUCTURE (have Time-series "1001" Blockettes)
c
c Frame#
c       +-------------------------------------------------------+
c   1   |  Fixed data header  | 1000 blockette | 1001 blockette |
c       |     (48 bytes)      |   (8 bytes)    |   (8 bytes)    |
c       +-------------------------------------------------------+
c   2   |       Data Frame                                      |
c       +-------------------------------------------------------+
c   3   |       Data Frame                                      |
c       +-------------------------------------------------------+
c               .
c               .
c               .
c       +-------------------------------------------------------+
c   7   |       Data Frame                                      |
c       +-------------------------------------------------------+
c
c
c  SEED "other" RECORD STRUCTURE (All other Blockettes)
c
c Frame#
c       +-------------------------------------------------------+
c   1   |  Fixed data header  |   Blockette   |    Blockette... |
c       |     (48 bytes)      | (8-32 bytes)  |   (8-32 bytes)  |
c       +-------------------------------------------------------+
c   2   | ... more blockette(s)                                 |
c       +-------------------------------------------------------+
c   3   |       Garbage                                         |
c       +-------------------------------------------------------+
c               .
c               .
c               .
c       +-------------------------------------------------------+
c   7   |       Garbage                                         |
c       +-------------------------------------------------------+
c
c Data Blockette structure:
c
c       FIELD                   TYPE            LENGTH (bytes)
c       Blockette type          Byte            2       (interpret as int)
c       Byte of next blockette  Byte            2       (interpret as int)
c       Data fields             depends on blockette type
c        .
c        .
c        .
c
c Known Blockette types:
c
c       NUMBER          TYPE
c       100             sample rate
c       200             generic event detection
c       201             Murdoch-Hutt event detection
c       202             LOG-Z event detection
C       300             step calibration
C       310             sine calibration
C       320             pseudo-random calibration
C       390             generic calibration
C       395             calibration abort
C       400             Beam (no used by Quanterra)
C       405             Beam delay (no used by Quanterra)
c       500             time stamp
c       1000            Data format description
c       1001            Data (time-series)
*/

/**
 *
 * Return a SeedHeader intepreted from this byte array.
 * Just a passthru method that calls  SeedHeader.parseSeedHeader(buff).
 */
public static SeedHeader parseSeedHeader (byte[] buff)  {

  return SeedHeader.parseSeedHeader(buff);
    }
/**
 * Create a WFsegment using the information in this SEED header.
 * This method will not populate the data part of the WFsegment.
 * You must call decompress() to interpret and load timeseries.
 * Returns null if buffer did not contain a readable SEED header.
 */
public static WFSegment createWFSegment (byte[] buffer)  {
  header = SeedHeader.parseSeedHeader(buffer);
  if (header == null)  return null;
  return header.createWFsegment();
}

/** Get the SEED header object for the last byte buffer processed.
 *  Will be null if nothing processed or the data type is not "D" */
    public static SeedHeader getHeader() {
      return header;
    }
/**
 * Decode a 10-byte Seed BTIME and convert to "UNIX" epoch time
    INTEGER*2     YR
    INTEGER*2     JDAY
    BYTE          HOUR
    BYTE          MIN
    BYTE          SEC
    BYTE          %FILL
    INTEGER*2     MSEC            ! ms * 10
 */
    public static double seedTime (byte[] buff, int idx)
    {
  int yr	 = Bits.byteToInt2 ( buff, idx+0);
  int jday = Bits.byteToInt2 ( buff, idx+2);
  int hr	 = buff[idx+4];
  int mn	 = buff[idx+5];
  int sec	 = buff[idx+6];
  int frac = Bits.byteToInt2 ( buff, idx+8);

// DateTime class keeps precision to nanoseconds (0.000001),
// native Java time only good to milliseconds (0.001)
  double fsec = (double) sec + (double)frac/10000.0;
  DateTime datetime = new DateTime (yr, 0, jday, hr, mn, fsec);

  double dt0 = datetime.getEpochSeconds();

  return dt0;
    }

/**
 * Decompress one SEED packet (Only handles Steim1 compression at this time)
 * Assumes you've already parsed the header because
 */
/*    static float[] decompress (SeedHeader header, byte[] buff) {
       return decompress (header.dataOffset, header.sampleCount,
        header.framesInRecord, buff);
    }
*/

 /*
      float pt[] = new float[header.sampleCount];

// Decode the X0 & Xn in the 1st frame, these are
// the "forward" and "reverse integration constants" (first and last values)
// Note that the first sample will be a repeat of the last sample of the
// previous SEED packet.

  firstValue = Bits.byteToInt4 (buff, header.dataOffset+4);	// the first value
  lastValue  = Bits.byteToInt4 (buff, header.dataOffset+8);

// Decode the remaining frames
// Doing this in two steps simplifies the logic because the first sample is given
// both explicity and as a difference

  int offset;
  int nword;
  int bytePtr;
  int idx = -1;
  int lastIndex = header.sampleCount - 1;     //	nSamples -1;

// Step 1: Load differences into the 'pt[]' array
  int frameNo = 0;

// NOTE: the total samples in the Seed record may be less then what it can hold.
//       :. we must believe the value given in the header and passed as nSamples.
     try {
//	while (idx < lastIndex) {
  while (frameNo < header.framesInRecord) {

    offset = header.dataOffset + (frameNo * FrameSize);	// byte offset to frame
       // decode the compression map that tells us how may bytes are used
       // for each data value here. It always has 16 values.
    map = compressionMap (buff, offset);

  // handle bytes in groups of 4-bytes (1 longword)
    for (nword=0; nword<map.length; nword++) {

      bytePtr = offset + (nword*4);	// step 4 bytes each time

      if (map[nword] == 0) {		// not compressed data, skip it
      } else if (map[nword]  == 1) {	// four 1-byte samples
      pt[++idx] = buff[bytePtr];
      pt[++idx] = buff[bytePtr+1];
      pt[++idx] = buff[bytePtr+2];
      pt[++idx] = buff[bytePtr+3];
      } else if (map[nword] == 2) {	// two 2-byte samples
      pt[++idx] = Bits.byteToInt2 (buff, bytePtr);
      pt[++idx] = Bits.byteToInt2 (buff, bytePtr+2);
      } else if (map[nword] == 3) {	// one 4-byte sample
      pt[++idx] = Bits.byteToInt4 (buff, bytePtr);
      }
    } // end of for (nword=0...

    frameNo++;        // next frame

  }   // end of while ....
     // catch attempt to write more samples than there should be
     } catch (IndexOutOfBoundsException ex) {
         System.out.println (ex.toString());
         idx--;
     }

// Step 2: Calculate actual values from differences

  pt[0] = firstValue;
  for (int i=1; i<=idx; i++) {
      pt[i] = pt[i-1] + pt[i];
  }

// sanity checks
  if (idx+1 != header.sampleCount) {
      System.out.println (" * Sample count mismatch: header said = "
        + header.sampleCount + "  decompressed count = " + (idx+1));
  }

  if (pt[idx] != lastValue) {
      System.out.println (" * Last value mismatch: Xn = " + lastValue
      + "  last decompressed value = " + pt[idx]);
  }

  return pt;

    }	// end of decompress()
    */

/*
    static float[] decompressSteim1 (int dataOffset, int sampleCount,
             byte[] buff) {
  boolean swap = false;
  int[] samps = new int[sampleCount];

  try {
    int dataBytes = buff.length-dataOffset;
    byte[] data = new byte[dataBytes];
    System.arraycopy(buff, dataOffset, data, 0, dataBytes);

    samps = Steim1.decode(data, sampleCount, swap);
  } catch (Exception ex) {
          System.out.println (ex.toString());
    return new float[0];
  }

  float pt[] = new float[sampleCount];
  // convert int to float
  for (int i = 0; i < sampleCount; i++) {
     pt[i] = (float) samps[i];
  }
  return pt;
    }
*/
/**
 * Decompress one SEED packet (Only handles Steim1 or 2 compression at this time)
 * Assumes you've already parsed the header. Byte-swaps if necessary. Returns
 * float[0] array on error.
 */
    static float[] decompress (SeedHeader header, byte[] buff) {

  // determine if you must byte-swap the timeseries
  boolean swap = false;
  if (header.wordOrder == 0) swap = true;

  int[] samps = new int[header.sampleCount];

  try {
    int dataBytes = buff.length-header.dataOffset;
    byte[] data = new byte[dataBytes];
    System.arraycopy(buff, header.dataOffset, data, 0, dataBytes);

    if (header.encodingFormat == SeedEncodingFormat.STEIM1) {
    samps = Steim1.decode(data, header.sampleCount, swap);
    } else if (header.encodingFormat == SeedEncodingFormat.STEIM2) {
      samps = Steim2.decode(data, header.sampleCount, swap);
    } else {
            throw new SeedReaderException("Can't decode data format: "+header.encodingFormat);
    }
  } catch (Exception ex) {
          System.out.println (ex.toString());
    return new float[0];
  }

  // convert int to float
  float pt[] = new float[header.sampleCount];
  for (int i = 0; i < header.sampleCount; i++) {
     pt[i] = (float) samps[i];
  }
  return pt;
    }

/**
 * Decompress one SEED packet (Only handles Steim1 compression at this time)
 * Assumes you've already parsed the header because
 */
 /*
    static float[] decompress (int dataOffset, int sampleCount,
             int framesInRecord, byte[] buff) {

    return decompressSteim1(dataOffset, sampleCount,
                buff);
    }
*/

/*
      float pt[] = new float[sampleCount];

// Decode the X0 & Xn in the 1st frame, these are
// the "forward" and "reverse integration constants" (first and last values)
// Note that the first sample will be a repeat of the last sample of the
// previous SEED packet.

  float firstValue = Bits.byteToInt4 (buff, dataOffset+4);	// the first value
  float lastValue  = Bits.byteToInt4 (buff, dataOffset+8);

// Decode the remaining frames
// Doing this in two steps simplifies the logic because the first sample is given
// both explicity and as a difference

  int offset;
  int nword;
  int bytePtr;
  int idx = -1;
  //int lastIndex = header.sampleCount - 1;     //	nSamples -1;
  int lastIndex = sampleCount - 1;     //	nSamples -1;

// Step 1: Load differences into the 'pt[]' array
  int frameNo = 0;

// NOTE: the total samples in the Seed record may be less then what it can hold.
//       :. we must believe the value given in the header and passed as nSamples.
     try {
  while (idx < lastIndex) {
//	while (frameNo < framesInRecord) {

    offset = dataOffset + (frameNo * FrameSize);	// byte offset to frame
       // decode the compression map that tells us how may bytes are used
       // for each data value here. It always has 16 values.
    map = compressionMap (buff, offset);

  // handle bytes in groups of 4-bytes (1 longword)
    for (nword=0; nword<map.length; nword++) {

      bytePtr = offset + (nword*4);	// step 4 bytes each time

      if (map[nword] == 0) {		// not compressed data, skip it
      } else if (map[nword]  == 1) {	// four 1-byte samples
      pt[++idx] = buff[bytePtr];
      pt[++idx] = buff[bytePtr+1];
      pt[++idx] = buff[bytePtr+2];
      pt[++idx] = buff[bytePtr+3];
      } else if (map[nword] == 2) {	// two 2-byte samples
      pt[++idx] = Bits.byteToInt2 (buff, bytePtr);
      pt[++idx] = Bits.byteToInt2 (buff, bytePtr+2);
      } else if (map[nword] == 3) {	// one 4-byte sample
      pt[++idx] = Bits.byteToInt4 (buff, bytePtr);
      }
    } // end of for (nword=0...

    frameNo++;        // next frame

  }   // end of while ....
     // catch attempt to write more samples than there should be
     } catch (IndexOutOfBoundsException ex) {
         System.out.println (ex.toString());
         idx--;
     }

// Step 2: Calculate actual values from differences

  pt[0] = firstValue;
  for (int i=1; i<=idx; i++) {
      pt[i] = pt[i-1] + pt[i];
  }

// sanity checks
  if (idx+1 != sampleCount) {
      System.out.println (" * Sample count mismatch: header said = "
        + sampleCount + "  decompressed count = " + (idx+1));
  }

  if (pt[idx] != lastValue) {
      System.out.println (" * Last value mismatch: Xn = " + lastValue
      + "  last decompressed value = " + pt[idx]);
  }

  return pt;

    }	// end of decompress()

*/

/**
 * Decode the "nibbles" in the first 32 bits of the frame that encode how
 * many bytes are in each difference. the arg
 * 'offset' is the address of the first byte of the first frame. Bytes are counted
 * from 0 :. this value will typically be 64. <p>

 The compression map is a "longword" (4 bytes) and is interpreted as 16 2-byte
 nibbles.  Each nibble (C00 - C15) tells you the amount of compression in the
 corrisponding 4-byte longword in the data portion of the frame.

 <tt>
 |     byte #0   |    byte #1    |    byte #2    |    byte #3    |
 +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
 |C00|C01|C02|C03|C04|C05|C06|C07|C08|C09|C10|C11|C12|C13|C14|C15|
 +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+

 16 nibbles = 32 bits = 4 bytes

 Cxx = 0   - special, the matching longword is not compressed data but
             something else (like a compression map value)
 Cxx = 1   - four 1-byte differences
 Cxx = 2   - two 2-byte differences
 Cxx = 3   - one 4-byte difference
 </tt>
 See Seed Manual Pg. 121-122.
 */
/*
    public static int[] compressionMap (byte[] buff, int offset)
    {
  imap = -1;

  for (int nbyte=offset; nbyte<offset+4; nbyte++)
  {
   for (int nib=0; nib<4; nib++)   // for each nibble
   {
    map[++imap] = buff[nbyte] >> (6-(nib*2)) & 0x03;

//	  System.out.println (" imap=" + imap +  "  val=" + map[imap]);
   }
  }

  return (map);

    }

    public final void finalize() {
      try {
        if (csStmt != null) csStmt.close();
      } catch (SQLException ex) {}
      csStmt =  null;
    }
*/

/// --------------------------------------------------------------------------
/**
 * Main for testing
 */

    public static void main (String args[])
    {
        Vector segVector  = new Vector();   // ref to vector of WF segments

  int evid;

  if (args.length > 0)	// translate epoch second on command-line value
  {
    Integer val = Integer.valueOf(args[0]);	    // convert arg String to 'double'
    evid =  val.intValue();
  } else {
      //	    evid = 9500724;	    // test event
      //evid = 9749085;
            evid = 2218477;

      System.out.println("Syntax: SeedReader <evid> <mode> <dbhost> <doublestart> <doubleend>");
  }

        System.out.println ("Making connection... ");
        DataSource jc = new TestDataSource();

  // Make a persistent instance so the FTP connection can be reused

  SeedReader reader = new SeedReader();

  //	reader.setFileAccess();
        Waveform wfi = Waveform.create();

  System.out.println ("evid = " +evid+ " filesAreLocal= "+ wfi.filesAreLocal());
  ArrayList wfList = (ArrayList) wfi.getBySolutionId(evid);

  System.out.println( "wf list length = "+ wfList.size());

        Waveform wf [] = new Waveform[wfList.size()];
        wfList.toArray(wf);
        if (wf == null)
      {
    System.out.println ("No waveforms in dbase for evid "+evid);
    System.exit(0);
      }

  // local read
  int totalBytes = 0;

  int count=0;
  int status = args.length;
//        String mode = "LOCAL";
        String mode = "DB";
        if (status > 0) mode = args[1].toUpperCase();
    System.out.println("Getting waveforms source input mode: " + mode);
        double wfStartTime = 0.;
        double wfEndTime = 0.;
        double startTime = 0.;
        double endTime   = 99999999999.;
        if (args.length > 3) {
          startTime = Double.parseDouble(args[3]);
        }
        if (args.length > 4) {
          endTime = Double.parseDouble(args[4]);
        }

//        boolean verbose = false;
        boolean verbose = true;
        if (args.length > 3) {
          verbose = true;
          //verbose = Boolean.valueOf(args[5]).booleanValue();
        }

  BenchMark bm  = new BenchMark();
  BenchMark bm2 = new BenchMark();
        for (int i = 0; i<wf.length; i++)
        {
        if (verbose) System.out.print ("Count =  "+ ++count + " reading... "+ wf[i].toString() );
            wfStartTime = wf[i].getStart().doubleValue();
            wfEndTime   = wf[i].getEnd().doubleValue();
      System.out.println("Requested window: " +
          EpochTime.epochToString(wfStartTime) + " to " +
    EpochTime.epochToString(wfEndTime));
      bm.reset();
            if (mode.equals("LOCAL")) {
        status = reader.getDataLocal(wf[i]);
//	    else if (mode.equals("REMOTE")) status = reader.getDataRemote(wf[i]);
//            else if (mode.equals("DB")) {
            } else if (mode.equals("REMOTE") || mode.equals("DB")) {
              //startTime = Math.min(startTime+20.,endTime);
              //endTime   = Math.max(endTime-20.,startTime);
              double sTime   = Math.max(wfStartTime,startTime);
              double eTime   = Math.min(wfEndTime, endTime);
              if (eTime > sTime) {
                if (verbose) System.out.println(" requested time window data len secs: " + (eTime - sTime)) ;
                status = reader.getDataFromDataSource(wf[i],sTime,eTime);
              }
              else if (verbose) System.out.println("** Note requested start time > end time of waveform.");
            }
            else status = getData(wf[i]);
      if (status > 0)
    if (verbose) System.out.println ("Read " + wf[i].getSegmentList().size() + " segments for " + wf[i].toString() );
      if (verbose) bm.print("BenchMark: ");
            // Dump segments
            if (args.length > 5) {
              WFSegment [] wfseg = wf[i].getArray();
              for (int j = 0; j < wfseg.length; j++) {
                 if (verbose) System.out.println ( wfseg[j].toString() );
              }
            }
            wfseg = null;
            wf[i] = null;
        }
  bm2.print("TOTAL BenchMark: ");
  bm2.reset();
    }
} // end of class
