package org.trinet.jasi;

import org.trinet.jdbc.datatypes.DataDouble;

/** The gain of a single channel. It has two parts a value and units.
* Gain is usually the relationship between digital counts and actual ground motion
* for a seismic sensor. <p>
* For example, CI_CBK_EHZ is a short-period velocity transducer with e a gain of
* 8966788 and units of 'Units.CntCMS' ("counts/cm/sec"). So one cm/sec equals
* 8966788 digital counts or one count = 1/8966788 cm/sec. Thus, this is a
* very high-gain station (i.e. it magnifies the ground motion alot.<p>
* CI_CTC_HLZ is an accelerometer. It has a gain of 5356 and units of
* "counts/cm/sec^2". Its a strong motion instrument (low-gain). */

public class ChannelGain implements Cloneable, java.io.Serializable {

/** The actual gain value. */
     DataDouble gain = new DataDouble();
     /** The units of the gain value. Default is counts/cm/sec.
     * @See: Units */
     int units = Units.CntCMS;    // counts/cm/sec

     public ChannelGain() {
     }

     public ChannelGain(ChannelGain cgain) {
            gain.setValue(cgain.doubleValue());
            units = cgain.getUnits();
     }

     /** Return the units string, e.g. "counts/cm/sec", "counts/cm/sec^2"
     * @See: Units */
     public String getUnitsString() {
       return Units.getString(units);
     }

     /** Return the value of the units type.
     *
     * @See: Units
     */
     public int getUnits() {
       return units;
     }
     /** Set the units of the gain. Returns false is the string is not a legal
     * units description.
     * @See: Units */
     public void setUnits(int iunits) {
       units = iunits;
     }
     /** Set the units of the gain. Returns false is the string is not a legal
     * units description.
     * @See: Units */
     public boolean setUnits(String sunits) {
        if (Units.isLegal(sunits)) {
          units = Units.getInt(sunits);
          return true;
        }
        return false;
     }

     /** Return true if the gain value is null. */
     public boolean isNull () {
       return gain.isNull();
     }
     /** Return the gain value. Returns NaN if the value is null.*/
     public double doubleValue() {
       return gain.doubleValue();
     }

     /** Set the gain to this value. Units should be set also but will default to
     * counts/cm/sec. */
     public void setValue(Object gainValue) {
       gain.setValue(gainValue);
     }

     public String toString (){
       return gain + " " +getUnitsString();
     }

     public Object clone() {
       ChannelGain chanGain = null;
       try {
         chanGain = (ChannelGain) super.clone();
       }
       catch (CloneNotSupportedException ex) {
         ex.printStackTrace();
       }
       chanGain.gain = (DataDouble) gain.clone();
       return chanGain;
     }
}
